package ru.yandex.intranet.d.services.delivery.provide;

import java.math.BigDecimal;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.common.collect.Sets;
import com.yandex.ydb.table.transaction.TransactionMode;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuple2;

import ru.yandex.intranet.d.dao.Tenants;
import ru.yandex.intranet.d.datasource.model.YdbTableClient;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.model.accounts.AccountModel;
import ru.yandex.intranet.d.model.accounts.AccountSpaceModel;
import ru.yandex.intranet.d.model.accounts.AccountsQuotasModel;
import ru.yandex.intranet.d.model.accounts.AccountsQuotasOperationsModel;
import ru.yandex.intranet.d.model.accounts.OperationChangesModel;
import ru.yandex.intranet.d.model.accounts.OperationErrorKind;
import ru.yandex.intranet.d.model.accounts.OperationInProgressModel;
import ru.yandex.intranet.d.model.accounts.OperationOrdersModel;
import ru.yandex.intranet.d.model.accounts.OperationSource;
import ru.yandex.intranet.d.model.delivery.DeliverableDeltaModel;
import ru.yandex.intranet.d.model.delivery.DeliverableFolderOperationModel;
import ru.yandex.intranet.d.model.delivery.DeliverableMetaHistoryModel;
import ru.yandex.intranet.d.model.delivery.DeliverableMetaRequestModel;
import ru.yandex.intranet.d.model.delivery.provide.DeliveryAndProvideDestinationModel;
import ru.yandex.intranet.d.model.delivery.provide.DeliveryAndProvideModel;
import ru.yandex.intranet.d.model.delivery.provide.DeliveryAndProvideOperationListModel;
import ru.yandex.intranet.d.model.delivery.provide.DeliveryAndProvideOperationModel;
import ru.yandex.intranet.d.model.delivery.provide.DeliveryAndProvideOperationWithIdModel;
import ru.yandex.intranet.d.model.delivery.provide.DeliveryAndProvideRequestModel;
import ru.yandex.intranet.d.model.folders.AccountHistoryModel;
import ru.yandex.intranet.d.model.folders.AccountsHistoryModel;
import ru.yandex.intranet.d.model.folders.FolderModel;
import ru.yandex.intranet.d.model.folders.FolderOperationLogModel;
import ru.yandex.intranet.d.model.folders.FolderOperationType;
import ru.yandex.intranet.d.model.folders.OperationPhase;
import ru.yandex.intranet.d.model.folders.ProvisionHistoryModel;
import ru.yandex.intranet.d.model.folders.ProvisionsByResource;
import ru.yandex.intranet.d.model.folders.QuotasByAccount;
import ru.yandex.intranet.d.model.folders.QuotasByResource;
import ru.yandex.intranet.d.model.providers.AccountsSettingsModel;
import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.model.quotas.QuotaModel;
import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.model.resources.segmentations.ResourceSegmentationModel;
import ru.yandex.intranet.d.model.resources.segments.ResourceSegmentModel;
import ru.yandex.intranet.d.model.units.UnitModel;
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel;
import ru.yandex.intranet.d.model.users.UserModel;
import ru.yandex.intranet.d.services.delivery.model.DeliverableMetaRequest;
import ru.yandex.intranet.d.services.delivery.model.DeliverableRequestGroupKey;
import ru.yandex.intranet.d.services.delivery.model.DeliveryQuotas;
import ru.yandex.intranet.d.services.delivery.model.ValidatedQuotas;
import ru.yandex.intranet.d.services.delivery.model.provide.DeliveryAndProvideDestination;
import ru.yandex.intranet.d.services.delivery.model.provide.DeliveryAndProvideDictionary;
import ru.yandex.intranet.d.services.delivery.model.provide.DeliveryAndProvideRequest;
import ru.yandex.intranet.d.services.delivery.model.provide.DeliveryStatusProvideDictionary;
import ru.yandex.intranet.d.services.delivery.model.provide.ProvideDictionary;
import ru.yandex.intranet.d.services.delivery.model.provide.ProvideState;
import ru.yandex.intranet.d.services.delivery.model.provide.ProvideStateStatus;
import ru.yandex.intranet.d.services.delivery.model.provide.UpdateDictionary;
import ru.yandex.intranet.d.services.integration.providers.ProvidersIntegrationService;
import ru.yandex.intranet.d.services.integration.providers.Response;
import ru.yandex.intranet.d.services.integration.providers.rest.model.AccountDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.AccountsSpaceKeyRequestDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.GetAccountRequestDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.KnownAccountProvisionsDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.KnownProvisionDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.ProvisionRequestDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.ResourceKeyRequestDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.SegmentKeyRequestDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.UpdateProvisionRequestDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.UpdateProvisionResponseDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.UserIdDto;
import ru.yandex.intranet.d.services.operations.OperationsObservabilityService;
import ru.yandex.intranet.d.services.operations.model.ReceivedAccount;
import ru.yandex.intranet.d.services.operations.model.ReceivedLastUpdate;
import ru.yandex.intranet.d.services.operations.model.ValidatedReceivedAccount;
import ru.yandex.intranet.d.services.operations.model.ValidatedReceivedProvision;
import ru.yandex.intranet.d.services.operations.model.ValidatedReceivedUpdatedProvision;
import ru.yandex.intranet.d.util.result.ErrorCollection;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.util.result.TypedError;
import ru.yandex.intranet.d.util.units.Units;
import ru.yandex.intranet.d.web.model.delivery.provide.DeliveryAndProvideRequestDto;
import ru.yandex.intranet.d.web.model.delivery.status.DeliveryStatusDto;

import static ru.yandex.intranet.d.model.folders.OperationPhase.SUBMIT;
import static ru.yandex.intranet.d.services.delivery.model.provide.ProvideStateStatus.FAILURE;

/**
 * Delivery and provide service implementation.
 *
 * @author Evgenii Serov <evserov@yandex-team.ru>
 */
@Component
public class DeliveryAndProvideService {
    private static final Logger LOG = LoggerFactory.getLogger(DeliveryAndProvideService.class);
    private static final int MAX_RETRY_COUNT = 3;

    private final DeliveryAndProvidePreValidationService preValidationService;
    private final DeliveryAndProvideValidationService validationService;
    private final DeliveryAndProvideStoreService storeService;
    private final DeliveryAndProvideStatusService statusService;
    private final ProvidersIntegrationService providersIntegrationService;
    private final OperationsObservabilityService operationsObservabilityService;
    private final YdbTableClient tableClient;
    private final MessageSource messages;

    @SuppressWarnings("ParameterNumber")
    public DeliveryAndProvideService(
            DeliveryAndProvidePreValidationService preValidationService,
            DeliveryAndProvideValidationService validationService,
            DeliveryAndProvideStoreService storeService,
            DeliveryAndProvideStatusService statusService,
            ProvidersIntegrationService providersIntegrationService,
            OperationsObservabilityService operationsObservabilityService,
            YdbTableClient tableClient,
            @Qualifier("messageSource") MessageSource messages
    ) {
        this.preValidationService = preValidationService;
        this.validationService = validationService;
        this.storeService = storeService;
        this.statusService = statusService;
        this.providersIntegrationService = providersIntegrationService;
        this.operationsObservabilityService = operationsObservabilityService;
        this.tableClient = tableClient;
        this.messages = messages;
    }

    public Mono<Result<DeliveryStatusDto>> deliverAndProvide(DeliveryAndProvideRequestDto requestDto,
                                                             Locale locale) {
        return preValidationService.preValidateProvideRequest(requestDto, locale).andThenMono(delivery ->
                        tableClient.usingSessionMonoRetryable(session ->
                                session.usingTxMonoRetryable(TransactionMode.SERIALIZABLE_READ_WRITE, ts ->
                                        validateExistingDelivery(delivery, locale, ts)
                                                .switchIfEmpty(deliverAndPrepareProvide(delivery, locale, ts)))))
                .flatMap(res -> res.andThenMono(result -> result.getDeliveryStatusDto() != null
                        ? Mono.just(Result.success(result.getDeliveryStatusDto()))
                        : provide(Objects.requireNonNull(result.getProvideDictionary()), locale)));
    }

    private Mono<Result<DeliveryStatusProvideDictionary>>
    validateExistingDelivery(DeliveryAndProvideRequest request, Locale locale, YdbTxSession ts) {
        return storeService.getDeliveryById(ts, request.getDeliveryId()).flatMap(finishedDeliveryO ->
                finishedDeliveryO.map(deliveryAndProvideModel ->
                                getExistingDeliveryStatus(request, deliveryAndProvideModel, locale)
                                        .map(res -> res.apply(status ->
                                                new DeliveryStatusProvideDictionary(status, null))))
                        .orElseGet(Mono::empty));
    }

    private Mono<Result<DeliveryStatusProvideDictionary>>
    deliverAndPrepareProvide(DeliveryAndProvideRequest request, Locale locale, YdbTxSession ts) {
        return storeService.loadDictionaries(ts, request)
                .flatMap(dict -> validationService.validateRequest(request, dict, locale).andThenMono(u ->
                        loadQuotas(ts, request, dict).flatMap(quotas ->
                                validationService.validateQuotas(request, quotas, dict, locale)
                                        .applyMono(validatedQuotas -> loadProvisions(ts, dict).flatMap(provisions ->
                                                applyDelivery(ts, dict, request, validatedQuotas, provisions))))))
                .map(res -> res.apply(provideDictionary -> new DeliveryStatusProvideDictionary(null,
                        provideDictionary)));
    }

    public Mono<Result<DeliveryStatusDto>> getExistingDeliveryStatus(DeliveryAndProvideRequest request,
                                                                     DeliveryAndProvideModel delivery,
                                                                     Locale locale) {
        DeliveryAndProvideRequestModel existingRequest = delivery.getRequest();
        Map<DeliverableRequestGroupKey, DeliverableFolderOperationModel> folderOperationByDeliverableKey =
                existingRequest.getDeliverables().stream().filter(d -> d.getFolderOperationModel() != null)
                        .collect(Collectors.toMap(DeliverableRequestGroupKey::new,
                                DeliveryAndProvideDestinationModel::getFolderOperationModel, (a, b) -> a));
        DeliveryAndProvideRequestModel newRequest = fromRequestToModel(request, folderOperationByDeliverableKey);
        if (!existingRequest.equals(newRequest)) {
            return Mono.just(Result.failure(ErrorCollection.builder().addError(TypedError.invalid(messages
                    .getMessage("errors.delivery.request.mismatch", null, locale))).build()));
        }
        return statusService.getDeliveryOperationStatuses(List.of(newRequest.getDeliveryId()), locale)
                .map(res -> res.andThen(response -> Result.success(response.getDeliveries().get(0))));
    }

    private Mono<DeliveryQuotas> loadQuotas(YdbTxSession session, DeliveryAndProvideRequest request,
                                            DeliveryAndProvideDictionary dictionary) {
        Set<QuotaModel.Key> keySet = new HashSet<>();
        request.getDeliverables().forEach(deliverable -> {
            ResourceModel resource = dictionary.getResources().get(deliverable.getResourceId());
            ProviderModel provider = dictionary.getProviders().get(deliverable.getProviderId());
            keySet.add(new QuotaModel.Key(deliverable.getFolderId(), provider.getId(), resource.getId()));
        });

        if (keySet.isEmpty()) {
            return Mono.just(new DeliveryQuotas(Map.of()));
        }

        return storeService.getQuotasByKeys(session, keySet).map(l -> new DeliveryQuotas(l.stream()
                .collect(Collectors.toMap(QuotaModel::toKey, Function.identity()))));
    }

    private Mono<List<AccountsQuotasModel>> loadProvisions(YdbTxSession session,
                                                           DeliveryAndProvideDictionary dictionary) {
        Set<String> accountIds = dictionary.getAccounts().values().stream()
                .map(AccountModel::getId)
                .collect(Collectors.toSet());

        if (accountIds.isEmpty()) {
            return Mono.just(List.of());
        }

        return storeService.getAccountQuotasByIds(session, accountIds);
    }

    private Mono<ProvideDictionary> applyDelivery(YdbTxSession session,
                                                  DeliveryAndProvideDictionary dictionary,
                                                  DeliveryAndProvideRequest request,
                                                  ValidatedQuotas quotas,
                                                  List<AccountsQuotasModel> provisions) {
        Instant now = Instant.now();
        Map<DeliverableRequestGroupKey, FolderOperationLogModel> opLogsByKey =
                prepareOpLogs(dictionary, request, quotas, now);
        List<FolderOperationLogModel> opLogs = new ArrayList<>(opLogsByKey.values());
        List<QuotaModel> updatedQuotas = new ArrayList<>(quotas.getUpdatedQuotas().values());

        Map<String, List<DeliveryAndProvideDestination>> providesByAccount = getProvidesByAccount(request);

        Map<String, AccountsQuotasOperationsModel> accountsQuotasOperationsByAccountId =
                prepareAccountsOperations(dictionary, request, provisions, now, providesByAccount, opLogs);
        List<AccountsQuotasOperationsModel> accountsQuotasOperationsModels =
                new ArrayList<>(accountsQuotasOperationsByAccountId.values());

        List<OperationInProgressModel> accountsOpInProgress =
                prepareAccountsOperationsInProgress(dictionary, accountsQuotasOperationsByAccountId);

        List<FolderOperationLogModel> opLogsForProvide =
                prepareOpLogsForProvide(dictionary, request, provisions, now, providesByAccount,
                        accountsQuotasOperationsByAccountId);
        opLogs.addAll(opLogsForProvide);

        DeliveryAndProvideModel deliveryRequestsModel = getDeliveryAndProvideModel(dictionary, request,
                opLogsByKey, accountsQuotasOperationsByAccountId);

        List<FolderModel> foldersToUpsert = updateNextOpLogOrder(dictionary, opLogs);

        accountsQuotasOperationsModels.forEach(operationsObservabilityService::observeOperationSubmitted);

        return storeService.upsertFolders(session, foldersToUpsert)
                .then(Mono.defer(() -> storeService.upsertQuotas(session, updatedQuotas)))
                .then(Mono.defer(() -> storeService.upsertAccountOperation(session, accountsQuotasOperationsModels)))
                .then(Mono.defer(() -> storeService.upsertAccountOperationInProgress(session, accountsOpInProgress)))
                .then(Mono.defer(() -> storeService.upsertOperationLog(session, opLogs)))
                .then(Mono.defer(() -> storeService.upsertDelivery(session, deliveryRequestsModel)))
                .then(Mono.defer(() -> storeService.loadDictionariesForProvide(session, dictionary,
                        deliveryRequestsModel, accountsQuotasOperationsByAccountId, accountsOpInProgress)));
    }

    private Map<String, List<DeliveryAndProvideDestination>> getProvidesByAccount(DeliveryAndProvideRequest request) {
        return request.getDeliverables().stream()
                .collect(Collectors.groupingBy(DeliveryAndProvideDestination::getAccountId,
                        Collectors.toList()));
    }

    @NotNull
    private DeliveryAndProvideModel getDeliveryAndProvideModel(
            DeliveryAndProvideDictionary dictionary,
            DeliveryAndProvideRequest request,
            Map<DeliverableRequestGroupKey, FolderOperationLogModel> opLogsByKey,
            Map<String, AccountsQuotasOperationsModel> accountsQuotasOperationsByAccountId) {
        var folderOperationByDeliverableKey = opLogsByKey.entrySet().stream()
                .collect(Collectors.toMap(Map.Entry::getKey, e -> DeliverableFolderOperationModel.builder()
                        .id(e.getValue().getId())
                        .folderId(e.getValue().getFolderId())
                        .operationDateTime(e.getValue().getOperationDateTime())
                        .build()));

        Collection<DeliveryAndProvideOperationListModel> deliveryAndProvideOperationListModels =
                prepareOperationModels(dictionary, accountsQuotasOperationsByAccountId);

        return new DeliveryAndProvideModel.Builder()
                .deliveryId(request.getDeliveryId())
                .request(fromRequestToModel(request, folderOperationByDeliverableKey))
                .tenantId(Tenants.DEFAULT_TENANT_ID)
                .addOperations(deliveryAndProvideOperationListModels)
                .build();
    }

    private Map<DeliverableRequestGroupKey, FolderOperationLogModel> prepareOpLogs(
            DeliveryAndProvideDictionary dictionary, DeliveryAndProvideRequest request, ValidatedQuotas quotas,
            Instant now) {
        Map<DeliverableRequestGroupKey, FolderOperationLogModel> result = new HashMap<>();
        Map<DeliverableRequestGroupKey, List<DeliveryAndProvideDestination>> deliverablesByKey =
                request.getDeliverables().stream()
                        .collect(Collectors.groupingBy(DeliverableRequestGroupKey::new, Collectors.toList()));
        List<DeliverableRequestGroupKey> keys = deliverablesByKey.keySet().stream()
                .sorted(DeliverableRequestGroupKey.COMPARATOR).toList();

        Map<String, Long> nextOpLogOrders = new HashMap<>();
        Map<QuotaModel.Key, Long> currentQuotaAmounts = new HashMap<>();
        Map<QuotaModel.Key, Long> updatedQuotaAmounts = new HashMap<>();

        keys.forEach(key -> {
            List<DeliveryAndProvideDestination> deliverables = deliverablesByKey.get(key);
            String targetFolderId = key.getFolderId().orElseThrow();
            deliverables.forEach(deliverable -> {
                String resourceId = deliverable.getResourceId();
                ResourceModel resource = dictionary.getResources().get(resourceId);
                QuotaModel.Key quotaKey = new QuotaModel.Key(deliverable.getFolderId(), deliverable.getProviderId(),
                        resourceId);
                long currentQuotaAmount = updatedQuotaAmounts.computeIfAbsent(quotaKey, k -> {
                    if (quotas.getOriginalQuotas().containsKey(k)) {
                        QuotaModel quota = quotas.getOriginalQuotas().get(k);
                        return quota.getQuota() != null ? quota.getQuota() : 0L;
                    }
                    return 0L;
                });
                UnitsEnsembleModel unitsEnsemble = dictionary.getUnitsEnsembles().get(resource.getUnitsEnsembleId());
                UnitModel unit = unitsEnsemble.unitByKey(deliverable.getDelta().getUnitKey()).orElseThrow();
                long deltaAmount = Units.convertFromApi(deliverable.getDelta().getAmount(), resource,
                        unitsEnsemble, unit).orElseThrow();
                long updatedQuotaAmount = Units.add(currentQuotaAmount, deltaAmount).orElseThrow();
                updatedQuotaAmounts.put(quotaKey, updatedQuotaAmount);
            });
            long nextOpLogOrder = nextOpLogOrders.computeIfAbsent(targetFolderId,
                    id -> dictionary.getFolders().get(id).getNextOpLogOrder());
            nextOpLogOrders.put(targetFolderId, nextOpLogOrder + 1L);
            updatedQuotaAmounts.keySet().forEach(updatedQuotaKey ->
                    currentQuotaAmounts.computeIfAbsent(updatedQuotaKey, k -> {
                        if (quotas.getOriginalQuotas().containsKey(updatedQuotaKey)) {
                            QuotaModel quota = quotas.getOriginalQuotas().get(updatedQuotaKey);
                            return quota.getQuota() != null ? quota.getQuota() : 0L;
                        } else {
                            return 0L;
                        }
                    }));
            Map<String, Long> oldQuotas = new HashMap<>();
            Map<String, Long> newQuotas = new HashMap<>();
            updatedQuotaAmounts.keySet().forEach(quotaKey -> {
                long newQuota = updatedQuotaAmounts.get(quotaKey);
                long oldQuota = currentQuotaAmounts.get(quotaKey);
                if (newQuota != oldQuota) {
                    oldQuotas.put(quotaKey.getResourceId(), oldQuota);
                    newQuotas.put(quotaKey.getResourceId(), newQuota);
                }
            });
            UserModel author = dictionary.getUsersByUid().get(request.getAuthorUid());
            FolderOperationLogModel opLog = FolderOperationLogModel.builder()
                    .setTenantId(Tenants.DEFAULT_TENANT_ID)
                    .setFolderId(targetFolderId)
                    .setOperationDateTime(now)
                    .setId(UUID.randomUUID().toString())
                    .setProviderRequestId(null)
                    .setOperationType(FolderOperationType.QUOTA_DELIVERY)
                    .setAuthorUserId(author.getId())
                    .setAuthorUserUid(author.getPassportUid().orElse(null))
                    .setAuthorProviderId(null)
                    .setSourceFolderOperationsLogId(null)
                    .setDestinationFolderOperationsLogId(null)
                    .setOldFolderFields(null)
                    .setOldQuotas(new QuotasByResource(oldQuotas))
                    .setOldBalance(new QuotasByResource(Map.of()))
                    .setOldProvisions(new QuotasByAccount(Map.of()))
                    .setOldAccounts(null)
                    .setNewFolderFields(null)
                    .setNewQuotas(new QuotasByResource(newQuotas))
                    .setNewBalance(new QuotasByResource(Map.of()))
                    .setNewProvisions(new QuotasByAccount(Map.of()))
                    .setActuallyAppliedProvisions(null)
                    .setNewAccounts(null)
                    .setAccountsQuotasOperationsId(null)
                    .setQuotasDemandsId(null)
                    .setOperationPhase(null)
                    .setOrder(nextOpLogOrder)
                    .setCommentId(null)
                    .setDeliveryMeta(DeliverableMetaHistoryModel.builder()
                            .deliveryId(request.getDeliveryId())
                            .quotaRequestId(key.getMeta().getQuotaRequestId())
                            .campaignId(key.getMeta().getCampaignId())
                            .bigOrderIds(key.getMeta().getBigOrderId())
                            .build())
                    .build();
            result.put(key, opLog);
            currentQuotaAmounts.putAll(updatedQuotaAmounts);
        });
        return result;
    }

    private List<FolderOperationLogModel> prepareOpLogsForProvide(
            DeliveryAndProvideDictionary dictionary, DeliveryAndProvideRequest request,
            List<AccountsQuotasModel> provisions, Instant now,
            Map<String, List<DeliveryAndProvideDestination>> providesByAccount,
            Map<String, AccountsQuotasOperationsModel> accountsQuotasOperationsByAccountId) {
        List<FolderOperationLogModel> result = new ArrayList<>();

        UserModel author = dictionary.getUsersByUid().get(request.getAuthorUid());

        DeliverableMetaRequest meta = request.getDeliverables().get(0).getMeta();
        Map<String, List<Long>> orderIdsByAccountId = new HashMap<>();
        request.getDeliverables().forEach(d ->
                orderIdsByAccountId.computeIfAbsent(d.getAccountId(), k -> new ArrayList<>())
                        .add(d.getMeta().getBigOrderId()));

        Map<String, Map<String, AccountsQuotasModel>> oldAccountsQuotaByResourceIdByAccountId = provisions.stream()
                .collect(Collectors.groupingBy(AccountsQuotasModel::getAccountId,
                        Collectors.toMap(AccountsQuotasModel::getResourceId, Function.identity())));

        providesByAccount.forEach((accountId, models) -> {
            DeliveryAndProvideDestination deliveryAndProvideDestination = models.get(0);
            String folderId = deliveryAndProvideDestination.getFolderId();
            AccountModel accountModel = dictionary.getAccounts().get(accountId);
            AccountsQuotasOperationsModel accountsQuotasOperationsModel =
                    accountsQuotasOperationsByAccountId.get(accountId);

            Map<String, AccountsQuotasModel> oldAccountsQuotaByResourceId =
                    oldAccountsQuotaByResourceIdByAccountId.getOrDefault(accountId, Map.of());

            Map<String, ProvisionHistoryModel> oldProvisionHistoryModelByResource = new HashMap<>();
            Map<String, ProvisionHistoryModel> newProvisionHistoryModelByResource = new HashMap<>();

            models.forEach(model -> {
                String resourceId = model.getResourceId();

                AccountsQuotasModel accountsQuotasModel = oldAccountsQuotaByResourceId.get(resourceId);
                ProvisionHistoryModel oldProvisionHistoryModel = accountsQuotasModel == null
                        ? new ProvisionHistoryModel(0L, null)
                        : new ProvisionHistoryModel(accountsQuotasModel.getProvidedQuota(),
                        accountsQuotasModel.getLastReceivedProvisionVersion().orElse(null));
                oldProvisionHistoryModelByResource.put(resourceId, oldProvisionHistoryModel);

                ResourceModel resource = dictionary.getResources().get(resourceId);
                UnitsEnsembleModel unitsEnsemble = dictionary.getUnitsEnsembles().get(resource.getUnitsEnsembleId());
                UnitModel unit = unitsEnsemble.unitByKey(model.getDelta().getUnitKey()).orElseThrow();
                long deltaProvisionAmount = Units.convertFromApi(model.getDelta().getAmount(), resource,
                        unitsEnsemble, unit).orElseThrow();
                long newProvision = oldProvisionHistoryModel.getProvision() + deltaProvisionAmount;
                long newVersion = oldProvisionHistoryModel.getVersion().orElse(-1L) + 1;
                ProvisionHistoryModel newProvisionHistoryModel = new ProvisionHistoryModel(newProvision, newVersion);
                newProvisionHistoryModelByResource.compute(resourceId, (k, v) -> v == null ? newProvisionHistoryModel
                        : new ProvisionHistoryModel(newProvision + v.getProvision(), newVersion));
            });

            var oldAccountQuotasByAccountMap = Map.of(accountModel.getId(),
                    new ProvisionsByResource(oldProvisionHistoryModelByResource));
            var newAccountQuotasByAccountMap = Map.of(accountModel.getId(),
                    new ProvisionsByResource(newProvisionHistoryModelByResource));

            FolderOperationLogModel folderOperationLogModel = FolderOperationLogModel.builder()
                    .setTenantId(Tenants.DEFAULT_TENANT_ID)
                    .setFolderId(folderId)
                    .setOperationDateTime(now)
                    .setId(UUID.randomUUID().toString())
                    .setProviderRequestId(accountsQuotasOperationsModel.getLastRequestId().orElseThrow())
                    .setOperationType(FolderOperationType.DELIVER_PROVIDE_QUOTAS_TO_ACCOUNT)
                    .setAuthorUserId(author.getId())
                    .setAuthorUserUid(author.getPassportUid().orElse(null))
                    .setAuthorProviderId(null)
                    .setSourceFolderOperationsLogId(null)
                    .setDestinationFolderOperationsLogId(null)
                    .setOldFolderFields(null)
                    .setOldQuotas(new QuotasByResource(Map.of()))
                    .setOldBalance(new QuotasByResource(Map.of()))
                    .setOldProvisions(new QuotasByAccount(oldAccountQuotasByAccountMap))
                    .setOldAccounts(null)
                    .setNewFolderFields(null)
                    .setNewQuotas(new QuotasByResource(Map.of()))
                    .setNewBalance(new QuotasByResource(Map.of()))
                    .setNewProvisions(new QuotasByAccount(newAccountQuotasByAccountMap))
                    .setActuallyAppliedProvisions(null)
                    .setNewAccounts(null)
                    .setAccountsQuotasOperationsId(accountsQuotasOperationsModel.getOperationId())
                    .setQuotasDemandsId(null)
                    .setOperationPhase(SUBMIT)
                    .setOrder(accountsQuotasOperationsModel.getOrders().getSubmitOrder())
                    .setCommentId(null)
                    .setDeliveryMeta(DeliverableMetaHistoryModel.builder()
                            .deliveryId(request.getDeliveryId())
                            .quotaRequestId(meta.getQuotaRequestId())
                            .campaignId(meta.getCampaignId())
                            .bigOrderIds(orderIdsByAccountId.getOrDefault(accountId, List.of()))
                            .build())
                    .build();

            result.add(folderOperationLogModel);
        });

        return result;
    }

    private Map<String, AccountsQuotasOperationsModel> prepareAccountsOperations(
            DeliveryAndProvideDictionary dictionary, DeliveryAndProvideRequest request,
            List<AccountsQuotasModel> provisions, Instant now,
            Map<String, List<DeliveryAndProvideDestination>> providesByAccount, List<FolderOperationLogModel> opLogs) {
        Map<String, AccountsQuotasOperationsModel> result = new HashMap<>();

        Map<String, Map<String, Long>> oldProvisionByResourceIdByAccountId = provisions.stream()
                .collect(Collectors.groupingBy(AccountsQuotasModel::getAccountId,
                        Collectors.toMap(AccountsQuotasModel::getResourceId, AccountsQuotasModel::getProvidedQuota)));

        UserModel author = dictionary.getUsersByUid().get(request.getAuthorUid());

        Map<String, Long> nextOpLogOrders = new HashMap<>();
        opLogs.forEach(opLog -> nextOpLogOrders.compute(opLog.getFolderId(), (k, v) ->
                v != null && v > opLog.getOrder() ? v : opLog.getOrder()));
        nextOpLogOrders.replaceAll((k, v) -> v + 1);

        providesByAccount.forEach((accountId, models) -> {
            Map<String, Long> oldProvisionsByResourceId =
                    oldProvisionByResourceIdByAccountId.getOrDefault(accountId, Map.of());

            AccountModel accountModel = dictionary.getAccounts().get(accountId);
            String folderId = accountModel.getFolderId();
            long nextOpLogOrder = nextOpLogOrders.get(folderId);
            nextOpLogOrders.put(folderId, nextOpLogOrder + 1L);


            Map<String, OperationChangesModel.Provision> provisionMap = new HashMap<>();
            Map<String, OperationChangesModel.Provision> frozenProvisionMap = new HashMap<>();
            models.forEach(model -> {
                String resourceId = model.getResourceId();
                ResourceModel resource = dictionary.getResources().get(resourceId);
                UnitsEnsembleModel unitsEnsemble = dictionary.getUnitsEnsembles().get(resource.getUnitsEnsembleId());
                UnitModel unit = unitsEnsemble.unitByKey(model.getDelta().getUnitKey()).orElseThrow();
                long deltaProvisionAmount = Units.convertFromApi(model.getDelta().getAmount(), resource,
                        unitsEnsemble, unit).orElseThrow();

                long oldProvisionAmount = oldProvisionsByResourceId.getOrDefault(resourceId, 0L);
                long newProvisionAmount = oldProvisionAmount + deltaProvisionAmount;

                provisionMap.compute(resourceId, (k, v) -> v == null
                        ? new OperationChangesModel.Provision(resourceId, newProvisionAmount)
                        : new OperationChangesModel.Provision(resourceId, newProvisionAmount + v.getAmount()));
                frozenProvisionMap.compute(resourceId, (k, v) -> v == null
                        ? new OperationChangesModel.Provision(resourceId, deltaProvisionAmount)
                        : new OperationChangesModel.Provision(resourceId, deltaProvisionAmount + v.getAmount()));
            });

            AccountsQuotasOperationsModel accountsQuotasOperationsModel = AccountsQuotasOperationsModel.builder()
                    .setTenantId(Tenants.DEFAULT_TENANT_ID)
                    .setOperationId(UUID.randomUUID().toString())
                    .setLastRequestId(UUID.randomUUID().toString())
                    .setCreateDateTime(now)
                    .setOperationSource(OperationSource.USER)
                    .setOperationType(AccountsQuotasOperationsModel.OperationType.DELIVER_AND_UPDATE_PROVISION)
                    .setAuthorUserId(author.getId())
                    .setAuthorUserUid(author.getPassportUid().orElse(null))
                    .setProviderId(accountModel.getProviderId())
                    .setAccountsSpaceId(accountModel.getAccountsSpacesId().orElse(null))
                    .setUpdateDateTime(null)
                    .setRequestStatus(AccountsQuotasOperationsModel.RequestStatus.WAITING)
                    .setErrorMessage(null)
                    .setRequestedChanges(OperationChangesModel.builder()
                            .accountId(accountModel.getId())
                            .updatedProvisions(new ArrayList<>(provisionMap.values()))
                            .frozenProvisions(new ArrayList<>(frozenProvisionMap.values()))
                            .deliveryId(request.getDeliveryId())
                            .build())
                    .setOrders(OperationOrdersModel.builder().submitOrder(nextOpLogOrder).build())
                    .setErrorKind(null)
                    .build();

            result.put(accountId, accountsQuotasOperationsModel);
        });

        return result;
    }

    private List<OperationInProgressModel> prepareAccountsOperationsInProgress(
            DeliveryAndProvideDictionary dictionary,
            Map<String, AccountsQuotasOperationsModel> accountsQuotasOperationsByAccountId) {
        List<OperationInProgressModel> result = new ArrayList<>();

        accountsQuotasOperationsByAccountId.forEach((accountId, accountsQuotasOperationsModel) -> {
            AccountModel accountModel = dictionary.getAccounts().get(accountId);

            OperationInProgressModel operationInProgressModel = OperationInProgressModel.builder()
                    .tenantId(Tenants.DEFAULT_TENANT_ID)
                    .operationId(accountsQuotasOperationsModel.getOperationId())
                    .folderId(accountModel.getFolderId())
                    .accountId(accountId)
                    .retryCounter(0L)
                    .build();

            result.add(operationInProgressModel);
        });

        return  result;
    }

    private List<FolderModel> updateNextOpLogOrder(DeliveryAndProvideDictionary dictionary,
                                                   Collection<FolderOperationLogModel> deliveryOpLogs) {
        List<FolderModel> result = new ArrayList<>();
        Map<String, Long> nextOpLogOrders = new HashMap<>();
        deliveryOpLogs.forEach(opLog -> {
            String folderId = opLog.getFolderId();
            long nextOpLogOrder = nextOpLogOrders.computeIfAbsent(folderId, id -> dictionary.getFolders().get(id)
                    .getNextOpLogOrder());
            nextOpLogOrders.put(folderId, nextOpLogOrder + 1L);
        });
        nextOpLogOrders.forEach((k, v) ->
                result.add(dictionary.getFolders().get(k).toBuilder().setNextOpLogOrder(v).build()));
        return result;
    }

    @NotNull
    private Collection<DeliveryAndProvideOperationListModel> prepareOperationModels(
            DeliveryAndProvideDictionary dictionary,
            Map<String, AccountsQuotasOperationsModel> accountsQuotasOperationsByAccountId) {
        return accountsQuotasOperationsByAccountId.entrySet().stream()
                .map(e -> {
                    String accountId = e.getKey();
                    AccountModel accountModel = dictionary.getAccounts().get(accountId);
                    FolderModel folderModel = dictionary.getFolders().get(accountModel.getFolderId());

                    DeliveryAndProvideOperationModel deliveryAndProvideOperationModel =
                            new DeliveryAndProvideOperationModel(0, e.getValue().getOperationId());
                    return new DeliveryAndProvideOperationListModel(folderModel.getServiceId(), folderModel.getId(),
                            accountId, List.of(deliveryAndProvideOperationModel));
                })
                .collect(Collectors.toList());
    }

    private DeliveryAndProvideRequestModel fromRequestToModel(
            DeliveryAndProvideRequest request,
            Map<DeliverableRequestGroupKey, DeliverableFolderOperationModel> folderOperationByDeliverableKey) {
        DeliveryAndProvideRequestModel.Builder builder = new DeliveryAndProvideRequestModel.Builder();
        builder.deliveryId(request.getDeliveryId());
        builder.authorUid(request.getAuthorUid());
        request.getDeliverables().forEach(deliverable -> {
            DeliveryAndProvideDestinationModel.Builder deliverableBuilder =
                    new DeliveryAndProvideDestinationModel.Builder();
            deliverableBuilder.serviceId(deliverable.getServiceId());
            deliverableBuilder.providerId(deliverable.getProviderId());
            deliverableBuilder.folderId(deliverable.getFolderId());
            deliverableBuilder.accountId(deliverable.getAccountId());
            deliverableBuilder.resourceId(deliverable.getResourceId());
            deliverableBuilder.delta(DeliverableDeltaModel.builder()
                    .amount(deliverable.getDelta().getAmount())
                    .unitKey(deliverable.getDelta().getUnitKey())
                    .build());
            deliverableBuilder.meta(DeliverableMetaRequestModel.builder()
                    .quotaRequestId(deliverable.getMeta().getQuotaRequestId())
                    .campaignId(deliverable.getMeta().getCampaignId())
                    .bigOrderId(deliverable.getMeta().getBigOrderId())
                    .build());
            DeliverableFolderOperationModel deliverableFolderOperationModel =
                    folderOperationByDeliverableKey.get(new DeliverableRequestGroupKey(deliverable));
            if (deliverableFolderOperationModel != null) {
                deliverableBuilder.folderOperationModel(DeliverableFolderOperationModel.builder()
                        .id(deliverableFolderOperationModel.getId())
                        .folderId(deliverableFolderOperationModel.getFolderId())
                        .operationDateTime(deliverableFolderOperationModel.getOperationDateTime())
                        .build());
            }
            builder.addDeliverable(deliverableBuilder.build());
        });
        return builder.build();
    }

    private Mono<Result<DeliveryStatusDto>> provide(ProvideDictionary provideDictionary, Locale locale) {
        DeliveryAndProvideModel deliveryRequestsModel = provideDictionary.getDeliveryRequestsModel();
        String deliveryId = deliveryRequestsModel.getDeliveryId();
        return Flux.fromIterable(getOperations(provideDictionary))
                .concatMap(op -> getAccountQuotasByResourceIdByAccountByFolder(provideDictionary).flatMap(provision -> {
                    DeliveryAndProvideOperationWithIdModel deliverOperation =
                            new DeliveryAndProvideOperationWithIdModel(deliveryId, op.getServiceId(),
                                    op.getFolderId(), op.getAccountId(), op.getOperations());

                    UpdateProvisionRequestDto updateProvisionRequestDto =
                            prepareUpdateProvisionRequestDto(provideDictionary, provision, op);

                    return updateProvisionRequest(deliverOperation, updateProvisionRequestDto, provideDictionary,
                            new AtomicInteger(MAX_RETRY_COUNT), locale).map(Result::success);
                }).onErrorResume(e -> Mono.just(Result.failure(ErrorCollection.builder()
                        .addError(TypedError.invalid(e.getMessage())).build())))).collectList()
                .map(results -> results.stream().map(res -> res.match(Function.identity(), e -> {
                    LOG.error("Deliver and provide updateProvisionRequest fails" + e.toString());
                    return new ProvideState(FAILURE);
                })).collect(Collectors.toList()))
                .then(statusService.getDeliveryOperationStatuses(List.of(deliveryId), locale))
                .map(r -> r.apply(deliveryStatusResponseDto -> deliveryStatusResponseDto.getDeliveries()
                        .stream().findFirst().orElseThrow()));
    }

    private List<DeliveryAndProvideOperationListModel> getOperations(ProvideDictionary provideDictionary) {
        Map<String, AccountModel> accountsById = provideDictionary.getAccountsById();
        return provideDictionary.getDeliveryRequestsModel().getOperations().stream()
                .sorted(Comparator.comparing(a -> accountsById.get(a.getAccountId()).getOuterAccountIdInProvider()))
                .collect(Collectors.toList());
    }

    private Mono<Map<String, Map<String, Map<String, AccountsQuotasModel>>>>
    getAccountQuotasByResourceIdByAccountByFolder(ProvideDictionary provideDictionary) {
        Function<ProvideDictionary, Set<String>> accountIdsFunc =
                (dict) -> new HashSet<>(dict.getAccountsById().keySet());
        return tableClient.usingSessionMonoRetryable(session ->
                session.usingTxMonoRetryable(TransactionMode.SERIALIZABLE_READ_WRITE, ts ->
                        storeService.getAccountQuotasByIds(ts, accountIdsFunc.apply(provideDictionary))
                                .map(provisions -> provisions.stream()
                                        .collect(Collectors.groupingBy(AccountsQuotasModel::getFolderId,
                                                Collectors.groupingBy(AccountsQuotasModel::getAccountId,
                                                        Collectors.toMap(AccountsQuotasModel::getResourceId,
                                                                Function.identity())))))));
    }

    private UpdateProvisionRequestDto prepareUpdateProvisionRequestDto(
            ProvideDictionary provideDictionary,
            Map<String, Map<String, Map<String, AccountsQuotasModel>>> provisionsMap,
            DeliveryAndProvideOperationListModel deliveryAndProvideOperationListModel) {
        String folderId = deliveryAndProvideOperationListModel.getFolderId();
        String accountId = deliveryAndProvideOperationListModel.getAccountId();
        long serviceId = deliveryAndProvideOperationListModel.getServiceId();

        UserModel userModel = provideDictionary.getAuthor();
        UserIdDto author = new UserIdDto(userModel.getPassportUid().orElse(null),
                userModel.getPassportLogin().orElse(null));

        AccountsQuotasOperationsModel accountsQuotasOperationsModel =
                provideDictionary.getAccountsQuotasOperationsByAccountId().get(accountId);
        String operationId = accountsQuotasOperationsModel.getOperationId();

        AccountModel accountModel = provideDictionary.getAccountsById().get(accountId);
        AccountSpaceModel accountSpaceModel = accountModel.getAccountsSpacesId()
                .map(id -> provideDictionary.getAccountSpacesById().get(id))
                .orElse(null);
        Optional<AccountsSpaceKeyRequestDto> accountsSpaceKeyO = createAccountsSpaceKey(accountSpaceModel,
                provideDictionary);

        List<ProvisionRequestDto> updatedProvisions = prepareUpdatedProvisions(provideDictionary, provisionsMap,
                deliveryAndProvideOperationListModel);

        List<KnownAccountProvisionsDto> knownProvisions = prepareKnownProvisions(
                accountModel,
                accountsQuotasOperationsModel,
                provideDictionary,
                provisionsMap
        );

        return new UpdateProvisionRequestDto(
                folderId,
                serviceId,
                updatedProvisions,
                knownProvisions,
                author,
                operationId,
                accountsSpaceKeyO.orElse(null)
        );
    }

    private List<ProvisionRequestDto> prepareUpdatedProvisions(
            ProvideDictionary provideDictionary,
            Map<String, Map<String, Map<String, AccountsQuotasModel>>> accountQuotasByResourceIdByAccountByFolder,
            DeliveryAndProvideOperationListModel deliveryAndProvideOperationListModel) {
        List<ProvisionRequestDto> updatedProvisions = new ArrayList<>();
        Set<String> updatedResourceIds = new HashSet<>();

        Map<String, ResourceModel> resourcesById = provideDictionary.getResourcesById();
        Map<String, UnitsEnsembleModel> unitsEnsemblesById = provideDictionary.getUnitsEnsemblesById();

        String accountId = deliveryAndProvideOperationListModel.getAccountId();
        String folderId = deliveryAndProvideOperationListModel.getFolderId();
        AccountsQuotasOperationsModel accountsQuotasOperationsModel =
                provideDictionary.getAccountsQuotasOperationsByAccountId().get(accountId);

        accountsQuotasOperationsModel.getRequestedChanges().getUpdatedProvisions()
                .ifPresent(provisions -> provisions.forEach(provision -> {
                    String resourceId = provision.getResourceId();
                    updatedResourceIds.add(resourceId);

                    ResourceModel resource = resourcesById.get(resourceId);
                    UnitsEnsembleModel unitsEnsemble =
                            unitsEnsemblesById.get(resource.getUnitsEnsembleId());
                    ResourceKeyRequestDto resourceKey = prepareResourceKey(resource, provideDictionary);
                    Tuple2<BigDecimal, UnitModel> valueAndUnit = Units.convertToApi(provision.getAmount(),
                            resource, unitsEnsemble);
                    updatedProvisions.add(new ProvisionRequestDto(resourceKey, valueAndUnit.getT1().longValue(),
                            valueAndUnit.getT2().getKey()));
                }));

        Map<String, AccountsQuotasModel> accountProvisions =
                accountQuotasByResourceIdByAccountByFolder.getOrDefault(folderId, Map.of())
                        .getOrDefault(accountId, Map.of());
        accountProvisions.forEach((resourceId, provision) -> {
            long providedAmount = provision.getProvidedQuota() != null ? provision.getProvidedQuota() : 0L;
            if (updatedResourceIds.contains(resourceId) || providedAmount == 0L) {
                return;
            }

            ResourceModel resource = resourcesById.get(resourceId);
            if (resource == null || !resource.isManaged() || resource.isReadOnly() || resource.isDeleted()) {
                return;
            }

            UnitsEnsembleModel unitsEnsemble = unitsEnsemblesById.get(resource.getUnitsEnsembleId());
            ResourceKeyRequestDto resourceKey = prepareResourceKey(resource, provideDictionary);
            Tuple2<BigDecimal, UnitModel> valueAndUnit = Units.convertToApi(providedAmount, resource, unitsEnsemble);
            updatedProvisions.add(new ProvisionRequestDto(resourceKey, valueAndUnit.getT1().longValue(),
                    valueAndUnit.getT2().getKey()));
        });

        return updatedProvisions;
    }

    private ResourceKeyRequestDto prepareResourceKey(ResourceModel resource,
                                                     ProvideDictionary provideDictionary) {
        Map<String, ResourceSegmentationModel> resourceSegmentationsById =
                provideDictionary.getResourceSegmentationsById();
        Map<String, ResourceSegmentModel> resourceSegmentsById = provideDictionary.getResourceSegmentsById();
        String resourceTypeKey = provideDictionary.getResourceTypesById().get(resource.getResourceTypeId()).getKey();
        Map<String, String> segmentationKeys = new HashMap<>();
        Map<String, String> segmentKeys = new HashMap<>();
        resource.getSegments().forEach(s -> {
            String segmentationId = s.getSegmentationId();
            String segmentId = s.getSegmentId();
            segmentationKeys.put(segmentationId, resourceSegmentationsById.get(segmentationId).getKey());
            segmentKeys.put(segmentId, resourceSegmentsById.get(segmentId).getKey());
        });
        Set<String> accountSpaceSegmentationIds = new HashSet<>();
        if (resource.getAccountsSpacesId() != null) {
            provideDictionary.getAccountSpacesById().get(resource.getAccountsSpacesId()).getSegments().forEach(s ->
                    accountSpaceSegmentationIds.add(s.getSegmentationId()));
        }
        List<SegmentKeyRequestDto> segmentation = new ArrayList<>();
        resource.getSegments().forEach(resourceSegment -> {
            if (accountSpaceSegmentationIds.contains(resourceSegment.getSegmentationId())) {
                return;
            }
            segmentation.add(new SegmentKeyRequestDto(
                    segmentationKeys.get(resourceSegment.getSegmentationId()),
                    segmentKeys.get(resourceSegment.getSegmentId())));
        });
        return new ResourceKeyRequestDto(resourceTypeKey, segmentation);
    }

    private Optional<AccountsSpaceKeyRequestDto> createAccountsSpaceKey(AccountSpaceModel accountsSpace,
                                                                        ProvideDictionary provideDictionary) {
        if (accountsSpace == null) {
            return Optional.empty();
        }

        Map<String, ResourceSegmentationModel> resourceSegmentationsById =
                provideDictionary.getResourceSegmentationsById();
        Map<String, ResourceSegmentModel> resourceSegmentsById = provideDictionary.getResourceSegmentsById();

        List<SegmentKeyRequestDto> segmentKeyRequestDtoList = accountsSpace.getSegments().stream()
                .map(s -> new SegmentKeyRequestDto(
                        resourceSegmentationsById.get(s.getSegmentationId())
                                .getKey(),
                        resourceSegmentsById.get(s.getSegmentId())
                                .getKey()
                ))
                .collect(Collectors.toList());

        return Optional.of(new AccountsSpaceKeyRequestDto(segmentKeyRequestDtoList));
    }

    private List<KnownAccountProvisionsDto> prepareKnownProvisions(
            AccountModel targetAccount,
            AccountsQuotasOperationsModel operation,
            ProvideDictionary provideDictionary, Map<String,
            Map<String, Map<String, AccountsQuotasModel>>> accountQuotasByResourceIdByAccountByFolder) {
        Map<String, ResourceModel> resourcesById = provideDictionary.getResourcesById();
        Map<String, UnitsEnsembleModel> unitsEnsemblesById = provideDictionary.getUnitsEnsemblesById();

        Set<String> targetResourceIds = new HashSet<>();
        operation.getRequestedChanges().getUpdatedProvisions().ifPresent(provisions -> provisions
                .forEach(provision -> targetResourceIds.add(provision.getResourceId())));

        Map<String, Map<String, AccountsQuotasModel>> accountQuotasByResourceIdByAccount =
                accountQuotasByResourceIdByAccountByFolder.getOrDefault(targetAccount.getFolderId(), Map.of());
        Map<String, AccountsQuotasModel> accountsQuotasByResourceId =
                accountQuotasByResourceIdByAccount.getOrDefault(targetAccount.getId(), Map.of());
        accountsQuotasByResourceId.forEach((resourceId, provision) -> {
            ResourceModel resource = resourcesById.get(resourceId);
            if (resource == null || !resource.isManaged() || resource.isReadOnly() || resource.isDeleted()) {
                return;
            }
            if ((provision.getProvidedQuota() != null ? provision.getProvidedQuota() : 0L) != 0L) {
                targetResourceIds.add(resourceId);
            }
        });

        List<KnownAccountProvisionsDto> result = new ArrayList<>();

        accountQuotasByResourceIdByAccount.keySet().stream()
                .map(accountId -> provideDictionary.getAccountsById().get(accountId))
                .filter(account -> filterAccount(targetAccount, account))
                .forEach(account -> {
                    Map<String, AccountsQuotasModel> provisionsByResourceId =
                            accountQuotasByResourceIdByAccount.getOrDefault(account.getId(), Map.of());
                    Set<String> collectedResourceIds = new HashSet<>();
                    List<KnownProvisionDto> knownProvisions = new ArrayList<>();

                    provisionsByResourceId.forEach((resourceId, provision) -> {
                        long providedAmount = provision.getProvidedQuota() != null ? provision.getProvidedQuota() : 0L;
                        collectedResourceIds.add(resourceId);
                        if (providedAmount == 0 && !targetResourceIds.contains(resourceId)) {
                            return;
                        }

                        ResourceModel resource = resourcesById.get(resourceId);
                        if (resource == null) {
                            return;
                        }
                        UnitsEnsembleModel unitsEnsemble = unitsEnsemblesById.get(resource.getUnitsEnsembleId());
                        ResourceKeyRequestDto resourceKey = prepareResourceKey(resource, provideDictionary);

                        Tuple2<BigDecimal, UnitModel> valueAndUnit = Units.convertToApi(providedAmount,
                                resource, unitsEnsemble);
                        knownProvisions.add(new KnownProvisionDto(resourceKey, valueAndUnit.getT1().longValue(),
                                valueAndUnit.getT2().getKey()));
                    });

                    Set<String> remainingResourceIds = Sets.difference(targetResourceIds, collectedResourceIds);

                    remainingResourceIds.forEach(resourceId -> {
                        ResourceModel resource = resourcesById.get(resourceId);
                        if (resource == null) {
                            return;
                        }
                        UnitsEnsembleModel unitsEnsemble = unitsEnsemblesById.get(resource.getUnitsEnsembleId());
                        ResourceKeyRequestDto resourceKey = prepareResourceKey(resource, provideDictionary);
                        Tuple2<BigDecimal, UnitModel> valueAndUnit = Units.convertToApi(0L, resource, unitsEnsemble);
                        knownProvisions.add(new KnownProvisionDto(resourceKey, valueAndUnit.getT1().longValue(),
                                valueAndUnit.getT2().getKey()));
                    });

                    result.add(new KnownAccountProvisionsDto(account.getOuterAccountIdInProvider(), knownProvisions));
                });

        return result;
    }

    private boolean filterAccount(AccountModel targetAccount,
                                  AccountModel accountModel) {
        return !accountModel.isDeleted()
                && targetAccount.getProviderId().equals(accountModel.getProviderId())
                && targetAccount.getAccountsSpacesId().equals(accountModel.getAccountsSpacesId());

    }

    private Mono<ProvideState> updateProvisionRequest(
            DeliveryAndProvideOperationWithIdModel deliveryAndProvideOperationList,
            UpdateProvisionRequestDto request,
            ProvideDictionary provideDictionary,
            AtomicInteger retryCounter,
            Locale locale) {
        String accountId = deliveryAndProvideOperationList.getAccountId();
        AccountModel accountModel = provideDictionary.getAccountsById().get(accountId);
        String outerAccountId = accountModel.getOuterAccountIdInProvider();
        ProviderModel provider = provideDictionary.getProvidersById().get(accountModel.getProviderId());
        return providersIntegrationService.updateProvision(outerAccountId, provider, request, locale).flatMap(r ->
                r.match(response -> {
                    if (response.isSuccess()) {
                        return updateOnSuccess(unpack(response), deliveryAndProvideOperationList, provideDictionary,
                                locale);
                    }
                    if (!isVersionConflict(response)
                            && !isRetryable(response)) {
                        return doUpdateProvisionRequestIncorrect(response, deliveryAndProvideOperationList,
                                provideDictionary, locale);
                    }
                    if (is5xx(response)) {
                        if (isRetryable(response) && canRetry(retryCounter)) {
                            return updateProvisionRequest(deliveryAndProvideOperationList, request, provideDictionary,
                                    retryCounter, locale);
                        } else {
                            operationsObservabilityService.observeOperationTransientFailure(
                                    provideDictionary.getAccountsQuotasOperationsByAccountId()
                                            .get(deliveryAndProvideOperationList.getAccountId()));
                            return Mono.just(new ProvideState(ProvideStateStatus.IN_PROGRESS));
                        }
                    } else {
                        if (isVersionConflict(response)) {
                            AtomicInteger getAccountRetryCounter = new AtomicInteger(MAX_RETRY_COUNT);
                            return doOnVersionConflict(deliveryAndProvideOperationList, provideDictionary,
                                    getAccountRetryCounter, locale);
                        } else {
                            if (isRetryable(response) && canRetry(retryCounter)) {
                                return updateProvisionRequest(deliveryAndProvideOperationList, request,
                                        provideDictionary, retryCounter, locale);
                            } else {
                                operationsObservabilityService.observeOperationTransientFailure(
                                        provideDictionary.getAccountsQuotasOperationsByAccountId()
                                                .get(deliveryAndProvideOperationList.getAccountId()));

                                return Mono.just(new ProvideState(ProvideStateStatus.IN_PROGRESS));
                            }
                        }
                    }
                }, e -> {
                    operationsObservabilityService.observeOperationTransientFailure(
                            provideDictionary.getAccountsQuotasOperationsByAccountId()
                                    .get(deliveryAndProvideOperationList.getAccountId()));

                    LOG.warn("Failed to send provision update request: {}", e);
                    return Mono.just(new ProvideState(FAILURE));
                })).onErrorResume(e -> {
            operationsObservabilityService.observeOperationTransientFailure(
                    provideDictionary.getAccountsQuotasOperationsByAccountId()
                            .get(deliveryAndProvideOperationList.getAccountId()));
            LOG.warn("Failed to send and process provision update request", e);
            return Mono.just(new ProvideState(FAILURE));
        });
    }

    private boolean canRetry(AtomicInteger retryCounter) {
        return retryCounter.decrementAndGet() >= 0;
    }

    private <R> boolean isVersionConflict(Response<R> response) {
        return validationService.isVersionConflict(response);
    }

    private <R> boolean isRetryable(Response<R> updateProvisionResponseDtoResponse) {
        return validationService.isRetryable(updateProvisionResponseDtoResponse);
    }

    private <R> boolean is5xx(Response<R> updateProvisionResponseDtoResponse) {
        return validationService.is5xx(updateProvisionResponseDtoResponse);
    }

    private <R> R unpack(
            Response<R> response
    ) {
        return response.<Optional<R>>match(
                (result, requestId) -> Optional.ofNullable(result),
                throwable -> Optional.empty(),
                (providerError, requestId) -> Optional.empty()
        ).orElseThrow();
    }

    private Mono<ProvideState> updateOnSuccess(
            UpdateProvisionResponseDto responseDto,
            DeliveryAndProvideOperationWithIdModel operationListModel,
            ProvideDictionary provideDictionary, Locale locale) {
        return tableClient.usingSessionMonoRetryable(session -> session.usingTxMonoRetryable(
                TransactionMode.SERIALIZABLE_READ_WRITE, ts -> validateReceivedUpdatedProvision(ts,
                        responseDto, operationListModel, provideDictionary, locale).flatMap(r -> r.match(provision ->
                        storeService.loadUpdateDictionaries(ts, operationListModel.getAccountId(), provideDictionary)
                                .flatMap(updateDictionary -> applySuccessProvide(ts, provision, updateDictionary,
                                        provideDictionary))
                                .onErrorResume(e -> {
                                    LOG.warn("Failed to save successful deliver provision update", e);
                                    return Mono.just(new ProvideState(FAILURE));
                                }), e -> {
                    LOG.warn("Failed to validate deliver provision response" + e.toString());
                    return Mono.just(new ProvideState(FAILURE));
                }))));
    }

    public Mono<Result<ValidatedReceivedUpdatedProvision>> validateReceivedUpdatedProvision(
            YdbTxSession ts, UpdateProvisionResponseDto responseDto,
            DeliveryAndProvideOperationWithIdModel operationListModel,
            ProvideDictionary provideDictionary,
            Locale locale) {
        return validationService.validateUpdatedProvision(responseDto, locale).andThenMono(provision -> {
            Set<String> userUidsToLoad = new HashSet<>();
            Set<String> userLoginsToLoad = new HashSet<>();
            Set<String> operationIdsToLoad = new HashSet<>();

            provision.getProvisions().forEach(p -> p.getLastUpdate().flatMap(ReceivedLastUpdate::getAuthor)
                    .ifPresent(a -> {
                        a.getPassportUid().ifPresent(userUidsToLoad::add);
                        a.getStaffLogin().ifPresent(userLoginsToLoad::add);
                    }));

            provision.getProvisions().forEach(p -> p.getLastUpdate().flatMap(ReceivedLastUpdate::getOperationId)
                    .ifPresent(operationIdsToLoad::add));

            return storeService.loadUsers(ts, userUidsToLoad, userLoginsToLoad).flatMap(users ->
                    storeService.loadOperations(ts, operationIdsToLoad).map(operations ->
                            validationService.validateReceivedUpdatedProvision(provision, operationListModel,
                                    provideDictionary, users, operations, locale)));
        });
    }

    private Mono<ProvideState> applySuccessProvide(YdbTxSession ts,
                                                   ValidatedReceivedUpdatedProvision provision,
                                                   UpdateDictionary updateDictionary,
                                                   ProvideDictionary provideDictionary) {
        AccountModel account = updateDictionary.getAccount();
        String accountId = account.getId();
        AccountsQuotasOperationsModel operation = updateDictionary.getAccountsQuotasOperations();
        String operationId = operation.getOperationId();
        String knownFolderId = updateDictionary.getFolder().getId();
        String knownProviderId = account.getProviderId();

        AccountsSettingsModel accountsSettings =
                provideDictionary.getProvidersById().get(knownProviderId).getAccountsSettings();
        boolean provisionsVersionedSeparately = accountsSettings.isPerProvisionVersionSupported();
        boolean accountAndProvisionsVersionedTogether = accountsSettings.isPerAccountVersionSupported()
                && !accountsSettings.isPerProvisionVersionSupported();

        List<OperationChangesModel.Provision> targetProvisions = operation.getRequestedChanges()
                .getUpdatedProvisions().orElse(List.of());
        List<OperationChangesModel.Provision> frozenProvisions = operation.getRequestedChanges()
                .getFrozenProvisions().orElse(List.of());
        Map<String, OperationChangesModel.Provision> frozenProvisionsByResourceId = frozenProvisions.stream()
                .collect(Collectors.toMap(OperationChangesModel.Provision::getResourceId, Function.identity()));

        Map<String, AccountsQuotasModel> knownProvisionsByResourceId = updateDictionary
                .getProvisionByResourceIdByAccount().getOrDefault(accountId, Map.of());
        Map<String, ValidatedReceivedProvision> receivedProvisionByResourceId = provision.getProvisions()
                .stream().collect(Collectors.toMap(v -> v.getResource().getId(), Function.identity()));

        List<AccountsQuotasModel> updatedProvisions = new ArrayList<>();
        List<QuotaModel> updatedQuotas = new ArrayList<>();

        Instant now = Instant.now();
        String opLogId = UUID.randomUUID().toString();

        FolderOperationLogModel.Builder opLogBuilder = prepareFolderOpLogBuilder(updateDictionary, now, operationId,
                knownFolderId, provideDictionary, opLogId);

        FolderModel updatedFolder = updateDictionary.getFolder().toBuilder()
                .setNextOpLogOrder(updateDictionary.getFolder().getNextOpLogOrder() + 1L)
                .build();

        Optional<AccountModel> updatedAccount = prepareUpdatedAccountOnRetry(updateDictionary, accountId,
                provision, accountAndProvisionsVersionedTogether, opLogBuilder);

        processUpdatedProvisions(updateDictionary, now, operationId, knownFolderId, accountId,
                knownProviderId, provisionsVersionedSeparately, targetProvisions, frozenProvisionsByResourceId,
                knownProvisionsByResourceId, receivedProvisionByResourceId, updatedProvisions, updatedQuotas,
                opLogBuilder);

        FolderOperationLogModel opLog = opLogBuilder.build();

        AccountsQuotasOperationsModel updatedOperation = new AccountsQuotasOperationsModel
                .Builder(operation)
                .setRequestStatus(AccountsQuotasOperationsModel.RequestStatus.OK)
                .setUpdateDateTime(now)
                .setOrders(OperationOrdersModel.builder(operation.getOrders())
                        .closeOrder(updateDictionary.getFolder().getNextOpLogOrder())
                        .build())
                .setErrorKind(null)
                .build();

        OperationInProgressModel operationInProgressModel = updateDictionary.getOperationInProgressModel();

        operationsObservabilityService.observeOperationFinished(updatedOperation);

        return storeService.upsertOperation(ts, updatedOperation)
                .then(Mono.defer(() -> storeService.deleteOperationInProgress(ts, operationInProgressModel)))
                .then(Mono.defer(() -> storeService.upsertQuotas(ts, updatedQuotas)))
                .then(Mono.defer(() -> storeService.upsertFolders(ts, List.of(updatedFolder))))
                .then(Mono.defer(() -> storeService.upsertOperationLog(ts, List.of(opLog))))
                .then(Mono.defer(() -> storeService.upsertProvisions(ts, updatedProvisions)))
                .then(Mono.defer(() -> updatedAccount.isPresent()
                        ? storeService.upsertAccount(ts, updatedAccount.get())
                        : Mono.empty()).thenReturn(new ProvideState(ProvideStateStatus.SUCCESS)));
    }

    private FolderOperationLogModel.Builder prepareFolderOpLogBuilder(
            UpdateDictionary updateDictionary, Instant now, String operationId, String knownFolderId,
            ProvideDictionary provideDictionary, String opLogId) {
        return FolderOperationLogModel.builder()
                .setTenantId(Tenants.DEFAULT_TENANT_ID)
                .setFolderId(knownFolderId)
                .setOperationDateTime(now)
                .setId(opLogId)
                .setProviderRequestId(null)
                .setOperationType(FolderOperationType.DELIVER_PROVIDE_QUOTAS_TO_ACCOUNT)
                .setAuthorUserId(provideDictionary.getAuthor().getId())
                .setAuthorUserUid(provideDictionary.getAuthor().getPassportUid().orElse(null))
                .setAuthorProviderId(null)
                .setSourceFolderOperationsLogId(null)
                .setDestinationFolderOperationsLogId(null)
                .setOldFolderFields(null)
                .setNewFolderFields(null)
                .setOldQuotas(new QuotasByResource(Map.of()))
                .setNewQuotas(new QuotasByResource(Map.of()))
                .setAccountsQuotasOperationsId(operationId)
                .setQuotasDemandsId(null)
                .setOperationPhase(OperationPhase.CLOSE)
                .setOrder(updateDictionary.getFolder().getNextOpLogOrder());
    }

    private Optional<AccountModel> prepareUpdatedAccountOnRetry(UpdateDictionary updateDictionary,
                                                                String accountId,
                                                                ValidatedReceivedUpdatedProvision provision,
                                                                boolean accountAndProvisionsVersionedTogether,
                                                                FolderOperationLogModel.Builder opLogBuilder) {
        if (accountAndProvisionsVersionedTogether && provision.getAccountVersion().isPresent()) {
            AccountModel.Builder updatedAccountBuilder = new AccountModel.Builder(updateDictionary.getAccount())
                    .setLastReceivedVersion(provision.getAccountVersion().get())
                    .setVersion(updateDictionary.getAccount().getVersion() + 1L);
            AccountHistoryModel.Builder oldAccountBuilder = AccountHistoryModel.builder()
                    .lastReceivedVersion(updateDictionary.getAccount().getLastReceivedVersion().orElse(null))
                    .version(updateDictionary.getAccount().getVersion());
            AccountHistoryModel.Builder newAccountBuilder = AccountHistoryModel.builder()
                    .lastReceivedVersion(provision.getAccountVersion().get())
                    .version(updateDictionary.getAccount().getVersion() + 1L);
            Optional<AccountModel> updatedAccount = Optional.of(updatedAccountBuilder.build());
            opLogBuilder.setOldAccounts(new AccountsHistoryModel(Map.of(accountId, oldAccountBuilder.build())));
            opLogBuilder.setNewAccounts(new AccountsHistoryModel(Map.of(accountId, newAccountBuilder.build())));
            return updatedAccount;
        } else {
            return Optional.empty();
        }
    }

    @SuppressWarnings("ParameterNumber")
    private void processUpdatedProvisions(
            UpdateDictionary updateDictionary,
            Instant now,
            String operationId,
            String knownFolderId,
            String knownAccountId,
            String knownProviderId,
            boolean provisionsVersionedSeparately,
            List<OperationChangesModel.Provision> targetProvisions,
            Map<String, OperationChangesModel.Provision> frozenProvisionsByResourceId,
            Map<String, AccountsQuotasModel> knownProvisionsByResourceId,
            Map<String, ValidatedReceivedProvision> receivedProvisionByResourceId,
            List<AccountsQuotasModel> updatedProvisions,
            List<QuotaModel> updatedQuotas,
            FolderOperationLogModel.Builder opLogBuilder) {
        Map<String, ProvisionHistoryModel> oldProvisionsByResourceId = new HashMap<>();
        Map<String, ProvisionHistoryModel> newProvisionsByResourceId = new HashMap<>();
        Map<String, ProvisionHistoryModel> actualProvisionsByResourceId = new HashMap<>();
        Map<String, Long> oldBalanceByResourceId = new HashMap<>();
        Map<String, Long> newBalanceByResourceId = new HashMap<>();
        processUpdatedProvisions(updateDictionary, now, operationId, knownFolderId, knownAccountId,
                knownProviderId, provisionsVersionedSeparately, targetProvisions, frozenProvisionsByResourceId,
                knownProvisionsByResourceId, receivedProvisionByResourceId, updatedProvisions, updatedQuotas,
                oldProvisionsByResourceId, newProvisionsByResourceId, actualProvisionsByResourceId,
                oldBalanceByResourceId, newBalanceByResourceId);
        if (!oldProvisionsByResourceId.isEmpty() || !newProvisionsByResourceId.isEmpty()) {
            opLogBuilder.setOldProvisions(new QuotasByAccount(Map.of(knownAccountId,
                    new ProvisionsByResource(oldProvisionsByResourceId))));
            opLogBuilder.setNewProvisions(new QuotasByAccount(Map.of(knownAccountId,
                    new ProvisionsByResource(newProvisionsByResourceId))));
        } else {
            opLogBuilder.setOldProvisions(new QuotasByAccount(Map.of()));
            opLogBuilder.setNewProvisions(new QuotasByAccount(Map.of()));
        }
        if (!actualProvisionsByResourceId.isEmpty()) {
            opLogBuilder.setActuallyAppliedProvisions(new QuotasByAccount(Map.of(knownAccountId,
                    new ProvisionsByResource(actualProvisionsByResourceId))));
        }
        if (!oldBalanceByResourceId.isEmpty() || !newBalanceByResourceId.isEmpty()) {
            opLogBuilder.setOldBalance(new QuotasByResource(oldBalanceByResourceId));
            opLogBuilder.setNewBalance(new QuotasByResource(newBalanceByResourceId));
        } else {
            opLogBuilder.setOldBalance(new QuotasByResource(Map.of()));
            opLogBuilder.setNewBalance(new QuotasByResource(Map.of()));
        }
    }

    @SuppressWarnings("ParameterNumber")
    private void processUpdatedProvisions(
            UpdateDictionary updateDictionary,
            Instant now,
            String operationId,
            String knownFolderId,
            String knownAccountId,
            String knownProviderId,
            boolean provisionsVersionedSeparately,
            List<OperationChangesModel.Provision> targetProvisions,
            Map<String, OperationChangesModel.Provision> frozenProvisionsByResourceId,
            Map<String, AccountsQuotasModel> knownProvisionsByResourceId,
            Map<String, ValidatedReceivedProvision> receivedProvisionByResourceId,
            List<AccountsQuotasModel> updatedProvisions,
            List<QuotaModel> updatedQuotas,
            Map<String, ProvisionHistoryModel> oldProvisionsByResourceId,
            Map<String, ProvisionHistoryModel> newProvisionsByResourceId,
            Map<String, ProvisionHistoryModel> actualProvisionsByResourceId,
            Map<String, Long> oldBalanceByResourceId,
            Map<String, Long> newBalanceByResourceId) {
        targetProvisions.forEach(targetProvision -> {
            String resourceId = targetProvision.getResourceId();
            AccountsQuotasModel knownProvision = knownProvisionsByResourceId.get(resourceId);
            QuotaModel knownQuota = updateDictionary.getQuotasByResourceId().get(resourceId);
            OperationChangesModel.Provision frozenProvision = frozenProvisionsByResourceId.get(resourceId);
            ValidatedReceivedProvision receivedProvision = receivedProvisionByResourceId.get(resourceId);
            Long receivedVersion = receivedProvision != null ? receivedProvision.getQuotaVersion().orElse(null) : null;
            Long knownVersion = knownProvision != null
                    ? knownProvision.getLastReceivedProvisionVersion().orElse(null) : null;
            long knownProvidedAmount = knownProvision != null ? (knownProvision.getProvidedQuota() != null
                    ? knownProvision.getProvidedQuota() : 0L) : 0L;
            long targetProvidedAmount = targetProvision.getAmount();
            long receivedProvidedAmount = receivedProvision != null ? receivedProvision.getProvidedAmount() : 0L;
            long receivedAllocatedAmount = receivedProvision != null ? receivedProvision.getAllocatedAmount() : 0L;
            long targetFrozenAmount = frozenProvision != null ? frozenProvision.getAmount() : 0L;
            long knownFrozenAmount = knownQuota != null ? knownQuota.getFrozenQuota() : 0L;
            long knownBalance = knownQuota != null
                    ? (knownQuota.getBalance() != null ? knownQuota.getBalance() : 0L) : 0L;
            processUpdatedProvidedAmounts(now, operationId, knownFolderId, knownAccountId, knownProviderId,
                    provisionsVersionedSeparately, updatedProvisions, oldProvisionsByResourceId,
                    newProvisionsByResourceId, actualProvisionsByResourceId, resourceId, knownProvision,
                    receivedVersion, knownVersion, knownProvidedAmount, targetProvidedAmount, receivedProvidedAmount,
                    receivedAllocatedAmount);
            processUpdatedBalance(updateDictionary, operationId, knownFolderId, knownAccountId, knownProviderId,
                    updatedQuotas, oldBalanceByResourceId, newBalanceByResourceId, resourceId, knownQuota,
                    receivedProvidedAmount, targetFrozenAmount, knownFrozenAmount, knownBalance);
        });
    }

    @SuppressWarnings("ParameterNumber")
    private void processUpdatedProvidedAmounts(
            Instant now,
            String operationId,
            String knownFolderId,
            String knownAccountId,
            String knownProviderId,
            boolean provisionsVersionedSeparately,
            List<AccountsQuotasModel> updatedProvisions,
            Map<String, ProvisionHistoryModel> oldProvisionsByResourceId,
            Map<String, ProvisionHistoryModel> newProvisionsByResourceId,
            Map<String, ProvisionHistoryModel> actualProvisionsByResourceId,
            String resourceId,
            AccountsQuotasModel knownProvision,
            Long receivedVersion,
            Long knownVersion,
            long knownProvidedAmount,
            long targetProvidedAmount,
            long receivedProvidedAmount,
            long receivedAllocatedAmount) {
        if (knownProvision != null) {
            AccountsQuotasModel updatedProvision = new AccountsQuotasModel.Builder(knownProvision)
                    .setProvidedQuota(receivedProvidedAmount)
                    .setAllocatedQuota(receivedAllocatedAmount)
                    .setLastProvisionUpdate(now)
                    .setLatestSuccessfulProvisionOperationId(operationId)
                    .setLastReceivedProvisionVersion(provisionsVersionedSeparately
                            ? receivedVersion
                            : knownProvision.getLastReceivedProvisionVersion().orElse(null))
                    .build();
            updatedProvisions.add(updatedProvision);
        } else {
            AccountsQuotasModel newProvision = new AccountsQuotasModel.Builder()
                    .setTenantId(Tenants.DEFAULT_TENANT_ID)
                    .setAccountId(knownAccountId)
                    .setResourceId(resourceId)
                    .setProvidedQuota(receivedProvidedAmount)
                    .setAllocatedQuota(receivedAllocatedAmount)
                    .setFolderId(knownFolderId)
                    .setProviderId(knownProviderId)
                    .setLastProvisionUpdate(now)
                    .setLatestSuccessfulProvisionOperationId(operationId)
                    .setLastReceivedProvisionVersion(provisionsVersionedSeparately
                            ? receivedVersion : null)
                    .build();
            updatedProvisions.add(newProvision);
        }
        if (receivedProvidedAmount != knownProvidedAmount || !Objects.equals(receivedVersion, knownVersion)) {
            oldProvisionsByResourceId.put(resourceId, new ProvisionHistoryModel(knownProvidedAmount,
                    knownVersion));
            actualProvisionsByResourceId.put(resourceId, new ProvisionHistoryModel(receivedProvidedAmount,
                    receivedVersion));
        }
        if (targetProvidedAmount != knownProvidedAmount) {
            oldProvisionsByResourceId.put(resourceId, new ProvisionHistoryModel(knownProvidedAmount,
                    knownVersion));
            newProvisionsByResourceId.put(resourceId, new ProvisionHistoryModel(targetProvidedAmount,
                    targetProvidedAmount == receivedProvidedAmount ? receivedVersion : knownVersion));
        }
    }

    @SuppressWarnings("ParameterNumber")
    private void processUpdatedBalance(
            UpdateDictionary updateDictionary,
            String operationId,
            String knownFolderId,
            String knownAccountId,
            String knownProviderId,
            List<QuotaModel> updatedQuotas,
            Map<String, Long> oldBalanceByResourceId,
            Map<String, Long> newBalanceByResourceId,
            String resourceId,
            QuotaModel knownQuota,
            long receivedProvidedAmount,
            long targetFrozenAmount,
            long knownFrozenAmount,
            long knownBalance) {
        long updatedFrozenAmount = knownQuota != null ? knownQuota.getFrozenQuota() : 0L;
        if (targetFrozenAmount != 0L) {
            if (targetFrozenAmount <= knownFrozenAmount && knownQuota != null) {
                updatedFrozenAmount = knownFrozenAmount - targetFrozenAmount;
            } else {
                LOG.error("Frozen quota {} for resource {} in folder {} is less then value {} stored for "
                                + "deliver and provide operation {}",
                        knownFrozenAmount, resourceId, knownFolderId, targetFrozenAmount, operationId);
            }
        }
        long updatedBalance = knownQuota != null ? knownQuota.getQuota() : 0L;
        for (Map<String, AccountsQuotasModel> provisionsByResourceId :
                updateDictionary.getProvisionByResourceIdByAccount().values()) {
            AccountsQuotasModel currentProvision = provisionsByResourceId.get(resourceId);
            long currentProvidedAmount = currentProvision != null
                    ? (currentProvision.getProvidedQuota() != null ? currentProvision.getProvidedQuota() : 0L) : 0L;
            if (currentProvidedAmount == 0L) {
                continue;
            }
            if (currentProvision.getAccountId().equals(knownAccountId)) {
                continue;
            }
            Optional<Long> balanceUpdateO = Units.subtract(updatedBalance, currentProvidedAmount);
            if (balanceUpdateO.isPresent()) {
                updatedBalance = balanceUpdateO.get();
            } else {
                LOG.error("Underflow while updating balance of resource {} in folder {} for operation {}",
                        resourceId, knownFolderId, operationId);
            }
        }
        Optional<Long> balanceUpdateO = Units.subtract(updatedBalance, receivedProvidedAmount);
        if (balanceUpdateO.isPresent()) {
            updatedBalance = balanceUpdateO.get();
        } else {
            LOG.error("Underflow while updating balance of resource {} in folder {} for operation {}",
                    resourceId, knownFolderId, operationId);
        }
        Optional<Long> freezeUpdateO = Units.subtract(updatedBalance, updatedFrozenAmount);
        if (freezeUpdateO.isPresent()) {
            updatedBalance = freezeUpdateO.get();
        } else {
            LOG.error("Underflow while updating balance of resource {} in folder {} for operation {}",
                    resourceId, knownFolderId, operationId);
        }
        if (knownBalance != updatedBalance || knownFrozenAmount != updatedFrozenAmount) {
            if (knownQuota != null) {
                QuotaModel updatedQuota = QuotaModel.builder(knownQuota)
                        .frozenQuota(updatedFrozenAmount)
                        .balance(updatedBalance)
                        .build();
                updatedQuotas.add(updatedQuota);
            } else {
                QuotaModel newQuota = QuotaModel.builder()
                        .tenantId(Tenants.DEFAULT_TENANT_ID)
                        .folderId(knownFolderId)
                        .providerId(knownProviderId)
                        .resourceId(resourceId)
                        .quota(0L)
                        .balance(updatedBalance)
                        .frozenQuota(updatedFrozenAmount)
                        .build();
                updatedQuotas.add(newQuota);
            }
        }
        if (knownBalance != updatedBalance) {
            oldBalanceByResourceId.put(resourceId, knownBalance);
            newBalanceByResourceId.put(resourceId, updatedBalance);
        }
    }

    private Mono<ProvideState> doUpdateProvisionRequestIncorrect(
            Response<UpdateProvisionResponseDto> response,
            DeliveryAndProvideOperationWithIdModel deliveryAndProvideOperationList,
            ProvideDictionary provideDictionary, Locale locale) {
        String errorMessage = validationService.toErrorMessage(response)
                .orElseGet(() -> messages.getMessage("errors.bad.request", null, locale));

        return updateProvisionRequestIncorrect(errorMessage, deliveryAndProvideOperationList, provideDictionary)
                .onErrorResume(e -> {
                    LOG.warn("Failed to save failed deliver and provision update", e);
                    return Mono.just(new ProvideState(FAILURE));
                });
    }

    private Mono<ProvideState> updateProvisionRequestIncorrect(
            String errorMessage, DeliveryAndProvideOperationWithIdModel listModel,
            ProvideDictionary provideDict) {
        return tableClient.usingSessionMonoRetryable(session -> session.usingTxMonoRetryable(
                TransactionMode.SERIALIZABLE_READ_WRITE, ts -> {
                    String accountId = listModel.getAccountId();
                    return storeService.loadUpdateFailureDictionaries(ts, provideDict, accountId).flatMap(dict -> {
                        AccountsQuotasOperationsModel operation = dict.getAccountsQuotasOperations();
                        Instant now = Instant.now();
                        AccountsQuotasOperationsModel updatedOperation = new AccountsQuotasOperationsModel
                                .Builder(operation)
                                .setRequestStatus(AccountsQuotasOperationsModel.RequestStatus.ERROR)
                                .setUpdateDateTime(now)
                                .setErrorMessage(errorMessage)
                                .setErrorKind(OperationErrorKind.INVALID_ARGUMENT)
                                .build();

                        List<QuotaModel> quotas = dict.getQuotas();
                        List<QuotaModel> unfreezedQuotas = unfreezeQuotas(operation, quotas);

                        OperationInProgressModel opInProg = dict.getOperationInProgress();

                        operationsObservabilityService.observeOperationFinished(updatedOperation);

                        return storeService.upsertQuotas(ts, unfreezedQuotas)
                                .then(Mono.defer(() -> storeService.finishOperation(ts, opInProg, updatedOperation)))
                                .thenReturn(new ProvideState(FAILURE));
                    });
                }));
    }

    private Mono<ProvideState> updateProvisionRequestIncorrect(YdbTxSession ts, String errorMessage,
                                                               UpdateDictionary updateDictionary) {
        AccountsQuotasOperationsModel operation = updateDictionary.getAccountsQuotasOperations();
        Instant now = Instant.now();
        AccountsQuotasOperationsModel updatedOperation = new AccountsQuotasOperationsModel
                .Builder(operation)
                .setRequestStatus(AccountsQuotasOperationsModel.RequestStatus.ERROR)
                .setUpdateDateTime(now)
                .setErrorMessage(errorMessage)
                .setErrorKind(OperationErrorKind.FAILED_PRECONDITION)
                .build();

        List<QuotaModel> quotas = new ArrayList<>(updateDictionary.getQuotasByResourceId().values());
        List<QuotaModel> unfreezedQuotas = unfreezeQuotas(operation, quotas);

        OperationInProgressModel opInProg = updateDictionary.getOperationInProgressModel();

        operationsObservabilityService.observeOperationFinished(updatedOperation);

        return storeService.upsertQuotas(ts, unfreezedQuotas)
                .then(Mono.defer(() -> storeService.finishOperation(ts, opInProg, updatedOperation)))
                .thenReturn(new ProvideState(FAILURE));
    }

    private List<QuotaModel> unfreezeQuotas(AccountsQuotasOperationsModel operation, List<QuotaModel> quotas) {
        List<QuotaModel> unfreezedQuotas = new ArrayList<>();
        List<OperationChangesModel.Provision> frozenProvisions = operation.getRequestedChanges()
                .getFrozenProvisions().orElse(List.of());
        Map<String, OperationChangesModel.Provision> frozenByResource = frozenProvisions.stream()
                .collect(Collectors.toMap(OperationChangesModel.Provision::getResourceId, Function.identity()));

        quotas.forEach(quota -> {
            if (!frozenByResource.containsKey(quota.getResourceId())) {
                return;
            }

            OperationChangesModel.Provision frozen = frozenByResource.get(quota.getResourceId());
            if (frozen.getAmount() == 0L) {
                return;
            }

            if (frozen.getAmount() <= quota.getFrozenQuota()) {
                if (Units.add(quota.getBalance(), frozen.getAmount()).isEmpty()) {
                    LOG.error("Balance overflow ({} frozen + {} balance) while unfreezing "
                                    + "quota for resource {} in folder {}, operation {}", frozen.getAmount(),
                            quota.getBalance(), quota.getResourceId(), quota.getFolderId(), operation.getOperationId());
                    return;
                }
                QuotaModel unfreezedQuota = QuotaModel.builder(quota)
                        .frozenQuota(quota.getFrozenQuota() - frozen.getAmount())
                        .balance(quota.getBalance() + frozen.getAmount())
                        .build();
                unfreezedQuotas.add(unfreezedQuota);
            } else {
                LOG.error("Frozen quota {} for resource {} in folder {} is less then value {} stored for operation {}",
                        quota.getFrozenQuota(), quota.getResourceId(), quota.getFolderId(), frozen.getAmount(),
                        operation.getOperationId());
            }
        });

        return unfreezedQuotas;
    }

    private Mono<ProvideState> doOnVersionConflict(DeliveryAndProvideOperationWithIdModel listModel,
                                                   ProvideDictionary provideDictionary,
                                                   AtomicInteger retryCounter, Locale locale) {
        String accountId = listModel.getAccountId();
        AccountModel account = provideDictionary.getAccountsById().get(accountId);
        ProviderModel provider = provideDictionary.getProvidersById().get(account.getProviderId());
        GetAccountRequestDto request = getGetAccountRequestDto(account, provider, provideDictionary);
        return providersIntegrationService.getAccount(account.getOuterAccountIdInProvider(), provider, request, locale)
                .flatMap(r -> r.match(accountDtoResponse -> {
                    if (accountDtoResponse.isSuccess()) {
                        return onSuccessProvidersIntegrationServiceGetAccount(accountDtoResponse,
                                listModel, provideDictionary, locale);
                    } else {
                        return onErrorProvidersIntegrationServiceGetAccount(accountDtoResponse,
                                listModel, provideDictionary, retryCounter, locale);
                    }
                }, e -> {
                    operationsObservabilityService.observeOperationTransientFailure(
                            provideDictionary.getAccountsQuotasOperationsByAccountId()
                                    .get(listModel.getAccountId()));

                    LOG.warn("Failed to resolve version conflict for deliver and provision update: {}", e);
                    return Mono.just(new ProvideState(FAILURE));
                })).onErrorResume(e -> {
                    operationsObservabilityService.observeOperationTransientFailure(
                            provideDictionary.getAccountsQuotasOperationsByAccountId()
                                    .get(listModel.getAccountId()));

                    LOG.warn("Failed to resolve version conflict for deliver and provision update", e);
                    return Mono.just(new ProvideState(FAILURE));
                });
    }

    private GetAccountRequestDto getGetAccountRequestDto(AccountModel account,
                                                         ProviderModel provider,
                                                         ProvideDictionary provideDictionary) {
        Boolean includeDeleted = provider.getAccountsSettings().isDeleteSupported()
                && provider.getAccountsSettings().isSoftDeleteSupported() ? true : null;

        AccountSpaceModel accountSpace = account.getAccountsSpacesId()
                .map(accountSpaceId -> provideDictionary.getAccountSpacesById().get(accountSpaceId))
                .orElse(null);
        Optional<AccountsSpaceKeyRequestDto> accountsSpaceKeyO = createAccountsSpaceKey(accountSpace,
                provideDictionary);

        FolderModel folder = provideDictionary.getFoldersById().get(account.getFolderId());

        return new GetAccountRequestDto(true, includeDeleted,
                folder.getId(), folder.getServiceId(), accountsSpaceKeyO.orElse(null));
    }

    private Mono<ProvideState> onErrorProvidersIntegrationServiceGetAccount(
            Response<AccountDto> accountDtoResponse,
            DeliveryAndProvideOperationWithIdModel listModel,
            ProvideDictionary provideDictionary,
            AtomicInteger retryCounter,
            Locale locale) {
        if ((!is5xx(accountDtoResponse)
                && !isRetryable(accountDtoResponse))
                || !canRetry(retryCounter)) {
            operationsObservabilityService.observeOperationTransientFailure(
                    provideDictionary.getAccountsQuotasOperationsByAccountId()
                            .get(listModel.getAccountId()));
            return Mono.just(new ProvideState(FAILURE));
        } else {
            return doOnVersionConflict(listModel, provideDictionary, retryCounter, locale);
        }
    }

    private Mono<ProvideState> onSuccessProvidersIntegrationServiceGetAccount(
            Response<AccountDto> accountDtoResponse,
            DeliveryAndProvideOperationWithIdModel listModel,
            ProvideDictionary provideDict, Locale locale) {
        return validationService.validateAccount(unpack(accountDtoResponse), locale).match(account ->
                tableClient.usingSessionMonoRetryable(session -> session.usingTxMonoRetryable(
                        TransactionMode.SERIALIZABLE_READ_WRITE, ts -> validateReceivedAccount(ts,
                                account, listModel, provideDict, locale).flatMap(r -> r.match(validatedAccount -> {
                            String accountId = listModel.getAccountId();
                            return storeService.loadUpdateDictionaries(ts, accountId, provideDict).flatMap(dict -> {
                                if (validationService.isProvisionChangeApplied(validatedAccount, dict, provideDict)) {
                                    return applySuccessProvide(ts, dict, provideDict, validatedAccount);
                                } else {
                                    String errorMessage = validationService.toErrorMessage(accountDtoResponse)
                                            .orElse(null);
                                    return updateProvisionRequestIncorrect(ts, errorMessage, dict)
                                            .onErrorResume(e -> {
                                                operationsObservabilityService.observeOperationTransientFailure(
                                                        provideDict.getAccountsQuotasOperationsByAccountId()
                                                                .get(listModel.getAccountId()));
                                                LOG.warn("Failed to save failed deliver and provision update", e);
                                                return Mono.just(new ProvideState(FAILURE));
                                            });
                                }
                            });
                        }, e -> {
                            operationsObservabilityService.observeOperationTransientFailure(
                                    provideDict.getAccountsQuotasOperationsByAccountId()
                                            .get(listModel.getAccountId()));
                            LOG.warn("Failed to process account refresh response after provision update for " +
                                    "deliver and provision update operation {}", e);
                            return Mono.just(new ProvideState(FAILURE));
                        })))), e -> {
            operationsObservabilityService.observeOperationTransientFailure(
                    provideDict.getAccountsQuotasOperationsByAccountId()
                            .get(listModel.getAccountId()));
            LOG.warn("Failed to validate AccountDto deliver and provision update {}", e);
            return Mono.just(new ProvideState(FAILURE));
        });
    }

    public Mono<Result<ValidatedReceivedAccount>> validateReceivedAccount(
            YdbTxSession ts, ReceivedAccount account, DeliveryAndProvideOperationWithIdModel listModel,
            ProvideDictionary provideDictionary, Locale locale) {
        String externalAccountIdToLoad = account.getAccountId();
        String folderIdToLoad = account.getFolderId();
        Set<String> userUidsToLoad = new HashSet<>();
        Set<String> userLoginsToLoad = new HashSet<>();
        Set<String> operationIdsToLoad = new HashSet<>();
        account.getLastUpdate().flatMap(ReceivedLastUpdate::getAuthor).ifPresent(a -> {
            a.getPassportUid().ifPresent(userUidsToLoad::add);
            a.getStaffLogin().ifPresent(userLoginsToLoad::add);
        });
        account.getProvisions().forEach(p -> p.getLastUpdate().flatMap(ReceivedLastUpdate::getAuthor).ifPresent(a -> {
            a.getPassportUid().ifPresent(userUidsToLoad::add);
            a.getStaffLogin().ifPresent(userLoginsToLoad::add);
        }));
        account.getLastUpdate().flatMap(ReceivedLastUpdate::getOperationId).ifPresent(operationIdsToLoad::add);
        account.getProvisions().forEach(p -> p.getLastUpdate().flatMap(ReceivedLastUpdate::getOperationId)
                .ifPresent(operationIdsToLoad::add));

        String accountId = listModel.getAccountId();
        AccountModel accountModel = provideDictionary.getAccountsById().get(accountId);
        String accountsSpaceId = accountModel.getAccountsSpacesId().orElse(null);
        return storeService.loadUsers(ts, userUidsToLoad, userLoginsToLoad).flatMap(users ->
                storeService.loadOperations(ts, operationIdsToLoad).flatMap(operations ->
                        storeService.getFoldersByIds(ts, Set.of(folderIdToLoad)).flatMap(folders ->
                                storeService.loadAccountByExternalId(ts, accountModel.getProviderId(),
                                                accountsSpaceId, externalAccountIdToLoad)
                                        .map(accountO -> {
                                            FolderModel folderModel = folders.stream()
                                                    .filter(folder -> folder.getId().equals(folderIdToLoad))
                                                    .findFirst()
                                                    .orElse(null);
                                            return validationService.validateReceivedAccount(account, users,
                                                    operations, folderModel, accountO.orElse(null),
                                                    listModel, provideDictionary, locale);
                                        }))));
    }

    private Mono<ProvideState> applySuccessProvide(YdbTxSession ts, UpdateDictionary updateDictionary,
                                                   ProvideDictionary provideDictionary,
                                                   ValidatedReceivedAccount receivedAccount) {
        AccountModel account = updateDictionary.getAccount();
        String accountId = account.getId();
        AccountsQuotasOperationsModel operation = updateDictionary.getAccountsQuotasOperations();
        String operationId = operation.getOperationId();
        String knownFolderId = updateDictionary.getFolder().getId();
        String knownAccountId = account.getProviderId();
        String knownProviderId = account.getProviderId();

        AccountsSettingsModel accountsSettings = provideDictionary.getProvidersById().get(knownProviderId)
                .getAccountsSettings();
        boolean provisionsVersionedSeparately = accountsSettings.isPerProvisionVersionSupported();
        boolean accountAndProvisionsVersionedTogether = accountsSettings.isPerAccountVersionSupported()
                && !accountsSettings.isPerProvisionVersionSupported();

        List<OperationChangesModel.Provision> targetProvisions = operation.getRequestedChanges()
                .getUpdatedProvisions().orElse(List.of());
        List<OperationChangesModel.Provision> frozenProvisions = operation.getRequestedChanges()
                .getFrozenProvisions().orElse(List.of());
        Map<String, OperationChangesModel.Provision> frozenProvisionsByResourceId = frozenProvisions.stream()
                .collect(Collectors.toMap(OperationChangesModel.Provision::getResourceId, Function.identity()));

        Map<String, AccountsQuotasModel> knownProvisionsByResourceId = updateDictionary
                .getProvisionByResourceIdByAccount().getOrDefault(accountId, Map.of());
        Map<String, ValidatedReceivedProvision> receivedProvisionByResourceId = receivedAccount.getProvisions()
                .stream().collect(Collectors.toMap(v -> v.getResource().getId(), Function.identity()));

        List<AccountsQuotasModel> updatedProvisions = new ArrayList<>();
        List<QuotaModel> updatedQuotas = new ArrayList<>();

        Instant now = Instant.now();
        String opLogId = UUID.randomUUID().toString();

        FolderOperationLogModel.Builder opLogBuilder = prepareFolderOpLogBuilder(updateDictionary, now, operationId,
                knownFolderId, provideDictionary, opLogId);
        FolderModel updatedFolder = updateDictionary.getFolder().toBuilder()
                .setNextOpLogOrder(updateDictionary.getFolder().getNextOpLogOrder() + 1L)
                .build();

        Optional<AccountModel> updatedAccount = prepareUpdatedAccountOnRefresh(updateDictionary, knownAccountId,
                receivedAccount, accountAndProvisionsVersionedTogether, opLogBuilder, accountsSettings);

        processUpdatedProvisions(updateDictionary, now, operationId, knownFolderId, knownAccountId,
                knownProviderId, provisionsVersionedSeparately, targetProvisions, frozenProvisionsByResourceId,
                knownProvisionsByResourceId, receivedProvisionByResourceId, updatedProvisions, updatedQuotas,
                opLogBuilder);

        FolderOperationLogModel opLog = opLogBuilder.build();

        AccountsQuotasOperationsModel updatedOperation = new AccountsQuotasOperationsModel
                .Builder(operation)
                .setRequestStatus(AccountsQuotasOperationsModel.RequestStatus.OK)
                .setUpdateDateTime(now)
                .setOrders(OperationOrdersModel.builder(operation.getOrders())
                        .closeOrder(updateDictionary.getFolder().getNextOpLogOrder())
                        .build())
                .setErrorKind(null)
                .build();

        OperationInProgressModel operationInProgressModel = updateDictionary.getOperationInProgressModel();

        operationsObservabilityService.observeOperationFinished(updatedOperation);

        return storeService.upsertOperation(ts, updatedOperation)
                .then(Mono.defer(() -> storeService.deleteOperationInProgress(ts, operationInProgressModel)))
                .then(Mono.defer(() -> storeService.upsertQuotas(ts, updatedQuotas)))
                .then(Mono.defer(() -> storeService.upsertFolders(ts, List.of(updatedFolder))))
                .then(Mono.defer(() -> storeService.upsertOperationLog(ts, List.of(opLog))))
                .then(Mono.defer(() -> storeService.upsertProvisions(ts, updatedProvisions)))
                .then(Mono.defer(() -> updatedAccount.isPresent()
                        ? storeService.upsertAccount(ts, updatedAccount.get())
                        : Mono.empty()).thenReturn(new ProvideState(ProvideStateStatus.SUCCESS)));
    }

    private Optional<AccountModel> prepareUpdatedAccountOnRefresh(
            UpdateDictionary updateDictionary, String knownAccountId, ValidatedReceivedAccount receivedAccount,
            boolean accountAndProvisionsVersionedTogether, FolderOperationLogModel.Builder opLogBuilder,
            AccountsSettingsModel accountsSettings) {
        if (accountAndProvisionsVersionedTogether && receivedAccount.getAccountVersion().isPresent()) {
            AccountModel account = updateDictionary.getAccount();
            AccountModel.Builder updatedAccountBuilder = new AccountModel.Builder(account)
                    .setLastReceivedVersion(receivedAccount.getAccountVersion().get())
                    .setVersion(account.getVersion() + 1L);
            AccountHistoryModel.Builder oldAccountBuilder = AccountHistoryModel.builder()
                    .lastReceivedVersion(account.getLastReceivedVersion().orElse(null))
                    .version(account.getVersion());
            AccountHistoryModel.Builder newAccountBuilder = AccountHistoryModel.builder()
                    .lastReceivedVersion(receivedAccount.getAccountVersion().get())
                    .version(account.getVersion() + 1L);
            if (accountsSettings.isDisplayNameSupported()
                    && !account.getDisplayName().equals(receivedAccount.getDisplayName())) {
                updatedAccountBuilder.setDisplayName(receivedAccount.getDisplayName().orElse(null));
                oldAccountBuilder.displayName(account.getDisplayName().orElse(null));
                newAccountBuilder.displayName(receivedAccount.getDisplayName().orElse(null));
            }
            if (accountsSettings.isKeySupported()
                    && !account.getOuterAccountKeyInProvider().equals(receivedAccount.getKey())) {
                updatedAccountBuilder.setOuterAccountKeyInProvider(receivedAccount.getKey().orElse(null));
                oldAccountBuilder.outerAccountKeyInProvider(account
                        .getOuterAccountKeyInProvider().orElse(null));
                newAccountBuilder.outerAccountKeyInProvider(receivedAccount.getKey().orElse(null));
            }
            Optional<AccountModel> updatedAccount = Optional.of(updatedAccountBuilder.build());
            opLogBuilder.setOldAccounts(new AccountsHistoryModel(Map.of(knownAccountId, oldAccountBuilder.build())));
            opLogBuilder.setNewAccounts(new AccountsHistoryModel(Map.of(knownAccountId, newAccountBuilder.build())));
            return updatedAccount;
        } else {
            return Optional.empty();
        }
    }
}
