package ru.yandex.intranet.d.services.delivery.provide;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.yandex.ydb.table.transaction.TransactionMode;
import org.apache.commons.lang3.builder.ReflectionToStringBuilder;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuples;

import ru.yandex.intranet.d.dao.Tenants;
import ru.yandex.intranet.d.dao.accounts.AccountsDao;
import ru.yandex.intranet.d.dao.accounts.AccountsQuotasOperationsDao;
import ru.yandex.intranet.d.datasource.model.YdbTableClient;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.i18n.Locales;
import ru.yandex.intranet.d.loaders.providers.AccountSpacesLoader;
import ru.yandex.intranet.d.loaders.providers.ProvidersLoader;
import ru.yandex.intranet.d.loaders.resources.ResourcesLoader;
import ru.yandex.intranet.d.loaders.resources.segmentations.ResourceSegmentationsLoader;
import ru.yandex.intranet.d.loaders.resources.segments.ResourceSegmentsLoader;
import ru.yandex.intranet.d.loaders.units.UnitsEnsemblesLoader;
import ru.yandex.intranet.d.model.WithTenant;
import ru.yandex.intranet.d.model.accounts.AccountModel;
import ru.yandex.intranet.d.model.accounts.AccountSpaceModel;
import ru.yandex.intranet.d.model.accounts.AccountsQuotasOperationsModel;
import ru.yandex.intranet.d.model.accounts.OperationChangesModel;
import ru.yandex.intranet.d.model.delivery.DeliverableFolderOperationModel;
import ru.yandex.intranet.d.model.delivery.DeliverableMetaRequestModel;
import ru.yandex.intranet.d.model.delivery.provide.DeliveryAndProvideDestinationModel;
import ru.yandex.intranet.d.model.delivery.provide.DeliveryAndProvideModel;
import ru.yandex.intranet.d.model.delivery.provide.DeliveryAndProvideOperationListModel;
import ru.yandex.intranet.d.model.delivery.provide.DeliveryAndProvideOperationModel;
import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.model.resources.ResourceSegmentSettingsModel;
import ru.yandex.intranet.d.model.resources.segmentations.ResourceSegmentationModel;
import ru.yandex.intranet.d.model.resources.segments.ResourceSegmentModel;
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel;
import ru.yandex.intranet.d.services.quotas.QuotasHelper;
import ru.yandex.intranet.d.util.DisplayUtil;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.web.model.delivery.DeliverableFolderOperationDto;
import ru.yandex.intranet.d.web.model.delivery.provide.DeliveryAndProvideMetaResponseDto;
import ru.yandex.intranet.d.web.model.delivery.status.DeliveryStatusAccountDto;
import ru.yandex.intranet.d.web.model.delivery.status.DeliveryStatusDto;
import ru.yandex.intranet.d.web.model.delivery.status.DeliveryStatusOperationDto;
import ru.yandex.intranet.d.web.model.delivery.status.DeliveryStatusProviderDto;
import ru.yandex.intranet.d.web.model.delivery.status.DeliveryStatusRequestedQuotasDto;
import ru.yandex.intranet.d.web.model.delivery.status.DeliveryStatusResourceDto;
import ru.yandex.intranet.d.web.model.delivery.status.DeliveryStatusResponseDto;
import ru.yandex.intranet.d.web.model.operations.OperationStatusDto;
import ru.yandex.intranet.d.web.model.resources.ResourceSegmentationSegmentDto;

/**
 * Service for get statuses of delivery and provide operations
 *
 * @author Evgenii Serov <evserov@yandex-team.ru>
 */
@Component
public class DeliveryAndProvideStatusService {

    private final DeliveryAndProvidePreValidationService preValidationService;
    private final DeliveryAndProvideStoreService storeService;
    private final YdbTableClient tableClient;
    private final AccountsQuotasOperationsDao accountsQuotasOperationsDao;
    private final AccountsDao accountsDao;
    private final ProvidersLoader providersLoader;
    private final ResourcesLoader resourcesLoader;
    private final UnitsEnsemblesLoader unitsEnsemblesLoader;
    private final ResourceSegmentsLoader resourceSegmentsLoader;
    private final ResourceSegmentationsLoader resourceSegmentationsLoader;
    private final AccountSpacesLoader accountSpacesLoader;

    @SuppressWarnings("checkstyle:ParameterNumber")
    public DeliveryAndProvideStatusService(DeliveryAndProvidePreValidationService preValidationService,
                                           DeliveryAndProvideStoreService storeService,
                                           YdbTableClient tableClient,
                                           AccountsQuotasOperationsDao accountsQuotasOperationsDao,
                                           AccountsDao accountsDao,
                                           ProvidersLoader providersLoader,
                                           ResourcesLoader resourcesLoader,
                                           UnitsEnsemblesLoader unitsEnsemblesLoader,
                                           ResourceSegmentsLoader resourceSegmentsLoader,
                                           ResourceSegmentationsLoader resourceSegmentationsLoader,
                                           AccountSpacesLoader accountSpacesLoader) {
        this.preValidationService = preValidationService;
        this.storeService = storeService;
        this.tableClient = tableClient;
        this.accountsQuotasOperationsDao = accountsQuotasOperationsDao;
        this.accountsDao = accountsDao;
        this.providersLoader = providersLoader;
        this.resourcesLoader = resourcesLoader;
        this.unitsEnsemblesLoader = unitsEnsemblesLoader;
        this.resourceSegmentsLoader = resourceSegmentsLoader;
        this.resourceSegmentationsLoader = resourceSegmentationsLoader;
        this.accountSpacesLoader = accountSpacesLoader;
    }

    public Mono<Result<DeliveryStatusResponseDto>> getDeliveryOperationStatuses(
            List<String> deliveryIds,
            Locale locale
    ) {
        return preValidationService.validateRequiredListUuids(deliveryIds, locale).andThenMono(u ->
                inTx(ts -> storeService.getDeliveriesByIds(ts, deliveryIds)
            .flatMap(deliveries ->
                accountsQuotasOperationsDao.getAllByIds(
                        ts,
                        Tenants.DEFAULT_TENANT_ID,
                        deliveries.stream()
                                .filter(delivery -> !delivery.getOperations().isEmpty())
                                .flatMap(d -> toOperationIds(d.getOperations()))
                                .collect(Collectors.toList()))
            .flatMap(operations ->
                    accountsDao.getAllByIdsWithDeleted(
                            ts,
                            operations.stream()
                                    .filter(op -> op.getRequestedChanges().getAccountId().isPresent())
                                    .map(op -> op.getRequestedChanges().getAccountId().get())
                                    .distinct()
                                    .collect(Collectors.toList()),
                            Tenants.DEFAULT_TENANT_ID)
            .flatMap(accounts ->
                    providersLoader.getProvidersByIds(
                            ts,
                            (operations.stream()
                                    .map(AccountsQuotasOperationsModel::getProviderId)
                                    .distinct()
                                    .map(id -> Tuples.of(id, Tenants.DEFAULT_TENANT_ID))
                                    .collect(Collectors.toList())))
            .flatMap(providers ->
                    resourcesLoader.getResourcesByIds(
                            ts,
                            operations.stream()
                                    .map(AccountsQuotasOperationsModel::getRequestedChanges)
                                    .flatMap(reqChanges ->
                                            reqChanges.getFrozenProvisions().orElse(List.of()).stream()
                                                    .map(OperationChangesModel.Provision::getResourceId)
                                                    .distinct()
                                                    .map(id -> Tuples.of(id, Tenants.DEFAULT_TENANT_ID)))
                                    .collect(Collectors.toList()))
            .flatMap(resources ->
                    unitsEnsemblesLoader.getUnitsEnsemblesByIds(
                            ts,
                            resources.stream()
                                    .map(ResourceModel::getUnitsEnsembleId)
                                    .distinct()
                                    .map(id -> Tuples.of(id, Tenants.DEFAULT_TENANT_ID))
                                    .collect(Collectors.toList()))
            .flatMap(unitEnsembles ->
                    accountSpacesLoader.getAccountSpaces(
                            ts,
                            providers.stream()
                                    .map(p -> new WithTenant<>(p.getTenantId(), p.getId()))
                                    .collect(Collectors.toSet()))
            .flatMap(accountSpaces ->
                    resourceSegmentsLoader.getResourceSegmentsByIds(
                            ts,
                            Stream.concat(
                                    resources.stream()
                                            .flatMap(r -> r.getSegments().stream()
                                                    .map(ResourceSegmentSettingsModel::getSegmentId)),
                                    accountSpaces.stream()
                                            .flatMap(a -> a.getSegments().stream())
                                            .map(ResourceSegmentSettingsModel::getSegmentId))
                                    .distinct()
                                    .map(id -> Tuples.of(id, Tenants.DEFAULT_TENANT_ID))
                                    .collect(Collectors.toList()))
            .flatMap(resourceSegments ->
                    resourceSegmentationsLoader.getResourceSegmentationsByIds(
                            ts,
                            Stream.concat(
                                    resources.stream()
                                            .flatMap(r -> r.getSegments().stream()
                                                    .map(ResourceSegmentSettingsModel::getSegmentationId)),
                                    accountSpaces.stream()
                                            .flatMap(a -> a.getSegments().stream())
                                            .map(ResourceSegmentSettingsModel::getSegmentationId))
                                    .distinct()
                                    .map(id -> Tuples.of(id, Tenants.DEFAULT_TENANT_ID))
                                    .collect(Collectors.toList()))
            .map(resourceSegmentations ->
                    toDeliveryStatusResponse(
                            deliveries,
                            operations,
                            accounts,
                            providers,
                            resources,
                            unitEnsembles,
                            resourceSegments,
                            resourceSegmentations,
                            accountSpaces,
                            locale)
            )))))))))));
    }

    @SuppressWarnings("ParameterNumber")
    Result<DeliveryStatusResponseDto> toDeliveryStatusResponse(
            Collection<DeliveryAndProvideModel> deliveries,
            Collection<AccountsQuotasOperationsModel> operations,
            Collection<AccountModel> accounts,
            Collection<ProviderModel> providers,
            Collection<ResourceModel> resources,
            Collection<UnitsEnsembleModel> unitEnsembles,
            Collection<ResourceSegmentModel> segments,
            Collection<ResourceSegmentationModel> segmentations,
            Collection<AccountSpaceModel> accountSpaces,
            Locale locale
    ) {
        Map<String, AccountsQuotasOperationsModel> operationsByIds = operations.stream()
                .collect(Collectors.toMap(AccountsQuotasOperationsModel::getOperationId, Function.identity()));
        Map<String, ResourceModel> resourcesByIds = resources.stream()
                .collect(Collectors.toMap(ResourceModel::getId, Function.identity()));
        Map<String, UnitsEnsembleModel> unitEnsemblesByIds = unitEnsembles.stream()
                .collect(Collectors.toMap(UnitsEnsembleModel::getId, Function.identity()));
        Map<String, ResourceSegmentModel> segmentsByIds = segments.stream()
                .collect(Collectors.toMap(ResourceSegmentModel::getId, Function.identity()));
        Map<String, ResourceSegmentationModel> segmentationsByIds = segmentations.stream()
                .collect(Collectors.toMap(ResourceSegmentationModel::getId, Function.identity()));
        Map<String, AccountModel> accountsByIds = accounts.stream()
                .collect(Collectors.toMap(AccountModel::getId, Function.identity()));
        Map<String, ProviderModel> providersByIds = providers.stream()
                .collect(Collectors.toMap(ProviderModel::getId, Function.identity()));
        Map<String, AccountSpaceModel> accountSpacesByIds = accountSpaces.stream()
                .collect(Collectors.toMap(AccountSpaceModel::getId, Function.identity()));
        Map<DeliveryAccountProviderKey, List<DeliverableMetaRequestModel>> metaByDeliveryAccountResource =
                new HashMap<>();
        for (DeliveryAndProvideModel delivery : deliveries) {
            for (DeliveryAndProvideDestinationModel destination : delivery.getRequest().getDeliverables()) {
                metaByDeliveryAccountResource.compute(
                        new DeliveryAccountProviderKey(delivery.getRequest().getDeliveryId(),
                                destination.getAccountId(), destination.getProviderId()), (k, v) -> v != null ? v
                                : new ArrayList<>()).add(destination.getMeta());
            }
        }

        List<DeliveryStatusDto> deliveryStatusDtos = new ArrayList<>();
        for (DeliveryAndProvideModel delivery: deliveries) {
            Map<String, Set<DeliverableFolderOperationModel>> folderOpByAccountId = delivery.getRequest()
                    .getDeliverables().stream()
                    .filter(d -> d.getFolderOperationModel() != null)
                    .collect(Collectors.groupingBy(DeliveryAndProvideDestinationModel::getAccountId,
                            Collectors.mapping(DeliveryAndProvideDestinationModel::getFolderOperationModel,
                                    Collectors.toSet())));

            List<DeliveryStatusOperationDto> operationDtos = delivery.getOperations().isEmpty() ? List.of() :
                    toOperationIds(delivery.getOperations())
                            .map(operationsByIds::get)
                            .map(operation -> new DeliveryStatusOperationDto(
                                    operation.getOperationId(),
                                    operation.getRequestedChanges().getAccountId().orElse(null),
                                    operation.getProviderId(),
                                    operation.getRequestedChanges().getFrozenProvisions().orElse(List.of()).stream()
                                            .map(d -> {
                                                String resourceId = d.getResourceId();
                                                ResourceModel resourceModel = resourcesByIds.get(resourceId);
                                                return new DeliveryStatusRequestedQuotasDto(
                                                        resourceId,
                                                        QuotasHelper.getAmountDto(
                                                                BigDecimal.valueOf(d.getAmount()),
                                                                resourceModel,
                                                                unitEnsemblesByIds.get(
                                                                        resourceModel.getUnitsEnsembleId()),
                                                                locale));
                                            })
                                            .collect(Collectors.toList()),
                                    toMetaDto(metaByDeliveryAccountResource.get(new DeliveryAccountProviderKey(
                                            delivery.getDeliveryId(),
                                            operation.getRequestedChanges().getAccountId().orElse(null),
                                            operation.getProviderId()
                                    ))),
                                    toFolderOperationDto(folderOpByAccountId.get(operation.getRequestedChanges()
                                            .getAccountId().orElse(null))),
                                    operation.getCreateDateTime(),
                                    operation.getUpdateDateTime().orElse(null),
                                    toStatus(operation),
                                    operation.getErrorMessage().orElse(null),
                                    operation.getErrorKind().orElse(null)))
                            .collect(Collectors.toList());
            List<DeliveryStatusAccountDto> accountDtos = operationDtos.stream()
                    .map(DeliveryStatusOperationDto::getAccountId)
                    .filter(Objects::nonNull)
                    .distinct()
                    .map(accountsByIds::get)
                    .map(account -> new DeliveryStatusAccountDto(
                            account.getId(),
                            account.getProviderId(),
                            DisplayUtil.getAccountDisplayString(account),
                            toAccountSegmentsDto(
                                    account, accountSpacesByIds, segmentsByIds, segmentationsByIds, locale)))
                    .collect(Collectors.toList());
            List<DeliveryStatusProviderDto> providerDtos = operationDtos.stream()
                    .map(DeliveryStatusOperationDto::getProviderId)
                    .distinct()
                    .map(providersByIds::get)
                    .map(provider -> new DeliveryStatusProviderDto(
                            provider.getId(),
                            Locales.select(provider.getNameEn(), provider.getNameRu(), locale)))
                    .collect(Collectors.toList());
            List<DeliveryStatusResourceDto> resourceDtos = operationDtos.stream()
                    .flatMap(op -> op.getRequestedQuotas().stream()
                            .map(DeliveryStatusRequestedQuotasDto::getResourceId))
                    .distinct()
                    .map(resourcesByIds::get)
                    .map(resource -> new DeliveryStatusResourceDto(
                            resource.getId(),
                            Locales.select(resource.getNameEn(), resource.getNameRu(), locale),
                            toResourceSegmentsDto(resource, segmentsByIds, segmentationsByIds, locale)))
                    .collect(Collectors.toList());
            deliveryStatusDtos.add(new DeliveryStatusDto(
                    delivery.getDeliveryId(),
                    operationDtos,
                    accountDtos,
                    providerDtos,
                    resourceDtos));
        }
        return Result.success(new DeliveryStatusResponseDto(deliveryStatusDtos));
    }

    private Set<DeliverableFolderOperationDto> toFolderOperationDto(
            Set<DeliverableFolderOperationModel> operationModels) {
        if (operationModels == null || operationModels.isEmpty()) {
            return null;
        }

        return operationModels.stream()
                .map(deliverableFolderOperationModel -> DeliverableFolderOperationDto.builder()
                        .id(deliverableFolderOperationModel.getId())
                        .folderId(deliverableFolderOperationModel.getFolderId())
                        .timestamp(deliverableFolderOperationModel.getOperationDateTime())
                        .build())
                .collect(Collectors.toSet());
    }

    private <T> Mono<T> inTx(Function<YdbTxSession, Mono<T>> inTx) {
        return tableClient.usingSessionMonoRetryable(session ->
                session.usingTxMonoRetryable(TransactionMode.SERIALIZABLE_READ_WRITE, inTx));
    }

    private DeliveryAndProvideMetaResponseDto toMetaDto(List<DeliverableMetaRequestModel> meta) {
        if (meta == null || meta.isEmpty()) {
            return null;
        }

        Set<Long> bigOrderIds = meta.stream()
                .map(DeliverableMetaRequestModel::getBigOrderId)
                .collect(Collectors.toSet());

        DeliverableMetaRequestModel deliverableMetaRequestModel = meta.get(0);
        return new DeliveryAndProvideMetaResponseDto.Builder()
                .addBigOrderIds(bigOrderIds)
                .quotaRequestId(deliverableMetaRequestModel.getQuotaRequestId())
                .campaignId(deliverableMetaRequestModel.getCampaignId())
                .build();
    }

    private Set<ResourceSegmentationSegmentDto> toAccountSegmentsDto(
            AccountModel account,
            Map<String, AccountSpaceModel> accountSpacesByIds,
            Map<String, ResourceSegmentModel> segmentsByIds,
            Map<String, ResourceSegmentationModel> segmentationsByIds,
            Locale locale
    ) {
        return account.getAccountsSpacesId()
                .map(accountSpacesByIds::get)
                .map(AccountSpaceModel::getSegments)
                .orElse(Set.of())
                .stream()
                .map(s -> new ResourceSegmentationSegmentDto(
                        s,
                        segmentationsByIds,
                        segmentsByIds,
                        locale))
                .collect(Collectors.toSet());
    }

    private Set<ResourceSegmentationSegmentDto> toResourceSegmentsDto(
            ResourceModel resource,
            Map<String, ResourceSegmentModel> segmentsByIds,
            Map<String, ResourceSegmentationModel> segmentationsByIds,
            Locale locale
    ) {
        return resource.getSegments().stream()
                .map(s -> new ResourceSegmentationSegmentDto(
                        s,
                        segmentationsByIds,
                        segmentsByIds,
                        locale))
                .collect(Collectors.toSet());
    }

    private OperationStatusDto toStatus(AccountsQuotasOperationsModel operation) {
        if (operation.getRequestStatus().isEmpty()) {
            return OperationStatusDto.IN_PROGRESS;
        }

        return switch (operation.getRequestStatus().get()) {
            case WAITING -> OperationStatusDto.IN_PROGRESS;
            case OK -> OperationStatusDto.SUCCESS;
            case ERROR -> OperationStatusDto.FAILURE;
        };
    }

    private static final class DeliveryAccountProviderKey {

        private final String deliveryId;
        private final String accountId;
        private final String providerId;

        DeliveryAccountProviderKey(String deliveryId, String accountId, String providerId) {
            this.deliveryId = deliveryId;
            this.accountId = accountId;
            this.providerId = providerId;
        }

        public String getDeliveryId() {
            return deliveryId;
        }

        public String getAccountId() {
            return accountId;
        }

        public String getProviderId() {
            return providerId;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            DeliveryAccountProviderKey that = (DeliveryAccountProviderKey) o;
            return Objects.equals(deliveryId, that.deliveryId) &&
                    Objects.equals(accountId, that.accountId) &&
                    Objects.equals(providerId, that.providerId);
        }

        @Override
        public int hashCode() {
            return Objects.hash(deliveryId, accountId, providerId);
        }

        @Override
        public String toString() {
            return ReflectionToStringBuilder.toString(this);
        }
    }

    private Stream<String> toOperationIds(
            List<DeliveryAndProvideOperationListModel> deliveryAndProvideOperationListModels) {
        return deliveryAndProvideOperationListModels.stream()
                .filter(model -> !model.getOperations().isEmpty())
                .map(model -> model.getOperations().stream()
                        .max(Comparator.comparingInt(DeliveryAndProvideOperationModel::getVersion)).orElseThrow()
                        .getOperationId());
    }
}
