package ru.yandex.intranet.d.services.delivery.provide;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.google.common.collect.Lists;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuple2;
import reactor.util.function.Tuples;

import ru.yandex.intranet.d.dao.Tenants;
import ru.yandex.intranet.d.dao.accounts.AccountsDao;
import ru.yandex.intranet.d.dao.accounts.AccountsQuotasDao;
import ru.yandex.intranet.d.dao.accounts.AccountsQuotasOperationsDao;
import ru.yandex.intranet.d.dao.accounts.OperationsInProgressDao;
import ru.yandex.intranet.d.dao.delivery.DeliveriesAndProvidedRequestsDao;
import ru.yandex.intranet.d.dao.folders.FolderDao;
import ru.yandex.intranet.d.dao.folders.FolderOperationLogDao;
import ru.yandex.intranet.d.dao.quotas.QuotasDao;
import ru.yandex.intranet.d.dao.resources.ResourcesDao;
import ru.yandex.intranet.d.dao.services.ServicesDao;
import ru.yandex.intranet.d.dao.users.UsersDao;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.loaders.providers.AccountSpacesLoader;
import ru.yandex.intranet.d.loaders.providers.ProvidersLoader;
import ru.yandex.intranet.d.loaders.resources.ResourcesLoader;
import ru.yandex.intranet.d.loaders.resources.segmentations.ResourceSegmentationsLoader;
import ru.yandex.intranet.d.loaders.resources.segments.ResourceSegmentsLoader;
import ru.yandex.intranet.d.loaders.resources.types.ResourceTypesLoader;
import ru.yandex.intranet.d.loaders.units.UnitsEnsemblesLoader;
import ru.yandex.intranet.d.loaders.users.UsersLoader;
import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.WithTenant;
import ru.yandex.intranet.d.model.accounts.AccountModel;
import ru.yandex.intranet.d.model.accounts.AccountSpaceModel;
import ru.yandex.intranet.d.model.accounts.AccountsQuotasModel;
import ru.yandex.intranet.d.model.accounts.AccountsQuotasOperationsModel;
import ru.yandex.intranet.d.model.accounts.OperationInProgressModel;
import ru.yandex.intranet.d.model.delivery.provide.DeliveryAndProvideModel;
import ru.yandex.intranet.d.model.folders.FolderModel;
import ru.yandex.intranet.d.model.folders.FolderOperationLogModel;
import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.model.quotas.QuotaModel;
import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.model.resources.ResourceSegmentSettingsModel;
import ru.yandex.intranet.d.model.resources.segmentations.ResourceSegmentationModel;
import ru.yandex.intranet.d.model.resources.segments.ResourceSegmentModel;
import ru.yandex.intranet.d.model.resources.types.ResourceTypeModel;
import ru.yandex.intranet.d.model.services.ServiceWithStatesModel;
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel;
import ru.yandex.intranet.d.model.users.UserModel;
import ru.yandex.intranet.d.services.delivery.model.provide.DeliveryAndProvideDictionary;
import ru.yandex.intranet.d.services.delivery.model.provide.DeliveryAndProvideRequest;
import ru.yandex.intranet.d.services.delivery.model.provide.FailureDictionary;
import ru.yandex.intranet.d.services.delivery.model.provide.ProvideDictionary;
import ru.yandex.intranet.d.services.delivery.model.provide.UpdateDictionary;
import ru.yandex.intranet.d.services.operations.model.ExternalAccountsSpaceKey;
import ru.yandex.intranet.d.services.operations.model.ExternalResourceKey;
import ru.yandex.intranet.d.services.operations.model.ExternalSegmentKey;

/**
 * Delivery store service implementation.
 *
 * @author Evgenii Serov <evserov@yandex-team.ru>
 */
@Component
public class DeliveryAndProvideStoreService {

    private final FolderDao folderDao;
    private final QuotasDao quotasDao;
    private final AccountsDao accountsDao;
    private final ServicesDao servicesDao;
    private final FolderOperationLogDao folderOperationLogDao;
    private final DeliveriesAndProvidedRequestsDao deliveriesAndProvidedRequestsDao;
    private final AccountsQuotasDao accountsQuotasDao;
    private final AccountsQuotasOperationsDao accountsQuotasOperationsDao;
    private final OperationsInProgressDao operationsInProgressDao;
    private final UsersDao usersDao;
    private final ResourcesDao resourcesDao;

    private final UsersLoader usersLoader;
    private final ProvidersLoader providersLoader;
    private final ResourcesLoader resourcesLoader;
    private final ResourceTypesLoader resourceTypesLoader;
    private final UnitsEnsemblesLoader unitsEnsemblesLoader;
    private final ResourceSegmentsLoader resourceSegmentsLoader;
    private final ResourceSegmentationsLoader resourceSegmentationsLoader;
    private final AccountSpacesLoader accountSpacesLoader;

    @SuppressWarnings("ParameterNumber")
    public DeliveryAndProvideStoreService(FolderDao folderDao,
                                          QuotasDao quotasDao,
                                          AccountsDao accountsDao,
                                          ServicesDao servicesDao,
                                          FolderOperationLogDao folderOperationLogDao,
                                          DeliveriesAndProvidedRequestsDao deliveriesAndProvidedRequestsDao,
                                          AccountsQuotasDao accountsQuotasDao,
                                          AccountsQuotasOperationsDao accountsQuotasOperationsDao,
                                          OperationsInProgressDao operationsInProgressDao,
                                          UsersDao usersDao,
                                          ResourcesDao resourcesDao,
                                          UsersLoader usersLoader,
                                          ProvidersLoader providersLoader,
                                          ResourcesLoader resourcesLoader,
                                          ResourceTypesLoader resourceTypesLoader,
                                          UnitsEnsemblesLoader unitsEnsemblesLoader,
                                          ResourceSegmentsLoader resourceSegmentsLoader,
                                          ResourceSegmentationsLoader resourceSegmentationsLoader,
                                          AccountSpacesLoader accountSpacesLoader) {
        this.folderDao = folderDao;
        this.quotasDao = quotasDao;
        this.accountsDao = accountsDao;
        this.servicesDao = servicesDao;
        this.folderOperationLogDao = folderOperationLogDao;
        this.deliveriesAndProvidedRequestsDao = deliveriesAndProvidedRequestsDao;
        this.accountsQuotasDao = accountsQuotasDao;
        this.accountsQuotasOperationsDao = accountsQuotasOperationsDao;
        this.operationsInProgressDao = operationsInProgressDao;
        this.usersDao = usersDao;
        this.resourcesDao = resourcesDao;
        this.usersLoader = usersLoader;
        this.providersLoader = providersLoader;
        this.resourcesLoader = resourcesLoader;
        this.resourceTypesLoader = resourceTypesLoader;
        this.unitsEnsemblesLoader = unitsEnsemblesLoader;
        this.resourceSegmentsLoader = resourceSegmentsLoader;
        this.resourceSegmentationsLoader = resourceSegmentationsLoader;
        this.accountSpacesLoader = accountSpacesLoader;
    }

    public Mono<Optional<DeliveryAndProvideModel>> getDeliveryById(YdbTxSession session, String deliveryId) {
        return deliveriesAndProvidedRequestsDao.getById(session, deliveryId, Tenants.DEFAULT_TENANT_ID);
    }

    public Mono<List<DeliveryAndProvideModel>> getDeliveriesByIds(YdbTxSession session, List<String> ids) {
        return deliveriesAndProvidedRequestsDao.getByIds(session, ids, Tenants.DEFAULT_TENANT_ID);
    }

    public Mono<List<FolderModel>> getFoldersByIds(YdbTxSession session, Set<String> folderIds) {
        return folderDao.getByIds(session, folderIds.stream()
                .map(id -> new WithTenant<>(Tenants.DEFAULT_TENANT_ID, id)).collect(Collectors.toList()));
    }

    public Mono<List<ResourceModel>> getResourcesByIds(YdbTxSession session, Set<String> resourcesIds) {
        return resourcesLoader.getResourcesByIds(session, resourcesIds.stream()
                .map(id -> Tuples.of(id, Tenants.DEFAULT_TENANT_ID)).collect(Collectors.toList()));
    }

    public Mono<List<ServiceWithStatesModel>> getServicesByIds(YdbTxSession session, Set<Long> serviceIds) {
        return servicesDao.getServiceStatesByIds(session, List.copyOf(serviceIds));
    }

    public Mono<List<UnitsEnsembleModel>> getUnitsEnsemblesByIds(YdbTxSession session, Set<String> unitsEnsemblesIds) {
        return unitsEnsemblesLoader.getUnitsEnsemblesByIds(session, unitsEnsemblesIds.stream()
                .map(id -> Tuples.of(id, Tenants.DEFAULT_TENANT_ID)).collect(Collectors.toList()));
    }

    public Mono<List<UserModel>> getUserById(YdbTxSession session, String uid) {
        return usersLoader.getUserByPassportUid(session, uid, Tenants.DEFAULT_TENANT_ID)
                .map(u -> u.stream().collect(Collectors.toList()));
    }

    public Mono<List<ProviderModel>> getProvidersByIds(YdbTxSession session, Set<String> providerIds) {
        return providersLoader.getProvidersByIds(session, providerIds.stream()
                .map(id -> Tuples.of(id, Tenants.DEFAULT_TENANT_ID)).collect(Collectors.toList()));
    }

    public Mono<List<AccountModel>> getAccountsByIds(YdbTxSession session, Set<String> accountIds) {
        return accountsDao.getByIds(session, accountIds.stream()
                .map(id -> new WithTenant<>(Tenants.DEFAULT_TENANT_ID, id)).collect(Collectors.toList()));
    }

    public Mono<List<AccountSpaceModel>> getAccountsSpacesByProviderIds(YdbTxSession session, Set<String> providerIds) {
        return accountSpacesLoader.getAccountSpaces(session, providerIds.stream()
                .map(id -> new WithTenant<>(Tenants.DEFAULT_TENANT_ID, id)).collect(Collectors.toSet()));
    }

    public Mono<List<ResourceSegmentModel>> getResourceSegmentsByIds(YdbTxSession session, Set<String> segmentIds) {
        return resourceSegmentsLoader.getResourceSegmentsByIds(session, segmentIds.stream()
                .map(id -> Tuples.of(id, Tenants.DEFAULT_TENANT_ID)).collect(Collectors.toList()));
    }

    public Mono<List<ResourceSegmentationModel>> getResourceSegmentationsByIds(YdbTxSession session,
                                                                               Set<String> segmentationIds) {
        return resourceSegmentationsLoader.getResourceSegmentationsByIds(session, segmentationIds.stream()
                .map(id -> Tuples.of(id, Tenants.DEFAULT_TENANT_ID)).collect(Collectors.toList()));
    }

    public Mono<List<QuotaModel>> getQuotasByKeys(YdbTxSession session, Set<QuotaModel.Key> keys) {
        return quotasDao.getByKeys(session, keys.stream()
                .map(key -> new WithTenant<>(Tenants.DEFAULT_TENANT_ID, key)).collect(Collectors.toList()));
    }

    public Mono<Void> upsertDelivery(YdbTxSession session, DeliveryAndProvideModel delivery) {
        return deliveriesAndProvidedRequestsDao.upsertOneRetryable(session, delivery).then();
    }

    public Mono<Void> upsertQuotas(YdbTxSession session, List<QuotaModel> quotas) {
        if (quotas.isEmpty()) {
            return Mono.empty();
        }
        return quotasDao.upsertAllRetryable(session, quotas).then();
    }

    public Mono<Void> upsertOperationLog(YdbTxSession session, List<FolderOperationLogModel> logs) {
        if (logs.isEmpty()) {
            return Mono.empty();
        }
        return folderOperationLogDao.upsertAllRetryable(session, logs).then();
    }

    public Mono<Void> upsertFolders(YdbTxSession session, List<FolderModel> folders) {
        if (folders.isEmpty()) {
            return Mono.empty();
        }
        return folderDao.upsertAllRetryable(session, folders).then();
    }

    public Mono<DeliveryAndProvideDictionary> loadDictionaries(YdbTxSession session,
                                                               DeliveryAndProvideRequest request) {
        String authorUid = request.getAuthorUid();
        Set<Long> serviceIds = new HashSet<>();
        Set<String> providerIds = new HashSet<>();
        Set<String> folderIds = new HashSet<>();
        Set<String> resourceIds = new HashSet<>();
        Set<String> accountIds = new HashSet<>();
        request.getDeliverables().forEach(deliverable -> {
            serviceIds.add(deliverable.getServiceId());
            providerIds.add(deliverable.getProviderId());
            folderIds.add(deliverable.getFolderId());
            resourceIds.add(deliverable.getResourceId());
            accountIds.add(deliverable.getAccountId());
        });
        return getFoldersByIds(session, folderIds)
                .flatMap(folders -> getResourcesByIds(session, resourceIds)
                .flatMap(resources -> getServicesByIds(session, serviceIds)
                .flatMap(services -> getUserById(session, authorUid)
                .flatMap(users -> getProvidersByIds(session, providerIds)
                .flatMap(providers -> getAccountsByIds(session, accountIds)
                .flatMap(accounts -> getUnitsEnsemblesByIds(session, resources.stream()
                        .map(ResourceModel::getUnitsEnsembleId).collect(Collectors.toSet()))
                .flatMap(unitEnsembles -> getAccountsSpacesByProviderIds(session, providerIds)
                .flatMap(accountSpaces -> getResourceSegmentationsByIds(session, Stream.concat(
                        resources.stream()
                                .flatMap(r -> r.getSegments().stream()
                                        .map(ResourceSegmentSettingsModel::getSegmentationId)),
                        accountSpaces.stream()
                                .flatMap(a -> a.getSegments().stream())
                                .map(ResourceSegmentSettingsModel::getSegmentationId)).collect(Collectors.toSet()))
                .flatMap(resourceSegmentations -> getResourceSegmentsByIds(session, Stream.concat(
                        resources.stream()
                                .flatMap(r -> r.getSegments().stream()
                                        .map(ResourceSegmentSettingsModel::getSegmentId)),
                        accountSpaces.stream()
                                .flatMap(a -> a.getSegments().stream())
                                .map(ResourceSegmentSettingsModel::getSegmentId)).collect(Collectors.toSet()))
                .map(resourceSegments -> buildDictionaries(folders, resources, services, unitEnsembles, users,
                        accounts, providers, accountSpaces, resourceSegmentations, resourceSegments)))))))))));
    }

    @SuppressWarnings("ParameterNumber")
    private DeliveryAndProvideDictionary buildDictionaries(List<FolderModel> folders,
                                                           List<ResourceModel> resources,
                                                           List<ServiceWithStatesModel> services,
                                                           List<UnitsEnsembleModel> unitsEnsembles,
                                                           List<UserModel> users,
                                                           List<AccountModel> accounts,
                                                           List<ProviderModel> providers,
                                                           List<AccountSpaceModel> accountSpaces,
                                                           List<ResourceSegmentationModel> resourceSegmentations,
                                                           List<ResourceSegmentModel> resourceSegments) {
        Map<String, ResourceModel> resourcesById = resources.stream()
                .collect(Collectors.toMap(ResourceModel::getId, Function.identity()));
        Map<String, AccountModel> accountsById = accounts.stream()
                .collect(Collectors.toMap(AccountModel::getId, Function.identity()));
        Map<String, UnitsEnsembleModel> unitsEnsemblesById = unitsEnsembles.stream()
                .collect(Collectors.toMap(UnitsEnsembleModel::getId, Function.identity()));
        Map<String, FolderModel> foldersById = folders.stream()
                .collect(Collectors.toMap(FolderModel::getId, Function.identity()));
        Map<String, AccountSpaceModel> accountSpacesById = accountSpaces.stream()
                .collect(Collectors.toMap(AccountSpaceModel::getId, Function.identity()));
        Map<Long, ServiceWithStatesModel> servicesById = services.stream()
                .collect(Collectors.toMap(ServiceWithStatesModel::getId, Function.identity()));
        Map<String, ResourceSegmentationModel> resourceSegmentationsById = resourceSegmentations.stream()
                .collect(Collectors.toMap(ResourceSegmentationModel::getId, Function.identity()));
        Map<String, UserModel> usersByUid = users.stream().filter(u -> u.getPassportUid().isPresent())
                .collect(Collectors.toMap(r -> r.getPassportUid().get(), Function.identity()));
        Map<String, ResourceSegmentModel> resourceSegmentsById = resourceSegments.stream()
                .collect(Collectors.toMap(ResourceSegmentModel::getId, Function.identity()));
        Map<String, ProviderModel> providersById = providers.stream()
                .collect(Collectors.toMap(ProviderModel::getId, Function.identity()));
        return new DeliveryAndProvideDictionary(resourcesById, unitsEnsemblesById, foldersById, servicesById,
                providersById, usersByUid, accountsById, accountSpacesById, resourceSegmentationsById,
                resourceSegmentsById);
    }

    public Mono<Void> upsertAccountOperation(YdbTxSession session,
                                             List<AccountsQuotasOperationsModel> accountsQuotasOperationsModel) {
        if (accountsQuotasOperationsModel.isEmpty()) {
            return Mono.empty();
        }
        return accountsQuotasOperationsDao.upsertAllRetryable(session, accountsQuotasOperationsModel).then();
    }

    public Mono<Void> upsertAccountOperationInProgress(YdbTxSession session,
                                                       List<OperationInProgressModel> operationInProgressModels) {
        if (operationInProgressModels.isEmpty()) {
            return Mono.empty();
        }
        return operationsInProgressDao.upsertAllRetryable(session, operationInProgressModels).then();
    }

    public Mono<List<AccountsQuotasModel>> getAccountQuotasByIds(YdbTxSession session, Set<String> accountIds) {
        return accountsQuotasDao.getAllByAccountIds(session, Tenants.DEFAULT_TENANT_ID, accountIds);
    }

    public Mono<ProvideDictionary> loadDictionariesForProvide(
            YdbTxSession ts,
            DeliveryAndProvideDictionary dictionary,
            DeliveryAndProvideModel deliveryRequestsModel,
            Map<String, AccountsQuotasOperationsModel> accountsQuotasOperationsByAccountId,
            List<OperationInProgressModel> accountsOpInProgress) {
        Function<List<AccountModel>, Set<String>> resourceIdsFunc =
                (accounts) -> accounts.stream().map(AccountModel::getProviderId)
                        .collect(Collectors.toSet());
        Function<List<ResourceModel>, Set<String>> resourceTypeIdsFun = (resources) -> resources.stream()
                .map(ResourceModel::getResourceTypeId).collect(Collectors.toSet());
        Function<List<ResourceModel>, Set<String>> unitsEnsembleIdsFun = (resources) -> resources.stream()
                .map(ResourceModel::getUnitsEnsembleId).collect(Collectors.toSet());
        Function<List<ResourceModel>, Set<String>> segmentationIdsFun = (resources) -> resources.stream()
                .flatMap(r -> r.getSegments().stream().map(ResourceSegmentSettingsModel::getSegmentationId))
                .collect(Collectors.toSet());
        Function<List<ResourceModel>, Set<String>> segmentIdsFun = (resources) -> resources.stream()
                .flatMap(r -> r.getSegments().stream().map(ResourceSegmentSettingsModel::getSegmentId))
                .collect(Collectors.toSet());

        Set<String> folderIds = dictionary.getFolders().values().stream()
                .map(FolderModel::getId)
                .collect(Collectors.toSet());

        Map<String, OperationInProgressModel> accountOpInProgressByAccountId = accountsOpInProgress.stream()
                .collect(Collectors.toMap(a -> a.getAccountId().orElseThrow(), Function.identity()));

        return getAccountsByFolderIds(ts, folderIds).flatMap(accounts -> getResourcesForProviders(ts,
                resourceIdsFunc.apply(accounts)).flatMap(resources ->  getResourceTypesByResourceTypeIds(ts,
                resourceTypeIdsFun.apply(resources)).flatMap(resourceTypes -> getUnitsEnsemblesByIds(ts,
                unitsEnsembleIdsFun.apply(resources)).flatMap(unitEnsembles -> getResourceSegmentationsByIds(ts,
                segmentationIdsFun.apply(resources)).flatMap(resourceSegmentations -> getResourceSegmentsByIds(ts,
                segmentIdsFun.apply(resources)).map(resourceSegments -> buildProvideDictionary(dictionary, accounts,
                resources, resourceTypes, unitEnsembles, resourceSegmentations, resourceSegments, deliveryRequestsModel,
                accountsQuotasOperationsByAccountId, accountOpInProgressByAccountId)))))));
    }

    public Mono<List<AccountModel>> getAccountsByFolderIds(YdbTxSession session, Set<String> folderIds) {
        return accountsDao.getAllByFolderIds(session, folderIds.stream()
                .map(id -> new WithTenant<>(Tenants.DEFAULT_TENANT_ID, id))
                .collect(Collectors.toList()), false);
    }

    public Mono<List<ResourceTypeModel>> getResourceTypesByResourceTypeIds(YdbTxSession session,
                                                                           Set<String> resourceTypeIds) {
        return resourceTypesLoader.getResourceTypesByIds(session, resourceTypeIds.stream()
                .map(id -> Tuples.of(id, Tenants.DEFAULT_TENANT_ID))
                .collect(Collectors.toList()));
    }

    @SuppressWarnings("ParameterNumber")
    private ProvideDictionary buildProvideDictionary(
            DeliveryAndProvideDictionary deliveryAndProvideDictionary,
            List<AccountModel> accounts,
            List<ResourceModel> resources,
            List<ResourceTypeModel> resourceTypes,
            List<UnitsEnsembleModel> unitEnsembles,
            List<ResourceSegmentationModel> resourceSegmentations,
            List<ResourceSegmentModel> resourceSegments,
            DeliveryAndProvideModel deliveryRequestsModel,
            Map<String, AccountsQuotasOperationsModel> deliveryAccountsQuotasOperationsByAccountId,
            Map<String, OperationInProgressModel> accountOpInProgressByAccountId) {
        Map<String, ResourceModel> resourcesById = resources.stream()
                .collect(Collectors.toMap(ResourceModel::getId, Function.identity()));
        Map<String, ResourceTypeModel> resourceTypesById = resourceTypes.stream()
                .collect(Collectors.toMap(ResourceTypeModel::getId, Function.identity()));
        Map<String, UnitsEnsembleModel> unitsEnsemblesById = unitEnsembles.stream()
                .collect(Collectors.toMap(UnitsEnsembleModel::getId, Function.identity()));
        Map<String, FolderModel> foldersById = Map.copyOf(deliveryAndProvideDictionary.getFolders());
        Map<Long, ServiceWithStatesModel> servicesById = Map.copyOf(deliveryAndProvideDictionary.getServices());
        Map<String, ProviderModel> providersById = Map.copyOf(deliveryAndProvideDictionary.getProviders());
        UserModel author = deliveryAndProvideDictionary.getUsersByUid()
                .get(deliveryRequestsModel.getRequest().getAuthorUid());
        Map<String, AccountModel> accountsById = accounts.stream()
                .collect(Collectors.toMap(AccountModel::getId, Function.identity()));
        Map<String, AccountSpaceModel> accountSpacesById = Map.copyOf(deliveryAndProvideDictionary.getAccountSpaces());
        Map<String, ResourceSegmentationModel> deliveryResourceSegmentations =
                deliveryAndProvideDictionary.getResourceSegmentations();
        Map<String, ResourceSegmentationModel> resourceSegmentationsById =
                Stream.of(deliveryResourceSegmentations.values(), resourceSegmentations).flatMap(Collection::stream)
                        .collect(Collectors.toMap(ResourceSegmentationModel::getId, Function.identity(), (a, b) -> a));
        Map<String, ResourceSegmentModel> deliveryResourceSegments = deliveryAndProvideDictionary.getResourceSegments();
        Map<String, ResourceSegmentModel> resourceSegmentsById =
                Stream.of(deliveryResourceSegments.values(), resourceSegments).flatMap(Collection::stream)
                        .collect(Collectors.toMap(ResourceSegmentModel::getId, Function.identity(), (a, b) -> a));
        Map<String, AccountsQuotasOperationsModel> accountsQuotasOperationsByAccountId =
                Map.copyOf(deliveryAccountsQuotasOperationsByAccountId);

        Function<Set<ResourceSegmentSettingsModel>, Set<ExternalSegmentKey>> toExternalKeys = s -> s.stream()
                .map(segment -> {
                    String segmentationId = segment.getSegmentationId();
                    String segmentId = segment.getSegmentId();
                    ResourceSegmentationModel resourceSegmentationModel =
                            resourceSegmentationsById.get(segmentationId);
                    ResourceSegmentModel resourceSegmentModel = resourceSegmentsById.get(segmentId);
                    return new ExternalSegmentKey(resourceSegmentationModel.getKey(),
                            resourceSegmentModel.getKey());
                }).collect(Collectors.toSet());

        Map<ExternalAccountsSpaceKey, AccountSpaceModel> accountsSpaceIdByExternalKey =
                accountSpacesById.values().stream()
                        .collect(Collectors.toMap(accountSpace -> {
                            Set<ExternalSegmentKey> segments = toExternalKeys.apply(accountSpace.getSegments());
                            return new ExternalAccountsSpaceKey(segments);
                        }, Function.identity(), (a, b) -> a));

        Map<ExternalResourceKey, ResourceModel> resourceByExternalKey = resourcesById.values().stream()
                .collect(Collectors.toMap(r -> {
                    String resourceTypeId = r.getResourceTypeId();
                    ResourceTypeModel resourceType = resourceTypesById.get(resourceTypeId);
                    String key = resourceType.getKey();
                    Set<ExternalSegmentKey> segments = toExternalKeys.apply(r.getSegments());
                    return new ExternalResourceKey(key, segments);
                }, Function.identity(), (a, b) -> a));

        return new ProvideDictionary(resourcesById, resourceByExternalKey, resourceTypesById, unitsEnsemblesById,
                foldersById, servicesById, providersById, author, accountsById, accountSpacesById,
                resourceSegmentationsById, resourceSegmentsById, accountsQuotasOperationsByAccountId,
                accountOpInProgressByAccountId, accountsSpaceIdByExternalKey, deliveryRequestsModel);
    }

    public Mono<UpdateDictionary> loadUpdateDictionaries(
            YdbTxSession ts,
            String accountId,
            ProvideDictionary provideDictionary) {
        OperationInProgressModel opInPr =
                provideDictionary.getAccountOpInProgressByAccountId().get(accountId);
        return operationsInProgressDao.getById(ts, opInPr.getKey(), Tenants.DEFAULT_TENANT_ID).flatMap(newOppInPrO -> {
            if (newOppInPrO.isEmpty()) {
                return Mono.error(new IllegalArgumentException("Operation in progress missed" + opInPr.getKey()));
            }

            OperationInProgressModel inProgress = newOppInPrO.get();
            String operationId = inProgress.getOperationId();
            return loadOperations(ts, Set.of(operationId)).flatMap(operations -> {
                Map<String, AccountsQuotasOperationsModel> operationById = operations.stream()
                        .collect(Collectors.toMap(AccountsQuotasOperationsModel::getOperationId, Function.identity()));

                if (!operationById.containsKey(operationId)) {
                    return Mono.error(new IllegalArgumentException("Operation missed " + operationId));
                }

                return accountsDao.getByIdWithDeleted(ts, accountId, Tenants.DEFAULT_TENANT_ID).flatMap(accountO -> {
                    if (accountO.isEmpty()) {
                        return Mono.error(new IllegalArgumentException("Provide and update provision missing account "
                                + accountId));
                    }
                    AccountModel account = accountO.get();
                    return folderDao.getById(ts, account.getFolderId(), Tenants.DEFAULT_TENANT_ID).flatMap(folderO -> {
                        if (folderO.isEmpty()) {
                            return Mono.error(new IllegalArgumentException("Provide and update provision missing " +
                                    "folder " + account.getFolderId()));
                        }

                        return loadUpdateDictionaries(ts, newOppInPrO.get(), operationById.get(operationId), account,
                                folderO.get());
                    });
                });
            });
        });
    }

    private Mono<UpdateDictionary> loadUpdateDictionaries(
            YdbTxSession ts,
            OperationInProgressModel operationInProgressModel,
            AccountsQuotasOperationsModel accountsQuotasOperations,
            AccountModel account,
            FolderModel folder) {
        return accountsDao.getByFoldersForProvider(ts, Tenants.DEFAULT_TENANT_ID, account.getProviderId(),
                Set.of(folder.getId()), account.getAccountsSpacesId().orElse(null)).flatMap(accounts -> {
            Map<String, AccountModel> accountsById = Stream.concat(accounts.stream(), Stream.of(account))
                    .collect(Collectors.toMap(AccountModel::getId, Function.identity(), (a, b) -> a));
            return quotasDao.getByFoldersAndProvider(ts, List.of(folder.getId()), Tenants.DEFAULT_TENANT_ID,
                    account.getProviderId()).flatMap(quotas -> {
                Map<String, QuotaModel> quotasByResourceId = quotas.stream().collect(
                        Collectors.toMap(QuotaModel::getResourceId, Function.identity()));
                Set<String> notRemovedAccountIds = accountsById.values().stream()
                        .filter(acc -> !acc.isDeleted()).map(AccountModel::getId)
                        .collect(Collectors.toSet());
                return accountsQuotasDao.getAllByAccountIds(ts, Tenants.DEFAULT_TENANT_ID, notRemovedAccountIds)
                        .map(provisions -> {
                            var provisionByResourceIdByAccount = provisions.stream()
                                    .collect(Collectors.groupingBy(AccountsQuotasModel::getAccountId,
                                            Collectors.toMap(AccountsQuotasModel::getResourceId, Function.identity())));

                            return new UpdateDictionary(account, folder, quotasByResourceId,
                                    provisionByResourceIdByAccount, accountsById, operationInProgressModel,
                                    accountsQuotasOperations);
                        });
            });
        });
    }

    public Mono<List<UserModel>> loadUsers(YdbTxSession ts, Set<String> uids, Set<String> logins) {
        if (logins.isEmpty() && uids.isEmpty()) {
            return Mono.just(List.of());
        }
        List<List<String>> partitionedLogins = Lists.partition(new ArrayList<>(logins), 250);
        List<List<String>> partitionedUids = Lists.partition(new ArrayList<>(uids), 250);
        List<Tuple2<List<Tuple2<String, TenantId>>, List<Tuple2<String, TenantId>>>> zipped = new ArrayList<>();
        for (int i = 0; i < Math.max(partitionedLogins.size(), partitionedUids.size()); i++) {
            List<Tuple2<String, TenantId>> loginPairs = i < partitionedLogins.size()
                    ? partitionedLogins.get(i).stream().map(v -> Tuples.of(v, Tenants.DEFAULT_TENANT_ID))
                    .collect(Collectors.toList())
                    : List.of();
            List<Tuple2<String, TenantId>> uidPairs = i < partitionedUids.size()
                    ? partitionedUids.get(i).stream().map(v -> Tuples.of(v, Tenants.DEFAULT_TENANT_ID))
                    .collect(Collectors.toList())
                    : List.of();
            zipped.add(Tuples.of(uidPairs, loginPairs));
        }
        return Flux.fromIterable(zipped)
                .concatMap(v -> usersDao.getByExternalIds(ts, v.getT1(), v.getT2(), List.of()))
                .collectList().map(l -> l.stream().flatMap(Collection::stream).collect(Collectors.toList()));
    }

    public Mono<List<AccountsQuotasOperationsModel>> loadOperations(YdbTxSession session, Set<String> operationIds) {
        if (operationIds.isEmpty()) {
            return Mono.just(List.of());
        }
        List<WithTenant<String>> unitsEnsembleIdsToLoad = operationIds.stream()
                .map(id -> new WithTenant<>(Tenants.DEFAULT_TENANT_ID, id)).collect(Collectors.toList());
        return Flux.fromIterable(Lists.partition(unitsEnsembleIdsToLoad, 500))
                .concatMap(v -> accountsQuotasOperationsDao.getByIds(session, v))
                .collectList().map(l -> l.stream().flatMap(Collection::stream).collect(Collectors.toList()));
    }

    public Mono<AccountsQuotasOperationsModel> upsertOperation(YdbTxSession ts,
                                                               AccountsQuotasOperationsModel operation) {
        return accountsQuotasOperationsDao.upsertOneRetryable(ts, operation);
    }

    public Mono<Void> deleteOperationInProgress(YdbTxSession ts, OperationInProgressModel inProgress) {
        return operationsInProgressDao.deleteOneRetryable(ts, new WithTenant<>(inProgress.getTenantId(),
                inProgress.getKey())).then();
    }

    public Mono<Void> upsertProvisions(YdbTxSession ts, List<AccountsQuotasModel> provisions) {
        return accountsQuotasDao.upsertAllRetryable(ts, provisions).then();
    }

    public Mono<Void> upsertAccount(YdbTxSession ts, AccountModel account) {
        return accountsDao.upsertOneRetryable(ts, account).then();
    }


    public Mono<FailureDictionary> loadUpdateFailureDictionaries(
            YdbTxSession ts, ProvideDictionary provideDictionary, String accountId) {
        OperationInProgressModel opInPr =
                provideDictionary.getAccountOpInProgressByAccountId().get(accountId);
        String providerId = provideDictionary.getAccountsById().get(accountId).getProviderId();

        return operationsInProgressDao.getById(ts, opInPr.getKey(), Tenants.DEFAULT_TENANT_ID).flatMap(newOppInPrO -> {
            if (newOppInPrO.isEmpty()) {
                return Mono.error(new IllegalArgumentException("Operation in progress missed " + opInPr.getKey()));
            }
            OperationInProgressModel inProgress = newOppInPrO.get();
            String operationId = inProgress.getOperationId();
            return loadOperations(ts, Set.of(operationId)).flatMap(operations -> {
                Map<String, AccountsQuotasOperationsModel> operationById = operations.stream()
                        .collect(Collectors.toMap(AccountsQuotasOperationsModel::getOperationId, Function.identity()));

                if (!operationById.containsKey(operationId)) {
                    return Mono.error(new IllegalArgumentException("Operation missed " + operationId));
                }

                return loadUpdateProvisionQuotas(ts, accountId, providerId).map(quotas ->
                        toFailureDictionaries(inProgress, operationById.get(operationId), quotas));
            });
        });
    }

    public Mono<List<QuotaModel>> loadUpdateProvisionQuotas(YdbTxSession ts,
                                                            String accountId, String  providerId) {
        return accountsDao.getByIdWithDeleted(ts, accountId, Tenants.DEFAULT_TENANT_ID).flatMap(accountO -> {
            if (accountO.isEmpty()) {
                return Mono.error(new IllegalArgumentException("Operation references missing account " + accountId));
            }

            return quotasDao.getByFoldersAndProvider(ts, List.of(accountO.get().getFolderId()),
                    Tenants.DEFAULT_TENANT_ID, providerId);
        });
    }

    private FailureDictionary toFailureDictionaries(OperationInProgressModel operationInProgress,
                                                    AccountsQuotasOperationsModel accountsQuotasOperations,
                                                    List<QuotaModel> quotas) {
        return new FailureDictionary(operationInProgress, accountsQuotasOperations, quotas);
    }

    public Mono<Void> finishOperation(YdbTxSession ts, OperationInProgressModel operationInProgress,
                                      AccountsQuotasOperationsModel updatedOperation) {
        WithTenant<OperationInProgressModel.Key> inProgressKeyToDelete =
                new WithTenant<>(operationInProgress.getTenantId(), operationInProgress.getKey());

        return operationsInProgressDao.deleteOneRetryable(ts, inProgressKeyToDelete)
                .then(Mono.defer(() -> accountsQuotasOperationsDao.upsertOneRetryable(ts, updatedOperation)))
                .then();
    }

    public Mono<Optional<AccountModel>> loadAccountByExternalId(YdbTxSession ts, String providerId,
                                                                String accountsSpaceId, String externalAccountId) {
        return accountsDao.getAllByExternalId(ts, new WithTenant<>(Tenants.DEFAULT_TENANT_ID,
                new AccountModel.ExternalId(providerId, externalAccountId, accountsSpaceId)));
    }

    public Mono<List<ResourceModel>> getResourcesForProviders(YdbTxSession ts, Collection<String> providerIds) {
        return resourcesDao.getAllByProviders(ts, providerIds, Tenants.DEFAULT_TENANT_ID, true);
    }
}
