package ru.yandex.intranet.d.services.delivery.provide;

import java.time.Instant;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import com.google.common.collect.Sets;
import io.grpc.Status;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import org.springframework.stereotype.Component;

import ru.yandex.intranet.d.dao.Tenants;
import ru.yandex.intranet.d.model.accounts.AccountModel;
import ru.yandex.intranet.d.model.accounts.AccountSpaceModel;
import ru.yandex.intranet.d.model.accounts.AccountsQuotasModel;
import ru.yandex.intranet.d.model.accounts.AccountsQuotasOperationsModel;
import ru.yandex.intranet.d.model.accounts.OperationChangesModel;
import ru.yandex.intranet.d.model.delivery.provide.DeliveryAndProvideOperationWithIdModel;
import ru.yandex.intranet.d.model.folders.FolderModel;
import ru.yandex.intranet.d.model.providers.AccountsSettingsModel;
import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.model.quotas.QuotaModel;
import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.model.services.ServiceWithStatesModel;
import ru.yandex.intranet.d.model.units.UnitModel;
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel;
import ru.yandex.intranet.d.model.users.UserModel;
import ru.yandex.intranet.d.services.delivery.model.DeliveryQuotas;
import ru.yandex.intranet.d.services.delivery.model.ValidatedQuotas;
import ru.yandex.intranet.d.services.delivery.model.provide.DeliveryAndProvideDestination;
import ru.yandex.intranet.d.services.delivery.model.provide.DeliveryAndProvideDictionary;
import ru.yandex.intranet.d.services.delivery.model.provide.DeliveryAndProvideRequest;
import ru.yandex.intranet.d.services.delivery.model.provide.ProvideDictionary;
import ru.yandex.intranet.d.services.delivery.model.provide.UpdateDictionary;
import ru.yandex.intranet.d.services.integration.providers.ProviderError;
import ru.yandex.intranet.d.services.integration.providers.Response;
import ru.yandex.intranet.d.services.integration.providers.rest.model.AccountDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.AccountsSpaceKeyResponseDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.ErrorMessagesDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.LastUpdateDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.ProvisionDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.ResourceKeyResponseDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.SegmentKeyResponseDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.UpdateProvisionResponseDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.UserIdDto;
import ru.yandex.intranet.d.services.operations.model.ExternalAccountsSpaceKey;
import ru.yandex.intranet.d.services.operations.model.ExternalResourceKey;
import ru.yandex.intranet.d.services.operations.model.ExternalSegmentKey;
import ru.yandex.intranet.d.services.operations.model.ReceivedAccount;
import ru.yandex.intranet.d.services.operations.model.ReceivedAccountsSpaceKey;
import ru.yandex.intranet.d.services.operations.model.ReceivedLastUpdate;
import ru.yandex.intranet.d.services.operations.model.ReceivedProvision;
import ru.yandex.intranet.d.services.operations.model.ReceivedResourceKey;
import ru.yandex.intranet.d.services.operations.model.ReceivedSegmentKey;
import ru.yandex.intranet.d.services.operations.model.ReceivedUpdatedProvision;
import ru.yandex.intranet.d.services.operations.model.ReceivedUserId;
import ru.yandex.intranet.d.services.operations.model.ValidatedReceivedAccount;
import ru.yandex.intranet.d.services.operations.model.ValidatedReceivedLastUpdate;
import ru.yandex.intranet.d.services.operations.model.ValidatedReceivedProvision;
import ru.yandex.intranet.d.services.operations.model.ValidatedReceivedUpdatedProvision;
import ru.yandex.intranet.d.services.validators.AbcServiceValidator;
import ru.yandex.intranet.d.util.Uuids;
import ru.yandex.intranet.d.util.result.ErrorCollection;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.util.result.TypedError;
import ru.yandex.intranet.d.util.units.Units;

import static io.grpc.Status.Code.INTERNAL;

/**
 * Delivery validation service
 *
 * @author Evgenii Serov <evserov@yandex-team.ru>
 */
@Component
public class DeliveryAndProvideValidationService {
    private final MessageSource messages;

    public DeliveryAndProvideValidationService(@Qualifier("messageSource") MessageSource messages) {
        this.messages = messages;
    }

    public Result<Void> validateRequest(DeliveryAndProvideRequest request, DeliveryAndProvideDictionary dictionary,
                                        Locale locale) {
        ErrorCollection.Builder errors = ErrorCollection.builder();
        if (!dictionary.getUsersByUid().containsKey(request.getAuthorUid())) {
            errors.addError("authorUid", TypedError.invalid(messages
                    .getMessage("errors.user.not.found", null, locale)));
        } else {
            UserModel author = dictionary.getUsersByUid().get(request.getAuthorUid());
            if (author.isDeleted() || author.getStaffDismissed().orElse(false)) {
                errors.addError("authorUid", TypedError.invalid(messages
                        .getMessage("errors.user.not.found", null, locale)));
            }
        }
        for (int i = 0; i < request.getDeliverables().size(); i++) {
            DeliveryAndProvideDestination destination = request.getDeliverables().get(i);
            validateDestination(destination, dictionary, errors, i, locale);
        }
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build());
        }
        return Result.success(null);
    }

    private void validateDestination(DeliveryAndProvideDestination deliverable, DeliveryAndProvideDictionary dictionary,
                                     ErrorCollection.Builder errors, int i, Locale locale) {
        if (!dictionary.getResources().containsKey(deliverable.getResourceId())) {
            errors.addError("deliverables." + i + ".resourceId", TypedError.invalid(messages
                    .getMessage("errors.resource.not.found", null, locale)));
        } else {
            ResourceModel resource = dictionary.getResources().get(deliverable.getResourceId());
            ProviderModel provider = dictionary.getProviders().get(deliverable.getProviderId());
            if (provider == null) {
                errors.addError("deliverables." + i + ".providerId", TypedError.invalid(messages
                        .getMessage("errors.provider.not.found", null, locale)));
            } else if (resource.isDeleted() || provider.isDeleted() ||
                    !resource.getProviderId().equals(deliverable.getProviderId())) {
                errors.addError("deliverables." + i + ".resourceId", TypedError.invalid(messages
                        .getMessage("errors.resource.not.found", null, locale)));
            } else if (!resource.isManaged()) {
                errors.addError("deliverables." + i + ".resourceId", TypedError.invalid(messages
                        .getMessage("errors.resource.not.managed", null, locale)));
            } else if (!provider.isManaged()) {
                errors.addError("deliverables." + i + ".providerId", TypedError.invalid(messages
                        .getMessage("errors.provider.is.not.managed", null, locale)));
            } else if (resource.isReadOnly()) {
                errors.addError("deliverables." + i + ".resourceId", TypedError.invalid(messages
                        .getMessage("errors.resource.is.read.only", null, locale)));
            } else if (provider.isReadOnly()) {
                errors.addError("deliverables." + i + ".providerId", TypedError.invalid(messages
                        .getMessage("errors.provider.is.read.only", null, locale)));
            }
            UnitsEnsembleModel unitsEnsemble = dictionary.getUnitsEnsembles().get(resource.getUnitsEnsembleId());
            Optional<UnitModel> unitO = unitsEnsemble.unitByKey(deliverable.getDelta().getUnitKey());
            if (unitO.isEmpty()) {
                errors.addError("deliverables." + i + ".delta.unitKey", TypedError.invalid(messages
                        .getMessage("errors.unit.not.found", null, locale)));
            }
        }

        if (!dictionary.getServices().containsKey(deliverable.getServiceId())) {
            errors.addError("deliverables." + i + ".serviceId", TypedError.invalid(messages
                    .getMessage("errors.service.not.found", null, locale)));
        } else {
            ServiceWithStatesModel service = dictionary.getServices().get(deliverable.getServiceId());
            if (!service.isExportable()) {
                errors.addError("deliverables." + i + ".serviceId", TypedError.invalid(messages
                        .getMessage("errors.service.is.non.exportable", null, locale)));
            } else if (!isValidServiceState(service)) {
                errors.addError("deliverables." + i + ".serviceId", TypedError.invalid(messages
                        .getMessage("errors.service.bad.status", null, locale)));
            }
        }

        if (!dictionary.getFolders().containsKey(deliverable.getFolderId()) ||
                dictionary.getFolders().get(deliverable.getFolderId()).getServiceId() != deliverable.getServiceId()) {
            errors.addError("deliverables." + i + ".folderId", TypedError.invalid(messages
                    .getMessage("errors.folder.not.found", null, locale)));
        } else {
            FolderModel folder = dictionary.getFolders().get(deliverable.getFolderId());
            if (folder.isDeleted()) {
                errors.addError("deliverables." + i + ".folderId", TypedError.invalid(messages
                        .getMessage("errors.folder.not.found", null, locale)));
            }
        }

        if (!dictionary.getAccounts().containsKey(deliverable.getAccountId()) ||
                !dictionary.getAccounts().get(deliverable.getAccountId()).getFolderId()
                        .equals(deliverable.getFolderId())) {
            errors.addError("deliverables." + i + ".accountId", TypedError.invalid(messages
                    .getMessage("errors.account.not.found", null, locale)));
        } else {
            AccountModel account = dictionary.getAccounts().get(deliverable.getAccountId());
            if (account.isDeleted()) {
                errors.addError("deliverables." + i + ".accountId", TypedError.invalid(messages
                        .getMessage("errors.account.not.found", null, locale)));
            }
        }
    }

    private boolean isValidServiceState(ServiceWithStatesModel service) {
        boolean isValidState = AbcServiceValidator.ALLOWED_SERVICE_STATES.contains(service.getState());
        boolean isValidReadOnlyState = service.getReadOnlyState() == null ||
                AbcServiceValidator.ALLOWED_SERVICE_READONLY_STATES.contains(service.getReadOnlyState());
        return isValidState && isValidReadOnlyState;
    }

    public Result<ValidatedQuotas> validateQuotas(DeliveryAndProvideRequest request,
                                                  DeliveryQuotas originalQuotas,
                                                  DeliveryAndProvideDictionary dictionary,
                                                  Locale locale) {
        Map<QuotaModel.Key, QuotaModel> updatedQuotas = new HashMap<>();
        ErrorCollection.Builder errors = ErrorCollection.builder();
        Map<QuotaModel.Key, Long> updatedQuotaAmounts = new HashMap<>();
        Map<QuotaModel.Key, Long> updatedFrozenQuota = new HashMap<>();
        for (int i = 0; i < request.getDeliverables().size(); i++) {
            DeliveryAndProvideDestination deliverable = request.getDeliverables().get(i);
            ResourceModel resource = dictionary.getResources().get(deliverable.getResourceId());
            ProviderModel provider = dictionary.getProviders().get(deliverable.getProviderId());
            QuotaModel.Key quotaKey = new QuotaModel.Key(deliverable.getFolderId(), provider.getId(), resource.getId());
            long currentQuotaAmount = updatedQuotaAmounts.computeIfAbsent(quotaKey, k -> {
                if (originalQuotas.getQuotas().containsKey(k)) {
                    QuotaModel quota = originalQuotas.getQuotas().get(k);
                    return quota.getQuota() != null ? quota.getQuota() : 0L;
                }
                return 0L;
            });
            long currentFrozenQuotaAmount = updatedFrozenQuota.computeIfAbsent(quotaKey, k -> {
                if (originalQuotas.getQuotas().containsKey(k)) {
                    QuotaModel quota = originalQuotas.getQuotas().get(k);
                    return quota.getFrozenQuota();
                }
                return 0L;
            });
            UnitsEnsembleModel unitsEnsemble = dictionary.getUnitsEnsembles().get(resource.getUnitsEnsembleId());
            UnitModel unit = unitsEnsemble.unitByKey(deliverable.getDelta().getUnitKey()).orElseThrow();
            Optional<Long> deltaAmountO = Units.convertFromApi(deliverable.getDelta().getAmount(), resource,
                    unitsEnsemble, unit);
            if (deltaAmountO.isEmpty()) {
                errors.addError("deliverables." + i + ".delta.amount", TypedError.invalid(messages
                        .getMessage("errors.value.can.not.be.converted.to.base.unit", null, locale)));
            } else {
                long deltaAmount = deltaAmountO.get();
                Optional<Long> updatedQuotaAmount = Units.add(currentQuotaAmount, deltaAmount);
                Optional<Long> updatedFrozenQuotaAmount = Units.add(currentFrozenQuotaAmount, deltaAmount);
                if (updatedQuotaAmount.isEmpty() || updatedFrozenQuotaAmount.isEmpty()) {
                    errors.addError("deliverables." + i + ".delta.amount", TypedError.invalid(messages
                            .getMessage("errors.number.out.of.range", null, locale)));
                } else {
                    updatedQuotaAmounts.put(quotaKey, updatedQuotaAmount.get());
                    updatedFrozenQuota.put(quotaKey, updatedFrozenQuotaAmount.get());
                }
            }
        }

        Set<QuotaModel.Key> keys = Sets.union(updatedQuotaAmounts.keySet(), updatedFrozenQuota.keySet());
        keys.forEach(key -> {
            long updatedQuotaAmount = updatedQuotaAmounts.getOrDefault(key, 0L);
            long updatedFrozenQuotaAmount = updatedFrozenQuota.getOrDefault(key, 0L);
            QuotaModel.Builder builder;
            if (originalQuotas.getQuotas().containsKey(key)) {
                QuotaModel originalQuota = originalQuotas.getQuotas().get(key);
                builder = QuotaModel.builder(originalQuota)
                        .quota(updatedQuotaAmount)
                        .frozenQuota(updatedFrozenQuotaAmount);
            } else {
                builder = QuotaModel.builder()
                        .tenantId(Tenants.DEFAULT_TENANT_ID)
                        .folderId(key.getFolderId())
                        .providerId(key.getProviderId())
                        .resourceId(key.getResourceId())
                        .quota(updatedQuotaAmount)
                        .balance(0L)
                        .frozenQuota(updatedFrozenQuotaAmount);
            }
            updatedQuotas.put(key, builder.build());
        });
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build());
        }
        return Result.success(new ValidatedQuotas(originalQuotas.getQuotas(), updatedQuotas));
    }

    public Result<ReceivedUpdatedProvision> validateUpdatedProvision(UpdateProvisionResponseDto response,
                                                                     Locale locale) {
        ReceivedUpdatedProvision.Builder builder = ReceivedUpdatedProvision.builder();
        ErrorCollection.Builder errorsBuilder = ErrorCollection.builder();
        response.getProvisions().ifPresent(provisions -> {
            for (int i = 0; i < provisions.size(); i++) {
                ReceivedProvision.Builder provisionBuilder = ReceivedProvision.builder();
                ErrorCollection.Builder provisionErrorsBuilder = ErrorCollection.builder();
                validateProvision(provisions.get(i), provisionBuilder, provisionErrorsBuilder,
                        "provisions." + i + ".", locale);
                if (provisionErrorsBuilder.hasAnyErrors()) {
                    errorsBuilder.add(provisionErrorsBuilder);
                } else {
                    builder.addProvision(provisionBuilder.build());
                }
            }
        });
        response.getAccountVersion().ifPresent(builder::accountVersion);
        response.getAccountsSpaceKey().ifPresent(accountsSpaceKey -> {
            ReceivedAccountsSpaceKey.Builder accountsSpaceKeyBuilder = ReceivedAccountsSpaceKey.builder();
            ErrorCollection.Builder accountsSpaceKeyErrorsBuilder = ErrorCollection.builder();
            validateAccountsSpaceKey(accountsSpaceKey, accountsSpaceKeyBuilder, accountsSpaceKeyErrorsBuilder,
                    "accountsSpaceKey.", locale);
            if (accountsSpaceKeyErrorsBuilder.hasAnyErrors()) {
                errorsBuilder.add(accountsSpaceKeyErrorsBuilder);
            } else {
                builder.accountsSpaceKey(accountsSpaceKeyBuilder.build());
            }
        });
        if (errorsBuilder.hasAnyErrors()) {
            return Result.failure(errorsBuilder.build());
        }
        return Result.success(builder.build());
    }

    private void validateProvision(ProvisionDto provision, ReceivedProvision.Builder provisionBuilder,
                                   ErrorCollection.Builder errorsBuilder, String keyPrefix, Locale locale) {
        if (provision.getResourceKey().isEmpty()) {
            errorsBuilder.addError(keyPrefix + "resourceKey", TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        } else {
            ReceivedResourceKey.Builder resourceKeyBuilder = ReceivedResourceKey.builder();
            ErrorCollection.Builder resourceKeyErrorsBuilder = ErrorCollection.builder();
            validateResourceKey(provision.getResourceKey().get(), resourceKeyBuilder,
                    resourceKeyErrorsBuilder, keyPrefix + "resourceKey.", locale);
            if (resourceKeyErrorsBuilder.hasAnyErrors()) {
                errorsBuilder.add(resourceKeyErrorsBuilder);
            } else {
                provisionBuilder.resourceKey(resourceKeyBuilder.build());
            }
        }
        if (provision.getProvidedAmount().isEmpty()) {
            errorsBuilder.addError(keyPrefix + "providedAmount", TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        } else {
            provisionBuilder.providedAmount(provision.getProvidedAmount().get());
        }
        if (provision.getAllocatedAmount().isEmpty()) {
            errorsBuilder.addError(keyPrefix + "allocatedAmount", TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        } else {
            provisionBuilder.allocatedAmount(provision.getAllocatedAmount().get());
        }
        if (provision.getProvidedAmountUnitKey().isEmpty()) {
            errorsBuilder.addError(keyPrefix + "providedAmountUnitKey", TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        } else {
            provisionBuilder.providedAmountUnitKey(provision.getProvidedAmountUnitKey().get());
        }
        if (provision.getAllocatedAmountUnitKey().isEmpty()) {
            errorsBuilder.addError(keyPrefix + "allocatedAmountUnitKey", TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        } else {
            provisionBuilder.allocatedAmountUnitKey(provision.getAllocatedAmountUnitKey().get());
        }
        provision.getQuotaVersion().ifPresent(provisionBuilder::quotaVersion);
        provision.getLastUpdate().ifPresent(lastUpdate -> {
            ReceivedLastUpdate.Builder lastUpdateBuilder = ReceivedLastUpdate.builder();
            ErrorCollection.Builder lastUpdateErrorsBuilder = ErrorCollection.builder();
            validateLastUpdate(lastUpdate, lastUpdateBuilder, lastUpdateErrorsBuilder,
                    keyPrefix + "lastUpdate.", locale);
            if (lastUpdateErrorsBuilder.hasAnyErrors()) {
                errorsBuilder.add(lastUpdateErrorsBuilder);
            } else {
                provisionBuilder.lastUpdate(lastUpdateBuilder.build());
            }
        });
    }

    private void validateResourceKey(ResourceKeyResponseDto resourceKey,
                                     ReceivedResourceKey.Builder resourceKeyBuilder,
                                     ErrorCollection.Builder errorsBuilder, String keyPrefix, Locale locale) {
        if (resourceKey.getResourceTypeKey().isEmpty() || resourceKey.getResourceTypeKey().get().isEmpty()) {
            errorsBuilder.addError(keyPrefix + "resourceTypeKey", TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        } else {
            resourceKeyBuilder.resourceTypeKey(resourceKey.getResourceTypeKey().get());
        }
        resourceKey.getSegmentation().ifPresent(segmentation -> {
            for (int i = 0; i < segmentation.size(); i++) {
                SegmentKeyResponseDto segmentKey = segmentation.get(i);
                ReceivedSegmentKey.Builder segmentKeyBuilder = ReceivedSegmentKey.builder();
                ErrorCollection.Builder segmentKeyErrorsBuilder = ErrorCollection.builder();
                validateSegmentKey(segmentKey, segmentKeyBuilder, segmentKeyErrorsBuilder,
                        keyPrefix + "segmentation." + i + ".", locale);
                if (segmentKeyErrorsBuilder.hasAnyErrors()) {
                    errorsBuilder.add(segmentKeyErrorsBuilder);
                } else {
                    resourceKeyBuilder.addSegment(segmentKeyBuilder.build());
                }
            }
        });
    }

    private void validateSegmentKey(SegmentKeyResponseDto segmentKey,
                                    ReceivedSegmentKey.Builder segmentKeyBuilder,
                                    ErrorCollection.Builder errorsBuilder, String keyPrefix, Locale locale) {
        if (segmentKey.getSegmentKey().isEmpty() || segmentKey.getSegmentKey().get().isEmpty()) {
            errorsBuilder.addError(keyPrefix + "segmentKey", TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        } else {
            segmentKeyBuilder.segmentKey(segmentKey.getSegmentKey().get());
        }
        if (segmentKey.getSegmentationKey().isEmpty() || segmentKey.getSegmentationKey().get().isEmpty()) {
            errorsBuilder.addError(keyPrefix + "segmentationKey", TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        } else {
            segmentKeyBuilder.segmentationKey(segmentKey.getSegmentationKey().get());
        }
    }

    private void validateLastUpdate(LastUpdateDto lastUpdate,
                                    ReceivedLastUpdate.Builder lastUpdateBuilder,
                                    ErrorCollection.Builder errorsBuilder, String keyPrefix, Locale locale) {
        lastUpdate.getTimestamp().ifPresent(t -> lastUpdateBuilder.timestamp(Instant.ofEpochMilli(t)));
        lastUpdate.getOperationId().ifPresent(lastUpdateBuilder::operationId);
        lastUpdate.getAuthor().ifPresent(userId -> {
            ReceivedUserId.Builder userIdBuilder = ReceivedUserId.builder();
            ErrorCollection.Builder userIdErrorsBuilder = ErrorCollection.builder();
            validateUserId(userId, userIdBuilder, userIdErrorsBuilder, keyPrefix + "author", locale);
            if (userIdErrorsBuilder.hasAnyErrors()) {
                errorsBuilder.add(userIdErrorsBuilder);
            } else {
                lastUpdateBuilder.author(userIdBuilder.build());
            }
        });
    }

    private void validateUserId(UserIdDto userId,
                                ReceivedUserId.Builder userIdBuilder,
                                ErrorCollection.Builder errorsBuilder, String keyPrefix, Locale locale) {
        userId.getStaffLogin().ifPresent(userIdBuilder::staffLogin);
        userId.getPassportUid().ifPresent(userIdBuilder::passportUid);
    }

    private void validateAccountsSpaceKey(AccountsSpaceKeyResponseDto accountsSpaceKey,
                                          ReceivedAccountsSpaceKey.Builder accountsSpaceKeyBuilder,
                                          ErrorCollection.Builder errorsBuilder, String keyPrefix, Locale locale) {
        accountsSpaceKey.getSegmentation().ifPresent(segmentation -> {
            for (int i = 0; i < segmentation.size(); i++) {
                SegmentKeyResponseDto segmentKey = segmentation.get(i);
                ReceivedSegmentKey.Builder segmentKeyBuilder = ReceivedSegmentKey.builder();
                ErrorCollection.Builder segmentKeyErrorsBuilder = ErrorCollection.builder();
                validateSegmentKey(segmentKey, segmentKeyBuilder, segmentKeyErrorsBuilder,
                        keyPrefix + "segmentation." + i + ".", locale);
                if (segmentKeyErrorsBuilder.hasAnyErrors()) {
                    errorsBuilder.add(segmentKeyErrorsBuilder);
                } else {
                    accountsSpaceKeyBuilder.addSegment(segmentKeyBuilder.build());
                }
            }
        });
    }

    public Result<ValidatedReceivedUpdatedProvision> validateReceivedUpdatedProvision(
            ReceivedUpdatedProvision provision, DeliveryAndProvideOperationWithIdModel operationListModel,
            ProvideDictionary provideDictionary, List<UserModel> users, List<AccountsQuotasOperationsModel> operations,
            Locale locale) {
        Map<String, UserModel> usersByUid = users.stream()
                .collect(Collectors.toMap(u -> u.getPassportUid().orElseThrow(), Function.identity()));
        Map<String, UserModel> usersByLogin = users.stream()
                .collect(Collectors.toMap(u -> u.getPassportLogin().orElseThrow(), Function.identity()));
        Map<String, AccountsQuotasOperationsModel> operationsById = operations.stream()
                .collect(Collectors.toMap(AccountsQuotasOperationsModel::getOperationId, Function.identity()));
        ValidatedReceivedUpdatedProvision.Builder builder = ValidatedReceivedUpdatedProvision.builder();
        ErrorCollection.Builder errors = ErrorCollection.builder();
        validateReceivedUpdatedProvision(provision, operationListModel, provideDictionary, usersByUid, usersByLogin,
                operationsById, builder, errors, locale);
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build());
        } else {
            return Result.success(builder.build());
        }
    }

    @SuppressWarnings("ParameterNumber")
    private void validateReceivedUpdatedProvision(ReceivedUpdatedProvision receivedProvision,
                                                  DeliveryAndProvideOperationWithIdModel operationListModel,
                                                  ProvideDictionary provideDictionary,
                                                  Map<String, UserModel> usersByUid,
                                                  Map<String, UserModel> usersByLogin,
                                                  Map<String, AccountsQuotasOperationsModel> operationsById,
                                                  ValidatedReceivedUpdatedProvision.Builder builder,
                                                  ErrorCollection.Builder errors, Locale locale) {
        receivedProvision.getAccountVersion().ifPresent(builder::accountVersion);
        validateAccountsSpace(receivedProvision::getAccountsSpaceKey, operationListModel, provideDictionary,
                builder::accountsSpace, errors, "", locale);
        for (int i = 0; i < receivedProvision.getProvisions().size(); i++) {
            ReceivedProvision provision = receivedProvision.getProvisions().get(i);
            ValidatedReceivedProvision.Builder provisionBuilder = ValidatedReceivedProvision.builder();
            ErrorCollection.Builder provisionErrors = ErrorCollection.builder();
            validateProvision(receivedProvision.getAccountsSpaceKey().orElse(null), provision, usersByUid,
                    usersByLogin, operationsById, provideDictionary, provisionBuilder, provisionErrors,
                    "provisions." + i + ".", locale);
            if (provisionErrors.hasAnyErrors()) {
                errors.add(provisionErrors);
            } else {
                builder.addProvision(provisionBuilder.build());
            }
        }
    }

    private void validateAccountsSpace(Supplier<Optional<ReceivedAccountsSpaceKey>> accountsSpaceKeySupplier,
                                       DeliveryAndProvideOperationWithIdModel operationListModel,
                                       ProvideDictionary provideDictionary,
                                       Consumer<AccountSpaceModel> builder,
                                       ErrorCollection.Builder errors,
                                       String fieldKeyPrefix,
                                       Locale locale) {
        String accountId = operationListModel.getAccountId();
        AccountModel account = provideDictionary.getAccountsById().get(accountId);
        String providerId = account.getProviderId();
        ProviderModel provider = provideDictionary.getProvidersById().get(providerId);
        Optional<ReceivedAccountsSpaceKey> accountsSpaceO = accountsSpaceKeySupplier.get();
        if (accountsSpaceO.isPresent()) {
            if (provider.isAccountsSpacesSupported()) {
                Optional<AccountSpaceModel> accountsSpace = getTargetAccountsSpace(
                        accountsSpaceO.get(), provideDictionary.getAccountsSpaceByExternalKey());
                if (accountsSpace.isPresent()) {
                    Optional<String> accountsSpacesId = account.getAccountsSpacesId();
                    AccountSpaceModel accountSpace =
                            provideDictionary.getAccountSpacesById().get(accountsSpacesId.orElseThrow());
                    if (accountsSpace.get().getId()
                            .equals(accountSpace.getId())) {
                        builder.accept(accountsSpace.get());
                    } else {
                        errors.addError(fieldKeyPrefix + "accountsSpaceKey", TypedError
                                .invalid(messages.getMessage("errors.accounts.space.mismatch", null, locale)));
                    }
                } else {
                    errors.addError(fieldKeyPrefix + "accountsSpaceKey", TypedError
                            .invalid(messages.getMessage("errors.accounts.space.not.found", null, locale)));
                }
            } else {
                errors.addError(fieldKeyPrefix + "accountsSpaceKey", TypedError
                        .invalid(messages.getMessage("errors.accounts.spaces.is.not.supported", null, locale)));
            }
        } else {
            if (provider.isAccountsSpacesSupported()) {
                errors.addError(fieldKeyPrefix + "accountsSpaceKey",
                        TypedError.invalid(messages.getMessage("errors.field.is.required", null, locale)));
            }
        }
    }

    private Optional<AccountSpaceModel> getTargetAccountsSpace(
            ReceivedAccountsSpaceKey accountsSpaceKey,
            Map<ExternalAccountsSpaceKey, AccountSpaceModel> externalAccountsSpaceIndex) {
        List<ReceivedSegmentKey> segments = accountsSpaceKey.getSegmentation();
        Map<String, List<String>> segmentsByKey = segments
                .stream().collect(Collectors.groupingBy(ReceivedSegmentKey::getSegmentationKey,
                        Collectors.mapping(ReceivedSegmentKey::getSegmentKey, Collectors.toList())));
        boolean duplicateSegments = segmentsByKey.values().stream().anyMatch(v -> v.size() > 1);
        if (duplicateSegments) {
            return Optional.empty();
        }
        Set<ExternalSegmentKey> segmentKeys = segments.stream()
                .map(s -> new ExternalSegmentKey(s.getSegmentationKey(),
                        s.getSegmentKey())).collect(Collectors.toSet());
        ExternalAccountsSpaceKey externalKey = new ExternalAccountsSpaceKey(segmentKeys);
        if (!externalAccountsSpaceIndex.containsKey(externalKey)) {
            return Optional.empty();
        }
        return Optional.of(externalAccountsSpaceIndex.get(externalKey));
    }

    @SuppressWarnings("ParameterNumber")
    private void validateProvision(@Nullable ReceivedAccountsSpaceKey accountsSpaceKey,
                                   ReceivedProvision provision,
                                   Map<String, UserModel> usersByUid,
                                   Map<String, UserModel> usersByLogin,
                                   Map<String, AccountsQuotasOperationsModel> operationsById,
                                   ProvideDictionary provideDictionary,
                                   ValidatedReceivedProvision.Builder provisionBuilder,
                                   ErrorCollection.Builder provisionErrors,
                                   String fieldKeyPrefix,
                                   Locale locale) {
        provision.getQuotaVersion().ifPresent(provisionBuilder::quotaVersion);
        if (provision.getLastUpdate().isPresent()) {
            ValidatedReceivedLastUpdate.Builder lastUpdateBuilder = ValidatedReceivedLastUpdate.builder();
            ErrorCollection.Builder lastUpdateErrors = ErrorCollection.builder();
            validateLastUpdate(provision.getLastUpdate().get(), usersByUid, usersByLogin, operationsById,
                    lastUpdateBuilder, lastUpdateErrors, fieldKeyPrefix + "lastUpdate.", locale);
            if (lastUpdateErrors.hasAnyErrors()) {
                provisionErrors.add(lastUpdateErrors);
            } else {
                provisionBuilder.lastUpdate(lastUpdateBuilder.build());
            }
        }
        Optional<ResourceModel> resource = getTargetResource(accountsSpaceKey, provision, provideDictionary);
        if (resource.isEmpty()) {
            provisionErrors.addError(fieldKeyPrefix + "resourceKey",  TypedError.invalid(messages
                    .getMessage("errors.resource.not.found", null, locale)));
            return;
        } else {
            provisionBuilder.resource(resource.get());
        }
        ResourceModel resourceModel = resource.get();
        UnitsEnsembleModel unitsEnsemble =
                provideDictionary.getUnitsEnsemblesById().get(resourceModel.getUnitsEnsembleId());
        Optional<UnitModel> providedUnit = unitsEnsemble
                .unitByKey(provision.getProvidedAmountUnitKey());
        if (providedUnit.isEmpty()) {
            provisionErrors.addError(fieldKeyPrefix + "providedAmountUnitKey",  TypedError.invalid(messages
                    .getMessage("errors.unit.not.found", null, locale)));
        } else {
            Optional<Long> providedAmount = Units.convertFromApi(provision.getProvidedAmount(),
                    resourceModel, unitsEnsemble, providedUnit.get());
            if (providedAmount.isEmpty()) {
                provisionErrors.addError(fieldKeyPrefix + "providedAmount",  TypedError.invalid(messages
                        .getMessage("errors.value.can.not.be.converted.to.base.unit", null, locale)));
            } else {
                provisionBuilder.providedAmount(providedAmount.get());
            }
        }
        Optional<UnitModel> allocatedUnit = unitsEnsemble
                .unitByKey(provision.getAllocatedAmountUnitKey());
        if (allocatedUnit.isEmpty()) {
            provisionErrors.addError(fieldKeyPrefix + "allocatedAmountUnitKey",  TypedError.invalid(messages
                    .getMessage("errors.unit.not.found", null, locale)));
        } else {
            Optional<Long> allocatedAmount = Units.convertFromApi(provision.getAllocatedAmount(),
                    resourceModel, unitsEnsemble, allocatedUnit.get());
            if (allocatedAmount.isEmpty()) {
                provisionErrors.addError(fieldKeyPrefix + "allocatedAmount",  TypedError.invalid(messages
                        .getMessage("errors.value.can.not.be.converted.to.base.unit", null, locale)));
            } else {
                provisionBuilder.allocatedAmount(allocatedAmount.get());
            }
        }
    }

    @SuppressWarnings("ParameterNumber")
    private void validateLastUpdate(ReceivedLastUpdate lastUpdate,
                                    Map<String, UserModel> usersByUid,
                                    Map<String, UserModel> usersByLogin,
                                    Map<String, AccountsQuotasOperationsModel> operationsById,
                                    ValidatedReceivedLastUpdate.Builder lastUpdateBuilder,
                                    ErrorCollection.Builder lastUpdateErrors,
                                    String fieldKeyPrefix,
                                    Locale locale) {
        lastUpdate.getTimestamp().ifPresent(lastUpdateBuilder::timestamp);
        lastUpdate.getOperationId().ifPresent(lastUpdateBuilder::operationId);
        if (lastUpdate.getOperationId().isPresent()) {
            AccountsQuotasOperationsModel operation = operationsById.get(lastUpdate.getOperationId().get());
            if (operation != null) {
                lastUpdateBuilder.operation(operation);
            }
        }
        if (lastUpdate.getAuthor().isPresent() && (lastUpdate.getAuthor().get().getPassportUid().isPresent()
                || lastUpdate.getAuthor().get().getStaffLogin().isPresent())) {
            UserModel authorByUid = null;
            if (lastUpdate.getAuthor().get().getPassportUid().isPresent()) {
                authorByUid = usersByUid.get(lastUpdate.getAuthor().get().getPassportUid().get());
            }
            UserModel authorByLogin = null;
            if (lastUpdate.getAuthor().get().getStaffLogin().isPresent()) {
                authorByLogin = usersByLogin.get(lastUpdate.getAuthor().get().getStaffLogin().get());
            }
            if ((authorByUid == null && authorByLogin == null) || (authorByUid != null
                    && authorByLogin != null && !authorByUid.getId().equals(authorByLogin.getId()))) {
                lastUpdateErrors.addError(fieldKeyPrefix + "author", TypedError
                        .invalid(messages.getMessage("errors.user.not.found", null, locale)));
            } else {
                lastUpdateBuilder.author(authorByUid != null ? authorByUid : authorByLogin);
            }
        }
    }

    private Optional<ResourceModel> getTargetResource(@Nullable ReceivedAccountsSpaceKey accountsSpaceKey,
                                                      ReceivedProvision provision,
                                                      ProvideDictionary provideDictionary) {
        String resourceTypeKey = provision.getResourceKey().getResourceTypeKey();
        List<ReceivedSegmentKey> segments = new ArrayList<>();
        if (accountsSpaceKey != null) {
            segments.addAll(accountsSpaceKey.getSegmentation());
        }
        segments.addAll(provision.getResourceKey().getSegmentation());
        Map<String, List<String>> segmentsByKey = segments
                .stream().collect(Collectors.groupingBy(ReceivedSegmentKey::getSegmentationKey,
                        Collectors.mapping(ReceivedSegmentKey::getSegmentKey, Collectors.toList())));
        boolean duplicateSegments = segmentsByKey.values().stream().anyMatch(v -> v.size() > 1);
        if (duplicateSegments) {
            return Optional.empty();
        }
        Set<ExternalSegmentKey> segmentKeys = segments.stream()
                .map(s -> new ExternalSegmentKey(s.getSegmentationKey(), s.getSegmentKey()))
                .collect(Collectors.toSet());
        ExternalResourceKey externalKey = new ExternalResourceKey(resourceTypeKey, segmentKeys);
        Map<ExternalResourceKey, ResourceModel> resourceByExternalKey = provideDictionary.getResourceByExternalKey();
        if (!resourceByExternalKey.containsKey(externalKey)) {
            return Optional.empty();
        }
        return Optional.of(resourceByExternalKey.get(externalKey));
    }

    public  <R> boolean is5xx(Response<R> response) {
        return response.match(new Response.Cases<>() {
            @Override
            public Boolean success(R result, String requestId) {
                return false;
            }

            @Override
            public Boolean failure(Throwable error) {
                return false;
            }

            @Override
            public Boolean error(ProviderError error, String requestId) {
                return error.match(new ProviderError.Cases<>() {
                    @Override
                    public Boolean httpError(int statusCode) {
                        return is5xx(statusCode);
                    }

                    @Override
                    public Boolean httpExtendedError(int statusCode, ErrorMessagesDto errors) {
                        return is5xx(statusCode);
                    }

                    @Override
                    public Boolean grpcError(Status.Code statusCode, String message) {
                        return is5xx(statusCode);
                    }

                    @Override
                    public Boolean grpcExtendedError(Status.Code statusCode, String message,
                                                     Map<String, String> badRequestDetails) {
                        return is5xx(statusCode);
                    }
                });
            }
        });
    }

    private boolean is5xx(int code) {
        return code >= 500;
    }

    private boolean is5xx(Status.Code code) {
        return INTERNAL.equals(code);
    }

    public <R> boolean isVersionConflict(Response<R> response) {
        return response.match(new Response.Cases<>() {
            @Override
            public Boolean success(R result, String requestId) {
                return false;
            }

            @Override
            public Boolean failure(Throwable error) {
                return false;
            }

            @Override
            public Boolean error(ProviderError error, String requestId) {
                return error.isConflict();
            }
        });
    }

    public  <R> boolean isRetryable(Response<R> updateProvisionResponseDtoResponse) {
        return updateProvisionResponseDtoResponse.match(new Response.Cases<>() {
            @Override
            public Boolean success(R result, String requestId) {
                return false;
            }

            @Override
            public Boolean failure(Throwable error) {
                return false;
            }

            @Override
            public Boolean error(ProviderError error, String requestId) {
                return error.isRetryable();
            }
        });
    }

    public <R> Optional<String> toErrorMessage(Response<R> response) {
        return response.match(new Response.Cases<>() {
            @Override
            public Optional<String> success(R result, String requestId) {
                return Optional.empty();
            }

            @Override
            public Optional<String> failure(Throwable error) {
                return Optional.ofNullable(error.getMessage());
            }

            @Override
            public Optional<String> error(ProviderError error, String requestId) {
                return Optional.ofNullable(error.match(new ProviderError.Cases<>() {
                    @Override
                    public String httpError(int statusCode) {
                        return "HttpError with code " + statusCode;
                    }

                    @Override
                    public String httpExtendedError(int statusCode, ErrorMessagesDto errors) {
                        return errors.getMessage().orElse(null);
                    }

                    @Override
                    public String grpcError(Status.Code statusCode, String message) {
                        return message;
                    }

                    @Override
                    public String grpcExtendedError(Status.Code statusCode, String message,
                                                    Map<String, String> badRequestDetails) {
                        return message;
                    }
                }));
            }
        });
    }

    @SuppressWarnings("ParameterNumber")
    public Result<ValidatedReceivedAccount> validateReceivedAccount(
            ReceivedAccount receivedAccount, List<UserModel> users,
            List<AccountsQuotasOperationsModel> operations,
            @Nullable FolderModel folder,
            @Nullable AccountModel account,
            DeliveryAndProvideOperationWithIdModel listModel,
            ProvideDictionary provideDictionary, Locale locale) {
        Map<String, UserModel> usersByUid = users.stream()
                .collect(Collectors.toMap(u -> u.getPassportUid().get(), Function.identity()));
        Map<String, UserModel> usersByLogin = users.stream()
                .collect(Collectors.toMap(u -> u.getPassportLogin().get(), Function.identity()));
        Map<String, AccountsQuotasOperationsModel> operationsById = operations.stream()
                .collect(Collectors.toMap(AccountsQuotasOperationsModel::getOperationId, Function.identity()));
        ValidatedReceivedAccount.Builder builder = ValidatedReceivedAccount.builder();
        ErrorCollection.Builder errors = ErrorCollection.builder();
        validateReceivedAccount(receivedAccount, listModel, provideDictionary, usersByUid, usersByLogin, operationsById,
                folder, account, builder, errors, "", locale);
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build());
        } else {
            return Result.success(builder.build());
        }
    }

    @SuppressWarnings("ParameterNumber")
    private void validateReceivedAccount(
            ReceivedAccount receivedAccount,
            DeliveryAndProvideOperationWithIdModel listModel,
            ProvideDictionary provideDictionary,
            Map<String, UserModel> usersByUid,
            Map<String, UserModel> usersByLogin,
            Map<String, AccountsQuotasOperationsModel> operationsById,
            @Nullable FolderModel folder,
            @Nullable AccountModel account,
            ValidatedReceivedAccount.Builder builder,
            ErrorCollection.Builder errors,
            String fieldKeyPrefix,
            Locale locale) {
        builder.accountId(receivedAccount.getAccountId());
        receivedAccount.getKey().ifPresent(builder::key);
        receivedAccount.getDisplayName().ifPresent(builder::displayName);
        builder.deleted(receivedAccount.isDeleted());
        if (folder == null) {
            errors.addError(fieldKeyPrefix + "folderId",  TypedError.invalid(messages
                    .getMessage("errors.folder.not.found", null, locale)));
        } else {
            builder.folder(folder);
        }
        receivedAccount.getAccountVersion().ifPresent(builder::accountVersion);
        if (account != null) {
            builder.account(account);
        }
        validateAccountsSpace(receivedAccount::getAccountsSpaceKey, listModel, provideDictionary,
                builder::accountsSpace, errors, fieldKeyPrefix, locale);
        for (int i = 0; i < receivedAccount.getProvisions().size(); i++) {
            ReceivedProvision provision = receivedAccount.getProvisions().get(i);
            ValidatedReceivedProvision.Builder provisionBuilder = ValidatedReceivedProvision.builder();
            ErrorCollection.Builder provisionErrors = ErrorCollection.builder();
            validateProvision(receivedAccount.getAccountsSpaceKey().orElse(null), provision, usersByUid,
                    usersByLogin, operationsById, provideDictionary, provisionBuilder, provisionErrors,
                    fieldKeyPrefix + "provisions." + i + ".", locale);
            if (provisionErrors.hasAnyErrors()) {
                errors.add(provisionErrors);
            } else {
                builder.addProvision(provisionBuilder.build());
            }
        }
        if (receivedAccount.getLastUpdate().isPresent()) {
            ValidatedReceivedLastUpdate.Builder lastUpdateBuilder = ValidatedReceivedLastUpdate.builder();
            ErrorCollection.Builder lastUpdateErrors = ErrorCollection.builder();
            validateLastUpdate(receivedAccount.getLastUpdate().get(), usersByUid, usersByLogin, operationsById,
                    lastUpdateBuilder, lastUpdateErrors, fieldKeyPrefix + "lastUpdate.", locale);
            if (lastUpdateErrors.hasAnyErrors()) {
                errors.add(lastUpdateErrors);
            } else {
                builder.lastUpdate(lastUpdateBuilder.build());
            }
        }
        builder.freeTier(receivedAccount.isFreeTier());
    }

    public Result<ReceivedAccount> validateAccount(AccountDto account, Locale locale) {
        ReceivedAccount.Builder accountBuilder = ReceivedAccount.builder();
        ErrorCollection.Builder errorsBuilder = ErrorCollection.builder();
        validateAccount(account, accountBuilder, errorsBuilder, "", locale);
        if (errorsBuilder.hasAnyErrors()) {
            return Result.failure(errorsBuilder.build());
        }
        return Result.success(accountBuilder.build());
    }

    private void validateAccount(AccountDto account, ReceivedAccount.Builder accountBuilder,
                                 ErrorCollection.Builder errorsBuilder, String keyPrefix, Locale locale) {
        if (account.getAccountId().isEmpty() || account.getAccountId().get().isEmpty()) {
            errorsBuilder.addError(keyPrefix + "accountId", TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        } else {
            accountBuilder.accountId(account.getAccountId().get());
        }
        account.getKey().ifPresent(accountBuilder::key);
        account.getDisplayName().ifPresent(accountBuilder::displayName);
        if (account.getFolderId().isEmpty() || account.getFolderId().get().isEmpty()) {
            errorsBuilder.addError(keyPrefix + "folderId", TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        } else {
            if (!Uuids.isValidUuid(account.getFolderId().get())) {
                errorsBuilder.addError(keyPrefix + "folderId", TypedError.invalid(messages
                        .getMessage("errors.folder.not.found", null, locale)));
            } else {
                accountBuilder.folderId(account.getFolderId().get());
            }
        }
        accountBuilder.deleted(account.getDeleted().orElse(false));
        account.getProvisions().ifPresent(provisions -> {
            for (int i = 0; i < provisions.size(); i++) {
                ReceivedProvision.Builder provisionBuilder = ReceivedProvision.builder();
                ErrorCollection.Builder provisionErrorsBuilder = ErrorCollection.builder();
                validateProvision(provisions.get(i), provisionBuilder, provisionErrorsBuilder,
                        keyPrefix + "provisions." + i + ".", locale);
                if (provisionErrorsBuilder.hasAnyErrors()) {
                    errorsBuilder.add(provisionErrorsBuilder);
                } else {
                    accountBuilder.addProvision(provisionBuilder.build());
                }
            }
        });
        account.getAccountVersion().ifPresent(accountBuilder::accountVersion);
        account.getLastUpdate().ifPresent(lastUpdate -> {
            ReceivedLastUpdate.Builder lastUpdateBuilder = ReceivedLastUpdate.builder();
            ErrorCollection.Builder lastUpdateErrorsBuilder = ErrorCollection.builder();
            validateLastUpdate(lastUpdate, lastUpdateBuilder, lastUpdateErrorsBuilder,
                    keyPrefix + "lastUpdate.", locale);
            if (lastUpdateErrorsBuilder.hasAnyErrors()) {
                errorsBuilder.add(lastUpdateErrorsBuilder);
            } else {
                accountBuilder.lastUpdate(lastUpdateBuilder.build());
            }
        });
        account.getAccountsSpaceKey().ifPresent(accountsSpaceKey -> {
            ReceivedAccountsSpaceKey.Builder accountsSpaceKeyBuilder = ReceivedAccountsSpaceKey.builder();
            ErrorCollection.Builder accountsSpaceKeyErrorsBuilder = ErrorCollection.builder();
            validateAccountsSpaceKey(accountsSpaceKey, accountsSpaceKeyBuilder, accountsSpaceKeyErrorsBuilder,
                    keyPrefix + "accountsSpaceKey.", locale);
            if (accountsSpaceKeyErrorsBuilder.hasAnyErrors()) {
                errorsBuilder.add(accountsSpaceKeyErrorsBuilder);
            } else {
                accountBuilder.accountsSpaceKey(accountsSpaceKeyBuilder.build());
            }
        });
    }

    public boolean isProvisionChangeApplied(ValidatedReceivedAccount receivedAccount, UpdateDictionary dict,
                                            ProvideDictionary provideDict) {
        AccountModel account = dict.getAccount();
        AccountsQuotasOperationsModel operation =
                provideDict.getAccountsQuotasOperationsByAccountId().get(account.getId());
        ProviderModel provider = provideDict.getProvidersById().get(account.getProviderId());
        AccountsSettingsModel accountsSettings = provider.getAccountsSettings();
        boolean provisionOperationIdSupported = accountsSettings.isPerProvisionLastUpdateSupported();
        boolean accountAndProvisionsVersionedTogether = accountsSettings.isPerAccountVersionSupported()
                && !accountsSettings.isPerProvisionVersionSupported();
        boolean provisionsVersionedSeparately = accountsSettings.isPerProvisionVersionSupported();
        boolean hasProvisionOperationIds = receivedAccount.getProvisions().stream()
                .allMatch(p -> p.getLastUpdate().isPresent() && p.getLastUpdate().get().getOperationId().isPresent());
        boolean hasCommonVersion = receivedAccount.getAccountVersion().isPresent();
        boolean hasProvisionVersions = receivedAccount.getProvisions().stream()
                .allMatch(p -> p.getQuotaVersion().isPresent());
        if (provisionOperationIdSupported && hasProvisionOperationIds) {
            return hasOperationId(receivedAccount, operation, dict, accountAndProvisionsVersionedTogether,
                    provisionsVersionedSeparately);
        }
        if (accountAndProvisionsVersionedTogether && hasCommonVersion) {
            return isExactValuesMatchAndFreshCommonVersion(receivedAccount, operation, dict);
        }
        if (provisionsVersionedSeparately && hasProvisionVersions) {
            return isExactValuesMatchAndFreshProvisionVersions(receivedAccount, operation, dict);
        }
        return isExactValuesMatch(receivedAccount, operation, dict);
    }

    private boolean hasOperationId(ValidatedReceivedAccount receivedAccount,
                                   AccountsQuotasOperationsModel operation,
                                   UpdateDictionary updateDictionary,
                                   boolean accountAndProvisionsVersionedTogether,
                                   boolean provisionsVersionedSeparately) {
        AccountModel account = updateDictionary.getAccount();
        if (accountAndProvisionsVersionedTogether) {
            Optional<Long> knownVersionO = account.getLastReceivedVersion();
            Optional<Long> receivedVersionO = receivedAccount.getAccountVersion();
            if (knownVersionO.isPresent() && receivedVersionO.isPresent()) {
                if (knownVersionO.get() >= receivedVersionO.get()) {
                    return false;
                }
            }
            if (isMismatchForRemainingResources(receivedAccount, operation, updateDictionary)) {
                return false;
            }
        }
        Map<String, OperationChangesModel.Provision> targetProvisionByResourceId = operation.getRequestedChanges()
                .getUpdatedProvisions().orElse(List.of()).stream()
                .collect(Collectors.toMap(OperationChangesModel.Provision::getResourceId, Function.identity()));
        Set<String> targetResourceIds = targetProvisionByResourceId.keySet();
        Map<String, ValidatedReceivedProvision> receivedProvisionsByResourceId = receivedAccount.getProvisions()
                .stream().collect(Collectors.toMap(v -> v.getResource().getId(), Function.identity()));
        if (provisionsVersionedSeparately) {
            boolean versionMatch = targetResourceIds.stream().allMatch(resourceId -> {
                AccountsQuotasModel knownProvision = updateDictionary.getProvisionByResourceIdByAccount()
                        .getOrDefault(account.getId(), Map.of())
                        .get(resourceId);
                ValidatedReceivedProvision receivedProvision = receivedProvisionsByResourceId.get(resourceId);
                Optional<Long> knownVersionO = knownProvision != null
                        ? knownProvision.getLastReceivedProvisionVersion() : Optional.empty();
                Optional<Long> receivedVersionO = receivedProvision != null
                        ? receivedProvision.getQuotaVersion() : Optional.empty();
                if (knownVersionO.isPresent() && receivedVersionO.isPresent()) {
                    return knownVersionO.get() < receivedVersionO.get();
                }
                return true;
            });
            if (!versionMatch) {
                return false;
            }
        }
        String knownFolderId = updateDictionary.getFolder().getId();
        String receivedFolderId = receivedAccount.getFolder().getId();
        boolean foldersMatch = knownFolderId.equals(receivedFolderId);
        boolean knownDeletion = updateDictionary.getAccount().isDeleted();
        boolean receivedDeletion = receivedAccount.isDeleted();
        boolean deletionMatch = knownDeletion == receivedDeletion;
        String targetOperationId = operation.getOperationId();
        boolean opIdMatch = targetResourceIds.stream().allMatch(resourceId -> {
            ValidatedReceivedProvision receivedProvision = receivedProvisionsByResourceId.get(resourceId);
            OperationChangesModel.Provision targetProvision = targetProvisionByResourceId.get(resourceId);
            if (receivedProvision == null) {
                return targetProvision.getAmount() == 0L;
            }
            return receivedProvision.getLastUpdate().flatMap(ValidatedReceivedLastUpdate::getOperationId)
                    .map(targetOperationId::equals).orElse(false);
        });
        return opIdMatch && foldersMatch && deletionMatch;
    }

    private boolean isMismatchForRemainingResources(ValidatedReceivedAccount receivedAccount,
                                                    AccountsQuotasOperationsModel operation,
                                                    UpdateDictionary updateDictionary) {
        Map<String, ValidatedReceivedProvision> receivedProvisionsByResourceId = receivedAccount.getProvisions()
                .stream().collect(Collectors.toMap(v -> v.getResource().getId(), Function.identity()));
        Set<String> targetResourceIds = operation.getRequestedChanges().getUpdatedProvisions()
                .orElse(List.of()).stream().map(OperationChangesModel.Provision::getResourceId)
                .collect(Collectors.toSet());
        Set<String> receivedResourceIds = receivedProvisionsByResourceId.keySet();
        AccountModel account = updateDictionary.getAccount();
        Set<String> knowResourceIds = updateDictionary.getProvisionByResourceIdByAccount()
                .getOrDefault(account.getId(), Map.of()).keySet();
        Set<String> remainingResourceIds = Sets.difference(Sets.union(knowResourceIds, receivedResourceIds),
                targetResourceIds);
        return remainingResourceIds.stream().anyMatch(resourceId -> {
            ValidatedReceivedProvision receivedProvision = receivedProvisionsByResourceId.get(resourceId);
            AccountsQuotasModel knownProvision = updateDictionary.getProvisionByResourceIdByAccount()
                    .getOrDefault(account.getId(), Map.of()).get(resourceId);
            long receivedAmount = receivedProvision != null ? receivedProvision.getProvidedAmount() : 0L;
            long knownAmount = knownProvision != null ? (knownProvision.getProvidedQuota() != null
                    ? knownProvision.getProvidedQuota() : 0L) : 0L;
            return receivedAmount != knownAmount;
        });
    }

    private boolean isExactValuesMatchAndFreshCommonVersion(
            ValidatedReceivedAccount receivedAccount,
            AccountsQuotasOperationsModel operation,
            UpdateDictionary updateDictionary) {
        Optional<Long> receivedVersionO = receivedAccount.getAccountVersion();
        Optional<Long> knownVersionO = updateDictionary.getAccount().getLastReceivedVersion();
        if (receivedVersionO.isPresent() && knownVersionO.isPresent()
                && receivedVersionO.get() <= knownVersionO.get()) {
            return false;
        }
        if (isMismatchForRemainingResources(receivedAccount, operation, updateDictionary)) {
            return false;
        }
        return isExactValuesMatch(receivedAccount, operation, updateDictionary);
    }

    private boolean isExactValuesMatch(ValidatedReceivedAccount receivedAccount,
                                       AccountsQuotasOperationsModel operation,
                                       UpdateDictionary updateDictionary) {
        List<OperationChangesModel.Provision> targetProvisions = operation.getRequestedChanges()
                .getUpdatedProvisions().orElse(List.of());
        Map<String, ValidatedReceivedProvision> receivedProvisionsByResourceId = receivedAccount.getProvisions()
                .stream().collect(Collectors.toMap(v -> v.getResource().getId(), Function.identity()));
        boolean valuesMatch = targetProvisions.stream().allMatch(targetProvision -> {
            String targetResourceId = targetProvision.getResourceId();
            long targetAmount = targetProvision.getAmount();
            ValidatedReceivedProvision receivedProvision = receivedProvisionsByResourceId.get(targetResourceId);
            long receivedAmount = receivedProvision != null ? receivedProvision.getProvidedAmount() : 0L;
            return targetAmount == receivedAmount;
        });
        String knownFolderId = updateDictionary.getFolder().getId();
        String receivedFolderId = receivedAccount.getFolder().getId();
        boolean foldersMatch = knownFolderId.equals(receivedFolderId);
        boolean knownDeletion = updateDictionary.getAccount().isDeleted();
        boolean receivedDeletion = receivedAccount.isDeleted();
        boolean deletionMatch = knownDeletion == receivedDeletion;
        return valuesMatch && foldersMatch && deletionMatch;
    }

    private boolean isExactValuesMatchAndFreshProvisionVersions(
            ValidatedReceivedAccount receivedAccount,
            AccountsQuotasOperationsModel operation,
            UpdateDictionary updateDictionary) {
        String accountId = updateDictionary.getAccount().getId();
        Map<String, AccountsQuotasModel> knownProvisions = updateDictionary
                .getProvisionByResourceIdByAccount().getOrDefault(accountId, Map.of());
        boolean receivedVersionsAreFresh = receivedAccount.getProvisions().stream().allMatch(receivedProvision -> {
            Optional<Long> receivedVersionO = receivedProvision.getQuotaVersion();
            AccountsQuotasModel knownProvision = knownProvisions.get(receivedProvision.getResource().getId());
            Optional<Long> knownVersionO = knownProvision != null
                    ? knownProvision.getLastReceivedProvisionVersion() : Optional.empty();
            if (receivedVersionO.isPresent() && knownVersionO.isPresent()
                    && receivedVersionO.get() <= knownVersionO.get()) {
                return false;
            }
            return true;
        });
        if (!receivedVersionsAreFresh) {
            return false;
        }
        return isExactValuesMatch(receivedAccount, operation, updateDictionary);
    }
}
