package ru.yandex.intranet.d.services.elements;

import java.util.Locale;
import java.util.Optional;
import java.util.UUID;
import java.util.function.Function;

import javax.annotation.Nullable;

import com.yandex.ydb.table.transaction.TransactionMode;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuples;

import ru.yandex.intranet.d.dao.folders.FolderDao;
import ru.yandex.intranet.d.datasource.model.WithTxId;
import ru.yandex.intranet.d.datasource.model.YdbQueryChain;
import ru.yandex.intranet.d.datasource.model.YdbQueryChain.Validator;
import ru.yandex.intranet.d.datasource.model.YdbTableClient;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.folders.FolderModel;
import ru.yandex.intranet.d.model.folders.FolderType;
import ru.yandex.intranet.d.services.validators.AbcServiceValidator;
import ru.yandex.intranet.d.util.paging.Page;
import ru.yandex.intranet.d.util.result.Result;

/**
 * Useful methods for service to operate with folders.
 *
 * @author Vladimir Zaytsev <vzay@yandex-team.ru>
 * @since 16.09.2020
 */
@Component
public class FolderServiceElement {
    private final YdbTableClient tableClient;
    private final StringContinuationTokenPager pager;
    private final FolderDao folderDao;
    private final YdbQueryChain<GetOrCreateDefaultFolderQueryParams, FolderModel> getOrCreateDefaultFolderQueryChain;
    private final YdbQueryChain<ListFoldersByServiceParams, Page<FolderModel>> listFoldersByServiceQueryChain;
    private final AbcServiceValidator abcServiceValidator;

    public FolderServiceElement(
            YdbTableClient tableClient,
            StringContinuationTokenPager pager,
            FolderDao folderDao,
            AbcServiceValidator abcServiceValidator
    ) {
        this.tableClient = tableClient;
        this.pager = pager;
        this.folderDao = folderDao;
        this.abcServiceValidator = abcServiceValidator;
        this.getOrCreateDefaultFolderQueryChain = getOrCreateDefaultFolderQueryChain();
        this.listFoldersByServiceQueryChain = listFoldersByServiceQueryChain();
    }

    public Mono<Result<FolderModel>> getOrCreateDefaultFolder(TenantId tenantId, long serviceId, Locale locale) {
        return tableClient.usingSessionMonoRetryable(session ->
                getOrCreateDefaultFolderQueryChain.validateAndExecute(session, TransactionMode.SERIALIZABLE_READ_WRITE,
                        new GetOrCreateDefaultFolderQueryParams(tenantId, serviceId),
                        getOrCreateDefaultFolderValidator(locale)
                ));
    }

    public <T extends HasServiceId> Validator<T> getOrCreateDefaultFolderValidator(Locale locale) {
        return (ts, params) -> abcServiceValidator
                .validateAbcService(Optional.of(params.getServiceId()), locale, ts, "serviceId")
                .map(r -> r.apply(ignored -> params));
    }

    public Mono<Result<Page<FolderModel>>> listFoldersByService(
            TenantId tenantId,
            long serviceId,
            boolean includeDeleted,
            int limit,
            @Nullable String fromId,
            Locale locale
    ) {
        return tableClient.usingSessionMonoRetryable(session ->
                listFoldersByServiceQueryChain.validateAndExecute(session, TransactionMode.SERIALIZABLE_READ_WRITE,
                        new ListFoldersByServiceParams(
                                tenantId,
                                serviceId,
                                includeDeleted,
                                limit,
                                fromId
                        ),
                        getOrCreateDefaultFolderValidator(locale)
                )
        );
    }

    private YdbQueryChain<GetOrCreateDefaultFolderQueryParams, FolderModel> getOrCreateDefaultFolderQueryChain() {
        return YdbQueryChain
                .start((YdbQueryChain.Stage<GetOrCreateDefaultFolderQueryParams, ParamsAndModel>) (ts, params) ->
                        folderDao.getDefaultFolderTx(ts, params.tenantId, params.serviceId)
                                .map(d -> d.map(w -> new ParamsAndModel(params, w))))
                .append((ts, pm) -> pm.model
                        .map(folderModel -> new WithTxId<>(folderModel, ts.getId()))
                        .map(folderModelAndTxId ->
                                ts.getTxControl().isCommitTx() ?
                                        ts.commitTransaction().thenReturn(folderModelAndTxId.toClosed()) :
                                        Mono.just(folderModelAndTxId)
                        )
                        .orElseGet(() ->
                                folderDao.upsertOneTxRetryable(ts, FolderModel.newBuilder()
                                        .setTenantId(pm.params.tenantId)
                                        .setId(UUID.randomUUID().toString())
                                        .setServiceId(pm.params.serviceId)
                                        .setFolderType(FolderType.COMMON_DEFAULT_FOR_SERVICE)
                                        .setDisplayName("default")
                                        .setNextOpLogOrder(1L)
                                        .build()
                                )
                        )
                );
    }

    private YdbQueryChain<ListFoldersByServiceParams, Page<FolderModel>> listFoldersByServiceQueryChain() {
        return YdbQueryChain
                .start((YdbTxSession ts, ListFoldersByServiceParams params) ->
                        folderDao.listFoldersByServiceIdWithoutDefaultTx(
                                ts,
                                params.tenantId,
                                params.serviceId,
                                params.includeDeleted,
                                params.limit + 1,
                                params.fromId
                        ).map(r -> r.map(folders -> Tuples.of(params,
                                pager.toPage(folders, params.limit, FolderModel::getId)
                        )))
                )
                .combine(getOrCreateDefaultFolderQueryChain,
                        t -> {
                            ListFoldersByServiceParams params = t.getT1();
                            return new GetOrCreateDefaultFolderQueryParams(params.tenantId, params.serviceId);
                        },
                        (t, defaultFolder) -> {
                            ListFoldersByServiceParams params = t.getT1();
                            Page<FolderModel> page = t.getT2();
                            Function<FolderModel, String> idGetter = params.limit > 1 ?
                                    FolderModel::getId :
                                    ignored -> "";
                            return params.fromId == null ?
                                    pager.addFirst(page, defaultFolder, params.limit, idGetter) :
                                    page;
                        }
                );
    }

    interface HasServiceId {
        long getServiceId();
    }

    public static class GetOrCreateDefaultFolderQueryParams implements HasServiceId {
        private final TenantId tenantId;
        private final long serviceId;

        private GetOrCreateDefaultFolderQueryParams(TenantId tenantId, long serviceId) {
            this.tenantId = tenantId;
            this.serviceId = serviceId;
        }

        @Override
        public long getServiceId() {
            return serviceId;
        }
    }


    public static class ListFoldersByServiceParams implements HasServiceId {
        private final TenantId tenantId;
        private final long serviceId;
        private final boolean includeDeleted;
        private final int limit;
        @Nullable
        private final String fromId;

        private ListFoldersByServiceParams(
                TenantId tenantId,
                long serviceId,
                boolean includeDeleted,
                int limit,
                @Nullable String fromId
        ) {
            this.tenantId = tenantId;
            this.serviceId = serviceId;
            this.includeDeleted = includeDeleted;
            this.limit = limit;
            this.fromId = fromId;
        }

        @Override
        public long getServiceId() {
            return serviceId;
        }
    }

    private static class ParamsAndModel {
        private final GetOrCreateDefaultFolderQueryParams params;
        private final Optional<FolderModel> model;

        private ParamsAndModel(GetOrCreateDefaultFolderQueryParams params, Optional<FolderModel> model) {
            this.params = params;
            this.model = model;
        }
    }
}
