package ru.yandex.intranet.d.services.elements;

import java.util.Arrays;
import java.util.List;
import java.util.Locale;
import java.util.function.Function;

import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import org.springframework.stereotype.Component;

import ru.yandex.intranet.d.services.validators.TextValidator;
import ru.yandex.intranet.d.util.JsonReader;
import ru.yandex.intranet.d.util.JsonWriter;
import ru.yandex.intranet.d.util.ObjectMapperHolder;
import ru.yandex.intranet.d.util.paging.ContinuationTokens;
import ru.yandex.intranet.d.util.paging.Page;
import ru.yandex.intranet.d.util.paging.PageRequest;
import ru.yandex.intranet.d.util.paging.StringContinuationToken;
import ru.yandex.intranet.d.util.result.Result;

/**
 * Convert models list to page with StringContinuationToken.
 *
 * @author Vladimir Zaytsev <vzay@yandex-team.ru>
 * @since 17.09.2020
 */
@Component
public class StringContinuationTokenPager {
    private final MessageSource messages;
    private final TextValidator textValidator;
    private final JsonReader<StringContinuationToken> continuationTokenReader;
    private final JsonWriter<StringContinuationToken> continuationTokenWriter;

    public StringContinuationTokenPager(
            @Qualifier("messageSource") MessageSource messages,
            TextValidator textValidator,
            @Qualifier("continuationTokensJsonObjectMapper") ObjectMapperHolder objectMapper
    ) {
        this.messages = messages;
        this.textValidator = textValidator;
        this.continuationTokenReader = new JsonReader<>(objectMapper.getObjectMapper(), StringContinuationToken.class);
        this.continuationTokenWriter = new JsonWriter<>(objectMapper.getObjectMapper(), StringContinuationToken.class);
    }

    public <M> Page<M> toPage(
            List<M> values, int limit, Function<M, String> idGetter
    ) {
        return ContinuationTokens.toPage(
                values, limit, m -> new StringContinuationToken(idGetter.apply(m)), continuationTokenWriter
        );
    }

    public <M> Page<M> addFirst(Page<M> page, M item, int limit, Function<M, String> idGetter) {
        List<M> items = page.getItems();
        int size = items.size();
        //noinspection unchecked
        M[] arr = (M[]) new Object[size + 1];
        arr = items.toArray(arr);
        System.arraycopy(arr, 0, arr, 1, size);
        arr[0] = item;
        return toPage(Arrays.asList(arr), limit, idGetter);
    }

    public Result<PageRequest.Validated<StringContinuationToken>> validatePageRequest(
            PageRequest pageRequest, Locale locale
    ) {
        return pageRequest.validate(continuationTokenReader, messages, locale).andThen(validated -> validated
                .getContinuationToken()
                .map(t -> validateContinuationToken(locale, t))
                .orElse(Result.success(null))
                .apply(u -> validated));
    }

    private Result<Void> validateContinuationToken(Locale locale, StringContinuationToken t) {
        if (t.getId() != null && t.getId().isEmpty()) {
            return Result.success(null);
        }
        return textValidator.validateId(t.getId(), locale, "errors.invalid.continuation.token");
    }
}
