package ru.yandex.intranet.d.services.folders;

import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;

import com.google.common.collect.Sets;
import com.yandex.ydb.table.transaction.TransactionMode;
import org.apache.commons.lang3.mutable.MutableLong;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuple2;
import reactor.util.function.Tuples;

import ru.yandex.intranet.d.dao.Tenants;
import ru.yandex.intranet.d.dao.folders.FolderDao;
import ru.yandex.intranet.d.dao.providers.ProvidersDao;
import ru.yandex.intranet.d.dao.services.ServicesDao;
import ru.yandex.intranet.d.datasource.model.WithTxId;
import ru.yandex.intranet.d.datasource.model.YdbSession;
import ru.yandex.intranet.d.datasource.model.YdbTableClient;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.WithTenant;
import ru.yandex.intranet.d.model.folders.FolderModel;
import ru.yandex.intranet.d.model.folders.FolderType;
import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.model.users.UserModel;
import ru.yandex.intranet.d.services.elements.FolderServiceElement;
import ru.yandex.intranet.d.services.security.SecurityManagerService;
import ru.yandex.intranet.d.services.validators.AbcServiceValidator;
import ru.yandex.intranet.d.services.validators.TextValidator;
import ru.yandex.intranet.d.services.validators.VersionValidator;
import ru.yandex.intranet.d.util.JsonReader;
import ru.yandex.intranet.d.util.JsonWriter;
import ru.yandex.intranet.d.util.ObjectMapperHolder;
import ru.yandex.intranet.d.util.Uuids;
import ru.yandex.intranet.d.util.paging.ContinuationTokens;
import ru.yandex.intranet.d.util.paging.Page;
import ru.yandex.intranet.d.util.paging.PageRequest;
import ru.yandex.intranet.d.util.paging.StringContinuationToken;
import ru.yandex.intranet.d.util.result.ErrorCollection;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.util.result.TypedError;
import ru.yandex.intranet.d.web.model.folders.FrontFolderInputDto;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;

import static ru.yandex.intranet.d.dao.Tenants.getTenantId;

/**
 * Folder service implementation.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class FolderService {
    private static final int MAX_NAME_LENGTH = 255;
    private static final int MAX_DESCRIPTION_LENGTH = 1023;
    private static final int MAX_TAG_LENGTH = 127;
    private static final int BATCH_SIZE = 100;

    private final YdbTableClient tableClient;
    private final MessageSource messages;
    private final FolderDao folderDao;
    private final ServicesDao servicesDao;
    private final AbcServiceValidator abcServiceValidator;
    private final TextValidator textValidator;
    private final VersionValidator versionValidator;
    private final SecurityManagerService securityManagerService;
    private final JsonReader<StringContinuationToken> continuationTokenReader;
    private final JsonWriter<StringContinuationToken> continuationTokenWriter;
    private final FolderServiceElement folderServiceElement;
    private final ProvidersDao providersDao;

    @SuppressWarnings("checkstyle:ParameterNumber")
    public FolderService(
            YdbTableClient tableClient,
            @Qualifier("messageSource") MessageSource messages,
            FolderDao folderDao,
            ServicesDao servicesDao,
            AbcServiceValidator abcServiceValidator,
            TextValidator textValidator,
            VersionValidator versionValidator,
            SecurityManagerService securityManagerService,
            @Qualifier("continuationTokensJsonObjectMapper") ObjectMapperHolder objectMapper,
            FolderServiceElement folderServiceElement,
            ProvidersDao providersDao
    ) {
        this.tableClient = tableClient;
        this.messages = messages;
        this.folderDao = folderDao;
        this.servicesDao = servicesDao;
        this.versionValidator = versionValidator;
        this.securityManagerService = securityManagerService;
        this.continuationTokenReader = new JsonReader<>(objectMapper.getObjectMapper(), StringContinuationToken.class);
        this.continuationTokenWriter = new JsonWriter<>(objectMapper.getObjectMapper(), StringContinuationToken.class);
        this.abcServiceValidator = abcServiceValidator;
        this.textValidator = textValidator;
        this.folderServiceElement = folderServiceElement;
        this.providersDao = providersDao;
    }

    public Mono<Result<FolderModel>> getFolder(String folderId, YaUserDetails currentUser, Locale locale) {
        TenantId tenantId = getTenantId(currentUser);
        return securityManagerService.checkReadPermissions(currentUser, locale).flatMap(res -> res.andThenMono(u ->
                tableClient.usingSessionMonoRetryable(session ->
                        folderDao.getById(
                                session.asTxCommitRetryable(TransactionMode.ONLINE_READ_ONLY), folderId, tenantId
                        ).map(folder -> folder.isEmpty() || folder.get().isDeleted() ? notFound(locale) :
                                Result.success(folder.get())
                        ).flatMap(result -> result.andThenMono(folder ->
                                securityManagerService.checkReadPermissions(
                                        folder.getServiceId(), currentUser, locale, folder
                                ))
                        )
                )));
    }

    public Mono<Result<List<FolderModel>>> getFolders(List<String> folderIds, boolean withDeleted,
                                                      YaUserDetails currentUser, Locale locale) {
        if (folderIds.size() > 1000) {
            return Mono.just(Result.failure(ErrorCollection.builder()
                    .addError(TypedError.badRequest(messages
                            .getMessage("errors.limit.is.too.large", null, locale)))
                    .build()));
        }

        TenantId tenantId = getTenantId(currentUser);
        return securityManagerService.checkReadPermissions(currentUser, locale).flatMap(res -> res.andThenMono(u ->
                tableClient.usingSessionMonoRetryable(session -> folderDao.getByIds(
                        session.asTxCommitRetryable(TransactionMode.ONLINE_READ_ONLY), folderIds, tenantId)
                ).map(folders -> withDeleted ? folders : folders.stream()
                        .filter(f -> !f.isDeleted()).collect(Collectors.toList())
                ).flatMap(folders -> securityManagerService.checkReadPermissions(folders.stream()
                        .map(FolderModel::getId).collect(Collectors.toList()), currentUser, locale, folders))
        ));
    }

    public Mono<Result<FolderModel>> getReservedFolderOfService(long serviceId,
                                                                YaUserDetails currentUser,
                                                                Locale locale) {
        TenantId tenantId = getTenantId(currentUser);
        return securityManagerService.checkReadPermissions(currentUser, locale).flatMap(res -> res.andThenMono(u ->
                tableClient.usingSessionMonoRetryable(session ->
                        folderDao.getAllReservedFoldersByServiceIds(
                                session.asTxCommitRetryable(TransactionMode.ONLINE_READ_ONLY),
                                Collections.singletonList(new WithTenant<>(tenantId, serviceId)))
                ).flatMap(folders -> {
                    if (folders.isEmpty()) {
                        return Mono.just(Result.failure(ErrorCollection.builder().addError(
                                TypedError.notFound(messages
                                        .getMessage("errors.provider.reserved.folder.not.found",
                                                null,
                                                locale)))
                                .build()));
                    }
                    FolderModel folder = folders.get(0);
                    return securityManagerService.checkReadPermissions(folder.getId(), currentUser, locale, folder);
                })));
    }

    private Result<FolderModel> notFound(Locale locale) {
        return Result.failure(ErrorCollection.builder().addError(
                TypedError.notFound(messages
                        .getMessage("errors.folder.not.found", null, locale)))
                .build());
    }

    public Mono<Result<Page<FolderModel>>> listFoldersByService(
            long serviceId,
            boolean includeDeleted,
            PageRequest pageRequest,
            YaUserDetails currentUser,
            Locale locale
    ) {
        return securityManagerService.checkReadPermissions(serviceId, currentUser, locale)
                .then(pageRequest.validate(continuationTokenReader, messages, locale)
                        .andThenMono((PageRequest.Validated<StringContinuationToken> p) -> {
                            int limit = p.getLimit();
                            String fromId = p.getContinuationToken().map(StringContinuationToken::getId).orElse(null);
                            return folderServiceElement.listFoldersByService(
                                    getTenantId(currentUser),
                                    serviceId,
                                    includeDeleted,
                                    limit,
                                    fromId,
                                    locale
                            );
                        })
                );
    }

    public Mono<Result<FolderModel>> createFolder(
            FrontFolderInputDto folder,
            YaUserDetails currentUser,
            Locale locale
    ) {
        if (currentUser == null || !currentUser.getUser().map(UserModel::getDAdmin).orElse(false)) {
            return Mono.just(Result.failure(ErrorCollection.builder().addError(TypedError.forbidden(messages
                    .getMessage("errors.access.denied", null, locale))).build()));
        }
        String newId = UUID.randomUUID().toString();
        return securityManagerService.checkWritePermissions(folder.getServiceId(), currentUser, locale).flatMap(
                res -> res.andThenMono(v -> tableClient.usingSessionMonoRetryable(session ->
                        session.usingTxMonoRetryable(TransactionMode.SERIALIZABLE_READ_WRITE, ts ->
                                validateAndBuildFolderModel(folder, newId, getTenantId(currentUser), locale, ts, 0L,
                                        1L)
                                        .flatMap(r -> r.andThenMono(validated ->
                                                        folderDao.upsertOneRetryable(ts, validated)
                                                                .thenReturn(Result.success(validated))
                                                )
                                        )
                        )
                ))
        );
    }

    public Mono<Result<FolderModel>> updateFolder(
            String folderId,
            FrontFolderInputDto folderInput,
            YaUserDetails currentUser,
            Locale locale
    ) {
        if (currentUser == null || !currentUser.getUser().map(UserModel::getDAdmin).orElse(false)) {
            return Mono.just(Result.failure(ErrorCollection.builder().addError(TypedError.forbidden(messages
                    .getMessage("errors.access.denied", null, locale))).build()));
        }
        TenantId tenantId = getTenantId(currentUser);
        return securityManagerService.checkWritePermissions(
                folderInput.getServiceId(), currentUser, locale).flatMap(res -> res.andThenMono(v ->
                tableClient.usingSessionMonoRetryable(session ->
                        session.usingCompTxRetryable(
                                ts ->
                                        folderDao.getByIdStartTx(ts, folderId, tenantId),
                                (tx, oldFolder) ->
                                        validateUpdate(tx, locale, oldFolder, folderInput),
                                (ts, result) -> result.andThenMono(validated ->
                                        folderDao.upsertOneRetryable(ts, validated)
                                                .thenReturn(Result.success(validated))
                                )
                        )
                ))
        );
    }

    public Mono<Result<FolderModel>> deleteFolder(
            String folderId,
            YaUserDetails currentUser,
            Locale locale
    ) {
        if (currentUser == null || !currentUser.getUser().map(UserModel::getDAdmin).orElse(false)) {
            return Mono.just(Result.failure(ErrorCollection.builder().addError(TypedError.forbidden(messages
                    .getMessage("errors.access.denied", null, locale))).build()));
        }
        TenantId tenantId = getTenantId(currentUser);
        return tableClient.usingSessionMonoRetryable(session ->
                session.usingCompTxRetryable(
                        ts ->
                                folderDao.getByIdStartTx(ts, folderId, tenantId),
                        (tx, validatedFolder) ->
                                validateDelete(tx, locale, validatedFolder, currentUser)
                                .flatMap(result -> result.andThenMono(folder -> {
                                    FolderModel deletedFolder = folder.toBuilder()
                                            .setDeleted(true)
                                            .setVersion(folder.getVersion() + 1)
                                            .setNextOpLogOrder(folder.getNextOpLogOrder() + 1L)
                                            .build();
                                    return folderDao.removeRetryable(
                                            tx,
                                            deletedFolder.getId(),
                                            deletedFolder.getTenantId(),
                                            deletedFolder.getVersion()
                                    ).then(providersDao.getAllByTenant(tx, deletedFolder.getTenantId(), false)
                                            .map(WithTxId::get)
                                            .flatMap(providers -> {
                                                Optional<ProviderModel> provider = providers.stream()
                                                        .filter(p -> deletedFolder.getId()
                                                                .equals(p.getReserveFolderId().orElse(null)))
                                                        .findAny();
                                                if (provider.isPresent()) {
                                                    ProviderModel newProvider = ProviderModel.builder(provider.get())
                                                            .reserveFolderId(null)
                                                            .build();
                                                    return providersDao.updateProviderRetryable(tx, newProvider);
                                                } else {
                                                    return Mono.empty();
                                                }
                                            })
                                    ).thenReturn(Result.success(deletedFolder));
                                })),
                        (ts, result) -> result.match(s -> ts.commitTransaction().thenReturn(Result.success(s)),
                                e -> ts.commitTransaction().thenReturn(Result.failure(e)))
                )
        );
    }

    public Mono<Result<Page<FolderModel>>> listFolders(boolean includeDeleted,
                                                       PageRequest pageRequest,
                                                       YaUserDetails currentUser,
                                                       Locale locale) {
        // TODO Special implementation is required for external users
        // TODO External users are allowed to obtain folders only for some services
        return securityManagerService.checkReadPermissions(currentUser, locale).flatMap(r -> r.andThenMono(u -> {
            Result<PageRequest.Validated<StringContinuationToken>> pageValidation
                    = pageRequest.validate(continuationTokenReader, messages, locale);
            return pageValidation.andThenDo(p -> validateContinuationToken(p, locale)).andThenMono(p -> {
                int limit = p.getLimit();
                String fromId = p.getContinuationToken().map(StringContinuationToken::getId).orElse(null);
                return tableClient.usingSessionMonoRetryable(session ->
                        folderDao
                                .listFolders(immediateTx(session),
                                        Tenants.DEFAULT_TENANT_ID, fromId, limit + 1, includeDeleted)
                                .map(values -> values.size() > limit
                                        ? Page.page(values.subList(0, limit),
                                        prepareToken(values.get(limit - 1)))
                                        : Page.lastPage(values))
                                .map(Result::success));
            });
        }));
    }

    public Mono<Long> addDefaultFolders() {
        return tableClient.usingSessionMonoRetryable(session -> servicesDao.getAllServiceIds(session)
                .buffer(BATCH_SIZE)
                .concatMap(this::addDefaultFolders)
                .collect(Collectors.summingLong(value -> value)));
    }

    public Mono<Result<FolderModel>> createReservedFolder(String newFolderId, long serviceId, TenantId tenantId,
                                                          YdbTxSession ts, Locale locale) {
        FrontFolderInputDto newFolder = new FrontFolderInputDto(0L, serviceId, "reserved", "Reserved",
                Collections.emptySet());
        return validateAndBuildFolderModel(newFolder, newFolderId, tenantId, locale, ts, 0L, 1L)
                .map(r -> r.andThen(folder -> Result.success(
                        folder.toBuilder().setFolderType(FolderType.PROVIDER_RESERVE).build())))
                .flatMap(r -> r.andThenMono(validated -> folderDao.upsertOneRetryable(ts, validated)
                        .thenReturn(Result.success(validated))));
    }

    private Mono<Result<FolderModel>> validateUpdate(
            YdbTxSession tx, Locale locale, Optional<FolderModel> oldFolderOptional, FrontFolderInputDto folderInput
    ) {
        if (oldFolderOptional.isEmpty()) {
            return Mono.just(notFound(locale));
        }
        FolderModel oldFolder = oldFolderOptional.get();

        Tuple2<Long, ErrorCollection> newVersion = validateVersion(oldFolder, folderInput, locale);
        if (newVersion.getT2().hasAnyErrors()) {
            return Mono.just(Result.failure(newVersion.getT2()));
        }

        if (folderInput.getServiceId() != oldFolder.getServiceId()) {
            // todo поддержать перенос фолдера в другой сервис
            return Mono.just(Result.failure(ErrorCollection.builder().addError(
                    TypedError.locked(messages
                            .getMessage("errors.folder.move.impossible", null, locale)))
                    .build()));
        }

        return validateAndBuildFolderModel(
                folderInput, oldFolder.getId(), oldFolder.getTenantId(), locale, tx, newVersion.getT1(),
                oldFolder.getNextOpLogOrder() + 1L
        );
    }

    private Tuple2<Long, ErrorCollection> validateVersion(
            FolderModel oldFolder, FrontFolderInputDto folderInput, Locale locale
    ) {
        MutableLong newVersion = new MutableLong();
        ErrorCollection.Builder errors = ErrorCollection.builder();
        versionValidator.validate(
                () -> Optional.of(folderInput.getVersion()),
                oldFolder::getVersion,
                newVersion::setValue,
                errors,
                "version",
                locale
        );
        return Tuples.of(newVersion.getValue(), errors.build());
    }

    private Mono<Result<FolderModel>> validateDelete(
            YdbTxSession tx, Locale locale, Optional<FolderModel> folderOptional,
            YaUserDetails currentUser) {
        if (folderOptional.isEmpty()) {
            return Mono.just(notFound(locale));
        }
        FolderModel folder = folderOptional.get();
        if (folder.getFolderType() == FolderType.COMMON_DEFAULT_FOR_SERVICE) {
            return Mono.just(Result.failure(ErrorCollection.builder().addError(
                    TypedError.locked(messages
                            .getMessage("errors.folder.delete.default", null, locale)))
                    .build()));
        }
        // todo проверить, что в фолдере нет квот, особенно спущенных
        return securityManagerService.checkWritePermissions(folder.getServiceId(), currentUser, locale, folder);
    }

    private Mono<Result<FolderModel>> validateAndBuildFolderModel(
            FrontFolderInputDto folder, String newId, TenantId tenantId, Locale locale, YdbTxSession ts,
            long newVersion, long newNextOpLogOrder
    ) {
        return abcServiceValidator.validateAbcService(
                Optional.of(folder.getServiceId()), locale, ts, "serviceId",
                AbcServiceValidator.ALLOWED_SERVICE_STATES, AbcServiceValidator.ALLOWED_SERVICE_READONLY_STATES, false
        ).map(serviceR -> {
            FolderModel.Builder builder = FolderModel.newBuilder();
            ErrorCollection.Builder errors = ErrorCollection.builder();
            builder
                    .setId(newId)
                    .setTenantId(tenantId)
                    .setVersion(newVersion)
                    .setNextOpLogOrder(newNextOpLogOrder);
            serviceR.doOnSuccess(s -> builder.setServiceId(s.getId()));
            serviceR.doOnFailure(errors::add);
            textValidator
                    .validateAndSet(
                            () -> Optional.ofNullable(folder.getDisplayName()), builder::setDisplayName,
                            errors, "displayName", MAX_NAME_LENGTH, locale
                    )
                    .validateAndSet(
                            () -> Optional.ofNullable(folder.getDescription()), builder::setDescription,
                            errors, "description", MAX_DESCRIPTION_LENGTH, locale
                    );
            textValidator.validateAndSetTags(folder.getTags(), builder::setTags, errors, "tags", MAX_TAG_LENGTH,
                    locale);

            return errors.hasAnyErrors() ?
                    Result.failure(errors.build()) :
                    Result.success(builder.build());
        });
    }

    private Result<Void> validateId(String folderId, Locale locale) {
        if (!Uuids.isValidUuid(folderId)) {
            ErrorCollection error = ErrorCollection.builder().addError(TypedError.notFound(messages
                    .getMessage("errors.folder.not.found", null, locale)))
                    .build();
            return Result.failure(error);
        }
        return Result.success(null);
    }

    private Result<Void> validateContinuationToken(PageRequest.Validated<StringContinuationToken> pageRequest,
                                                   Locale locale) {
        if (pageRequest.getContinuationToken().isEmpty()) {
            return Result.success(null);
        }
        return validateId(pageRequest.getContinuationToken().get().getId(), locale);
    }

    private YdbTxSession immediateTx(YdbSession session) {
        return session.asTxCommitRetryable(TransactionMode.STALE_READ_ONLY);
    }

    private String prepareToken(FolderModel lastItem) {
        return ContinuationTokens.encode(new StringContinuationToken(lastItem.getId()),
                continuationTokenWriter);
    }

    private Flux<Long> addDefaultFolders(List<Long> serviceIds) {
        if (serviceIds.isEmpty()) {
            return Flux.just(0L);
        }
        Set<Tuple2<Long, String>> ids = serviceIds.stream()
                .map(i -> Tuples.of(i, Tenants.DEFAULT_TENANT_ID.getId())).collect(Collectors.toSet());
        Set<Long> existingServices = new HashSet<>(serviceIds);
        Map<Long, String> newFolderIds = existingServices.stream()
                .collect(Collectors.toMap(v -> v, v -> UUID.randomUUID().toString()));
        return tableClient.usingSessionMonoRetryable(innerSession -> innerSession.usingCompTxMonoRetryable(
                TransactionMode.SERIALIZABLE_READ_WRITE,
                ts -> folderDao.getServiceIdsWithExistingDefaultFolderTx(ts, ids).map(WithTxId::asTuple),
                (ts, v) -> Mono.just(v),
                (ts, v) -> {
                    Set<Long> servicesWithMissingFolders = Sets.difference(existingServices, v);
                    if (servicesWithMissingFolders.isEmpty()) {
                        return ts.commitTransaction().thenReturn(0L);
                    }
                    List<FolderModel> newFolders = servicesWithMissingFolders.stream().map(serviceId ->
                            FolderModel.newBuilder()
                                    .setTenantId(Tenants.DEFAULT_TENANT_ID)
                                    .setId(newFolderIds.get(serviceId))
                                    .setServiceId(serviceId)
                                    .setFolderType(FolderType.COMMON_DEFAULT_FOR_SERVICE)
                                    .setDisplayName("default")
                                    .setNextOpLogOrder(1L)
                                    .build()).collect(Collectors.toList());
                    return folderDao.upsertAllRetryable(ts, newFolders).thenReturn((long) newFolders.size());
                }
        )).flux();
    }

}
