package ru.yandex.intranet.d.services.folders.history;

import java.math.BigDecimal;
import java.util.Collection;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.jetbrains.annotations.Nullable;

import ru.yandex.intranet.d.i18n.Locales;
import ru.yandex.intranet.d.model.accounts.AccountModel;
import ru.yandex.intranet.d.model.delivery.DeliverableMetaHistoryModel;
import ru.yandex.intranet.d.model.folders.AccountHistoryModel;
import ru.yandex.intranet.d.model.folders.AccountsHistoryModel;
import ru.yandex.intranet.d.model.folders.FolderHistoryFieldsModel;
import ru.yandex.intranet.d.model.folders.FolderOperationLogModel;
import ru.yandex.intranet.d.model.folders.ProvisionHistoryModel;
import ru.yandex.intranet.d.model.folders.ProvisionsByResource;
import ru.yandex.intranet.d.model.folders.QuotasByAccount;
import ru.yandex.intranet.d.model.folders.QuotasByResource;
import ru.yandex.intranet.d.model.folders.TransferMetaHistoryModel;
import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.model.services.ServiceMinimalModel;
import ru.yandex.intranet.d.model.units.UnitModel;
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel;
import ru.yandex.intranet.d.model.users.UserModel;
import ru.yandex.intranet.d.services.quotas.QuotasHelper;
import ru.yandex.intranet.d.services.units.UnitsComparator;
import ru.yandex.intranet.d.util.FrontStringUtil;
import ru.yandex.intranet.d.util.paging.Page;
import ru.yandex.intranet.d.web.model.AbcServiceDto;
import ru.yandex.intranet.d.web.model.AccountDto;
import ru.yandex.intranet.d.web.model.AmountDto;
import ru.yandex.intranet.d.web.model.PageDto;
import ru.yandex.intranet.d.web.model.ProviderDto;
import ru.yandex.intranet.d.web.model.ResourceDto;
import ru.yandex.intranet.d.web.model.UserDto;
import ru.yandex.intranet.d.web.model.accounts.AccountReserveTypeDto;
import ru.yandex.intranet.d.web.model.folders.front.history.FrontFolderOperationLogDto;
import ru.yandex.intranet.d.web.model.folders.front.history.FrontFolderOperationLogPageDto;
import ru.yandex.intranet.d.web.model.resources.AccountsSpaceDto;

import static ru.yandex.intranet.d.web.util.ModelDtoConverter.providerDtoFromModel;

/**
 * Folder history page builder
 *
 * @author Denis Blokhin <denblo@yandex-team.ru>
 */

public class FolderHistoryPageBuilder {
    private Page<FolderOperationLogModel> page;
    private List<UserModel> users;
    private List<ProviderModel> providers;
    private List<AccountModel> accounts;
    private List<ServiceMinimalModel> services;

    private Map<String, ResourceModel> resourcesById;
    private Map<String, UnitsEnsembleModel> ensemblesById;
    private Map<String, AccountsSpaceDto> accountsSpacesById;

    public FolderHistoryPageBuilder setPage(Page<FolderOperationLogModel> page) {
        this.page = page;
        return this;
    }

    public FolderHistoryPageBuilder setUsers(List<UserModel> users) {
        this.users = users;
        return this;
    }

    public FolderHistoryPageBuilder setProviders(List<ProviderModel> providers) {
        this.providers = providers;
        return this;
    }

    public FolderHistoryPageBuilder setResources(List<ResourceModel> resourceModels) {
        resourcesById = resourceModels.stream()
                .collect(Collectors.toMap(ResourceModel::getId, Function.identity()));
        return this;
    }

    public FolderHistoryPageBuilder setAccounts(List<AccountModel> accountModels) {
        this.accounts = accountModels;
        return this;
    }

    public FolderHistoryPageBuilder setServices(List<ServiceMinimalModel> services) {
        this.services = services;
        return this;
    }

    public FolderHistoryPageBuilder setUnitEnsembles(List<UnitsEnsembleModel> ensembles) {
        ensemblesById = ensembles.stream()
                .collect(Collectors.toMap(UnitsEnsembleModel::getId, Function.identity()));
        return this;
    }

    public FolderHistoryPageBuilder setAccountsSpaces(List<AccountsSpaceDto> accountsSpaceDtos) {
        accountsSpacesById = accountsSpaceDtos.stream()
                .collect(Collectors.toMap(AccountsSpaceDto::getId, Function.identity()));
        return this;
    }

    public FrontFolderOperationLogPageDto build(Locale locale) {
        final Map<String, UserDto> userById = users.stream()
                .collect(Collectors.toMap(UserModel::getId, u -> toView(u, locale)));

        final Map<String, ProviderDto> providerById = providers.stream()
                .collect(Collectors.toMap(ProviderModel::getId, p -> providerDtoFromModel(p, locale)));

        final Map<String, ResourceDto> resourcesDtoById = resourcesById.values().stream()
                .collect(Collectors.toMap(ResourceModel::getId, r -> toView(r, locale)));

        final Map<String, AccountDto> accountsById = accounts.stream()
                .collect(Collectors.toMap(AccountModel::getId, this::toView));

        final Map<Long, AbcServiceDto> servicesById = services.stream()
                .collect(Collectors.toMap(ServiceMinimalModel::getId, s -> toView(s, locale)));

        final List<FrontFolderOperationLogDto> historyItems = page.getItems().stream()
                .map(i -> toView(i, locale))
                .collect(Collectors.toList());

        final String token = page.getContinuationToken().orElse(null);

        return new FrontFolderOperationLogPageDto(new PageDto<>(historyItems, token), userById, providerById,
                resourcesDtoById, accountsById, servicesById, accountsSpacesById);
    }

    private UserDto toView(UserModel user, Locale locale) {
        String firstName = Locales.select(user.getFirstNameEn(), user.getFirstNameRu(), locale);
        String lastName = Locales.select(user.getLastNameEn(), user.getLastNameRu(), locale);

        return new UserDto(user.getPassportUid().orElse(null), user.getPassportLogin().orElse(null), firstName,
                lastName, user.getGender());
    }

    private ResourceDto toView(ResourceModel resource, Locale locale) {
        return new ResourceDto(resource, locale);
    }

    private AccountDto toView(AccountModel account) {
        return AccountDto.fromModel(account);
    }

    private FrontFolderOperationLogDto toView(FolderOperationLogModel folderOperationLogModel, Locale locale) {
        return new FrontFolderOperationLogDto(
                folderOperationLogModel.getIdentity().getId(),
                folderOperationLogModel.getFolderId(),
                folderOperationLogModel.getIdentity().getOperationDateTime(),
                folderOperationLogModel.getOperationType(),
                folderOperationLogModel.getAuthorUserId().orElse(null),
                folderOperationLogModel.getAuthorProviderId().orElse(null),
                folderOperationLogModel.getSourceFolderOperationsLogId().orElse(null),
                folderOperationLogModel.getDestinationFolderOperationsLogId().orElse(null),
                toView(folderOperationLogModel.getOldFolderFields().orElse(null)),
                toView(folderOperationLogModel.getNewFolderFields().orElse(null)),
                toView(folderOperationLogModel.getOldBalance(), locale),
                toView(folderOperationLogModel.getNewBalance(), locale),
                toView(folderOperationLogModel.getOldBalance(), folderOperationLogModel.getNewBalance(), locale),
                toView(folderOperationLogModel.getOldProvisions(), locale),
                toView(folderOperationLogModel.getNewProvisions(), locale),
                toView(folderOperationLogModel.getOldProvisions(), folderOperationLogModel.getNewProvisions(), locale),
                toView(folderOperationLogModel.getOldQuotas(), locale),
                toView(folderOperationLogModel.getNewQuotas(), locale),
                toView(folderOperationLogModel.getOldQuotas(), folderOperationLogModel.getNewQuotas(), locale),
                folderOperationLogModel.getAccountsQuotasOperationsId().orElse(null),
                toView(folderOperationLogModel.getOldAccounts().orElse(null)),
                toView(folderOperationLogModel.getNewAccounts().orElse(null)),
                folderOperationLogModel.getProviderRequestId().orElse(null),
                folderOperationLogModel.getQuotasDemandsId().orElse(null),
                folderOperationLogModel.getOperationPhase().orElse(null),
                folderOperationLogModel.getOrder(),
                toView(folderOperationLogModel.getDeliveryMeta().orElse(null)),
                toView(folderOperationLogModel.getTransferMeta().orElse(null))
        );
    }

    @Nullable
    private FrontFolderOperationLogDto.TransferMetaHistoryDto toView(TransferMetaHistoryModel transferMeta) {
        if (transferMeta == null) {
            return null;
        }
        return new FrontFolderOperationLogDto.TransferMetaHistoryDto(
                transferMeta.getTransferRequestId(),
                transferMeta.getRoleInTransfer(),
                transferMeta.getAnotherParticipants().stream()
                        .map(it -> new FrontFolderOperationLogDto.TransferMetaHistoryDto.AnotherDto(
                                it.getServiceId(),
                                it.getFolderId(),
                                it.getAccountId()
                        ))
                        .collect(Collectors.toSet())
        );
    }

    @Nullable
    private FrontFolderOperationLogDto.DeliverableMetaHistoryDto toView(
            @Nullable DeliverableMetaHistoryModel deliveryMeta) {
        if (deliveryMeta == null) {
            return null;
        }
        return new FrontFolderOperationLogDto.DeliverableMetaHistoryDto(deliveryMeta.getQuotaRequestId(),
                deliveryMeta.getCampaignId(), deliveryMeta.getBigOrderIds(), deliveryMeta.getDeliveryId());
    }

    @Nullable
    private FrontFolderOperationLogDto.Accounts toView(@Nullable AccountsHistoryModel accounts) {
        if (accounts == null) {
            return null;
        }
        return new FrontFolderOperationLogDto.Accounts(accounts.getAccounts().entrySet().stream()
                .collect(Collectors.toMap(Map.Entry::getKey, e -> toView(e.getValue()))));
    }

    private FrontFolderOperationLogDto.Account toView(AccountHistoryModel account) {
        return new FrontFolderOperationLogDto.Account(
                account.getVersion().orElse(null),
                account.getProviderId().orElse(null),
                account.getOuterAccountIdInProvider().orElse(null),
                account.getOuterAccountKeyInProvider().orElse(null),
                account.getFolderId().orElse(null),
                account.getDisplayName().orElse(null),
                account.getLastReceivedVersion().orElse(null),
                account.getAccountsSpacesId(),
                AccountReserveTypeDto.fromModel(account.getReserveType().orElse(null)));
    }

    private FrontFolderOperationLogDto.ProvisionsByAccountId toView(QuotasByAccount quotasByAccount, Locale locale) {
        return new FrontFolderOperationLogDto.ProvisionsByAccountId(quotasByAccount.asMap().entrySet().stream()
                .collect(Collectors.toMap(Map.Entry::getKey, e -> toView(e.getValue(), locale))));
    }

    private FrontFolderOperationLogDto.ProvisionsByResourceId toView(ProvisionsByResource provisionsByResource,
                                                                     Locale locale) {
        return new FrontFolderOperationLogDto.ProvisionsByResourceId(provisionsByResource.asMap().entrySet().stream()
                .collect(Collectors.toMap(Map.Entry::getKey, e -> toView(e.getValue(), e.getKey(), locale))));
    }

    private FrontFolderOperationLogDto.Provision toView(ProvisionHistoryModel historyModel,
                                                        String resourceId, Locale locale) {
        return new FrontFolderOperationLogDto.Provision(
                getAmountDto(resourceId, historyModel.getProvision(), locale),
                historyModel.getVersion().orElse(null)
        );
    }

    private FrontFolderOperationLogDto.AmountByResourceId toView(QuotasByResource quotasByResource,
                                                                 Locale locale) {
        Map<String, AmountDto> amountByResourceId = quotasByResource.asMap().entrySet().stream()
                .collect(Collectors.toMap(Map.Entry::getKey, e -> getAmountDto(e.getKey(), e.getValue(), locale)));

        return new FrontFolderOperationLogDto.AmountByResourceId(amountByResourceId);
    }


    private AmountDto getAmountDto(String resourceId, Long rawAmount, Locale locale) {
        ResourceModel resource = resourcesById.get(resourceId);
        BigDecimal amount = BigDecimal.valueOf(rawAmount);
        if (resource == null) {
            return new AmountDto(FrontStringUtil.toString(amount), "", FrontStringUtil.toString(amount), "");
        }
        UnitsEnsembleModel unitsEnsembleModel = ensemblesById.get(resource.getUnitsEnsembleId());
        if (unitsEnsembleModel == null) {
            return new AmountDto(FrontStringUtil.toString(amount), "", FrontStringUtil.toString(amount), "");
        }

        Optional<UnitModel> unitModel = unitsEnsembleModel.unitById(resource.getBaseUnitId());

        if (unitModel.isEmpty()) {
            return new AmountDto(FrontStringUtil.toString(amount), "", FrontStringUtil.toString(amount), "");
        }

        UnitModel baseUnit = unitModel.get();
        List<UnitModel> sortedUnits = unitsEnsembleModel.getUnits().stream()
                .sorted(UnitsComparator.INSTANCE).collect(Collectors.toList());
        List<UnitModel> allowedSortedUnits = QuotasHelper.getAllowedUnits(resource, sortedUnits);
        return QuotasHelper.getAmountDto(amount, allowedSortedUnits, baseUnit, baseUnit, baseUnit, baseUnit, locale);
    }


    @Nullable
    private FrontFolderOperationLogDto.FoldersFields toView(@Nullable FolderHistoryFieldsModel folderFields) {
        if (folderFields == null) {
            return null;
        }
        return new FrontFolderOperationLogDto.FoldersFields(
                folderFields.getServiceId().orElse(null),
                folderFields.getVersion().orElse(null),
                folderFields.getDisplayName().orElse(null),
                folderFields.getDescription().orElse(null),
                folderFields.getFolderType().orElse(null),
                folderFields.getTags().orElse(null)
        );
    }

    private AbcServiceDto toView(ServiceMinimalModel service, Locale locale) {
        String serviceName = Locales.select(service.getNameEn(), service.getName(), locale);
        return new AbcServiceDto(service.getId(), serviceName);
    }

    private FrontFolderOperationLogDto.AmountByResourceId toView(QuotasByResource oldQuotasByResource,
                                                                 QuotasByResource newQuotasByResource,
                                                                 Locale locale) {
        Map<String, AmountDto> amountByResourceId = getDiffAmountMap(oldQuotasByResource.asMap(),
                newQuotasByResource.asMap())
                .entrySet().stream()
                .collect(Collectors.toMap(Map.Entry::getKey, e -> getAmountDto(e.getKey(), e.getValue(), locale)));

        return new FrontFolderOperationLogDto.AmountByResourceId(amountByResourceId);
    }

    private Map<String, Long> getDiffAmountMap(Map<String, Long> oldAmountByResource,
                                               Map<String, Long> newAmountByResource) {
        Set<String> keys = Stream.of(oldAmountByResource.keySet(), newAmountByResource.keySet())
                .flatMap(Collection::stream)
                .collect(Collectors.toSet());
        return keys.stream()
                .collect(Collectors.toMap(
                        k -> k,
                        v -> newAmountByResource.getOrDefault(v, 0L) - oldAmountByResource.getOrDefault(v, 0L)
                ));
    }

    private FrontFolderOperationLogDto.ProvisionDeltasByAccountId toView(QuotasByAccount oldQuotasByAccount,
                                                                         QuotasByAccount newQuotasByAccount,
                                                                         Locale locale) {
        Map<String, ProvisionsByResource> oldQuotasByAccountMap = oldQuotasByAccount.asMap();
        Map<String, ProvisionsByResource> newQuotasByAccountMap = newQuotasByAccount.asMap();
        Set<String> accountsKeySet = Stream.of(oldQuotasByAccountMap.keySet(), newQuotasByAccountMap.keySet())
                .flatMap(Collection::stream)
                .collect(Collectors.toSet());

        return new FrontFolderOperationLogDto.ProvisionDeltasByAccountId(accountsKeySet.stream()
                .collect(Collectors.toMap(k -> k, v -> toView(oldQuotasByAccountMap.get(v),
                        newQuotasByAccountMap.get(v), locale))));
    }

    private FrontFolderOperationLogDto.ProvisionDeltasByResourceId toView(
            @Nullable ProvisionsByResource oldProvisionsByResource,
            @Nullable ProvisionsByResource newProvisionsByResource,
            Locale locale
    ) {
        Map<String, AmountDto> amountByResourceId = getDiffAmountMap(toAmountByResourceMap(oldProvisionsByResource),
                toAmountByResourceMap(newProvisionsByResource))
                .entrySet().stream()
                .collect(Collectors.toMap(Map.Entry::getKey, e -> getAmountDto(e.getKey(), e.getValue(), locale)));

        return new FrontFolderOperationLogDto.ProvisionDeltasByResourceId(amountByResourceId);
    }

    private static Map<String, Long> toAmountByResourceMap(@Nullable ProvisionsByResource provisionsByResource) {
        return provisionsByResource == null
                ? Map.of()
                : provisionsByResource.asMap().entrySet().stream()
                .collect(Collectors.toMap(Map.Entry::getKey, e -> e.getValue().getProvision()));
    }
}
