package ru.yandex.intranet.d.services.folders.history;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.google.common.collect.Lists;
import com.yandex.ydb.table.transaction.TransactionMode;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuple2;
import reactor.util.function.Tuples;

import ru.yandex.intranet.d.dao.Tenants;
import ru.yandex.intranet.d.dao.accounts.AccountsDao;
import ru.yandex.intranet.d.dao.folders.FolderDao;
import ru.yandex.intranet.d.dao.folders.FolderOperationLogDao;
import ru.yandex.intranet.d.dao.services.ServicesDao;
import ru.yandex.intranet.d.dao.users.UsersDao;
import ru.yandex.intranet.d.datasource.Ydb;
import ru.yandex.intranet.d.datasource.model.YdbTableClient;
import ru.yandex.intranet.d.loaders.providers.AccountSpacesLoader;
import ru.yandex.intranet.d.loaders.providers.ProvidersLoader;
import ru.yandex.intranet.d.loaders.resources.ResourcesLoader;
import ru.yandex.intranet.d.loaders.units.UnitsEnsemblesLoader;
import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.WithTenant;
import ru.yandex.intranet.d.model.accounts.AccountModel;
import ru.yandex.intranet.d.model.accounts.AccountSpaceModel;
import ru.yandex.intranet.d.model.folders.FolderOperationLogModel;
import ru.yandex.intranet.d.model.folders.TransferMetaHistoryModel;
import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.model.services.ServiceMinimalModel;
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel;
import ru.yandex.intranet.d.model.users.UserModel;
import ru.yandex.intranet.d.services.resources.AccountsSpacesUtils;
import ru.yandex.intranet.d.services.resources.ExpandedAccountsSpaces;
import ru.yandex.intranet.d.services.security.SecurityManagerService;
import ru.yandex.intranet.d.util.JsonReader;
import ru.yandex.intranet.d.util.JsonWriter;
import ru.yandex.intranet.d.util.ObjectMapperHolder;
import ru.yandex.intranet.d.util.paging.ContinuationTokens;
import ru.yandex.intranet.d.util.paging.Page;
import ru.yandex.intranet.d.util.paging.PageRequest;
import ru.yandex.intranet.d.util.result.ErrorCollection;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.util.result.TypedError;
import ru.yandex.intranet.d.web.model.SortOrderDto;
import ru.yandex.intranet.d.web.model.folders.front.history.FrontFolderOperationLogPageDto;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;

import static java.util.stream.Stream.concat;

/**
 * Folder history service
 *
 * @author Denis Blokhin <denblo@yandex-team.ru>
 */
@Component
public class FolderHistoryService {

    private final FolderOperationLogDao folderOperationLogDao;
    private final YdbTableClient tableClient;
    private final SecurityManagerService securityManagerService;
    private final MessageSource messages;
    private final JsonReader<FolderHistoryContinuationToken> continuationTokenReader;
    private final JsonWriter<FolderHistoryContinuationToken> continuationTokenWriter;
    private final UsersDao usersDao;
    private final ProvidersLoader providersLoader;
    private final ResourcesLoader resourcesLoader;
    private final AccountsDao accountsDao;
    private final FolderDao folderDao;
    private final ServicesDao servicesDao;
    private final UnitsEnsemblesLoader unitsEnsemblesLoader;
    private final AccountsSpacesUtils accountsSpacesUtils;
    private final AccountSpacesLoader accountSpacesLoader;

    @SuppressWarnings("checkstyle:ParameterNumber")
    public FolderHistoryService(FolderOperationLogDao folderOperationLogDao, YdbTableClient tableClient,
                                SecurityManagerService securityManagerService,
                                @Qualifier("messageSource") MessageSource messages,
                                @Qualifier("continuationTokensJsonObjectMapper") ObjectMapperHolder objectMapper,
                                UsersDao usersDao,
                                ProvidersLoader providersLoader, ResourcesLoader resourcesLoader,
                                AccountsDao accountsDao, FolderDao folderDao, ServicesDao servicesDao,
                                UnitsEnsemblesLoader unitsEnsemblesLoader, AccountsSpacesUtils accountsSpacesUtils,
                                AccountSpacesLoader accountSpacesLoader) {
        this.folderOperationLogDao = folderOperationLogDao;
        this.tableClient = tableClient;
        this.securityManagerService = securityManagerService;
        this.messages = messages;
        this.continuationTokenReader = new JsonReader<>(objectMapper.getObjectMapper(),
                FolderHistoryContinuationToken.class);
        this.continuationTokenWriter = new JsonWriter<>(objectMapper.getObjectMapper(),
                FolderHistoryContinuationToken.class);
        this.usersDao = usersDao;
        this.providersLoader = providersLoader;
        this.resourcesLoader = resourcesLoader;
        this.accountsDao = accountsDao;
        this.folderDao = folderDao;
        this.servicesDao = servicesDao;
        this.unitsEnsemblesLoader = unitsEnsemblesLoader;
        this.accountsSpacesUtils = accountsSpacesUtils;
        this.accountSpacesLoader = accountSpacesLoader;
    }

    public Mono<Result<FrontFolderOperationLogPageDto>> getFolderOperationLogsPage(String folderId,
                                                                                   PageRequest pageRequest,
                                                                                   SortOrderDto sortOrderDto,
                                                                                   YaUserDetails currentUser,
                                                                                   Locale locale) {
        return securityManagerService.checkReadPermissions(currentUser, locale).flatMap(res -> res.andThenMono(u -> {
            Result<PageRequest.Validated<FolderHistoryContinuationToken>> pageValidation
                    = pageRequest.validate(continuationTokenReader, messages, locale);

            final TenantId tenantId = Tenants.getTenantId(currentUser);

            return pageValidation.andThenMono(p -> tableClient.usingSessionMonoRetryable(session -> folderDao.getById(
                    session.asTxCommitRetryable(TransactionMode.STALE_READ_ONLY), folderId, tenantId))
                    .flatMap(folder -> folder.map(f ->
                            securityManagerService.checkReadPermissions(f.getId(), currentUser, locale, p))
                            .orElseGet(() -> Mono.just(Result.failure(ErrorCollection.builder().addError(
                                    TypedError.notFound(messages
                                            .getMessage("errors.folder.not.found", null, locale)))
                                    .build()))
                            )
                    )
            )
                    .flatMap(r -> r.andThenMono(p -> loadPage(p, folderId, sortOrderDto, locale, tenantId)));
        }));
    }

    private Mono<Result<FrontFolderOperationLogPageDto>> loadPage(
            PageRequest.Validated<FolderHistoryContinuationToken> pageRequest, String folderId,
            SortOrderDto sortOrderDto, Locale locale, TenantId tenantId) {
        int limit = pageRequest.getLimit();
        final FolderHistoryContinuationToken token = pageRequest.getContinuationToken().orElse(null);

        return tableClient.usingSessionMonoRetryable(session -> folderOperationLogDao
                .getPageByFolder(session.asTxCommitRetryable(TransactionMode.STALE_READ_ONLY),
                        tenantId, folderId, token, sortOrderDto, limit + 1)
                .map(values -> values.size() > limit
                        ? Page.page(values.subList(0, limit),
                        prepareToken(values.get(limit - 1)))
                        : Page.lastPage(values))
                .flatMap(page -> Mono.zip(
                        getUsers(page, tenantId),
                        getResources(page, tenantId).flatMap(rs -> withEnsembles(rs, tenantId)),
                        getAccounts(page, tenantId).flatMap(accounts -> withAccountsSpaces(accounts, page, tenantId)),
                        getServices(page)
                ).flatMap(params -> getProviders(page, params.getT3().getT1(), params.getT2().getT1(), tenantId).map(
                        providers -> Result.success(new FolderHistoryPageBuilder()
                            .setPage(page)
                            .setUsers(params.getT1())
                            .setProviders(providers)
                            .setResources(params.getT2().getT1())
                            .setAccounts(params.getT3().getT1())
                            .setServices(params.getT4())
                            .setUnitEnsembles(params.getT2().getT2())
                            .setAccountsSpaces(accountsSpacesUtils.toDtos(params.getT3().getT2(), locale))
                            .build(locale))
                )))
        );
    }

    private Mono<Tuple2<List<ResourceModel>, List<UnitsEnsembleModel>>> withEnsembles(
            List<ResourceModel> resourceModels, TenantId tenantId) {

        List<Tuple2<String, TenantId>> ids = resourceModels.stream()
                .map(ResourceModel::getUnitsEnsembleId)
                .distinct()
                .map(id -> Tuples.of(id, tenantId))
                .collect(Collectors.toList());
        return unitsEnsemblesLoader.getUnitsEnsemblesByIdsImmediate(ids)
                .map(ensembles -> Tuples.of(resourceModels, ensembles));
    }

    private Mono<List<AccountModel>> getAccounts(Page<FolderOperationLogModel> page, TenantId tenantId) {
        final List<WithTenant<String>> ids = page.getItems().stream()
                .flatMap(fm -> concat(concat(concat(concat(
                        fm.getNewProvisions().asMap().keySet().stream(),
                        fm.getOldProvisions().asMap().keySet().stream()),
                        fm.getActuallyAppliedProvisions().stream().flatMap(p -> p.asMap().keySet().stream())
                ), concat(
                        fm.getOldAccounts().stream().flatMap(h -> h.getAccounts().keySet().stream()),
                        fm.getNewAccounts().stream().flatMap(h -> h.getAccounts().keySet().stream())
                )),
                        fm.getTransferMeta().stream()
                            .flatMap(it -> it.getAnotherParticipants().stream())
                            .map(TransferMetaHistoryModel.Another::getAccountId).filter(Objects::nonNull)
                ))
                .distinct()
                .map(uid -> new WithTenant<>(tenantId, uid))
                .collect(Collectors.toList());

        return sequencePartitionLoad(ids, idsPart -> tableClient.usingSessionMonoRetryable(session ->
                accountsDao.getByIdsWithDeleted(session.asTxCommitRetryable(TransactionMode.STALE_READ_ONLY),
                        idsPart)));
    }

    private static <T, K> Mono<List<T>> sequencePartitionLoad(List<K> ids, Function<List<K>, Mono<List<T>>> loader) {

        if (ids.isEmpty()) {
            return Mono.just(Collections.emptyList());
        }

        return Lists.partition(ids, (int) Ydb.MAX_RESPONSE_ROWS)
                .stream()
                .map(loader)
                .reduce((m1, m2) -> m1.flatMap(p1 -> m2.map(p2 -> {
                    final ArrayList<T> sum = new ArrayList<>(p1.size() + p2.size());
                    sum.addAll(p1);
                    sum.addAll(p2);
                    return sum;
                }))).orElseGet(() -> Mono.just(Collections.emptyList()));
    }

    private Mono<List<ResourceModel>> getResources(Page<FolderOperationLogModel> page, TenantId tenantId) {
        final List<Tuple2<String, TenantId>> ids = page.getItems().stream()
                .flatMap(fm -> Stream.of(
                        fm.getNewQuotas().asMap().keySet().stream(),
                        fm.getOldQuotas().asMap().keySet().stream(),
                        fm.getNewProvisions().asMap().values().stream().flatMap(qr -> qr.asMap().keySet().stream()),
                        fm.getOldProvisions().asMap().values().stream().flatMap(qr -> qr.asMap().keySet().stream()),
                        fm.getOldBalance().asMap().keySet().stream(),
                        fm.getNewBalance().asMap().keySet().stream(),
                        fm.getActuallyAppliedProvisions().stream()
                                .flatMap(p -> p.asMap().values().stream().flatMap(qr -> qr.asMap().keySet().stream()))
                        ).flatMap(Function.identity())
                )
                .distinct()
                .map(uid -> Tuples.of(uid, tenantId))
                .collect(Collectors.toList());

        return sequencePartitionLoad(ids, resourcesLoader::getResourcesByIdsImmediate);
    }

    private Mono<List<ProviderModel>> getProviders(Page<FolderOperationLogModel> page, List<AccountModel> accounts,
                                                   List<ResourceModel> resources, TenantId tenantId) {
        Set<String> pageProviderIds = page.getItems().stream().flatMap(m -> extractProviderIdsFromOpLog(m).stream())
                .collect(Collectors.toSet());
        Set<String> accountsProviderIds = accounts.stream().map(AccountModel::getProviderId)
                .collect(Collectors.toSet());
        Set<String> resourcesProviderIds = resources.stream().map(ResourceModel::getProviderId)
                .collect(Collectors.toSet());
        List<Tuple2<String, TenantId>> ids = Stream.concat(Stream.concat(pageProviderIds.stream(),
                                accountsProviderIds.stream()), resourcesProviderIds.stream())
                .distinct().map(id -> Tuples.of(id, tenantId)).collect(Collectors.toList());
        return sequencePartitionLoad(ids, providersLoader::getProvidersByIdsImmediate);
    }

    private List<String> extractProviderIdsFromOpLog(FolderOperationLogModel log) {
        List<String> ids = new ArrayList<>();
        log.getAuthorProviderId().ifPresent(ids::add);
        log.getNewAccounts().ifPresent(a -> a.getAccounts().values().forEach(m -> m.getProviderId()
                .ifPresent(ids::add)));
        log.getOldAccounts().ifPresent(a -> a.getAccounts().values().forEach(m -> m.getProviderId()
                .ifPresent(ids::add)));
        return ids;
    }

    private Mono<List<UserModel>> getUsers(Page<FolderOperationLogModel> page, TenantId tenantId) {
        final List<Tuple2<String, TenantId>> ids = page.getItems().stream()
                .map(fm -> fm.getAuthorUserId().orElse(null))
                .filter(Objects::nonNull)
                .distinct()
                .map(id -> Tuples.of(id, tenantId))
                .collect(Collectors.toList());

        return sequencePartitionLoad(ids, idsPart -> tableClient.usingSessionMonoRetryable(session ->
                usersDao.getByIds(session.asTxCommitRetryable(TransactionMode.STALE_READ_ONLY), idsPart)));
    }

    private String prepareToken(FolderOperationLogModel item) {
        return ContinuationTokens.encode(new FolderHistoryContinuationToken(item.getOperationDateTime(), item.getId()),
                continuationTokenWriter);
    }

    private Mono<List<ServiceMinimalModel>> getServices(Page<FolderOperationLogModel> page) {
        final List<Long> ids = page.getItems().stream()
                .flatMap(fm -> {
                    var buf = concat(
                            fm.getNewFolderFields().stream(),
                            fm.getOldFolderFields().stream()
                    ).flatMap(it -> it.getServiceId().stream());
                    buf = concat(buf,
                            fm.getTransferMeta().stream()
                                    .flatMap(it -> it.getAnotherParticipants().stream())
                                    .map(TransferMetaHistoryModel.Another::getServiceId)
                    );
                    return buf;
                })
                .distinct()
                .collect(Collectors.toList());

        return sequencePartitionLoad(ids, idsPart -> tableClient.usingSessionMonoRetryable(session ->
                servicesDao.getByIdsMinimal(session.asTxCommitRetryable(TransactionMode.STALE_READ_ONLY), idsPart)));
    }

    private Mono<Tuple2<List<AccountModel>, ExpandedAccountsSpaces<List<AccountSpaceModel>>>>
    withAccountsSpaces(List<AccountModel> accounts, Page<FolderOperationLogModel> page, TenantId tenantId) {

        Set<String> accountSpacesIds = new HashSet<>();
        Set<WithTenant<String>> providerIds = new HashSet<>();

        for (AccountModel account : accounts) {
            if (account.getAccountsSpacesId().isPresent()) {
                accountSpacesIds.add(account.getAccountsSpacesId().get());
                providerIds.add(new WithTenant<>(tenantId, account.getProviderId()));
            }
        }

        page.getItems().stream().flatMap(history -> concat(
                history.getNewAccounts().stream(),
                history.getOldAccounts().stream())
        ).flatMap(accs -> accs.getAccounts().values().stream())
                .filter(account -> Objects.nonNull(account.getAccountsSpacesId())
                        && account.getProviderId().isPresent())
                .forEach(account -> {
                    accountSpacesIds.add(account.getAccountsSpacesId());
                    providerIds.add(new WithTenant<>(tenantId, account.getProviderId().get()));
                });

        return tableClient.usingSessionMonoRetryable(session -> accountSpacesLoader
                .getAccountSpaces(session.asTxCommitRetryable(TransactionMode.STALE_READ_ONLY), providerIds))
                .map(spaces -> spaces.stream()
                        .filter(space -> accountSpacesIds.contains(space.getId()))
                        .collect(Collectors.toList()))
                .flatMap(accountsSpacesUtils::expandCollection)
                .map(spaces -> Tuples.of(accounts, spaces));
    }
}
