package ru.yandex.intranet.d.services.gc;

import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.util.List;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;

import com.google.common.collect.Lists;
import com.yandex.ydb.table.transaction.TransactionMode;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.dao.accounts.AccountsQuotasDao;
import ru.yandex.intranet.d.dao.quotas.QuotasDao;
import ru.yandex.intranet.d.datasource.model.YdbTableClient;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.model.WithTenant;
import ru.yandex.intranet.d.model.accounts.AccountsQuotasModel;
import ru.yandex.intranet.d.model.quotas.QuotaModel;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

/**
 * Zero quotas and provisions GC service implementation.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class QuotasProvisionsGCService {

    private static final Logger LOG = LoggerFactory.getLogger(QuotasProvisionsGCService.class);

    private static final Duration PROVISIONS_COOL_DOWN_PERIOD = Duration.ofDays(1L);
    private static final long SCAN_LIMIT = 10000L;
    private static final Duration SCAN_TIMEOUT = Duration.ofMinutes(5L);
    private static final int GC_PAGE_SIZE = 1000;
    private static final String DURATION_SINCE_LAST_SUCCESS = "cron.jobs.duration_since_last_success_millis";
    private static final String JOB = "job";
    private static final String JOB_NAME = "QuotasProvisionsGCJob";

    private final YdbTableClient tableClient;
    private final QuotasDao quotasDao;
    private final AccountsQuotasDao accountsQuotasDao;
    private final AtomicReference<Instant> lastSuccess = new AtomicReference<>(Instant.now());

    public QuotasProvisionsGCService(YdbTableClient tableClient,
                                     QuotasDao quotasDao,
                                     AccountsQuotasDao accountsQuotasDao) {
        this.tableClient = tableClient;
        this.quotasDao = quotasDao;
        this.accountsQuotasDao = accountsQuotasDao;
        MetricRegistry.root().lazyGaugeInt64(DURATION_SINCE_LAST_SUCCESS,
                Labels.of(JOB, JOB_NAME), () -> Duration.between(lastSuccess.get(), Instant.now()).toMillis());
    }

    public Mono<Void> gcQuotasProvisions(Clock clock) {
        Instant now = Instant.now(clock);
        Instant lastUpdateThreshold = now.minus(PROVISIONS_COOL_DOWN_PERIOD);
        return tableClient.usingSessionMonoRetryable(session ->
                quotasDao.scanZeroQuotas(session, SCAN_LIMIT, SCAN_TIMEOUT).collectList().flatMap(ids ->
                        Flux.fromIterable(Lists.partition(ids, GC_PAGE_SIZE)).concatMap(page ->
                                session.usingTxMonoRetryable(TransactionMode.SERIALIZABLE_READ_WRITE,
                                        ts -> collectQuotasPage(ts, page))).collectList()
                                .doOnSuccess(counts -> {
                                    long total = counts.stream().mapToLong(v -> v).sum();
                                    if (total > 0) {
                                        LOG.info("Successfully collected {} zero quotas", total);
                                    }
                                }).then())
                        .then(accountsQuotasDao.scanZeroAccountsQuotas(session, lastUpdateThreshold,
                                SCAN_LIMIT, SCAN_TIMEOUT).collectList().flatMap(ids ->
                        Flux.fromIterable(Lists.partition(ids, GC_PAGE_SIZE)).concatMap(page ->
                                session.usingTxMonoRetryable(TransactionMode.SERIALIZABLE_READ_WRITE,
                                        ts -> collectAccountsQuotasPage(ts, page, lastUpdateThreshold))).collectList()
                                .doOnSuccess(counts -> {
                                    long total = counts.stream().mapToLong(v -> v).sum();
                                    if (total > 0) {
                                        LOG.info("Successfully collected {} zero provisions", total);
                                    }
                                })
                                .then()))).doOnSuccess(v -> lastSuccess.set(Instant.now()));
    }

    private Mono<Long> collectQuotasPage(YdbTxSession session, List<WithTenant<QuotaModel.Key>> page) {
        return quotasDao.getByKeys(session, page).flatMap(quotas -> {
            List<QuotaModel> zeroQuotas = quotas.stream().filter(this::isZeroQuota).collect(Collectors.toList());
            return quotasDao.deleteQuotasModelsRetryable(session, zeroQuotas).thenReturn((long) zeroQuotas.size());
        });
    }

    private boolean isZeroQuota(QuotaModel quota) {
        return (quota.getQuota() == null || quota.getQuota() == 0L)
                && (quota.getBalance() == null || quota.getBalance() == 0L)
                && quota.getFrozenQuota() == 0L;
    }

    private Mono<Long> collectAccountsQuotasPage(YdbTxSession session,
                                                 List<WithTenant<AccountsQuotasModel.Identity>> page,
                                                 Instant lastUpdateThreshold) {
        return accountsQuotasDao.getByIds(session, page).flatMap(provisions -> {
            List<AccountsQuotasModel> zeroProvisions = provisions.stream()
                    .filter(provision -> isZeroAccountQuota(provision, lastUpdateThreshold))
                    .collect(Collectors.toList());
            return accountsQuotasDao.removeAllModelsRetryable(session, zeroProvisions)
                    .thenReturn((long) zeroProvisions.size());
        });
    }

    private boolean isZeroAccountQuota(AccountsQuotasModel quota, Instant lastUpdateThreshold) {
        return (quota.getProvidedQuota() == null || quota.getProvidedQuota() == 0L)
                && (quota.getAllocatedQuota() == null || quota.getAllocatedQuota() == 0L)
                && (quota.getLastProvisionUpdate() == null
                        || quota.getLastProvisionUpdate().isBefore(lastUpdateThreshold));
    }

}
