package ru.yandex.intranet.d.services.idm;

import java.util.Map;
import java.util.function.Predicate;
import java.util.function.UnaryOperator;
import java.util.stream.Collectors;

import com.yandex.ydb.table.transaction.TransactionMode;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.dao.Tenants;
import ru.yandex.intranet.d.dao.users.UsersDao;
import ru.yandex.intranet.d.datasource.model.WithTxId;
import ru.yandex.intranet.d.datasource.model.YdbTableClient;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.loaders.users.UsersLoader;
import ru.yandex.intranet.d.model.users.UserModel;
import ru.yandex.intranet.d.web.model.idm.IdmResult;
import ru.yandex.intranet.d.web.model.idm.IdmResultRoleGroup;
import ru.yandex.intranet.d.web.model.idm.IdmResultUsersAnswer;
import ru.yandex.intranet.d.web.model.idm.IdmUpdate;
import ru.yandex.intranet.d.web.model.idm.RoleGroup;
import ru.yandex.intranet.d.web.model.idm.UserRoles;


/**
 * Methods for sync with IDM
 *
 * @author Ruslan Kadriev <aqru@yandex-team.ru>
 * @since 06.10.2020
 */
@Component
public class IdmService {
    private static final Logger LOG = LoggerFactory.getLogger(IdmService.class);

    private static final  String ROLE_SLUG = "group";
    private static final  String ROLE_NAME = "группа";
    private static final  String ADMIN_ROLE = "admin";
    private static final  String ADMIN_ROLE_NAME = "администратор";
    private static final Map<String, String> ADMIN_ROLE_MAP = Map.of(ROLE_SLUG, ADMIN_ROLE);
    private static final IdmResultRoleGroup ROLES_INFO = new IdmResultRoleGroup(
            RoleGroup.builder()
                    .slug(ROLE_SLUG)
                    .name(ROLE_NAME)
                    .addRole(ADMIN_ROLE, ADMIN_ROLE_NAME)
                    .build()
    );

    private final UsersDao usersDao;
    private final YdbTableClient tableClient;
    private final UsersLoader usersLoader;

    public IdmService(UsersDao usersDao, YdbTableClient tableClient, UsersLoader usersLoader) {
        this.usersDao = usersDao;
        this.tableClient = tableClient;
        this.usersLoader = usersLoader;
    }

    public Mono<IdmResult> getRolesInfo() {
        return Mono.just(ROLES_INFO);
    }

    public Mono<IdmResult> getAllRoles() {
        return tableClient.usingSessionMonoRetryable(
                session -> usersDao.getDAdmins(session.asTxCommitRetryable(TransactionMode.ONLINE_READ_ONLY),
                        Tenants.DEFAULT_TENANT_ID)
                        .map(
                                users -> users.stream()
                                        .map(user -> UserRoles.fromUserModel(user, ADMIN_ROLE_MAP))
                                        .collect(Collectors.toList())
                        )
                        .map(IdmResultUsersAnswer::new)
        );
    }

    public Mono<IdmResult> addRole(IdmUpdate idmUpdate) {
        String uid = idmUpdate.getUid();
        return checkRole(idmUpdate.getRoleMap())
                .switchIfEmpty(updateRole(uid, UserModel::getDAdmin, String.format(
                        "User with uid = [%s] already have role.", uid),
                        (user) -> user.copyBuilder().dAdmin(Boolean.TRUE).build())
                );
    }

    public Mono<IdmResult> removeRole(IdmUpdate idmUpdate) {
        String uid = idmUpdate.getUid();
        return checkRole(idmUpdate.getRoleMap())
                .switchIfEmpty(updateRole(uid, (user) -> !user.getDAdmin(), String.format(
                        "User with uid = [%s] already haven't role.", uid),
                        (user) -> user.copyBuilder().dAdmin(Boolean.FALSE).build())
                );
    }

    private Mono<IdmResult> updateRole(String uid, Predicate<UserModel> noNeedToUpdatePredicate,
                                       String noNeedToUpdateMessage, UnaryOperator<UserModel> userUpdatePreparerOp) {
        return tableClient.usingSessionMonoRetryable(session -> session.usingCompTxMonoRetryable(
                TransactionMode.SERIALIZABLE_READ_WRITE,
                ts -> usersDao.getByPassportUidTx(ts, uid, Tenants.DEFAULT_TENANT_ID).map(WithTxId::asTuple),
                (ts, userO) -> Mono.just(userO),
                (ts, userO) -> {
                    if (userO.isPresent()) {
                        final UserModel userModel = userO.get();

                        if (noNeedToUpdatePredicate.test(userModel)) {
                            return ts.commitTransaction()
                                    .thenReturn(IdmResult.newWarningMessageOkIdmResult(noNeedToUpdateMessage));
                        }

                        return updateUser(userUpdatePreparerOp.apply(userModel), ts);
                    } else {
                        final String message =
                                String.format("User with uid = [%s] not found.", uid);
                        LOG.warn(message);
                        return ts.commitTransaction()
                                .thenReturn(IdmResult.newFatalMessageIdmResult(message));
                    }
                }
        ));
    }

    private Mono<IdmResult> checkRole(Map<String, String> role) {
        if (role == null) {
            return Mono.just(IdmResult.newFatalMessageIdmResult("Role must be provided."));
        }

        final String value = role.get(ROLE_SLUG);
        if (!ADMIN_ROLE.equals(value)) {
            return Mono.just(IdmResult.newFatalMessageIdmResult("Role not existing."));
        }

        return Mono.empty();
    }

    private Mono<IdmResult> updateUser(UserModel userModel, YdbTxSession ydbTxSession) {
        return usersDao.updateUserRetryable(ydbTxSession, userModel)
                .doOnSuccess(m -> usersLoader.updateUser(userModel))
                .thenReturn(IdmResult.newOkIdmResult());
    }
}
