package ru.yandex.intranet.d.services.imports;

import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Function;

import ru.yandex.intranet.d.web.model.imports.AccountSpaceIdentityDto;
import ru.yandex.intranet.d.web.model.imports.SegmentKey;

/**
 * Accounts space identity by string id or set of segments.
 *
 * @author Vladimir Zaytsev <vzay@yandex-team.ru>
 * @since 24.12.2020
 */
public abstract class AccountSpaceIdentity {
    private static final AccountSpaceIdentity EMPTY = new AccountSpaceIdentity() {
        @Override
        public <T> Optional<T> map(Function<String, T> accountSpaceIdMapper,
                                   Function<List<SegmentKey>, T> segmentsMapper) {
            return Optional.empty();
        }

        @Override
        public boolean isEmpty() {
            return true;
        }

        @Override
        public boolean equals(Object o) {
            return this == o;
        }

        @Override
        public int hashCode() {
            return 0;
        }

        @Override
        public String toString() {
            return "AccountSpaceIdentity{null}";
        }
    };

    private AccountSpaceIdentity() {
    }

    public abstract <T> Optional<T> map(
            Function<String, T> accountSpaceIdMapper, Function<List<SegmentKey>, T> segmentsMapper
    );

    public void consume(Consumer<String> accountSpaceIdConsumer, Consumer<List<SegmentKey>> segmentsConsumer) {
        map(accountSpaceId -> {
            accountSpaceIdConsumer.accept(accountSpaceId);
            return null;
        }, segmentKeys -> {
            segmentsConsumer.accept(segmentKeys);
            return null;
        });
    }

    public abstract boolean isEmpty();

    public boolean isPresent() {
        return !isEmpty();
    }

    public static AccountSpaceIdentity of() {
        return EMPTY;
    }

    public static AccountSpaceIdentity of(String accountSpaceId) {
        return new AccountSpaceIdentity() {
            @Override
            public <T> Optional<T> map(
                    Function<String, T> accountSpaceIdMapper,
                    Function<List<SegmentKey>, T> segmentsMapper
            ) {
                return Optional.ofNullable(accountSpaceIdMapper.apply(accountSpaceId));
            }

            @Override
            public boolean isEmpty() {
                return false;
            }

            @Override
            public boolean equals(Object o) {
                if (this == o) {
                    return true;
                }
                if (o == null || getClass() != o.getClass()) {
                    return false;
                }
                AccountSpaceIdentity that = (AccountSpaceIdentity) o;
                return that.map(
                        accountSpaceId2 -> Objects.equals(accountSpaceId, accountSpaceId2),
                        segment -> false
                ).orElse(false);
            }

            @Override
            public int hashCode() {
                return Objects.hash(accountSpaceId);
            }

            @Override
            public String toString() {
                return "AccountSpaceIdentity{" +
                        "accountSpaceId='" + accountSpaceId + '\'' +
                        '}';
            }
        };
    }

    public static AccountSpaceIdentity of(List<SegmentKey> segments) {
        return new AccountSpaceIdentity() {
            @Override
            public <T> Optional<T> map(
                    Function<String, T> accountSpaceIdMapper,
                    Function<List<SegmentKey>, T> segmentsMapper
            ) {
                return Optional.ofNullable(segmentsMapper.apply(segments));
            }

            @Override
            public boolean isEmpty() {
                return false;
            }

            @Override
            public boolean equals(Object o) {
                if (this == o) {
                    return true;
                }
                if (o == null || getClass() != o.getClass()) {
                    return false;
                }
                AccountSpaceIdentity that = (AccountSpaceIdentity) o;
                return that.map(
                        accountSpaceId -> false,
                        segments2 -> Objects.equals(segments, segments2)
                ).orElse(false);
            }

            @Override
            public int hashCode() {
                return Objects.hash(segments);
            }

            @Override
            public String toString() {
                return "AccountSpaceIdentity{" +
                        "segments=" + segments +
                        '}';
            }
        };
    }

    public static AccountSpaceIdentity fromDto(AccountSpaceIdentityDto dto) {
        if (dto == null) {
            return of();
        }
        if (dto.getAccountSpaceId() != null) {
            return of(dto.getAccountSpaceId());
        }
        if (dto.getSegments() != null) {
            return of(dto.getSegments());
        }
        return of();
    }
}
