package ru.yandex.intranet.d.services.imports;

import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import com.google.common.collect.Streams;
import reactor.util.function.Tuple2;
import reactor.util.function.Tuples;

import ru.yandex.intranet.d.model.resources.segmentations.ResourceSegmentationModel.ProviderKey;
import ru.yandex.intranet.d.web.model.imports.ImportFailureDto;
import ru.yandex.intranet.d.web.model.imports.ImportFolderDto;

/**
 * Pre-validated import info.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public final class PreValidatedImport {

    private final List<ImportFailureDto> failures;
    private final List<ImportFolderDto> quotas;
    private final Set<String> providerIds;
    private final Set<String> resourceIds;
    private final Set<ProviderWithResourceIdentity> providerWithResourceIdentitySet;
    private final Set<AccountExternalId> accountIds;
    private final Set<String> folderIds;
    private final Set<Long> serviceIds;
    private final Map<AccountExternalId, String> preGeneratedAccountIds;
    private final Map<Long, String> preGeneratedFolderIds;
    private final Map<String, Integer> folderIndices;
    private final Map<Long, Integer> serviceIndices;
    private final Map<String, Set<AccountSpaceIdentity>> accountSpaceIdsByProviderId;
    private final Map<ProviderKey, Set<String>> segmentKeys;

    @SuppressWarnings("ParameterNumber")
    public PreValidatedImport(List<ImportFailureDto> failures, List<ImportFolderDto> quotas, Set<String> providerIds,
                              Set<String> resourceIds,
                              Set<ProviderWithResourceIdentity> providerWithResourceIdentitySet,
                              Set<AccountExternalId> accountIds, Set<String> folderIds, Set<Long> serviceIds,
                              Map<AccountExternalId, String> preGeneratedAccountIds,
                              Map<Long, String> preGeneratedFolderIds, Map<String, Integer> folderIndices,
                              Map<Long, Integer> serviceIndices) {
        this.failures = failures;
        this.quotas = quotas;
        this.providerIds = providerIds;
        this.resourceIds = resourceIds;
        this.providerWithResourceIdentitySet = providerWithResourceIdentitySet;
        this.accountIds = accountIds;
        this.folderIds = folderIds;
        this.serviceIds = serviceIds;
        this.preGeneratedAccountIds = preGeneratedAccountIds;
        this.preGeneratedFolderIds = preGeneratedFolderIds;
        this.folderIndices = folderIndices;
        this.serviceIndices = serviceIndices;

        var resourceIdentityAccountSpaceSt = providerWithResourceIdentitySet.stream()
                .filter(resourceIdentity -> resourceIdentity.getAccountSpace().isPresent())
                .collect(Collectors.groupingBy(ProviderWithResourceIdentity::getProviderId,
                        Collectors.mapping(resourceIdentity -> resourceIdentity.getAccountSpace().orElseThrow(),
                                Collectors.toList())))
                .entrySet().stream();
        var resourceAccountSpaceSt = accountIds.stream()
                .collect(Collectors.groupingBy(AccountExternalId::getProviderId,
                        Collectors.mapping(AccountExternalId::getAccountSpaceIdentity, Collectors.toList())))
                .entrySet().stream();
        this.accountSpaceIdsByProviderId = Streams.concat(resourceIdentityAccountSpaceSt, resourceAccountSpaceSt)
                .collect(Collectors.toMap(Map.Entry::getKey, entry -> new HashSet<>(entry.getValue()), (a1, a2) -> {
                    a1.addAll(a2);
                    return a1;
                }));

        var resourceIdentitySegmentSt = providerWithResourceIdentitySet.stream()
                .flatMap(resourceIdentity -> resourceIdentity.getResource().getSegmentKeyBySegmentationKey().entrySet()
                        .stream().map(segment -> Tuples.of(resourceIdentity.getProviderId(), segment)))
                .map(tuple -> Tuples.of(new ProviderKey(tuple.getT1(), tuple.getT2().getKey()),
                        tuple.getT2().getValue()));
        var accountSpacesSegmentsSt = accountSpaceIdsByProviderId.entrySet().stream()
                .flatMap(providerIdEntry -> providerIdEntry.getValue().stream().flatMap(accountSpaceIdentity ->
                        accountSpaceIdentity.map(s -> null, Collection::stream).orElse(null))
                        .map(segmentKey -> Tuples.of(
                                new ProviderKey(providerIdEntry.getKey(), segmentKey.getSegmentationKey()),
                                segmentKey.getSegmentKey()
                        ))
                );
        this.segmentKeys = Streams.concat(resourceIdentitySegmentSt, accountSpacesSegmentsSt)
                .collect(Collectors.groupingBy(Tuple2::getT1, Collectors.mapping(Tuple2::getT2, Collectors.toSet())));
    }

    public List<ImportFailureDto> getFailures() {
        return failures;
    }

    public List<ImportFolderDto> getQuotas() {
        return quotas;
    }

    public Set<String> getProviderIds() {
        return providerIds;
    }

    public Set<String> getResourceIds() {
        return resourceIds;
    }

    public Set<ProviderWithResourceIdentity> getProviderWithResourceIdentitySet() {
        return providerWithResourceIdentitySet;
    }

    public Set<AccountExternalId> getAccountIds() {
        return accountIds;
    }

    public Set<String> getFolderIds() {
        return folderIds;
    }

    public Set<Long> getServiceIds() {
        return serviceIds;
    }

    public Map<AccountExternalId, String> getPreGeneratedAccountIds() {
        return preGeneratedAccountIds;
    }

    public Map<Long, String> getPreGeneratedFolderIds() {
        return preGeneratedFolderIds;
    }

    public Map<String, Integer> getFolderIndices() {
        return folderIndices;
    }

    public Map<Long, Integer> getServiceIndices() {
        return serviceIndices;
    }

    public Map<String, Set<AccountSpaceIdentity>> getAccountSpaceIdsByProviderId() {
        return accountSpaceIdsByProviderId;
    }

    public Map<ProviderKey, Set<String>> getSegmentKeys() {
        return segmentKeys;
    }

    public static final class AccountExternalId {
        private final String providerId;
        private final String outerAccountIdInProvider;
        private final AccountSpaceIdentity accountSpaceIdentity;

        public AccountExternalId(String providerId, String outerAccountIdInProvider,
                                 AccountSpaceIdentity accountSpaceIdentity) {
            this.providerId = providerId;
            this.outerAccountIdInProvider = outerAccountIdInProvider;
            this.accountSpaceIdentity = accountSpaceIdentity;
        }

        public String getProviderId() {
            return providerId;
        }

        public String getOuterAccountIdInProvider() {
            return outerAccountIdInProvider;
        }

        public AccountSpaceIdentity getAccountSpaceIdentity() {
            return accountSpaceIdentity;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            AccountExternalId that = (AccountExternalId) o;
            return Objects.equals(providerId, that.providerId) && Objects.equals(outerAccountIdInProvider,
                    that.outerAccountIdInProvider) && Objects.equals(accountSpaceIdentity, that.accountSpaceIdentity);
        }

        @Override
        public int hashCode() {
            return Objects.hash(providerId, outerAccountIdInProvider, accountSpaceIdentity);
        }

        @Override
        public String toString() {
            return "AccountExternalId{" +
                    "providerId='" + providerId + '\'' +
                    ", outerAccountIdInProvider='" + outerAccountIdInProvider + '\'' +
                    ", accountSpaceIdentity=" + accountSpaceIdentity +
                    '}';
        }
    }
}
