package ru.yandex.intranet.d.services.imports;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nullable;

import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuple2;
import reactor.util.function.Tuple3;
import reactor.util.function.Tuples;

import ru.yandex.intranet.d.dao.Tenants;
import ru.yandex.intranet.d.dao.accounts.AccountsDao;
import ru.yandex.intranet.d.dao.accounts.AccountsQuotasDao;
import ru.yandex.intranet.d.dao.folders.FolderDao;
import ru.yandex.intranet.d.dao.providers.ProvidersDao;
import ru.yandex.intranet.d.dao.quotas.QuotasDao;
import ru.yandex.intranet.d.dao.resources.ResourcesDao;
import ru.yandex.intranet.d.dao.services.ServicesDao;
import ru.yandex.intranet.d.dao.units.UnitsEnsemblesDao;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.loaders.providers.AccountSpacesLoader;
import ru.yandex.intranet.d.loaders.resources.segmentations.ResourceSegmentationsLoaderByKey;
import ru.yandex.intranet.d.loaders.resources.segments.ResourceSegmentsLoaderByKey;
import ru.yandex.intranet.d.loaders.resources.types.ResourceTypesLoader;
import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.WithTenant;
import ru.yandex.intranet.d.model.accounts.AccountModel;
import ru.yandex.intranet.d.model.accounts.AccountSpaceModel;
import ru.yandex.intranet.d.model.accounts.AccountsQuotasModel;
import ru.yandex.intranet.d.model.folders.FolderModel;
import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.model.quotas.QuotaModel;
import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.model.resources.ResourceSegmentSettingsModel;
import ru.yandex.intranet.d.model.resources.segmentations.ResourceSegmentationModel;
import ru.yandex.intranet.d.model.resources.segments.ResourceSegmentModel;
import ru.yandex.intranet.d.model.resources.segments.ResourceSegmentModel.SegmentationAndKey;
import ru.yandex.intranet.d.model.resources.types.ResourceTypeModel;
import ru.yandex.intranet.d.model.services.ServiceWithStatesModel;
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel;
import ru.yandex.intranet.d.services.imports.PreValidatedImport.AccountExternalId;
import ru.yandex.intranet.d.services.integration.providers.rest.model.ResourceComplexKey;
import ru.yandex.intranet.d.web.model.imports.SegmentKey;

/**
 * QuotasImportsLoadingService.
 *
 * @author Vladimir Zaytsev <vzay@yandex-team.ru>
 * @since 30.12.2020
 */
@Component
public class QuotasImportsLoadingService {
    private final ProvidersDao providersDao;
    private final ResourcesDao resourcesDao;
    private final FolderDao folderDao;
    private final AccountsDao accountsDao;
    private final ServicesDao servicesDao;
    private final UnitsEnsemblesDao unitsEnsemblesDao;
    private final QuotasDao quotasDao;
    private final AccountsQuotasDao accountsQuotasDao;
    private final AccountSpacesLoader accountSpacesLoader;
    private final ResourceSegmentationsLoaderByKey resourceSegmentationsLoaderByKey;
    private final ResourceSegmentsLoaderByKey resourceSegmentsLoaderByKey;
    private final ResourceTypesLoader resourceTypesLoader;

    @SuppressWarnings("ParameterNumber")
    public QuotasImportsLoadingService(
            ProvidersDao providersDao,
            ResourcesDao resourcesDao,
            FolderDao folderDao,
            AccountsDao accountsDao,
            ServicesDao servicesDao,
            UnitsEnsemblesDao unitsEnsemblesDao,
            QuotasDao quotasDao,
            AccountsQuotasDao accountsQuotasDao,
            AccountSpacesLoader accountSpacesLoader,
            ResourceSegmentationsLoaderByKey resourceSegmentationsLoaderByKey,
            ResourceSegmentsLoaderByKey resourceSegmentsLoaderByKey,
            ResourceTypesLoader resourceTypesLoader
    ) {
        this.providersDao = providersDao;
        this.resourcesDao = resourcesDao;
        this.folderDao = folderDao;
        this.accountsDao = accountsDao;
        this.servicesDao = servicesDao;
        this.unitsEnsemblesDao = unitsEnsemblesDao;
        this.quotasDao = quotasDao;
        this.accountsQuotasDao = accountsQuotasDao;
        this.accountSpacesLoader = accountSpacesLoader;
        this.resourceSegmentationsLoaderByKey = resourceSegmentationsLoaderByKey;
        this.resourceSegmentsLoaderByKey = resourceSegmentsLoaderByKey;
        this.resourceTypesLoader = resourceTypesLoader;
    }

    public Mono<ValidatedDirectories> loadDirectories(PreValidatedImport quotasToImport,
                                                      YdbTxSession session,
                                                      boolean withDeletedResourceTypes) {
        return loadProviders(session, quotasToImport.getProviderIds())
                .flatMap(providers -> loadSegmentations(session, quotasToImport.getSegmentKeys())
                .flatMap(segmentations -> loadAccountSpaces(session, quotasToImport.getAccountSpaceIdsByProviderId(),
                    segmentations)
                .flatMap(accountSpaces -> loadResourceTypes(session,
                        quotasToImport.getProviderWithResourceIdentitySet(),
                        withDeletedResourceTypes)
                .flatMap(resourcesTypes -> loadResourcesByIdentity(session, accountSpaces, segmentations,
                    resourcesTypes, quotasToImport.getProviderWithResourceIdentitySet())
                .flatMap(resourcesByComplexkey -> loadResources(session, quotasToImport.getResourceIds())
                .flatMap(resources -> loadAccounts(session, accountSpaces, quotasToImport.getAccountIds())
                .flatMap(accounts -> loadFolders(session, quotasToImport.getFolderIds())
                .flatMap(folders -> loadServices(session, Stream.concat(quotasToImport.getServiceIds().stream(),
                        folders.values().stream().map(FolderModel::getServiceId)).collect(Collectors.toSet()))
                .flatMap(services -> loadUnitsEnsembles(session, Stream.concat(resources.values().stream(),
                        resourcesByComplexkey.values().stream()).collect(Collectors.toSet()))
                .flatMap(unitsEnsembles -> loadDefaultFolders(session, services.values())
                .map(defaultFolders -> {
                    Map<String, FolderModel> mergedFolders = new HashMap<>(folders);
                    defaultFolders.forEach((k, v) -> mergedFolders.put(v.getId(), v));
                    Map<String, ResourceModel> mergedResources = new HashMap<>(resources);
                    resourcesByComplexkey.values().forEach(resourceModel ->
                            mergedResources.put(resourceModel.getId(), resourceModel));
                    return new ValidatedDirectories(
                            providers, accountSpaces, resourcesByComplexkey, mergedResources, accounts, mergedFolders,
                            services, unitsEnsembles, defaultFolders, segmentations
                    );
                })))))))))));
    }

    private Mono<Map<ResourceSegmentationModel.ProviderKey, SegmentationInfo>> loadSegmentations(
            YdbTxSession session, Map<ResourceSegmentationModel.ProviderKey, Set<String>> segmentsBySegmentations
    ) {
        return resourceSegmentationsLoaderByKey
                .getResourceSegmentationByKeys(session, segmentsBySegmentations.keySet().stream()
                        .map(providerKey -> Tuples.of(providerKey, Tenants.DEFAULT_TENANT_ID))
                        .collect(Collectors.toList())
                )
                .map(segmentations -> segmentations.stream().collect(Collectors.toMap(
                        ResourceSegmentationModel::getProviderKey, Function.identity()))
                )
                .flatMap(segmentations -> {
                    Map<String, ResourceSegmentationModel.ProviderKey> segmentationsKeysById =
                            segmentations.values().stream().collect(Collectors.toMap(
                                    ResourceSegmentationModel::getId, ResourceSegmentationModel::getProviderKey));
                    return resourceSegmentsLoaderByKey
                            .getResourceSegmentsByKeys(session, segmentsBySegmentations.entrySet().stream()
                                    .flatMap(entry -> segmentations.containsKey(entry.getKey()) ?
                                            entry.getValue().stream().map(segmentKey -> Tuples.of(
                                                    new SegmentationAndKey(
                                                            segmentations.get(entry.getKey()).getId(),
                                                            segmentKey
                                                    ),
                                                    Tenants.DEFAULT_TENANT_ID
                                            ))
                                            : Stream.empty())
                                    .collect(Collectors.toList())
                            )
                            .map(segments -> segments.stream().collect(
                                    Collectors.groupingBy(
                                            segment -> segmentationsKeysById.get(segment.getSegmentationId()),
                                            Collectors.toMap(ResourceSegmentModel::getKey, Function.identity())
                                    ))
                            )
                            .map(providerKeyMap -> providerKeyMap.entrySet().stream()
                                    .map(providerKeyEntry -> Tuples.of(
                                            providerKeyEntry.getKey(),
                                            new SegmentationInfo(
                                                    segmentations.get(providerKeyEntry.getKey()),
                                                    providerKeyEntry.getValue()
                                            ))
                                    )
                                    .collect(Collectors.toMap(Tuple2::getT1, Tuple2::getT2))
                            );
                });
    }

    public Mono<Map<AccountSpaceIdentity, AccountSpaceModel>> loadAccountSpaces(
            YdbTxSession session,
            Map<String, Set<AccountSpaceIdentity>> accountSpaceIdsByProviderId,
            Map<ResourceSegmentationModel.ProviderKey, SegmentationInfo> segmentations
    ) {
        return Flux.fromIterable(accountSpaceIdsByProviderId.entrySet()).concatMap(entry -> {
            String providerId = entry.getKey();
            Set<AccountSpaceIdentity> accountSpaceIds = entry.getValue();
            return getAccountSpaces(session, Tenants.DEFAULT_TENANT_ID, providerId, accountSpaceIds,
                    segmentations);
        })
                .collect(HashMap::new, Map::putAll);
    }

    private Mono<Map<AccountSpaceIdentity, AccountSpaceModel>> getAccountSpaces(
            YdbTxSession session,
            TenantId tenantId, String providerId, Set<AccountSpaceIdentity> accountSpaceIds,
            Map<ResourceSegmentationModel.ProviderKey, SegmentationInfo> segmentations
    ) {
        return Flux.fromIterable(accountSpaceIds)
                .concatMap(id -> id
                        .map(
                                stringId -> accountSpacesLoader.getAccountSpaces(
                                        session, tenantId, providerId, stringId),
                                segmentKeys -> getAccountSpacesByKey(segmentations,
                                        session, tenantId, providerId, segmentKeys)
                        )
                        .orElse(Mono.empty())
                        .map(accountSpaces -> accountSpaces.flatMap(l ->
                                l.size() == 1 ? Optional.ofNullable(l.get(0)) : Optional.empty()
                        ))
                        .flatMap(Mono::justOrEmpty)
                        .map(accountSpace -> Tuples.of(id, accountSpace))
                )
                .collect(HashMap::new, (map, tuple) -> map.put(tuple.getT1(), tuple.getT2()));
    }

    public Mono<Optional<List<AccountSpaceModel>>> getAccountSpacesByKey(
            Map<ResourceSegmentationModel.ProviderKey, SegmentationInfo> segmentations,
            YdbTxSession session,
            TenantId tenantId, String providerId, List<SegmentKey> segmentKeys
    ) {
        Set<ResourceSegmentSettingsModel> segments = segmentKeys.stream()
                .map((Function<SegmentKey,
                        Optional<ResourceSegmentSettingsModel>>) segmentKey -> {
                    SegmentationInfo segmentationInfo = segmentations.get(new ResourceSegmentationModel.ProviderKey(
                            providerId,
                            segmentKey.getSegmentationKey()
                    ));
                    if (segmentationInfo == null) {
                        return Optional.empty();
                    }
                    ResourceSegmentationModel segmentationModel = segmentationInfo.getSegmentationModel();
                    ResourceSegmentModel segmentModel =
                            segmentationInfo.getSegmentsByKey().get(segmentKey.getSegmentKey());
                    if (segmentModel == null) {
                        return Optional.empty();
                    }
                    return Optional.of(new ResourceSegmentSettingsModel(
                            segmentationModel.getId(),
                            segmentModel.getId()
                    ));
                })
                .filter(Optional::isPresent)
                .map(Optional::get)
                .collect(Collectors.toSet());
        return accountSpacesLoader.getAccountSpaces(session, tenantId, providerId, segments);
    }

    public Mono<Map<String, ProviderModel>> loadProviders(YdbTxSession session, Set<String> providerIds) {
        if (providerIds.isEmpty()) {
            return Mono.just(Map.of());
        }
        List<Tuple2<String, TenantId>> ids = providerIds.stream().map(id -> Tuples.of(id, Tenants.DEFAULT_TENANT_ID))
                .collect(Collectors.toList());
        return Flux.fromIterable(Lists.partition(ids, 500)).concatMap(v -> providersDao.getByIds(session, v))
                .collectList().map(l -> l.stream().flatMap(Collection::stream)
                        .collect(Collectors.toMap(ProviderModel::getId, Function.identity())));
    }

    public Mono<Map<String, String>> loadResourceTypes(
            YdbTxSession session,
            Set<ProviderWithResourceIdentity> resourceIdentitySet,
            boolean withDeleted) {
        if (resourceIdentitySet.isEmpty()) {
            return Mono.just(Map.of());
        }
        List<Tuple2<Tuple2<String, String>, TenantId>> resourceTypes = resourceIdentitySet.stream()
                .map(resourceIdentity -> Tuples.of(Tuples.of(resourceIdentity.getProviderId(),
                        resourceIdentity.getResource().getResourceTypeKey()),
                        Tenants.DEFAULT_TENANT_ID
                ))
                .distinct()
                .collect(Collectors.toList());

        return resourceTypesLoader.getResourceTypesByProviderAndKeys(session, resourceTypes)
                .map(resourceTypeModels -> resourceTypeModels.stream()
                        .filter(resourceType -> withDeleted || !resourceType.isDeleted())
                        .collect(Collectors.toMap(ResourceTypeModel::getKey, ResourceTypeModel::getId)));
    }

    public Mono<Map<ResourceComplexKey, ResourceModel>>
    loadResourcesByIdentity(YdbTxSession session, Map<AccountSpaceIdentity, AccountSpaceModel> accountSpaces,
                            Map<ResourceSegmentationModel.ProviderKey, SegmentationInfo> segmentations,
                            Map<String, String> resourcesTypes,
                            Set<ProviderWithResourceIdentity> resourceIdentitySet
    ) {
        if (resourceIdentitySet.isEmpty()) {
            return Mono.just(Map.of());
        }
        Map<Tuple3<String, Optional<String>, Set<ResourceSegmentSettingsModel>>, ResourceComplexKey>
                resourceComplexKeyMap = new HashMap<>();
        List<ResourceIdentity> resourceIdentities = resourceIdentitySet.stream()
                .map((Function<ProviderWithResourceIdentity, Optional<ResourceIdentity>>) resourceIdentity -> {
                    String providerId = resourceIdentity.getProviderId();
                    ResourceComplexKey resourceComplexKey =  resourceIdentity.getResource();
                    Optional<AccountSpaceIdentity> accountSpaceIdentity = resourceIdentity.getAccountSpace();

                    Set<ResourceSegmentSettingsModel> segmentSettingsModels = new HashSet<>();

                    for (Map.Entry<String, String> segmentKeyBySegmentationKey :
                            resourceComplexKey.getSegmentKeyBySegmentationKey().entrySet()) {
                        SegmentationInfo segmentationInfo = segmentations.get(
                                new ResourceSegmentationModel.ProviderKey(
                                        providerId,
                                        segmentKeyBySegmentationKey.getKey()
                                ));
                        if (segmentationInfo == null) {
                            return Optional.empty();
                        }
                        ResourceSegmentationModel segmentationModel = segmentationInfo.getSegmentationModel();
                        ResourceSegmentModel segmentModel =
                                segmentationInfo.getSegmentsByKey().get(segmentKeyBySegmentationKey.getValue());
                        if (segmentModel == null) {
                            return Optional.empty();
                        }
                        segmentSettingsModels.add(new ResourceSegmentSettingsModel(
                                segmentationModel.getId(),
                                segmentModel.getId()
                        ));
                    }

                    String accountSpaceId = null;
                    if (accountSpaceIdentity.isPresent()) {
                        AccountSpaceModel accountSpaceModel = accountSpaces.get(accountSpaceIdentity.get());
                        if (accountSpaceModel == null) {
                            return Optional.empty();
                        }

                        accountSpaceId = accountSpaceModel.getId();
                        segmentSettingsModels.addAll(accountSpaceModel.getSegments());
                    }

                    String resourceTypeId = resourcesTypes.get(resourceComplexKey.getResourceTypeKey());
                    if (resourceTypeId == null) {
                        return Optional.empty();
                    }

                    resourceComplexKeyMap.put(Tuples.of(resourceTypeId, Optional.ofNullable(accountSpaceId),
                            segmentSettingsModels), resourceComplexKey);

                    return Optional.of(new ResourceIdentity(
                            resourceTypeId,
                            Tenants.DEFAULT_TENANT_ID,
                            accountSpaceId,
                            segmentSettingsModels,
                            providerId
                    ));
                })
                .filter(Optional::isPresent)
                .map(Optional::get)
                .collect(Collectors.toList());

        return resourcesDao.getAllByIdentities(session, resourceIdentities, false)
                .map(resourceModels -> resourceModels.stream()
                        .filter(resourceModel -> resourceComplexKeyMap.containsKey(Tuples.of(
                                resourceModel.getResourceTypeId(),
                                Optional.ofNullable(resourceModel.getAccountsSpacesId()),
                                resourceModel.getSegments())))
                        .collect(Collectors.toMap(resourceModel -> resourceComplexKeyMap.get(Tuples.of(
                                resourceModel.getResourceTypeId(),
                                Optional.ofNullable(resourceModel.getAccountsSpacesId()), resourceModel.getSegments())),
                                Function.identity(), (r1, r2) -> r1)));
    }

    public Mono<Map<String, ResourceModel>> loadResources(YdbTxSession session, Set<String> resourceIds) {
        if (resourceIds.isEmpty()) {
            return Mono.just(Map.of());
        }
        List<Tuple2<String, TenantId>> ids = resourceIds.stream().map(id -> Tuples.of(id, Tenants.DEFAULT_TENANT_ID))
                .collect(Collectors.toList());
        return Flux.fromIterable(Lists.partition(ids, 500)).concatMap(v -> resourcesDao.getByIds(session, v))
                .collectList().map(l -> l.stream().flatMap(Collection::stream)
                        .collect(Collectors.toMap(ResourceModel::getId, Function.identity())));
    }

    public Mono<Map<AccountExternalId, AccountModel>> loadAccounts(
            YdbTxSession session,
            Map<AccountSpaceIdentity, AccountSpaceModel> accountSpaces,
            Set<AccountExternalId> accountIds
    ) {
        if (accountIds.isEmpty()) {
            return Mono.just(Map.of());
        }
        Map<AccountModel.ExternalId, AccountExternalId> modelIdsToTransferIds = accountIds.stream()
                .map(accountExternalId -> {
                    AccountSpaceModel accountSpace = accountSpaces.get(accountExternalId.getAccountSpaceIdentity());
                    return Tuples.of(accountExternalId, new AccountModel.ExternalId(
                            accountExternalId.getProviderId(),
                            accountExternalId.getOuterAccountIdInProvider(),
                            accountSpace == null ? null : accountSpace.getId()
                    ));
                })
                .collect(Collectors.toMap(Tuple2::getT2, Tuple2::getT1));
        List<WithTenant<AccountModel.ExternalId>> ids = modelIdsToTransferIds.keySet().stream().map(id ->
                new WithTenant<>(Tenants.DEFAULT_TENANT_ID, id)).collect(Collectors.toList());
        return Flux.fromIterable(Lists.partition(ids, 500))
                .concatMap(v -> accountsDao.getAllByExternalIds(session, v))
                .collectList()
                .map(l -> l.stream().flatMap(Collection::stream)
                        .collect(Collectors.toMap(
                                account -> modelIdsToTransferIds.get(account.toExternalId()),
                                Function.identity())
                        )
                );
    }

    public Mono<Map<String, FolderModel>> loadFolders(YdbTxSession session, Set<String> folderIds) {
        if (folderIds.isEmpty()) {
            return Mono.just(Map.of());
        }
        List<WithTenant<String>> ids = folderIds.stream().map(id ->
                new WithTenant<>(Tenants.DEFAULT_TENANT_ID, id)).collect(Collectors.toList());
        return Flux.fromIterable(Lists.partition(ids, 500)).concatMap(v -> folderDao.getByIds(session, v))
                .collectList().map(l -> l.stream().flatMap(Collection::stream)
                        .collect(Collectors.toMap(FolderModel::getId, Function.identity())));
    }

    public Mono<Map<Long, ServiceWithStatesModel>> loadServices(YdbTxSession session, Set<Long> serviceIds) {
        if (serviceIds.isEmpty()) {
            return Mono.just(Map.of());
        }
        List<Long> ids = new ArrayList<>(serviceIds);
        return Flux.fromIterable(Lists.partition(ids, 500))
                .concatMap(v -> servicesDao.getServiceStatesByIds(session, v))
                .collectList().map(l -> l.stream().flatMap(Collection::stream)
                        .collect(Collectors.toMap(ServiceWithStatesModel::getId, Function.identity())));
    }

    public Mono<Map<String, UnitsEnsembleModel>> loadUnitsEnsembles(YdbTxSession session,
                                                                    Collection<ResourceModel> resources) {
        if (resources.isEmpty()) {
            return Mono.just(Map.of());
        }
        List<Tuple2<String, TenantId>> ids = resources.stream().map(r -> Tuples.of(r.getUnitsEnsembleId(),
                Tenants.DEFAULT_TENANT_ID)).collect(Collectors.toList());
        return Flux.fromIterable(Lists.partition(ids, 500))
                .concatMap(v -> unitsEnsemblesDao.getByIds(session, v))
                .collectList().map(l -> l.stream().flatMap(Collection::stream)
                        .collect(Collectors.toMap(UnitsEnsembleModel::getId, Function.identity())));
    }

    public Mono<Map<Long, FolderModel>> loadDefaultFolders(YdbTxSession session,
                                                           Collection<ServiceWithStatesModel> services) {
        if (services.isEmpty()) {
            return Mono.just(Map.of());
        }
        List<WithTenant<Long>> ids = services.stream().map(s -> new WithTenant<>(Tenants.DEFAULT_TENANT_ID,
                s.getId())).collect(Collectors.toList());
        return Flux.fromIterable(Lists.partition(ids, 500))
                .concatMap(v -> folderDao.getAllDefaultFoldersByServiceIds(session, v))
                .collectList().map(l -> l.stream().flatMap(Collection::stream)
                        .collect(Collectors.toMap(FolderModel::getServiceId, Function.identity())));
    }

    public Mono<ExistingAccountsCounts> loadAccountsCounts(ValidatedImport quotasToImport, YdbTxSession session) {
        List<WithTenant<AccountsDao.FolderIdProviderId>> ids = new ArrayList<>();
        quotasToImport.getQuotas().forEach(importFolder -> {
            String folderId = null;
            if (importFolder.getFolderId().isPresent()) {
                folderId = importFolder.getFolderId().get();
            } else if (importFolder.getServiceId().isPresent()) {
                FolderModel defaultFolder = quotasToImport.getDirectories().getDefaultFolders()
                        .get(importFolder.getServiceId().get());
                if (defaultFolder != null) {
                    folderId = defaultFolder.getId();
                }
            }
            if (folderId != null) {
                for (ImportAccount importAccount : importFolder.getAccounts()) {
                    ids.add(new WithTenant<>(Tenants.DEFAULT_TENANT_ID,
                            new AccountsDao.FolderIdProviderId(folderId, importAccount.getProviderId())));
                }
            }
        });
        if (ids.isEmpty()) {
            return Mono.just(new ExistingAccountsCounts(Map.of()));
        }
        return Flux.fromIterable(Lists.partition(ids, 500))
                .concatMap(v -> accountsDao.countByFolderIdProviderId(session, v))
                .collectList().map(l -> {
                    Map<String, Map<String, Long>> counts = new HashMap<>();
                    l.forEach(m -> m.forEach((k, v) -> counts.computeIfAbsent(k.getFolderId(),
                            x -> new HashMap<>()).put(k.getProviderId(), v)));
                    return new ExistingAccountsCounts(counts);
                });
    }

    public Mono<ValidatedImportWithQuotas> loadImportedQuotas(YdbTxSession session,
                                                              ValidatedImport validatedImport) {
        Set<QuotaModel.Key> quotaKeys = new HashSet<>();
        Set<AccountsQuotasDao.FolderProviderAccountsSpaceResource> provisionKeys = new HashSet<>();
        validatedImport.getQuotas().forEach(importFolder -> {
            Optional<String> folderIdO = Optional.empty();
            if (importFolder.getFolderId().isPresent()) {
                folderIdO = importFolder.getFolderId();
            } else if (importFolder.getServiceId().isPresent()) {
                FolderModel defaultFolder = validatedImport.getDirectories().getDefaultFolders()
                        .get(importFolder.getServiceId().get());
                if (defaultFolder != null) {
                    folderIdO = Optional.of(defaultFolder.getId());
                }
            }
            folderIdO.ifPresent(folderId -> {
                importFolder.getResourceQuotas().forEach(resource -> {
                        quotaKeys.add(new QuotaModel.Key(folderId,
                                resource.getProviderId(), resource.getResourceId()));
                        provisionKeys.add(new AccountsQuotasDao.FolderProviderAccountsSpaceResource(folderId,
                                resource.getProviderId(), resource.getAccountsSpaceId().orElse(null),
                                resource.getResourceId()));
                });
                importFolder.getAccounts().forEach(account -> account.getProvisions()
                        .forEach(provision -> {
                            quotaKeys.add(new QuotaModel.Key(folderId,
                                account.getProviderId(), provision.getResourceId()));
                            provisionKeys.add(new AccountsQuotasDao.FolderProviderAccountsSpaceResource(folderId,
                                    account.getProviderId(), provision.getAccountsSpaceId().orElse(null),
                                    provision.getResourceId()));
                        }));
            });
        });
        return loadQuotas(session, quotaKeys).flatMap(quotas -> loadProvidedQuotas(session, provisionKeys)
                .flatMap(provisions -> {
                    Set<String> accountIdsByProvidedQuotas = provisions.values().stream()
                            .map(AccountsQuotasModel::getAccountId).collect(Collectors.toSet());
                    Set<String> loadedAccountIds = validatedImport.getDirectories().getAccounts().values()
                            .stream().map(AccountModel::getId).collect(Collectors.toSet());
                    Set<String> accountIdsToLoad = Sets.difference(accountIdsByProvidedQuotas, loadedAccountIds);
                    return loadAccountsByIds(session, accountIdsToLoad).map(loadedAccounts -> {
                        Map<String, AccountModel> accounts = new HashMap<>(loadedAccounts);
                        validatedImport.getDirectories().getAccounts().values()
                                .forEach(account -> accounts.put(account.getId(), account));
                        return new ValidatedImportWithQuotas(validatedImport, quotas, provisions, accounts);
                    });
                }));
    }

    public Mono<Map<QuotaModel.Key, QuotaModel>> loadQuotas(YdbTxSession session, Set<QuotaModel.Key> quotaKeys) {
        if (quotaKeys.isEmpty()) {
            return Mono.just(Map.of());
        }
        List<WithTenant<QuotaModel.Key>> ids = quotaKeys.stream()
                .map(key -> new WithTenant<>(Tenants.DEFAULT_TENANT_ID, key)).collect(Collectors.toList());
        return Flux.fromIterable(Lists.partition(ids, 500))
                .concatMap(v -> quotasDao.getByKeys(session, v))
                .collectList().map(l -> l.stream().flatMap(Collection::stream)
                        .collect(Collectors.toMap(QuotaModel::toKey, Function.identity())));
    }

    public Mono<Map<AccountsQuotasModel.Identity, AccountsQuotasModel>> loadProvidedQuotas(
            YdbTxSession session, Set<AccountsQuotasDao.FolderProviderAccountsSpaceResource> keys) {
        if (keys.isEmpty()) {
            return Mono.just(Map.of());
        }
        List<WithTenant<AccountsQuotasDao.FolderProviderAccountsSpaceResource>> ids = keys.stream()
                .map(key -> new WithTenant<>(Tenants.DEFAULT_TENANT_ID, key)).collect(Collectors.toList());
        return accountsQuotasDao.getByTenantFolderProviderResource(session, ids).map(l -> l.stream()
                .collect(Collectors.toMap(v -> new AccountsQuotasModel.Identity(v.getAccountId(), v.getResourceId()),
                        Function.identity())));
    }

    public Mono<Map<String, AccountModel>> loadAccountsByIds(YdbTxSession session, Set<String> accountIds) {
        if (accountIds.isEmpty()) {
            return Mono.just(Map.of());
        }
        List<WithTenant<String>> ids = accountIds.stream().map(id -> new WithTenant<>(Tenants.DEFAULT_TENANT_ID, id))
                .collect(Collectors.toList());
        return Flux.fromIterable(Lists.partition(ids, 500))
                .concatMap(v -> accountsDao.getByIds(session, v))
                .collectList().map(l -> l.stream().flatMap(Collection::stream)
                        .collect(Collectors.toMap(AccountModel::getId, Function.identity())));
    }

    public Mono<AccountsMoveInfo> loadAccountsMoveInfo(YdbTxSession session, ValidatedImportWithQuotas importData) {
        Map<AccountModel.ExternalId, MovingAccount> movingAccounts = findMovingAccounts(importData);
        if (movingAccounts.isEmpty()) {
            return Mono.just(new AccountsMoveInfo(new HashMap<>(), new HashMap<>()));
        }
        Set<String> sourceFolderIds = movingAccounts.values().stream().map(MovingAccount::getSourceFolderId)
                .collect(Collectors.toSet());
        return loadSourceFolders(session, sourceFolderIds, importData)
                .flatMap(sourceFolders -> loadResourceIdsByAccountIds(session, movingAccounts.keySet(), importData)
                        .flatMap(accountsResources -> loadResources(session, accountsResources.values(), importData)
                                .flatMap(resources -> loadProvidedQuotas(session, movingAccounts, accountsResources,
                                        resources)
                                        .flatMap(provisions -> loadQuotas(session, movingAccounts, accountsResources,
                                                resources)
                                                .map(quotas ->
                                                        prepareAccountsMoveInfo(movingAccounts, sourceFolders,
                                                                accountsResources, resources,
                                                                provisions, quotas, importData))))));
    }

    private Map<AccountModel.ExternalId, MovingAccount> findMovingAccounts(ValidatedImportWithQuotas importData) {
        Map<AccountModel.ExternalId, MovingAccount> movingAccounts = new HashMap<>();
        importData.getValidatedImport().getQuotas().forEach(importFolder -> {
            String destinationFolderId = null;
            String destinationFolderToCreateId = null;
            if (importFolder.getFolderId().isPresent()) {
                destinationFolderId = importFolder.getFolderId().get();
            } else if (importFolder.getServiceId().isPresent()) {
                destinationFolderId = Optional.ofNullable(importData.getValidatedImport().getDirectories()
                        .getDefaultFolders().get(importFolder.getServiceId().get()))
                        .map(FolderModel::getId).orElse(null);
                if (destinationFolderId == null) {
                    destinationFolderToCreateId = importData.getValidatedImport()
                            .getPreGeneratedFolderIds().get(importFolder.getServiceId().get());
                }
            }
            for (ImportAccount importAccount : importFolder.getAccounts()) {
                String accountSpaceId = getAccountSpaceId(
                        importData.getValidatedImport().getDirectories().getAccountSpaces(),
                        importAccount.getAccountSpaceId()
                );
                AccountModel.ExternalId externalId = new AccountModel.ExternalId(importAccount.getProviderId(),
                        importAccount.getId(), accountSpaceId);
                AccountModel existingAccount = importData.getValidatedImport().getDirectories()
                        .getAccountsByExternalId().get(externalId);
                if (existingAccount != null) {
                    String sourceFolderId = existingAccount.getFolderId();
                    if ((sourceFolderId != null && destinationFolderId == null) || (sourceFolderId != null
                            && destinationFolderId != null && !sourceFolderId.equals(destinationFolderId))) {
                        movingAccounts.put(externalId, new MovingAccount(sourceFolderId, destinationFolderId,
                                destinationFolderToCreateId, existingAccount.getId()));
                    }
                }
            }
        });
        return movingAccounts;
    }

    @Nullable
    private String getAccountSpaceId(
            Map<AccountSpaceIdentity, AccountSpaceModel> accountSpaces,
            AccountSpaceIdentity accountSpaceIdentity
    ) {
        if (accountSpaceIdentity.isEmpty()) {
            return null;
        }
        AccountSpaceModel accountSpaceModel = accountSpaces.get(accountSpaceIdentity);
        if (accountSpaceModel == null) {
            return null;
        }
        return accountSpaceModel.getId();
    }

    private AccountsDeleteInfo prepareAccountsDeleteInfo(
            Set<AccountModel.ExternalId> deletingAccounts,
            Map<AccountModel.ExternalId, List<AccountsQuotasModel>> provisions) {
        Map<AccountModel.ExternalId, AccountDeleteInfo> deletedAccounts = new HashMap<>();
        deletingAccounts.forEach(k -> {
            Map<AccountsQuotasModel.Identity, AccountsQuotasModel> provisionsMap = provisions
                    .getOrDefault(k, Collections.emptyList()).stream()
                    .collect(Collectors.toMap(AccountsQuotasModel::getIdentity, Function.identity()));
            AccountDeleteInfo deleteInfo = new AccountDeleteInfo(provisionsMap);
            deletedAccounts.put(k, deleteInfo);
        });
        return new AccountsDeleteInfo(deletedAccounts);
    }

    @SuppressWarnings("ParameterNumber")
    private AccountsMoveInfo prepareAccountsMoveInfo(
            Map<AccountModel.ExternalId, MovingAccount> movingAccounts,
            Map<String, FolderModel> sourceFolders,
            Map<AccountModel.ExternalId, Set<String>> accountResources,
            Map<String, ResourceModel> resources,
            Map<AccountModel.ExternalId, List<AccountsQuotasModel>> provisions,
            Map<AccountModel.ExternalId, List<QuotaModel>> quotas,
            ValidatedImportWithQuotas importData) {
        Map<AccountModel.ExternalId, AccountMoveInfo> movedAccounts = new HashMap<>();
        Map<String, FolderModel> foldersById = new HashMap<>();
        movingAccounts.forEach((k, v) -> {
            FolderModel sourceFolder = sourceFolders.get(v.getSourceFolderId());
            FolderModel destinationFolder = v.getDestinationFolderId().map(id -> importData.getValidatedImport()
                    .getDirectories().getFolders().get(id)).orElse(null);
            Set<ResourceModel> currentAccountResources = accountResources.getOrDefault(k, Collections.emptySet())
                    .stream().map(resources::get).collect(Collectors.toSet());
            Map<AccountsQuotasModel.Identity, AccountsQuotasModel> sourceProvisions = provisions
                    .getOrDefault(k, Collections.emptyList()).stream()
                    .filter(p -> p.getFolderId().equals(sourceFolder.getId()))
                    .collect(Collectors.toMap(AccountsQuotasModel::getIdentity, Function.identity()));
            Map<AccountsQuotasModel.Identity, AccountsQuotasModel> destinationProvisions = destinationFolder != null
                    ? provisions.getOrDefault(k, Collections.emptyList()).stream()
                    .filter(p -> p.getFolderId().equals(destinationFolder.getId()))
                    .collect(Collectors.toMap(AccountsQuotasModel::getIdentity, Function.identity()))
                    : Map.of();
            Map<QuotaModel.Key, QuotaModel> sourceQuotas = quotas.getOrDefault(k, Collections.emptyList()).stream()
                    .filter(q -> q.getFolderId().equals(sourceFolder.getId()))
                    .collect(Collectors.toMap(QuotaModel::toKey, Function.identity()));
            Map<QuotaModel.Key, QuotaModel> destinationQuotas = destinationFolder != null
                    ? quotas.getOrDefault(k, Collections.emptyList()).stream()
                    .filter(q -> q.getFolderId().equals(destinationFolder.getId()))
                    .collect(Collectors.toMap(QuotaModel::toKey, Function.identity()))
                    : Map.of();
            AccountMoveInfo moveInfo = new AccountMoveInfo(sourceFolder, destinationFolder,
                    v.getDestinationFolderToCreateId().orElse(null), currentAccountResources,
                    sourceProvisions, destinationProvisions, sourceQuotas, destinationQuotas, v.getAccountId());
            movedAccounts.put(k, moveInfo);
            if (!foldersById.containsKey(sourceFolder.getId())) {
                foldersById.put(sourceFolder.getId(), sourceFolder);
            }
            if (destinationFolder != null && !foldersById.containsKey(destinationFolder.getId())) {
                foldersById.put(destinationFolder.getId(), destinationFolder);
            }
        });
        return new AccountsMoveInfo(movedAccounts, foldersById);
    }

    private Set<AccountModel.ExternalId> findDeletingAccounts(ValidatedImportWithQuotas importData) {
        Set<AccountModel.ExternalId> deletingAccounts = new HashSet<>();
        importData.getValidatedImport().getQuotas().forEach(importFolder -> {
            for (ImportAccount importAccount : importFolder.getAccounts()) {
                String accountSpaceId = getAccountSpaceId(
                        importData.getValidatedImport().getDirectories().getAccountSpaces(),
                        importAccount.getAccountSpaceId()
                );
                AccountModel.ExternalId externalId = new AccountModel.ExternalId(importAccount.getProviderId(),
                        importAccount.getId(), accountSpaceId);
                AccountModel existingAccount = importData.getValidatedImport().getDirectories()
                        .getAccountsByExternalId().get(externalId);
                if (existingAccount != null) {
                    deletingAccounts.add(externalId);
                }
            }
        });
        return deletingAccounts;
    }

    public Mono<Map<String, FolderModel>> loadSourceFolders(YdbTxSession session,
                                                            Set<String> sourceFolderIds,
                                                            ValidatedImportWithQuotas importData) {
        Map<String, FolderModel> knownSourceFolders = new HashMap<>();
        Set<String> folderIdsToLoad = new HashSet<>();
        Map<String, FolderModel> foldersById = importData.getValidatedImport().getDirectories().getFolders();
        Map<String, FolderModel> defaultFoldersById = new HashMap<>();
        importData.getValidatedImport().getDirectories().getDefaultFolders().values()
                .forEach(m -> defaultFoldersById.put(m.getId(), m));
        sourceFolderIds.forEach(id -> {
            FolderModel folderById = foldersById.get(id);
            if (folderById != null) {
                knownSourceFolders.put(id, folderById);
                return;
            }
            FolderModel defaultFolderById = defaultFoldersById.get(id);
            if (defaultFolderById != null) {
                knownSourceFolders.put(id, defaultFolderById);
                return;
            }
            folderIdsToLoad.add(id);
        });
        if (folderIdsToLoad.isEmpty()) {
            return Mono.just(knownSourceFolders);
        }
        return loadFolders(session, folderIdsToLoad).map(loadedFolders -> {
            if (knownSourceFolders.isEmpty()) {
                return loadedFolders;
            }
            Map<String, FolderModel> result = new HashMap<>(loadedFolders);
            result.putAll(knownSourceFolders);
            return result;
        });
    }

    public Mono<Map<AccountModel.ExternalId, Set<String>>> loadResourceIdsByAccountIds(
            YdbTxSession session,
            Set<AccountModel.ExternalId> movingAccounts,
            ValidatedImportWithQuotas importData) {
        Set<String> accountIds = new HashSet<>();
        Map<String, AccountModel.ExternalId> externalIdByInternalId = new HashMap<>();
        movingAccounts.forEach(externalId -> {
            AccountModel account = importData.getValidatedImport().getDirectories().getAccountsByExternalId()
                    .get(externalId);
            if (account != null) {
                accountIds.add(account.getId());
                externalIdByInternalId.put(account.getId(), externalId);
            }
        });
        List<WithTenant<String>> ids = accountIds.stream().map(id -> new WithTenant<>(Tenants.DEFAULT_TENANT_ID, id))
                .collect(Collectors.toList());
        if (ids.isEmpty()) {
            return Mono.just(Map.of());
        }
        return accountsQuotasDao.getResourceIdsByAccountIds(session, ids).map(resourceIds -> {
            Map<String, Set<String>> byAccountId = resourceIds.stream().map(WithTenant::getIdentity)
                    .collect(Collectors.groupingBy(AccountsQuotasDao.AccountResource::getAccountId,
                            Collectors.mapping(AccountsQuotasDao.AccountResource::getResourceId, Collectors.toSet())));
            Map<AccountModel.ExternalId, Set<String>> result = new HashMap<>();
            byAccountId.forEach((k, v) -> result.put(externalIdByInternalId.get(k), v));
            return result;
        });
    }

    public Mono<Map<String, ResourceModel>> loadResources(YdbTxSession session,
                                                          Collection<Set<String>> resourceIds,
                                                          ValidatedImportWithQuotas importData) {
        Map<String, ResourceModel> knownResources = new HashMap<>();
        Set<String> resourceIdsToLoad = new HashSet<>();
        Set<String> resourceIdsSet = resourceIds.stream().flatMap(Collection::stream)
                .collect(Collectors.toSet());
        resourceIdsSet.forEach(id -> {
            ResourceModel resource = importData.getValidatedImport().getDirectories().getResources().get(id);
            if (resource != null) {
                knownResources.put(id, resource);
            } else {
                resourceIdsToLoad.add(id);
            }
        });
        if (resourceIdsToLoad.isEmpty()) {
            return Mono.just(knownResources);
        }
        return loadResources(session, resourceIdsToLoad).map(loadedResources -> {
            if (knownResources.isEmpty()) {
                return loadedResources;
            }
            Map<String, ResourceModel> result = new HashMap<>(loadedResources);
            result.putAll(knownResources);
            return result;
        });
    }

    public Mono<Map<AccountModel.ExternalId, List<AccountsQuotasModel>>> loadProvidedQuotas(
            YdbTxSession session,
            Map<AccountModel.ExternalId, MovingAccount> accounts,
            Map<AccountModel.ExternalId, Set<String>> resourcesByAccount,
            Map<String, ResourceModel> resources) {
        List<WithTenant<AccountsQuotasDao.FolderProviderAccountsSpaceResource>> params = new ArrayList<>();
        Map<FolderProviderResource, Set<AccountModel.ExternalId>> accountsReverseLookup
                = new HashMap<>();
        accounts.forEach((k, v) -> resourcesByAccount.getOrDefault(k, Collections.emptySet()).forEach(resourceId -> {
            ResourceModel resource = resources.get(resourceId);
            String providerId = resource.getProviderId();
            String accountsSpaceId = resource.getAccountsSpacesId();
            AccountsQuotasDao.FolderProviderAccountsSpaceResource sourceId =
                    new AccountsQuotasDao.FolderProviderAccountsSpaceResource(v.getSourceFolderId(), providerId,
                            accountsSpaceId, resourceId);
            FolderProviderResource sourceKey =
                    new FolderProviderResource(v.getSourceFolderId(), providerId, resourceId);
            params.add(new WithTenant<>(Tenants.DEFAULT_TENANT_ID, sourceId));
            accountsReverseLookup.computeIfAbsent(sourceKey, x -> new HashSet<>()).add(k);
            if (v.getDestinationFolderId().isPresent()) {
                AccountsQuotasDao.FolderProviderAccountsSpaceResource destinationId
                        = new AccountsQuotasDao.FolderProviderAccountsSpaceResource(v.getDestinationFolderId().get(),
                        providerId, accountsSpaceId, resourceId);
                FolderProviderResource destinationKey
                        = new FolderProviderResource(v.getDestinationFolderId().get(),
                        providerId, resourceId);
                params.add(new WithTenant<>(Tenants.DEFAULT_TENANT_ID, destinationId));
                accountsReverseLookup.computeIfAbsent(destinationKey, x -> new HashSet<>()).add(k);
            }
        }));
        if (params.isEmpty()) {
            return Mono.just(Map.of());
        }
        return accountsQuotasDao.getByTenantFolderProviderResource(session, params).map(l -> {
            Map<AccountModel.ExternalId, List<AccountsQuotasModel>> result = new HashMap<>();
            l.forEach(quota -> {
                FolderProviderResource lookupKey = new FolderProviderResource(quota.getFolderId(),
                        quota.getProviderId(), quota.getResourceId());
                accountsReverseLookup.getOrDefault(lookupKey, Collections.emptySet()).forEach(externalId ->
                        result.computeIfAbsent(externalId, x -> new ArrayList<>()).add(quota));
            });
            return result;
        });
    }

    public Mono<Map<AccountModel.ExternalId, List<QuotaModel>>> loadQuotas(
            YdbTxSession session,
            Map<AccountModel.ExternalId, MovingAccount> accounts,
            Map<AccountModel.ExternalId, Set<String>> resourcesByAccount,
            Map<String, ResourceModel> resources) {
        Set<QuotaModel.Key> keysToLoad = new HashSet<>();
        Map<QuotaModel.Key, Set<AccountModel.ExternalId>> accountsReverseLookup = new HashMap<>();
        accounts.forEach((k, v) -> resourcesByAccount.getOrDefault(k, Collections.emptySet()).forEach(resourceId -> {
            String providerId = resources.get(resourceId).getProviderId();
            QuotaModel.Key sourceKey = new QuotaModel.Key(v.getSourceFolderId(), providerId, resourceId);
            keysToLoad.add(sourceKey);
            accountsReverseLookup.computeIfAbsent(sourceKey, x -> new HashSet<>()).add(k);
            if (v.getDestinationFolderId().isPresent()) {
                QuotaModel.Key destinationId = new QuotaModel.Key(v.getDestinationFolderId().get(), providerId,
                        resourceId);
                keysToLoad.add(destinationId);
                accountsReverseLookup.computeIfAbsent(destinationId, x -> new HashSet<>()).add(k);
            }
        }));
        return loadQuotas(session, keysToLoad).map(l -> {
            Map<AccountModel.ExternalId, List<QuotaModel>> result = new HashMap<>();
            l.values().forEach(quota -> {
                QuotaModel.Key lookupKey = new QuotaModel.Key(quota.getFolderId(), quota.getProviderId(),
                        quota.getResourceId());
                accountsReverseLookup.getOrDefault(lookupKey, Collections.emptySet()).forEach(externalId ->
                        result.computeIfAbsent(externalId, x -> new ArrayList<>()).add(quota));
            });
            return result;
        });
    }

    public Mono<AccountsDeleteInfo> loadAccountDeleteInfo(YdbTxSession session,
                                                          ValidatedImportWithQuotas importData) {
        Set<AccountModel.ExternalId> deletingAccounts = findDeletingAccounts(importData);
        if (deletingAccounts.isEmpty()) {
            return Mono.just(new AccountsDeleteInfo(new HashMap<>()));
        }
        return loadResourceIdsByAccountIds(session, deletingAccounts, importData)
                .flatMap(accountsResources -> loadProvidedQuotasForAccount(session, deletingAccounts,
                        accountsResources, importData)
                        .map(provisions -> prepareAccountsDeleteInfo(deletingAccounts, provisions)));
    }

    private Mono<Map<AccountModel.ExternalId, List<AccountsQuotasModel>>> loadProvidedQuotasForAccount(
            YdbTxSession session,
            Set<AccountModel.ExternalId> accounts,
            Map<AccountModel.ExternalId, Set<String>> resourcesByAccount,
            ValidatedImportWithQuotas importData) {
        List<WithTenant<AccountsQuotasModel.Identity>> ids = new ArrayList<>();
        Map<AccountsQuotasModel.Identity, Set<AccountModel.ExternalId>> accountsReverseLookup = new HashMap<>();
        accounts.forEach(k -> {
            AccountModel account = importData.getValidatedImport().getDirectories().getAccountsByExternalId().get(k);
            if (account == null) {
                return;
            }
            resourcesByAccount.getOrDefault(k, Collections.emptySet()).forEach(resourceId -> {
                AccountsQuotasModel.Identity provisionId =
                        new AccountsQuotasModel.Identity(account.getId(), resourceId);
                ids.add(new WithTenant<>(Tenants.DEFAULT_TENANT_ID, provisionId));
                accountsReverseLookup.computeIfAbsent(provisionId, x -> new HashSet<>()).add(k);
            });
        });
        if (ids.isEmpty()) {
            return Mono.just(Map.of());
        }
        return Flux.fromIterable(Lists.partition(ids, 500))
                .concatMap(v -> accountsQuotasDao.getByIds(session, v))
                .collectList().map(l -> {
                    Map<AccountModel.ExternalId, List<AccountsQuotasModel>> result = new HashMap<>();
                    l.stream().flatMap(Collection::stream).forEach(quota -> {
                        AccountsQuotasModel.Identity lookupId = quota.getIdentity();
                        accountsReverseLookup.getOrDefault(lookupId, Collections.emptySet()).forEach(externalId ->
                                result.computeIfAbsent(externalId, x -> new ArrayList<>()).add(quota));
                    });
                    return result;
                });
    }
}
