package ru.yandex.intranet.d.services.imports;

import java.time.Instant;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.yandex.ydb.table.transaction.TransactionMode;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuple2;
import reactor.util.function.Tuples;

import ru.yandex.intranet.d.dao.Tenants;
import ru.yandex.intranet.d.dao.accounts.AccountsDao;
import ru.yandex.intranet.d.dao.accounts.AccountsQuotasDao;
import ru.yandex.intranet.d.dao.folders.FolderDao;
import ru.yandex.intranet.d.dao.folders.FolderOperationLogDao;
import ru.yandex.intranet.d.dao.quotas.QuotasDao;
import ru.yandex.intranet.d.datasource.model.YdbTableClient;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.model.accounts.AccountModel;
import ru.yandex.intranet.d.model.accounts.AccountSpaceModel;
import ru.yandex.intranet.d.model.accounts.AccountsQuotasModel;
import ru.yandex.intranet.d.model.folders.AccountHistoryModel;
import ru.yandex.intranet.d.model.folders.AccountsHistoryModel;
import ru.yandex.intranet.d.model.folders.FolderHistoryFieldsModel;
import ru.yandex.intranet.d.model.folders.FolderModel;
import ru.yandex.intranet.d.model.folders.FolderOperationLogModel;
import ru.yandex.intranet.d.model.folders.FolderOperationType;
import ru.yandex.intranet.d.model.folders.FolderType;
import ru.yandex.intranet.d.model.folders.ProvisionHistoryModel;
import ru.yandex.intranet.d.model.folders.ProvisionsByResource;
import ru.yandex.intranet.d.model.folders.QuotasByAccount;
import ru.yandex.intranet.d.model.folders.QuotasByResource;
import ru.yandex.intranet.d.model.quotas.QuotaModel;
import ru.yandex.intranet.d.util.result.ErrorCollection;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.util.result.TypedError;
import ru.yandex.intranet.d.web.model.imports.ImportDto;
import ru.yandex.intranet.d.web.model.imports.ImportFailureDto;
import ru.yandex.intranet.d.web.model.imports.ImportResultDto;
import ru.yandex.intranet.d.web.model.imports.ImportSuccessKeyDto;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;

/**
 * Quotas imports service implementation.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class QuotasImportsService {

    private final FolderDao folderDao;
    private final AccountsDao accountsDao;
    private final QuotasDao quotasDao;
    private final AccountsQuotasDao accountsQuotasDao;
    private final FolderOperationLogDao folderOperationLogDao;
    private final YdbTableClient tableClient;
    private final QuotasImportsValidationService validationService;

    @SuppressWarnings("ParameterNumber")
    public QuotasImportsService(FolderDao folderDao,
                                AccountsDao accountsDao,
                                QuotasDao quotasDao,
                                AccountsQuotasDao accountsQuotasDao,
                                FolderOperationLogDao folderOperationLogDao,
                                YdbTableClient tableClient,
                                QuotasImportsValidationService validationService) {
        this.folderDao = folderDao;
        this.accountsDao = accountsDao;
        this.quotasDao = quotasDao;
        this.accountsQuotasDao = accountsQuotasDao;
        this.folderOperationLogDao = folderOperationLogDao;
        this.tableClient = tableClient;
        this.validationService = validationService;
    }

    public Mono<Result<ImportResultDto>> importQuotas(ImportDto quotasToImport, YaUserDetails currentUser,
                                                      Locale locale) {
        return validationService.checkWritePermissions(currentUser, locale)
                .andThen(v -> validationService.preValidateGlobal(quotasToImport, locale))
                .apply(v -> validationService.preValidateLocal(quotasToImport, v, locale))
                .andThenMono(v -> doValidateAndImport(v, currentUser, locale));

    }

    private Mono<Result<ImportResultDto>> doValidateAndImport(PreValidatedImport quotasToImport,
                                                              YaUserDetails currentUser,
                                                              Locale locale) {
        return tableClient.usingSessionMonoRetryable(session -> session
                .usingTxMonoRetryable(TransactionMode.SERIALIZABLE_READ_WRITE,
                        ts -> validationService.doValidate(quotasToImport, ts, locale).flatMap(v -> v
                                .andThenMono(i -> validationService.checkWritePermissions(i, currentUser, locale))
                                .flatMap(result -> result.andThenMono(u -> validationService.doValidateQuotas(u, ts,
                                        locale)
                                        .flatMap(r -> r.andThenMono(t -> doImport(t, ts, currentUser))))))));
    }

    private Mono<Result<ImportResultDto>> doImport(PreparedImportData quotasToImport,
                                                   YdbTxSession session,
                                                   YaUserDetails currentUser) {
        if (quotasToImport.getImportedQuotas().isEmpty() && !quotasToImport.getImportFailures().isEmpty()) {
            return Mono.just(Result.failure(toErrorCollection(quotasToImport.getImportFailures(),
                    quotasToImport.getFolderIndices(), quotasToImport.getServiceIndices())));
        }
        Instant now = Instant.now();
        List<QuotaImportApplication> quotasToChange = quotasToImport.getImportApplication().getQuotas().values()
                .stream().filter(QuotaImportApplication::hasChanges).collect(Collectors.toList());
        List<QuotaModel> quotasToUpsert = quotasToChange.stream()
                .map(this::toUpdatedQuota).collect(Collectors.toList());
        Map<ProvisionImportKey, ProvisionImportApplication> provisionsToChange = quotasToImport.getImportApplication()
                .getProvisions().entrySet().stream().filter(e -> e.getValue().hasChanges()).collect(Collectors
                        .toMap(Map.Entry::getKey, Map.Entry::getValue));
        Map<AccountsQuotasModel.Identity, Set<Tuple2<ProvisionImportKey,
                ProvisionImportApplication>>> groupedProvisionsToChange = provisionsToChange.entrySet().stream()
                .map(e -> Tuples.of(e.getKey(), e.getValue())).collect(Collectors.groupingBy(e ->
                                new AccountsQuotasModel.Identity(e.getT2().getAccountId(), e.getT2().getResourceId()),
                        Collectors.toSet()));
        List<AccountsQuotasModel> provisionsToUpsert = findProvisionsToUpsert(groupedProvisionsToChange, now);
        Map<String, Map<String, AccountHistoryModel>> oldAccounts = new HashMap<>();
        Map<String, Map<String, AccountHistoryModel>> newAccounts = new HashMap<>();
        List<AccountModel> accountsToUpsert = prepareAccountsToUpsert(quotasToImport, now, oldAccounts, newAccounts);
        List<FolderModel> foldersToUpsert = prepareFoldersToUpsert(quotasToImport);
        Map<String, FolderModel> foldersToIncrementNextOpLogId = new HashMap<>();
        List<FolderOperationLogModel> historyToUpsert = prepareLog(quotasToImport, quotasToChange, provisionsToChange,
                currentUser, foldersToUpsert, foldersToIncrementNextOpLogId, now, oldAccounts, newAccounts);
        addNextOpLogIdIncrements(foldersToUpsert, foldersToIncrementNextOpLogId);
        List<ImportSuccessKeyDto> successfullyImported = quotasToImport.getImportedQuotas().stream()
                .map(folder -> new ImportSuccessKeyDto(folder.getFolderId().orElse(null),
                        folder.getServiceId().orElse(null))).collect(Collectors.toList());
        Result<ImportResultDto> result = Result.success(new ImportResultDto(successfullyImported,
                quotasToImport.getImportFailures()));
        return folderDao.upsertAllRetryable(session, foldersToUpsert)
                .then(accountsDao.upsertAllRetryable(session, accountsToUpsert)
                        .then(quotasDao.upsertAllRetryable(session, quotasToUpsert)
                                .then(accountsQuotasDao.upsertAllRetryable(session, provisionsToUpsert)
                                        .then(folderOperationLogDao.upsertAllRetryable(session, historyToUpsert)
                                                .thenReturn(result)))));
    }

    private void addNextOpLogIdIncrements(List<FolderModel> foldersToUpsert,
                                          Map<String, FolderModel> foldersToIncrementNextOpLogId) {
        if (foldersToIncrementNextOpLogId.isEmpty()) {
            return;
        }
        Set<String> currentFolderIdsToUpsert = foldersToUpsert.stream().map(FolderModel::getId)
                .collect(Collectors.toSet());
        foldersToIncrementNextOpLogId.values().forEach(folder -> {
            if (!currentFolderIdsToUpsert.contains(folder.getId())) {
                FolderModel updatedFolder = folder.toBuilder()
                        .setNextOpLogOrder(folder.getNextOpLogOrder() + 1L)
                        .build();
                foldersToUpsert.add(updatedFolder);
            }
        });
    }

    private ErrorCollection toErrorCollection(List<ImportFailureDto> importFailures,
                                              Map<String, Integer> folderIndices,
                                              Map<Long, Integer> serviceIndices) {
        ErrorCollection.Builder result = ErrorCollection.builder();
        importFailures.forEach(failure -> {
            int index = getFolderIndex(folderIndices, serviceIndices, failure.getFolderId().orElse(null),
                    failure.getServiceId().orElse(null));
            String keyPrefix = "quotas." + index + ".";
            failure.getErrors().forEach(e -> result.addError(TypedError.invalid(e)));
            failure.getFieldErrors().forEach((k, v) -> v.forEach(t -> result.addError(keyPrefix + k,
                    TypedError.invalid(t))));
        });
        return result.build();
    }

    private int getFolderIndex(Map<String, Integer> folderIndices, Map<Long, Integer> serviceIndices,
                               String folderId, Long serviceId) {
        int index = -1;
        if (folderId != null) {
            index = folderIndices.getOrDefault(folderId, -1);
        }
        if (serviceId != null) {
            index = serviceIndices.getOrDefault(serviceId, -1);
        }
        return index;
    }

    private List<FolderModel> prepareFoldersToUpsert(PreparedImportData quotasToImport) {
        List<FolderModel> result = new ArrayList<>();
        quotasToImport.getPreGeneratedFolderIds().forEach((serviceId, folderId) -> {
            if (quotasToImport.getImportDirectories().getDefaultFolders().containsKey(serviceId)) {
                return;
            }
            FolderModel newFolder = FolderModel.newBuilder()
                    .setTenantId(Tenants.DEFAULT_TENANT_ID)
                    .setId(folderId)
                    .setServiceId(serviceId)
                    .setFolderType(FolderType.COMMON_DEFAULT_FOR_SERVICE)
                    .setDisplayName("default")
                    .setNextOpLogOrder(1L)
                    .build();
            result.add(newFolder);
        });
        return result;
    }

    private List<AccountModel> prepareAccountsToUpsert(PreparedImportData quotasToImport, Instant now,
                                                       Map<String, Map<String, AccountHistoryModel>> oldAccounts,
                                                       Map<String, Map<String, AccountHistoryModel>> newAccounts) {
        List<AccountModel> result = new ArrayList<>();
        quotasToImport.getImportedQuotas().forEach(q -> {
            String folderId;
            if (q.getFolderId().isPresent()) {
                folderId = q.getFolderId().get();
            } else {
                FolderModel folder = quotasToImport.getImportDirectories()
                        .getDefaultFolders().get(q.getServiceId().get());
                if (folder != null) {
                    folderId = folder.getId();
                } else {
                    folderId = quotasToImport.getPreGeneratedFolderIds().get(q.getServiceId().get());
                }
            }
            q.getAccounts().forEach(importedAccount -> {
                AccountSpaceModel accountSpace = quotasToImport.getImportDirectories().getAccountSpaces()
                        .get(importedAccount.getAccountSpaceId());
                String accountSpaceId = accountSpace != null ? accountSpace.getId() : null;
                PreValidatedImport.AccountExternalId accountExternalId = new PreValidatedImport.AccountExternalId(
                        importedAccount.getProviderId(),
                        importedAccount.getId(),
                        importedAccount.getAccountSpaceId()
                );
                AccountModel existingAccount = quotasToImport.getImportDirectories().getAccounts()
                        .get(accountExternalId);
                if (existingAccount == null) {
                    String accountId = quotasToImport.getPreGeneratedAccountIds().get(accountExternalId);
                    AccountModel newAccount = new AccountModel.Builder()
                            .setTenantId(Tenants.DEFAULT_TENANT_ID)
                            .setId(accountId)
                            .setVersion(0L)
                            .setProviderId(importedAccount.getProviderId())
                            .setAccountsSpacesId(accountSpaceId)
                            .setOuterAccountIdInProvider(importedAccount.getId())
                            .setOuterAccountKeyInProvider(importedAccount.getKey().orElse(null))
                            .setFolderId(folderId)
                            .setDisplayName(importedAccount.getDisplayName().orElse(null))
                            .setDeleted(importedAccount.isDeleted())
                            .setLastAccountUpdate(now)
                            // TODO Support versions here
                            .setLastReceivedVersion(null)
                            .setLatestSuccessfulAccountOperationId(null)
                            .setFreeTier(importedAccount.isFreeTier())
                            .build();
                    result.add(newAccount);
                    AccountHistoryModel newAccountHistory = AccountHistoryModel.builder()
                            .version(0L)
                            .providerId(importedAccount.getProviderId())
                            .accountsSpacesId(accountSpaceId)
                            .outerAccountIdInProvider(importedAccount.getId())
                            .outerAccountKeyInProvider(importedAccount.getKey().orElse(null))
                            .folderId(folderId)
                            .displayName(importedAccount.getDisplayName().orElse(null))
                            .deleted(importedAccount.isDeleted())
                            // TODO Support versions here
                            .lastReceivedVersion(null)
                            .build();
                    newAccounts.computeIfAbsent(folderId, k -> new HashMap<>()).put(accountId, newAccountHistory);
                } else {
                    boolean accountMoved = quotasToImport.getAccountsMoveInfo().getMovedAccounts()
                            .containsKey(existingAccount.toExternalId());
                    boolean deletedChanged = existingAccount.isDeleted() != importedAccount.isDeleted();
                    boolean nameChanged = !existingAccount.getDisplayName().equals(importedAccount.getDisplayName());
                    if (!accountMoved && !deletedChanged && !nameChanged) {
                        return;
                    }
                    AccountModel updatedAccount = new AccountModel.Builder(existingAccount)
                            .setTenantId(Tenants.DEFAULT_TENANT_ID)
                            .setId(existingAccount.getId())
                            .setVersion(existingAccount.getVersion() + 1L)
                            .setProviderId(existingAccount.getProviderId())
                            .setAccountsSpacesId(accountSpaceId)
                            .setOuterAccountIdInProvider(existingAccount.getOuterAccountIdInProvider())
                            .setOuterAccountKeyInProvider(existingAccount.getOuterAccountKeyInProvider().orElse(null))
                            .setFolderId(folderId)
                            .setDisplayName(importedAccount.getDisplayName().orElse(null))
                            .setDeleted(importedAccount.isDeleted())
                            .setLastAccountUpdate(now)
                            // TODO Support versions here
                            .setLastReceivedVersion(null)
                            .setLatestSuccessfulAccountOperationId(existingAccount
                                    .getLatestSuccessfulAccountOperationId().orElse(null))
                            .setFreeTier(existingAccount.isFreeTier())
                            .setReserveType(existingAccount.getReserveType().orElse(null))
                            .build();
                    result.add(updatedAccount);
                    AccountHistoryModel.Builder oldAccountBuilder = AccountHistoryModel.builder();
                    AccountHistoryModel.Builder newAccountBuilder = AccountHistoryModel.builder();
                    oldAccountBuilder.version(existingAccount.getVersion());
                    newAccountBuilder.version(existingAccount.getVersion() + 1L);
                    if (accountMoved) {
                        oldAccountBuilder.folderId(existingAccount.getFolderId());
                        newAccountBuilder.folderId(folderId);
                    }
                    if (deletedChanged) {
                        oldAccountBuilder.deleted(existingAccount.isDeleted());
                        newAccountBuilder.deleted(importedAccount.isDeleted());
                    }
                    if (nameChanged) {
                        oldAccountBuilder.displayName(existingAccount.getDisplayName().orElse(null));
                        newAccountBuilder.displayName(importedAccount.getDisplayName().orElse(null));
                    }
                    // TODO Support versions here
                    if (accountMoved) {
                        oldAccounts.computeIfAbsent(existingAccount.getFolderId(), k -> new HashMap<>())
                                .put(existingAccount.getId(), oldAccountBuilder.build());
                        newAccounts.computeIfAbsent(existingAccount.getFolderId(), k -> new HashMap<>())
                                .put(existingAccount.getId(), newAccountBuilder.build());
                    }
                    oldAccounts.computeIfAbsent(folderId, k -> new HashMap<>())
                            .put(existingAccount.getId(), oldAccountBuilder.build());
                    newAccounts.computeIfAbsent(folderId, k -> new HashMap<>())
                            .put(existingAccount.getId(), newAccountBuilder.build());
                }
            });
        });
        return result;
    }

    private List<AccountsQuotasModel> findProvisionsToUpsert(
            Map<AccountsQuotasModel.Identity, Set<Tuple2<ProvisionImportKey, ProvisionImportApplication>>> map,
            Instant now) {
        List<AccountsQuotasModel> result = new ArrayList<>();
        map.values().forEach(v -> {
            if (v.size() > 1) {
                Tuple2<ProvisionImportKey, ProvisionImportApplication> destinationApplication = v.stream()
                        .filter(t -> !t.getT2().isMoveSource()).findFirst().get();
                long provided = destinationApplication.getT2().getImportedProvidedQuota()
                        .orElseGet(() -> destinationApplication.getT2().getOriginalProvidedQuota().orElse(0L));
                long allocated = destinationApplication.getT2().getImportedAllocatedQuota()
                        .orElseGet(() -> destinationApplication.getT2().getOriginalAllocatedQuota().orElse(0L));
                String latestSuccessfulProvisionOperationId = destinationApplication.getT2()
                        .getOriginalLatestSuccessfulProvisionOperationId().orElse(null);
                result.add(new AccountsQuotasModel.Builder()
                        .setTenantId(Tenants.DEFAULT_TENANT_ID)
                        .setAccountId(destinationApplication.getT2().getAccountId())
                        .setResourceId(destinationApplication.getT2().getResourceId())
                        .setProvidedQuota(provided)
                        .setAllocatedQuota(allocated)
                        .setFolderId(destinationApplication.getT2().getFolderId())
                        .setProviderId(destinationApplication.getT2().getProviderId())
                        .setLastProvisionUpdate(now)
                        // TODO Add version support here
                        .setLastReceivedProvisionVersion(null)
                        .setLatestSuccessfulProvisionOperationId(latestSuccessfulProvisionOperationId)
                        .build());
                return;
            }
            Tuple2<ProvisionImportKey, ProvisionImportApplication> application = v.iterator().next();
            long provided = application.getT2().getImportedProvidedQuota()
                    .orElseGet(() -> application.getT2().getOriginalProvidedQuota().orElse(0L));
            long allocated = application.getT2().getImportedAllocatedQuota()
                    .orElseGet(() -> application.getT2().getOriginalAllocatedQuota().orElse(0L));
            String latestSuccessfulProvisionOperationId = application.getT2()
                    .getOriginalLatestSuccessfulProvisionOperationId().orElse(null);
            result.add(new AccountsQuotasModel.Builder()
                    .setTenantId(Tenants.DEFAULT_TENANT_ID)
                    .setAccountId(application.getT2().getAccountId())
                    .setResourceId(application.getT2().getResourceId())
                    .setProvidedQuota(provided)
                    .setAllocatedQuota(allocated)
                    .setFolderId(application.getT2().getFolderId())
                    .setProviderId(application.getT2().getProviderId())
                    .setLastProvisionUpdate(now)
                    // TODO Add version support here
                    .setLastReceivedProvisionVersion(null)
                    .setLatestSuccessfulProvisionOperationId(latestSuccessfulProvisionOperationId)
                    .build());
        });
        return result;
    }

    @SuppressWarnings("ParameterNumber")
    private List<FolderOperationLogModel> prepareLog(
            PreparedImportData quotasToImport,
            List<QuotaImportApplication> quotasToChange,
            Map<ProvisionImportKey, ProvisionImportApplication> provisionsToChange,
            YaUserDetails currentUser,
            List<FolderModel> foldersToUpsert,
            Map<String, FolderModel> foldersToIncrementNextOpLogIdOutput,
            Instant now,
            Map<String, Map<String, AccountHistoryModel>> oldAccounts,
            Map<String, Map<String, AccountHistoryModel>> newAccounts) {
        Map<String, FolderModel> foldersToUpsertById = foldersToUpsert.stream()
                .collect(Collectors.toMap(FolderModel::getId, Function.identity()));
        Map<String, FolderOperationLogModel.Builder> history = new HashMap<>();
        Map<String, Map<String, Long>> oldQuotas = new HashMap<>();
        Map<String, Map<String, Long>> newQuotas = new HashMap<>();
        Map<String, Map<String, Long>> oldBalances = new HashMap<>();
        Map<String, Map<String, Long>> newBalances = new HashMap<>();
        Map<String, Map<String, Map<String, ProvisionHistoryModel>>> oldProvisions = new HashMap<>();
        Map<String, Map<String, Map<String, ProvisionHistoryModel>>> newProvisions = new HashMap<>();
        quotasToChange.forEach(application -> {
            if (!application.hasQuotaChanges()) {
                return;
            }
            FolderModel existingFolder = quotasToImport.getImportDirectories()
                    .getFolders().get(application.getFolderId());
            FolderModel newFolder = foldersToUpsertById.get(application.getFolderId());
            history.computeIfAbsent(application.getFolderId(),
                    id -> newLogBuilder(id, now, currentUser, existingFolder, newFolder,
                            foldersToIncrementNextOpLogIdOutput, application.getProviderId()));
            oldQuotas.computeIfAbsent(application.getFolderId(), k -> new HashMap<>())
                    .put(application.getResourceId(), application.getOriginalQuota().orElse(0L));
            newQuotas.computeIfAbsent(application.getFolderId(), k -> new HashMap<>())
                    .put(application.getResourceId(), application.getImportedQuota().orElse(0L));

        });
        quotasToChange.forEach(application -> {
            if (!application.hasBalanceChanges()) {
                return;
            }
            FolderModel existingFolder = quotasToImport.getImportDirectories()
                    .getFolders().containsKey(application.getFolderId())
                    ? quotasToImport.getImportDirectories().getFolders().get(application.getFolderId())
                    : quotasToImport.getAccountsMoveInfo().getFoldersById().get(application.getFolderId());
            FolderModel newFolder = foldersToUpsertById.get(application.getFolderId());
            history.computeIfAbsent(application.getFolderId(),
                    id -> newLogBuilder(id, now, currentUser, existingFolder, newFolder,
                            foldersToIncrementNextOpLogIdOutput, application.getProviderId()));
            oldBalances.computeIfAbsent(application.getFolderId(), k -> new HashMap<>())
                    .put(application.getResourceId(), application.getOriginalBalance().orElse(0L));
            newBalances.computeIfAbsent(application.getFolderId(), k -> new HashMap<>())
                    .put(application.getResourceId(), application.getActualBalance().longValueExact());
        });
        provisionsToChange.values().forEach(application -> {
            if (!application.hasProvidedChanges()) {
                return;
            }
            FolderModel importedFolder = quotasToImport.getImportDirectories()
                    .getFolders().get(application.getFolderId());
            FolderModel existingFolder = importedFolder != null
                    ? importedFolder
                    : quotasToImport.getAccountsMoveInfo().getFoldersById().get(application.getFolderId());
            FolderModel newFolder = foldersToUpsertById.get(application.getFolderId());
            history.computeIfAbsent(application.getFolderId(),
                    id -> newLogBuilder(id, now, currentUser, existingFolder, newFolder,
                            foldersToIncrementNextOpLogIdOutput, application.getProviderId()));
            oldProvisions.computeIfAbsent(application.getFolderId(), x -> new HashMap<>())
                    .computeIfAbsent(application.getAccountId(), x -> new HashMap<>())
                    .put(application.getResourceId(), new ProvisionHistoryModel(application
                            .getOriginalProvidedQuota().orElse(0L), application
                            .getOriginalLastReceivedProvisionVersion().orElse(null)));
            newProvisions.computeIfAbsent(application.getFolderId(), x -> new HashMap<>())
                    .computeIfAbsent(application.getAccountId(), x -> new HashMap<>())
                    .put(application.getResourceId(), new ProvisionHistoryModel(application.getImportedProvidedQuota()
                            // TODO Add version support here
                            .orElse(0L), null));
        });
        oldQuotas.forEach((folderId, quotas) -> history.get(folderId).setOldQuotas(new QuotasByResource(quotas)));
        newQuotas.forEach((folderId, quotas) -> history.get(folderId).setNewQuotas(new QuotasByResource(quotas)));
        oldBalances.forEach((folderId, balances) -> history.get(folderId)
                .setOldBalance(new QuotasByResource(balances)));
        newBalances.forEach((folderId, balances) -> history.get(folderId)
                .setNewBalance(new QuotasByResource(balances)));
        oldProvisions.forEach((folderId, provisions) -> {
            Map<String, ProvisionsByResource> provisionsByAccount = new HashMap<>();
            provisions.forEach((accountId, provisionsByResource) -> {
                provisionsByAccount.put(accountId, new ProvisionsByResource(provisionsByResource));
            });
            history.get(folderId).setOldProvisions(new QuotasByAccount(provisionsByAccount));
        });
        newProvisions.forEach((folderId, provisions) -> {
            Map<String, ProvisionsByResource> provisionsByAccount = new HashMap<>();
            provisions.forEach((accountId, provisionsByResource) -> {
                provisionsByAccount.put(accountId, new ProvisionsByResource(provisionsByResource));
            });
            history.get(folderId).setNewProvisions(new QuotasByAccount(provisionsByAccount));
        });
        oldAccounts.forEach((folderId, accounts) -> {
            FolderOperationLogModel.Builder builder = findFolderHistoryBuilder(folderId, history,
                    foldersToIncrementNextOpLogIdOutput, quotasToImport, foldersToUpsertById, currentUser, now,
                    accounts);
            builder.setOldAccounts(new AccountsHistoryModel(accounts));
        });
        newAccounts.forEach((folderId, accounts) -> {
            FolderOperationLogModel.Builder builder = findFolderHistoryBuilder(folderId, history,
                    foldersToIncrementNextOpLogIdOutput, quotasToImport, foldersToUpsertById, currentUser, now,
                    accounts);
            builder.setNewAccounts(new AccountsHistoryModel(accounts));
        });
        return history.values().stream().map(FolderOperationLogModel.Builder::build).collect(Collectors.toList());
    }

    @SuppressWarnings("ParameterNumber")
    private FolderOperationLogModel.Builder findFolderHistoryBuilder(
            String folderId, Map<String, FolderOperationLogModel.Builder> history,
            Map<String, FolderModel> foldersToIncrementNextOpLogIdOutput,
            PreparedImportData quotasToImport, Map<String, FolderModel> foldersToUpsertById,
            YaUserDetails currentUser, Instant now, Map<String, AccountHistoryModel> accounts
    ) {
        FolderModel importedFolder = quotasToImport.getImportDirectories().getFolders().get(folderId);
        FolderModel existingFolder = importedFolder != null
                ? importedFolder
                : quotasToImport.getAccountsMoveInfo().getFoldersById().get(folderId);
        FolderModel newFolder = foldersToUpsertById.get(folderId);
        return history.computeIfAbsent(folderId,
                id -> newLogBuilder(id, now, currentUser, existingFolder, newFolder,
                        foldersToIncrementNextOpLogIdOutput,
                        accounts.entrySet().stream().findFirst().flatMap(e -> e.getValue().getProviderId()).orElse(null)
                ));
    }

    private FolderOperationLogModel.Builder newLogBuilder(String folderId, Instant now, YaUserDetails currentUser,
                                                          FolderModel existingFolder, FolderModel newFolder,
                                                          Map<String, FolderModel> foldersToIncrementNextOpLogId,
                                                          String providerId) {
        FolderOperationLogModel.Builder result = new FolderOperationLogModel.Builder()
                .setTenantId(Tenants.DEFAULT_TENANT_ID)
                .setId(UUID.randomUUID().toString())
                .setFolderId(folderId)
                .setOperationDateTime(now)
                .setProviderRequestId(null)
                .setOperationType(FolderOperationType.QUOTA_IMPORT)
                .setSourceFolderOperationsLogId(null)
                .setDestinationFolderOperationsLogId(null)
                .setAccountsQuotasOperationsId(null)
                .setQuotasDemandsId(null)
                .setOldQuotas(new QuotasByResource(Map.of()))
                .setNewQuotas(new QuotasByResource(Map.of()))
                .setOldProvisions(new QuotasByAccount(Map.of()))
                .setNewProvisions(new QuotasByAccount(Map.of()))
                .setOldBalance(new QuotasByResource(Map.of()))
                .setNewBalance(new QuotasByResource(Map.of()));
        if (currentUser.getUser().isPresent()) {
            result.setAuthorUserId(currentUser.getUser().get().getId());
            currentUser.getUser().get().getPassportUid().ifPresent(result::setAuthorUserUid);
        }
        if (currentUser.getProviders().size() == 1) {
            result.setAuthorProviderId(currentUser.getProviders().get(0).getId());
        } else if (currentUser.getProviders().size() > 1) {
            currentUser.getProviders().stream()
                    .filter(providerModel -> providerModel.getId().equals(providerId)).findFirst()
                    .ifPresent(providerModel -> result.setAuthorProviderId(providerModel.getId()));
        }
        // todo: if (currentUser.getTvmServiceId().isPresent()), set new field AuthorProviderTvmServiceId
        prepareFolderHistoryFields(result, existingFolder, newFolder);
        if (existingFolder == null) {
            result.setOrder(0L);
        } else {
            result.setOrder(existingFolder.getNextOpLogOrder());
            foldersToIncrementNextOpLogId.put(existingFolder.getId(), existingFolder);
        }
        return result;
    }

    private void prepareFolderHistoryFields(FolderOperationLogModel.Builder logBuilder,
                                            FolderModel existingFolder, FolderModel newFolder) {
        if (existingFolder == null) {
            logBuilder.setNewFolderFields(FolderHistoryFieldsModel.builder()
                    .serviceId(newFolder.getServiceId())
                    .version(newFolder.getVersion())
                    .displayName(newFolder.getDisplayName())
                    .description(newFolder.getDescription().orElse(null))
                    .deleted(newFolder.isDeleted())
                    .folderType(newFolder.getFolderType())
                    .tags(newFolder.getTags())
                    .build());
        }
    }

    private QuotaModel toUpdatedQuota(QuotaImportApplication application) {
        long quota = application.getImportedQuota().orElseGet(() -> application.getOriginalQuota().orElse(0L));
        return new QuotaModel(Tenants.DEFAULT_TENANT_ID, application.getFolderId(), application.getProviderId(),
                application.getResourceId(), quota, application.getActualBalance().longValueExact(), 0);
    }

}
