package ru.yandex.intranet.d.services.imports;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.dao.accounts.AccountsDao;
import ru.yandex.intranet.d.dao.accounts.AccountsQuotasDao;
import ru.yandex.intranet.d.dao.folders.FolderDao;
import ru.yandex.intranet.d.dao.providers.ProvidersDao;
import ru.yandex.intranet.d.dao.quotas.QuotasDao;
import ru.yandex.intranet.d.dao.resources.ResourcesDao;
import ru.yandex.intranet.d.dao.services.ServicesDao;
import ru.yandex.intranet.d.dao.units.UnitsEnsemblesDao;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.model.accounts.AccountModel;
import ru.yandex.intranet.d.model.accounts.AccountSpaceModel;
import ru.yandex.intranet.d.model.accounts.AccountsQuotasModel;
import ru.yandex.intranet.d.model.folders.FolderModel;
import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.model.quotas.QuotaModel;
import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.model.services.ServiceWithStatesModel;
import ru.yandex.intranet.d.model.units.UnitModel;
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel;
import ru.yandex.intranet.d.services.imports.PreValidatedImport.AccountExternalId;
import ru.yandex.intranet.d.services.integration.providers.rest.model.ResourceComplexKey;
import ru.yandex.intranet.d.services.security.SecurityManagerService;
import ru.yandex.intranet.d.util.Uuids;
import ru.yandex.intranet.d.util.result.ErrorCollection;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.util.result.TypedError;
import ru.yandex.intranet.d.util.units.Units;
import ru.yandex.intranet.d.web.model.imports.AccountSpaceIdentityDto;
import ru.yandex.intranet.d.web.model.imports.ImportAccountDto;
import ru.yandex.intranet.d.web.model.imports.ImportAccountProvisionDto;
import ru.yandex.intranet.d.web.model.imports.ImportDto;
import ru.yandex.intranet.d.web.model.imports.ImportFailureDto;
import ru.yandex.intranet.d.web.model.imports.ImportFolderDto;
import ru.yandex.intranet.d.web.model.imports.ImportResourceDto;
import ru.yandex.intranet.d.web.model.imports.ResourceIdentityDto;
import ru.yandex.intranet.d.web.model.imports.SegmentKey;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;

/**
 * Quotas imports validation service implementation.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class QuotasImportsValidationService {

    private static final long IMPORTED_OBJECT_COUNT_LIMIT = 500;
    private static final int ACCOUNT_KEY_LIMIT = 1024;
    private static final int ACCOUNT_ID_LIMIT = 1024;
    private static final int ACCOUNT_DISPLAY_NAME_LIMIT = 2048;

    private final MessageSource messages;
    private final SecurityManagerService securityManagerService;
    private final QuotasImportsLoadingService quotasImportsLoadingService;

    @SuppressWarnings("ParameterNumber")
    public QuotasImportsValidationService(@Qualifier("messageSource") MessageSource messages,
                                          ProvidersDao providersDao,
                                          ResourcesDao resourcesDao,
                                          FolderDao folderDao,
                                          AccountsDao accountsDao,
                                          ServicesDao servicesDao,
                                          UnitsEnsemblesDao unitsEnsemblesDao,
                                          QuotasDao quotasDao,
                                          AccountsQuotasDao accountsQuotasDao,
                                          SecurityManagerService securityManagerService,
                                          QuotasImportsLoadingService quotasImportsLoadingService
    ) {
        this.messages = messages;
        this.securityManagerService = securityManagerService;
        this.quotasImportsLoadingService = quotasImportsLoadingService;
    }

    public Result<Void> checkWritePermissions(YaUserDetails currentUser, Locale locale) {
        return securityManagerService.checkImportPermissions(currentUser, locale);
    }

    public Result<ImportIndices> preValidateGlobal(ImportDto quotasToImport, Locale locale) {
        if (quotasToImport.getQuotas().isEmpty() || quotasToImport.getQuotas().get().isEmpty()) {
            ErrorCollection error = ErrorCollection.builder().addError("quotas", TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale))).build();
            return Result.failure(error);
        }
        List<ImportFolderDto> folderQuotas = quotasToImport.getQuotas().get();
        Set<String> seenFolders = new HashSet<>();
        Set<Long> seenServices = new HashSet<>();
        Map<String, Set<String>> seenAccountIdsByProvider = new HashMap<>();
        Map<String, Set<String>> seenNonDeletedAccountKeysByProvider = new HashMap<>();
        long totalEntities = 0L;
        ErrorCollection.Builder errors = ErrorCollection.builder();
        Map<String, Integer> folderIndices = new HashMap<>();
        Map<Long, Integer> serviceIndices = new HashMap<>();
        for (int i = 0; i < folderQuotas.size(); i++) {
            ImportFolderDto folderQuota = folderQuotas.get(i);
            String fieldKey = "quotas." + i;
            if (folderQuota == null) {
                errors.addError(fieldKey, TypedError.invalid(messages
                        .getMessage("errors.field.is.required", null, locale)));
                continue;
            }
            if ((folderQuota.getFolderId().isEmpty() || folderQuota.getFolderId().get().isBlank())
                    && folderQuota.getServiceId().isEmpty()) {
                errors.addError(fieldKey, TypedError.invalid(messages
                        .getMessage("errors.folder.id.or.service.id.is.required", null, locale)));
                continue;
            }
            if (folderQuota.getFolderId().isPresent() && folderQuota.getServiceId().isPresent()) {
                errors.addError(fieldKey, TypedError.invalid(messages
                        .getMessage("errors.folder.id.and.service.id.can.not.be.both.supplied", null, locale)));
                continue;
            }
            Optional<String> folderIdO = folderQuota.getFolderId();
            if (folderIdO.isPresent()) {
                folderIndices.put(folderIdO.get(), i);
            }
            Optional<Long> serviceIdO = folderQuota.getServiceId();
            if (serviceIdO.isPresent()) {
                serviceIndices.put(serviceIdO.get(), i);
            }
            if (folderIdO.isPresent() && seenFolders.contains(folderIdO.get())) {
                errors.addError(fieldKey, TypedError.invalid(messages
                        .getMessage("errors.duplicate.folder.ids.are.not.allowed", null, locale)));
            } else {
                folderIdO.ifPresent(seenFolders::add);
            }
            if (serviceIdO.isPresent() && seenServices.contains(serviceIdO.get())) {
                errors.addError(fieldKey, TypedError.invalid(messages
                        .getMessage("errors.duplicate.service.ids.are.not.allowed", null, locale)));
            } else {
                serviceIdO.ifPresent(seenServices::add);
            }
            folderQuota.getAccounts().ifPresent(accounts -> {
                for (int j = 0; j < accounts.size(); j++) {
                    ImportAccountDto account = accounts.get(j);
                    if (account == null || account.getProviderId().isEmpty()) {
                        continue;
                    }
                    String fieldKeyPrefix = fieldKey + ".accounts." + j;
                    String providerId = account.getProviderId().get();
                    account.getId().ifPresent(accountId -> {
                        Set<String> seenAccountIds = seenAccountIdsByProvider.computeIfAbsent(providerId,
                                k -> new HashSet<>());
                        if (seenAccountIds.contains(accountId)) {
                            errors.addError(fieldKeyPrefix + ".id", TypedError.invalid(messages
                                    .getMessage("errors.duplicate.account.ids.are.not.allowed", null, locale)));
                        } else {
                            seenAccountIds.add(accountId);
                        }
                    });
                    account.getKey().ifPresent(accountKey -> {
                        Set<String> seenNonDeletedAccountKeys = seenNonDeletedAccountKeysByProvider
                                .computeIfAbsent(providerId, k -> new HashSet<>());
                        boolean deleted = account.getDeleted().orElse(false);
                        if (!deleted && seenNonDeletedAccountKeys.contains(accountKey)) {
                            errors.addError(fieldKeyPrefix + ".key", TypedError.invalid(messages
                                    .getMessage("errors.duplicate.account.keys.are.not.allowed", null, locale)));
                        } else if (!deleted) {
                            seenNonDeletedAccountKeys.add(accountKey);
                        }
                    });
                }
            });
            totalEntities += folderQuota.getResourceQuotas().map(List::size).orElse(0);
            totalEntities += folderQuota.getAccounts().map(List::size).orElse(0);
            totalEntities += folderQuota.getAccounts().map(list -> list.stream().filter(Objects::nonNull)
                    .mapToLong(v -> v.getProvisions().map(List::size).orElse(0)).sum()).orElse(0L);
        }
        if (totalEntities > IMPORTED_OBJECT_COUNT_LIMIT) {
            errors.addError(TypedError.invalid(messages
                    .getMessage("errors.too.many.imported.objects", null, locale)));
        }
        if (totalEntities == 0L) {
            errors.addError(TypedError.invalid(messages
                    .getMessage("errors.no.imported.objects", null, locale)));
        }
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build());
        }
        return Result.success(new ImportIndices(folderIndices, serviceIndices));
    }

    public PreValidatedImport preValidateLocal(ImportDto quotasToImport, ImportIndices importIndices, Locale locale) {
        List<ImportFailureDto> failures = new ArrayList<>();
        List<ImportFolderDto> quotas = new ArrayList<>();
        Set<String> providerIds = new HashSet<>();
        Set<String> resourceIds = new HashSet<>();
        Set<ProviderWithResourceIdentity> providerResourceIdentitySet = new HashSet<>();
        Set<AccountExternalId> accountIds = new HashSet<>();
        Set<String> folderIds = new HashSet<>();
        Set<Long> serviceIds = new HashSet<>();
        Map<AccountExternalId, String> preGeneratedAccountIds = new HashMap<>();
        Map<Long, String> preGeneratedFolderIds = new HashMap<>();
        List<ImportFolderDto> folderQuotas = quotasToImport.getQuotas().get();
        folderQuotas.forEach(folderQuota -> {
            ErrorCollection.Builder errors = ErrorCollection.builder();
            String folderId = folderQuota.getFolderId().orElse(null);
            Long serviceId = folderQuota.getServiceId().orElse(null);
            if (folderQuota.getFolderId().isPresent() && !Uuids.isValidUuid(folderId)) {
                errors.addError("folderId", TypedError.invalid(messages
                        .getMessage("errors.folder.not.found", null, locale)));
            }
            folderQuota.getResourceQuotas().ifPresent(resources -> errors.add(validateResources(resources, locale)));
            folderQuota.getAccounts().ifPresent(accounts -> errors.add(validateAccounts(accounts, locale)));
            if (errors.hasAnyErrors()) {
                failures.add(toImportFailure(folderId, serviceId, errors.build()));
            } else {
                quotas.add(folderQuota);
                folderQuota.getFolderId().ifPresent(folderIds::add);
                folderQuota.getServiceId().ifPresent(id -> {
                    serviceIds.add(id);
                    preGeneratedFolderIds.put(id, UUID.randomUUID().toString());
                });
                folderQuota.getResourceQuotas().ifPresent(resources -> resources.forEach(resource -> {
                    resource.getResourceId().ifPresent(resourceIds::add);
                    resource.getResourceIdentity().ifPresent(resourceIdentityDto -> resource.getProviderId().ifPresent(
                            providerId -> providerResourceIdentitySet.add(
                                    new ProviderWithResourceIdentity(providerId,
                                            new ResourceComplexKey(resourceIdentityDto),
                                            resourceIdentityDto.getAccountSpaceIdentityDto()
                                                    .map(AccountSpaceIdentity::fromDto).orElse(null)))));
                    resource.getProviderId().ifPresent(providerIds::add);
                }));
                folderQuota.getAccounts().ifPresent(accounts -> accounts.forEach(account -> {
                    if (account.getId().isPresent() && account.getProviderId().isPresent()) {
                        AccountExternalId accountId = new AccountExternalId(
                                account.getProviderId().get(),
                                account.getId().get(),
                                AccountSpaceIdentity.fromDto(account.getAccountSpaceIdentity().orElse(null))
                        );
                        boolean added = accountIds.add(accountId);
                        if (added) {
                            preGeneratedAccountIds.put(accountId, UUID.randomUUID().toString());
                        }
                    }
                    account.getProviderId().ifPresent(providerIds::add);
                    account.getProvisions().ifPresent(provisions -> provisions.forEach(provision -> {
                        provision.getResourceId().ifPresent(resourceIds::add);
                        provision.getResourceIdentity().ifPresent(resourceIdentityDto -> account.getProviderId()
                                .ifPresent(providerId -> providerResourceIdentitySet.add(
                                        new ProviderWithResourceIdentity(providerId,
                                                new ResourceComplexKey(resourceIdentityDto),
                                                resourceIdentityDto.getAccountSpaceIdentityDto()
                                                        .map(AccountSpaceIdentity::fromDto).orElse(null)))));
                    }));
                }));
            }
        });
        return new PreValidatedImport(failures, quotas, providerIds, resourceIds, providerResourceIdentitySet,
                accountIds, folderIds, serviceIds, preGeneratedAccountIds, preGeneratedFolderIds,
                importIndices.getFolderIndices(), importIndices.getServiceIndices());
    }

    private ErrorCollection validateResources(List<ImportResourceDto> resources, Locale locale) {
        ErrorCollection.Builder errors = ErrorCollection.builder();
        for (int i = 0; i < resources.size(); i++) {
            ImportResourceDto resource = resources.get(i);
            if (resource == null) {
                errors.addError("resourceQuotas." + i, TypedError.invalid(messages
                        .getMessage("errors.field.is.required", null, locale)));
                continue;
            }
            if (resource.getResourceId().isPresent() && resource.getResourceIdentity().isPresent()) {
                errors.addError("resourceQuotas." + i + ".resourceId", TypedError.invalid(messages
                        .getMessage("errors.resource.either.id.or.identity.allowed", null, locale)));
                errors.addError("resourceQuotas." + i + ".resourceIdentity", TypedError.invalid(messages
                        .getMessage("errors.resource.either.id.or.identity.allowed", null, locale)));
            }
            if (resource.getResourceId().isPresent()) {
                validateUuid(resource::getResourceId, errors, "resourceQuotas." + i + ".resourceId",
                        "errors.resource.not.found", locale);
            } else {
                validateIdentity(resource::getResourceIdentity, errors,
                        "resourceQuotas." + i + ".resourceIdentity", locale);
            }
            validateUuid(resource::getProviderId, errors, "resourceQuotas." + i + ".providerId",
                    "errors.provider.not.found", locale);
            validateNonNegativeValue(resource::getQuota, errors,
                    "resourceQuotas." + i + ".quota", locale);
            validateNonNegativeValue(resource::getBalance, errors,
                    "resourceQuotas." + i + ".balance", locale);
            validateNonBlank(resource::getQuotaUnitKey, errors,
                    "resourceQuotas." + i + ".quotaUnitKey", locale);
            validateNonBlank(resource::getBalanceUnitKey, errors,
                    "resourceQuotas." + i + ".balanceUnitKey", locale);
        }
        return errors.build();
    }

    private ErrorCollection validateAccounts(List<ImportAccountDto> accounts, Locale locale) {
        ErrorCollection.Builder errors = ErrorCollection.builder();
        for (int i = 0; i < accounts.size(); i++) {
            ImportAccountDto account = accounts.get(i);
            if (account == null) {
                errors.addError("accounts." + i, TypedError.invalid(messages
                        .getMessage("errors.field.is.required", null, locale)));
                continue;
            }
            validateNonBlankWithLength(account::getId, errors, ACCOUNT_ID_LIMIT,
                    "accounts." + i + ".id", locale);
            validateNonBlankOrAbsentWithLength(account::getKey, errors, ACCOUNT_KEY_LIMIT,
                    "accounts." + i + ".key", locale);
            validateNonBlankOrAbsentWithLength(account::getDisplayName, errors, ACCOUNT_DISPLAY_NAME_LIMIT,
                    "accounts." + i + ".displayName", locale);
            validateUuid(account::getProviderId, errors, "accounts." + i + ".providerId",
                    "errors.provider.not.found", locale);
            boolean deletedAccount = account.getDeleted().orElse(false);
            String fieldKeyPrefix = "accounts." + i + "provisions.";
            account.getProvisions().ifPresent(provisions -> {
                for (int j = 0; j < provisions.size(); j++) {
                    ImportAccountProvisionDto provision = provisions.get(j);
                    if (provision == null) {
                        errors.addError(fieldKeyPrefix + j, TypedError.invalid(messages
                                .getMessage("errors.field.is.required", null, locale)));
                        continue;
                    }
                    if (provision.getResourceId().isPresent()) {
                        validateUuid(provision::getResourceId, errors, fieldKeyPrefix + j + ".resourceId",
                                "errors.resource.not.found", locale);
                    } else {
                        validateIdentity(provision::getResourceIdentity, errors,
                                "resourceQuotas." + j + ".resourceIdentity", locale);
                    }
                    if (deletedAccount) {
                        validateZeroValue(provision::getProvided, errors,
                                fieldKeyPrefix + j + ".provided", locale);
                        validateZeroValue(provision::getAllocated, errors,
                                fieldKeyPrefix + j + ".allocated", locale);
                    } else {
                        validateNonNegativeValue(provision::getProvided, errors,
                                fieldKeyPrefix + j + ".provided", locale);
                        validateNonNegativeValue(provision::getAllocated, errors,
                                fieldKeyPrefix + j + ".allocated", locale);
                    }
                    validateNonBlank(provision::getProvidedUnitKey, errors,
                            fieldKeyPrefix + j + ".providedUnitKey", locale);
                    validateNonBlank(provision::getAllocatedUnitKey, errors,
                            fieldKeyPrefix + j + ".allocatedUnitKey", locale);
                }
            });
        }
        return errors.build();
    }

    public Mono<Result<ValidatedImport>> checkWritePermissions(ValidatedImport quotasToImport,
                                                               YaUserDetails currentUser,
                                                               Locale locale) {
        Set<String> providerIds = new HashSet<>();
        quotasToImport.getQuotas().forEach(f -> {
            f.getResourceQuotas().forEach(q -> providerIds.add(q.getProviderId()));
            f.getAccounts().forEach(a -> providerIds.add(a.getProviderId()));
        });
        Set<ProviderModel> providers = providerIds.stream()
                .map(id -> quotasToImport.getDirectories().getProviders().get(id)).collect(Collectors.toSet());
        return securityManagerService.checkImportPermissions(currentUser, providers, locale)
                .map(result -> result.andThen(v -> Result.success(quotasToImport)));
    }

    public Mono<Result<ValidatedImport>> doValidate(PreValidatedImport quotasToImport,
                                                    YdbTxSession session,
                                                    Locale locale) {
        return quotasImportsLoadingService.loadDirectories(quotasToImport, session, false).map(directories -> {
            Map<ResourceComplexKey, ResourceModel> resourcesByComplexKey = directories.getResourcesByComplexKey();
            List<ImportFolder> quotas = new ArrayList<>();
            List<ImportFailureDto> failures = new ArrayList<>(quotasToImport.getFailures());
            quotasToImport.getQuotas().forEach(folderQuota -> {
                ErrorCollection.Builder errors = ErrorCollection.builder();
                validateFolderId(folderQuota::getFolderId, directories.getFolders(), directories.getServices(),
                        errors, "folderId", locale);
                validateServiceId(folderQuota::getServiceId, directories.getServices(), errors, "serviceId", locale);
                Optional<String> targetFolderId = getTargetFolderId(folderQuota, directories);
                boolean importToDefaultFolder = folderQuota.getServiceId().isPresent();
                folderQuota.getResourceQuotas().ifPresent(resources -> {
                    for (int i = 0; i < resources.size(); i++) {
                        ImportResourceDto importedResource = resources.get(i);
                        if (importedResource.getResourceId().isEmpty()) {
                            importedResource = validateImportResourceIdentity(importedResource, directories, errors, i,
                                    locale);
                        }
                        validateImportResource(importedResource, directories, errors, i, locale);
                    }
                });
                folderQuota.getAccounts().ifPresent(accounts -> {
                    for (int i = 0; i < accounts.size(); i++) {
                        ImportAccountDto importedAccount = accounts.get(i);
                        validateImportAccount(importedAccount, directories, errors, i, locale,
                                targetFolderId.orElse(null), importToDefaultFolder);
                    }
                    validateAccountsPerProviderPerFolderCount(directories, accounts, errors, locale);
                });
                String folderId = folderQuota.getFolderId().orElse(null);
                Long serviceId = folderQuota.getServiceId().orElse(null);
                if (errors.hasAnyErrors()) {
                    failures.add(toImportFailure(folderId, serviceId, errors.build()));
                } else {
                    List<ImportResource> resourceQuotas = new ArrayList<>();
                    List<ImportAccount> accounts = new ArrayList<>();
                    folderQuota.getResourceQuotas().ifPresent(r -> r.forEach(resource -> {
                        String resourceId = resource.getResourceId().isPresent()
                                ? resource.getResourceId().get()
                                : resourcesByComplexKey.get(
                                        new ResourceComplexKey(resource.getResourceIdentity().get())).getId();
                        String accountsSpaceId = directories.getResources().get(resourceId).getAccountsSpacesId();
                        ImportResource importResource = new ImportResource(resourceId,
                                resource.getProviderId().get(), resource.getQuota().get(),
                                resource.getQuotaUnitKey().get(), resource.getBalance().get(),
                                resource.getBalanceUnitKey().get(), accountsSpaceId);
                        resourceQuotas.add(importResource);
                    }));
                    folderQuota.getAccounts().ifPresent(a -> a.forEach(account -> {
                        List<ImportAccountProvision> provisions = new ArrayList<>();
                        account.getProvisions().ifPresent(p -> p.forEach(provision -> {
                            String resourceId = provision
                                    .getResourceId().isPresent() ? provision.getResourceId().get()
                                    : resourcesByComplexKey.get(
                                            new ResourceComplexKey(provision.getResourceIdentity().get())).getId();
                            String accountsSpaceId = directories.getResources().get(resourceId).getAccountsSpacesId();
                            ImportAccountProvision importProvision = new ImportAccountProvision(resourceId,
                                    provision.getProvided().get(), provision.getProvidedUnitKey().get(),
                                    provision.getAllocated().get(), provision.getAllocatedUnitKey().get(),
                                    accountsSpaceId);
                            provisions.add(importProvision);
                        }));
                        ImportAccount importAccount = new ImportAccount(account.getId().get(),
                                account.getKey().orElse(null), account.getDisplayName().orElse(null),
                                account.getDeleted().orElse(false), account.getProviderId().get(), provisions,
                                AccountSpaceIdentity.fromDto(account.getAccountSpaceIdentity().orElse(null)),
                                account.isFreeTier().orElse(false));
                        accounts.add(importAccount);
                    }));
                    ImportFolder importFolder = new ImportFolder(folderId, serviceId, resourceQuotas, accounts);
                    quotas.add(importFolder);
                }
            });
            return Result.success(new ValidatedImport(directories, failures, quotas,
                    quotasToImport.getPreGeneratedAccountIds(), quotasToImport.getPreGeneratedFolderIds(),
                    quotasToImport.getFolderIndices(), quotasToImport.getServiceIndices()));
        });
    }

    public void validateAccountsPerProviderPerFolderCount(ValidatedDirectories directories,
                                                          List<ImportAccountDto> accounts,
                                                          ErrorCollection.Builder errors,
                                                          Locale locale) {
        Map<String, Set<ImportAccountDto>> accountByProvider = accounts.stream()
                .filter(a -> a.getProviderId().isPresent())
                .filter(a -> !a.getDeleted().orElse(false))
                .collect(Collectors.groupingBy(v -> v.getProviderId().get(), Collectors.toSet()));
        boolean moreAccountsThanAllowed = accountByProvider.entrySet().stream().anyMatch(entry -> {
            if (!directories.getProviders().containsKey(entry.getKey())) {
                return false;
            }
            ProviderModel provider = directories.getProviders().get(entry.getKey());
            return !provider.isMultipleAccountsPerFolder() && entry.getValue().size() > 1;
        });
        if (moreAccountsThanAllowed) {
            errors.addError("accounts", TypedError.invalid(messages.getMessage(
                    "errors.no.more.than.one.account.per.folder.is.allowed.for.the.provider", null, locale)));
        }
    }

    public Mono<Result<PreparedImportData>> doValidateQuotas(ValidatedImport quotasToImport,
                                                             YdbTxSession session,
                                                             Locale locale) {
        return quotasImportsLoadingService.loadImportedQuotas(session, quotasToImport)
                .flatMap(loaded -> quotasImportsLoadingService.loadAccountsMoveInfo(session, loaded)
                .flatMap(moveInfo -> quotasImportsLoadingService.loadAccountDeleteInfo(session, loaded)
                .flatMap(deleteInfo -> quotasImportsLoadingService.loadAccountsCounts(
                        quotasToImport, session
                ).map(accountsCounts -> {
                    ErrorCollection.Builder errors = ErrorCollection.builder();
                    List<ImportFolder> importedQuotas = new ArrayList<>();
                    List<ImportFailureDto> importFailures = new ArrayList<>(loaded
                            .getValidatedImport().getFailures());
                    Map<QuotaModel.Key, QuotaImportApplication> appliedQuotas = new HashMap<>();
                    Map<ProvisionImportKey, ProvisionImportApplication> appliedProvisions = new HashMap<>();
                    applyImport(appliedQuotas, appliedProvisions, importedQuotas, importFailures,
                            loaded, moveInfo, deleteInfo, accountsCounts, errors, locale);
                    ImportApplication importApplication = new ImportApplication(appliedQuotas, appliedProvisions);
                    if (errors.hasAnyErrors()) {
                        return Result.failure(errors.build());
                    }
                    return Result.success(new PreparedImportData(importedQuotas, importFailures,
                            loaded.getValidatedImport().getPreGeneratedAccountIds(),
                            loaded.getValidatedImport().getPreGeneratedFolderIds(),
                            loaded.getValidatedImport().getDirectories(), moveInfo, importApplication,
                            loaded.getValidatedImport().getFolderIndices(),
                            loaded.getValidatedImport().getServiceIndices()

                    ));
                })))
        );
    }

    private void validateUuid(Supplier<Optional<String>> supplier,
                              ErrorCollection.Builder errors,
                              String fieldKey,
                              String invalidUuidMessage,
                              Locale locale) {
        Optional<String> id = supplier.get();
        if (id.isEmpty() || id.get().isBlank()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
            return;
        }
        if (!Uuids.isValidUuid(id.get())) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage(invalidUuidMessage, null, locale)));
        }
    }

    private void validateIdentity(Supplier<Optional<ResourceIdentityDto>> supplier,
                                  ErrorCollection.Builder errors,
                                  String fieldKey,
                                  Locale locale) {
        Optional<ResourceIdentityDto> resourceIdentityDtoO = supplier.get();
        if (resourceIdentityDtoO.isEmpty()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
            return;
        }
        ResourceIdentityDto resourceIdentityDto = resourceIdentityDtoO.get();

        String resourceTypeKey = resourceIdentityDto.getResourceTypeKey().orElse(null);
        if (resourceTypeKey == null || resourceTypeKey.isBlank()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.resource.type.not.found", null, locale)));
        }

        Optional<AccountSpaceIdentityDto> accountSpaceIdentityDtoO = resourceIdentityDto.getAccountSpaceIdentityDto();
        if (accountSpaceIdentityDtoO.isPresent()) {
            AccountSpaceIdentityDto accountSpaceIdentityDto = accountSpaceIdentityDtoO.get();

            String accountSpaceId = accountSpaceIdentityDto.getAccountSpaceId();
            List<SegmentKey> segments = accountSpaceIdentityDto.getSegments();
            if ((accountSpaceId == null && segments == null) || (accountSpaceId != null && accountSpaceId.isBlank())) {
                errors.addError(fieldKey, TypedError.invalid(messages
                        .getMessage("errors.accounts.space.not.found", null, locale)));
            }

            if ((accountSpaceId == null && segments == null) || (segments != null && segments.isEmpty())) {
                errors.addError(fieldKey, TypedError.invalid(messages
                        .getMessage("errors.accounts.space.not.found", null, locale)));
            } else if (segments != null) {
                validateSegmentKey(segments, errors, fieldKey, locale);
            }
        }

        resourceIdentityDto.getSegments().ifPresent(segmentKeys -> validateSegmentKey(segmentKeys, errors, fieldKey,
                locale));
    }

    private void validateSegmentKey(List<SegmentKey> segments, ErrorCollection.Builder errors, String fieldKey,
                                    Locale locale) {
        segments.forEach(segmentKey -> {
            String segmentationKey = segmentKey.getSegmentationKey();
            if (segmentationKey == null || segmentationKey.isBlank()) {
                errors.addError(fieldKey, TypedError.invalid(messages
                        .getMessage("errors.resource.segmentation.not.found", null, locale)));
            }

            String segmentKey1 = segmentKey.getSegmentKey();
            if (segmentKey1 == null || segmentKey1.isBlank()) {
                errors.addError(fieldKey, TypedError.invalid(messages
                        .getMessage("errors.resource.segment.not.found", null, locale)));
            }
        });
    }

    private void validateNonNegativeValue(Supplier<Optional<Long>> supplier,
                                          ErrorCollection.Builder errors,
                                          String fieldKey,
                                          Locale locale) {
        Optional<Long> value = supplier.get();
        if (value.isEmpty()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
            return;
        }
        if (value.get() < 0) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.number.must.be.non.negative", null, locale)));
        }
    }

    private void validateZeroValue(Supplier<Optional<Long>> supplier,
                                   ErrorCollection.Builder errors,
                                   String fieldKey,
                                   Locale locale) {
        Optional<Long> value = supplier.get();
        if (value.isEmpty()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
            return;
        }
        if (value.get() != 0) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.number.must.be.zero", null, locale)));
        }
    }

    private void validateNonBlank(Supplier<Optional<String>> supplier,
                                  ErrorCollection.Builder errors,
                                  String fieldKey,
                                  Locale locale) {
        Optional<String> value = supplier.get();
        if (value.isEmpty() || value.get().isBlank()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        }
    }

    private void validateNonBlankOrAbsentWithLength(Supplier<Optional<String>> supplier,
                                                    ErrorCollection.Builder errors,
                                                    int lengthLimit,
                                                    String fieldKey,
                                                    Locale locale) {
        Optional<String> value = supplier.get();
        if (value.isPresent() && value.get().isBlank()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.non.blank.text.is.required", null, locale)));
            return;
        }
        if (value.isPresent() && value.get().length() > lengthLimit) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.text.is.too.long", null, locale)));
        }
    }

    private void validateNonBlankWithLength(Supplier<Optional<String>> supplier,
                                            ErrorCollection.Builder errors,
                                            int lengthLimit,
                                            String fieldKey,
                                            Locale locale) {
        Optional<String> value = supplier.get();
        if (value.isEmpty() || value.get().isBlank()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
            return;
        }
        if (value.get().length() > lengthLimit) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.text.is.too.long", null, locale)));
        }
    }

    private ImportFailureDto toImportFailure(String folderId, Long serviceId, ErrorCollection errors) {
        Set<String> errorsSet = errors.getErrors().stream().map(TypedError::getError).collect(Collectors.toSet());
        Map<String, Set<String>> fieldErrors = errors.getFieldErrors().entrySet().stream()
                .collect(Collectors.toMap(Map.Entry::getKey, e -> e.getValue().stream().map(TypedError::getError)
                        .collect(Collectors.toSet())));
        return new ImportFailureDto(folderId, serviceId, errorsSet, fieldErrors);
    }

    private Optional<String> getTargetFolderId(ImportFolderDto folderQuota, ValidatedDirectories directories) {
        if (folderQuota.getFolderId().isPresent()
                && directories.getFolders().containsKey(folderQuota.getFolderId().get())) {
            return folderQuota.getFolderId();
        }
        if (folderQuota.getServiceId().isPresent()
                && directories.getDefaultFolders().containsKey(folderQuota.getServiceId().get())) {
            return Optional.ofNullable(directories.getDefaultFolders().get(folderQuota.getServiceId().get()).getId());
        }
        return Optional.empty();
    }

    private void validateFolderId(Supplier<Optional<String>> supplier,
                                  Map<String, FolderModel> folders,
                                  Map<Long, ServiceWithStatesModel> services,
                                  ErrorCollection.Builder errors,
                                  String fieldKey,
                                  Locale locale) {
        Optional<String> folderIdO = supplier.get();
        if (folderIdO.isEmpty()) {
            return;
        }
        FolderModel folder = folders.get(folderIdO.get());
        if (folder == null || folder.isDeleted()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.folder.not.found", null, locale)));
            return;
        }
        validateService(services.get(folder.getServiceId()), errors, fieldKey, locale);
    }

    private void validateServiceId(Supplier<Optional<Long>> supplier,
                                   Map<Long, ServiceWithStatesModel> services,
                                   ErrorCollection.Builder errors,
                                   String fieldKey,
                                   Locale locale) {
        Optional<Long> serviceIdO = supplier.get();
        if (serviceIdO.isEmpty()) {
            return;
        }
        validateService(services.get(serviceIdO.get()), errors, fieldKey, locale);
    }

    private void validateService(ServiceWithStatesModel service, ErrorCollection.Builder errors,
                                 String fieldKey, Locale locale) {
        if (service == null) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.service.not.found", null, locale)));
        }
    }

    private ImportResourceDto validateImportResourceIdentity(ImportResourceDto importedResource,
                                                             ValidatedDirectories directories,
                                                             ErrorCollection.Builder errors,
                                                             int resourceIndex,
                                                             Locale locale) {
        ResourceModel resourceModel = directories.getResourcesByComplexKey().get(
                new ResourceComplexKey(importedResource.getResourceIdentity().get()));
        if (resourceModel == null) {
            errors.addError("resourceQuotas." + resourceIndex + ".resourceIdentity", TypedError.invalid(messages
                    .getMessage("errors.resource.not.found", null, locale)));
            return importedResource;
        } else {
            return new ImportResourceDto(
                    resourceModel.getId(),
                    null,
                    importedResource.getProviderId().orElse(null),
                    importedResource.getQuota().orElse(null),
                    importedResource.getQuotaUnitKey().orElse(null),
                    importedResource.getBalance().orElse(null),
                    importedResource.getBalanceUnitKey().orElse(null)
            );
        }
    }

    private void validateImportResource(ImportResourceDto importedResource,
                                        ValidatedDirectories directories,
                                        ErrorCollection.Builder errors,
                                        int resourceIndex,
                                        Locale locale) {
        Optional<String> providerId = importedResource.getProviderId();
        Optional<ProviderModel> providerO = Optional.empty();
        if (providerId.isPresent()) {
            ProviderModel provider = directories.getProviders().get(providerId.get());
            providerO = Optional.ofNullable(provider);
            if (provider == null || provider.isDeleted()) {
                errors.addError("resourceQuotas." + resourceIndex + ".providerId", TypedError.invalid(messages
                        .getMessage("errors.provider.not.found", null, locale)));
            }
        }
        if (providerO.isPresent() && !providerO.get().isImportAllowed()) {
            errors.addError("resourceQuotas." + resourceIndex + ".providerId", TypedError.invalid(messages
                    .getMessage("errors.import.is.not.allowed", null, locale)));
        }
        Optional<String> resourceId = importedResource.getResourceId();
        Optional<ResourceModel> resourceO = Optional.empty();
        if (resourceId.isPresent()) {
            ResourceModel resource = directories.getResources().get(resourceId.get());
            resourceO = Optional.ofNullable(resource);
            if (resource == null || resource.isDeleted()
                    || (providerO.isPresent() && !resource.getProviderId().equals(providerO.get().getId()))) {
                errors.addError("resourceQuotas." + resourceIndex + ".resourceId", TypedError.invalid(messages
                        .getMessage("errors.resource.not.found", null, locale)));
            }
        }
        if (resourceO.isPresent()) {
            UnitsEnsembleModel unitsEnsemble = directories.getUnitsEnsembles()
                    .get(resourceO.get().getUnitsEnsembleId());
            if (unitsEnsemble == null || unitsEnsemble.isDeleted()) {
                errors.addError("resourceQuotas." + resourceIndex, TypedError
                        .invalid(messages.getMessage("errors.units.ensemble.not.found", null, locale)));
            } else {
                Optional<String> balanceUnitKey = importedResource.getBalanceUnitKey();
                if (balanceUnitKey.isPresent()) {
                    Optional<UnitModel> unitO = unitsEnsemble.unitByKey(balanceUnitKey.get());
                    if (unitO.isEmpty()) {
                        errors.addError("resourceQuotas." + resourceIndex + ".balanceUnitKey", TypedError
                                .invalid(messages.getMessage("errors.unit.not.found", null, locale)));
                    }
                }
                Optional<String> quotaUnitKey = importedResource.getQuotaUnitKey();
                if (quotaUnitKey.isPresent()) {
                    Optional<UnitModel> unitO = unitsEnsemble.unitByKey(quotaUnitKey.get());
                    if (unitO.isEmpty()) {
                        errors.addError("resourceQuotas." + resourceIndex + ".quotaUnitKey", TypedError
                                .invalid(messages.getMessage("errors.unit.not.found", null, locale)));
                    }
                }
            }
        }
    }

    private void validateImportAccount(ImportAccountDto importedAccount,
                                       ValidatedDirectories directories,
                                       ErrorCollection.Builder errors,
                                       int accountIndex,
                                       Locale locale,
                                       String targetFolderId,
                                       boolean importToDefaultFolder) {
        Optional<String> externalIdO = importedAccount.getId();
        Optional<String> providerId = importedAccount.getProviderId();
        Optional<ProviderModel> providerO = Optional.empty();
        if (providerId.isPresent()) {
            ProviderModel provider = directories.getProviders().get(providerId.get());
            providerO = Optional.ofNullable(provider);
            if (provider == null || provider.isDeleted()) {
                errors.addError("accounts." + accountIndex + ".providerId", TypedError
                        .invalid(messages.getMessage("errors.provider.not.found", null, locale)));
            }
        }
        if (providerO.isPresent() && !providerO.get().isImportAllowed()) {
            errors.addError("accounts." + accountIndex, TypedError.invalid(messages
                    .getMessage("errors.import.is.not.allowed", null, locale)));
        }
        if (externalIdO.isPresent() && providerId.isPresent()) {
            AccountExternalId externalAccountId = new AccountExternalId(
                    providerId.get(),
                    externalIdO.get(),
                    importedAccount.getAccountSpaceIdentity().map(AccountSpaceIdentity::fromDto).orElse(null)
            );
            AccountModel existingAccount = directories.getAccounts().get(externalAccountId);
            if (existingAccount != null) {
                if (importedAccount.getKey().isPresent() && existingAccount.getOuterAccountKeyInProvider().isPresent()
                        && !Objects.equals(importedAccount.getKey().get(),
                        existingAccount.getOuterAccountKeyInProvider().get())) {
                    errors.addError("accounts." + accountIndex + ".key", TypedError
                            .invalid(messages.getMessage("errors.account.key.is.immutable", null, locale)));
                }
            }
            if (providerO.isPresent()) {
                ProviderModel provider = providerO.get();
                if (importedAccount.getDisplayName().isEmpty()
                        && provider.getAccountsSettings().isDisplayNameSupported()) {
                    errors.addError("accounts." + accountIndex + ".displayName", TypedError.invalid(messages
                            .getMessage("errors.account.display.name.is.required", null, locale)));
                }
                if (importedAccount.getDisplayName().isPresent()
                        && !provider.getAccountsSettings().isDisplayNameSupported()) {
                    errors.addError("accounts." + accountIndex + ".displayName", TypedError.invalid(messages
                            .getMessage("errors.account.display.name.is.not.supported", null, locale)));
                }
                if (importedAccount.getKey().isEmpty()
                        && provider.getAccountsSettings().isKeySupported()) {
                    errors.addError("accounts." + accountIndex + ".key", TypedError.invalid(messages
                            .getMessage("errors.account.key.is.required", null, locale)));
                }
                if (importedAccount.getKey().isPresent()
                        && !provider.getAccountsSettings().isKeySupported()) {
                    errors.addError("accounts." + accountIndex + ".displayName", TypedError.invalid(messages
                            .getMessage("errors.account.key.is.not.supported", null, locale)));
                }
                if (importedAccount.getDeleted().orElse(false)
                        && (!provider.getAccountsSettings().isDeleteSupported()
                        || !provider.getAccountsSettings().isSoftDeleteSupported())) {
                    errors.addError("accounts." + accountIndex + ".deleted", TypedError.invalid(messages
                            .getMessage("errors.account.deletion.is.not.supported", null, locale)));
                }
                if (existingAccount != null) {
                    if (importedAccount.getDisplayName().isPresent() && existingAccount.getDisplayName().isPresent()
                            && !provider.getAccountsSettings().isRenameSupported()
                            && !Objects.equals(importedAccount.getDisplayName().get(),
                            existingAccount.getDisplayName().get())) {
                        errors.addError("accounts." + accountIndex + ".displayName", TypedError.invalid(messages
                                .getMessage("errors.account.rename.is.not.supported", null, locale)));
                    }
                    boolean accountMoved = (targetFolderId != null
                            && !Objects.equals(targetFolderId, existingAccount.getFolderId()))
                            || (importToDefaultFolder && targetFolderId == null);
                    if (!provider.getAccountsSettings().isMoveSupported() && accountMoved) {
                        errors.addError("accounts." + accountIndex, TypedError.invalid(messages
                                .getMessage("errors.account.move.is.not.supported", null, locale)));
                    }
                    if (!importedAccount.getDeleted().orElse(false) && existingAccount.isDeleted()) {
                        errors.addError("accounts." + accountIndex, TypedError.invalid(messages
                                .getMessage("errors.account.can.not.be.restored", null, locale)));
                    }
                }
            }
        }
        if (importedAccount.getProvisions().isPresent()) {
            String accountSpaceId = null;
            if (importedAccount.getAccountSpaceIdentity().isPresent()) {
                AccountSpaceModel accountSpace = directories.getAccountSpaces().get(
                        AccountSpaceIdentity.fromDto(importedAccount.getAccountSpaceIdentity().get())
                );
                if (accountSpace == null) {
                    errors.addError("accounts." + accountIndex + ".accountSpaceIdentity",
                            TypedError.invalid(messages.getMessage("errors.accounts.space.not.found", null, locale)));
                    return;
                }
                accountSpaceId = accountSpace.getId();
            }
            List<ImportAccountProvisionDto> provisions = importedAccount.getProvisions().get();
            for (int i = 0; i < provisions.size(); i++) {
                ImportAccountProvisionDto provision = provisions.get(i);

                if (provision.getResourceId().isEmpty()) {
                    provision = validateImportAccountProvisionIdentity(provision, directories, errors, i, locale);
                }

                validateImportProvision(provision, directories, providerO.orElse(null),
                        errors, accountIndex, i, locale, accountSpaceId);
            }
        }
    }

    private ImportAccountProvisionDto validateImportAccountProvisionIdentity(ImportAccountProvisionDto provision,
                                                                             ValidatedDirectories directories,
                                                                             ErrorCollection.Builder errors,
                                                                             int resourceIndex,
                                                                             Locale locale) {
        ResourceModel resourceModel = directories.getResourcesByComplexKey().get(
                new ResourceComplexKey(provision.getResourceIdentity().get()));
        if (resourceModel == null) {
            errors.addError("accounts." + resourceIndex + ".provisions", TypedError.invalid(messages
                    .getMessage("errors.resource.not.found", null, locale)));
            return provision;
        } else {
            return  new ImportAccountProvisionDto(
                    resourceModel.getId(),
                    null,
                    provision.getProvided().orElse(null),
                    provision.getProvidedUnitKey().orElse(null),
                    provision.getAllocated().orElse(null),
                    provision.getAllocatedUnitKey().orElse(null)
            );
        }
    }

    @SuppressWarnings("ParameterNumber")
    private void validateImportProvision(ImportAccountProvisionDto importedProvision,
                                         ValidatedDirectories directories,
                                         ProviderModel provider,
                                         ErrorCollection.Builder errors,
                                         int accountIndex,
                                         int provisionIndex,
                                         Locale locale,
                                         String accountSpaceId) {
        Optional<String> resourceId = importedProvision.getResourceId();
        Optional<ResourceModel> resourceO = Optional.empty();
        if (resourceId.isPresent()) {
            ResourceModel resource = directories.getResources().get(resourceId.get());
            resourceO = Optional.ofNullable(resource);
            if (resource == null || resource.isDeleted()
                    || (provider != null && !resource.getProviderId().equals(provider.getId()))) {
                errors.addError("accounts." + accountIndex + ".provisions." + provisionIndex + ".resourceId",
                        TypedError.invalid(messages.getMessage("errors.resource.not.found", null, locale)));
            }
        }
        if (resourceO.isPresent()) {
            ResourceModel resourceModel = resourceO.get();
            if (!Objects.equals(accountSpaceId, resourceModel.getAccountsSpacesId())) {
                errors.addError("accounts." + accountIndex + ".provisions." + provisionIndex, TypedError.invalid(
                        messages.getMessage("errors.provided.resource.account.space.not.match.account", null, locale)));
            }
            UnitsEnsembleModel unitsEnsemble = directories.getUnitsEnsembles()
                    .get(resourceModel.getUnitsEnsembleId());
            if (unitsEnsemble == null || unitsEnsemble.isDeleted()) {
                errors.addError("accounts." + accountIndex + ".provisions." + provisionIndex, TypedError
                        .invalid(messages.getMessage("errors.units.ensemble.not.found", null, locale)));
            } else {
                Optional<String> providedUnitKey = importedProvision.getProvidedUnitKey();
                if (providedUnitKey.isPresent()) {
                    Optional<UnitModel> unitO = unitsEnsemble.unitByKey(providedUnitKey.get());
                    if (unitO.isEmpty()) {
                        errors.addError("accounts." + accountIndex + ".provisions." + provisionIndex
                                + ".providedUnitKey", TypedError.invalid(messages
                                .getMessage("errors.unit.not.found", null, locale)));
                    }
                }
                Optional<String> allocatedUnitKey = importedProvision.getAllocatedUnitKey();
                if (allocatedUnitKey.isPresent()) {
                    Optional<UnitModel> unitO = unitsEnsemble.unitByKey(allocatedUnitKey.get());
                    if (unitO.isEmpty()) {
                        errors.addError("accounts." + accountIndex + ".provisions." + provisionIndex
                                + ".allocatedUnitKey", TypedError.invalid(messages
                                .getMessage("errors.unit.not.found", null, locale)));
                    }
                }
            }
        }
    }

    @SuppressWarnings("ParameterNumber")
    private void applyImport(Map<QuotaModel.Key, QuotaImportApplication> appliedQuotas,
                             Map<ProvisionImportKey, ProvisionImportApplication> appliedProvisions,
                             List<ImportFolder> importedQuotas,
                             List<ImportFailureDto> importFailures,
                             ValidatedImportWithQuotas quotasToImport,
                             AccountsMoveInfo accountsMoveInfo,
                             AccountsDeleteInfo deleteInfo,
                             ExistingAccountsCounts accountsCounts,
                             ErrorCollection.Builder globalErrors,
                             Locale locale) {
        Map<String, Set<QuotaModel>> loadedQuotasByDestinationFolder = new HashMap<>();
        quotasToImport.getQuotas().values().forEach(q -> loadedQuotasByDestinationFolder
                .computeIfAbsent(q.getFolderId(), x -> new HashSet<>()).add(q));
        Map<String, Set<AccountsQuotasModel>> loadedProvisionByDestinationFolder = new HashMap<>();
        quotasToImport.getProvisions().values().forEach(p -> loadedProvisionByDestinationFolder
                .computeIfAbsent(p.getFolderId(), x -> new HashSet<>()).add(p));
        boolean hasMoves = !accountsMoveInfo.getMovedAccounts().isEmpty();
        Map<QuotaModel.Key, QuotaImportApplication> allAppliedQuotas = new HashMap<>();
        Map<ProvisionImportKey, ProvisionImportApplication> allAppliedProvisions = new HashMap<>();
        Set<String> destinationFolderIdsWithErrors = new HashSet<>();
        Map<String, Integer> folderIndices = new HashMap<>();
        Map<String, Map<String, Integer>> folderResourceIndices = new HashMap<>();
        quotasToImport.getValidatedImport().getQuotas().forEach(importedFolder -> applyImportForFolder(appliedQuotas,
                appliedProvisions, importedQuotas, importFailures, quotasToImport, deleteInfo, accountsCounts,
                loadedQuotasByDestinationFolder, loadedProvisionByDestinationFolder, allAppliedQuotas,
                allAppliedProvisions, folderIndices, folderResourceIndices, destinationFolderIdsWithErrors,
                importedFolder, hasMoves, locale));
        if (hasMoves) {
            applyImportForMoves(appliedQuotas, appliedProvisions, importedQuotas, quotasToImport, accountsMoveInfo,
                    accountsCounts, allAppliedQuotas, allAppliedProvisions, importFailures,
                    destinationFolderIdsWithErrors, folderIndices, folderResourceIndices, globalErrors, locale);
        }
    }

    @SuppressWarnings("ParameterNumber")
    private void applyImportForMoves(Map<QuotaModel.Key, QuotaImportApplication> appliedQuotas,
                                     Map<ProvisionImportKey, ProvisionImportApplication> appliedProvisions,
                                     List<ImportFolder> importedQuotas,
                                     ValidatedImportWithQuotas quotasToImport,
                                     AccountsMoveInfo accountsMoveInfo,
                                     ExistingAccountsCounts accountsCounts,
                                     Map<QuotaModel.Key, QuotaImportApplication> allAppliedQuotas,
                                     Map<ProvisionImportKey, ProvisionImportApplication> allAppliedProvisions,
                                     List<ImportFailureDto> importFailures,
                                     Set<String> destinationFolderIdsWithErrors,
                                     Map<String, Integer> folderIndices,
                                     Map<String, Map<String, Integer>> folderResourceIndices,
                                     ErrorCollection.Builder globalErrors,
                                     Locale locale) {
        importFailures.forEach(failure -> {
            int index = getFolderIndex(quotasToImport, failure.getFolderId().orElse(null),
                    failure.getServiceId().orElse(null));
            String keyPrefix = "quotas." + index + ".";
            failure.getErrors().forEach(e -> globalErrors.addError(TypedError.invalid(e)));
            failure.getFieldErrors().forEach((k, v) -> v.forEach(t -> globalErrors.addError(keyPrefix + k,
                    TypedError.invalid(t))));
        });
        quotasToImport.getValidatedImport().getQuotas().forEach(importedFolder -> {
            int index = getFolderIndex(quotasToImport, importedFolder.getFolderId().orElse(null),
                    importedFolder.getServiceId().orElse(null));
            String keyPrefix = "quotas." + index + ".";
            String destinationFolderId;
            if (importedFolder.getFolderId().isPresent()) {
                destinationFolderId = importedFolder.getFolderId().get();
            } else {
                FolderModel folder = quotasToImport.getValidatedImport().getDirectories()
                        .getDefaultFolders().get(importedFolder.getServiceId().get());
                if (folder != null) {
                    destinationFolderId = folder.getId();
                } else {
                    destinationFolderId = quotasToImport.getValidatedImport().getPreGeneratedFolderIds()
                            .get(importedFolder.getServiceId().get());
                }
            }
            Map<String, Set<AccountMoveInfo>> accountMoveInfoBySourceFolder = accountsMoveInfo.getMovedAccounts()
                    .values().stream().collect(Collectors.groupingBy(i -> i.getSourceFolder().getId(),
                            Collectors.toSet()));
            validateAccountsCounts(accountsCounts, importedFolder, destinationFolderId, quotasToImport,
                    accountMoveInfoBySourceFolder, keyPrefix, globalErrors, locale);
        });
        Map<ProvisionImportKey, ProvisionImportApplication> delayedAppliedProvisions = new HashMap<>();
        accountsMoveInfo.getMovedAccounts().forEach((k, v) -> {
            String sourceFolderId = v.getSourceFolder().getId();
            String destinationFolderId = v.getDestinationFolder()
                    .map(FolderModel::getId).orElseGet(() -> v.getDestinationFolderToCreateId().get());
            String accountId = v.getAccountId();
            Set<String> resourceIds = v.getAccountResources().stream().map(ResourceModel::getId)
                    .collect(Collectors.toSet());
            v.getSourceProvisions().forEach((i, p) -> {
                ProvisionImportKey destinationKey = new ProvisionImportKey(p.getAccountId(),
                        p.getResourceId(), destinationFolderId);
                ProvisionImportKey sourceKey = new ProvisionImportKey(p.getAccountId(),
                        p.getResourceId(), sourceFolderId);
                if (p.getFolderId().equals(sourceFolderId) && p.getAccountId().equals(accountId)
                        && resourceIds.contains(p.getResourceId())) {
                    ProvisionImportApplication destinationApplication = new ProvisionImportApplication(
                            p.getAccountId(), p.getResourceId(), p.getProviderId(),
                            destinationFolderId, null, null,
                            p.getProvidedQuota(), p.getAllocatedQuota(), false, null, null);
                    if (!allAppliedProvisions.containsKey(destinationKey)) {
                        allAppliedProvisions.put(destinationKey, destinationApplication);
                    }
                    ProvisionImportApplication sourceApplication = new ProvisionImportApplication(p.getAccountId(),
                            p.getResourceId(), p.getProviderId(), sourceFolderId, p.getProvidedQuota(),
                            p.getAllocatedQuota(), 0L, 0L, true,
                            p.getLatestSuccessfulProvisionOperationId().orElse(null),
                            p.getLastReceivedProvisionVersion().orElse(null));
                    if (!allAppliedProvisions.containsKey(sourceKey)) {
                        allAppliedProvisions.put(sourceKey, sourceApplication);
                    }
                } else {
                    ProvisionImportKey key = new ProvisionImportKey(p.getAccountId(),
                            p.getResourceId(), p.getFolderId());
                    ProvisionImportApplication application = new ProvisionImportApplication(p.getAccountId(),
                            p.getResourceId(), p.getProviderId(), p.getFolderId(), p.getProvidedQuota(),
                            p.getAllocatedQuota(), null, null, false,
                            p.getLatestSuccessfulProvisionOperationId().orElse(null),
                            p.getLastReceivedProvisionVersion().orElse(null));
                    delayedAppliedProvisions.put(key, application);
                }
            });
            v.getDestinationProvisions().forEach((i, p) -> {
                ProvisionImportKey key = new ProvisionImportKey(p.getAccountId(),
                        p.getResourceId(), p.getFolderId());
                ProvisionImportApplication application = new ProvisionImportApplication(p.getAccountId(),
                        p.getResourceId(), p.getProviderId(), p.getFolderId(), p.getProvidedQuota(),
                        p.getAllocatedQuota(), null, null, false,
                        p.getLatestSuccessfulProvisionOperationId().orElse(null),
                        p.getLastReceivedProvisionVersion().orElse(null));
                delayedAppliedProvisions.put(key, application);
            });
        });
        delayedAppliedProvisions.forEach((k, v) -> {
            if (!allAppliedProvisions.containsKey(k)) {
                allAppliedProvisions.put(k, v);
            }
        });
        quotasToImport.getProvisions().forEach((k, v) -> {
            ProvisionImportKey key = new ProvisionImportKey(v.getAccountId(),
                    v.getResourceId(), v.getFolderId());
            ProvisionImportApplication application = new ProvisionImportApplication(v.getAccountId(),
                    v.getResourceId(), v.getProviderId(), v.getFolderId(), v.getProvidedQuota(),
                    v.getAllocatedQuota(), null, null, false,
                    v.getLatestSuccessfulProvisionOperationId().orElse(null),
                    v.getLastReceivedProvisionVersion().orElse(null));
            if (!allAppliedProvisions.containsKey(key)) {
                allAppliedProvisions.put(key, application);
            }
        });
        quotasToImport.getQuotas().forEach((k, v) -> {
            QuotaImportApplication application = new QuotaImportApplication(v.getFolderId(),
                    v.getProviderId(), v.getResourceId(), v.getQuota(),
                    v.getBalance(), null, null);
            if (!allAppliedQuotas.containsKey(k)) {
                allAppliedQuotas.put(k, application);
            }
        });
        accountsMoveInfo.getMovedAccounts().forEach((k, v) -> {
            v.getSourceQuotas().forEach((i, q) -> {
                QuotaImportApplication application = new QuotaImportApplication(q.getFolderId(),
                        q.getProviderId(), q.getResourceId(), q.getQuota(),
                        q.getBalance(), null, null);
                if (!allAppliedQuotas.containsKey(i)) {
                    allAppliedQuotas.put(i, application);
                }
            });
            v.getDestinationQuotas().forEach((i, q) -> {
                QuotaImportApplication application = new QuotaImportApplication(q.getFolderId(),
                        q.getProviderId(), q.getResourceId(), q.getQuota(),
                        q.getBalance(), null, null);
                if (!allAppliedQuotas.containsKey(i)) {
                    allAppliedQuotas.put(i, application);
                }
            });
        });
        sumBalances(allAppliedQuotas, allAppliedProvisions);
        if (destinationFolderIdsWithErrors.isEmpty()) {
            validateBalances(allAppliedQuotas, folderIndices, folderResourceIndices, globalErrors, locale);
        }
        if (!globalErrors.hasAnyErrors()) {
            importedQuotas.addAll(quotasToImport.getValidatedImport().getQuotas());
            appliedQuotas.putAll(allAppliedQuotas);
            appliedProvisions.putAll(allAppliedProvisions);
        }
    }

    @SuppressWarnings("ParameterNumber")
    private void applyImportForFolder(
            Map<QuotaModel.Key, QuotaImportApplication> appliedQuotas,
            Map<ProvisionImportKey, ProvisionImportApplication> appliedProvisions,
            List<ImportFolder> importedQuotas,
            List<ImportFailureDto> importFailures,
            ValidatedImportWithQuotas quotasToImport,
            AccountsDeleteInfo deleteInfo,
            ExistingAccountsCounts accountsCounts,
            Map<String, Set<QuotaModel>> loadedQuotasByDestinationFolder,
            Map<String, Set<AccountsQuotasModel>> loadedProvisionByDestinationFolder,
            Map<QuotaModel.Key, QuotaImportApplication> allAppliedQuotas,
            Map<ProvisionImportKey, ProvisionImportApplication> allAppliedProvisions,
            Map<String, Integer> folderIndices,
            Map<String, Map<String, Integer>> folderResourceIndices,
            Set<String> destinationFolderIdsWithErrors,
            ImportFolder importedFolder,
            boolean hasMoves,
            Locale locale) {
        ErrorCollection.Builder errors = ErrorCollection.builder();
        Map<QuotaModel.Key, QuotaImportApplication> folderAppliedQuotas = new HashMap<>();
        Map<ProvisionImportKey, ProvisionImportApplication> folderAppliedProvisions = new HashMap<>();
        String destinationFolderId = null;
        if (importedFolder.getFolderId().isPresent()) {
            destinationFolderId = importedFolder.getFolderId().get();
        } else if (importedFolder.getServiceId().isPresent()) {
            destinationFolderId = Optional.ofNullable(quotasToImport.getValidatedImport().getDirectories()
                    .getDefaultFolders().get(importedFolder.getServiceId().get()))
                    .map(FolderModel::getId).orElse(null);
            if (destinationFolderId == null) {
                destinationFolderId = quotasToImport.getValidatedImport()
                        .getPreGeneratedFolderIds().get(importedFolder.getServiceId().get());
            }
        }
        int folderIndex = getFolderIndex(quotasToImport, importedFolder.getFolderId().orElse(null),
                importedFolder.getServiceId().orElse(null));
        folderIndices.put(destinationFolderId, folderIndex);
        Map<String, Integer> resourceIndices = new HashMap<>();
        applyImportQuotasForFolder(quotasToImport, loadedQuotasByDestinationFolder, allAppliedQuotas,
                folderAppliedQuotas, importedFolder, destinationFolderId, folderResourceIndices, hasMoves,
                resourceIndices, errors, locale);
        applyImportProvisionsForFolder(quotasToImport, deleteInfo, loadedProvisionByDestinationFolder,
                allAppliedProvisions, folderAppliedProvisions, importedFolder, destinationFolderId, hasMoves,
                errors, locale);
        if (!hasMoves) {
            validateAccountsCounts(accountsCounts, importedFolder, destinationFolderId, quotasToImport,
                    errors, locale);
        }
        if (errors.hasAnyErrors()) {
            importFailures.add(toImportFailure(importedFolder.getFolderId().orElse(null),
                    importedFolder.getServiceId().orElse(null), errors.build()));
            destinationFolderIdsWithErrors.add(destinationFolderId);
        } else {
            if (!hasMoves) {
                sumBalances(folderAppliedQuotas, folderAppliedProvisions);
                validateBalances(folderAppliedQuotas, resourceIndices, errors, locale);
                if (errors.hasAnyErrors()) {
                    importFailures.add(toImportFailure(importedFolder.getFolderId().orElse(null),
                            importedFolder.getServiceId().orElse(null), errors.build()));
                    destinationFolderIdsWithErrors.add(destinationFolderId);
                } else {
                    importedQuotas.add(importedFolder);
                    appliedQuotas.putAll(folderAppliedQuotas);
                    appliedProvisions.putAll(folderAppliedProvisions);
                }
            }
        }
    }

    private void validateAccountsCounts(ExistingAccountsCounts accountsCounts, ImportFolder importedFolder,
                                        String destinationFolderId, ValidatedImportWithQuotas quotasToImport,
                                        ErrorCollection.Builder errors, Locale locale) {
        Map<String, Set<ImportAccount>> accountsByProvider = importedFolder.getAccounts().stream()
                .collect(Collectors.groupingBy(ImportAccount::getProviderId, Collectors.toSet()));
        Map<String, Long> totalNewAccountsByProvider = new HashMap<>();
        accountsByProvider.forEach((providerId, accounts) -> {
            long totalNewAccounts = 0L;
            for (ImportAccount account : accounts) {
                AccountExternalId externalId = new AccountExternalId(
                        account.getProviderId(), account.getId(), account.getAccountSpaceId());
                boolean accountExists = quotasToImport.getValidatedImport().getDirectories().getAccounts()
                        .containsKey(externalId);
                boolean accountMoved = accountExists && !quotasToImport.getValidatedImport().getDirectories()
                        .getAccounts().get(externalId).getFolderId().equals(destinationFolderId);
                if (!account.isDeleted() && (!accountExists || accountMoved)) {
                    totalNewAccounts++;
                }
            }
            totalNewAccountsByProvider.put(providerId, totalNewAccounts);
        });
        Map<String, Long> totalExistingAccountsByProvider = accountsCounts.getCounts()
                .getOrDefault(destinationFolderId, Map.of());
        boolean moreAccountsThanAllowed = totalNewAccountsByProvider.entrySet().stream().anyMatch(e -> {
            long existingAccounts = totalExistingAccountsByProvider.getOrDefault(e.getKey(), 0L);
            long newAccounts = e.getValue();
            ProviderModel provider = quotasToImport.getValidatedImport().getDirectories()
                    .getProviders().get(e.getKey());
            return !provider.isMultipleAccountsPerFolder() && existingAccounts + newAccounts > 1;
        });
        if (moreAccountsThanAllowed) {
            errors.addError("accounts", TypedError.invalid(messages.getMessage(
                    "errors.no.more.than.one.account.per.folder.is.allowed.for.the.provider", null, locale)));
        }
    }

    @SuppressWarnings("ParameterNumber")
    private void validateAccountsCounts(ExistingAccountsCounts accountsCounts, ImportFolder importedFolder,
                                        String destinationFolderId, ValidatedImportWithQuotas quotasToImport,
                                        Map<String, Set<AccountMoveInfo>> accountMoveInfoBySourceFolder,
                                        String fieldKeyPrefix, ErrorCollection.Builder errors, Locale locale) {
        Map<String, Set<ImportAccount>> accountsByProvider = importedFolder.getAccounts().stream()
                .collect(Collectors.groupingBy(ImportAccount::getProviderId, Collectors.toSet()));
        Map<String, Long> totalNewAccountsByProvider = new HashMap<>();
        Map<String, Long> totalMovedOutAccountsByProvider = new HashMap<>();
        accountsByProvider.forEach((providerId, accounts) -> {
            long totalNewAccounts = 0L;
            for (ImportAccount account : accounts) {
                AccountExternalId externalId = new AccountExternalId(
                        account.getProviderId(), account.getId(), account.getAccountSpaceId());
                boolean accountExists = quotasToImport.getValidatedImport().getDirectories().getAccounts()
                        .containsKey(externalId);
                boolean accountMoved = accountExists && !quotasToImport.getValidatedImport().getDirectories()
                        .getAccounts().get(externalId).getFolderId().equals(destinationFolderId);
                if (!account.isDeleted() && (!accountExists || accountMoved)) {
                    totalNewAccounts++;
                }
            }
            totalNewAccountsByProvider.put(providerId, totalNewAccounts);
        });
        Set<AccountMoveInfo> movedOutInfos = accountMoveInfoBySourceFolder.getOrDefault(destinationFolderId,
                Collections.emptySet());
        Map<String, Set<AccountModel>> movedOutAccountsByProvider = new HashMap<>();
        movedOutInfos.forEach(info -> {
            AccountModel movedOutAccount = quotasToImport.getAccounts().get(info.getAccountId());
            movedOutAccountsByProvider.computeIfAbsent(movedOutAccount.getProviderId(), x -> new HashSet<>())
                    .add(movedOutAccount);
        });
        movedOutAccountsByProvider.forEach((providerId, set) -> totalMovedOutAccountsByProvider
                .put(providerId, (long) set.size()));
        Map<String, Long> totalExistingAccountsByProvider = accountsCounts.getCounts()
                .getOrDefault(destinationFolderId, Map.of());
        boolean moreAccountsThanAllowed = totalNewAccountsByProvider.entrySet().stream().anyMatch(e -> {
            long existingAccounts = totalExistingAccountsByProvider.getOrDefault(e.getKey(), 0L);
            long movedOutAccounts = totalMovedOutAccountsByProvider.getOrDefault(e.getKey(), 0L);
            long newAccounts = e.getValue();
            ProviderModel provider = quotasToImport.getValidatedImport().getDirectories()
                    .getProviders().get(e.getKey());
            return !provider.isMultipleAccountsPerFolder() && existingAccounts - movedOutAccounts + newAccounts > 1;
        });
        if (moreAccountsThanAllowed) {
            errors.addError(fieldKeyPrefix + ".accounts", TypedError.invalid(messages.getMessage(
                    "errors.no.more.than.one.account.per.folder.is.allowed.for.the.provider", null, locale)));
        }
    }

    @SuppressWarnings("ParameterNumber")
    private void applyImportProvisionsForFolder(
            ValidatedImportWithQuotas quotasToImport,
            AccountsDeleteInfo deleteInfo,
            Map<String, Set<AccountsQuotasModel>> loadedProvisionByDestinationFolder,
            Map<ProvisionImportKey, ProvisionImportApplication> allAppliedProvisions,
            Map<ProvisionImportKey, ProvisionImportApplication> folderAppliedProvisions,
            ImportFolder importedFolder,
            String destinationFolderId,
            boolean hasMoves,
            ErrorCollection.Builder errors,
            Locale locale) {
        for (int i = 0; i < importedFolder.getAccounts().size(); i++) {
            Map<ProvisionImportKey, ProvisionImportApplication> accountAppliedProvisions
                    = new HashMap<>();
            ImportAccount importedAccount = importedFolder.getAccounts().get(i);
            AccountExternalId externalAccountId = new AccountExternalId(
                    importedAccount.getProviderId(), importedAccount.getId(), importedAccount.getAccountSpaceId());
            AccountModel existingAccount = quotasToImport.getValidatedImport().getDirectories()
                    .getAccounts().get(externalAccountId);
            String accountId;
            if (existingAccount != null) {
                accountId = existingAccount.getId();
            } else {
                accountId = quotasToImport.getValidatedImport()
                        .getPreGeneratedAccountIds().get(externalAccountId);
            }
            for (int j = 0; j < importedAccount.getProvisions().size(); j++) {
                ImportAccountProvision importedProvision = importedAccount.getProvisions().get(j);
                AccountsQuotasModel.Identity key = new AccountsQuotasModel.Identity(accountId,
                        importedProvision.getResourceId());
                ProvisionImportKey importKey = new ProvisionImportKey(accountId,
                        importedProvision.getResourceId(), destinationFolderId);
                AccountsQuotasModel existingProvision = quotasToImport.getProvisions().get(key);
                if (existingProvision != null && !existingProvision.getFolderId().equals(destinationFolderId)) {
                    existingProvision = null;
                }
                Long originalProvided = existingProvision != null ? existingProvision.getProvidedQuota() : null;
                Long originalAllocated = existingProvision != null ? existingProvision.getAllocatedQuota() : null;
                String originalLatestSuccessfulProvisionOperationId = existingProvision != null
                        ? existingProvision.getLatestSuccessfulProvisionOperationId().orElse(null) : null;
                Long originalLastReceivedProvisionVersion = existingProvision != null
                        ? existingProvision.getLastReceivedProvisionVersion().orElse(null) : null;
                ResourceModel resource = quotasToImport.getValidatedImport().getDirectories().getResources()
                        .get(importedProvision.getResourceId());
                UnitsEnsembleModel unitsEnsemble = quotasToImport.getValidatedImport().getDirectories()
                        .getUnitsEnsembles().get(resource.getUnitsEnsembleId());
                UnitModel providedUnit = unitsEnsemble.unitByKey(importedProvision.getProvidedUnitKey()).get();
                UnitModel allocatedUnit = unitsEnsemble.unitByKey(importedProvision.getAllocatedUnitKey()).get();
                Optional<Long> importedProvided = Units.convertFromApi(importedProvision.getProvided(),
                        resource, unitsEnsemble, providedUnit);
                Optional<Long> importedAllocated = Units.convertFromApi(importedProvision.getAllocated(),
                        resource, unitsEnsemble, allocatedUnit);
                if (importedProvided.isEmpty()) {
                    errors.addError("accounts." + i + ".provisions." + j + ".provided", TypedError
                            .invalid(messages.getMessage("errors.value.can.not.be.converted.to.base.unit",
                                    null, locale)));
                }
                if (importedAllocated.isEmpty()) {
                    errors.addError("accounts." + i + ".provisions." + j + ".allocated", TypedError
                            .invalid(messages.getMessage("errors.value.can.not.be.converted.to.base.unit",
                                    null, locale)));
                }
                if (importedProvided.isPresent() && importedAllocated.isPresent()) {
                    ProvisionImportApplication application = new ProvisionImportApplication(accountId,
                            resource.getId(), resource.getProviderId(), destinationFolderId, originalProvided,
                            originalAllocated, importedProvided.get(), importedAllocated.get(), false,
                            originalLatestSuccessfulProvisionOperationId, originalLastReceivedProvisionVersion);
                    folderAppliedProvisions.put(importKey, application);
                    accountAppliedProvisions.put(importKey, application);
                    if (hasMoves) {
                        allAppliedProvisions.put(importKey, application);
                    }
                }
            }
            if (importedAccount.isDeleted() && existingAccount != null) {
                validateAccountsDeletion(accountAppliedProvisions, destinationFolderId,
                        deleteInfo.getDeletedAccounts().get(existingAccount.toExternalId()), errors, i, locale);
            }
        }
        loadedProvisionByDestinationFolder.getOrDefault(destinationFolderId, Collections.emptySet()).forEach(p -> {
            ProvisionImportKey key = new ProvisionImportKey(p.getAccountId(), p.getResourceId(), p.getFolderId());
            ProvisionImportApplication application = new ProvisionImportApplication(p.getAccountId(),
                    p.getResourceId(), p.getProviderId(), p.getFolderId(), p.getProvidedQuota(),
                    p.getAllocatedQuota(), null, null, false,
                    p.getLatestSuccessfulProvisionOperationId().orElse(null),
                    p.getLastReceivedProvisionVersion().orElse(null));
            if (!folderAppliedProvisions.containsKey(key)) {
                folderAppliedProvisions.put(key, application);
            }
        });
    }

    @SuppressWarnings("ParameterNumber")
    private void applyImportQuotasForFolder(
            ValidatedImportWithQuotas quotasToImport,
            Map<String, Set<QuotaModel>> loadedQuotasByDestinationFolder,
            Map<QuotaModel.Key, QuotaImportApplication> allAppliedQuotas,
            Map<QuotaModel.Key, QuotaImportApplication> folderAppliedQuotas,
            ImportFolder importedFolder,
            String destinationFolderId,
            Map<String, Map<String, Integer>> folderResourceIndices,
            boolean hasMoves,
            Map<String, Integer> resourceIndices,
            ErrorCollection.Builder errors,
            Locale locale) {
        for (int i = 0; i < importedFolder.getResourceQuotas().size(); i++) {
            ImportResource importedResource = importedFolder.getResourceQuotas().get(i);
            resourceIndices.put(importedResource.getResourceId(), i);
            folderResourceIndices.computeIfAbsent(destinationFolderId, x -> new HashMap<>())
                    .put(importedResource.getResourceId(), i);
            QuotaModel.Key key = new QuotaModel.Key(destinationFolderId, importedResource.getProviderId(),
                    importedResource.getResourceId());
            QuotaModel existingQuota = quotasToImport.getQuotas().get(key);
            Long originalQuota = existingQuota != null ? existingQuota.getQuota() : null;
            Long originalBalance = existingQuota != null ? existingQuota.getBalance() : null;
            ResourceModel resource = quotasToImport.getValidatedImport().getDirectories().getResources()
                    .get(importedResource.getResourceId());
            UnitsEnsembleModel unitsEnsemble = quotasToImport.getValidatedImport().getDirectories()
                    .getUnitsEnsembles().get(resource.getUnitsEnsembleId());
            UnitModel quotaUnit = unitsEnsemble.unitByKey(importedResource.getQuotaUnitKey()).get();
            UnitModel balanceUnit = unitsEnsemble.unitByKey(importedResource.getBalanceUnitKey()).get();
            Optional<Long> importedQuota = Units.convertFromApi(importedResource.getQuota(), resource,
                    unitsEnsemble, quotaUnit);
            Optional<Long> importedBalance = Units.convertFromApi(importedResource.getBalance(), resource,
                    unitsEnsemble, balanceUnit);
            if (importedQuota.isEmpty()) {
                errors.addError("resourceQuotas." + i + ".quota", TypedError.invalid(messages
                        .getMessage("errors.value.can.not.be.converted.to.base.unit", null, locale)));
            }
            if (importedBalance.isEmpty()) {
                errors.addError("resourceQuotas." + i + ".balance", TypedError.invalid(messages
                        .getMessage("errors.value.can.not.be.converted.to.base.unit", null, locale)));
            }
            if (importedQuota.isPresent() && importedBalance.isPresent()) {
                QuotaImportApplication application = new QuotaImportApplication(destinationFolderId,
                        importedResource.getProviderId(), importedResource.getResourceId(), originalQuota,
                        originalBalance, importedQuota.get(), importedBalance.get());
                folderAppliedQuotas.put(key, application);
                if (hasMoves) {
                    allAppliedQuotas.put(key, application);
                }
            }
        }
        loadedQuotasByDestinationFolder.getOrDefault(destinationFolderId, Collections.emptySet()).forEach(q -> {
            QuotaImportApplication application = new QuotaImportApplication(q.getFolderId(),
                    q.getProviderId(), q.getResourceId(), q.getQuota(),
                    q.getBalance(), null, null);
            QuotaModel.Key key = q.toKey();
            if (!folderAppliedQuotas.containsKey(key)) {
                folderAppliedQuotas.put(key, application);
            }
        });
    }

    private int getFolderIndex(ValidatedImportWithQuotas quotasToImport, String folderId, Long serviceId) {
        int index = -1;
        if (folderId != null) {
            index = quotasToImport.getValidatedImport().getFolderIndices().getOrDefault(folderId, -1);
        }
        if (serviceId != null) {
            index = quotasToImport.getValidatedImport().getServiceIndices().getOrDefault(serviceId, -1);
        }
        return index;
    }

    private void sumBalances(Map<QuotaModel.Key, QuotaImportApplication> folderAppliedQuotas,
                             Map<ProvisionImportKey, ProvisionImportApplication> folderAppliedProvisions) {
        folderAppliedQuotas.forEach((k, v) -> v.setActualBalance(BigDecimal.valueOf(v.getImportedQuota()
                .orElseGet(() -> v.getOriginalQuota().orElse(0L)))));
        folderAppliedProvisions.forEach((k, v) -> {
            QuotaModel.Key quotaKey = new QuotaModel.Key(v.getFolderId(), v.getProviderId(), v.getResourceId());
            QuotaImportApplication quotaApplication = folderAppliedQuotas.computeIfAbsent(quotaKey,
                    x -> emptyApplication(x.getFolderId(), x.getProviderId(), x.getResourceId()));
            quotaApplication.setActualBalance(quotaApplication.getActualBalance().subtract(BigDecimal.valueOf(v
                    .getImportedProvidedQuota().orElseGet(() -> v.getOriginalProvidedQuota().orElse(0L)))));
        });
    }

    private QuotaImportApplication emptyApplication(String folderId, String providerId, String resourceId) {
        return new QuotaImportApplication(folderId, providerId, resourceId, null, null, null, null);
    }

    private void validateBalances(Map<QuotaModel.Key, QuotaImportApplication> folderAppliedQuotas,
                                  Map<String, Integer> resourceIndices,
                                  ErrorCollection.Builder errors,
                                  Locale locale) {
        folderAppliedQuotas.forEach((k, v) -> {
            int resourceIndex = resourceIndices.getOrDefault(k.getResourceId(), -1);
            if (v.getActualBalance().compareTo(BigDecimal.ZERO) < 0) {
                String fieldKey = resourceIndex >= 0 ? "resourceQuotas." + resourceIndex : "resourceQuotas";
                errors.addError(fieldKey, TypedError.invalid(messages
                        .getMessage("errors.negative.balance.is.not.allowed", null, locale)));
            }
            if (v.getImportedBalance().isPresent() && BigDecimal.valueOf(v.getImportedBalance().get())
                    .compareTo(v.getActualBalance()) != 0) {
                String fieldKey = resourceIndex >= 0
                        ? "resourceQuotas." + resourceIndex + ".balance" : "resourceQuotas";
                errors.addError(fieldKey, TypedError.invalid(messages
                        .getMessage("errors.imported.balance.does.not.match.with.actual.balance", null, locale)));
            }
        });
    }

    private void validateBalances(Map<QuotaModel.Key, QuotaImportApplication> folderAppliedQuotas,
                                  Map<String, Integer> folderIndices,
                                  Map<String, Map<String, Integer>> folderResourceIndices,
                                  ErrorCollection.Builder errors,
                                  Locale locale) {
        folderAppliedQuotas.forEach((k, v) -> {
            int folderIndex = folderIndices.getOrDefault(v.getFolderId(), -1);
            int resourceIndex = folderResourceIndices.getOrDefault(v.getFolderId(), Collections.emptyMap())
                    .getOrDefault(v.getResourceId(), -1);
            if (v.getActualBalance().compareTo(BigDecimal.ZERO) < 0) {
                String fieldKey = resourceIndex >= 0
                        ? "quotas." + folderIndex + ".resourceQuotas." + resourceIndex
                        : "quotas." + folderIndex + ".resourceQuotas";
                errors.addError(fieldKey, TypedError.invalid(messages
                        .getMessage("errors.negative.balance.is.not.allowed", null, locale)));
            }
            if (v.getImportedBalance().isPresent() && BigDecimal.valueOf(v.getImportedBalance().get())
                    .compareTo(v.getActualBalance()) != 0) {
                String fieldKey = resourceIndex >= 0
                        ? "quotas." + folderIndex + ".resourceQuotas." + resourceIndex + ".balance"
                        : "quotas." + folderIndex + ".resourceQuotas";
                errors.addError(fieldKey, TypedError.invalid(messages.getMessage(
                        "errors.imported.balance.does.not.match.with.actual.balance", null, locale)));
            }
        });
    }

    private void validateAccountsDeletion(
            Map<ProvisionImportKey, ProvisionImportApplication> accountAppliedProvisions,
            String destinationFolderId,
            AccountDeleteInfo deleteInfo,
            ErrorCollection.Builder errors,
            int accountIndex,
            Locale locale) {
        deleteInfo.getProvisions().forEach((k, v) -> {
            ProvisionImportKey importKey = new ProvisionImportKey(v.getAccountId(), v.getResourceId(),
                    destinationFolderId);
            if (!accountAppliedProvisions.containsKey(importKey)) {
                ProvisionImportApplication application = new ProvisionImportApplication(v.getAccountId(),
                        v.getResourceId(), v.getProviderId(), v.getFolderId(), v.getProvidedQuota(),
                        v.getAllocatedQuota(), null, null, false,
                        v.getLatestSuccessfulProvisionOperationId().orElse(null),
                        v.getLastReceivedProvisionVersion().orElse(null));
                accountAppliedProvisions.put(importKey, application);
            }
        });
        boolean hasNonZero = accountAppliedProvisions.values().stream().anyMatch(v -> {
            long provided = v.getImportedProvidedQuota().orElseGet(() -> v.getOriginalProvidedQuota().orElse(0L));
            long allocated = v.getImportedAllocatedQuota().orElseGet(() -> v.getOriginalAllocatedQuota().orElse(0L));
            return provided != 0L || allocated != 0L;
        });
        if (hasNonZero) {
            errors.addError("accounts." + accountIndex, TypedError.invalid(messages.getMessage(
                    "errors.deleted.accounts.may.not.have.non.zero.provisions.or.allocations", null, locale)));
        }
    }
}
