package ru.yandex.intranet.d.services.integration.providers;

import org.slf4j.Logger;

import ru.yandex.intranet.d.model.TenantId;

/**
 * OperationLog.
 *
 * @author Petr Surkov <petrsurkov@yandex-team.ru>
 */
public class OperationLog<Req, Res> {
    private final String name;
    private final TenantId tenantId;
    private final String operationId;
    private final String requestId;
    private final Req request;
    private final Response<Res> response;
    private final boolean logRequest;
    private final boolean logSuccessResponse;

    @SuppressWarnings("ParameterNumber")
    public OperationLog(
            String name, TenantId tenantId, String operationId, String requestId,
            Req request, Response<Res> response, boolean logRequest, boolean logSuccessResponse) {
        this.name = name;
        this.tenantId = tenantId;
        this.operationId = operationId;
        this.requestId = requestId;
        this.request = request;
        this.response = response;
        this.logRequest = logRequest;
        this.logSuccessResponse = logSuccessResponse;
    }

    public String getName() {
        return this.name;
    }

    public TenantId getTenantId() {
        return this.tenantId;
    }

    public String getOperationId() {
        return this.operationId;
    }

    public String getRequestId() {
        return this.requestId;
    }

    public Req getRequest() {
        return this.request;
    }

    public Response<Res> getResponse() {
        return this.response;
    }

    public boolean isLogRequest() {
        return this.logRequest;
    }

    public boolean isLogSuccessResponse() {
        return this.logSuccessResponse;
    }

    public void writeTo(Logger logger) {
        logger.info("Provider request \"{}\", opId: {}, requestId: {}",
                name, operationId, requestId);
        if (logRequest) {
            logger.info("opId: {}, requestId: {}, request: {}",
                    operationId, requestId, request);
        }
        if (response == null) {
            return;
        }
        response.match(new Response.Cases<Res, Void>() {
            @Override
            public Void success(Res result, String requestId) {
                logger.info("Success on provider \"{}\", opId: {}, requestId: {}",
                        name, operationId, requestId);
                if (logSuccessResponse) {
                    logger.info("Success on provider \"{}\", opId: {}, requestId: {}, result: {}",
                            name, operationId, requestId, result);
                }
                return null;
            }

            @Override
            public Void failure(Throwable error) {
                logger.error("Failure on provider \"{}\", opId: {}",
                        name, operationId);
                logger.error("Failure on provider \"{}\", opId: {}, error: {}",
                        name, operationId, error);
                return null;
            }

            @Override
            public Void error(ProviderError error, String requestId) {
                logger.error("Failure on provider \"{}\", opId: {}, requestId: {}",
                        name, operationId, requestId);
                logger.error("Failure on provider \"{}\", opId: {}, requestId: {}, error: {}",
                        name, operationId, requestId, error);
                return null;
            }
        });
    }

    public static class RequestStage<Req> {
        private final String name;
        private final TenantId tenantId;
        private final String operationId;
        private final Req request;
        private final boolean logRequest;
        private final boolean logSuccessResponse;

        public RequestStage(String name,
                            TenantId tenantId,
                            String operationId,
                            Req request,
                            boolean logRequest,
                            boolean logSuccessResponse) { // safe initialization
            this.name = name;
            this.tenantId = tenantId;
            this.operationId = operationId;
            this.request = request;
            this.logRequest = logRequest;
            this.logSuccessResponse = logSuccessResponse;
        }
    }

    public static class ResponseStage<Req, Res> {
        private final RequestStage<Req> requestStage;
        private final String requestId;
        private final Response<Res> response;

        public ResponseStage(RequestStage<Req> requestStage,
                             String requestId,
                             Response<Res> response) { // safe initialization
            this.requestStage = requestStage;
            this.requestId = requestId;
            this.response = response;
        }

        public OperationLog<Req, Res> buildLog() {
            return new OperationLog<>(requestStage.name,
                    requestStage.tenantId,
                    requestStage.operationId,
                    requestId,
                    requestStage.request,
                    response,
                    requestStage.logRequest,
                    requestStage.logSuccessResponse);
        }
    }
}
