package ru.yandex.intranet.d.services.integration.providers;

import java.util.Map;
import java.util.Objects;

import com.google.rpc.Code;
import io.grpc.Status;
import org.springframework.http.HttpStatus;

import ru.yandex.intranet.d.services.integration.providers.rest.model.ErrorMessagesDto;

/**
 * Provider error.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public interface ProviderError {

    static ProviderError httpError(int statusCode) {
        return new ProviderError() {
            @Override
            public boolean isRetryable() {
                return ProviderError.isRetryableCode(statusCode);
            }

            @Override
            public boolean isConflict() {
                return ProviderError.isConflictCode(statusCode);
            }

            @Override
            public <R> R match(Cases<R> cases) {
                return cases.httpError(statusCode);
            }

            @Override
            public int hashCode() {
                return Objects.hash(statusCode);
            }

            @Override
            public boolean equals(Object obj) {
                if (this == obj) {
                    return true;
                }
                if (!(obj instanceof ProviderError)) {
                    return false;
                }
                final ProviderError error = (ProviderError) obj;
                return error.match(new Cases<>() {
                    @Override
                    public Boolean httpError(int otherStatusCode) {
                        return statusCode == otherStatusCode;
                    }

                    @Override
                    public Boolean httpExtendedError(int otherStatusCode, ErrorMessagesDto otherErrors) {
                        return false;
                    }

                    @Override
                    public Boolean grpcError(Status.Code otherStatusCode, String otherMessage) {
                        return false;
                    }

                    @Override
                    public Boolean grpcExtendedError(Status.Code otherStatusCode, String otherMessage,
                                                     Map<String, String> otherBadRequestDetails) {
                        return false;
                    }
                });
            }

            @Override
            public String toString() {
                return "ProviderError::httpError{"
                        + "statusCode=" + statusCode
                        + '}';
            }

        };
    }

    static ProviderError httpExtendedError(int statusCode, ErrorMessagesDto errors) {
        return new ProviderError() {
            @Override
            public boolean isRetryable() {
                return ProviderError.isRetryableCode(statusCode);
            }

            @Override
            public boolean isConflict() {
                return ProviderError.isConflictCode(statusCode);
            }

            @Override
            public <R> R match(Cases<R> cases) {
                return cases.httpExtendedError(statusCode, errors);
            }

            @Override
            public int hashCode() {
                return Objects.hash(statusCode, errors);
            }

            @Override
            public boolean equals(Object obj) {
                if (this == obj) {
                    return true;
                }
                if (!(obj instanceof ProviderError)) {
                    return false;
                }
                final ProviderError error = (ProviderError) obj;
                return error.match(new Cases<>() {
                    @Override
                    public Boolean httpError(int otherStatusCode) {
                        return false;
                    }

                    @Override
                    public Boolean httpExtendedError(int otherStatusCode, ErrorMessagesDto otherErrors) {
                        return statusCode == otherStatusCode && Objects.equals(errors, otherErrors);
                    }

                    @Override
                    public Boolean grpcError(Status.Code otherStatusCode, String otherMessage) {
                        return false;
                    }

                    @Override
                    public Boolean grpcExtendedError(Status.Code otherStatusCode, String otherMessage,
                                                     Map<String, String> otherBadRequestDetails) {
                        return false;
                    }
                });
            }

            @Override
            public String toString() {
                return "ProviderError::httpExtendedError{"
                        + "statusCode=" + statusCode
                        + ", errors=" + errors
                        + '}';
            }

        };
    }

    static ProviderError grpcError(Status.Code statusCode, String message) {
        return new ProviderError() {
            @Override
            public boolean isRetryable() {
                return ProviderError.isRetryableCode(statusCode);
            }

            @Override
            public boolean isConflict() {
                return ProviderError.isConflictCode(statusCode);
            }

            @Override
            public <R> R match(Cases<R> cases) {
                return cases.grpcError(statusCode, message);
            }

            @Override
            public int hashCode() {
                return Objects.hash(statusCode, message);
            }

            @Override
            public boolean equals(Object obj) {
                if (this == obj) {
                    return true;
                }
                if (!(obj instanceof ProviderError)) {
                    return false;
                }
                final ProviderError error = (ProviderError) obj;
                return error.match(new Cases<>() {
                    @Override
                    public Boolean httpError(int otherStatusCode) {
                        return false;
                    }

                    @Override
                    public Boolean httpExtendedError(int otherStatusCode, ErrorMessagesDto otherErrors) {
                        return false;
                    }

                    @Override
                    public Boolean grpcError(Status.Code otherStatusCode, String otherMessage) {
                        return Objects.equals(statusCode, otherStatusCode) && Objects.equals(message, otherMessage);
                    }

                    @Override
                    public Boolean grpcExtendedError(Status.Code otherStatusCode, String otherMessage,
                                                     Map<String, String> otherBadRequestDetails) {
                        return false;
                    }
                });
            }

            @Override
            public String toString() {
                return "ProviderError::grpcError{"
                        + "statusCode=" + statusCode
                        + ", message='" + message + '\''
                        + '}';
            }

        };
    }

    static ProviderError grpcExtendedError(Status.Code statusCode, String message,
                                           Map<String, String> badRequestDetails) {
        return new ProviderError() {
            @Override
            public boolean isRetryable() {
                return ProviderError.isRetryableCode(statusCode);
            }

            @Override
            public boolean isConflict() {
                return ProviderError.isConflictCode(statusCode);
            }

            @Override
            public <R> R match(Cases<R> cases) {
                return cases.grpcExtendedError(statusCode, message, badRequestDetails);
            }

            @Override
            public int hashCode() {
                return Objects.hash(statusCode, message, badRequestDetails);
            }

            @Override
            public boolean equals(Object obj) {
                if (this == obj) {
                    return true;
                }
                if (!(obj instanceof ProviderError)) {
                    return false;
                }
                final ProviderError error = (ProviderError) obj;
                return error.match(new Cases<>() {
                    @Override
                    public Boolean httpError(int otherStatusCode) {
                        return false;
                    }

                    @Override
                    public Boolean httpExtendedError(int otherStatusCode, ErrorMessagesDto otherErrors) {
                        return false;
                    }

                    @Override
                    public Boolean grpcError(Status.Code otherStatusCode, String otherMessage) {
                        return false;
                    }

                    @Override
                    public Boolean grpcExtendedError(Status.Code otherStatusCode, String otherMessage,
                                                     Map<String, String> otherBadRequestDetails) {
                        return Objects.equals(statusCode, otherStatusCode) && Objects.equals(message, otherMessage)
                                && Objects.equals(badRequestDetails, otherBadRequestDetails);
                    }
                });
            }

            @Override
            public String toString() {
                return "ProviderError::grpcExtendedError{"
                        + "statusCode=" + statusCode
                        + ", message='" + message + '\''
                        + ", badRequestDetails=" + badRequestDetails
                        + '}';
            }

        };
    }

    boolean isRetryable();

    boolean isConflict();

    <R> R match(Cases<R> cases);

    interface Cases<O> {

        O httpError(int statusCode);

        O httpExtendedError(int statusCode, ErrorMessagesDto errors);

        O grpcError(Status.Code statusCode, String message);

        O grpcExtendedError(Status.Code statusCode, String message, Map<String, String> badRequestDetails);

    }

    static boolean isRetryableCode(int statusCode) {
        return statusCode == HttpStatus.TOO_MANY_REQUESTS.value()
                || statusCode == HttpStatus.INTERNAL_SERVER_ERROR.value()
                || statusCode == HttpStatus.BAD_GATEWAY.value()
                || statusCode == HttpStatus.SERVICE_UNAVAILABLE.value()
                || statusCode == HttpStatus.GATEWAY_TIMEOUT.value();
    }

    static boolean isRetryableCode(Code statusCode) {
        return statusCode == Code.INTERNAL || statusCode == Code.RESOURCE_EXHAUSTED
                || statusCode == Code.ABORTED || statusCode == Code.UNAVAILABLE
                || statusCode == Code.DEADLINE_EXCEEDED;
    }

    static boolean isRetryableCode(Status.Code statusCode) {
        return statusCode == Status.Code.INTERNAL
                || statusCode == Status.Code.RESOURCE_EXHAUSTED || statusCode == Status.Code.ABORTED
                || statusCode == Status.Code.UNAVAILABLE || statusCode == Status.Code.DEADLINE_EXCEEDED;
    }

    static boolean isConflictCode(int statusCode) {
        return statusCode == HttpStatus.CONFLICT.value() || statusCode == HttpStatus.PRECONDITION_FAILED.value();
    }

    static boolean isConflictCode(Code statusCode) {
        return statusCode == Code.ALREADY_EXISTS || statusCode == Code.FAILED_PRECONDITION;
    }

    static boolean isConflictCode(Status.Code statusCode) {
        return statusCode == Status.Code.ALREADY_EXISTS || statusCode == Status.Code.FAILED_PRECONDITION;
    }

}
