package ru.yandex.intranet.d.services.integration.providers

import mu.KotlinLogging
import ru.yandex.intranet.d.kotlin.AccountId
import ru.yandex.intranet.d.kotlin.ResourceId
import ru.yandex.intranet.d.model.accounts.AccountModel
import ru.yandex.intranet.d.model.accounts.AccountsQuotasModel
import ru.yandex.intranet.d.model.accounts.AccountsQuotasOperationsModel
import ru.yandex.intranet.d.model.folders.FolderModel
import ru.yandex.intranet.d.model.providers.ProviderModel
import ru.yandex.intranet.d.services.operations.model.ValidatedReceivedAccount

private val logger = KotlinLogging.logger {}

fun isProvisionChangesApplied(receivedAccount: ValidatedReceivedAccount,
                              operation: AccountsQuotasOperationsModel,
                              provider: ProviderModel,
                              account: AccountModel,
                              folder: FolderModel,
                              folderProvisions: Map<AccountId, Map<ResourceId, AccountsQuotasModel>>): Boolean {
    if (receivedAccount.account.isEmpty) {
        logger.error { "Received account $receivedAccount can't be matched with any known account" }
        return false
    }
    if (receivedAccount.account.get().id != account.id) {
        logger.error { "Received account $receivedAccount is different from requested account ${account.id}" }
        return false
    }
    if (receivedAccount.folder.id != folder.id || receivedAccount.folder.id != receivedAccount.account.get().folderId) {
        logger.warn { "Reserve account ${receivedAccount.account.get().id} was moved unexpectedly, " +
            "received folder is ${receivedAccount.folder.id}" }
        return false
    }
    val accountsSettings = provider.accountsSettings
    val provisionOperationIdSupported = accountsSettings.isPerProvisionLastUpdateSupported
    val accountAndProvisionsVersionedTogether = accountsSettings.isPerAccountVersionSupported
        && !accountsSettings.isPerProvisionVersionSupported
    val provisionsVersionedSeparately = accountsSettings.isPerProvisionVersionSupported
    val hasProvisionOperationIds = receivedAccount.provisions
        .all { it.lastUpdate.isPresent && it.lastUpdate.get().operationId.isPresent }
    val hasCommonVersion = receivedAccount.accountVersion.isPresent
    val hasProvisionVersions = receivedAccount.provisions.all { it.quotaVersion.isPresent }
    if (provisionOperationIdSupported && hasProvisionOperationIds) {
        return hasOperationId(receivedAccount, operation, account, folder, folderProvisions,
            accountAndProvisionsVersionedTogether, provisionsVersionedSeparately)
    }
    if (accountAndProvisionsVersionedTogether && hasCommonVersion) {
        return isExactValuesMatchAndFreshCommonVersion(receivedAccount, operation, folder, account, folderProvisions)
    }
    return if (provisionsVersionedSeparately && hasProvisionVersions) {
        isExactValuesMatchAndFreshProvisionVersions(receivedAccount, operation, folder, account, folderProvisions)
    } else {
        isExactValuesMatch(receivedAccount, operation, folder, account)
    }
}

private fun isExactValuesMatch(receivedAccount: ValidatedReceivedAccount,
                               operation: AccountsQuotasOperationsModel,
                               folder: FolderModel,
                               account: AccountModel): Boolean {
    val targetProvisions = operation.requestedChanges.updatedProvisions.orElse(emptyList())
    val receivedProvisionsByResourceId = receivedAccount.provisions.associateBy { it.resource.id }
    val valuesMatch = targetProvisions.all { targetProvision ->
        val targetResourceId = targetProvision.resourceId
        val targetAmount = targetProvision.amount
        val receivedProvision = receivedProvisionsByResourceId[targetResourceId]
        val receivedAmount = receivedProvision?.providedAmount ?: 0L
        targetAmount == receivedAmount
    }
    val knownFolderId = folder.id
    val receivedFolderId = receivedAccount.folder.id
    val foldersMatch = knownFolderId == receivedFolderId
    val knownDeletion = account.isDeleted
    val receivedDeletion = receivedAccount.isDeleted
    val deletionMatch = knownDeletion == receivedDeletion
    return valuesMatch && foldersMatch && deletionMatch
}

private fun isExactValuesMatchAndFreshCommonVersion(
    receivedAccount: ValidatedReceivedAccount,
    operation: AccountsQuotasOperationsModel,
    folder: FolderModel,
    account: AccountModel,
    folderProvisions: Map<AccountId, Map<ResourceId, AccountsQuotasModel>>
): Boolean {
    val receivedVersionO = receivedAccount.accountVersion
    val knownVersionO = account.lastReceivedVersion
    if (receivedVersionO.isPresent && knownVersionO.isPresent && receivedVersionO.get() <= knownVersionO.get()) {
        return false
    }
    return if (isMismatchForRemainingResources(receivedAccount, operation, account, folderProvisions)) {
        false
    } else {
        isExactValuesMatch(receivedAccount, operation, folder, account)
    }
}

private fun isMismatchForRemainingResources(receivedAccount: ValidatedReceivedAccount,
                                            operation: AccountsQuotasOperationsModel,
                                            account: AccountModel,
                                            folderProvisions: Map<AccountId, Map<ResourceId, AccountsQuotasModel>>
): Boolean {
    val receivedProvisionsByResourceId = receivedAccount.provisions.associateBy { it.resource.id }
    val targetResourceIds = operation.requestedChanges.updatedProvisions
        .orElse(emptyList()).map { it.resourceId }.toSet()
    val receivedResourceIds = receivedProvisionsByResourceId.keys
    val knowResourceIds = (folderProvisions[account.id] ?: emptyMap()).keys
    val remainingResourceIds = knowResourceIds.union(receivedResourceIds).subtract(targetResourceIds)
    return remainingResourceIds.any {
        val receivedProvision = receivedProvisionsByResourceId[it]
        val knownProvision = (folderProvisions[account.id] ?: emptyMap())[it]
        val receivedAmount = receivedProvision?.providedAmount ?: 0L
        val knownAmount = knownProvision?.providedQuota ?: 0L
        receivedAmount != knownAmount
    }
}

private fun isExactValuesMatchAndFreshProvisionVersions(
    receivedAccount: ValidatedReceivedAccount,
    operation: AccountsQuotasOperationsModel,
    folder: FolderModel,
    account: AccountModel,
    folderProvisions: Map<AccountId, Map<ResourceId, AccountsQuotasModel>>
): Boolean {
    val knownProvisions = folderProvisions[account.id] ?: emptyMap()
    val receivedVersionsAreFresh = receivedAccount.provisions.all {receivedProvision ->
        val receivedVersion = receivedProvision.quotaVersion.orElse(null)
        val knownProvision = knownProvisions[receivedProvision.resource.id]
        val knownVersion = knownProvision?.lastReceivedProvisionVersion?.orElse(null)
        !(receivedVersion != null && knownVersion != null && receivedVersion <= knownVersion)
    }
    return if (!receivedVersionsAreFresh) {
        false
    } else {
        isExactValuesMatch(receivedAccount, operation, folder, account)
    }
}

private fun hasOperationId(receivedAccount: ValidatedReceivedAccount,
                           operation: AccountsQuotasOperationsModel,
                           account: AccountModel,
                           folder: FolderModel,
                           folderProvisions: Map<AccountId, Map<ResourceId, AccountsQuotasModel>>,
                           accountAndProvisionsVersionedTogether: Boolean,
                           provisionsVersionedSeparately: Boolean): Boolean {
    if (accountAndProvisionsVersionedTogether) {
        val knownVersionO = account.lastReceivedVersion
        val receivedVersionO = receivedAccount.accountVersion
        if (knownVersionO.isPresent && receivedVersionO.isPresent) {
            if (knownVersionO.get() >= receivedVersionO.get()) {
                return false
            }
        }
        if (isMismatchForRemainingResources(receivedAccount, operation, account, folderProvisions)) {
            return false
        }
    }
    val targetProvisionByResourceId = operation.requestedChanges.updatedProvisions.orElse(emptyList())
        .associateBy { it.resourceId }
    val targetResourceIds = targetProvisionByResourceId.keys
    val receivedProvisionsByResourceId = receivedAccount.provisions.associateBy { it.resource.id }
    if (provisionsVersionedSeparately) {
        val versionMatch = targetResourceIds.all {
            val knownProvision = (folderProvisions[account.id] ?: emptyMap())[it]
            val receivedProvision = receivedProvisionsByResourceId[it]
            val knownVersion = knownProvision?.lastReceivedProvisionVersion?.orElse(null)
            val receivedVersion = receivedProvision?.quotaVersion?.orElse(null)
            if (knownVersion != null && receivedVersion != null) {
                knownVersion < receivedVersion
            } else {
                true
            }
        }
        if (!versionMatch) {
            return false
        }
    }
    val knownFolderId = folder.id
    val receivedFolderId = receivedAccount.folder.id
    val foldersMatch = knownFolderId == receivedFolderId
    val knownDeletion = account.isDeleted
    val receivedDeletion = receivedAccount.isDeleted
    val deletionMatch = knownDeletion == receivedDeletion
    val targetOperationId = operation.operationId
    val opIdMatch = targetResourceIds.all { resourceId ->
        val receivedProvision = receivedProvisionsByResourceId[resourceId]
        val targetProvision = targetProvisionByResourceId[resourceId]!!
        if (receivedProvision == null) {
            targetProvision.amount == 0L
        } else {
            receivedProvision.lastUpdate.flatMap { it.operationId }
                .map { targetOperationId.equals(it) }.orElse(false)
        }
    }
    return opIdMatch && foldersMatch && deletionMatch
}
