package ru.yandex.intranet.d.services.integration.providers

import org.springframework.context.MessageSource
import ru.yandex.intranet.d.kotlin.FolderId
import ru.yandex.intranet.d.kotlin.ResourceId
import ru.yandex.intranet.d.kotlin.ResourceTypeId
import ru.yandex.intranet.d.kotlin.SegmentId
import ru.yandex.intranet.d.kotlin.SegmentationId
import ru.yandex.intranet.d.kotlin.UnitsEnsembleId
import ru.yandex.intranet.d.model.accounts.AccountModel
import ru.yandex.intranet.d.model.accounts.AccountSpaceModel
import ru.yandex.intranet.d.model.accounts.AccountsQuotasOperationsModel
import ru.yandex.intranet.d.model.folders.FolderModel
import ru.yandex.intranet.d.model.providers.ProviderModel
import ru.yandex.intranet.d.model.resources.ResourceModel
import ru.yandex.intranet.d.model.resources.segmentations.ResourceSegmentationModel
import ru.yandex.intranet.d.model.resources.segments.ResourceSegmentModel
import ru.yandex.intranet.d.model.resources.types.ResourceTypeModel
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel
import ru.yandex.intranet.d.model.users.UserModel
import ru.yandex.intranet.d.services.integration.providers.rest.model.AccountDto
import ru.yandex.intranet.d.services.integration.providers.rest.model.AccountsSpaceKeyResponseDto
import ru.yandex.intranet.d.services.integration.providers.rest.model.LastUpdateDto
import ru.yandex.intranet.d.services.integration.providers.rest.model.ProvisionDto
import ru.yandex.intranet.d.services.integration.providers.rest.model.ResourceKeyResponseDto
import ru.yandex.intranet.d.services.integration.providers.rest.model.SegmentKeyResponseDto
import ru.yandex.intranet.d.services.integration.providers.rest.model.UpdateProvisionResponseDto
import ru.yandex.intranet.d.services.integration.providers.rest.model.UserIdDto
import ru.yandex.intranet.d.services.operations.model.ExpandedResource
import ru.yandex.intranet.d.services.operations.model.ExpandedSegment
import ru.yandex.intranet.d.services.operations.model.ExternalAccountsSpaceKey
import ru.yandex.intranet.d.services.operations.model.ExternalResourceKey
import ru.yandex.intranet.d.services.operations.model.ExternalSegmentKey
import ru.yandex.intranet.d.services.operations.model.ReceivedAccount
import ru.yandex.intranet.d.services.operations.model.ReceivedAccountsSpaceKey
import ru.yandex.intranet.d.services.operations.model.ReceivedLastUpdate
import ru.yandex.intranet.d.services.operations.model.ReceivedProvision
import ru.yandex.intranet.d.services.operations.model.ReceivedResourceKey
import ru.yandex.intranet.d.services.operations.model.ReceivedSegmentKey
import ru.yandex.intranet.d.services.operations.model.ReceivedUpdatedProvision
import ru.yandex.intranet.d.services.operations.model.ReceivedUserId
import ru.yandex.intranet.d.services.operations.model.ValidatedReceivedAccount
import ru.yandex.intranet.d.services.operations.model.ValidatedReceivedLastUpdate
import ru.yandex.intranet.d.services.operations.model.ValidatedReceivedProvision
import ru.yandex.intranet.d.services.operations.model.ValidatedReceivedUpdatedProvision
import ru.yandex.intranet.d.util.Uuids
import ru.yandex.intranet.d.util.result.ErrorCollection
import ru.yandex.intranet.d.util.result.Result
import ru.yandex.intranet.d.util.result.TypedError
import ru.yandex.intranet.d.util.units.Units
import java.time.Instant
import java.util.*

fun validateUpdateProvisionResponseDto(response: UpdateProvisionResponseDto,
                                       messages: MessageSource, locale: Locale): Result<ReceivedUpdatedProvision> {
    val builder = ReceivedUpdatedProvision.builder()
    val errorsBuilder = ErrorCollection.builder()
    response.provisions.ifPresent {
        it.forEachIndexed { i, provision ->
            val provisionBuilder = ReceivedProvision.builder()
            val provisionErrorsBuilder = ErrorCollection.builder()
            validateProvision(provision, provisionBuilder, provisionErrorsBuilder, "provisions.$i.",
                messages, locale)
            if (provisionErrorsBuilder.hasAnyErrors()) {
                errorsBuilder.add(provisionErrorsBuilder)
            } else {
                builder.addProvision(provisionBuilder.build())
            }
        }
    }
    response.accountVersion.ifPresent { builder.accountVersion(it) }
    response.accountsSpaceKey.ifPresent {
        val accountsSpaceKeyBuilder = ReceivedAccountsSpaceKey.builder()
        val accountsSpaceKeyErrorsBuilder = ErrorCollection.builder()
        validateAccountsSpaceKey(it, accountsSpaceKeyBuilder, accountsSpaceKeyErrorsBuilder, messages, locale)
        if (accountsSpaceKeyErrorsBuilder.hasAnyErrors()) {
            errorsBuilder.add(accountsSpaceKeyErrorsBuilder)
        } else {
            builder.accountsSpaceKey(accountsSpaceKeyBuilder.build())
        }
    }
    return if (errorsBuilder.hasAnyErrors()) {
        Result.failure(errorsBuilder.build())
    } else {
        Result.success(builder.build())
    }
}

fun getUsersFromReceivedUpdatedProvision(provision: ReceivedUpdatedProvision): Set<ReceivedUserId> {
    val result = mutableSetOf<ReceivedUserId>()
    provision.provisions.forEach { p -> p.lastUpdate.ifPresent { update -> update.author
        .ifPresent { author -> result.add(author) } } }
    return result
}

fun getOperationsFromReceivedUpdatedProvision(provision: ReceivedUpdatedProvision): Set<String> {
    val result = mutableSetOf<String>()
    provision.provisions.forEach { p -> p.lastUpdate.ifPresent { update -> update.operationId
        .ifPresent { op -> result.add(op) } } }
    return result
}

fun validateReceivedUpdatedProvision(provision: ReceivedUpdatedProvision,
                                     users: List<UserModel>,
                                     operations: List<AccountsQuotasOperationsModel>,
                                     provider: ProviderModel,
                                     accountsSpace: AccountSpaceModel?,
                                     resources: Map<ResourceId, ResourceModel>,
                                     segmentations: Map<SegmentationId, ResourceSegmentationModel>,
                                     segments: Map<SegmentId, ResourceSegmentModel>,
                                     resourceTypes: Map<ResourceTypeId, ResourceTypeModel>,
                                     unitsEnsembles: Map<UnitsEnsembleId, UnitsEnsembleModel>,
                                     messages: MessageSource,
                                     locale: Locale): Result<ValidatedReceivedUpdatedProvision> {
    val usersByUid = users.associateBy { it.passportUid.orElseThrow() }
    val usersByLogin = users.associateBy { it.passportLogin.get() }
    val operationsById = operations.associateBy { it.operationId }
    val resourceIndex = prepareResourceIndex(resources.values, resourceTypes, segmentations, segments, unitsEnsembles)
    val builder = ValidatedReceivedUpdatedProvision.builder()
    val errors = ErrorCollection.builder()
    validateReceivedUpdatedProvision(provision, usersByUid, usersByLogin, operationsById, provider, accountsSpace,
        segmentations, segments, resourceIndex, builder, errors, messages, locale)
    return if (errors.hasAnyErrors()) {
        Result.failure(errors.build())
    } else {
        Result.success(builder.build())
    }
}

fun validateProviderAccountDto(account: AccountDto, messages: MessageSource, locale: Locale): Result<ReceivedAccount> {
    val accountBuilder = ReceivedAccount.builder()
    val errorsBuilder = ErrorCollection.builder()
    validateAccount(account, accountBuilder, errorsBuilder, messages, locale)
    return if (errorsBuilder.hasAnyErrors()) {
        Result.failure(errorsBuilder.build())
    } else {
        Result.success(accountBuilder.build())
    }
}

fun getUsersFromReceivedAccount(account: ReceivedAccount): Set<ReceivedUserId> {
    val result = mutableSetOf<ReceivedUserId>()
    account.lastUpdate.ifPresent{ update -> update.author.ifPresent { author -> result.add(author) } }
    account.provisions.forEach { p -> p.lastUpdate.ifPresent { update -> update.author
        .ifPresent { author -> result.add(author) } } }
    return result
}

fun getOperationsFromReceivedAccount(account: ReceivedAccount): Set<String> {
    val result = mutableSetOf<String>()
    account.provisions.forEach { p -> p.lastUpdate.ifPresent { update -> update.operationId
        .ifPresent { op -> result.add(op) } } }
    account.lastUpdate.ifPresent { update -> update.operationId.ifPresent { op -> result.add(op) } }
    return result
}

fun getFolderFromReceivedAccount(account: ReceivedAccount): FolderId {
    return account.folderId
}

fun getExternalAccountIdFromReceivedAccount(account: ReceivedAccount): String {
    return account.accountId
}

fun validateReceivedAccount(receivedAccount: ReceivedAccount,
                            users: List<UserModel>,
                            operations: List<AccountsQuotasOperationsModel>,
                            accountFolder: FolderModel?,
                            existingAccount: AccountModel?,
                            provider: ProviderModel,
                            accountsSpace: AccountSpaceModel?,
                            resources: Map<ResourceId, ResourceModel>,
                            segmentations: Map<SegmentationId, ResourceSegmentationModel>,
                            segments: Map<SegmentId, ResourceSegmentModel>,
                            resourceTypes: Map<ResourceTypeId, ResourceTypeModel>,
                            unitsEnsembles: Map<UnitsEnsembleId, UnitsEnsembleModel>,
                            messages: MessageSource,
                            locale: Locale): Result<ValidatedReceivedAccount> {
    val usersByUid = users.associateBy { it.passportUid.orElseThrow() }
    val usersByLogin = users.associateBy { it.passportLogin.orElseThrow() }
    val operationsById = operations.associateBy { it.operationId }
    val resourceIndex = prepareResourceIndex(resources.values, resourceTypes, segmentations, segments, unitsEnsembles)
    val builder = ValidatedReceivedAccount.builder()
    val errors = ErrorCollection.builder()
    validateReceivedAccount(receivedAccount, usersByUid, usersByLogin, operationsById, accountFolder,
        existingAccount, provider, accountsSpace, segmentations, segments, resourceIndex, builder,
        errors, messages, locale)
    return if (errors.hasAnyErrors()) {
        Result.failure(errors.build())
    } else {
        Result.success(builder.build())
    }
}

private fun validateReceivedAccount(
    receivedAccount: ReceivedAccount,
    usersByUid: Map<String, UserModel>,
    usersByLogin: Map<String, UserModel>,
    operationsById: Map<String, AccountsQuotasOperationsModel>,
    accountFolder: FolderModel?,
    existingAccount: AccountModel?,
    provider: ProviderModel,
    accountsSpace: AccountSpaceModel?,
    segmentations: Map<SegmentationId, ResourceSegmentationModel>,
    segments: Map<SegmentId, ResourceSegmentModel>,
    resourceIndex: Map<ExternalResourceKey, ExpandedResource>,
    builder: ValidatedReceivedAccount.Builder,
    errors: ErrorCollection.Builder,
    messages: MessageSource,
    locale: Locale
) {
    builder.accountId(receivedAccount.accountId)
    receivedAccount.key.ifPresent { builder.key(it) }
    receivedAccount.displayName.ifPresent { builder.displayName(it) }
    builder.deleted(receivedAccount.isDeleted)
    if (accountFolder == null) {
        errors.addError("folderId", TypedError.invalid(messages
            .getMessage("errors.folder.not.found", null, locale)))
    } else {
        builder.folder(accountFolder)
    }
    receivedAccount.accountVersion.ifPresent { builder.accountVersion(it) }
    if (existingAccount != null) {
        builder.account(existingAccount)
    }
    validateAccountsSpace(receivedAccount.accountsSpaceKey.orElse(null), { builder.accountsSpace(it) }, provider,
        accountsSpace, segmentations, segments, errors, messages, locale)
    val seenResourceKeys = mutableSetOf<ResourceId>()
    for (i in receivedAccount.provisions.indices) {
        val provision = receivedAccount.provisions[i]
        val provisionBuilder = ValidatedReceivedProvision.builder()
        val provisionErrors = ErrorCollection.builder()
        validateProvision(receivedAccount.accountsSpaceKey.orElse(null), provision, usersByUid,
            usersByLogin, operationsById, resourceIndex, provisionBuilder, provisionErrors,
            "provisions.${i}.", messages, locale)
        if (provisionErrors.hasAnyErrors()) {
            errors.add(provisionErrors)
        } else {
            val validatedProvision = provisionBuilder.build()
            if (!seenResourceKeys.contains(validatedProvision.resource.id)) {
                builder.addProvision(validatedProvision)
                seenResourceKeys.add(validatedProvision.resource.id)
            } else {
                errors.addError("provisions.${i}", TypedError.invalid(messages
                    .getMessage("errors.resource.id.duplicate", null, locale)))
            }
        }
    }
    if (receivedAccount.lastUpdate.isPresent) {
        val lastUpdateBuilder = ValidatedReceivedLastUpdate.builder()
        val lastUpdateErrors = ErrorCollection.builder()
        validateLastUpdate(receivedAccount.lastUpdate.get(), usersByUid, usersByLogin, operationsById,
            lastUpdateBuilder, lastUpdateErrors, "lastUpdate.", messages, locale)
        if (lastUpdateErrors.hasAnyErrors()) {
            errors.add(lastUpdateErrors)
        } else {
            builder.lastUpdate(lastUpdateBuilder.build())
        }
    }
    builder.freeTier(receivedAccount.isFreeTier)
}

private fun validateAccount(account: AccountDto, accountBuilder: ReceivedAccount.Builder,
                            errorsBuilder: ErrorCollection.Builder, messages: MessageSource, locale: Locale) {
    if (account.accountId.isEmpty || account.accountId.get().isEmpty()) {
        errorsBuilder.addError("accountId", TypedError.invalid(messages
            .getMessage("errors.field.is.required", null, locale)))
    } else {
        accountBuilder.accountId(account.accountId.get())
    }
    account.key.ifPresent { accountBuilder.key(it) }
    account.displayName.ifPresent { accountBuilder.displayName(it) }
    if (account.folderId.isEmpty || account.folderId.get().isEmpty()) {
        errorsBuilder.addError("folderId", TypedError.invalid(messages
            .getMessage("errors.field.is.required", null, locale)))
    } else {
        if (!Uuids.isValidUuid(account.folderId.get())) {
            errorsBuilder.addError("folderId", TypedError.invalid(messages
                .getMessage("errors.folder.not.found", null, locale)))
        } else {
            accountBuilder.folderId(account.folderId.get())
        }
    }
    accountBuilder.deleted(account.deleted.orElse(false))
    account.provisions.ifPresent {
        it.forEachIndexed { i, provision ->
            val provisionBuilder = ReceivedProvision.builder()
            val provisionErrorsBuilder = ErrorCollection.builder()
            validateProvision(provision, provisionBuilder, provisionErrorsBuilder,
                "provisions.${i}.", messages, locale)
            if (provisionErrorsBuilder.hasAnyErrors()) {
                errorsBuilder.add(provisionErrorsBuilder)
            } else {
                accountBuilder.addProvision(provisionBuilder.build())
            }
        }
    }
    account.accountVersion.ifPresent {accountBuilder.accountVersion(it) }
    account.lastUpdate.ifPresent {
        val lastUpdateBuilder = ReceivedLastUpdate.builder()
        prepareLastUpdate(it, lastUpdateBuilder)
        accountBuilder.lastUpdate(lastUpdateBuilder.build())
    }
    account.accountsSpaceKey.ifPresent {
        val accountsSpaceKeyBuilder = ReceivedAccountsSpaceKey.builder()
        val accountsSpaceKeyErrorsBuilder = ErrorCollection.builder()
        validateAccountsSpaceKey(it, accountsSpaceKeyBuilder, accountsSpaceKeyErrorsBuilder, messages, locale)
        if (accountsSpaceKeyErrorsBuilder.hasAnyErrors()) {
            errorsBuilder.add(accountsSpaceKeyErrorsBuilder)
        } else {
            accountBuilder.accountsSpaceKey(accountsSpaceKeyBuilder.build())
        }
    }
}

private fun validateProvision(provision: ProvisionDto, provisionBuilder: ReceivedProvision.Builder,
                              errorsBuilder: ErrorCollection.Builder, keyPrefix: String,
                              messages: MessageSource, locale: Locale) {
    if (provision.resourceKey.isEmpty) {
        errorsBuilder.addError(keyPrefix + "resourceKey", TypedError.invalid(messages
            .getMessage("errors.field.is.required", null, locale)))
    } else {
        val resourceKeyBuilder = ReceivedResourceKey.builder()
        val resourceKeyErrorsBuilder = ErrorCollection.builder()
        validateResourceKey(provision.resourceKey.get(), resourceKeyBuilder,
            resourceKeyErrorsBuilder, "${keyPrefix}resourceKey.", messages, locale)
        if (resourceKeyErrorsBuilder.hasAnyErrors()) {
            errorsBuilder.add(resourceKeyErrorsBuilder)
        } else {
            provisionBuilder.resourceKey(resourceKeyBuilder.build())
        }
    }
    if (provision.providedAmount.isEmpty) {
        errorsBuilder.addError("${keyPrefix}providedAmount", TypedError.invalid(messages
            .getMessage("errors.field.is.required", null, locale)))
    } else {
        provisionBuilder.providedAmount(provision.providedAmount.get())
    }
    if (provision.allocatedAmount.isEmpty) {
        errorsBuilder.addError("${keyPrefix}allocatedAmount", TypedError.invalid(messages
            .getMessage("errors.field.is.required", null, locale)))
    } else {
        provisionBuilder.allocatedAmount(provision.allocatedAmount.get())
    }
    if (provision.providedAmountUnitKey.isEmpty) {
        errorsBuilder.addError("${keyPrefix}providedAmountUnitKey", TypedError.invalid(messages
            .getMessage("errors.field.is.required", null, locale)))
    } else {
        provisionBuilder.providedAmountUnitKey(provision.providedAmountUnitKey.get())
    }
    if (provision.allocatedAmountUnitKey.isEmpty) {
        errorsBuilder.addError("${keyPrefix}allocatedAmountUnitKey", TypedError.invalid(messages
            .getMessage("errors.field.is.required", null, locale)))
    } else {
        provisionBuilder.allocatedAmountUnitKey(provision.allocatedAmountUnitKey.get())
    }
    provision.quotaVersion.ifPresent {provisionBuilder.quotaVersion(it) }
    provision.lastUpdate.ifPresent {
        val lastUpdateBuilder = ReceivedLastUpdate.builder()
        val lastUpdateErrorsBuilder = ErrorCollection.builder()
        prepareLastUpdate(it, lastUpdateBuilder)
        if (lastUpdateErrorsBuilder.hasAnyErrors()) {
            errorsBuilder.add(lastUpdateErrorsBuilder)
        } else {
            provisionBuilder.lastUpdate(lastUpdateBuilder.build())
        }
    }
}

private fun validateResourceKey(resourceKey: ResourceKeyResponseDto,
                                resourceKeyBuilder: ReceivedResourceKey.Builder,
                                errorsBuilder: ErrorCollection.Builder, keyPrefix: String,
                                messages: MessageSource, locale: Locale) {
    if (resourceKey.resourceTypeKey.isEmpty || resourceKey.resourceTypeKey.get().isEmpty()) {
        errorsBuilder.addError("${keyPrefix}resourceTypeKey", TypedError.invalid(messages
            .getMessage("errors.field.is.required", null, locale)))
    } else {
        resourceKeyBuilder.resourceTypeKey(resourceKey.resourceTypeKey.get())
    }
    resourceKey.segmentation.ifPresent {
        it.forEachIndexed { i, segmentKey ->
            val segmentKeyBuilder = ReceivedSegmentKey.builder()
            val segmentKeyErrorsBuilder = ErrorCollection.builder()
            validateSegmentKey(segmentKey, segmentKeyBuilder, segmentKeyErrorsBuilder,
                keyPrefix + "segmentation.${i}.", messages, locale)
            if (segmentKeyErrorsBuilder.hasAnyErrors()) {
                errorsBuilder.add(segmentKeyErrorsBuilder)
            } else {
                resourceKeyBuilder.addSegment(segmentKeyBuilder.build())
            }
        }
    }
}

private fun validateSegmentKey(segmentKey: SegmentKeyResponseDto,
                               segmentKeyBuilder: ReceivedSegmentKey.Builder,
                               errorsBuilder: ErrorCollection.Builder, keyPrefix: String,
                               messages: MessageSource, locale: Locale) {
    if (segmentKey.segmentKey.isEmpty || segmentKey.segmentKey.get().isEmpty()) {
        errorsBuilder.addError("${keyPrefix}segmentKey", TypedError.invalid(messages
            .getMessage("errors.field.is.required", null, locale)))
    } else {
        segmentKeyBuilder.segmentKey(segmentKey.segmentKey.get())
    }
    if (segmentKey.segmentationKey.isEmpty || segmentKey.segmentationKey.get().isEmpty()) {
        errorsBuilder.addError("${keyPrefix}segmentationKey", TypedError.invalid(messages
            .getMessage("errors.field.is.required", null, locale)))
    } else {
        segmentKeyBuilder.segmentationKey(segmentKey.segmentationKey.get())
    }
}

private fun prepareLastUpdate(lastUpdate: LastUpdateDto,
                              lastUpdateBuilder: ReceivedLastUpdate.Builder) {
    lastUpdate.timestamp.ifPresent { t -> lastUpdateBuilder.timestamp(Instant.ofEpochMilli(t)) }
    lastUpdate.operationId.filter { it.isNotEmpty() }.ifPresent { lastUpdateBuilder.operationId(it) }
    lastUpdate.author.ifPresent { userId: UserIdDto ->
        val userIdBuilder = ReceivedUserId.builder()
        userId.staffLogin.filter { it.isNotEmpty() }.ifPresent { userIdBuilder.staffLogin(it) }
        userId.passportUid.filter { it.isNotEmpty() }.ifPresent{ userIdBuilder.passportUid(it) }
        lastUpdateBuilder.author(userIdBuilder.build())
    }
}

private fun validateAccountsSpaceKey(accountsSpaceKey: AccountsSpaceKeyResponseDto,
                                     accountsSpaceKeyBuilder: ReceivedAccountsSpaceKey.Builder,
                                     errorsBuilder: ErrorCollection.Builder,
                                     messages: MessageSource, locale: Locale) {
    accountsSpaceKey.segmentation.ifPresent {
        it.forEachIndexed { i, segmentKey ->
            val segmentKeyBuilder = ReceivedSegmentKey.builder()
            val segmentKeyErrorsBuilder = ErrorCollection.builder()
            validateSegmentKey(segmentKey!!, segmentKeyBuilder, segmentKeyErrorsBuilder,
                "accountsSpaceKey.segmentation.${i}.", messages, locale)
            if (segmentKeyErrorsBuilder.hasAnyErrors()) {
                errorsBuilder.add(segmentKeyErrorsBuilder)
            } else {
                accountsSpaceKeyBuilder.addSegment(segmentKeyBuilder.build())
            }
        }
    }

}

private fun validateReceivedUpdatedProvision(receivedProvision: ReceivedUpdatedProvision,
                                             usersByUid: Map<String, UserModel>,
                                             usersByLogin: Map<String, UserModel>,
                                             operationsById: Map<String, AccountsQuotasOperationsModel>,
                                             provider: ProviderModel,
                                             accountsSpace: AccountSpaceModel?,
                                             segmentations: Map<SegmentationId, ResourceSegmentationModel>,
                                             segments: Map<SegmentId, ResourceSegmentModel>,
                                             resourceIndex: Map<ExternalResourceKey, ExpandedResource>,
                                             builder: ValidatedReceivedUpdatedProvision.Builder,
                                             errors: ErrorCollection.Builder,
                                             messages: MessageSource,
                                             locale: Locale) {
    receivedProvision.accountVersion.ifPresent { builder.accountVersion(it) }
    validateAccountsSpace(receivedProvision.accountsSpaceKey.orElse(null), { builder.accountsSpace(it) },
        provider, accountsSpace, segmentations, segments, errors, messages, locale)
    val seenResourceKeys = mutableSetOf<ResourceId>()
    receivedProvision.provisions.forEachIndexed { i, provision ->
        val provisionBuilder = ValidatedReceivedProvision.builder()
        val provisionErrors = ErrorCollection.builder()
        validateProvision(receivedProvision.accountsSpaceKey.orElse(null), provision, usersByUid,
            usersByLogin, operationsById, resourceIndex, provisionBuilder, provisionErrors,
            "provisions.${i}.", messages, locale)
        if (provisionErrors.hasAnyErrors()) {
            errors.add(provisionErrors)
        } else {
            val validatedProvision = provisionBuilder.build()
            if (!seenResourceKeys.contains(validatedProvision.resource.id)) {
                builder.addProvision(validatedProvision)
                seenResourceKeys.add(validatedProvision.resource.id)
            } else {
                errors.addError("provisions.${i}", TypedError.invalid(messages
                    .getMessage("errors.resource.id.duplicate", null, locale)))
            }
        }
    }
}

private fun validateAccountsSpace(accountsSpaceKey: ReceivedAccountsSpaceKey?,
                                  builder: (AccountSpaceModel) -> Unit,
                                  provider: ProviderModel,
                                  accountsSpace: AccountSpaceModel?,
                                  segmentations: Map<SegmentationId, ResourceSegmentationModel>,
                                  segments: Map<SegmentId, ResourceSegmentModel>,
                                  errors: ErrorCollection.Builder,
                                  messages: MessageSource,
                                  locale: Locale) {
    if (accountsSpaceKey != null) {
        if (provider.isAccountsSpacesSupported && accountsSpace != null) {
            val expectedAccountsSpaceKey = toExternalAccountsSpaceKey(accountsSpace, segmentations, segments)
            val receivedAccountsSpaceKey = fromReceivedAccountsSpaceKey(accountsSpaceKey)
            if (receivedAccountsSpaceKey == null
                || !Objects.equals(expectedAccountsSpaceKey, receivedAccountsSpaceKey)
            ) {
                errors.addError("accountsSpaceKey", TypedError
                    .invalid(messages.getMessage("errors.accounts.space.mismatch", null, locale)))
            } else {
                builder(accountsSpace)
            }
        } else {
            errors.addError("accountsSpaceKey", TypedError
                .invalid(messages.getMessage("errors.accounts.spaces.is.not.supported", null, locale)))
        }
    } else {
        if (provider.isAccountsSpacesSupported && accountsSpace != null) {
            errors.addError("accountsSpaceKey",
                TypedError.invalid(messages.getMessage("errors.field.is.required", null, locale)))
        }
    }
}

fun validateReceivedProvisions(
    receivedProvisions: List<ReceivedProvision>,
    accountSpaceKey: ReceivedAccountsSpaceKey?,
    usersByUid: Map<String, UserModel>,
    usersByLogin: Map<String, UserModel>,
    operationsById: Map<String, AccountsQuotasOperationsModel>,
    resources: Collection<ResourceModel>,
    resourceTypes: Map<ResourceTypeId, ResourceTypeModel>,
    segmentations: Map<SegmentationId, ResourceSegmentationModel>,
    segments: Map<SegmentId, ResourceSegmentModel>,
    unitsEnsembles: Map<UnitsEnsembleId, UnitsEnsembleModel>,
    errors: ErrorCollection.Builder,
    messages: MessageSource,
    locale: Locale,
): List<ValidatedReceivedProvision> {
    val resourceIndex = prepareResourceIndex(resources, resourceTypes, segmentations, segments, unitsEnsembles)
    val result = mutableListOf<ValidatedReceivedProvision>()
    val seenResourceKeys = mutableSetOf<ResourceId>()
    receivedProvisions.forEachIndexed { i, provision ->
        val provisionBuilder = ValidatedReceivedProvision.builder()
        val provisionErrors = ErrorCollection.builder()
        validateProvision(accountSpaceKey, provision, usersByUid,
            usersByLogin, operationsById, resourceIndex, provisionBuilder, provisionErrors,
            "provisions.${i}.", messages, locale)
        if (provisionErrors.hasAnyErrors()) {
            errors.add(provisionErrors)
        } else {
            val validatedProvision = provisionBuilder.build()
            if (!seenResourceKeys.contains(validatedProvision.resource.id)) {
                result.add(validatedProvision)
                seenResourceKeys.add(validatedProvision.resource.id)
            } else {
                errors.addError("provisions.${i}", TypedError.invalid(messages
                    .getMessage("errors.resource.id.duplicate", null, locale)))
            }
        }
    }
    return result
}

private fun validateProvision(accountsSpaceKey: ReceivedAccountsSpaceKey?,
                              provision: ReceivedProvision,
                              usersByUid: Map<String, UserModel>,
                              usersByLogin: Map<String, UserModel>,
                              operationsById: Map<String, AccountsQuotasOperationsModel>,
                              resourceIndex: Map<ExternalResourceKey, ExpandedResource>,
                              provisionBuilder: ValidatedReceivedProvision.Builder,
                              provisionErrors: ErrorCollection.Builder,
                              fieldKeyPrefix: String,
                              messages: MessageSource,
                              locale: Locale) {
    provision.quotaVersion.ifPresent { provisionBuilder.quotaVersion(it) }
    if (provision.lastUpdate.isPresent) {
        val lastUpdateBuilder = ValidatedReceivedLastUpdate.builder()
        val lastUpdateErrors = ErrorCollection.builder()
        validateLastUpdate(provision.lastUpdate.get(), usersByUid, usersByLogin, operationsById,
            lastUpdateBuilder, lastUpdateErrors, "${fieldKeyPrefix}lastUpdate.", messages, locale)
        if (lastUpdateErrors.hasAnyErrors()) {
            provisionErrors.add(lastUpdateErrors)
        } else {
            provisionBuilder.lastUpdate(lastUpdateBuilder.build())
        }
    }
    val expandedResource = getTargetResource(accountsSpaceKey, provision, resourceIndex)
    if (expandedResource == null) {
        provisionErrors.addError("${fieldKeyPrefix}resourceKey", TypedError.invalid(messages
            .getMessage("errors.resource.not.found", null, locale)))
        return
    }
    provisionBuilder.resource(expandedResource.resource)
    val providedUnit = expandedResource.unitsEnsemble.unitByKey(provision.providedAmountUnitKey)
    if (providedUnit.isEmpty) {
        provisionErrors.addError("${fieldKeyPrefix}providedAmountUnitKey", TypedError.invalid(messages
            .getMessage("errors.unit.not.found", null, locale)))
    } else {
        val providedAmount = Units.convertFromApi(provision.providedAmount,
            expandedResource.resource, expandedResource.unitsEnsemble, providedUnit.get())
        if (providedAmount.isEmpty) {
            provisionErrors.addError("${fieldKeyPrefix}providedAmount", TypedError.invalid(messages
                .getMessage("errors.value.can.not.be.converted.to.base.unit", null, locale)))
        } else {
            provisionBuilder.providedAmount(providedAmount.get())
        }
    }
    val allocatedUnit = expandedResource.unitsEnsemble.unitByKey(provision.allocatedAmountUnitKey)
    if (allocatedUnit.isEmpty) {
        provisionErrors.addError("${fieldKeyPrefix}allocatedAmountUnitKey", TypedError.invalid(messages
            .getMessage("errors.unit.not.found", null, locale)))
    } else {
        val allocatedAmount = Units.convertFromApi(provision.allocatedAmount,
            expandedResource.resource, expandedResource.unitsEnsemble, allocatedUnit.get())
        if (allocatedAmount.isEmpty) {
            provisionErrors.addError("${fieldKeyPrefix}allocatedAmount", TypedError.invalid(messages
                .getMessage("errors.value.can.not.be.converted.to.base.unit", null, locale)))
        } else {
            provisionBuilder.allocatedAmount(allocatedAmount.get())
        }
    }
}

private fun validateLastUpdate(lastUpdate: ReceivedLastUpdate,
                               usersByUid: Map<String, UserModel>,
                               usersByLogin: Map<String, UserModel>,
                               operationsById: Map<String, AccountsQuotasOperationsModel>,
                               lastUpdateBuilder: ValidatedReceivedLastUpdate.Builder,
                               lastUpdateErrors: ErrorCollection.Builder,
                               fieldKeyPrefix: String,
                               messages: MessageSource,
                               locale: Locale) {
    lastUpdate.timestamp.ifPresent { lastUpdateBuilder.timestamp(it) }
    lastUpdate.operationId.ifPresent { lastUpdateBuilder.operationId(it) }
    if (lastUpdate.operationId.isPresent) {
        val operation = operationsById[lastUpdate.operationId.get()]
        if (operation != null) {
            lastUpdateBuilder.operation(operation)
        }
    }
    if (lastUpdate.author.isPresent && (lastUpdate.author.get().passportUid.isPresent
            || lastUpdate.author.get().staffLogin.isPresent)) {
        val authorByUid = if (lastUpdate.author.get().passportUid.isPresent) {
            usersByUid[lastUpdate.author.get().passportUid.get()]
        } else {
            null
        }
        val authorByLogin = if (lastUpdate.author.get().staffLogin.isPresent) {
            usersByLogin[lastUpdate.author.get().staffLogin.get()]
        } else {
            null
        }
        if ((authorByUid == null && authorByLogin == null) || (authorByUid != null && authorByLogin != null
                && authorByUid.id != authorByLogin.id)) {
            lastUpdateErrors.addError("${fieldKeyPrefix}author", TypedError
                .invalid(messages.getMessage("errors.user.not.found", null, locale)))
        } else {
            lastUpdateBuilder.author(authorByUid ?: authorByLogin)
        }
    }
}

private fun getTargetResource(accountsSpaceKey: ReceivedAccountsSpaceKey?,
                              provision: ReceivedProvision,
                              resourceIndex: Map<ExternalResourceKey, ExpandedResource>): ExpandedResource? {
    val resourceTypeKey = provision.resourceKey.resourceTypeKey
    val segments = mutableListOf<ReceivedSegmentKey>()
    if (accountsSpaceKey != null) {
        segments.addAll(accountsSpaceKey.segmentation)
    }
    segments.addAll(provision.resourceKey.segmentation)
    val segmentsByKey = segments.groupBy { it.segmentKey }
    if (segmentsByKey.values.any { it.size > 1 }) {
        return null
    }
    val externalKey = ExternalResourceKey(resourceTypeKey,
        segments.map { ExternalSegmentKey(it.segmentationKey, it.segmentKey) }.toSet())
    return if (!resourceIndex.containsKey(externalKey)) {
        null
    } else {
        resourceIndex[externalKey]
    }
}

private fun toExternalAccountsSpaceKey(accountsSpace: AccountSpaceModel,
                                       segmentations: Map<SegmentationId, ResourceSegmentationModel>,
                                       segments: Map<SegmentId, ResourceSegmentModel>): ExternalAccountsSpaceKey {
    return ExternalAccountsSpaceKey(accountsSpace.segments
        .map { ExternalSegmentKey(segmentations[it.segmentationId]!!.key, segments[it.segmentId]!!.key) }.toSet())
}

private fun fromReceivedAccountsSpaceKey(accountsSpaceKey: ReceivedAccountsSpaceKey): ExternalAccountsSpaceKey? {
    val segmentsBySegmentation = accountsSpaceKey.segmentation.groupBy { it.segmentationKey }
    if (segmentsBySegmentation.values.any { it.size > 1 }) {
        return null
    }
    return ExternalAccountsSpaceKey(accountsSpaceKey.segmentation
        .map { ExternalSegmentKey(it.segmentationKey, it.segmentKey) }.toSet())
}

private fun prepareResourceIndex(resources: Collection<ResourceModel>,
                                 resourceTypes: Map<ResourceTypeId, ResourceTypeModel>,
                                 segmentations: Map<SegmentationId, ResourceSegmentationModel>,
                                 segments: Map<SegmentId, ResourceSegmentModel>,
                                 unitsEnsembles: Map<UnitsEnsembleId, UnitsEnsembleModel>
): Map<ExternalResourceKey, ExpandedResource> {
    val result = mutableMapOf<ExternalResourceKey, ExpandedResource>()
    resources.filter { !it.isDeleted }.forEach { resource ->
        val resourceType = resourceTypes[resource.resourceTypeId]!!
        val expandedSegments = resource.segments.map { ExpandedSegment(segmentations[it.segmentationId]!!,
            segments[it.segmentId]!!) }.toSet()
        val unitsEnsemble = unitsEnsembles[resource.unitsEnsembleId]
        result[ExternalResourceKey(resourceType.key, resource.segments
            .map { ExternalSegmentKey(segmentations[it.segmentationId]!!.key, segments[it.segmentId]!!.key) }
            .toSet())] = ExpandedResource(resource, resourceType, expandedSegments, unitsEnsemble)
    }
    return result
}
