package ru.yandex.intranet.d.services.integration.providers;

import java.io.UncheckedIOException;
import java.time.Duration;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.function.BiConsumer;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import com.google.protobuf.Any;
import com.google.protobuf.Empty;
import com.google.protobuf.InvalidProtocolBufferException;
import com.google.protobuf.util.Timestamps;
import com.google.rpc.BadRequest;
import com.google.rpc.Status;
import io.grpc.CallOptions;
import io.grpc.Metadata;
import io.grpc.StatusException;
import io.grpc.StatusRuntimeException;
import io.grpc.protobuf.StatusProto;
import io.grpc.stub.StreamObserver;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.MessageSource;
import org.springframework.stereotype.Component;
import reactor.core.Exceptions;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuples;
import reactor.util.retry.Retry;
import reactor.util.retry.RetrySpec;

import ru.yandex.intranet.d.backend.service.provider_proto.Account;
import ru.yandex.intranet.d.backend.service.provider_proto.AccountsPageToken;
import ru.yandex.intranet.d.backend.service.provider_proto.AccountsServiceGrpc;
import ru.yandex.intranet.d.backend.service.provider_proto.AccountsSpaceKey;
import ru.yandex.intranet.d.backend.service.provider_proto.Amount;
import ru.yandex.intranet.d.backend.service.provider_proto.CompoundAccountsSpaceKey;
import ru.yandex.intranet.d.backend.service.provider_proto.CompoundResourceKey;
import ru.yandex.intranet.d.backend.service.provider_proto.CreateAccountAndProvideRequest;
import ru.yandex.intranet.d.backend.service.provider_proto.CreateAccountRequest;
import ru.yandex.intranet.d.backend.service.provider_proto.DeleteAccountRequest;
import ru.yandex.intranet.d.backend.service.provider_proto.GetAccountRequest;
import ru.yandex.intranet.d.backend.service.provider_proto.KnownAccountProvisions;
import ru.yandex.intranet.d.backend.service.provider_proto.KnownProvision;
import ru.yandex.intranet.d.backend.service.provider_proto.LastUpdate;
import ru.yandex.intranet.d.backend.service.provider_proto.ListAccountsByFolderRequest;
import ru.yandex.intranet.d.backend.service.provider_proto.ListAccountsByFolderResponse;
import ru.yandex.intranet.d.backend.service.provider_proto.ListAccountsRequest;
import ru.yandex.intranet.d.backend.service.provider_proto.ListAccountsResponse;
import ru.yandex.intranet.d.backend.service.provider_proto.MoveAccountRequest;
import ru.yandex.intranet.d.backend.service.provider_proto.MoveProvisionRequest;
import ru.yandex.intranet.d.backend.service.provider_proto.MoveProvisionResponse;
import ru.yandex.intranet.d.backend.service.provider_proto.PassportUID;
import ru.yandex.intranet.d.backend.service.provider_proto.Provision;
import ru.yandex.intranet.d.backend.service.provider_proto.ProvisionRequest;
import ru.yandex.intranet.d.backend.service.provider_proto.RenameAccountRequest;
import ru.yandex.intranet.d.backend.service.provider_proto.ResourceKey;
import ru.yandex.intranet.d.backend.service.provider_proto.ResourceSegmentKey;
import ru.yandex.intranet.d.backend.service.provider_proto.RevokeFreeTierRequest;
import ru.yandex.intranet.d.backend.service.provider_proto.StaffLogin;
import ru.yandex.intranet.d.backend.service.provider_proto.UpdateProvisionRequest;
import ru.yandex.intranet.d.backend.service.provider_proto.UpdateProvisionResponse;
import ru.yandex.intranet.d.backend.service.provider_proto.UserID;
import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.services.integration.providers.grpc.GrpcClient;
import ru.yandex.intranet.d.services.integration.providers.grpc.GrpcStubFactoryService;
import ru.yandex.intranet.d.services.integration.providers.grpc.RequestIdHolder;
import ru.yandex.intranet.d.services.integration.providers.grpc.RequestIdInterceptor;
import ru.yandex.intranet.d.services.integration.providers.rest.ProvidersClient;
import ru.yandex.intranet.d.services.integration.providers.rest.model.AccountDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.AccountsSpaceKeyRequestDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.AccountsSpaceKeyResponseDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.CreateAccountAndProvideRequestDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.CreateAccountRequestDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.DeleteAccountRequestDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.GetAccountRequestDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.KnownAccountProvisionsDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.KnownProvisionDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.LastUpdateDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.ListAccountsByFolderRequestDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.ListAccountsRequestDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.ListAccountsResponseDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.MoveAccountRequestDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.MoveProvisionRequestDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.MoveProvisionResponseDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.ProvisionDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.ProvisionRequestDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.RenameAccountRequestDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.ResourceKeyRequestDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.ResourceKeyResponseDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.RevokeFreeTierRequestDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.SegmentKeyRequestDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.SegmentKeyResponseDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.UpdateProvisionRequestDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.UpdateProvisionResponseDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.UserIdDto;
import ru.yandex.intranet.d.services.integration.providers.security.GrpcCallCredentialsSupplier;
import ru.yandex.intranet.d.services.integration.providers.security.ProviderAuthSupplier;
import ru.yandex.intranet.d.services.operations.OperationsRequestLogService;
import ru.yandex.intranet.d.util.AsyncMetrics;
import ru.yandex.intranet.d.util.result.ErrorCollection;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.util.result.TypedError;

/**
 * Providers integration service.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class ProvidersIntegrationService {

    private static final Logger LOG = LoggerFactory.getLogger(ProvidersIntegrationService.class);

    public static final Metadata.Key<String> REQUEST_ID_KEY = Metadata.Key
            .of("X-Request-ID", Metadata.ASCII_STRING_MARSHALLER);

    private final GrpcStubFactoryService grpcStubFactory;
    private final ProvidersClient providersClient;
    private final MessageSource messages;
    private final ProviderAuthSupplier providerAuthSupplier;
    private final GrpcCallCredentialsSupplier grpcCallCredentialsSupplier;
    private final long timeoutMillis;
    private final long maxAttempts;
    private final long minBackoffMillis;
    private final long responseSizeLimitBytes;
    private final long deadlineAfterMillis;
    private final IntegrationMetrics integrationMetrics;
    private final OperationsRequestLogService operationsRequestLogService;

    @SuppressWarnings("ParameterNumber")
    public ProvidersIntegrationService(
            GrpcStubFactoryService grpcStubFactory,
            ProvidersClient providersClient,
            @Qualifier("messageSource") MessageSource messages,
            ProviderAuthSupplier providerAuthSupplier,
            GrpcCallCredentialsSupplier grpcCallCredentialsSupplier,
            @Value("${providers.client.timeoutMs}") long timeoutMillis,
            @Value("${providers.client.maxAttempts}") long maxAttempts,
            @Value("${providers.client.minBackoffMs}") long minBackoffMillis,
            @Value("${providers.client.responseSizeLimitBytes}") long responseSizeLimitBytes,
            @Value("${providers.client.deadlineAfterMs}") long deadlineAfterMillis,
            IntegrationMetrics integrationMetrics,
            OperationsRequestLogService operationsRequestLogService) {
        this.grpcStubFactory = grpcStubFactory;
        this.providersClient = providersClient;
        this.messages = messages;
        this.providerAuthSupplier = providerAuthSupplier;
        this.grpcCallCredentialsSupplier = grpcCallCredentialsSupplier;
        this.timeoutMillis = timeoutMillis;
        this.maxAttempts = maxAttempts;
        this.minBackoffMillis = minBackoffMillis;
        this.responseSizeLimitBytes = responseSizeLimitBytes;
        this.deadlineAfterMillis = deadlineAfterMillis;
        this.integrationMetrics = integrationMetrics;
        this.operationsRequestLogService = operationsRequestLogService;
    }

    public Mono<Result<Response<UpdateProvisionResponseDto>>> updateProvision(String accountId, ProviderModel provider,
                                                                              UpdateProvisionRequestDto request,
                                                                              Locale locale) {
        OperationLog.RequestStage<UpdateProvisionRequestDto> log = new OperationLog.RequestStage<>(
                "updateProvision", provider.getTenantId(), request.getOperationId(),
                request, true, true);
        return selectImplementation(provider, locale,
                (stub, ticket) -> updateProvisionGrpc(ticket, new RequestIdSupplier(),
                        stub, provider.getId(), provider.getKey(), accountId, request, log),
                (baseUrl, ticket) -> providersClient.updateProvision(baseUrl, ticket, new RequestIdSupplier(),
                        accountId, provider.getId(), provider.getKey(), request, log));
    }

    public Mono<Result<Response<AccountDto>>> getAccount(String accountId, ProviderModel provider,
                                                         GetAccountRequestDto request, Locale locale) {
        final OperationLog.RequestStage<GetAccountRequestDto> log = new OperationLog.RequestStage<>(
                "getAccount", provider.getTenantId(), null, request, false, false);
        return selectImplementation(provider, locale,
                (stub, ticket) -> getAccountGrpc(ticket, new RequestIdSupplier(),
                        stub, provider.getId(), provider.getKey(), accountId, request, log),
                (baseUrl, ticket) -> providersClient.getAccount(baseUrl, ticket, new RequestIdSupplier(),
                        accountId, provider.getId(), provider.getKey(), request, log));
    }

    public Mono<Result<Response<AccountDto>>> createAccount(ProviderModel provider, CreateAccountRequestDto request,
                                                            Locale locale) {
        OperationLog.RequestStage<CreateAccountRequestDto> log = new OperationLog.RequestStage<>(
                "createAccount", provider.getTenantId(), request.getOperationId().orElse(null), request, true, true);
        return selectImplementation(provider, locale,
                (stub, ticket) -> createAccountGrpc(ticket, new RequestIdSupplier(),
                        stub, provider.getId(), provider.getKey(), request, log),
                (baseUrl, ticket) -> providersClient.createAccount(baseUrl, ticket, new RequestIdSupplier(),
                                provider.getId(), provider.getKey(), request, log));
    }

    public Mono<Result<Response<AccountDto>>> createAccountAndProvide(ProviderModel provider,
                                                                      CreateAccountAndProvideRequestDto request,
                                                                      Locale locale) {
        OperationLog.RequestStage<CreateAccountAndProvideRequestDto> log =
                new OperationLog.RequestStage<>(
                "createAccountAndProvide", provider.getTenantId(), request.getOperationId(),
                        request, true, true);
        return selectImplementation(provider, locale,
                (stub, ticket) -> createAccountAndProvideGrpc(ticket, new RequestIdSupplier(),
                        stub, provider.getId(), provider.getKey(), request, log),
                (baseUrl, ticket) -> providersClient
                        .createAccountAndProvide(baseUrl, ticket, new RequestIdSupplier(),
                                provider.getId(), provider.getKey(), request, log));
    }

    public Mono<Result<Response<Void>>> deleteAccount(String accountId, ProviderModel provider,
                                                      DeleteAccountRequestDto request, Locale locale) {
        final OperationLog.RequestStage<DeleteAccountRequestDto> log = new OperationLog.RequestStage<>(
                "deleteAccount", provider.getTenantId(), request.getOperationId(),
                request, false, false);
        return selectImplementation(provider, locale,
                (stub, ticket) -> deleteAccountGrpc(ticket, new RequestIdSupplier(), stub, provider.getId(),
                        provider.getKey(), accountId, request, log), (baseUrl, ticket) -> providersClient
                        .deleteAccount(baseUrl, ticket, new RequestIdSupplier(), accountId, provider.getId(),
                                provider.getKey(), request, log));
    }

    public Mono<Result<Response<AccountDto>>> renameAccount(String accountId, ProviderModel provider,
                                                            RenameAccountRequestDto request, Locale locale) {
        final OperationLog.RequestStage<RenameAccountRequestDto> log = new OperationLog.RequestStage<>(
                "renameAccount", provider.getTenantId(), request.getOperationId(),
                request, false, false);
        return selectImplementation(provider, locale,
                (stub, ticket) -> renameAccountGrpc(ticket, new RequestIdSupplier(),
                        stub, provider.getId(), provider.getKey(), accountId, request, log),
                (baseUrl, ticket) -> providersClient.renameAccount(baseUrl, ticket, new RequestIdSupplier(),
                        accountId, provider.getId(), provider.getKey(), request, log));
    }

    public Mono<Result<Response<AccountDto>>> moveAccount(String accountId, ProviderModel provider,
                                                          MoveAccountRequestDto request, Locale locale) {
        final OperationLog.RequestStage<MoveAccountRequestDto> log = new OperationLog.RequestStage<>(
                "moveAccount", provider.getTenantId(), request.getOperationId(),
                request, false, false);
        return selectImplementation(provider, locale,
                (stub, ticket) -> moveAccountGrpc(ticket, new RequestIdSupplier(),
                        stub, provider.getId(), provider.getKey(), accountId, request, log),
                (baseUrl, ticket) -> providersClient.moveAccount(baseUrl, ticket, new RequestIdSupplier(),
                                accountId, provider.getId(), provider.getKey(), request, log));
    }

    public Mono<Result<Response<MoveProvisionResponseDto>>> moveProvision(String sourceAccountId,
                                                                          ProviderModel provider,
                                                                          MoveProvisionRequestDto request,
                                                                          Locale locale) {
        OperationLog.RequestStage<MoveProvisionRequestDto> log = new OperationLog.RequestStage<>(
                "moveProvision", provider.getTenantId(), request.getOperationId(),
                request, true, true);
        return selectImplementation(provider, locale,
                (stub, ticket) -> moveProvisionGrpc(ticket, new RequestIdSupplier(),
                        stub, provider.getId(), provider.getKey(), sourceAccountId, request, log),
                (baseUrl, ticket) -> providersClient.moveProvision(baseUrl, ticket, new RequestIdSupplier(),
                        sourceAccountId, provider.getId(), provider.getKey(), request, log));
    }

    public Mono<Result<Response<ListAccountsResponseDto>>> listAccounts(ProviderModel provider,
                                                                        ListAccountsRequestDto request,
                                                                        Locale locale) {
        OperationLog.RequestStage<ListAccountsRequestDto> log = new OperationLog.RequestStage<>(
                "listAccounts", provider.getTenantId(), null, request, false, false);
        return selectImplementation(provider, locale,
                (stub, ticket) -> listAccountsGrpc(ticket, new RequestIdSupplier(),
                        stub, provider.getId(), provider.getKey(), request, log),
                (baseUrl, ticket) -> providersClient.listAccounts(baseUrl, ticket, new RequestIdSupplier(),
                        provider.getId(), provider.getKey(), request, log));
    }

    @SuppressWarnings("ParameterNumber")
    public Mono<Result<Response<ListAccountsResponseDto>>> listAccountsByFolder(ProviderModel provider,
                                                                                ListAccountsByFolderRequestDto request,
                                                                                Locale locale) {
        OperationLog.RequestStage<ListAccountsByFolderRequestDto> log = new OperationLog.RequestStage<>(
                "listAccountsByFolder", provider.getTenantId(), null,
                request, false, false);
        return selectImplementation(provider, locale,
                (stub, ticket) -> listAccountsByFolderGrpc(ticket, new RequestIdSupplier(), stub,
                        provider.getId(), provider.getKey(), request, log),
                (baseUrl, ticket) -> providersClient.listAccountsByFolder(baseUrl, ticket, new RequestIdSupplier(),
                        provider.getId(), provider.getKey(), request, log));
    }

    public Mono<Result<Response<AccountDto>>> revokeFreeTier(
            String accountId, ProviderModel provider, RevokeFreeTierRequestDto request, Locale locale
    ) {
        final OperationLog.RequestStage<RevokeFreeTierRequestDto> log = new OperationLog.RequestStage<>(
                "revokeFreeTier", provider.getTenantId(), request.getOperationId(),
                request, false, false);
        return selectImplementation(provider, locale,
                (stub, ticket) ->
                        revokeFreeTierGrpc(ticket, new RequestIdSupplier(), stub, provider.getId(), provider.getKey(),
                                accountId, request, log),
                (baseUrl, ticket) -> providersClient.revokeFreeTier(baseUrl, ticket, new RequestIdSupplier(),
                        accountId, provider.getId(), provider.getKey(), request, log));
    }

    @SuppressWarnings("ParameterNumber")
    private Mono<Response<UpdateProvisionResponseDto>> updateProvisionGrpc(
            String tvmTicket, RequestIdSupplier requestIdSupplier, AccountsServiceGrpc.AccountsServiceStub stub,
            String providerId, String providerKey, String accountId, UpdateProvisionRequestDto request,
            OperationLog.RequestStage<UpdateProvisionRequestDto> log) {
        return doGrpcCall(tvmTicket, requestIdSupplier, stub,
                (s, observer) -> s.updateProvision(toUpdateProvisionRequest(request, accountId, providerId), observer),
                this::toUpdateProvisionResponse, log,
                (d, r) -> integrationMetrics.afterUpdateProvision(providerKey, d, r));
    }

    @SuppressWarnings("ParameterNumber")
    private Mono<Response<AccountDto>> getAccountGrpc(
            String tvmTicket, RequestIdSupplier requestIdSupplier, AccountsServiceGrpc.AccountsServiceStub stub,
            String providerId, String providerKey, String accountId, GetAccountRequestDto request,
            OperationLog.RequestStage<GetAccountRequestDto> log) {
        return doGrpcCall(tvmTicket, requestIdSupplier, stub,
                (s, observer) -> s.getAccount(toGetAccountRequest(providerId, accountId, request), observer),
                this::toAccount, log,
                (d, r) -> integrationMetrics.afterGetAccount(providerKey, d, r));
    }

    @SuppressWarnings("ParameterNumber")
    private Mono<Response<AccountDto>> createAccountGrpc(
            String tvmTicket, RequestIdSupplier requestIdSupplier, AccountsServiceGrpc.AccountsServiceStub stub,
            String providerId, String providerKey, CreateAccountRequestDto request,
            OperationLog.RequestStage<CreateAccountRequestDto> log) {
        return doGrpcCall(tvmTicket, requestIdSupplier, stub,
                (s, observer) -> s.createAccount(toCreateAccountRequest(request, providerId), observer),
                this::toAccount, log,
                (d, r) -> integrationMetrics.afterCreateAccount(providerKey, d, r));
    }

    @SuppressWarnings("ParameterNumber")
    private Mono<Response<AccountDto>> createAccountAndProvideGrpc(
            String tvmTicket, RequestIdSupplier requestIdSupplier, AccountsServiceGrpc.AccountsServiceStub stub,
            String providerId, String providerKey, CreateAccountAndProvideRequestDto request,
            OperationLog.RequestStage<CreateAccountAndProvideRequestDto> log) {
        return doGrpcCall(tvmTicket, requestIdSupplier, stub,
                (s, observer) -> s.createAccountAndProvide(toCreateAccountAndProvideRequest(request, providerId),
                        observer),
                this::toAccount, log,
                (d, r) -> integrationMetrics.afterCreateAccountAndProvide(providerKey, d, r));
    }

    @SuppressWarnings("ParameterNumber")
    private Mono<Response<Void>> deleteAccountGrpc(
            String tvmTicket, RequestIdSupplier requestIdSupplier, AccountsServiceGrpc.AccountsServiceStub stub,
            String providerId, String providerKey, String accountId, DeleteAccountRequestDto request,
            OperationLog.RequestStage<DeleteAccountRequestDto> log) {

        return this.<Void, Empty>doGrpcCall(tvmTicket, requestIdSupplier, stub,
                (s, observer) -> s.deleteAccount(toDeleteAccountRequest(request, providerId, accountId), observer),
                e -> null, log,
                (d, r) -> integrationMetrics.afterDeleteAccount(providerKey, d, r));
    }

    @SuppressWarnings("ParameterNumber")
    private Mono<Response<AccountDto>> renameAccountGrpc(
            String tvmTicket, RequestIdSupplier requestIdSupplier, AccountsServiceGrpc.AccountsServiceStub stub,
            String providerId, String providerKey, String accountId, RenameAccountRequestDto request,
            OperationLog.RequestStage<RenameAccountRequestDto> log) {
        return doGrpcCall(tvmTicket, requestIdSupplier, stub,
                (s, observer) -> s.renameAccount(toRenameAccountRequest(request, providerId, accountId),
                        observer),
                this::toAccount, log,
                (d, r) -> integrationMetrics.afterRenameAccount(providerKey, d, r));
    }

    @SuppressWarnings("ParameterNumber")
    private Mono<Response<AccountDto>> moveAccountGrpc(
            String tvmTicket, RequestIdSupplier requestIdSupplier, AccountsServiceGrpc.AccountsServiceStub stub,
            String providerId, String providerKey, String sourceAccountId, MoveAccountRequestDto request,
            OperationLog.RequestStage<MoveAccountRequestDto> log) {
        return doGrpcCall(tvmTicket, requestIdSupplier, stub,
                (s, observer) -> s.moveAccount(toMoveAccountRequest(request, providerId, sourceAccountId),
                        observer),
                this::toAccount, log,
                (d, r) -> integrationMetrics.afterMoveAccount(providerKey, d, r));
    }

    @SuppressWarnings("ParameterNumber")
    private Mono<Response<MoveProvisionResponseDto>> moveProvisionGrpc(
            String tvmTicket, RequestIdSupplier requestIdSupplier, AccountsServiceGrpc.AccountsServiceStub stub,
            String providerId, String providerKey, String accountId, MoveProvisionRequestDto request,
            OperationLog.RequestStage<MoveProvisionRequestDto> log) {
        return doGrpcCall(tvmTicket, requestIdSupplier, stub,
                (s, observer) -> s.moveProvision(toMoveProvisionRequest(request, providerId, accountId),
                        observer),
                this::toMoveProvisionResponse, log,
                (d, r) -> integrationMetrics.afterMoveProvision(providerKey, d, r));
    }

    @SuppressWarnings("ParameterNumber")
    private Mono<Response<ListAccountsResponseDto>> listAccountsGrpc(
            String tvmTicket, RequestIdSupplier requestIdSupplier, AccountsServiceGrpc.AccountsServiceStub stub,
            String providerId, String providerKey, ListAccountsRequestDto request,
            OperationLog.RequestStage<ListAccountsRequestDto> log) {
        return doGrpcCall(tvmTicket, requestIdSupplier, stub,
                (s, observer) -> s.listAccounts(toListAccountsRequest(providerId, request), observer),
                this::toListAccountsResponse, log,
                (d, r) -> integrationMetrics.afterListAccounts(providerKey, d, r));
    }

    @SuppressWarnings("ParameterNumber")
    private Mono<Response<ListAccountsResponseDto>> listAccountsByFolderGrpc(
            String tvmTicket, RequestIdSupplier requestIdSupplier, AccountsServiceGrpc.AccountsServiceStub stub,
            String providerId, String providerKey, ListAccountsByFolderRequestDto request,
            OperationLog.RequestStage<ListAccountsByFolderRequestDto> log) {
        return doGrpcCall(tvmTicket, requestIdSupplier, stub,
                (s, observer) -> s.listAccountsByFolder(toListAccountsByFolderRequest(providerId, request), observer),
                this::toListAccountsByFolderResponse, log,
                (d, r) -> integrationMetrics.afterListAccountsByFolder(providerKey, d, r));
    }

    @SuppressWarnings("ParameterNumber")
    private Mono<Response<AccountDto>> revokeFreeTierGrpc(
            String tvmTicket,
            RequestIdSupplier requestIdSupplier,
            AccountsServiceGrpc.AccountsServiceStub stub,
            String providerId,
            String providerKey,
            String accountId,
            RevokeFreeTierRequestDto request,
            OperationLog.RequestStage<RevokeFreeTierRequestDto> log
    ) {
        return doGrpcCall(tvmTicket, requestIdSupplier, stub,
                (s, observer) -> s.revokeFreeTier(
                        toRevokeFreeTierRequest(request, providerId, accountId), observer
                ),
                this::toAccount, log,
                (d, r) -> integrationMetrics.afterRevokeFreeTier(providerKey, d, r));
    }

    @SuppressWarnings("ParameterNumber")
    private <T, U> Mono<Response<T>> doGrpcCall(
            String tvmTicket, RequestIdSupplier requestIdSupplier, AccountsServiceGrpc.AccountsServiceStub stub,
            BiConsumer<AccountsServiceGrpc.AccountsServiceStub, StreamObserver<U>> call,
            Function<U, T> responseMapper, @Nullable OperationLog.RequestStage<?> log,
            BiConsumer<Long, Boolean> metricConsumer) {
        return AsyncMetrics.metric(Mono.fromSupplier(RequestIdHolder::new)
                .flatMap(requestIdHolder -> GrpcClient.oneToOne(Mono.just(stub),
                (AccountsServiceGrpc.AccountsServiceStub s, StreamObserver<U> observer) -> call
                        .accept(prepareCallOptions(s, tvmTicket, requestIdSupplier, requestIdHolder), observer),
                CallOptions.DEFAULT)
                .doOnNext(r -> {
                    if (log != null) {
                        OperationLog<?, U> operationLog = new OperationLog.ResponseStage<>(log,
                                requestIdSupplier.getLastId(),
                                Response.success(r, requestIdHolder.getRequestId()))
                                .buildLog();
                        operationLog.writeTo(LOG);
                        if (operationLog.isLogRequest() && operationLog.getOperationId() != null) {
                            operationsRequestLogService.register(
                                    operationLog.getTenantId(), operationLog.getOperationId(),
                                    operationLog.getRequestId(), operationLog.getRequest(), responseMapper.apply(r)
                            );
                        }
                    }
                })
                .doOnError(error -> {
                    if (log != null) {
                        OperationLog<?, ?> operationLog = new OperationLog.ResponseStage<>(log,
                                requestIdSupplier.getLastId(),
                                toResponse(error))
                                .buildLog();
                        operationLog.writeTo(LOG);
                        if (operationLog.isLogRequest() && operationLog.getOperationId() != null) {
                            operationsRequestLogService.registerError(
                                    operationLog.getTenantId(), operationLog.getOperationId(),
                                    operationLog.getRequestId(), operationLog.getRequest(), error
                            );
                        }
                    }
                })
                .map(r -> Tuples.of(r, requestIdHolder)))
                .timeout(Duration.ofMillis(timeoutMillis)), metricConsumer)
                .retryWhen(retryRequest())
                .map(v -> Response.success(responseMapper.apply(v.getT1()), v.getT2().getRequestId()))
                .onErrorResume(error -> Mono.just(toResponse(error)));
    }

    private AccountsServiceGrpc.AccountsServiceStub prepareCallOptions(AccountsServiceGrpc.AccountsServiceStub s,
                                                                       String tvmTicket,
                                                                       RequestIdSupplier requestIdSupplier,
                                                                       RequestIdHolder requestIdHolder) {
        return s.withMaxInboundMessageSize((int) responseSizeLimitBytes)
                .withDeadlineAfter(deadlineAfterMillis, TimeUnit.MILLISECONDS)
                .withCallCredentials(grpcCallCredentialsSupplier.getCredentials(tvmTicket))
                .withOption(RequestIdInterceptor.REQUEST_ID_KEY, requestIdSupplier)
                .withOption(RequestIdInterceptor.REQUEST_ID_HOLDER_KEY, requestIdHolder);
    }

    private Mono<AccountsServiceGrpc.AccountsServiceStub> getGrpcClient(ProviderModel provider) {
        return grpcStubFactory.getClient(AccountsServiceGrpc::newStub, provider,
                AccountsServiceGrpc.AccountsServiceStub.class);
    }

    private <T> Mono<Result<T>> noUriError(Locale locale) {
        return Mono.just(Result.failure(ErrorCollection.builder().addError(TypedError.invalid(messages
                .getMessage("errors.no.provider.endpoint", null, locale))).build()));
    }

    private Mono<Result<String>> getProviderTicket(ProviderModel provider, Locale locale) {
        return providerAuthSupplier.getTicket(provider.getDestinationTvmId())
                .map(Result::success)
                .onErrorResume(e -> {
                    LOG.error("Failed to get TVM ticket for provider " + provider.getKey(), e);
                    return Mono.just(Result.failure(ErrorCollection.builder().addError(TypedError.invalid(messages
                            .getMessage("errors.can.not.obtain.tvm.ticket.for.provider", null, locale))).build()));
                });
    }

    private Retry retryRequest() {
        return RetrySpec.backoff(maxAttempts, Duration.ofMillis(minBackoffMillis)).filter(e -> {
            if (e instanceof StatusRuntimeException grpcException) {
                return ProviderError.isRetryableCode(grpcException.getStatus().getCode());
            }
            if (e instanceof StatusException grpcException) {
                return ProviderError.isRetryableCode(grpcException.getStatus().getCode());
            }
            return !Exceptions.isRetryExhausted(e);
        });
    }

    private <T> Response<T> toResponse(Throwable ex) {
        if (ex instanceof StatusRuntimeException) {
            return toResponse((StatusRuntimeException) ex);
        }
        if (ex.getCause() instanceof StatusRuntimeException) {
            return toResponse((StatusRuntimeException) ex.getCause());
        }
        if (ex instanceof StatusException) {
            return toResponse((StatusException) ex);
        }
        if (ex.getCause() instanceof StatusException) {
            return toResponse((StatusException) ex.getCause());
        }
        if (Exceptions.isRetryExhausted(ex) && ex.getCause() != null) {
            return Response.failure(ex.getCause());
        }
        return Response.failure(ex);
    }

    private <T> Response<T> toResponse(StatusRuntimeException ex) {
        String requestId = null;
        if (ex.getTrailers() != null && ex.getTrailers().containsKey(REQUEST_ID_KEY)) {
            requestId = ex.getTrailers().get(REQUEST_ID_KEY);
        }
        io.grpc.Status status = ex.getStatus();
        Status statusProto = StatusProto.fromThrowable(ex);
        return toResponse(requestId, status, statusProto);
    }

    private <T> Response<T> toResponse(StatusException ex) {
        String requestIdSupplier = null;
        if (ex.getTrailers() != null && ex.getTrailers().containsKey(REQUEST_ID_KEY)) {
            requestIdSupplier = ex.getTrailers().get(REQUEST_ID_KEY);
        }
        io.grpc.Status status = ex.getStatus();
        Status statusProto = StatusProto.fromThrowable(ex);
        return toResponse(requestIdSupplier, status, statusProto);
    }

    private <T> Response<T> toResponse(String requestId, io.grpc.Status status, Status statusProto) {
        List<BadRequest> badRequests = statusProto != null
                ? statusProto.getDetailsList().stream().filter(any -> any.is(BadRequest.class))
                .map(this::unpackBadRequest).toList()
                : Collections.emptyList();
        if (!badRequests.isEmpty()) {
            Map<String, String> badRequestDetails = new HashMap<>();
            badRequests.forEach(r -> r.getFieldViolationsList()
                    .forEach(v -> badRequestDetails.put(v.getField(), v.getDescription())));
            return Response.error(ProviderError.grpcExtendedError(status.getCode(),
                    status.getDescription(), badRequestDetails), requestId);
        }
        return Response.error(ProviderError.grpcError(status.getCode(),
                status.getDescription()), requestId);
    }

    private BadRequest unpackBadRequest(Any any) {
        try {
            return any.unpack(BadRequest.class);
        } catch (InvalidProtocolBufferException e) {
            throw new UncheckedIOException(e);
        }
    }

    private UpdateProvisionRequest toUpdateProvisionRequest(
            UpdateProvisionRequestDto request, String accountId, String providerId) {
        UpdateProvisionRequest.Builder builder = UpdateProvisionRequest.newBuilder();
        builder.setAccountId(accountId);
        builder.setFolderId(request.getFolderId());
        builder.setAbcServiceId(request.getAbcServiceId());
        builder.setAuthor(toUserId(request.getAuthor()));
        builder.setOperationId(request.getOperationId());
        builder.setProviderId(providerId);
        request.getUpdatedProvisions().forEach(v -> builder.addUpdatedProvisions(toProvisionRequest(v)));
        request.getKnownProvisions().forEach(v -> builder.addKnownProvisions(toKnownAccountProvisions(v)));
        request.getAccountsSpaceKey().ifPresent(key -> builder.setAccountsSpaceKey(toAccountsSpaceKey(key)));
        return builder.build();
    }

    private AccountsSpaceKey toAccountsSpaceKey(AccountsSpaceKeyRequestDto key) {
        AccountsSpaceKey.Builder builder = AccountsSpaceKey.newBuilder();
        key.getSegmentation().ifPresent(segmentation -> {
            CompoundAccountsSpaceKey.Builder keyBuilder = CompoundAccountsSpaceKey.newBuilder();
            segmentation.forEach(segment -> keyBuilder.addResourceSegmentKeys(toResourceSegmentKey(segment)));
            builder.setCompoundKey(keyBuilder.build());
        });
        return builder.build();
    }

    private ResourceSegmentKey toResourceSegmentKey(SegmentKeyRequestDto segmentKey) {
        ResourceSegmentKey.Builder builder = ResourceSegmentKey.newBuilder();
        builder.setResourceSegmentationKey(segmentKey.getSegmentationKey());
        builder.setResourceSegmentKey(segmentKey.getSegmentKey());
        return builder.build();
    }

    private UserID toUserId(UserIdDto userId) {
        UserID.Builder builder = UserID.newBuilder();
        userId.getPassportUid().ifPresent(uid -> builder
                .setPassportUid(PassportUID.newBuilder().setPassportUid(uid).build()));
        userId.getStaffLogin().ifPresent(login -> builder
                .setStaffLogin(StaffLogin.newBuilder().setStaffLogin(login).build()));
        return builder.build();
    }

    private ResourceKey toResourceKey(ResourceKeyRequestDto resourceKey) {
        ResourceKey.Builder builder = ResourceKey.newBuilder();
        if (resourceKey.getResourceTypeKey().isPresent() && resourceKey.getSegmentation().isPresent()) {
            CompoundResourceKey.Builder keyBuilder = CompoundResourceKey.newBuilder();
            keyBuilder.setResourceTypeKey(resourceKey.getResourceTypeKey().get());
            resourceKey.getSegmentation().get()
                    .forEach(segmentKey -> keyBuilder.addResourceSegmentKeys(toResourceSegmentKey(segmentKey)));
            builder.setCompoundKey(keyBuilder.build());
        }
        return builder.build();
    }

    private ProvisionRequest toProvisionRequest(ProvisionRequestDto request) {
        ProvisionRequest.Builder builder = ProvisionRequest.newBuilder();
        builder.setResourceKey(toResourceKey(request.getResourceKey()));
        builder.setProvided(Amount.newBuilder()
                .setValue(request.getProvidedAmount())
                .setUnitKey(request.getProvidedAmountUnitKey())
                .build());
        return builder.build();
    }

    private KnownAccountProvisions toKnownAccountProvisions(KnownAccountProvisionsDto provisions) {
        KnownAccountProvisions.Builder builder = KnownAccountProvisions.newBuilder();
        builder.setAccountId(provisions.getAccountId());
        provisions.getKnownProvisions().forEach(v -> builder.addKnownProvisions(toKnownProvision(v)));
        return builder.build();
    }

    private KnownProvision toKnownProvision(KnownProvisionDto provision) {
        KnownProvision.Builder builder = KnownProvision.newBuilder();
        builder.setResourceKey(toResourceKey(provision.getResourceKey()));
        builder.setProvided(Amount.newBuilder()
                .setValue(provision.getProvidedAmount())
                .setUnitKey(provision.getProvidedAmountUnitKey())
                .build());
        return builder.build();
    }

    @Nullable
    private AccountsSpaceKeyResponseDto toAccountsSpaceKeyResponse(AccountsSpaceKey key) {
        List<SegmentKeyResponseDto> segmentation = null;
        if (key.hasCompoundKey()) {
            CompoundAccountsSpaceKey compoundKey = key.getCompoundKey();
            segmentation = compoundKey.getResourceSegmentKeysList().stream().map(this::toSegmentKeyResponse)
                    .collect(Collectors.toList());
        }
        if (segmentation == null || segmentation.isEmpty()) {
            return null;
        }
        return new AccountsSpaceKeyResponseDto(segmentation);
    }

    private SegmentKeyResponseDto toSegmentKeyResponse(ResourceSegmentKey key) {
        return new SegmentKeyResponseDto(key.getResourceSegmentationKey(), key.getResourceSegmentKey());
    }

    private UpdateProvisionResponseDto toUpdateProvisionResponse(UpdateProvisionResponse response) {
        List<ProvisionDto> provisions = response.getProvisionsList().stream().map(this::toProvision)
                .collect(Collectors.toList());
        Long accountVersion = response.hasAccountVersion() ? response.getAccountVersion().getVersion() : null;
        AccountsSpaceKeyResponseDto accountsSpaceKey = response.hasAccountsSpaceKey()
                ? toAccountsSpaceKeyResponse(response.getAccountsSpaceKey()) : null;
        return new UpdateProvisionResponseDto(provisions, accountVersion, accountsSpaceKey);
    }

    private ResourceKeyResponseDto toResourceKeyResponse(ResourceKey key) {
        String resourceTypeKey = null;
        List<SegmentKeyResponseDto> segmentation = null;
        if (key.hasCompoundKey()) {
            resourceTypeKey = key.getCompoundKey().getResourceTypeKey();
            segmentation = key.getCompoundKey().getResourceSegmentKeysList().stream().map(this::toSegmentKeyResponse)
                    .collect(Collectors.toList());
        }
        return new ResourceKeyResponseDto(resourceTypeKey, segmentation);
    }

    private ProvisionDto toProvision(Provision provision) {
        return new ProvisionDto(toResourceKeyResponse(provision.getResourceKey()),
                provision.hasProvided() ? provision.getProvided().getValue() : null,
                provision.hasProvided() ? provision.getProvided().getUnitKey() : null,
                provision.hasAllocated() ? provision.getAllocated().getValue() : null,
                provision.hasAllocated() ? provision.getAllocated().getUnitKey() : null,
                provision.hasLastUpdate() ? toLastUpdate(provision.getLastUpdate()) : null,
                provision.hasVersion() ? provision.getVersion().getVersion() : null);
    }

    private LastUpdateDto toLastUpdate(LastUpdate lastUpdate) {
        return new LastUpdateDto(lastUpdate.hasTimestamp() ? Timestamps.toMillis(lastUpdate.getTimestamp()) : null,
                lastUpdate.hasAuthor() ? toUserId(lastUpdate.getAuthor()) : null,
                lastUpdate.getOperationId());
    }

    private UserIdDto toUserId(UserID userId) {
        return new UserIdDto(userId.hasPassportUid() ? userId.getPassportUid().getPassportUid() : null,
                userId.hasStaffLogin() ? userId.getStaffLogin().getStaffLogin() : null);
    }

    private GetAccountRequest toGetAccountRequest(String providerId, String accountId, GetAccountRequestDto request) {
        GetAccountRequest.Builder builder = GetAccountRequest.newBuilder();
        builder.setAccountId(accountId);
        builder.setWithProvisions(request.isWithProvisions());
        builder.setIncludeDeleted(request.getIncludeDeleted().orElse(false));
        builder.setFolderId(request.getFolderId());
        builder.setAbcServiceId(request.getAbcServiceId());
        builder.setProviderId(providerId);
        request.getAccountsSpaceKey().ifPresent(key -> builder.setAccountsSpaceKey(toAccountsSpaceKey(key)));
        return builder.build();
    }

    private AccountDto toAccount(Account account) {
        List<ProvisionDto> provisions = account.getProvisionsList().stream().map(this::toProvision)
                .collect(Collectors.toList());
        Long accountVersion = account.hasVersion() ? account.getVersion().getVersion() : null;
        AccountsSpaceKeyResponseDto accountsSpaceKey = account.hasAccountsSpaceKey()
                ? toAccountsSpaceKeyResponse(account.getAccountsSpaceKey()) : null;
        LastUpdateDto lastUpdate = account.hasLastUpdate() ? toLastUpdate(account.getLastUpdate()) : null;
        return new AccountDto(account.getAccountId(), account.getKey(), account.getDisplayName(),
                account.getFolderId(), account.getDeleted(), provisions, accountVersion, accountsSpaceKey,
                lastUpdate, account.getFreeTier());
    }

    private CreateAccountRequest toCreateAccountRequest(CreateAccountRequestDto request, String providerId) {
        CreateAccountRequest.Builder builder = CreateAccountRequest.newBuilder();
        builder.setFolderId(request.getFolderId());
        request.getKey().ifPresent(builder::setKey);
        request.getDisplayName().ifPresent(builder::setDisplayName);
        builder.setAbcServiceId(request.getAbcServiceId());
        builder.setAuthor(toUserId(request.getAuthor()));
        request.getOperationId().ifPresent(builder::setOperationId);
        builder.setProviderId(providerId);
        request.getAccountsSpaceKey().ifPresent(key -> builder.setAccountsSpaceKey(toAccountsSpaceKey(key)));
        request.getFreeTier().ifPresent(builder::setFreeTier);
        builder.setAbcServiceSlug(request.getAbcServiceSlug());
        return builder.build();
    }

    private CreateAccountAndProvideRequest toCreateAccountAndProvideRequest(CreateAccountAndProvideRequestDto request,
                                                                            String providerId) {
        List<ProvisionRequest> provisions = request.getProvisions()
                .stream().map(this::toProvisionRequest).collect(Collectors.toList());
        List<KnownAccountProvisions> knownProvisions = request
                .getKnownProvisions().stream().map(this::toKnownAccountProvisions).collect(Collectors.toList());
        CreateAccountAndProvideRequest.Builder builder = CreateAccountAndProvideRequest.newBuilder();
        builder.setFolderId(request.getFolderId());
        builder.setKey(request.getKey());
        builder.setDisplayName(request.getDisplayName());
        builder.setAbcServiceId(request.getAbcServiceId());
        builder.setAuthor(toUserId(request.getAuthor()));
        builder.setOperationId(request.getOperationId());
        builder.setProviderId(providerId);
        builder.addAllNewProvisions(provisions);
        builder.addAllKnownProvisions(knownProvisions);
        request.getAccountsSpaceKey().ifPresent(key -> builder.setAccountsSpaceKey(toAccountsSpaceKey(key)));
        request.getFreeTier().ifPresent(builder::setFreeTier);
        builder.setAbcServiceSlug(request.getAbcServiceSlug());
        return builder.build();
    }

    private DeleteAccountRequest toDeleteAccountRequest(DeleteAccountRequestDto request, String providerId,
                                                        String accountId) {
        DeleteAccountRequest.Builder builder = DeleteAccountRequest.newBuilder();
        builder.setAccountId(accountId);
        builder.setFolderId(request.getFolderId());
        builder.setAbcServiceId(request.getAbcServiceId());
        builder.setAuthor(toUserId(request.getAuthor()));
        builder.setOperationId(request.getOperationId());
        builder.setProviderId(providerId);
        request.getAccountsSpaceKey().ifPresent(key -> builder.setAccountsSpaceKey(toAccountsSpaceKey(key)));
        return builder.build();
    }

    private RenameAccountRequest toRenameAccountRequest(RenameAccountRequestDto request, String providerId,
                                                        String accountId) {
        RenameAccountRequest.Builder builder = RenameAccountRequest.newBuilder();
        builder.setAccountId(accountId);
        builder.setDisplayName(request.getDisplayName());
        builder.setFolderId(request.getFolderId());
        builder.setAbcServiceId(request.getAbcServiceId());
        builder.setAuthor(toUserId(request.getAuthor()));
        builder.setOperationId(request.getOperationId());
        builder.setProviderId(providerId);
        request.getAccountsSpaceKey().ifPresent(key -> builder.setAccountsSpaceKey(toAccountsSpaceKey(key)));
        return builder.build();
    }

    private MoveAccountRequest toMoveAccountRequest(MoveAccountRequestDto request, String providerId,
                                                    String accountId) {
        List<KnownAccountProvisions> knownSourceProvisions = request
                .getKnownSourceProvisions().stream().map(this::toKnownAccountProvisions).collect(Collectors.toList());
        List<KnownAccountProvisions> knownDstProvisions = request
                .getKnownDestinationProvisions().stream().map(this::toKnownAccountProvisions)
                .collect(Collectors.toList());
        MoveAccountRequest.Builder builder = MoveAccountRequest.newBuilder();
        builder.setAccountId(accountId);
        builder.setSourceFolderId(request.getSourceFolderId());
        builder.setDestinationFolderId(request.getDestinationFolderId());
        builder.setSourceAbcServiceId(request.getSourceAbcServiceId());
        builder.setDestinationAbcServiceId(request.getDestinationAbcServiceId());
        builder.addAllKnownSourceProvisions(knownSourceProvisions);
        builder.addAllKnownDestinationProvisions(knownDstProvisions);
        builder.setAuthor(toUserId(request.getAuthor()));
        builder.setOperationId(request.getOperationId());
        builder.setProviderId(providerId);
        request.getAccountsSpaceKey().ifPresent(key -> builder.setAccountsSpaceKey(toAccountsSpaceKey(key)));
        return builder.build();
    }

    private MoveProvisionRequest toMoveProvisionRequest(
            MoveProvisionRequestDto request, String providerId, String sourceAccountId) {
        List<KnownAccountProvisions> knownSourceProvisions = request
                .getKnownSourceProvisions().stream().map(this::toKnownAccountProvisions).collect(Collectors.toList());
        List<KnownAccountProvisions> knownDstProvisions = request
                .getKnownDestinationProvisions().stream().map(this::toKnownAccountProvisions)
                .collect(Collectors.toList());
        List<ProvisionRequest> sourceProvisions = request
                .getUpdatedSourceProvisions().stream().map(this::toProvisionRequest).collect(Collectors.toList());
        List<ProvisionRequest> destinationProvisions = request
                .getUpdatedDestinationProvisions().stream().map(this::toProvisionRequest).collect(Collectors.toList());
        MoveProvisionRequest.Builder builder = MoveProvisionRequest.newBuilder();
        builder.setSourceAccountId(sourceAccountId);
        builder.setDestinationAccountId(request.getDestinationAccountId());
        builder.setSourceFolderId(request.getSourceFolderId());
        builder.setDestinationFolderId(request.getDestinationFolderId());
        builder.setSourceAbcServiceId(request.getSourceAbcServiceId());
        builder.setDestinationAbcServiceId(request.getDestinationAbcServiceId());
        builder.addAllKnownSourceProvisions(knownSourceProvisions);
        builder.addAllKnownDestinationProvisions(knownDstProvisions);
        builder.addAllUpdatedSourceProvisions(sourceProvisions);
        builder.addAllUpdatedDestinationProvisions(destinationProvisions);
        builder.setAuthor(toUserId(request.getAuthor()));
        builder.setOperationId(request.getOperationId());
        builder.setProviderId(providerId);
        request.getAccountsSpaceKey().ifPresent(key -> builder.setAccountsSpaceKey(toAccountsSpaceKey(key)));
        return builder.build();
    }

    private MoveProvisionResponseDto toMoveProvisionResponse(MoveProvisionResponse response) {
        List<ProvisionDto> sourceProvisions = response.getSourceProvisionsList().stream().map(this::toProvision)
                .collect(Collectors.toList());
        List<ProvisionDto> destinationProvisions = response.getDestinationProvisionsList().stream()
                .map(this::toProvision).collect(Collectors.toList());
        AccountsSpaceKeyResponseDto accountsSpaceKey = response.hasAccountsSpaceKey()
                ? toAccountsSpaceKeyResponse(response.getAccountsSpaceKey()) : null;
        Long sourceAccountVersion = response.hasSourceAccountVersion()
                ? response.getSourceAccountVersion().getVersion() : null;
        Long destinationAccountVersion = response.hasDestinationAccountVersion()
                ? response.getDestinationAccountVersion().getVersion() : null;
        return new MoveProvisionResponseDto(sourceProvisions, destinationProvisions, accountsSpaceKey,
                sourceAccountVersion, destinationAccountVersion);
    }

    private ListAccountsRequest toListAccountsRequest(String providerId, ListAccountsRequestDto request) {
        ListAccountsRequest.Builder builder = ListAccountsRequest.newBuilder();
        builder.setLimit(request.getLimit());
        request.getPageToken().ifPresent(token -> builder
                .setPageToken(AccountsPageToken.newBuilder().setToken(token).build()));
        builder.setWithProvisions(request.isWithProvisions());
        builder.setIncludeDeleted(request.getIncludeDeleted().orElse(false));
        builder.setProviderId(providerId);
        request.getAccountsSpaceKey().ifPresent(key -> builder.setAccountsSpaceKey(toAccountsSpaceKey(key)));
        return builder.build();
    }

    private ListAccountsResponseDto toListAccountsResponse(ListAccountsResponse response) {
        List<AccountDto> accounts = response.getAccountsList().stream().map(this::toAccount)
                .collect(Collectors.toList());
        String nextPageToken = response.hasNextPageToken() ? response.getNextPageToken().getToken() : null;
        return new ListAccountsResponseDto(accounts, nextPageToken);
    }

    private ListAccountsByFolderRequest toListAccountsByFolderRequest(String providerId,
                                                                      ListAccountsByFolderRequestDto request) {
        ListAccountsByFolderRequest.Builder builder = ListAccountsByFolderRequest.newBuilder();
        builder.setFolderId(request.getFolderId());
        builder.setLimit(request.getLimit());
        request.getPageToken().ifPresent(token -> builder
                .setPageToken(AccountsPageToken.newBuilder().setToken(token).build()));
        builder.setWithProvisions(request.isWithProvisions());
        builder.setIncludeDeleted(request.getIncludeDeleted().orElse(false));
        builder.setAbcServiceId(request.getAbcServiceId());
        builder.setProviderId(providerId);
        request.getAccountsSpaceKey().ifPresent(key -> builder.setAccountsSpaceKey(toAccountsSpaceKey(key)));
        return builder.build();
    }

    private ListAccountsResponseDto toListAccountsByFolderResponse(ListAccountsByFolderResponse response) {
        List<AccountDto> accounts = response.getAccountsList().stream().map(this::toAccount)
                .collect(Collectors.toList());
        String nextPageToken = response.hasNextPageToken() ? response.getNextPageToken().getToken() : null;
        return new ListAccountsResponseDto(accounts, nextPageToken);
    }

    private RevokeFreeTierRequest toRevokeFreeTierRequest(
            RevokeFreeTierRequestDto request, String providerId, String accountId
    ) {
        RevokeFreeTierRequest.Builder builder = RevokeFreeTierRequest.newBuilder();
        builder.setAccountId(accountId);
        builder.setFolderId(request.getFolderId());
        builder.setAbcServiceId(request.getAbcServiceId());
        builder.setAuthor(toUserId(request.getAuthor()));
        builder.setOperationId(request.getOperationId());
        builder.setProviderId(providerId);
        request.getAccountsSpaceKey().ifPresent(key -> builder.setAccountsSpaceKey(toAccountsSpaceKey(key)));
        return builder.build();
    }

    private <T> Mono<Result<Response<T>>> selectImplementation(
            ProviderModel provider, Locale locale,
            BiFunction<AccountsServiceGrpc.AccountsServiceStub, String, Mono<Response<T>>> grpcImpl,
            BiFunction<String, String, Mono<Response<T>>> restImpl) {
        if (provider.getGrpcApiUri().isPresent()) {
            return getGrpcClient(provider)
                    .flatMap(stub -> getProviderTicket(provider, locale)
                            .flatMap(ticketR -> ticketR
                                    .andThenMono(ticket -> grpcImpl.apply(stub, ticket).map(Result::success))));
        } else if (provider.getRestApiUri().isPresent()) {
            String baseUrl = provider.getRestApiUri().get();
            return getProviderTicket(provider, locale)
                    .flatMap(ticketR -> ticketR
                            .andThenMono(ticket -> restImpl.apply(baseUrl, ticket).map(Result::success)));
        } else {
            return noUriError(locale);
        }
    }
}
