package ru.yandex.intranet.d.services.integration.providers

import java.util.function.BiFunction
import java.util.function.Function

/**
 * Response.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
sealed interface Response<T> {
    fun <R> match(cases: Cases<T, R>): R
    fun <R> match(onSuccess: BiFunction<T, String?, R>, onFailure: Function<Throwable, R>,
                  onError: BiFunction<ProviderError, String?, R>): R
    fun isSuccess(): Boolean
    fun isFailure(): Boolean
    fun isError(): Boolean
    fun requestId(): String?
    suspend fun <R> matchSuspend(onSuccess: suspend (result: T, requestId: String?) -> R,
                                 onFailure: suspend (error: Throwable) -> R,
                                 onError: suspend (error: ProviderError, requestId: String?) -> R): R
    interface Cases<I, O> {
        fun success(result: I, requestId: String?): O
        fun failure(error: Throwable): O
        fun error(error: ProviderError, requestId: String?): O
    }
    companion object {
        @JvmStatic
        fun <I> success(result: I, requestId: String?): Response<I> = SuccessResponse(result, requestId)
        @JvmStatic
        fun <I> failure(error: Throwable): Response<I> = FailureResponse(error)
        @JvmStatic
        fun <I> error(error: ProviderError, requestId: String?): Response<I> = ErrorResponse(error, requestId)
    }
}
private data class SuccessResponse<I>(val result: I, val requestId: String?): Response<I> {
    override fun <R> match(cases: Response.Cases<I, R>): R = cases.success(result, requestId)
    override fun <R> match(onSuccess: BiFunction<I, String?, R>, onFailure: Function<Throwable, R>,
                           onError: BiFunction<ProviderError, String?, R>): R = onSuccess.apply(result, requestId)
    override fun isSuccess(): Boolean = true
    override fun isFailure(): Boolean = false
    override fun isError(): Boolean = false
    override fun requestId(): String? = requestId
    override suspend fun <R> matchSuspend(onSuccess: suspend (result: I, requestId: String?) -> R,
                                          onFailure: suspend (error: Throwable) -> R,
                                          onError: suspend (error: ProviderError, requestId: String?) -> R): R
            = onSuccess(result, requestId)
}
private data class FailureResponse<I>(val error: Throwable): Response<I> {
    override fun <R> match(cases: Response.Cases<I, R>): R = cases.failure(error)
    override fun <R> match(onSuccess: BiFunction<I, String?, R>, onFailure: Function<Throwable, R>,
                           onError: BiFunction<ProviderError, String?, R>): R = onFailure.apply(error)
    override fun isSuccess(): Boolean = false
    override fun isFailure(): Boolean = true
    override fun isError(): Boolean = false
    override fun requestId(): String? = null
    override suspend fun <R> matchSuspend(onSuccess: suspend (result: I, requestId: String?) -> R,
                                          onFailure: suspend (error: Throwable) -> R,
                                          onError: suspend (error: ProviderError, requestId: String?) -> R): R
        = onFailure(error)
}
private data class ErrorResponse<I>(val error: ProviderError, val requestId: String?): Response<I> {
    override fun <R> match(cases: Response.Cases<I, R>): R = cases.error(error, requestId)
    override fun <R> match(onSuccess: BiFunction<I, String?, R>, onFailure: Function<Throwable, R>,
                           onError: BiFunction<ProviderError, String?, R>): R = onError.apply(error, requestId)
    override fun isSuccess(): Boolean = false
    override fun isFailure(): Boolean = false
    override fun isError(): Boolean = true
    override fun requestId(): String? = requestId
    override suspend fun <R> matchSuspend(onSuccess: suspend (result: I, requestId: String?) -> R,
                                          onFailure: suspend (error: Throwable) -> R,
                                          onError: suspend (error: ProviderError, requestId: String?) -> R): R
        = onError(error, requestId)
}
