package ru.yandex.intranet.d.services.integration.providers.rest;

import java.time.Duration;
import java.util.concurrent.TimeUnit;

import javax.annotation.Nullable;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectReader;
import io.netty.channel.ChannelOption;
import io.netty.handler.timeout.ReadTimeoutHandler;
import io.netty.handler.timeout.WriteTimeoutHandler;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.InvalidMediaTypeException;
import org.springframework.http.MediaType;
import org.springframework.stereotype.Component;
import org.springframework.web.reactive.function.client.ClientResponse;
import org.springframework.web.reactive.function.client.ExchangeFilterFunctions;
import org.springframework.web.reactive.function.client.WebClient;
import org.springframework.web.reactive.function.client.WebClientResponseException;
import org.springframework.web.util.UriComponentsBuilder;
import reactor.core.Exceptions;
import reactor.core.publisher.Mono;
import reactor.netty.http.client.HttpClient;
import reactor.util.retry.Retry;
import reactor.util.retry.RetrySpec;

import ru.yandex.intranet.d.services.integration.providers.IntegrationMetrics;
import ru.yandex.intranet.d.services.integration.providers.OperationLog;
import ru.yandex.intranet.d.services.integration.providers.ProviderError;
import ru.yandex.intranet.d.services.integration.providers.RequestIdSupplier;
import ru.yandex.intranet.d.services.integration.providers.Response;
import ru.yandex.intranet.d.services.integration.providers.rest.model.AccountDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.CreateAccountAndProvideRequestDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.CreateAccountRequestDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.DeleteAccountRequestDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.ErrorMessagesDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.GetAccountRequestDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.ListAccountsByFolderRequestDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.ListAccountsRequestDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.ListAccountsResponseDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.MoveAccountRequestDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.MoveProvisionRequestDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.MoveProvisionResponseDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.RenameAccountRequestDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.RevokeFreeTierRequestDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.UpdateProvisionRequestDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.UpdateProvisionResponseDto;
import ru.yandex.intranet.d.services.operations.OperationsRequestLogService;
import ru.yandex.intranet.d.util.AsyncMetrics;
import ru.yandex.intranet.d.util.ResolverHolder;
import ru.yandex.intranet.d.util.http.YaReactorClientHttpConnector;

/**
 * Providers API client.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class ProvidersClient {

    private static final Logger LOG = LoggerFactory.getLogger(ProvidersClient.class);

    private final String userAgent;
    private final WebClient client;
    private final long maxAttempts;
    private final long minBackoffMillis;
    private final long timeoutMillis;
    private final ObjectReader errorReader;
    private final RestCallCredentialsSupplier callCredentialsSupplier;
    private final IntegrationMetrics integrationMetrics;

    private final OperationsRequestLogService operationsRequestLogService;

    @SuppressWarnings("ParameterNumber")
    public ProvidersClient(@Value("${providers.client.connectTimeoutMs}") int connectTimeoutMillis,
                           @Value("${providers.client.readTimeoutMs}") long readTimeoutMillis,
                           @Value("${providers.client.writeTimeoutMs}") long writeTimeoutMillis,
                           @Value("${providers.client.responseSizeLimitBytes}") long responseSizeLimitBytes,
                           @Value("${providers.client.maxAttempts}") long maxAttempts,
                           @Value("${providers.client.minBackoffMs}") long minBackoffMillis,
                           @Value("${providers.client.timeoutMs}") long timeoutMillis,
                           @Value("${http.client.userAgent}") String userAgent,
                           ObjectMapper objectMapper,
                           RestCallCredentialsSupplier callCredentialsSupplier,
                           IntegrationMetrics integrationMetrics,
                           OperationsRequestLogService operationsRequestLogService) {
        this.userAgent = userAgent;
        this.client = createWebClient(connectTimeoutMillis, readTimeoutMillis, writeTimeoutMillis,
                responseSizeLimitBytes);
        this.maxAttempts = maxAttempts;
        this.minBackoffMillis = minBackoffMillis;
        this.timeoutMillis = timeoutMillis;
        this.errorReader = objectMapper.readerFor(ErrorMessagesDto.class);
        this.callCredentialsSupplier = callCredentialsSupplier;
        this.integrationMetrics = integrationMetrics;
        this.operationsRequestLogService = operationsRequestLogService;
    }

    @SuppressWarnings("ParameterNumber")
    public Mono<Response<UpdateProvisionResponseDto>> updateProvision(String baseUrl,
                                                                      String serviceTicket,
                                                                      RequestIdSupplier requestIdSupplier,
                                                                      String accountId,
                                                                      String providerId,
                                                                      String providerKey,
                                                                      UpdateProvisionRequestDto request,
                                                                      OperationLog.RequestStage<?> log) {
        UriComponentsBuilder builder = UriComponentsBuilder.fromHttpUrl(baseUrl)
                .pathSegment("quotaManagement", "v1", "providers", providerId, "accounts", accountId, "_provide");
        String uri = builder.toUriString();
        return AsyncMetrics.metric(Mono.fromSupplier(requestIdSupplier).flatMap(requestId -> callCredentialsSupplier
                .supplyCredentials(client.post().uri(uri), serviceTicket)
                .header("X-Request-ID", requestId)
                .accept(MediaType.APPLICATION_JSON)
                .header(HttpHeaders.USER_AGENT, userAgent)
                .contentType(MediaType.APPLICATION_JSON)
                .bodyValue(request)
                .exchangeToMono(response -> toResponse(response, UpdateProvisionResponseDto.class, uri)))
                .timeout(Duration.ofMillis(timeoutMillis)),
                        Response::isSuccess,
                        (duration, success) -> integrationMetrics.afterUpdateProvision(providerKey, duration, success))
                .doOnNext(response -> logOnSuccess(response, log, requestIdSupplier))
                .doOnError(error -> logOnError(error, log, requestIdSupplier))
                .retryWhen(retryRequest())
                .onErrorResume(this::toResponseMono);
    }

    @SuppressWarnings("ParameterNumber")
    public Mono<Response<AccountDto>> getAccount(String baseUrl,
                                                 String serviceTicket,
                                                 RequestIdSupplier requestIdSupplier,
                                                 String accountId,
                                                 String providerId,
                                                 String providerKey,
                                                 GetAccountRequestDto request,
                                                 OperationLog.RequestStage<?> log) {
        UriComponentsBuilder builder = UriComponentsBuilder.fromHttpUrl(baseUrl)
                .pathSegment("quotaManagement", "v1", "providers", providerId, "accounts", accountId, "_getOne");
        String uri = builder.toUriString();
        return AsyncMetrics.metric(Mono.fromSupplier(requestIdSupplier).flatMap(requestId -> callCredentialsSupplier
                .supplyCredentials(client.post()
                .uri(uri), serviceTicket)
                .header("X-Request-ID", requestId)
                .accept(MediaType.APPLICATION_JSON)
                .header(HttpHeaders.USER_AGENT, userAgent)
                .contentType(MediaType.APPLICATION_JSON)
                .bodyValue(request)
                .exchangeToMono(response -> toResponse(response, AccountDto.class, uri)))
                .timeout(Duration.ofMillis(timeoutMillis)),
                        Response::isSuccess,
                        (duration, success) -> integrationMetrics.afterGetAccount(providerKey, duration, success))
                .doOnNext(response -> logOnSuccess(response, log, requestIdSupplier))
                .doOnError(error -> logOnError(error, log, requestIdSupplier))
                .retryWhen(retryRequest())
                .onErrorResume(this::toResponseMono);
    }

    @SuppressWarnings("ParameterNumber")
    public Mono<Response<AccountDto>> createAccount(String baseUrl,
                                                    String serviceTicket,
                                                    RequestIdSupplier requestIdSupplier,
                                                    String providerId,
                                                    String providerKey,
                                                    CreateAccountRequestDto request,
                                                    OperationLog.RequestStage<?> log) {
        UriComponentsBuilder builder = UriComponentsBuilder.fromHttpUrl(baseUrl)
                .pathSegment("quotaManagement", "v1", "providers", providerId, "accounts");
        String uri = builder.toUriString();
        return AsyncMetrics.metric(Mono.fromSupplier(requestIdSupplier).flatMap(requestId -> callCredentialsSupplier
                .supplyCredentials(client.post()
                .uri(uri), serviceTicket)
                .header("X-Request-ID", requestId)
                .accept(MediaType.APPLICATION_JSON)
                .header(HttpHeaders.USER_AGENT, userAgent)
                .contentType(MediaType.APPLICATION_JSON)
                .bodyValue(request)
                .exchangeToMono(response -> toResponse(response, AccountDto.class, uri)))
                .timeout(Duration.ofMillis(timeoutMillis)),
                        Response::isSuccess,
                        (duration, success) -> integrationMetrics.afterCreateAccount(providerKey, duration, success))
                .doOnNext(response -> logOnSuccess(response, log, requestIdSupplier))
                .doOnError(error -> logOnError(error, log, requestIdSupplier))
                .retryWhen(retryRequest())
                .onErrorResume(this::toResponseMono);
    }

    @SuppressWarnings("ParameterNumber")
    public Mono<Response<AccountDto>> createAccountAndProvide(String baseUrl,
                                                              String serviceTicket,
                                                              RequestIdSupplier requestIdSupplier,
                                                              String providerId,
                                                              String providerKey,
                                                              CreateAccountAndProvideRequestDto request,
                                                              OperationLog.RequestStage<?> log) {
        UriComponentsBuilder builder = UriComponentsBuilder.fromHttpUrl(baseUrl)
                .pathSegment("quotaManagement", "v1", "providers", providerId, "accounts");
        String uri = builder.toUriString();
        return AsyncMetrics.metric(Mono.fromSupplier(requestIdSupplier).flatMap(requestId -> callCredentialsSupplier
                .supplyCredentials(client.post()
                .uri(uri), serviceTicket)
                .header("X-Request-ID", requestId)
                .accept(MediaType.APPLICATION_JSON)
                .header(HttpHeaders.USER_AGENT, userAgent)
                .contentType(MediaType.APPLICATION_JSON)
                .bodyValue(request)
                .exchangeToMono(response -> toResponse(response, AccountDto.class, uri)))
                .timeout(Duration.ofMillis(timeoutMillis)),
                        Response::isSuccess,
                        (duration, success) -> integrationMetrics.afterCreateAccountAndProvide(providerKey, duration,
                                success))
                .doOnNext(response -> logOnSuccess(response, log, requestIdSupplier))
                .doOnError(error -> logOnError(error, log, requestIdSupplier))
                .retryWhen(retryRequest())
                .onErrorResume(this::toResponseMono);
    }

    @SuppressWarnings("ParameterNumber")
    public Mono<Response<Void>> deleteAccount(String baseUrl,
                                              String serviceTicket,
                                              RequestIdSupplier requestIdSupplier,
                                              String accountId,
                                              String providerId,
                                              String providerKey,
                                              DeleteAccountRequestDto request,
                                              OperationLog.RequestStage<?> log) {
        UriComponentsBuilder builder = UriComponentsBuilder.fromHttpUrl(baseUrl)
                .pathSegment("quotaManagement", "v1", "providers", providerId, "accounts", accountId);
        String uri = builder.toUriString();
        return AsyncMetrics.metric(Mono.fromSupplier(requestIdSupplier).flatMap(requestId -> callCredentialsSupplier
                .supplyCredentials(client.method(HttpMethod.DELETE)
                .uri(uri), serviceTicket)
                .header("X-Request-ID", requestId)
                .accept(MediaType.APPLICATION_JSON)
                .header(HttpHeaders.USER_AGENT, userAgent)
                .contentType(MediaType.APPLICATION_JSON)
                .bodyValue(request)
                .exchangeToMono(response -> this.<Void>toResponse(response, uri)))
                .timeout(Duration.ofMillis(timeoutMillis)),
                        Response::isSuccess,
                        (duration, success) -> integrationMetrics.afterDeleteAccount(providerKey, duration, success))
                .doOnNext(response -> logOnSuccess(response, log, requestIdSupplier))
                .doOnError(error -> logOnError(error, log, requestIdSupplier))
                .retryWhen(retryRequest())
                .onErrorResume(this::toResponseMono);
    }

    @SuppressWarnings("ParameterNumber")
    public Mono<Response<AccountDto>> renameAccount(String baseUrl,
                                                    String serviceTicket,
                                                    RequestIdSupplier requestIdSupplier,
                                                    String accountId,
                                                    String providerId,
                                                    String providerKey,
                                                    RenameAccountRequestDto request,
                                                    OperationLog.RequestStage<?> log) {
        UriComponentsBuilder builder = UriComponentsBuilder.fromHttpUrl(baseUrl)
                .pathSegment("quotaManagement", "v1", "providers", providerId, "accounts", accountId, "_rename");
        String uri = builder.toUriString();
        return AsyncMetrics.metric(Mono.fromSupplier(requestIdSupplier).flatMap(requestId -> callCredentialsSupplier
                .supplyCredentials(client.post()
                .uri(uri), serviceTicket)
                .header("X-Request-ID", requestId)
                .accept(MediaType.APPLICATION_JSON)
                .header(HttpHeaders.USER_AGENT, userAgent)
                .contentType(MediaType.APPLICATION_JSON)
                .bodyValue(request)
                .exchangeToMono(response -> toResponse(response, AccountDto.class, uri)))
                .timeout(Duration.ofMillis(timeoutMillis)),
                        Response::isSuccess,
                        (duration, success) -> integrationMetrics.afterRenameAccount(providerKey, duration, success))
                .doOnNext(response -> logOnSuccess(response, log, requestIdSupplier))
                .doOnError(error -> logOnError(error, log, requestIdSupplier))
                .retryWhen(retryRequest())
                .onErrorResume(this::toResponseMono);
    }

    @SuppressWarnings("ParameterNumber")
    public Mono<Response<AccountDto>> moveAccount(String baseUrl,
                                                  String serviceTicket,
                                                  RequestIdSupplier requestIdSupplier,
                                                  String accountId,
                                                  String providerId,
                                                  String providerKey,
                                                  MoveAccountRequestDto request,
                                                  OperationLog.RequestStage<?> log) {
        UriComponentsBuilder builder = UriComponentsBuilder.fromHttpUrl(baseUrl)
                .pathSegment("quotaManagement", "v1", "providers", providerId, "accounts", accountId, "_move");
        String uri = builder.toUriString();
        return AsyncMetrics.metric(Mono.fromSupplier(requestIdSupplier).flatMap(requestId -> callCredentialsSupplier
                .supplyCredentials(client.post()
                .uri(uri), serviceTicket)
                .header("X-Request-ID", requestId)
                .accept(MediaType.APPLICATION_JSON)
                .header(HttpHeaders.USER_AGENT, userAgent)
                .contentType(MediaType.APPLICATION_JSON)
                .bodyValue(request)
                .exchangeToMono(response -> toResponse(response, AccountDto.class, uri)))
                .timeout(Duration.ofMillis(timeoutMillis)),
                        Response::isSuccess,
                        (duration, success) -> integrationMetrics.afterMoveAccount(providerKey, duration, success))
                .doOnNext(response -> logOnSuccess(response, log, requestIdSupplier))
                .doOnError(error -> logOnError(error, log, requestIdSupplier))
                .retryWhen(retryRequest())
                .onErrorResume(this::toResponseMono);
    }

    @SuppressWarnings("ParameterNumber")
    public Mono<Response<MoveProvisionResponseDto>> moveProvision(String baseUrl,
                                                                  String serviceTicket,
                                                                  RequestIdSupplier requestIdSupplier,
                                                                  String sourceAccountId,
                                                                  String providerId,
                                                                  String providerKey,
                                                                  MoveProvisionRequestDto request,
                                                                  OperationLog.RequestStage<?> log) {
        UriComponentsBuilder builder = UriComponentsBuilder.fromHttpUrl(baseUrl)
                .pathSegment("quotaManagement", "v1", "providers", providerId, "accounts",
                        sourceAccountId, "_moveProvision");
        String uri = builder.toUriString();
        return AsyncMetrics.metric(Mono.fromSupplier(requestIdSupplier).flatMap(requestId -> callCredentialsSupplier
                .supplyCredentials(client.post()
                .uri(uri), serviceTicket)
                .header("X-Request-ID", requestId)
                .accept(MediaType.APPLICATION_JSON)
                .header(HttpHeaders.USER_AGENT, userAgent)
                .contentType(MediaType.APPLICATION_JSON)
                .bodyValue(request)
                .exchangeToMono(response -> toResponse(response, MoveProvisionResponseDto.class, uri)))
                .timeout(Duration.ofMillis(timeoutMillis)),
                        Response::isSuccess,
                        (duration, success) -> integrationMetrics.afterMoveProvision(providerKey, duration, success))
                .doOnNext(response -> logOnSuccess(response, log, requestIdSupplier))
                .doOnError(error -> logOnError(error, log, requestIdSupplier))
                .retryWhen(retryRequest())
                .onErrorResume(this::toResponseMono);
    }

    @SuppressWarnings("ParameterNumber")
    public Mono<Response<ListAccountsResponseDto>> listAccounts(String baseUrl,
                                                                String serviceTicket,
                                                                RequestIdSupplier requestIdSupplier,
                                                                String providerId,
                                                                String providerKey,
                                                                ListAccountsRequestDto request,
                                                                OperationLog.RequestStage<?> log) {
        UriComponentsBuilder builder = UriComponentsBuilder.fromHttpUrl(baseUrl)
                .pathSegment("quotaManagement", "v1", "providers", providerId, "accounts", "_getPage")
                .queryParam("providerId", providerId);
        String uri = builder.toUriString();
        return AsyncMetrics.metric(Mono.fromSupplier(requestIdSupplier).flatMap(requestId -> callCredentialsSupplier
                .supplyCredentials(client.post()
                .uri(uri), serviceTicket)
                .header("X-Request-ID", requestId)
                .accept(MediaType.APPLICATION_JSON)
                .header(HttpHeaders.USER_AGENT, userAgent)
                .contentType(MediaType.APPLICATION_JSON)
                .bodyValue(request)
                .exchangeToMono(response -> toResponse(response, ListAccountsResponseDto.class, uri)))
                .timeout(Duration.ofMillis(timeoutMillis)),
                        Response::isSuccess,
                        (duration, success) -> integrationMetrics.afterListAccounts(providerKey, duration, success))
                .doOnNext(response -> logOnSuccess(response, log, requestIdSupplier))
                .doOnError(error -> logOnError(error, log, requestIdSupplier))
                .retryWhen(retryRequest())
                .onErrorResume(this::toResponseMono);
    }

    @SuppressWarnings("ParameterNumber")
    public Mono<Response<ListAccountsResponseDto>> listAccountsByFolder(String baseUrl,
                                                                        String serviceTicket,
                                                                        RequestIdSupplier requestIdSupplier,
                                                                        String providerId,
                                                                        String providerKey,
                                                                        ListAccountsByFolderRequestDto request,
                                                                        OperationLog.RequestStage<?> log) {
        UriComponentsBuilder builder = UriComponentsBuilder.fromHttpUrl(baseUrl)
                .pathSegment("quotaManagement", "v1", "providers", providerId, "accounts", "_getPage");
        String uri = builder.toUriString();
        return AsyncMetrics.metric(Mono.fromSupplier(requestIdSupplier).flatMap(requestId -> callCredentialsSupplier
                .supplyCredentials(client.post()
                .uri(uri), serviceTicket)
                .header("X-Request-ID", requestId)
                .accept(MediaType.APPLICATION_JSON)
                .header(HttpHeaders.USER_AGENT, userAgent)
                .contentType(MediaType.APPLICATION_JSON)
                .bodyValue(request)
                .exchangeToMono(response -> toResponse(response, ListAccountsResponseDto.class, uri)))
                .timeout(Duration.ofMillis(timeoutMillis)),
                        Response::isSuccess,
                        (duration, success) -> integrationMetrics.afterListAccountsByFolder(providerKey, duration,
                                success))
                .doOnNext(response -> logOnSuccess(response, log, requestIdSupplier))
                .doOnError(error -> logOnError(error, log, requestIdSupplier))
                .retryWhen(retryRequest())
                .onErrorResume(this::toResponseMono);
    }

    @SuppressWarnings("ParameterNumber")
    public Mono<Response<AccountDto>> revokeFreeTier(
            String baseUrl,
            String serviceTicket,
            RequestIdSupplier requestIdSupplier,
            String accountId,
            String providerId,
            String providerKey,
            RevokeFreeTierRequestDto request,
            OperationLog.RequestStage<?> log
    ) {
        UriComponentsBuilder builder = UriComponentsBuilder.fromHttpUrl(baseUrl).pathSegment(
                "quotaManagement", "v1", "providers", providerId, "accounts", accountId, "_revokeFreeTier");
        String uri = builder.toUriString();
        return AsyncMetrics.metric(Mono.fromSupplier(requestIdSupplier).flatMap(requestId -> callCredentialsSupplier
                .supplyCredentials(client.post()
                .uri(uri), serviceTicket)
                .header("X-Request-ID", requestId)
                .accept(MediaType.APPLICATION_JSON)
                .header(HttpHeaders.USER_AGENT, userAgent)
                .contentType(MediaType.APPLICATION_JSON)
                .bodyValue(request)
                .exchangeToMono(response -> toResponse(response, AccountDto.class, uri)))
                .timeout(Duration.ofMillis(timeoutMillis)),
                        Response::isSuccess,
                        (duration, success) -> integrationMetrics.afterRevokeFreeTier(providerKey, duration, success))
                .doOnNext(response -> logOnSuccess(response, log, requestIdSupplier))
                .doOnError(error -> logOnError(error, log, requestIdSupplier))
                .retryWhen(retryRequest())
                .onErrorResume(this::toResponseMono);
    }

    private static WebClient createWebClient(int connectTimeoutMillis,
                                             long readTimeoutMillis,
                                             long writeTimeoutMillis,
                                             long responseSizeLimitBytes) {
        HttpClient httpClient = HttpClient.create()
                .option(ChannelOption.CONNECT_TIMEOUT_MILLIS, connectTimeoutMillis)
                .resolver(ResolverHolder.RESOLVER_INSTANCE)
                .doOnConnected(connection -> {
                    connection.addHandlerLast(new ReadTimeoutHandler(readTimeoutMillis, TimeUnit.MILLISECONDS));
                    connection.addHandlerLast(new WriteTimeoutHandler(writeTimeoutMillis, TimeUnit.MILLISECONDS));
                });
        return WebClient.builder()
                .clientConnector(new YaReactorClientHttpConnector(httpClient))
                .filter(ExchangeFilterFunctions.limitResponseSize(responseSizeLimitBytes))
                .codecs(configurer -> configurer
                        .defaultCodecs()
                        .maxInMemorySize(Math.toIntExact(responseSizeLimitBytes)))
                .build();
    }

    private <T> Mono<Response<T>> toResponse(ClientResponse clientResponse, Class<T> responseClass, String uri) {
        int statusCode = clientResponse.rawStatusCode();
        if (isSuccess(statusCode)) {
            String requestId = clientResponse.headers().asHttpHeaders().getFirst("X-Request-ID");
            return clientResponse.bodyToMono(responseClass).map(value -> Response.success(value, requestId));
        }
        LOG.error("Received {} error code for {} provider API endpoint", statusCode, uri);
        return clientResponse.createException().flatMap(Mono::error);
    }

    private <T> Mono<Response<T>> toResponse(ClientResponse clientResponse, String uri) {
        int statusCode = clientResponse.rawStatusCode();
        if (isSuccess(statusCode)) {
            String requestId = clientResponse.headers().asHttpHeaders().getFirst("X-Request-ID");
            return clientResponse.releaseBody().thenReturn(Response.success(null, requestId));
        }
        LOG.error("Received {} error code for {} provider API endpoint", statusCode, uri);
        return clientResponse.createException().flatMap(Mono::error);
    }

    private <T> Mono<Response<T>> toResponseMono(Throwable error) {
        return Mono.just(toResponse(error));
    }

    private <T> Response<T> toResponse(Throwable error) {
        LOG.error("Provider API client error", error);
        if (error instanceof WebClientResponseException) {
            return toResponse((WebClientResponseException) error);
        }
        if (error.getCause() instanceof WebClientResponseException) {
            return toResponse((WebClientResponseException) error.getCause());
        }
        if (Exceptions.isRetryExhausted(error) && error.getCause() != null) {
            return Response.failure(error.getCause());
        }
        return Response.failure(error);
    }

    private boolean isSuccess(int statusCode) {
        return statusCode >= 200 && statusCode < 300;
    }

    private <T> Response<T> toResponse(WebClientResponseException ex) {
        int statusCode = ex.getRawStatusCode();
        String requestId = ex.getHeaders().getFirst("X-Request-ID");
        if (!(statusCode >= 400 && statusCode < 500) || !isJsonContent(ex)) {
            return Response.error(ProviderError.httpError(statusCode), requestId);
        }
        try {
            ErrorMessagesDto errorMessages = errorReader.readValue(ex.getResponseBodyAsString());
            return Response.error(ProviderError.httpExtendedError(statusCode, errorMessages), requestId);
        } catch (Exception e) {
            LOG.error("Failed to read provider API error", e);
            return Response.error(ProviderError.httpError(statusCode), requestId);
        }
    }

    private boolean isJsonContent(WebClientResponseException ex) {
        try {
            MediaType contentType = ex.getHeaders().getContentType();
            if (contentType == null) {
                return false;
            }
            return MediaType.APPLICATION_JSON.includes(contentType);
        } catch (InvalidMediaTypeException e) {
            LOG.error("Unexpected provider API response content type", e);
            return false;
        }
    }

    private Retry retryRequest() {
        return RetrySpec.backoff(maxAttempts, Duration.ofMillis(minBackoffMillis)).filter(e -> {
            if (e instanceof WebClientResponseException) {
                WebClientResponseException webException = (WebClientResponseException) e;
                return ProviderError.isRetryableCode(webException.getRawStatusCode());
            }
            return !Exceptions.isRetryExhausted(e);
        });
    }

    private <T> void logOnSuccess(Response<T> response,
                              @Nullable OperationLog.RequestStage<?> log,
                              RequestIdSupplier requestIdSupplier) {
        if (log != null) {
            OperationLog<?, T> operationLog = new OperationLog.ResponseStage<>(log,
                    requestIdSupplier.getLastId(), response).buildLog();
            operationLog.writeTo(LOG);
            if (operationLog.isLogRequest() && operationLog.getOperationId() != null) {
                operationsRequestLogService.register(
                        operationLog.getTenantId(), operationLog.getOperationId(),
                        operationLog.getRequestId(), operationLog.getRequest(), operationLog.getResponse()
                );
            }
        }
    }

    private void logOnError(Throwable error,
                            @Nullable OperationLog.RequestStage<?> log,
                            RequestIdSupplier requestIdSupplier) {
        if (log != null) {
            OperationLog<?, ?> operationLog = new OperationLog.ResponseStage<>(log,
                    requestIdSupplier.getLastId(),
                    toResponse(error)).buildLog();
            operationLog.writeTo(LOG);
            if (operationLog.isLogRequest() && operationLog.getOperationId() != null) {
                operationsRequestLogService.registerError(
                        operationLog.getTenantId(), operationLog.getOperationId(),
                        operationLog.getRequestId(), operationLog.getRequest(), error
                );
            }
        }
    }

}
