package ru.yandex.intranet.d.services.integration.yt

import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.beans.factory.annotation.Value
import org.springframework.context.annotation.Bean
import org.springframework.context.annotation.Configuration
import org.springframework.context.annotation.Profile
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor
import ru.yandex.intranet.d.util.MdcTaskDecorator
import ru.yandex.yt.ytclient.proxy.YtClient
import ru.yandex.yt.ytclient.rpc.RpcCompression
import ru.yandex.yt.ytclient.rpc.RpcCredentials
import ru.yandex.yt.ytclient.rpc.internal.Compression
import java.time.Duration

/**
 * YT reader implementation.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Configuration
open class YtConfiguration {

    @Bean("ytClientHahn")
    @Profile("dev", "testing", "production")
    open fun ytClientHahn(
        @Qualifier("ytClientExecutor") ytClientExecutor: ThreadPoolTaskExecutor,
        @Value("\${yt.client.hahn.clusterName}") clusterName: String,
        @Value("\${yt.client.hahn.user}") user: String,
        @Value("\${yt.client.hahn.token}") token: String
    ): YtClient {
        return YtClient.builder()
            .setCluster(clusterName)
            .setRpcCompression(RpcCompression(Compression.Lz4))
            .setRpcCredentials(RpcCredentials(user, token))
            .setHeavyExecutor(ytClientExecutor)
            .build()
    }

    @Bean("ytReaderHahn")
    @Profile("dev", "testing", "production")
    open fun hahnReaderImpl(@Qualifier("ytClientHahn") ytClient: YtClient,
                            @Value("\${yt.client.userAgent}") userAgent: String,
                            @Value("\${yt.client.hahn.requestTimeout}") requestTimeout: Duration,
                            @Value("\${yt.client.hahn.retries}") retries: Int,
                            @Value("\${yt.client.hahn.retryInitialInterval}") retryInitialInterval: Duration,
                            @Value("\${yt.client.hahn.retryMultiplier}") retryMultiplier: Double,
                            @Value("\${yt.client.hahn.retryRandomizationFactor}") retryRandomizationFactor: Double): YtReader {
        return YtReaderImpl(ytClient, userAgent, requestTimeout, retries, "ytReaderHahnRetries",
            retryInitialInterval.toMillis(), retryMultiplier, retryRandomizationFactor)
    }

    @Bean("ytReaderHahn")
    @Profile("local", "load-testing", "integration-tests", "test-recipe")
    open fun hahnReaderStub(): YtReader {
        return YtReaderStub()
    }

    @Bean("ytClientExecutor")
    @Profile("dev", "testing", "production")
    open fun ytClientExecutor(
        @Value("\${yt.client.executorShutdownTimeoutMs}") executorShutdownTimeoutMs: Long
    ): ThreadPoolTaskExecutor {
        val executor = ThreadPoolTaskExecutor()
        executor.setQueueCapacity(10000)
        executor.maxPoolSize = Runtime.getRuntime().availableProcessors()
        executor.corePoolSize = (Runtime.getRuntime().availableProcessors() / 3).coerceAtLeast(1)
        executor.isDaemon = true
        executor.threadNamePrefix = "yt-client-executor-"
        executor.setAwaitTerminationMillis(executorShutdownTimeoutMs)
        executor.setWaitForTasksToCompleteOnShutdown(true)
        executor.setTaskDecorator(MdcTaskDecorator())
        return executor
    }

}
