package ru.yandex.intranet.d.services.integration.yt

import io.github.resilience4j.core.IntervalFunction
import io.github.resilience4j.kotlin.retry.executeSuspendFunction
import io.github.resilience4j.kotlin.retry.retry
import io.github.resilience4j.retry.Retry
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.asFlow
import kotlinx.coroutines.flow.emitAll
import kotlinx.coroutines.flow.flow
import kotlinx.coroutines.future.await
import mu.KotlinLogging
import ru.yandex.inside.yt.kosher.common.GUID
import ru.yandex.inside.yt.kosher.cypress.YPath
import ru.yandex.inside.yt.kosher.impl.ytree.`object`.serializers.YTreeObjectSerializerFactory
import ru.yandex.intranet.d.util.buildRetry
import ru.yandex.yt.ytclient.proxy.YtClient
import ru.yandex.yt.ytclient.proxy.request.CheckClusterLiveness
import ru.yandex.yt.ytclient.proxy.request.ExistsNode
import ru.yandex.yt.ytclient.proxy.request.ReadTable
import java.time.Duration

private val logger = KotlinLogging.logger {}

/**
 * YT reader implementation.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
class YtReaderImpl(
    private val ytClient: YtClient,
    private val userAgent: String,
    private val requestTimeout: Duration,
    retries: Int,
    retryKey: String,
    retryInitialIntervalMillis: Long,
    retryMultiplier: Double,
    retryRandomizationFactor: Double
): YtReader {

    private val retry: Retry = buildRetry(retryKey, retries,
        IntervalFunction.ofExponentialRandomBackoff(retryInitialIntervalMillis, retryMultiplier,
            retryRandomizationFactor),
        { e: Throwable? -> e is Exception }, logger, "YT request")

    override suspend fun <T> readTable(path: YPath, clazz: Class<T>): Flow<T> = flow {
        val reader = ytClient.readTable(prepareReadTable(path, clazz)).await()
        try {
            while (reader.canRead()) {
                reader.readyEvent().await()
                while (true) {
                    emitAll(reader.read()?.asFlow() ?: break)
                }
            }
        } finally {
            reader.close().await()
        }
    }.retry(retry)

    override suspend fun checkClusterLiveness() {
        retry.executeSuspendFunction {
            ytClient.checkClusterLiveness(prepareCheckClusterLiveness()).await()
        }
    }

    override suspend fun checkNodeExists(node: YPath): Boolean {
        return retry.executeSuspendFunction {
            ytClient.existsNode(prepareExistsNode(node)).await()
        }
    }

    private fun <T> prepareReadTable(path: YPath, clazz: Class<T>): ReadTable<T> {
        val readTable = ReadTable(path, YTreeObjectSerializerFactory.forClass(clazz))
        readTable.setTimeout(requestTimeout)
        readTable.setRequestId(GUID.create())
        readTable.setTraceId(GUID.create())
        readTable.setUserAgent(userAgent)
        readTable.setUnordered(true)
        return readTable
    }

    private fun prepareCheckClusterLiveness(): CheckClusterLiveness {
        val checkClusterLiveness = CheckClusterLiveness()
        checkClusterLiveness.setTimeout(requestTimeout)
        checkClusterLiveness.setRequestId(GUID.create())
        checkClusterLiveness.setTraceId(GUID.create())
        checkClusterLiveness.setUserAgent(userAgent)
        checkClusterLiveness.setCheckCypressRoot(true)
        checkClusterLiveness.setCheckSecondaryMasterCells(true)
        return checkClusterLiveness
    }

    private fun prepareExistsNode(node: YPath): ExistsNode {
        val existsNode = ExistsNode(node)
        existsNode.setTimeout(requestTimeout)
        existsNode.setRequestId(GUID.create())
        existsNode.setTraceId(GUID.create())
        existsNode.setUserAgent(userAgent)
        return existsNode
    }

}
