package ru.yandex.intranet.d.services.legacy

import kotlinx.coroutines.reactive.awaitSingle
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.context.MessageSource
import org.springframework.stereotype.Component
import reactor.core.publisher.Mono
import ru.yandex.intranet.d.dao.services.ServicesDao
import ru.yandex.intranet.d.datasource.dbSessionRetryable
import ru.yandex.intranet.d.datasource.model.YdbTableClient
import ru.yandex.intranet.d.kotlin.binding
import ru.yandex.intranet.d.kotlin.mono
import ru.yandex.intranet.d.model.services.ServiceSlugWithParent
import ru.yandex.intranet.d.util.result.ErrorCollection
import ru.yandex.intranet.d.util.result.Result
import ru.yandex.intranet.d.util.result.TypedError
import ru.yandex.intranet.d.web.model.legacy.DispenserGetProjectsListResponseDto
import ru.yandex.intranet.d.web.model.legacy.DispenserGetProjectsResponseDto
import java.util.*

/**
 * Dispenser compatible projects
 *
 * @author Evgenii Serov <evserov@yandex-team.ru>
 */
@Component
class DispenserApiCompatibleProjectService (
    private val servicesDao: ServicesDao,
    private val tableClient: YdbTableClient,
    @Qualifier("messageSource") private val messages: MessageSource
) {
    fun getServicesBySlugMono(
        servicesSlug: List<String>,
        locale: Locale
    ): Mono<Result<DispenserGetProjectsListResponseDto>> {
        return mono { getServicesBySlug(servicesSlug, locale) }
    }

    private suspend fun getServicesBySlug(
        servicesSlug: List<String>,
        locale: Locale
    ): Result<DispenserGetProjectsListResponseDto> = binding {
        val servicesSlugsWithParent = getService(servicesSlug, locale).bind()!!
        val response = prepareResponse(servicesSlug, servicesSlugsWithParent)
        return Result.success(response)
    }

    private suspend fun getService(
        servicesSlug: List<String>,
        locale: Locale
    ): Result<List<ServiceSlugWithParent>> {
        val servicesSlugsWithParent = dbSessionRetryable(tableClient) {
            servicesDao.getServiceSlugsWithParent(roStaleSingleRetryableCommit(), servicesSlug).awaitSingle()
        }!!
        val countParentServices = servicesSlugsWithParent.map { it.parentSlug }.toSet().size
        if (servicesSlug.size != countParentServices) {
            return Result.failure(
                ErrorCollection.builder().addError(
                    TypedError.notFound(
                        messages.getMessage("errors.service.not.found", null, locale)
                    )
                ).build()
            )
        }
        return Result.success(servicesSlugsWithParent)
    }

    private fun prepareResponse(
        servicesSlug: List<String>,
        servicesSlugsWithParent: List<ServiceSlugWithParent>
    ): DispenserGetProjectsListResponseDto {
        val childSlugsBySlug: MutableMap<String, MutableList<String>> = HashMap()
        for (serviceSlugWithParent in servicesSlugsWithParent) {
            childSlugsBySlug.putIfAbsent(serviceSlugWithParent.parentSlug, ArrayList())
            if (serviceSlugWithParent.slug != null) {
                childSlugsBySlug[serviceSlugWithParent.parentSlug]!!.add(serviceSlugWithParent.slug)
            }
        }

        val result = ArrayList<DispenserGetProjectsResponseDto>()
        for (requestedSlug in servicesSlug) {
            val serviceWithChildSlugs = DispenserGetProjectsResponseDto(requestedSlug,
                childSlugsBySlug.getOrDefault(requestedSlug, ArrayList()))
            result.add(serviceWithChildSlugs)
        }
        return DispenserGetProjectsListResponseDto(result)
    }
}
