package ru.yandex.intranet.d.services.legacy

import kotlinx.coroutines.reactor.awaitSingle
import kotlinx.coroutines.reactor.awaitSingleOrNull
import mu.KotlinLogging
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.context.MessageSource
import org.springframework.stereotype.Component
import reactor.core.publisher.Mono
import reactor.util.function.Tuples
import ru.yandex.intranet.d.dao.Tenants
import ru.yandex.intranet.d.dao.accounts.AccountsDao
import ru.yandex.intranet.d.dao.accounts.AccountsQuotasDao
import ru.yandex.intranet.d.dao.folders.FolderDao
import ru.yandex.intranet.d.dao.providers.ProvidersDao
import ru.yandex.intranet.d.dao.quotas.QuotasDao
import ru.yandex.intranet.d.dao.services.ServicesDao
import ru.yandex.intranet.d.dao.users.UsersDao
import ru.yandex.intranet.d.datasource.dbSessionRetryable
import ru.yandex.intranet.d.datasource.model.YdbTableClient
import ru.yandex.intranet.d.kotlin.*
import ru.yandex.intranet.d.loaders.resources.ResourcesByKeysLoader
import ru.yandex.intranet.d.loaders.resources.ResourcesLoader
import ru.yandex.intranet.d.loaders.units.UnitsEnsemblesLoader
import ru.yandex.intranet.d.model.TenantId
import ru.yandex.intranet.d.model.accounts.AccountModel
import ru.yandex.intranet.d.model.accounts.AccountsQuotasModel
import ru.yandex.intranet.d.model.folders.FolderModel
import ru.yandex.intranet.d.model.providers.ProviderModel
import ru.yandex.intranet.d.model.quotas.QuotaModel
import ru.yandex.intranet.d.model.resources.ResourceModel
import ru.yandex.intranet.d.model.services.ServiceMinimalModel
import ru.yandex.intranet.d.model.services.ServiceModel
import ru.yandex.intranet.d.model.transfers.TransferRequestModel
import ru.yandex.intranet.d.model.transfers.TransferRequestStatus
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel
import ru.yandex.intranet.d.services.accounts.AccountLogicService
import ru.yandex.intranet.d.services.integration.providers.rest.model.ResourceComplexKey
import ru.yandex.intranet.d.services.quotas.ProvisionLogicService
import ru.yandex.intranet.d.services.quotas.ProvisionOperationResultStatus
import ru.yandex.intranet.d.services.transfer.TransferRequestLogicService
import ru.yandex.intranet.d.services.transfer.model.ExpandedTransferRequests
import ru.yandex.intranet.d.util.result.ErrorCollection
import ru.yandex.intranet.d.util.result.Result
import ru.yandex.intranet.d.util.result.TypedError
import ru.yandex.intranet.d.util.units.Units
import ru.yandex.intranet.d.web.controllers.legacy.dto.*
import ru.yandex.intranet.d.web.model.CreateAccountExpandedAnswerDto
import ru.yandex.intranet.d.web.model.folders.FrontAccountInputDto
import ru.yandex.intranet.d.web.model.legacy.DiQuotaKey
import ru.yandex.intranet.d.web.model.legacy.DispenserGetQuotasProvisionDto
import ru.yandex.intranet.d.web.model.legacy.DispenserGetQuotasResponseDto
import ru.yandex.intranet.d.web.model.quotas.ProvisionLiteDto
import ru.yandex.intranet.d.web.model.quotas.UpdateProvisionsRequestDto
import ru.yandex.intranet.d.web.model.transfers.TransferRequestTypeDto
import ru.yandex.intranet.d.web.model.transfers.front.FrontCreateQuotaResourceTransferDto
import ru.yandex.intranet.d.web.model.transfers.front.FrontCreateQuotaTransferDto
import ru.yandex.intranet.d.web.model.transfers.front.FrontCreateTransferRequestDto
import ru.yandex.intranet.d.web.model.transfers.front.FrontCreateTransferRequestParametersDto
import ru.yandex.intranet.d.web.security.model.YaUserDetails
import java.math.BigDecimal
import java.util.*

private const val MDB_KEY = "mdb"
private val logger = KotlinLogging.logger {}

@Component
class DispenserApiCompatibleProvisionService(
    private val servicesDao: ServicesDao,
    private val foldersDao: FolderDao,
    private val providersDao: ProvidersDao,
    private val accountsQuotasDao: AccountsQuotasDao,
    private val tableClient: YdbTableClient,
    private val resourcesLoader: ResourcesLoader,
    private val resourcesByKeysLoader: ResourcesByKeysLoader,
    private val unitsEnsemblesLoader: UnitsEnsemblesLoader,
    private val quotasDao: QuotasDao,
    private val provisionLogicService: ProvisionLogicService,
    private val accountsDao: AccountsDao,
    private val transferRequestLogicService: TransferRequestLogicService,
    private val usersDao: UsersDao,
    private val accountLogicService: AccountLogicService,
    @Qualifier("messageSource") private val messages: MessageSource
) {
    fun getMdbProvisionByServiceSlugMono(
        serviceSlug: String,
        currentUser: YaUserDetails,
        locale: Locale
    ): Mono<Result<DispenserGetQuotasResponseDto>> {
        return mono { getMdbProvisionByServiceSlug(serviceSlug, currentUser, locale) }
    }

    fun transferQuotaFromParentMono(
        params: TransferQuotaFromParentParams,
        currentUID: String,
        locale: Locale
    ): Mono<Result<DiQuota>> = mono {
        transferQuotaFromParent(params, currentUID, locale)
    }

    suspend fun transferQuotaFromParent(
        params: TransferQuotaFromParentParams,
        currentUID: String,
        locale: Locale
    ): Result<DiQuota> = binding {
        val tenantId = Tenants.DEFAULT_TENANT_ID
        val currentUser = getUser(currentUID, tenantId, locale).bind()!!
        if (params.serviceKey != MDB_KEY) {
            return Result.failure(
                ErrorCollection.builder().addError(
                    "serviceKey", TypedError.badRequest("Only mdb supported")
                ).build()
            )
        }
        val mdbProvider = getMdbProvider(tenantId).bind()!!
        val resource = getResource(tenantId, mdbProvider.id, params.resourceKey, locale).bind()!!
        val unitsEnsemble = getUnitsEnsemble(resource.unitsEnsembleId, tenantId, locale).bind()!!
        val requestUnit = unitsEnsemble.unitByKey(params.unit.dUnitKey).get()
        val baseUnit = unitsEnsemble.unitById(resource.baseUnitId).get()
        val maxValue = Units.convert(BigDecimal(params.maxValue), requestUnit, baseUnit).longValueExact()

        val service = getService(params.projectKey, locale).bind()!!
        val targetFolder = getDefaultFolder(service.id, tenantId, locale).bind()!!
        val targetAccountId = getOrCreateAccount(
            tenantId,
            folderId = targetFolder.id,
            providerId = mdbProvider.id,
            locale,
            currentUser,
            service,
        ).bind()!!

        // Поскольку MDB, как и все облачные провайдеры, не поддерживает moveProvision, делать придется в 3 операции:
        //1. Проверяем, есть ли нужное количество в целевом аккаунте
        val targetAccountQuotas = getAccountQuotas(targetFolder.id, mdbProvider.id, tenantId)
        val targetProvision = targetAccountQuotas.firstOrNull { it.resourceId == resource.id }
        if (targetProvision != null && targetProvision.providedQuota >= maxValue) {
            return Result.success(prepareResponse(resource, unitsEnsemble, targetProvision, service))
        }
        val needToProvide = maxValue - (targetProvision?.providedQuota ?: 0L)

        //2. Если нет, проверяем, есть ли нужное количество на балансе целевого фолдера
        val targetFolderQuota = getFolderQuotas(targetFolder.id, tenantId, resource.id)
        val targetFolderBalance = targetFolderQuota?.balance ?: 0
        if (targetFolderBalance >= needToProvide) {
            return provideQuota(
                maxValue,
                targetProvision,
                targetAccountQuotas,
                resource,
                service,
                targetAccountId,
                targetFolder,
                currentUser,
                locale,
                unitsEnsemble
            )
        }
        val needToTransfer = needToProvide - targetFolderBalance

        //3. Если нет, проверяем, есть ли нужное количество на балансе фолдера - источника
        val parentService = getParentService(service, locale).bind()!!
        val sourceFolder = getDefaultFolder(parentService.id, tenantId, locale).bind()!!
        val sourceFolderQuota = getFolderQuotas(sourceFolder.id, tenantId, resource.id)
        val sourceFolderBalance = sourceFolderQuota?.balance ?: 0
        if (sourceFolderBalance >= needToTransfer) {
            transferQuota(needToTransfer, sourceFolder, resource, targetFolder, currentUser, locale).bind()
            return provideQuota(
                maxValue,
                targetProvision,
                targetAccountQuotas,
                resource,
                service,
                targetAccountId,
                targetFolder,
                currentUser,
                locale,
                unitsEnsemble
            )
        }
        val needToLiftUp = needToTransfer - sourceFolderBalance

        //4. Если нет, поднимаем на баланс из аккаунта - источника (у MDB один на сервис)
        val sourceAccount = getAccount(
            tenantId,
            folderId = sourceFolder.id,
            providerId = mdbProvider.id,
            locale
        ).bind()!!
        val sourceAccountQuotas = getAccountQuotas(sourceFolder.id, mdbProvider.id, tenantId)
        val sourceProvision = sourceAccountQuotas.firstOrNull { it.resourceId == resource.id }
        if (sourceProvision == null || sourceProvision.providedQuota < needToLiftUp) {
            return Result.failure(
                ErrorCollection.builder().addError(
                    TypedError.badRequest("Not enough quota.")
                ).build()
            )
        }
        liftUpQuota(
            maxValue = needToLiftUp,
            sourceAccountQuotas = sourceAccountQuotas,
            resource = resource,
            service = parentService,
            targetAccount = sourceAccount,
            targetFolder = sourceFolder,
            currentUser = currentUser,
            locale = locale,
            unitsEnsemble = unitsEnsemble
        ).bind()
        transferQuota(needToTransfer, sourceFolder, resource, targetFolder, currentUser, locale).bind()
        return provideQuota(
            maxValue,
            targetProvision,
            targetAccountQuotas,
            resource,
            service,
            targetAccountId,
            targetFolder,
            currentUser,
            locale,
            unitsEnsemble
        )
    }

    private suspend fun getUser(
        currentUID: String,
        tenantId: TenantId,
        locale: Locale
    ): Result<YaUserDetails> {
        val user = dbSessionRetryable(tableClient) {
            usersDao.getByPassportUid(roStaleSingleRetryableCommit(), currentUID, tenantId).awaitSingle()
        }!!
        if (user.isEmpty) {
            return Result.failure(
                ErrorCollection.builder().addError(
                    TypedError.notFound(
                        messages.getMessage("errors.user.not.found", null, locale)
                    )
                ).build()
            )
        }
        return Result.success(YaUserDetails.fromUser(user.orElseThrow()))
    }

    private suspend fun transferQuota(
        maxValue: Long,
        sourceFolder: FolderModel,
        resource: ResourceModel,
        targetFolder: FolderModel,
        currentUser: YaUserDetails,
        locale: Locale
    ): Result<ExpandedTransferRequests<TransferRequestModel>> = binding {
        val transferRequest = transferRequestLogicService.create(
            FrontCreateTransferRequestDto(
                /* description = */ "Квота для нового сервиса",
                TransferRequestTypeDto.QUOTA_TRANSFER,
                FrontCreateTransferRequestParametersDto(
                    /* quotaTransfers = */ listOf(
                        FrontCreateQuotaTransferDto(
                            /* destinationFolderId = */ sourceFolder.id,
                            /* destinationServiceId = */ sourceFolder.serviceId.toString(),
                            /* resourceTransfers = */ listOf(
                                FrontCreateQuotaResourceTransferDto(
                                    /* resourceId = */ resource.id,
                                    /* delta = */ (-maxValue).toString(),
                                    /* deltaUnitId = */ resource.baseUnitId
                                )
                            )
                        ),
                        FrontCreateQuotaTransferDto(
                            /* destinationFolderId = */ targetFolder.id,
                            /* destinationServiceId = */ targetFolder.serviceId.toString(),
                            /* resourceTransfers = */ listOf(
                                FrontCreateQuotaResourceTransferDto(
                                    /* resourceId = */ resource.id,
                                    /* delta = */ maxValue.toString(),
                                    /* deltaUnitId = */ resource.baseUnitId
                                )
                            )
                        )
                    ),
                    /* reserveTransfer = */ null,
                    /* provisionTransfers = */ null
                ),
                /* addConfirmation = */ true,
                /* loanParameters = */ null
            ),
            currentUser,
            locale,
            publicApi = false,
            idempotencyKey = null,
            delayValidation = false
        ).bind()!!
        if (transferRequest.transferRequests.status != TransferRequestStatus.APPLIED) {
            return Result.failure(ErrorCollection.builder()
                .addError(
                    TypedError.badRequest("Transfer not applied")
                )
                .addDetail("transfer", transferRequest.transferRequests)
                .build()
            )
        }
        return Result.success(transferRequest)
    }

    private suspend fun provideQuota(
        maxValue: Long,
        targetQuota: AccountsQuotasModel?,
        targetAccountQuotas: List<AccountsQuotasModel>,
        resource: ResourceModel,
        service: ServiceModel,
        targetAccountId: AccountId,
        targetFolder: FolderModel,
        currentUser: YaUserDetails,
        locale: Locale,
        unitsEnsemble: UnitsEnsembleModel
    ): Result<DiQuota> = binding {
        val correctTargetAccountQuotas = if (targetQuota != null) {
            targetAccountQuotas
        } else {
            val res = targetAccountQuotas.toMutableList()
            res.add(
                AccountsQuotasModel.Builder()
                    .setAccountId(targetAccountId)
                    .setResourceId(resource.id)
                    .setProvidedQuota(0L)
                    .build()
            )
            res
        }
        val resources = getResources(correctTargetAccountQuotas, Tenants.DEFAULT_TENANT_ID, locale).bind()!!
        val resourcesById = resources.associateBy { it.id }
        val newTargetAccountQuotas = correctTargetAccountQuotas
            .filter { resourcesById[it.resourceId]!!.isManaged }
            .map {
                val itBaseUnitId = resourcesById[it.resourceId]!!.baseUnitId
                ProvisionLiteDto(
                    /* resourceId = */ it.resourceId,
                    /* providedAmount = */ if (it.resourceId == resource.id) {
                        maxValue
                    } else {
                        it.providedQuota
                    }.toString(),
                    /* providedAmountUnitId = */ if (it.resourceId == resource.id) {
                        resource.baseUnitId
                    } else {
                        itBaseUnitId
                    },
                    /* oldProvidedAmount = */ it.providedQuota.toString(),
                    /* oldProvidedAmountUnitId = */ itBaseUnitId,
                )
            }
        val provisionResult = provisionLogicService.updateProvision(
            UpdateProvisionsRequestDto(
                /* serviceId = */ service.id,
                /* accountId = */ targetAccountId,
                /* folderId = */ targetFolder.id,
                newTargetAccountQuotas
            ),
            currentUser,
            locale,
            publicApi = false,
            idempotencyKey = null,
            validateAllowedUnits = false
        ).bind()!!
        if (provisionResult.status != ProvisionOperationResultStatus.SUCCESS) {
            return Result.failure(
                ErrorCollection.builder().addError(
                    TypedError.invalid("Provision operation failed")
                ).build()
            )
        }
        val resultAccountsQuota = provisionResult.expandedProvisionResult.get().actualUpdatedQuotas.first {
            it.resourceId == resource.id
        }
        return Result.success(prepareResponse(resource, unitsEnsemble, resultAccountsQuota, service))
    }

    private suspend fun liftUpQuota(
        maxValue: Long,
        sourceAccountQuotas: List<AccountsQuotasModel>,
        resource: ResourceModel,
        service: ServiceMinimalModel,
        targetAccount: AccountModel,
        targetFolder: FolderModel,
        currentUser: YaUserDetails,
        locale: Locale,
        unitsEnsemble: UnitsEnsembleModel
    ): Result<DiQuota> = binding {
        val resources = getResources(sourceAccountQuotas, Tenants.DEFAULT_TENANT_ID, locale).bind()!!
        val resourcesById = resources.associateBy { it.id }
        val newTargetAccountQuotas = sourceAccountQuotas
            .filter { resourcesById[it.resourceId]!!.isManaged }
            .map {
                val itBaseUnitId = resourcesById[it.resourceId]!!.baseUnitId
                ProvisionLiteDto(
                    /* resourceId = */ it.resourceId,
                    /* providedAmount = */ if (it.resourceId == resource.id) {
                        it.providedQuota - maxValue
                    } else {
                        it.providedQuota
                    }.toString(),
                    /* providedAmountUnitId = */ if (it.resourceId == resource.id) {
                        resource.baseUnitId
                    } else {
                        itBaseUnitId
                    },
                    /* oldProvidedAmount = */ it.providedQuota.toString(),
                    /* oldProvidedAmountUnitId = */ itBaseUnitId,
                )
            }
        val provisionResult = provisionLogicService.updateProvision(
            UpdateProvisionsRequestDto(
                /* serviceId = */ service.id,
                /* accountId = */ targetAccount.id,
                /* folderId = */ targetFolder.id,
                newTargetAccountQuotas
            ),
            currentUser,
            locale,
            publicApi = false,
            idempotencyKey = null,
            validateAllowedUnits = false
        ).bind()!!
        if (provisionResult.status != ProvisionOperationResultStatus.SUCCESS) {
            return Result.failure(
                ErrorCollection.builder().addError(
                    TypedError.invalid("Provision operation failed")
                ).build()
            )
        }
        val resultAccountsQuota = provisionResult.expandedProvisionResult.get().actualUpdatedQuotas.first {
            it.resourceId == resource.id
        }
        return Result.success(prepareResponse(resource, unitsEnsemble, resultAccountsQuota, service))
    }

    private suspend fun getBaseUnitIdByResourceId(resourceIds: List<ResourceId>): Map<ResourceId, UnitId> {
        val resources: List<ResourceModel> = resourcesLoader.getResourcesByIdsImmediate(resourceIds.map {
            Tuples.of(it, Tenants.DEFAULT_TENANT_ID)
        }).awaitSingleOrNull()!!
        return resources.associateBy({ resource -> resource.id }, { it.baseUnitId })
    }

    private fun prepareResponse(
        resource: ResourceModel,
        unitsEnsemble: UnitsEnsembleModel,
        targetQuota: AccountsQuotasModel,
        service: ServiceModel
    ): DiQuota {
        return prepareResponse(resource, unitsEnsemble, targetQuota, ServiceMinimalModel(service))
    }

    private fun prepareResponse(
        resource: ResourceModel,
        unitsEnsemble: UnitsEnsembleModel,
        targetQuota: AccountsQuotasModel,
        service: ServiceMinimalModel
    ): DiQuota {
        val dispenserResourceKey = dResourceKeyToDispenser(resource.key)
        val unit = unitsEnsemble.unitById(resource.baseUnitId).get()
        val amount = DiAmount(
            value = targetQuota.providedQuota,
            unit = DiUnit.findByDKey(unit.key)
        )
        val amountActual = DiAmount(
            value = targetQuota.allocatedQuota,
            unit = DiUnit.findByDKey(unit.key)
        )
        return DiQuota(
            specification = DiQuotaSpec(
                key = dispenserResourceKeyToQuotaSpecKey(dispenserResourceKey),
                resource = DiResource(
                    key = dispenserResourceKey,
                    service = DiService(
                        key = "mdb"
                    ),
                    name = resource.nameEn,
                    mode = DiQuotingMode.DEFAULT,
                ),
                description = "",
                type = DiQuotaType.ABSOLUTE,
            ),
            project = DiProject(
                key = service.slug,
                name = service.name,
                abcServiceId = service.id.toInt()
            ),
            max = amount,
            actual = amountActual,
            ownMax = amount,
            ownActual = amountActual,
            segmentKeys = setOf(),
        )
    }

    data class TransferQuotaFromParentParams(
        val projectKey: String,
        val serviceKey: String,
        val resourceKey: String,
        val quotaSpecKey: String,
        val maxValue: Long = -1,
        val ownMaxValue: Long = -1,
        val unit: DiUnit,
        val segments: Set<String> = emptySet()
    )

    private suspend fun getMdbProvisionByServiceSlug(
        serviceSlug: String,
        currentUser: YaUserDetails,
        locale: Locale
    ): Result<DispenserGetQuotasResponseDto> = binding {

        val tenantId = Tenants.getTenantId(currentUser)
        val service = getService(serviceSlug, locale).bind()!!
        val defaultFolder = getDefaultFolder(service.id, tenantId, locale).bind()!!
        val mdbProvider = getMdbProvider(tenantId).bind()!!
        val accountsQuotas = getAccountQuotas(defaultFolder.id, mdbProvider.id, tenantId)
        if (accountsQuotas.isEmpty()) {
            return Result.success(prepareResponse(listOf(), listOf(), serviceSlug))
        }
        val resources = getResources(accountsQuotas, tenantId, locale).bind()!!
        val response =
            prepareResponse(accountsQuotas, resources, serviceSlug)
        return Result.success(response)
    }

    private suspend fun getService(
        serviceSlug: String,
        locale: Locale
    ): Result<ServiceModel> {
        val serviceO = dbSessionRetryable(tableClient) {
            servicesDao.getBySlug(roStaleSingleRetryableCommit(), serviceSlug).awaitSingle()
        }!!
        if (serviceO.isEmpty) {
            return Result.failure(
                ErrorCollection.builder().addError(
                    TypedError.notFound(
                        messages.getMessage("errors.service.not.found", null, locale)
                    )
                ).build()
            )
        }
        return Result.success(serviceO.orElseThrow())
    }

    private suspend fun getParentService(
        service: ServiceModel,
        locale: Locale
    ): Result<ServiceMinimalModel> {
        val parentServiceId = service.parentId
            ?: return Result.failure(
                ErrorCollection.builder().addError(
                    TypedError.notFound(
                        messages.getMessage("errors.service.not.found", null, locale)
                    )
                ).build()
            )
        val parentService = dbSessionRetryable(tableClient) {
            servicesDao.getByIdMinimal(roStaleSingleRetryableCommit(), parentServiceId).awaitSingleOrNull()
        }!!.get()
        if (parentService.isEmpty) {
            return Result.failure(
                ErrorCollection.builder().addError(
                    TypedError.notFound(
                        messages.getMessage("errors.service.not.found", null, locale)
                    )
                ).build()
            )
        }
        return Result.success(parentService.orElseThrow())
    }

    private suspend fun getFolderQuotas(foldrId: String, tenantId: TenantId, resourceId: String): QuotaModel? {
        return dbSessionRetryable(tableClient) {
            quotasDao.getByFolders(roStaleSingleRetryableCommit(), listOf(foldrId), tenantId)
                .awaitSingleOrNull()
                ?.firstOrNull{it.resourceId == resourceId}
        }
    }

    private suspend fun getAccount(
        tenantId: TenantId,
        folderId: String,
        providerId: String,
        locale: Locale
    ): Result<AccountModel> {
        val accountModels = dbSessionRetryable(tableClient) {
            accountsDao.getByFolderAndProvider(
                roStaleSingleRetryableCommit(),
                tenantId,
                folderId,
                providerId,
                /* from = */ null,
                /* limit = */ 1,
                /* withDeleted = */ false
            ).awaitSingleOrNull()
        }!!
        if (accountModels.isEmpty()) {
            logger.error { "Account for provider $providerId not found in folder $folderId." }
            return Result.failure(
                ErrorCollection.builder().addError(
                    TypedError.notFound(
                        messages.getMessage("errors.account.not.found", null, locale)
                    )
                ).build()
            )
        }
        return Result.success(accountModels[0])
    }

    private suspend fun getOrCreateAccount(
        tenantId: TenantId,
        folderId: String,
        providerId: String,
        locale: Locale,
        currentUser: YaUserDetails,
        service: ServiceModel
    ): Result<AccountId> = binding {
        val accountModels = dbSessionRetryable(tableClient) {
            accountsDao.getByFolderAndProvider(
                roStaleSingleRetryableCommit(),
                tenantId,
                folderId,
                providerId,
                /* from = */ null,
                /* limit = */ 1,
                /* withDeleted = */ false
            ).awaitSingleOrNull()
        }!!
        if (accountModels.isNotEmpty()) {
            return Result.success(accountModels[0].id)
        }

        val answer: CreateAccountExpandedAnswerDto = accountLogicService.createAccount(
            FrontAccountInputDto(
                folderId, // folderId
                providerId, // providerId
                service.slug + "/mdb", // accountName
                null, // accountKey
                null, // accountsSpaceId
                false, // freeTier
                null // reserveType
            ),
            currentUser,
            locale,
            idempotencyKey = folderId
        ).bind()!!
        return Result.success(answer.expandedProvider.accounts.firstOrNull()!!.account.id)
    }

    private suspend fun getResource(
        tenantId: TenantId,
        providerId: String,
        dispenserResourceKey: String,
        locale: Locale
    ): Result<ResourceModel> {
        val resourceComplexKey = ResourceComplexKey(dispenserResourceKeyToD(dispenserResourceKey), mapOf())
        return dbSessionRetryable(tableClient) {
            resourcesByKeysLoader.getResources(
                roStaleSingleRetryableCommit(),
                tenantId,
                providerId,
                listOf(resourceComplexKey),
                locale
            ).awaitSingleOrNull()
        }!![resourceComplexKey]!!
    }

    private suspend fun getUnitsEnsemble(
        unitsEnsembleId: String, tenantId: TenantId, locale: Locale
    ): Result<UnitsEnsembleModel> {
        val unitsEnsemble = unitsEnsemblesLoader.getUnitsEnsembleByIdImmediate(unitsEnsembleId, tenantId)
            .awaitSingleOrNull()!!
        if (unitsEnsemble.isEmpty) {
            return Result.failure(
                ErrorCollection.builder().addError(TypedError.notFound(
                        messages.getMessage("errors.units.ensemble.not.found", null, locale)
                )).build()
            )
        }
        return Result.success(unitsEnsemble.get())
    }

    private suspend fun getDefaultFolder(
        serviceId: ServiceId,
        tenantId: TenantId,
        locale: Locale
    ): Result<FolderModel> {
        val folderO = dbSessionRetryable(tableClient) {
            foldersDao.getDefaultFolderTx(roStaleSingleRetryableCommit(), tenantId, serviceId).awaitSingle().get()
        }!!
        if (folderO.isEmpty) {
            return Result.failure(
                ErrorCollection.builder().addError(
                    TypedError.notFound(
                        messages.getMessage("errors.folder.not.found", null, locale)
                    )
                ).build()
            )
        }
        return Result.success(folderO.orElseThrow())
    }

    private suspend fun getMdbProvider(
        tenantId: TenantId
    ): Result<ProviderModel> {
        val providerO = dbSessionRetryable(tableClient) {
            providersDao.getByKey(roStaleSingleRetryableCommit(), tenantId, MDB_KEY, false).awaitSingle()
        }!!
        if (providerO.isEmpty) {
            return Result.failure(
                ErrorCollection.builder().addError(
                    TypedError.notFound("Provider MDB not found.")
                ).build()
            )
        }
        return Result.success(providerO.orElseThrow())
    }

    private suspend fun getAccountQuotas(
        folderId: FolderId,
        providerId: ProviderId,
        tenantId: TenantId
    ): List<AccountsQuotasModel> {
        return dbSessionRetryable(tableClient) {
            accountsQuotasDao.getAllByFoldersAndProvider(
                roStaleSingleRetryableCommit(), tenantId, setOf(folderId), providerId, false
            ).awaitSingle().get()
        }!!
    }

    private suspend fun getResources(
        accountsQuotas: List<AccountsQuotasModel>,
        tenantId: TenantId,
        locale: Locale
    ): Result<List<ResourceModel>> {
        val resourceIdsWithTenantId = accountsQuotas.map { accountsQuota -> accountsQuota.resourceId }
            .distinct()
            .map { resourceId -> Tuples.of(resourceId, tenantId) }
            .toList()
        val resources = resourcesLoader.getResourcesByIdsImmediate(resourceIdsWithTenantId).awaitSingle()
        if (resourceIdsWithTenantId.size != resources.size) {
            return Result.failure(
                ErrorCollection.builder().addError(
                    TypedError.notFound(
                        messages.getMessage("errors.resource.not.found", null, locale)
                    )
                ).build()
            )
        }
        return Result.success(resources)
    }

    private fun prepareResponse(
        provisions: List<AccountsQuotasModel>,
        resources: List<ResourceModel>,
        serviceSlug: String
    ): DispenserGetQuotasResponseDto {
        val dispenserKeys: Map<ResourceId, DispenserKeys> = prepareKeys(resources)
        val responseProvisions = provisions.map { provision ->
            val key = DiQuotaKey(
                projectKey = serviceSlug,
                serviceKey = MDB_KEY,
                resourceKey = dispenserKeys[provision.resourceId]!!.key,
                quotaSpecKey = dispenserKeys[provision.resourceId]!!.quotaSpecKey,
                segmentKeys = setOf()
            )
            DispenserGetQuotasProvisionDto(
                key,
                max = provision.providedQuota,
                actual = provision.allocatedQuota,
                ownMax = 0L,
                ownActual = 0L
            )
        }
        return DispenserGetQuotasResponseDto(result = responseProvisions)
    }

    private fun prepareKeys(resources: List<ResourceModel>): Map<ResourceId, DispenserKeys> =
        resources.associate { resource ->
            val key = dResourceKeyToDispenser(resource.key)
            resource.id to DispenserKeys(key, dispenserResourceKeyToQuotaSpecKey(key))
        }

    private fun dResourceKeyToDispenser(dResourceKey: String): String {
        return if (dResourceKey == "cpu_cores") "cpu" else dResourceKey
    }

    private fun dispenserResourceKeyToQuotaSpecKey(dispenserResourceKey: String): String {
        return "$dispenserResourceKey-quota"
    }

    private fun dispenserResourceKeyToD(dispenserResourceKey: String): String {
        return if (dispenserResourceKey == "cpu") "cpu_cores" else dispenserResourceKey
    }

    data class DispenserKeys(
        val key: String,
        val quotaSpecKey: String
    )
}
