package ru.yandex.intranet.d.services.loans

import com.fasterxml.jackson.databind.ObjectReader
import com.fasterxml.jackson.databind.ObjectWriter
import kotlinx.coroutines.reactor.awaitSingle
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.context.MessageSource
import org.springframework.stereotype.Component
import reactor.core.publisher.Mono
import reactor.util.function.Tuples
import ru.yandex.intranet.d.dao.Tenants
import ru.yandex.intranet.d.dao.accounts.AccountsDao
import ru.yandex.intranet.d.dao.accounts.AccountsSpacesDao
import ru.yandex.intranet.d.dao.folders.FolderDao
import ru.yandex.intranet.d.dao.loans.LoansDao
import ru.yandex.intranet.d.dao.loans.LoansHistoryDao
import ru.yandex.intranet.d.dao.loans.ServiceLoansInDao
import ru.yandex.intranet.d.dao.loans.ServiceLoansOutDao
import ru.yandex.intranet.d.dao.services.ServicesDao
import ru.yandex.intranet.d.dao.users.UsersDao
import ru.yandex.intranet.d.datasource.dbSessionRetryable
import ru.yandex.intranet.d.datasource.model.YdbTableClient
import ru.yandex.intranet.d.kotlin.AccountId
import ru.yandex.intranet.d.kotlin.AccountsSpacesId
import ru.yandex.intranet.d.kotlin.FolderId
import ru.yandex.intranet.d.kotlin.LoanHistoryId
import ru.yandex.intranet.d.kotlin.LoanId
import ru.yandex.intranet.d.kotlin.ProviderId
import ru.yandex.intranet.d.kotlin.ResourceId
import ru.yandex.intranet.d.kotlin.ResourceTypeId
import ru.yandex.intranet.d.kotlin.SegmentId
import ru.yandex.intranet.d.kotlin.SegmentationId
import ru.yandex.intranet.d.kotlin.ServiceId
import ru.yandex.intranet.d.kotlin.UserId
import ru.yandex.intranet.d.kotlin.binding
import ru.yandex.intranet.d.kotlin.mono
import ru.yandex.intranet.d.loaders.providers.ProvidersLoader
import ru.yandex.intranet.d.loaders.resources.ResourcesLoader
import ru.yandex.intranet.d.loaders.resources.segmentations.ResourceSegmentationsLoader
import ru.yandex.intranet.d.loaders.resources.segments.ResourceSegmentsLoader
import ru.yandex.intranet.d.loaders.resources.types.ResourceTypesLoader
import ru.yandex.intranet.d.model.TenantId
import ru.yandex.intranet.d.model.accounts.AccountModel
import ru.yandex.intranet.d.model.accounts.AccountSpaceModel
import ru.yandex.intranet.d.model.folders.FolderModel
import ru.yandex.intranet.d.model.loans.LoanModel
import ru.yandex.intranet.d.model.loans.LoanStatus
import ru.yandex.intranet.d.model.loans.LoansHistoryModel
import ru.yandex.intranet.d.model.loans.ServiceLoanInModel
import ru.yandex.intranet.d.model.loans.ServiceLoanOutModel
import ru.yandex.intranet.d.model.providers.ProviderModel
import ru.yandex.intranet.d.model.resources.ResourceModel
import ru.yandex.intranet.d.model.resources.segmentations.ResourceSegmentationModel
import ru.yandex.intranet.d.model.resources.segments.ResourceSegmentModel
import ru.yandex.intranet.d.model.resources.types.ResourceTypeModel
import ru.yandex.intranet.d.model.services.ServiceMinimalModel
import ru.yandex.intranet.d.model.users.UserModel
import ru.yandex.intranet.d.services.security.SecurityManagerService
import ru.yandex.intranet.d.util.ObjectMapperHolder
import ru.yandex.intranet.d.util.Uuids
import ru.yandex.intranet.d.util.paging.ContinuationTokens
import ru.yandex.intranet.d.util.paging.PageRequest
import ru.yandex.intranet.d.util.result.ErrorCollection
import ru.yandex.intranet.d.web.security.model.YaUserDetails
import ru.yandex.intranet.d.util.result.Result
import ru.yandex.intranet.d.util.result.TypedError
import ru.yandex.intranet.d.web.model.dictionaries.FrontAccountDictionaryElementDto
import ru.yandex.intranet.d.web.model.dictionaries.FrontAccountsSpaceDictionaryElementDto
import ru.yandex.intranet.d.web.model.dictionaries.FrontFolderDictionaryElementDto
import ru.yandex.intranet.d.web.model.dictionaries.FrontProviderDictionaryElementDto
import ru.yandex.intranet.d.web.model.dictionaries.FrontResourceDictionaryElementDto
import ru.yandex.intranet.d.web.model.dictionaries.FrontResourceSegmentDictionaryElementDto
import ru.yandex.intranet.d.web.model.dictionaries.FrontResourceSegmentationDictionaryElementDto
import ru.yandex.intranet.d.web.model.dictionaries.FrontResourceTypeDictionaryElementDto
import ru.yandex.intranet.d.web.model.dictionaries.FrontServiceDictionaryElementDto
import ru.yandex.intranet.d.web.model.dictionaries.FrontUserDictionaryElementDto
import ru.yandex.intranet.d.web.model.loans.front.FrontSearchLoansRequestDto
import ru.yandex.intranet.d.web.model.loans.front.FrontSearchLoansResponseDependencies
import ru.yandex.intranet.d.web.model.loans.front.FrontSearchLoansResponseDto
import ru.yandex.intranet.d.web.model.loans.api.ApiGetLoansHistoryResponseDto
import ru.yandex.intranet.d.web.model.loans.history.FrontGetLoansHistoryResponseDependencies
import ru.yandex.intranet.d.web.model.loans.history.FrontGetLoansHistoryResponseDto
import ru.yandex.intranet.d.web.model.loans.api.ApiSearchLoansRequestDto
import ru.yandex.intranet.d.web.model.loans.api.ApiSearchLoansResponseDto
import ru.yandex.intranet.d.web.model.loans.LoanDirection
import ru.yandex.intranet.d.web.model.loans.api.ApiLoanDto
import ru.yandex.intranet.d.web.model.loans.api.ApiLoansHistoryDto
import ru.yandex.intranet.d.web.model.loans.front.FrontLoanDto
import ru.yandex.intranet.d.web.model.loans.front.FrontLoansHistoryDto
import java.time.Instant
import java.util.*

@Component
class LoansService(
    private val securityManagerService: SecurityManagerService,
    private val loansDao: LoansDao,
    private val serviceLoansInDao: ServiceLoansInDao,
    private val serviceLoansOutDao: ServiceLoansOutDao,
    private val tableClient: YdbTableClient,
    private val providersLoader: ProvidersLoader,
    private val servicesDao: ServicesDao,
    private val accountsDao: AccountsDao,
    private val foldersDao: FolderDao,
    private val resourcesLoader: ResourcesLoader,
    private val resourceTypesLoader: ResourceTypesLoader,
    private val segmentsLoader: ResourceSegmentsLoader,
    private val segmentationsLoader: ResourceSegmentationsLoader,
    private val accountsSpacesDao: AccountsSpacesDao,
    private val usersDao: UsersDao,
    private val loansHistoryDao: LoansHistoryDao,
    @Qualifier("continuationTokensJsonObjectMapper") private val objectMapper: ObjectMapperHolder,
    @Qualifier("messageSource") private val messages: MessageSource
) {
    private val searchLoansContinuationTokenReader: ObjectReader = objectMapper.objectMapper
        .readerFor(LoansPageToken::class.java)
    private val searchLoansContinuationTokenWriter: ObjectWriter = objectMapper.objectMapper
        .writerFor(LoansPageToken::class.java)
    private val getHistoryContinuationTokenReader: ObjectReader = objectMapper.objectMapper
        .readerFor(LoansHistoryPageToken::class.java)
    private val getHistoryContinuationTokenWriter: ObjectWriter = objectMapper.objectMapper
        .writerFor(LoansHistoryPageToken::class.java)


    fun searchLoansMono(
        request: FrontSearchLoansRequestDto, userDetails: YaUserDetails, locale: Locale
    ): Mono<Result<FrontSearchLoansResponseDto>> = mono {
        searchLoans(request, userDetails, locale)
    }

    fun searchLoansApiMono(
        request: ApiSearchLoansRequestDto, userDetails: YaUserDetails, locale: Locale
    ): Mono<Result<ApiSearchLoansResponseDto>> = mono {
        searchLoansApi(request, userDetails, locale)
    }

    fun getLoansHistoryMono(
        id: LoanId?, pageToken: String?, limit: Int?, userDetails: YaUserDetails, locale: Locale
    ): Mono<Result<FrontGetLoansHistoryResponseDto>> = mono {
        getLoansHistory(id, pageToken, limit, userDetails, locale)
    }

    fun getLoansHistoryApiMono(
        id: LoanId?, pageToken: String?, limit: Int?, userDetails: YaUserDetails, locale: Locale
    ): Mono<Result<ApiGetLoansHistoryResponseDto>> = mono {
        getLoansHistoryApi(id, pageToken, limit, userDetails, locale)
    }

    private suspend fun searchLoans(
        request: FrontSearchLoansRequestDto, userDetails: YaUserDetails, locale: Locale
    ): Result<FrontSearchLoansResponseDto> = binding {
        securityManagerService.checkReadPermissions(userDetails, locale).awaitSingle().bind()
        val validatedRequest = validateRequest(request, locale).bind()!!
        securityManagerService.checkReadPermissions(validatedRequest.serviceId, userDetails, locale)
            .awaitSingle().bind()
        val serviceLoansOrdered = getOrderedServiceLoans(
            validatedRequest.serviceId,
            validatedRequest.status,
            validatedRequest.direction,
            validatedRequest.from,
            validatedRequest.limit
        )
        val nextPageToken = getNextPageToken(serviceLoansOrdered, validatedRequest.limit)
        val loanIdsOrdered = serviceLoansOrdered.map { it.loanId }
        val loanModelsOrdered = getLoanModelsOrdered(loanIdsOrdered)
        val dependencies = loadSearchLoansResponseDependencies(loanModelsOrdered, locale)
        val loanDtosOrdered = loanModelsOrdered.map { FrontLoanDto(it) }
        return Result.success(FrontSearchLoansResponseDto(loanDtosOrdered, dependencies, nextPageToken))
    }

    private suspend fun searchLoansApi(
        request: ApiSearchLoansRequestDto, userDetails: YaUserDetails, locale: Locale
    ): Result<ApiSearchLoansResponseDto> = binding {
        securityManagerService.checkReadPermissions(userDetails, locale).awaitSingle().bind()
        val validatedRequest = validateRequest(request, locale).bind()!!
        securityManagerService.checkReadPermissions(validatedRequest.serviceId, userDetails, locale)
            .awaitSingle().bind()
        val serviceLoansOrdered = getOrderedServiceLoans(
            validatedRequest.serviceId,
            validatedRequest.status,
            validatedRequest.direction,
            validatedRequest.from,
            validatedRequest.limit
        )
        val nextPageToken = getNextPageToken(serviceLoansOrdered, validatedRequest.limit)
        val loanIdsOrdered = serviceLoansOrdered.map { it.loanId }
        val loanModelsOrdered = getLoanModelsOrdered(loanIdsOrdered)
        val loanDtosOrdered = loanModelsOrdered.map { ApiLoanDto(it) }
        return Result.success(ApiSearchLoansResponseDto(loanDtosOrdered, nextPageToken))
    }

    private suspend fun getLoansHistory(
        id: LoanId?, pageToken: String?, limit: Int?, userDetails: YaUserDetails, locale: Locale
    ): Result<FrontGetLoansHistoryResponseDto> = binding {
        securityManagerService.checkReadPermissions(userDetails, locale).awaitSingle().bind()
        val validatedRequest = validateGetHistoryRequest(id, pageToken, limit, locale).bind()!!
        val historyModels = getLoansHistoryModels(
            validatedRequest.loanId, validatedRequest.from, validatedRequest.limit
        )
        val nextPageToken = getNextPageHistoryToken(historyModels, validatedRequest.limit)
        val dependencies = loadGetLoansHistoryDependencies(historyModels, locale)
        val frontHistoryDtosOrdered = historyModels.map { FrontLoansHistoryDto(it) }
        return Result.success(FrontGetLoansHistoryResponseDto(frontHistoryDtosOrdered, dependencies, nextPageToken))
    }

    private suspend fun getLoansHistoryApi(
        id: LoanId?, pageToken: String?, limit: Int?, userDetails: YaUserDetails, locale: Locale
    ): Result<ApiGetLoansHistoryResponseDto> = binding {
        securityManagerService.checkReadPermissions(userDetails, locale).awaitSingle().bind()
        val validatedRequest = validateGetHistoryRequest(id, pageToken, limit, locale).bind()!!
        val historyModels = getLoansHistoryModels(
            validatedRequest.loanId, validatedRequest.from, validatedRequest.limit
        )
        val nextPageToken = getNextPageHistoryToken(historyModels, validatedRequest.limit)
        val frontHistoryDtosOrdered = historyModels.map { ApiLoansHistoryDto(it) }
        return Result.success(ApiGetLoansHistoryResponseDto(frontHistoryDtosOrdered, nextPageToken))
    }

    private fun validateRequest(
        request: FrontSearchLoansRequestDto, locale: Locale
    ): Result<ValidatedSearchLoansRequest> {
        val errors = ErrorCollection.builder()
        if (request.serviceId == null) {
            errors.addError(
                "serviceId", TypedError.invalid(
                    messages.getMessage("errors.field.is.required", null, locale)
                )
            )
        }
        if (request.direction == null) {
            errors.addError(
                "direction", TypedError.invalid(
                    messages.getMessage("errors.field.is.required", null, locale)
                )
            )
        }
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build())
        }
        val pageRequest = PageRequest(request.from, request.limit?.toLong())
        val validatedPageRequest: Result<PageRequest.Validated<LoansPageToken>> =
            pageRequest.validate(searchLoansContinuationTokenReader, messages, locale)
        val validatedPageToken: Result<PageRequest.Validated<SearchLoansPageToken>> = validatedPageRequest
            .andThen { validateGetLoansPageToken(it, locale) }
        return validatedPageToken.apply { r ->
            ValidatedSearchLoansRequest(
                request.serviceId!!, request.status, request.direction!!,
                r.continuationToken.orElse(null), r.limit
            )
        }
    }

    private fun validateRequest(
        request: ApiSearchLoansRequestDto, locale: Locale
    ): Result<ValidatedSearchLoansRequest> {
        val errors = ErrorCollection.builder()
        if (request.serviceId == null) {
            errors.addError(
                "serviceId", TypedError.invalid(
                    messages.getMessage("errors.field.is.required", null, locale)
                )
            )
        }
        if (request.direction == null) {
            errors.addError(
                "direction", TypedError.invalid(
                    messages.getMessage("errors.field.is.required", null, locale)
                )
            )
        }
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build())
        }
        val pageRequest = PageRequest(request.from, request.limit?.toLong())
        val validatedPageRequest: Result<PageRequest.Validated<LoansPageToken>> =
            pageRequest.validate(searchLoansContinuationTokenReader, messages, locale)
        val validatedPageToken: Result<PageRequest.Validated<SearchLoansPageToken>> = validatedPageRequest
            .andThen { validateGetLoansPageToken(it, locale) }
        return validatedPageToken.apply { r ->
            ValidatedSearchLoansRequest(
                request.serviceId!!, request.status, request.direction!!,
                r.continuationToken.orElse(null), r.limit
            )
        }
    }

    private fun validateGetHistoryRequest(
        loanId: LoanId?, pageToken: String?, limit: Int?, locale: Locale
    ): Result<ValidatedGetLoansHistoryRequest> {
        val errors = ErrorCollection.builder()
        if (loanId == null) {
            errors.addError(
                "loanId", TypedError.invalid(
                    messages.getMessage("errors.field.is.required", null, locale)
                )
            )
        } else if (!Uuids.isValidUuid(loanId)) {
            errors.addError(
                "loanId", TypedError.invalid(
                    messages.getMessage("errors.loan.not.found", null, locale)
                )
            )
        }
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build())
        }
        val pageRequest = PageRequest(pageToken, limit?.toLong())
        val validatedPageRequest: Result<PageRequest.Validated<LoansHistoryPageToken>> =
            pageRequest.validate(getHistoryContinuationTokenReader, messages, locale)
        val validatedPageToken: Result<PageRequest.Validated<GetLoansHistoryPageToken>> = validatedPageRequest
            .andThen { validateGetHistoryPageToken(it, locale) }
        return validatedPageToken.apply { r ->
            ValidatedGetLoansHistoryRequest(
                loanId!!, r.continuationToken.orElse(null), r.limit
            )
        }
    }

    private fun validateGetLoansPageToken(
        pageToken: PageRequest.Validated<LoansPageToken>, locale: Locale
    ): Result<PageRequest.Validated<SearchLoansPageToken>> {
        if (pageToken.continuationToken.isEmpty) {
            return Result.success(PageRequest.Validated(null, pageToken.limit))
        }
        val continuationToken = pageToken.continuationToken.orElseThrow()
        if (continuationToken.status == null || continuationToken.dueAt == null || continuationToken.loanId == null) {
            return Result.failure(
                ErrorCollection.builder().addError(
                    "from", TypedError.invalid(
                        messages.getMessage("errors.loan.not.found", null, locale)
                    )
                ).build()
            )
        }
        return Result.success(
            PageRequest.Validated(
                SearchLoansPageToken(continuationToken.status, continuationToken.dueAt, continuationToken.loanId),
                pageToken.limit
            )
        )
    }

    private fun validateGetHistoryPageToken(
        pageToken: PageRequest.Validated<LoansHistoryPageToken>, locale: Locale
    ): Result<PageRequest.Validated<GetLoansHistoryPageToken>> {
        if (pageToken.continuationToken.isEmpty) {
            return Result.success(PageRequest.Validated(null, pageToken.limit))
        }
        if (pageToken.continuationToken.get().eventTimestamp == null) {
            return Result.failure(
                ErrorCollection.builder().addError(
                    "pageToken", TypedError.invalid(
                        messages
                            .getMessage("errors.loan.not.found", null, locale)
                    )
                ).build()
            )
        }
        if (pageToken.continuationToken.get().historyId == null) {
            return Result.failure(
                ErrorCollection.builder().addError(
                    "pageToken", TypedError.invalid(
                        messages
                            .getMessage("errors.loan.not.found", null, locale)
                    )
                ).build()
            )
        }
        return Result.success(
            PageRequest.Validated(
                GetLoansHistoryPageToken(
                    pageToken.continuationToken.get().eventTimestamp!!,
                    pageToken.continuationToken.get().historyId!!
                ),
                pageToken.limit
            )
        )
    }

    private suspend fun getLoansHistoryModels(
        loanId: LoanId, pageToken: GetLoansHistoryPageToken?, limit: Int
    ): List<LoansHistoryModel> = dbSessionRetryable(tableClient) {
        if (pageToken == null) {
            loansHistoryDao.getByLoanFirstPage(
                roStaleSingleRetryableCommit(), Tenants.DEFAULT_TENANT_ID, loanId, limit.toLong()
            )
        } else {
            loansHistoryDao.getByLoanNextPage(
                roStaleSingleRetryableCommit(), Tenants.DEFAULT_TENANT_ID, loanId, pageToken.eventTimestamp,
                pageToken.historyId, limit.toLong()
            )
        }
    }!!

    private suspend fun getOrderedServiceLoans(
        serviceId: ServiceId, status: LoanStatus?, direction: LoanDirection,
        pageToken: SearchLoansPageToken?, limit: Int
    ): List<ServiceLoan> =
        when(direction) {
            LoanDirection.IN ->
                if (status == null) {
                    getServiceLoansIn(serviceId, pageToken, limit)
                } else {
                    getServiceLoansInByStatus(serviceId, status, pageToken, limit)
                }
            LoanDirection.OUT ->
                if (status == null) {
                    getServiceLoansOut(serviceId, pageToken, limit)
                } else {
                    getServiceLoansOutByStatus(serviceId, status, pageToken, limit)
                }
        }

    private suspend fun getServiceLoansInByStatus(
        serviceId: ServiceId, status: LoanStatus, pageToken: SearchLoansPageToken?, limit: Int
    ): List<ServiceLoan> = dbSessionRetryable(tableClient) {
        if (pageToken == null) {
            serviceLoansInDao.getByServiceStatusOrderByDueFirstPage(
                roStaleSingleRetryableCommit(), Tenants.DEFAULT_TENANT_ID, serviceId, status, limit
            )
        } else {
            serviceLoansInDao.getByServiceStatusOrderByDueNextPage(
                roStaleSingleRetryableCommit(), Tenants.DEFAULT_TENANT_ID, serviceId, status,
                pageToken.dueAt, pageToken.loanId, limit
            )
        }
    }!!.map { ServiceLoan(it) }

    private suspend fun getServiceLoansOutByStatus(
        serviceId: ServiceId, status: LoanStatus, pageToken: SearchLoansPageToken?, limit: Int
    ): List<ServiceLoan> = dbSessionRetryable(tableClient) {
        if (pageToken == null) {
            serviceLoansOutDao.getByServiceStatusOrderByDueFirstPage(
                roStaleSingleRetryableCommit(), Tenants.DEFAULT_TENANT_ID, serviceId, status, limit
            )
        } else {
            serviceLoansOutDao.getByServiceStatusOrderByDueNextPage(
                roStaleSingleRetryableCommit(), Tenants.DEFAULT_TENANT_ID, serviceId, status,
                pageToken.dueAt, pageToken.loanId, limit
            )
        }
    }!!.map { ServiceLoan(it) }

    private suspend fun getServiceLoansIn(
        serviceId: ServiceId, pageToken: SearchLoansPageToken?, limit: Int
    ): List<ServiceLoan> = dbSessionRetryable(tableClient) {
        if (pageToken == null) {
            serviceLoansInDao.getByServiceOrderByDueFirstPage(
                roStaleSingleRetryableCommit(), Tenants.DEFAULT_TENANT_ID, serviceId, limit
            )
        } else {
            serviceLoansInDao.getByServiceOrderByDueNextPage(
                roStaleSingleRetryableCommit(), Tenants.DEFAULT_TENANT_ID, serviceId, pageToken.status,
                pageToken.dueAt, pageToken.loanId, limit
            )
        }
    }!!.map { ServiceLoan(it) }

    private suspend fun getServiceLoansOut(
        serviceId: ServiceId, pageToken: SearchLoansPageToken?, limit: Int
    ): List<ServiceLoan> = dbSessionRetryable(tableClient) {
        if (pageToken == null) {
            serviceLoansOutDao.getByServiceOrderByDueFirstPage(
                roStaleSingleRetryableCommit(), Tenants.DEFAULT_TENANT_ID, serviceId, limit
            )
        } else {
            serviceLoansOutDao.getByServiceOrderByDueNextPage(
                roStaleSingleRetryableCommit(), Tenants.DEFAULT_TENANT_ID, serviceId, pageToken.status,
                pageToken.dueAt, pageToken.loanId, limit
            )
        }
    }!!.map { ServiceLoan(it) }

    private suspend fun getLoanModelsOrdered(idsOrdered: List<LoanId>): List<LoanModel> {
        val loanModelsByIds = dbSessionRetryable(tableClient) {
            loansDao.getByIds(roStaleSingleRetryableCommit(), idsOrdered, Tenants.DEFAULT_TENANT_ID)
        }!!.associateBy { it.id }
        return idsOrdered.mapNotNull { id -> loanModelsByIds[id] }
    }

    private fun getNextPageToken(serviceLoansOrdered: List<ServiceLoan>, limit: Int): String? =
        if (serviceLoansOrdered.size >= limit) {
            val lastLoan = serviceLoansOrdered.last()
            val status = lastLoan.status
            val dueAt = lastLoan.dueAt
            val loanId = lastLoan.loanId
            ContinuationTokens.encode(LoansPageToken(status, dueAt, loanId), searchLoansContinuationTokenWriter)
        } else {
            null
        }

    private fun getNextPageHistoryToken(historyModels: List<LoansHistoryModel>, limit: Int): String? =
        if (historyModels.size >= limit) {
            val eventTimestamp = historyModels.last().key.eventTimestamp
            val historyId = historyModels.last().key.id
            ContinuationTokens.encode(
                LoansHistoryPageToken(eventTimestamp, historyId), getHistoryContinuationTokenWriter
            )
        } else {
            null
        }

    private suspend fun loadSearchLoansResponseDependencies(
        loans: List<LoanModel>, locale: Locale
    ): FrontSearchLoansResponseDependencies {
        val resourceIds = extractResourceIds(loans)
        val resources = loadResources(resourceIds)
        val accountIds = extractAccountIds(loans)
        val accounts = loadAccounts(accountIds)
        val folderIds = extractFolderIds(loans, accounts.values)
        val folders = loadFolders(folderIds)
        val providerIds = extractProviderIds(loans, resources.values, accounts.values)
        val providers = loadProviders(providerIds)
        val serviceIds = extractServiceIds(loans)
        val services = loadServices(serviceIds)
        val userIds = extractUserIds(loans)
        val users = loadUsers(userIds)
        val resourceTypeIds = resources.values.map { it.resourceTypeId }.distinct()
        val resourceTypes = loadResourceTypes(resourceTypeIds)
        val accountsSpaceIds = extractAccountsSpaceIds(accounts.values, resources.values)
        val accountsSpaces = loadAccountsSpaces(accountsSpaceIds)
        val segmentationIds = extractSegmentationIds(resources.values, accountsSpaces.values)
        val segmentations = loadSegmentations(segmentationIds)
        val segmentIds = extractSegmentIds(resources.values, accountsSpaces.values)
        val segments = loadSegments(segmentIds)

        return FrontSearchLoansResponseDependencies(
            providers = providers.mapValues { FrontProviderDictionaryElementDto(it.value, locale) },
            accounts = accounts.mapValues { FrontAccountDictionaryElementDto(it.value) },
            services = services.mapValues { FrontServiceDictionaryElementDto(it.value, locale) },
            folders = folders.mapValues { FrontFolderDictionaryElementDto(it.value) },
            resources = resources.mapValues { FrontResourceDictionaryElementDto(it.value, locale) },
            resourceTypes = resourceTypes.mapValues { FrontResourceTypeDictionaryElementDto(it.value, locale) },
            accountsSpaces = accountsSpaces.mapValues { FrontAccountsSpaceDictionaryElementDto(it.value, locale) },
            segmentations = segmentations.mapValues { FrontResourceSegmentationDictionaryElementDto(it.value, locale) },
            segments = segments.mapValues { FrontResourceSegmentDictionaryElementDto(it.value, locale) },
            users = users.mapValues { FrontUserDictionaryElementDto(it.value, locale) }
        )
    }

    private suspend fun loadGetLoansHistoryDependencies(
        historyModels: List<LoansHistoryModel>, locale: Locale
    ): FrontGetLoansHistoryResponseDependencies {
        val resourceIds = extractResourceIdsFromHistoryModels(historyModels)
        val resources = loadResources(resourceIds)
        val accountIds = extractAccountIdsFromHistoryModels(historyModels)
        val accounts = loadAccounts(accountIds)
        val folderIds = extractFolderIdsFromHistoryModels(historyModels, accounts.values)
        val folders = loadFolders(folderIds)
        val providerIds = extractProviderIdsFromHistoryModels(historyModels, resources.values, accounts.values)
        val providers = loadProviders(providerIds)
        val serviceIds = extractServiceIdsFromHistoryModels(historyModels)
        val services = loadServices(serviceIds)
        val userIds = extractUserIdsFromHistoryModels(historyModels)
        val users = loadUsers(userIds)
        val resourceTypeIds = resources.values.map { it.resourceTypeId }.distinct()
        val resourceTypes = loadResourceTypes(resourceTypeIds)
        val accountsSpaceIds = extractAccountsSpaceIds(accounts.values, resources.values)
        val accountsSpaces = loadAccountsSpaces(accountsSpaceIds)
        val segmentationIds = extractSegmentationIds(resources.values, accountsSpaces.values)
        val segmentations = loadSegmentations(segmentationIds)
        val segmentIds = extractSegmentIds(resources.values, accountsSpaces.values)
        val segments = loadSegments(segmentIds)

        return FrontGetLoansHistoryResponseDependencies(
            providers = providers.mapValues { FrontProviderDictionaryElementDto(it.value, locale) },
            accounts = accounts.mapValues { FrontAccountDictionaryElementDto(it.value) },
            services = services.mapValues { FrontServiceDictionaryElementDto(it.value, locale) },
            folders = folders.mapValues { FrontFolderDictionaryElementDto(it.value) },
            resources = resources.mapValues { FrontResourceDictionaryElementDto(it.value, locale) },
            resourceTypes = resourceTypes.mapValues { FrontResourceTypeDictionaryElementDto(it.value, locale) },
            accountsSpaces = accountsSpaces.mapValues { FrontAccountsSpaceDictionaryElementDto(it.value, locale) },
            segmentations = segmentations.mapValues { FrontResourceSegmentationDictionaryElementDto(it.value, locale) },
            segments = segments.mapValues { FrontResourceSegmentDictionaryElementDto(it.value, locale) },
            users = users.mapValues { FrontUserDictionaryElementDto(it.value, locale) }
        )
    }

    private fun extractResourceIds(loans: Collection<LoanModel>): List<ResourceId> = loans.flatMap { loan ->
        setOfNotNull(
            loan.borrowedAmounts.amounts.map { it.resource },
            loan.dueAmounts.amounts.map { it.resource },
            loan.payOffAmounts.amounts.map { it.resource }
        )
    }.flatten().distinct()

    private fun extractResourceIdsFromHistoryModels(models: Collection<LoansHistoryModel>): List<ResourceId> =
        models.flatMap { model ->
            setOfNotNull(
                model.oldFields?.borrowedAmounts?.amounts?.map { it.resource },
                model.oldFields?.payOffAmounts?.amounts?.map { it.resource },
                model.oldFields?.dueAmounts?.amounts?.map { it.resource },
                model.newFields?.borrowedAmounts?.amounts?.map { it.resource },
                model.newFields?.payOffAmounts?.amounts?.map { it.resource },
                model.newFields?.dueAmounts?.amounts?.map { it.resource }
            )
        }.flatten().distinct()

    private fun extractAccountIds(loans: Collection<LoanModel>): List<ResourceId> = loans.flatMap { loan ->
        setOfNotNull(
            loan.borrowedFrom.account,
            loan.borrowedTo.account,
            loan.payOffFrom.account,
            loan.payOffTo.account
        )
    }.distinct()

    private fun extractAccountIdsFromHistoryModels(models: Collection<LoansHistoryModel>): List<ResourceId> =
        models.flatMap { model ->
            setOfNotNull(
                model.oldFields?.payOffFrom?.account,
                model.oldFields?.payOffTo?.account,
                model.newFields?.payOffFrom?.account,
                model.newFields?.payOffTo?.account,
            )
        }.distinct()

    private fun extractFolderIds(loans: Collection<LoanModel>, accounts: Collection<AccountModel>): List<AccountId> =
        loans.flatMap { loan ->
            setOfNotNull(
                loan.borrowedFrom.folder,
                loan.borrowedTo.folder,
                loan.payOffFrom.folder,
                loan.payOffTo.folder
            )
        }
            .plus(accounts.map { it.folderId })
            .distinct()

    private fun extractFolderIdsFromHistoryModels(
        models: Collection<LoansHistoryModel>, accounts: Collection<AccountModel>
    ): List<AccountId> = models.flatMap { model ->
        setOfNotNull(
            model.oldFields?.payOffFrom?.folder,
            model.oldFields?.payOffTo?.folder,
            model.newFields?.payOffFrom?.folder,
            model.newFields?.payOffTo?.folder,
        )
    }
        .plus(accounts.map { it.folderId })
        .distinct()

    private fun extractProviderIds(
        loans: Collection<LoanModel>, resources: Collection<ResourceModel>, accounts: Collection<AccountModel>
    ): List<ProviderId> = loans.flatMap { loan ->
        setOfNotNull(
            loan.borrowedFrom.provider,
            loan.borrowedTo.provider,
            loan.payOffFrom.provider,
            loan.payOffTo.provider,
            loan.requestedBy.provider
        ).plus(loan.requestApprovedBy.subjects.mapNotNull { it.provider })
    }
        .plus(accounts.map { it.providerId })
        .plus(resources.map { it.providerId })
        .distinct()

    private fun extractProviderIdsFromHistoryModels(
        models: Collection<LoansHistoryModel>, resources: Collection<ResourceModel>, accounts: Collection<AccountModel>
    ): List<AccountId> = models.flatMap { model ->
        setOfNotNull(
            model.eventAuthor.provider,
            model.oldFields?.payOffFrom?.provider,
            model.oldFields?.payOffTo?.provider,
            model.newFields?.payOffFrom?.provider,
            model.newFields?.payOffTo?.provider
        ).plus(model.eventApprovedBy?.subjects?.mapNotNull { it.provider } ?: listOf())
    }
        .plus(accounts.map { it.providerId })
        .plus(resources.map { it.providerId })
        .distinct()

    private fun extractServiceIds(loans: Collection<LoanModel>): List<ServiceId> = loans.flatMap { loan ->
        setOfNotNull(
            loan.borrowedFrom.service,
            loan.borrowedTo.service,
            loan.payOffFrom.service,
            loan.payOffTo.service
        )
    }.distinct()

    private fun extractServiceIdsFromHistoryModels(models: Collection<LoansHistoryModel>): List<ServiceId> =
        models.flatMap { model ->
            setOfNotNull(
                model.oldFields?.payOffFrom?.service,
                model.oldFields?.payOffTo?.service,
                model.newFields?.payOffFrom?.service,
                model.newFields?.payOffTo?.service
            )
        }.distinct()

    private fun extractUserIds(loans: Collection<LoanModel>): List<UserId> = loans.mapNotNull { it.requestedBy.user }
        .plus(loans.flatMap { loan -> loan.requestApprovedBy.subjects.mapNotNull { it.user } })
        .distinct()

    private fun extractUserIdsFromHistoryModels(models: Collection<LoansHistoryModel>): List<UserId> =
        models.mapNotNull { it.eventAuthor.user }
            .plus(models.flatMap { model -> model.eventApprovedBy?.subjects?.mapNotNull { it.user } ?: listOf() })
            .distinct()

    private fun extractAccountsSpaceIds(
        accounts: Collection<AccountModel>, resources: Collection<ResourceModel>
    ): List<AccountsSpacesId> = accounts.mapNotNull { it.accountsSpacesId.orElse(null) }
        .plus(resources.mapNotNull { it.accountsSpacesId })
        .distinct()

    private fun extractSegmentIds(
        resources: Collection<ResourceModel>,
        accountsSpaces: Collection<AccountSpaceModel>
    ): List<SegmentId> = resources.flatMap { r -> r.segments.map { it.segmentId } }
        .plus(accountsSpaces.flatMap { s -> s.segments.map { it.segmentId } })

    private fun extractSegmentationIds(
        resources: Collection<ResourceModel>,
        accountsSpaces: Collection<AccountSpaceModel>
    ): List<SegmentationId> = resources.flatMap { r -> r.segments.map { it.segmentationId } }
        .plus(accountsSpaces.flatMap { s -> s.segments.map { it.segmentationId } })

    private suspend fun loadResources(ids: List<ResourceId>): Map<ResourceId, ResourceModel> =
        resourcesLoader.getResourcesByIdsImmediate(ids.map { Tuples.of(it, Tenants.DEFAULT_TENANT_ID) }).awaitSingle()
            .associateBy { it.id }

    private suspend fun loadAccounts(ids: List<AccountId>): Map<AccountId, AccountModel> =
        dbSessionRetryable(tableClient) {
            ids.chunked(1000).flatMap { _ids ->
                accountsDao.getAllByIdsWithDeleted(roStaleSingleRetryableCommit(), _ids, Tenants.DEFAULT_TENANT_ID)
                    .awaitSingle()
            }
        }!!.associateBy { it.id }

    private suspend fun loadFolders(ids: List<FolderId>): Map<FolderId, FolderModel> = dbSessionRetryable(tableClient) {
        ids.chunked(1000).flatMap { _ids ->
            foldersDao.getByIds(roStaleSingleRetryableCommit(), _ids, Tenants.DEFAULT_TENANT_ID).awaitSingle()
        }
    }!!.associateBy { it.id }

    private suspend fun loadProviders(ids: List<ProviderId>): Map<ProviderId, ProviderModel> =
        providersLoader.getProvidersByIdsImmediate(ids.map { Tuples.of(it, Tenants.DEFAULT_TENANT_ID) }).awaitSingle()
            .associateBy { it.id }

    private suspend fun loadServices(ids: List<ServiceId>): Map<ServiceId, ServiceMinimalModel> =
        dbSessionRetryable(tableClient) {
            ids.chunked(1000).flatMap { _ids ->
                servicesDao.getByIdsMinimal(roStaleSingleRetryableCommit(), _ids).awaitSingle()
            }
        }!!.associateBy { it.id }

    private suspend fun loadUsers(ids: List<UserId>): Map<UserId, UserModel> = dbSessionRetryable(tableClient) {
        ids.chunked(1000).flatMap { _ids ->
            usersDao.getByIds(roStaleSingleRetryableCommit(), _ids.map { Tuples.of(it, Tenants.DEFAULT_TENANT_ID) })
                .awaitSingle()
        }
    }!!.associateBy { it.id }

    private suspend fun loadResourceTypes(ids: List<ResourceTypeId>): Map<ResourceTypeId, ResourceTypeModel> =
        resourceTypesLoader.getResourceTypesByIdsImmediate(ids.map { Tuples.of(it, Tenants.DEFAULT_TENANT_ID) })
            .awaitSingle()
            .associateBy { it.id }

    private suspend fun loadAccountsSpaces(ids: List<AccountsSpacesId>): Map<AccountsSpacesId, AccountSpaceModel> =
        dbSessionRetryable(tableClient) {
            ids.chunked(1000).flatMap { _ids ->
                accountsSpacesDao.getByIds(roStaleSingleRetryableCommit(), _ids, Tenants.DEFAULT_TENANT_ID)
                    .awaitSingle()
            }
        }!!.associateBy { it.id }

    private suspend fun loadSegmentations(ids: List<SegmentationId>): Map<SegmentationId, ResourceSegmentationModel> =
        segmentationsLoader.getResourceSegmentationsByIdsImmediate(ids.map { Tuples.of(it, Tenants.DEFAULT_TENANT_ID) })
            .awaitSingle()
            .associateBy { it.id }

    private suspend fun loadSegments(ids: List<SegmentId>): Map<SegmentId, ResourceSegmentModel> =
        segmentsLoader.getResourceSegmentsByIdsImmediate(ids.map { Tuples.of(it, Tenants.DEFAULT_TENANT_ID) })
            .awaitSingle()
            .associateBy { it.id }

    private data class ValidatedSearchLoansRequest(
        val serviceId: ServiceId,
        val status: LoanStatus?,
        val direction: LoanDirection,
        val from: SearchLoansPageToken?,
        val limit: Int
    )

    private data class ValidatedGetLoansHistoryRequest(
        val loanId: LoanId,
        val from: GetLoansHistoryPageToken?,
        val limit: Int
    )

    private data class SearchLoansPageToken(
        val status: LoanStatus,
        val dueAt: Instant,
        val loanId: LoanId
    )

    private data class GetLoansHistoryPageToken(
        val eventTimestamp: Instant,
        val historyId: LoanHistoryId
    )

    private data class ServiceLoan(
        val tenantId: TenantId,
        val serviceId: ServiceId,
        val status: LoanStatus,
        val dueAt: Instant,
        val loanId: LoanId
    ) {
        constructor(inModel: ServiceLoanInModel) : this(
            inModel.tenantId, inModel.serviceId, inModel.status, inModel.dueAt, inModel.loanId
        )

        constructor(outModel: ServiceLoanOutModel) : this(
            outModel.tenantId, outModel.serviceId, outModel.status, outModel.dueAt, outModel.loanId
        )
    }

}
