package ru.yandex.intranet.d.services.local;

import java.security.Principal;
import java.util.Locale;

import com.yandex.ydb.table.transaction.TransactionMode;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import org.springframework.context.annotation.Profile;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuple2;

import ru.yandex.intranet.d.dao.ReadinessDao;
import ru.yandex.intranet.d.datasource.model.YdbTableClient;
import ru.yandex.intranet.d.datasource.utils.ReadinessYdbTableClientHolder;
import ru.yandex.intranet.d.metrics.ProbeMetrics;
import ru.yandex.intranet.d.tms.HourglassSchedulerStarter;
import ru.yandex.intranet.d.util.response.Responses;
import ru.yandex.intranet.d.util.result.ErrorCollection;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.util.result.TypedError;
import ru.yandex.intranet.d.web.errors.Errors;
import ru.yandex.intranet.d.web.security.Auth;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;
import ru.yandex.intranet.d.web.security.tvm.TvmClient;
import ru.yandex.intranet.d.web.security.tvm.model.TvmStatus;

/**
 * Local service.
 *
 * @author Ruslan Kadriev <aqru@yandex-team.ru>
 */
@Component
@Profile({"dev", "testing", "production", "load-testing"})
public class LocalServiceImpl implements LocalService {
    private static final Logger LOG = LoggerFactory.getLogger(LocalServiceImpl.class);
    private static final int RETRY_COUNT_ERROR_LIMIT = 5;
    public static final ResponseEntity<String> OK = ResponseEntity.ok().contentType(MediaType.TEXT_PLAIN).body("OK");
    public static final ResponseEntity<String> ERROR = ResponseEntity.ok().contentType(MediaType.TEXT_PLAIN).body(
            "ERROR");
    public static final ResponseEntity<String> ERROR_5XX = ResponseEntity.status(HttpStatus.SERVICE_UNAVAILABLE)
            .contentType(MediaType.TEXT_PLAIN).body("ERROR");

    private final TvmClient tvmClient;
    private final YdbTableClient tableClient;
    private final ReadinessDao readinessDao;
    private final ProbeMetrics probeMetrics;
    private final MessageSource messages;
    private final HourglassSchedulerStarter schedulerStarter;

    public LocalServiceImpl(TvmClient tvmClient,
                            ReadinessYdbTableClientHolder tableClient,
                            ReadinessDao readinessDao,
                            ProbeMetrics probeMetrics,
                            @Qualifier("messageSource") MessageSource messages,
                            HourglassSchedulerStarter schedulerStarter) {
        this.tvmClient = tvmClient;
        this.tableClient = tableClient.getTableClient();
        this.readinessDao = readinessDao;
        this.probeMetrics = probeMetrics;
        this.messages = messages;
        this.schedulerStarter = schedulerStarter;
    }

    @Override
    public Mono<ResponseEntity<String>> getLiveness() {
        probeMetrics.onLivenessProbe();
        return Mono.just(ResponseEntity.ok().contentType(MediaType.TEXT_PLAIN).body("OK"));
    }

    @Override
    public Mono<ResponseEntity<String>> getReadiness(boolean errorIs5xx) {
        return getZip()
                .map(zip -> {
                    boolean isTvmError = false;
                    boolean isYdbError = false;

                    if (!zip.getT1().getStatus().equals(TvmStatus.Status.OK)) {
                        isTvmError = true;
                        LOG.warn(zip.getT1().getMessage());
                    }

                    if (!zip.getT2().getStatus().equals(ReadinessDao.Status.OK)) {
                        isYdbError = true;
                        LOG.warn(zip.getT2().getMessage());
                    }

                    probeMetrics.onReadinessProbe(isTvmError, isYdbError);

                    if (isTvmError || isYdbError) {
                        return errorIs5xx ? ERROR_5XX : ERROR;
                    }

                    return OK;
                });
    }

    private Mono<Tuple2<TvmStatus, ReadinessDao.StatusWithMessage>> getZip() {
        return Mono.zip(tvmClient.ping()
                        .onErrorResume(e -> Mono.just(new TvmStatus(TvmStatus.Status.ERROR, e.getMessage()))),
                tableClient.usingSessionMonoRetryable(session ->
                        readinessDao.ping(session.asTxCommitRetryable(TransactionMode.STALE_READ_ONLY)))
                        .onErrorResume(e -> Mono.just(ReadinessDao.StatusWithMessage.statusWithMessage(
                                ReadinessDao.Status.ERROR, e.getMessage())))
        );
    }

    @Override
    public Mono<ResponseEntity<?>> getAdminReadiness(Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        return checkAdminPermissions(currentUser, locale).andThenMono(v -> getZip()
                .map(zip -> {
                    ErrorCollection.Builder errors = ErrorCollection.builder();

                    boolean isTvmError = false;
                    boolean isYdbError = false;

                    if (!zip.getT1().getStatus().equals(TvmStatus.Status.OK)) {
                        errors.addDetail("TVM ping", zip.getT1().toString());
                        isTvmError = true;
                    }

                    if (!zip.getT2().getStatus().equals(ReadinessDao.Status.OK)) {
                        errors.addDetail("YDB ping", zip.getT2().toString());
                        isYdbError = true;
                    }

                    probeMetrics.onReadinessProbe(isTvmError, isYdbError);

                    ErrorCollection errorCollection = errors.build();
                    return errorCollection.hasAnyErrors()
                            ? errorCollection.toString()
                            : "OK";
                })
                .map(Result::success))
                .map(r -> r.match(Responses::okJson, Errors::toResponse));
    }

    private Result<Void> checkAdminPermissions(YaUserDetails currentUser, Locale locale) {
        if (currentUser.getUser().isEmpty() || !currentUser.getUser().get().getDAdmin()) {
            return Result.failure(ErrorCollection.builder().addError(TypedError
                    .forbidden(messages.getMessage("errors.access.denied", null, locale))).build());
        }
        return Result.success(null);
    }

    @Override
    public Mono<ResponseEntity<String>> getSchedulerStatus() {
        String status;
        if (schedulerStarter.isReady() || schedulerStarter.getRetryCount() < 1) {
            status = "OK";
        } else {
            if (schedulerStarter.getRetryCount() < RETRY_COUNT_ERROR_LIMIT) {
                status = "WARNING";
            } else {
                status = "ERROR";
            }
        }
        return Mono.just(ResponseEntity.ok().contentType(MediaType.TEXT_PLAIN).body(status));
    }
}
