package ru.yandex.intranet.d.services.notifications;

import java.util.Objects;
import java.util.Optional;

import com.google.common.base.Preconditions;

/**
 * Mail notification.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public final class MailNotification {

    private final String plainTextBody;
    private final String htmlBody;
    private final String subject;
    private final String from;
    private final String fromName;
    private final String to;
    private final String actualTo;

    public MailNotification(String plainTextBody,
                            String htmlBody,
                            String subject,
                            String from,
                            String fromName,
                            String to,
                            String actualTo) {
        this.plainTextBody = plainTextBody;
        this.htmlBody = htmlBody;
        this.subject = subject;
        this.from = from;
        this.fromName = fromName;
        this.to = to;
        this.actualTo = actualTo;
    }

    public static Builder builder() {
        return new Builder();
    }

    public String getPlainTextBody() {
        return plainTextBody;
    }

    public String getHtmlBody() {
        return htmlBody;
    }

    public String getSubject() {
        return subject;
    }

    public String getFrom() {
        return from;
    }

    public String getFromName() {
        return fromName;
    }

    public String getTo() {
        return to;
    }

    public Optional<String> getActualTo() {
        return Optional.ofNullable(actualTo);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        MailNotification that = (MailNotification) o;
        return Objects.equals(plainTextBody, that.plainTextBody) &&
                Objects.equals(htmlBody, that.htmlBody) &&
                Objects.equals(subject, that.subject) &&
                Objects.equals(from, that.from) &&
                Objects.equals(fromName, that.fromName) &&
                Objects.equals(to, that.to) &&
                Objects.equals(actualTo, that.actualTo);
    }

    @Override
    public int hashCode() {
        return Objects.hash(plainTextBody, htmlBody, subject, from, fromName, to, actualTo);
    }

    @Override
    public String toString() {
        return "MailNotification{" +
                "plainTextBody='" + plainTextBody + '\'' +
                ", htmlBody='" + htmlBody + '\'' +
                ", subject='" + subject + '\'' +
                ", from='" + from + '\'' +
                ", fromName='" + fromName + '\'' +
                ", to='" + to + '\'' +
                ", actualTo='" + actualTo + '\'' +
                '}';
    }

    public static final class Builder {

        private String plainTextBody;
        private String htmlBody;
        private String subject;
        private String from;
        private String fromName;
        private String to;
        private String actualTo;

        private Builder() {
        }

        public Builder plainTextBody(String plainTextBody) {
            this.plainTextBody = plainTextBody;
            return this;
        }

        public Builder htmlBody(String htmlBody) {
            this.htmlBody = htmlBody;
            return this;
        }

        public Builder subject(String subject) {
            this.subject = subject;
            return this;
        }

        public Builder from(String from) {
            this.from = from;
            return this;
        }

        public Builder fromName(String fromName) {
            this.fromName = fromName;
            return this;
        }

        public Builder to(String to) {
            this.to = to;
            return this;
        }

        public Builder actualTo(String actualTo) {
            this.actualTo = actualTo;
            return this;
        }

        public MailNotification build() {
            Preconditions.checkNotNull(plainTextBody, "PlainTextBody is required");
            Preconditions.checkNotNull(htmlBody, "HtmlBody is required");
            Preconditions.checkNotNull(subject, "Subject is required");
            Preconditions.checkNotNull(from, "From is required");
            Preconditions.checkNotNull(fromName, "FromName is required");
            Preconditions.checkNotNull(to, "To is required");
            return new MailNotification(plainTextBody, htmlBody, subject, from, fromName, to, actualTo);
        }

    }

}
