package ru.yandex.intranet.d.services.notifications;

import java.util.Locale;
import java.util.Optional;
import java.util.Set;

import javax.annotation.Nullable;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.MessageSource;
import org.springframework.stereotype.Component;
import org.thymeleaf.TemplateEngine;
import org.thymeleaf.context.Context;

import ru.yandex.intranet.d.i18n.Locales;
import ru.yandex.intranet.d.model.accounts.OperationErrorKind;
import ru.yandex.intranet.d.model.users.UserModel;
import ru.yandex.intranet.d.services.transfer.TextFormatterService;

/**
 * Notification mail generator.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class NotificationMailGenerator {

    private static final Logger LOG = LoggerFactory.getLogger(NotificationMailGenerator.class);

    private final TextFormatterService textFormatterService;
    private final TemplateEngine emailTemplateEngine;
    private final MessageSource emailMessageSource;
    private final boolean overrideEnabled;
    private final String overrideDestination;
    private final String fromAddress;
    private final Set<String> doNotDisturb;

    public NotificationMailGenerator(@Qualifier("emailTemplateEngine") TemplateEngine emailTemplateEngine,
                                     @Qualifier("emailMessageSource") MessageSource emailMessageSource,
                                     @Value("${notifications.mail.override.enabled}") boolean overrideEnabled,
                                     @Value("${notifications.mail.override.destination}") String overrideDestination,
                                     @Value("${notifications.mail.from}") String fromAddress,
                                     NotificationsMailExceptionsProperties mailExceptionsProperties,
                                     TextFormatterService textFormatterService) {
        this.textFormatterService = textFormatterService;
        this.emailTemplateEngine = emailTemplateEngine;
        this.emailMessageSource = emailMessageSource;
        this.overrideEnabled = overrideEnabled;
        this.overrideDestination = overrideDestination;
        this.fromAddress = fromAddress;
        this.doNotDisturb = mailExceptionsProperties.getDoNotDisturb();
    }

    public Optional<MailNotification> generateTransferRequestNotification(String transferRequestId,
                                                                          UserModel recipient) {
        if (recipient.getWorkEmail().isEmpty()) {
            LOG.warn("User {} does not have a work e-mail, skipping notification",
                    recipient.getPassportUid().orElse(recipient.getId()));
            return Optional.empty();
        }
        if (recipient.getPassportLogin().isPresent() && doNotDisturb.contains(recipient.getPassportLogin().get())) {
            LOG.info("Skipped e-mail notification for {} due to 'do not disturb' list",
                    recipient.getPassportUid().orElse(recipient.getId()));
            return Optional.empty();
        }
        Locale locale = Locales.forUser(recipient);
        String subject = emailMessageSource.getMessage("notification.mail.quota.transfer.subject", null, locale);
        String fromName = emailMessageSource.getMessage("notification.mail.from.name", null, locale);
        Context context = new Context(locale);
        context.setVariable("transferUrl", textFormatterService.buildTransferUrl(transferRequestId));
        context.setVariable("allTransfersUrl", textFormatterService.buildAllTransfersUrl());
        String htmlBody = emailTemplateEngine.process("html/quota_transfer.html", context);
        String plainTextBody = emailTemplateEngine.process("text/quota_transfer.txt", context);
        String toAddress = overrideEnabled ? overrideDestination : recipient.getWorkEmail().get();
        String actualTo = overrideEnabled ? recipient.getWorkEmail().get() : null;
        return Optional.of(MailNotification.builder()
                .htmlBody(htmlBody)
                .plainTextBody(plainTextBody)
                .subject(subject)
                .from(fromAddress)
                .to(toAddress)
                .actualTo(actualTo)
                .fromName(fromName)
                .build());
    }

    public String generateHtmlFailedOperationNotification(String operationId, @Nullable String transferRequestId,
                                                          @Nullable OperationErrorKind errorKind, Locale locale) {
        Context context = new Context(locale);
        if (transferRequestId != null) {
            context.setVariable("transferUrl", textFormatterService.buildTransferUrl(transferRequestId));
        }
        context.setVariable("operationUrl", textFormatterService.buildOperationApiUrl(operationId));
        context.setVariable("errorKind", errorKind != null ? errorKind.name() : "-");
        return transferRequestId != null
                ? emailTemplateEngine.process("html/transfer_operation_failure.html", context)
                : emailTemplateEngine.process("html/operation_failure.html", context);
    }

    public String generateTextFailedOperationNotification(String operationId, @Nullable String transferRequestId,
                                                          @Nullable OperationErrorKind errorKind, Locale locale) {
        Context context = new Context(locale);
        if (transferRequestId != null) {
            context.setVariable("transferUrl", textFormatterService.buildTransferUrl(transferRequestId));
        }
        context.setVariable("operationUrl", textFormatterService.buildOperationApiUrl(operationId));
        context.setVariable("errorKind", errorKind != null ? errorKind.name() : "-");
        return transferRequestId != null
                ? emailTemplateEngine.process("text/transfer_operation_failure.txt", context)
                : emailTemplateEngine.process("text/operation_failure.txt", context);
    }

}
