package ru.yandex.intranet.d.services.notifications;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.time.Duration;

import javax.mail.MessagingException;
import javax.mail.internet.MimeMessage;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.SmartLifecycle;
import org.springframework.context.annotation.Profile;
import org.springframework.mail.javamail.JavaMailSender;
import org.springframework.mail.javamail.MimeMessageHelper;
import org.springframework.stereotype.Component;
import reactor.core.Exceptions;
import reactor.core.publisher.Mono;
import reactor.core.scheduler.Scheduler;
import reactor.core.scheduler.Schedulers;
import reactor.util.retry.Retry;
import reactor.util.retry.RetrySpec;

/**
 * Notification mail sender service implementation.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
@Profile({"dev", "testing", "production"})
public class NotificationMailSenderImpl implements NotificationMailSender, SmartLifecycle {

    private static final Logger LOG = LoggerFactory.getLogger(NotificationMailSenderImpl.class);

    private static final int MAX_THREADS = 10;
    private static final int MAX_QUEUE = 10000;
    private static final int TTL_SECONDS = 60;
    private static final long RETRY_ATTEMPTS = 3;
    private static final Duration MIN_RETRY_BACKOFF = Duration.ofSeconds(1);
    private static final String ACTUAL_TO_HEADER = "X-Abcd-To";

    private final JavaMailSender mailSender;
    private final Scheduler scheduler;
    private volatile boolean running = false;

    public NotificationMailSenderImpl(JavaMailSender mailSender) {
        this.mailSender = mailSender;
        this.scheduler = Schedulers.newBoundedElastic(MAX_THREADS, MAX_QUEUE, "MailSenderPool", TTL_SECONDS, true);
    }

    @Override
    public void sendBackground(MailNotification notification) {
        try {
            Mono.just(notification)
                    .publishOn(scheduler)
                    .flatMap(this::send)
                    .subscribe(v -> { }, e -> LOG.error("Failed to send mail notification", e));
        } catch (Exception e) {
            LOG.error("Failed to send mail notification", e);
        }
    }

    private Mono<Void> send(MailNotification notification) {
        MimeMessage message = prepareMessage(notification);
        return Mono.<Void>fromRunnable(() -> {
            try {
                mailSender.send(message);
                LOG.info("Sent email message {}", message.getMessageID());
            } catch (MessagingException e) {
                throw new RuntimeException(e);
            }
        }).retryWhen(retryRequest());
    }

    private MimeMessage prepareMessage(MailNotification notification) {
        try {
            MimeMessage message = mailSender.createMimeMessage();
            MimeMessageHelper messageHelper = new MimeMessageHelper(message, true, "UTF-8");
            messageHelper.setValidateAddresses(true);
            messageHelper.setText(notification.getPlainTextBody(), notification.getHtmlBody());
            messageHelper.setSubject(notification.getSubject());
            messageHelper.setFrom(notification.getFrom(), notification.getFromName());
            messageHelper.setTo(notification.getTo());
            if (notification.getActualTo().isPresent()) {
                message.addHeader(ACTUAL_TO_HEADER, notification.getActualTo().get());
            }
            message.saveChanges();
            return message;
        } catch (MessagingException e) {
            throw new RuntimeException(e);
        } catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    @Override
    public void start() {
        running = true;
    }

    @Override
    public void stop() {
        scheduler.dispose();
        running = false;
    }

    @Override
    public boolean isRunning() {
        return running;
    }

    private Retry retryRequest() {
        return RetrySpec.backoff(RETRY_ATTEMPTS, MIN_RETRY_BACKOFF).filter(e -> !Exceptions.isRetryExhausted(e));
    }

}
