package ru.yandex.intranet.d.services.operations;

import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;
import java.util.function.Function;
import java.util.stream.Collectors;

import io.grpc.Status;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.MessageSource;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuple3;
import reactor.util.function.Tuples;

import ru.yandex.intranet.d.dao.Tenants;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.model.accounts.AccountModel;
import ru.yandex.intranet.d.model.accounts.AccountSpaceModel;
import ru.yandex.intranet.d.model.accounts.AccountsQuotasModel;
import ru.yandex.intranet.d.model.accounts.AccountsQuotasOperationsModel;
import ru.yandex.intranet.d.model.accounts.OperationChangesModel;
import ru.yandex.intranet.d.model.accounts.OperationErrorKind;
import ru.yandex.intranet.d.model.accounts.OperationOrdersModel;
import ru.yandex.intranet.d.model.folders.AccountHistoryModel;
import ru.yandex.intranet.d.model.folders.AccountsHistoryModel;
import ru.yandex.intranet.d.model.folders.FolderModel;
import ru.yandex.intranet.d.model.folders.FolderOperationLogModel;
import ru.yandex.intranet.d.model.folders.FolderOperationType;
import ru.yandex.intranet.d.model.folders.OperationPhase;
import ru.yandex.intranet.d.model.folders.ProvisionHistoryModel;
import ru.yandex.intranet.d.model.folders.ProvisionsByResource;
import ru.yandex.intranet.d.model.folders.QuotasByAccount;
import ru.yandex.intranet.d.model.folders.QuotasByResource;
import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.model.quotas.QuotaModel;
import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.services.accounts.ReserveAccountsService;
import ru.yandex.intranet.d.services.integration.providers.ProviderError;
import ru.yandex.intranet.d.services.integration.providers.Response;
import ru.yandex.intranet.d.services.integration.providers.rest.model.ErrorMessagesDto;
import ru.yandex.intranet.d.services.operations.model.CreateAccountApplicationContext;
import ru.yandex.intranet.d.services.operations.model.CreateAccountContext;
import ru.yandex.intranet.d.services.operations.model.CreateAccountOperationPostRefreshContext;
import ru.yandex.intranet.d.services.operations.model.CreateAccountOperationPreRefreshContext;
import ru.yandex.intranet.d.services.operations.model.CreateAccountOperationRefreshContext;
import ru.yandex.intranet.d.services.operations.model.CreateAccountOperationRetryContext;
import ru.yandex.intranet.d.services.operations.model.OperationCommonContext;
import ru.yandex.intranet.d.services.operations.model.OperationPostRefreshContext;
import ru.yandex.intranet.d.services.operations.model.OperationPreRefreshContext;
import ru.yandex.intranet.d.services.operations.model.OperationRefreshContext;
import ru.yandex.intranet.d.services.operations.model.OperationRetryContext;
import ru.yandex.intranet.d.services.operations.model.PostRetryRefreshResult;
import ru.yandex.intranet.d.services.operations.model.ReceivedAccount;
import ru.yandex.intranet.d.services.operations.model.RefreshResult;
import ru.yandex.intranet.d.services.operations.model.RetryResult;
import ru.yandex.intranet.d.services.operations.model.RetryableOperation;
import ru.yandex.intranet.d.services.operations.model.ValidatedReceivedAccount;
import ru.yandex.intranet.d.services.operations.model.ValidatedReceivedProvision;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.web.errors.Errors;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;

import static ru.yandex.intranet.d.services.operations.OperationUtils.isAnotherRoundAfterRefresh;

/**
 * Create account operations retry service implementation.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class CreateAccountOperationsRetryService implements BaseOperationRetryService {

    private static final Logger LOG = LoggerFactory.getLogger(CreateAccountOperationsRetryService.class);

    private final OperationsRetryIntegrationService integrationService;
    private final OperationsRetryValidationService validationService;
    private final OperationsRetryStoreService storeService;
    private final OperationsObservabilityService operationsObservabilityService;
    private final ReserveAccountsService reserveAccountsService;
    private final MessageSource messages;
    private long maxAsyncRetries;

    public CreateAccountOperationsRetryService(OperationsRetryIntegrationService integrationService,
                                               OperationsRetryValidationService validationService,
                                               OperationsRetryStoreService storeService,
                                               OperationsObservabilityService operationsObservabilityService,
                                               ReserveAccountsService reserveAccountsService,
                                               @Qualifier("messageSource") MessageSource messages,
                                               @Value("${providers.client.maxAsyncRetries}") long maxAsyncRetries
    ) {
        this.integrationService = integrationService;
        this.validationService = validationService;
        this.storeService = storeService;
        this.operationsObservabilityService = operationsObservabilityService;
        this.reserveAccountsService = reserveAccountsService;
        this.messages = messages;
        this.maxAsyncRetries = maxAsyncRetries;
    }

    @Override
    public Mono<Optional<CreateAccountOperationPreRefreshContext>> preRefreshOperation(
            YdbTxSession session,
            RetryableOperation operation,
            OperationCommonContext context,
            Instant now,
            Locale locale) {
        AccountsQuotasOperationsModel op = operation.getOperation();
        return storeService.loadCreateAccountContext(session, op).flatMap(createAccountContext -> {
            if (createAccountContext.getTargetAccount().isPresent()) {
                AccountsQuotasOperationsModel closedOperation = new AccountsQuotasOperationsModel.Builder(op)
                        .setUpdateDateTime(now)
                        .setRequestStatus(AccountsQuotasOperationsModel.RequestStatus.OK)
                        .setErrorKind(null)
                        .setErrorMessage(null)
                        .build();
                operationsObservabilityService.observeOperationFinished(closedOperation);
                return storeService.upsertOperation(session, closedOperation)
                        .then(Mono.defer(() -> storeService.deleteOperationsInProgress(session,
                                operation.getInProgress()))).thenReturn(Optional.empty());
            }
            return Mono.just(Optional.of(new CreateAccountOperationPreRefreshContext(context, createAccountContext)));
        });
    }

    @Override
    public Mono<Optional<CreateAccountOperationRefreshContext>> refreshOperation(
            RetryableOperation operation,
            OperationPreRefreshContext context,
            Instant now,
            Locale locale) {
        CreateAccountOperationPreRefreshContext preRefreshContext = (CreateAccountOperationPreRefreshContext) context;
        return integrationService.listAccountsByFolder(preRefreshContext.getCreateAccountContext(),
                preRefreshContext.getCommonContext(), locale).map(r ->
                        Optional.of(new CreateAccountOperationRefreshContext(preRefreshContext, r)));
    }

    @Override
    public Mono<Optional<CreateAccountOperationPostRefreshContext>> postRefreshOperation(
            YdbTxSession session,
            RetryableOperation operation,
            OperationRefreshContext context,
            Instant now,
            Locale locale) {
        CreateAccountOperationRefreshContext refreshContext = (CreateAccountOperationRefreshContext) context;
        return refreshContext.getRefreshResult().andThenMono(resp -> resp.match(
                (acc, requestId) -> validationService.validateReceivedAccounts(session, acc,
                        context.getCommonContext(), locale).map(r -> r.apply(a -> Response.success(a, requestId))),
                e -> Mono.just(Result.success(Response.<List<ValidatedReceivedAccount>>failure(e))),
                (e, requestId) -> Mono.just(Result.success(Response
                        .<List<ValidatedReceivedAccount>>error(e, requestId)))))
                .map(v -> Optional.of(new CreateAccountOperationPostRefreshContext(refreshContext, v)));
    }

    @Override
    public Mono<Optional<CreateAccountOperationRetryContext>> retryOperation(
            RetryableOperation operation,
            OperationPostRefreshContext context,
            Instant now,
            Locale locale) {
        CreateAccountOperationPostRefreshContext postRefreshContext
                = (CreateAccountOperationPostRefreshContext) context;
        RefreshResult refreshResult = getRefreshResult(operation, postRefreshContext);
        Optional<ValidatedReceivedAccount> alreadyCreatedAccount = refreshResult == RefreshResult.OPERATION_APPLIED
                ? getRefreshedAccount(operation, postRefreshContext) : Optional.empty();
        Optional<String> refreshError = getRefreshErrorDescription(postRefreshContext, refreshResult, locale);
        if (isNoRetryAfterRefresh(refreshResult, context.getCommonContext())) {
            return Mono.just(Optional.of(new CreateAccountOperationRetryContext(postRefreshContext, refreshResult,
                    alreadyCreatedAccount.orElse(null), refreshError.orElse(null), null,
                    null, null, null, null, null,
                    UUID.randomUUID().toString())));
        }
        CreateAccountContext createAccountContext = postRefreshContext.getPreRefreshContext().getCreateAccountContext();
        OperationCommonContext commonContext = postRefreshContext.getCommonContext();
        return integrationService.createAccount(createAccountContext, commonContext, operation.getOperation(), locale)
                .flatMap(createResult -> {
                    RetryResult retryResult = getRetryResult(operation, createResult);
                    Optional<ReceivedAccount> createdAccount = getCreatedAccount(createResult);
                    Optional<String> retryError = getRetryErrorDescription(createResult, retryResult, locale);
                    if (retryResult != RetryResult.SUCCESS) {
                        return integrationService.listAccountsByFolder(createAccountContext, commonContext, locale)
                                .map(postRetryRefresh -> {
                                    PostRetryRefreshResult postRetryRefreshResult
                                            = getPostRetryRefreshResult(operation, postRetryRefresh);
                                    Optional<List<ReceivedAccount>> postRetryRefreshedAccounts
                                            = getPostRetryRefreshedAccounts(postRetryRefresh);
                                    Optional<String> postRetryRefreshError = getPostRetryRefreshErrorDescription(
                                            postRetryRefresh, postRetryRefreshResult, locale);
                                    return Optional.of(new CreateAccountOperationRetryContext(postRefreshContext,
                                            refreshResult, alreadyCreatedAccount.orElse(null),
                                            refreshError.orElse(null), retryResult,
                                            createdAccount.orElse(null), retryError.orElse(null),
                                            postRetryRefreshResult, postRetryRefreshedAccounts.orElse(null),
                                            postRetryRefreshError.orElse(null), UUID.randomUUID().toString()));
                                });
                    }
                    return Mono.just(Optional.of(new CreateAccountOperationRetryContext(postRefreshContext,
                            refreshResult, alreadyCreatedAccount.orElse(null), refreshError.orElse(null),
                            retryResult, createdAccount.orElse(null), retryError.orElse(null),
                            null, null, null,
                            UUID.randomUUID().toString())));
                });
    }

    @Override
    public Mono<Void> postRetryOperation(YdbTxSession session,
                                         RetryableOperation operation,
                                         OperationRetryContext context,
                                         Instant now,
                                         Locale locale) {
        CreateAccountOperationRetryContext retryContext = (CreateAccountOperationRetryContext) context;
        if (isAnotherRoundAfterRefresh(retryContext.getRefreshResult())) {
            return planToNextRetryOperation(session, operation, retryContext, now, OperationErrorKind.EXPIRED);
        } else if (isOperationAppliedAfterRefresh(retryContext.getRefreshResult())
                && retryContext.getAlreadyCreatedAccount().isPresent()) {
            return completeOnRefreshAlreadyApplied(session, operation, retryContext, now);
        } else if (isRetryUnsafeAfterRefresh(retryContext.getRefreshResult(), retryContext.getCommonContext())) {
            return abortOperation(session, operation, retryContext, now, OperationErrorKind.UNDEFINED);
        } else if (retryContext.getRetryResult().get() == RetryResult.FATAL_FAILURE) {
            if (retryContext.getPostRetryRefreshResult().get() == PostRetryRefreshResult.FATAL_FAILURE) {
                return abortOperation(session, operation, retryContext, now, OperationErrorKind.INVALID_ARGUMENT);
            } else if (retryContext.getPostRetryRefreshResult().get() == PostRetryRefreshResult.NON_FATAL_FAILURE) {
                return planToNextRetryOperation(session, operation, retryContext, now,
                        OperationErrorKind.INVALID_ARGUMENT);
            } else if (retryContext.getPostRetryRefreshResult().get() == PostRetryRefreshResult.UNSUPPORTED) {
                return abortOperation(session, operation, retryContext, now, OperationErrorKind.INVALID_ARGUMENT);
            } else if (retryContext.getPostRetryRefreshResult().get() == PostRetryRefreshResult.SUCCESS) {
                return onRetryNoSuccessFinish(session, operation, retryContext, now, locale,
                        OperationErrorKind.INVALID_ARGUMENT);
            } else {
                return Mono.error(new IllegalArgumentException("Unexpected operation retry state for operation "
                        + operation + ": " + context));
            }
        } else if (retryContext.getRetryResult().get() == RetryResult.NON_FATAL_FAILURE) {
            return planToNextRetryOperation(session, operation, retryContext, now, OperationErrorKind.EXPIRED);
        } else if (retryContext.getRetryResult().get() == RetryResult.SUCCESS) {
            return onRetrySuccess(session, operation, retryContext, now, locale);
        } else if (retryContext.getRetryResult().get() == RetryResult.CONFLICT) {
            if (retryContext.getPostRetryRefreshResult().get() == PostRetryRefreshResult.FATAL_FAILURE) {
                return abortOperation(session, operation, retryContext, now, OperationErrorKind.ALREADY_EXISTS);
            } else if (retryContext.getPostRetryRefreshResult().get() == PostRetryRefreshResult.NON_FATAL_FAILURE) {
                return planToNextRetryOperation(session, operation, retryContext, now,
                        OperationErrorKind.ALREADY_EXISTS);
            } else if (retryContext.getPostRetryRefreshResult().get() == PostRetryRefreshResult.UNSUPPORTED) {
                return abortOperation(session, operation, retryContext, now, OperationErrorKind.ALREADY_EXISTS);
            } else if (retryContext.getPostRetryRefreshResult().get() == PostRetryRefreshResult.SUCCESS) {
                return onRetryNoSuccessFinish(session, operation, retryContext, now, locale,
                        OperationErrorKind.ALREADY_EXISTS);
            } else {
                return Mono.error(new IllegalArgumentException("Unexpected operation retry state for operation "
                        + operation + ": " + context));
            }
        } else {
            return Mono.error(new IllegalArgumentException("Unexpected operation retry state for operation "
                    + operation + ": " + context));
        }
    }

    @Override
    public Mono<Void> abortOperation(YdbTxSession session, AccountsQuotasOperationsModel operation, String comment,
                                     Instant now, YaUserDetails currentUser, Locale locale) {
        AccountsQuotasOperationsModel updatedOperation = new AccountsQuotasOperationsModel
                .Builder(operation)
                .setRequestStatus(AccountsQuotasOperationsModel.RequestStatus.ERROR)
                .setUpdateDateTime(now)
                .setErrorMessage(comment)
                .setErrorKind(OperationErrorKind.ABORTED)
                .build();
        operationsObservabilityService.observeOperationFinished(updatedOperation);
        return storeService.upsertOperation(session, updatedOperation)
                .then(Mono.defer(() -> storeService.deleteInProgressByOperationId(session,
                        operation.getOperationId())));
    }

    private Mono<Void> abortOperation(YdbTxSession session,
                                      RetryableOperation operation,
                                      CreateAccountOperationRetryContext retryContext,
                                      Instant now,
                                      OperationErrorKind errorKind) {
        if (!isOperationNotComplete(operation)) {
            return Mono.empty();
        }
        return saveAbortOperation(session, operation, retryContext, now, errorKind);
    }

    private Mono<Void> planToNextRetryOperation(
            YdbTxSession session,
            RetryableOperation operation,
            CreateAccountOperationRetryContext retryContext,
            Instant now,
            OperationErrorKind errorKind
    ) {
        Optional<Boolean> maxAsyncRetriesReached = operation.getInProgress().stream().findFirst().map(op ->
                op.getRetryCounter() >= maxAsyncRetries);
        if (maxAsyncRetriesReached.isPresent() && maxAsyncRetriesReached.get()) {
            return abortOperation(session, operation, retryContext, now, errorKind);
        }
        operationsObservabilityService.observeOperationTransientFailure(operation.getOperation());
        return storeService.incrementRetryCounter(session, operation.getInProgress());
    }

    private Mono<Void> saveAbortOperation(YdbTxSession session, RetryableOperation operation,
                                          CreateAccountOperationRetryContext retryContext, Instant now,
                                          OperationErrorKind errorKind) {
        String errorMessage = retryContext.getReceivedRetryError().orElse(retryContext
                .getReceivedPostRetryRefreshError().orElse(retryContext.getReceivedRefreshError().orElse(null)));
        AccountsQuotasOperationsModel updatedOperation = new AccountsQuotasOperationsModel
                .Builder(operation.getOperation())
                .setRequestStatus(AccountsQuotasOperationsModel.RequestStatus.ERROR)
                .setUpdateDateTime(now)
                .setErrorMessage(errorMessage)
                .setErrorKind(errorKind)
                .build();
        operationsObservabilityService.observeOperationFinished(updatedOperation);
        return storeService.finishOperation(session, operation.getInProgress(), updatedOperation);
    }

    private boolean isOperationNotComplete(RetryableOperation operation) {
        return operation.getOperation().getRequestStatus().isEmpty() || operation.getOperation()
                .getRequestStatus().get() == AccountsQuotasOperationsModel.RequestStatus.WAITING;
    }

    private Mono<Void> completeOnRefreshAlreadyApplied(YdbTxSession session,
                                                       RetryableOperation operation,
                                                       CreateAccountOperationRetryContext retryContext,
                                                       Instant now) {
        if (!isOperationNotComplete(operation)) {
            return Mono.empty();
        }
        ValidatedReceivedAccount alreadyCreatedAccount = retryContext.getAlreadyCreatedAccount().get();
        return storeService.loadCreateAccountApplicationContext(session, operation.getOperation(),
                alreadyCreatedAccount.getAccountId()).flatMap(appCtx ->
                applyOperation(session, operation, retryContext, alreadyCreatedAccount, appCtx, now));
    }

    private Mono<Void> applyOperation(YdbTxSession session,
                                      RetryableOperation operation,
                                      CreateAccountOperationRetryContext retryContext,
                                      ValidatedReceivedAccount receivedAccount,
                                      CreateAccountApplicationContext appCtx,
                                      Instant now) {
        if (appCtx.getTargetAccountByInternalId().isPresent()
                || appCtx.getTargetAccountByExternalId().isPresent()) {
            AccountsQuotasOperationsModel updatedOperation = new AccountsQuotasOperationsModel
                    .Builder(operation.getOperation())
                    .setRequestStatus(AccountsQuotasOperationsModel.RequestStatus.OK)
                    .setUpdateDateTime(now)
                    .setErrorKind(null)
                    .build();
            operationsObservabilityService.observeOperationFinished(updatedOperation);
            return storeService.finishOperation(session, operation.getInProgress(), updatedOperation);
        }
        OperationCommonContext commonContext = retryContext.getCommonContext();
        FolderModel folder = appCtx.getFolder();
        ProviderModel provider = commonContext.getProvider();
        AccountModel account = prepareNewAccount(operation, receivedAccount, folder, provider, now);
        return prepareDefaultQuotas(session, account, receivedAccount.getProvisions()).flatMap(defaultQuotas -> {
            List<QuotaModel> oldQuotas = defaultQuotas.getT1();
            List<QuotaModel> updatedQuotas = defaultQuotas.getT2();
            List<AccountsQuotasModel> accountQuotas = defaultQuotas.getT3();
            FolderModel updatedFolder = folder.toBuilder()
                    .setNextOpLogOrder(folder.getNextOpLogOrder() + 1L)
                    .build();
            String preGeneratedFolderOpLogId = retryContext.getPreGeneratedFolderOpLogId();
            AccountsQuotasOperationsModel updatedOperation = new AccountsQuotasOperationsModel
                    .Builder(operation.getOperation())
                    .setRequestStatus(AccountsQuotasOperationsModel.RequestStatus.OK)
                    .setUpdateDateTime(now)
                    .setOrders(OperationOrdersModel.builder(operation.getOperation().getOrders())
                            .closeOrder(folder.getNextOpLogOrder())
                            .build())
                    .setErrorKind(null)
                    .build();
            operationsObservabilityService.observeOperationFinished(updatedOperation);
            return reserveAccountsService.adjustForReserveConflictMono(session, account).flatMap(adjustedAccount -> {
                FolderOperationLogModel folderOpLog = prepareNewFolderOpLog(operation, commonContext, folder,
                        adjustedAccount, preGeneratedFolderOpLogId, now, oldQuotas, updatedQuotas, accountQuotas);
                    return storeService.createAccount(session, adjustedAccount, updatedFolder, folderOpLog,
                                    updatedQuotas, accountQuotas)
                    .then(Mono.defer(() -> reserveAccountsService.addReserveAccountMono(session, adjustedAccount)))
                    .then(Mono.defer(() -> storeService.finishOperation(
                            session, operation.getInProgress(), updatedOperation
                    )));
            });
        });
    }

    private Mono<Tuple3<List<QuotaModel>, List<QuotaModel>, List<AccountsQuotasModel>>> prepareDefaultQuotas(
            YdbTxSession session, AccountModel account, List<ValidatedReceivedProvision> provisions
    ) {
        if (provisions.isEmpty()) {
            return Mono.just(Tuples.of(List.of(), List.of(), List.of()));
        }
        return storeService.loadFolderQuotas(session, account.getFolderId(), account.getProviderId())
                .map(folderQuotas -> {
                    var folderQuotasByResourceId = folderQuotas.stream().collect(Collectors
                            .toMap(QuotaModel::getResourceId, Function.identity()));
                    List<AccountsQuotasModel> updatedProvisions = new ArrayList<>();
                    List<QuotaModel> oldQuotas = new ArrayList<>();
                    List<QuotaModel> updatedQuotas = new ArrayList<>();
                    for (ValidatedReceivedProvision provision : provisions) {
                        ResourceModel resource = provision.getResource();
                        String resourceId = resource.getId();
                        long provided = provision.getProvidedAmount();
                        updatedProvisions.add(new AccountsQuotasModel.Builder()
                                .setTenantId(account.getTenantId())
                                .setAccountId(account.getId())
                                .setResourceId(resourceId)
                                .setProvidedQuota(provided)
                                .setAllocatedQuota(provision.getAllocatedAmount())
                                .setFolderId(account.getFolderId())
                                .setProviderId(account.getProviderId())
                                .setLastProvisionUpdate(account.getLastAccountUpdate())
                                .setLatestSuccessfulProvisionOperationId(
                                        account.getLatestSuccessfulAccountOperationId().orElse(null))
                                .setLastReceivedProvisionVersion(null)
                                .build()
                        );
                        long defaultQuota = resource.getDefaultQuota().orElse(0L);
                        Optional<QuotaModel> oldQuotaModel = Optional.ofNullable(
                                folderQuotasByResourceId.get(resourceId));
                        long oldQuota = oldQuotaModel.map(QuotaModel::getQuota).orElse(0L);
                        long oldBalance = oldQuotaModel.map(QuotaModel::getBalance).orElse(0L);
                        long oldFrozenQuota = oldQuotaModel.map(QuotaModel::getFrozenQuota).orElse(0L);
                        QuotaModel newQuota = QuotaModel.builder()
                                .tenantId(account.getTenantId())
                                .folderId(account.getFolderId())
                                .providerId(account.getProviderId())
                                .resourceId(resourceId)
                                .quota(oldQuota + defaultQuota)
                                .balance(oldBalance + defaultQuota - provided)
                                .frozenQuota(oldFrozenQuota)
                                .build();
                        updatedQuotas.add(newQuota);
                        oldQuotaModel.ifPresent(oldQuotas::add);
                    }
                    return Tuples.of(oldQuotas, updatedQuotas, updatedProvisions);
                });
    }

    @SuppressWarnings("ParameterNumber")
    private FolderOperationLogModel prepareNewFolderOpLog(
            RetryableOperation operation,
            OperationCommonContext commonContext,
            FolderModel folder,
            AccountModel account,
            String preGeneratedFolderOpLogId,
            Instant now,
            List<QuotaModel> oldQuotas,
            List<QuotaModel> updatedQuotas,
            List<AccountsQuotasModel> accountQuotas
    ) {
        AccountHistoryModel accountHistoryModel = AccountHistoryModel.builder()
                .version(account.getVersion())
                .providerId(account.getProviderId())
                .outerAccountIdInProvider(account.getOuterAccountIdInProvider())
                .outerAccountKeyInProvider(account.getOuterAccountKeyInProvider().orElse(null))
                .folderId(account.getFolderId())
                .displayName(account.getDisplayName().orElse(null))
                .deleted(account.isDeleted())
                .lastReceivedVersion(account.getLastReceivedVersion().orElse(null))
                .accountsSpacesId(account.getAccountsSpacesId().orElse(null))
                .reserveType(account.getReserveType().orElse(null))
                .build();
        QuotasByResource newQuotasByResource = new QuotasByResource(updatedQuotas.stream()
                .collect(Collectors.toMap(QuotaModel::getResourceId, QuotaModel::getQuota)));
        QuotasByResource oldFolderQuotasByResource = new QuotasByResource(oldQuotas.stream()
                .collect(Collectors.toMap(QuotaModel::getResourceId, QuotaModel::getQuota)));
        QuotasByResource newBalancesByResource = new QuotasByResource(updatedQuotas.stream()
                .collect(Collectors.toMap(QuotaModel::getResourceId, QuotaModel::getBalance)));
        QuotasByResource oldBalancesByResource = new QuotasByResource(oldQuotas.stream()
                .collect(Collectors.toMap(QuotaModel::getResourceId, QuotaModel::getQuota)));
        QuotasByAccount quotasByAccount = new QuotasByAccount(accountQuotas.isEmpty() ? Map.of() :
                Map.of(account.getId(), new ProvisionsByResource(accountQuotas.stream().collect(Collectors.toMap(
                        AccountsQuotasModel::getResourceId,
                        q -> new ProvisionHistoryModel(
                                q.getProvidedQuota(),
                                q.getLastReceivedProvisionVersion().orElse(0L)
                        )
                )))));
        return FolderOperationLogModel.builder()
                .setTenantId(folder.getTenantId())
                .setFolderId(folder.getId())
                .setOperationDateTime(now)
                .setId(preGeneratedFolderOpLogId)
                .setProviderRequestId(operation.getOperation().getLastRequestId().orElse(null))
                .setOperationType(FolderOperationType.CREATE_ACCOUNT)
                .setAuthorUserId(operation.getOperation().getAuthorUserId())
                .setAuthorUserUid(commonContext.getAuthor().getPassportUid().orElse(null))
                .setAuthorProviderId(null)
                .setSourceFolderOperationsLogId(null)
                .setDestinationFolderOperationsLogId(null)
                .setOldFolderFields(null)
                .setNewFolderFields(null)
                .setOldQuotas(oldFolderQuotasByResource)
                .setNewQuotas(newQuotasByResource)
                .setOldBalance(oldBalancesByResource)
                .setNewBalance(newBalancesByResource)
                .setOldProvisions(new QuotasByAccount(Collections.emptyMap()))
                .setNewProvisions(quotasByAccount)
                .setActuallyAppliedProvisions(null)
                .setOldAccounts(null)
                .setNewAccounts(new AccountsHistoryModel(Map.of(account.getId(), accountHistoryModel)))
                .setAccountsQuotasOperationsId(operation.getOperation().getOperationId())
                .setQuotasDemandsId(null)
                .setOperationPhase(OperationPhase.CLOSE)
                .setOrder(folder.getNextOpLogOrder())
                .build();
    }

    private AccountModel prepareNewAccount(RetryableOperation operation,
                                           ValidatedReceivedAccount alreadyCreatedAccount,
                                           FolderModel folder,
                                           ProviderModel provider,
                                           Instant now) {
        OperationChangesModel.AccountCreateParams accountCreateParams = operation.getOperation()
                .getRequestedChanges().getAccountCreateParams().get();
        boolean accountKeySupported = provider.getAccountsSettings().isKeySupported();
        boolean accountDisplayNameSupported = provider.getAccountsSettings().isDisplayNameSupported();
        boolean accountAndProvisionsVersionedTogether = provider.getAccountsSettings()
                .isPerAccountVersionSupported() && !provider.getAccountsSettings()
                .isPerProvisionVersionSupported();
        boolean accountVersionedSeparately = provider.getAccountsSettings().isPerAccountVersionSupported()
                && provider.getAccountsSettings().isPerProvisionVersionSupported();
        return new AccountModel.Builder()
                .setTenantId(Tenants.DEFAULT_TENANT_ID)
                .setId(accountCreateParams.getAccountId())
                .setVersion(0L)
                .setProviderId(provider.getId())
                .setOuterAccountIdInProvider(alreadyCreatedAccount.getAccountId())
                .setOuterAccountKeyInProvider(accountKeySupported
                        ? alreadyCreatedAccount.getKey().orElse(null) : null)
                .setFolderId(folder.getId())
                .setDisplayName(accountDisplayNameSupported
                        ? alreadyCreatedAccount.getDisplayName().orElse(null) : null)
                .setDeleted(false)
                .setLastAccountUpdate(now)
                .setLastReceivedVersion(accountAndProvisionsVersionedTogether || accountVersionedSeparately
                        ? alreadyCreatedAccount.getAccountVersion().orElse(null) : null)
                .setLatestSuccessfulAccountOperationId(operation.getOperation().getOperationId())
                .setAccountsSpacesId(alreadyCreatedAccount.getAccountsSpace()
                        .map(AccountSpaceModel::getId).orElse(null))
                .setReserveType(accountCreateParams.getReserveType().orElse(null))
                .build();
    }

    private Mono<Void> onRetrySuccess(YdbTxSession session,
                                      RetryableOperation operation,
                                      CreateAccountOperationRetryContext retryContext,
                                      Instant now,
                                      Locale locale) {
        if (!isOperationNotComplete(operation)) {
            return Mono.empty();
        }
        ReceivedAccount receivedAccount = retryContext.getCreatedAccount().get();
        OperationCommonContext commonContext = retryContext.getCommonContext();
        String operationId = operation.getOperation().getOperationId();
        return validationService.validateReceivedAccount(session, receivedAccount, commonContext, locale)
                .flatMap(validatedR -> validatedR.match(validatedAccount ->
                                storeService.loadCreateAccountApplicationContext(session, operation.getOperation(),
                                        validatedAccount.getAccountId()).flatMap(appCtx ->
                                                applyOperation(session, operation, retryContext, validatedAccount,
                                                        appCtx, now)),
                    e -> {
                        LOG.error("Failed to process account creation response for operation {}: {}",
                                operationId, e);
                        return Mono.empty();
                    }));
    }

    private Optional<ValidatedReceivedAccount> getRefreshedAccount(
            List<ValidatedReceivedAccount> receivedAccounts,
            RetryableOperation operation,
            OperationCommonContext commonContext) {
        String operationId = operation.getOperation().getOperationId();
        boolean accountKeySupported = commonContext.getProvider().getAccountsSettings().isKeySupported();
        boolean operationIdSupported = commonContext.getProvider().getAccountsSettings()
                .isPerAccountLastUpdateSupported();
        if (!accountKeySupported && !operationIdSupported) {
            return Optional.empty();
        }
        Optional<String> targetAccountKey = operation.getOperation().getRequestedChanges().getAccountCreateParams()
                .flatMap(OperationChangesModel.AccountCreateParams::getKey);
        return receivedAccounts.stream().filter(account -> {
            boolean match = false;
            if (operationIdSupported) {
                match = account.getLastUpdate()
                        .flatMap(u -> u.getOperationId().map(operationId::equals)).orElse(false);
            }
            if (accountKeySupported) {
                match = match || account.getKey().flatMap(k -> targetAccountKey.map(k::equals)).orElse(false);
            }
            return match;
        }).findFirst();
    }

    private Mono<Void> onRetryNoSuccessFinish(YdbTxSession session,
                                              RetryableOperation operation,
                                              CreateAccountOperationRetryContext retryContext,
                                              Instant now,
                                              Locale locale,
                                              OperationErrorKind errorKind) {
        if (!isOperationNotComplete(operation)) {
            return Mono.empty();
        }
        List<ReceivedAccount> receivedAccounts = retryContext.getPostRetryRefreshedAccounts().get();
        OperationCommonContext commonContext = retryContext.getCommonContext();
        String operationId = operation.getOperation().getOperationId();
        return validationService.validateReceivedAccounts(session, receivedAccounts, commonContext, locale)
                .flatMap(validatedR -> validatedR.match(validatedAccounts -> {
                            Optional<ValidatedReceivedAccount> matchingAccountO = getRefreshedAccount(
                                    validatedAccounts, operation, commonContext);
                            if (matchingAccountO.isEmpty()) {
                                return saveAbortOperation(session, operation, retryContext, now, errorKind);
                            }
                            return storeService.loadCreateAccountApplicationContext(session, operation.getOperation(),
                                    matchingAccountO.get().getAccountId()).flatMap(appCtx ->
                                            applyOperation(session, operation, retryContext, matchingAccountO.get(),
                                                    appCtx, now));
                        },
                        e -> {
                            LOG.error("Failed to process accounts refresh response after account creation response" +
                                            "for operation {}: {}", operationId, e);
                            return Mono.empty();
                        }));
    }

    private Optional<List<ReceivedAccount>> getPostRetryRefreshedAccounts(
            Result<Response<List<ReceivedAccount>>> result) {
        return result.match(
                resp -> resp.match(
                        (acc, requestId) -> Optional.of(acc),
                        e -> Optional.empty(),
                        (e, requestId) -> Optional.empty()),
                e -> Optional.empty());
    }

    private PostRetryRefreshResult getPostRetryRefreshResult(RetryableOperation operation,
                                                             Result<Response<List<ReceivedAccount>>> result) {
        String operationId = operation.getOperation().getOperationId();
        return result.match(resp -> resp.match(
                (acc, requestId) -> PostRetryRefreshResult.SUCCESS,
                e -> {
                    LOG.error("Failed to refresh folder accounts after account creation for operation "
                            + operationId, e);
                    return PostRetryRefreshResult.NON_FATAL_FAILURE;
                },
                (e, requestId) -> {
                    LOG.error("Failed to refresh folder accounts after account creation for operation " +
                                    "{} (requestId = {}): {}", operationId, requestId, e);
                    return e.match(new ProviderError.Cases<PostRetryRefreshResult>() {
                        @Override
                        public PostRetryRefreshResult httpError(int statusCode) {
                            if (isRefreshNotImplemented(statusCode)) {
                                return PostRetryRefreshResult.UNSUPPORTED;
                            }
                            if (isRefreshFatalError(statusCode)) {
                                return PostRetryRefreshResult.FATAL_FAILURE;
                            }
                            return PostRetryRefreshResult.NON_FATAL_FAILURE;
                        }

                        @Override
                        public PostRetryRefreshResult httpExtendedError(int statusCode, ErrorMessagesDto errors) {
                            if (isRefreshNotImplemented(statusCode)) {
                                return PostRetryRefreshResult.UNSUPPORTED;
                            }
                            if (isRefreshFatalError(statusCode)) {
                                return PostRetryRefreshResult.FATAL_FAILURE;
                            }
                            return PostRetryRefreshResult.NON_FATAL_FAILURE;
                        }

                        @Override
                        public PostRetryRefreshResult grpcError(Status.Code statusCode, String message) {
                            if (isRefreshNotImplemented(statusCode)) {
                                return PostRetryRefreshResult.UNSUPPORTED;
                            }
                            if (isRefreshFatalError(statusCode)) {
                                return PostRetryRefreshResult.FATAL_FAILURE;
                            }
                            return PostRetryRefreshResult.NON_FATAL_FAILURE;
                        }

                        @Override
                        public PostRetryRefreshResult grpcExtendedError(Status.Code statusCode, String message,
                                                                        Map<String, String> badRequestDetails) {
                            if (isRefreshNotImplemented(statusCode)) {
                                return PostRetryRefreshResult.UNSUPPORTED;
                            }
                            if (isRefreshFatalError(statusCode)) {
                                return PostRetryRefreshResult.FATAL_FAILURE;
                            }
                            return PostRetryRefreshResult.NON_FATAL_FAILURE;
                        }
                    });
                }),
                e -> {
                    LOG.error("Failed to refresh folder accounts after account creation for operation {}: {}",
                            operationId, e);
                    return PostRetryRefreshResult.NON_FATAL_FAILURE;
            });
    }

    private Optional<ReceivedAccount> getCreatedAccount(Result<Response<ReceivedAccount>> retryResult) {
        return retryResult.match(
                resp -> resp.match(
                        (acc, requestId) -> Optional.of(acc),
                        e -> Optional.empty(),
                        (e, requestId) -> Optional.empty()),
                e -> Optional.empty());
    }

    private RetryResult getRetryResult(RetryableOperation operation, Result<Response<ReceivedAccount>> retryResult) {
        String operationId = operation.getOperation().getOperationId();
        return retryResult.match(
                resp -> resp.match(
                        (acc, requestId) -> RetryResult.SUCCESS,
                        e -> {
                            LOG.error("Failed to retry account creation for operation " + operationId, e);
                            return RetryResult.NON_FATAL_FAILURE;
                        },
                        (e, requestId) -> {
                            LOG.error("Failed to retry account creation for operation {} (requestId = {}): {}",
                                    operationId, requestId, e);
                            return e.match(new ProviderError.Cases<RetryResult>() {
                                @Override
                                public RetryResult httpError(int statusCode) {
                                    if (isRetryConflict(statusCode)) {
                                        return RetryResult.CONFLICT;
                                    }
                                    if (isRetryFatalError(statusCode)) {
                                        return RetryResult.FATAL_FAILURE;
                                    }
                                    return RetryResult.NON_FATAL_FAILURE;
                                }

                                @Override
                                public RetryResult httpExtendedError(int statusCode, ErrorMessagesDto errors) {
                                    if (isRetryConflict(statusCode)) {
                                        return RetryResult.CONFLICT;
                                    }
                                    if (isRetryFatalError(statusCode)) {
                                        return RetryResult.FATAL_FAILURE;
                                    }
                                    return RetryResult.NON_FATAL_FAILURE;
                                }

                                @Override
                                public RetryResult grpcError(Status.Code statusCode, String message) {
                                    if (isRetryConflict(statusCode)) {
                                        return RetryResult.CONFLICT;
                                    }
                                    if (isRetryFatalError(statusCode)) {
                                        return RetryResult.FATAL_FAILURE;
                                    }
                                    return RetryResult.NON_FATAL_FAILURE;
                                }

                                @Override
                                public RetryResult grpcExtendedError(Status.Code statusCode, String message,
                                                                     Map<String, String> badRequestDetails) {
                                    if (isRetryConflict(statusCode)) {
                                        return RetryResult.CONFLICT;
                                    }
                                    if (isRetryFatalError(statusCode)) {
                                        return RetryResult.FATAL_FAILURE;
                                    }
                                    return RetryResult.NON_FATAL_FAILURE;
                                }
                            });
                        }),
                e -> {
                    LOG.error("Failed to retry account creation for operation {}: {}", operationId, e);
                    return RetryResult.NON_FATAL_FAILURE;
                });
    }

    private boolean isRetryConflict(Status.Code statusCode) {
        return statusCode == Status.Code.ALREADY_EXISTS
                || statusCode == Status.Code.FAILED_PRECONDITION;
    }

    private boolean isRetryConflict(int statusCode) {
        return statusCode == HttpStatus.CONFLICT.value()
                || statusCode == HttpStatus.PRECONDITION_FAILED.value();
    }

    private boolean isRetryFatalError(int statusCode) {
        return statusCode == HttpStatus.BAD_REQUEST.value()
                || statusCode == HttpStatus.UNPROCESSABLE_ENTITY.value()
                || statusCode == HttpStatus.NOT_FOUND.value()
                || statusCode == HttpStatus.NOT_IMPLEMENTED.value()
                || statusCode == HttpStatus.METHOD_NOT_ALLOWED.value();
    }

    private boolean isRetryFatalError(Status.Code statusCode) {
        return statusCode == Status.Code.DATA_LOSS
                || statusCode == Status.Code.INVALID_ARGUMENT
                || statusCode == Status.Code.NOT_FOUND
                || statusCode == Status.Code.OUT_OF_RANGE
                || statusCode == Status.Code.UNIMPLEMENTED
                || statusCode == Status.Code.UNKNOWN;
    }

    private Optional<ValidatedReceivedAccount> getRefreshedAccount(
            RetryableOperation operation,
            CreateAccountOperationPostRefreshContext postRefreshContext) {
        String operationId = operation.getOperation().getOperationId();
        boolean accountKeySupported = postRefreshContext.getCommonContext().getProvider()
                .getAccountsSettings().isKeySupported();
        boolean operationIdSupported = postRefreshContext.getCommonContext().getProvider().getAccountsSettings()
                .isPerAccountLastUpdateSupported();
        if (!accountKeySupported && !operationIdSupported) {
            return Optional.empty();
        }
        Optional<String> targetAccountKey = operation.getOperation().getRequestedChanges().getAccountCreateParams()
                .flatMap(OperationChangesModel.AccountCreateParams::getKey);
        return postRefreshContext.getPostRefreshResult().match(resp -> resp.match((acc, requestId) ->
                        acc.stream().filter(account -> {
                            if (operationIdSupported) {
                                return account.getLastUpdate()
                                        .flatMap(u -> u.getOperationId().map(operationId::equals)).orElse(false);
                            } else if (accountKeySupported) {
                                return account.getKey().flatMap(k -> targetAccountKey.map(k::equals)).orElse(false);
                            } else {
                                return false;
                            }
                        }).findFirst(),
                e -> Optional.empty(),
                (e, requestId) -> Optional.empty()),
                err -> Optional.empty());
    }

    private RefreshResult getRefreshResult(RetryableOperation operation,
                                           CreateAccountOperationPostRefreshContext postRefreshContext) {
        String operationId = operation.getOperation().getOperationId();
        boolean accountKeySupported = postRefreshContext.getCommonContext().getProvider()
                .getAccountsSettings().isKeySupported();
        boolean operationIdSupported = postRefreshContext.getCommonContext().getProvider().getAccountsSettings()
                .isPerAccountLastUpdateSupported();
        if (!accountKeySupported && !operationIdSupported) {
            LOG.warn("Can not check if account was created in operation {}, not supported", operationId);
            return RefreshResult.UNDEFINED;
        }
        Optional<String> targetAccountKey = operation.getOperation().getRequestedChanges().getAccountCreateParams()
                .flatMap(OperationChangesModel.AccountCreateParams::getKey);
        return postRefreshContext.getPostRefreshResult().match(resp -> resp.match((acc, requestId) -> {
            boolean accountFound = acc.stream().anyMatch(account -> {
                if (operationIdSupported) {
                    return account.getLastUpdate()
                            .flatMap(u -> u.getOperationId().map(operationId::equals)).orElse(false);
                } else if (accountKeySupported) {
                    return account.getKey().flatMap(k -> targetAccountKey.map(k::equals)).orElse(false);
                } else {
                    return false;
                }
            });
            if (accountFound) {
                return RefreshResult.OPERATION_APPLIED;
            } else {
                return RefreshResult.OPERATION_NOT_APPLIED;
            }
        },
        e -> {
            LOG.error("Failed to refresh folder accounts for operation " + operationId + " retry", e);
            return RefreshResult.NON_FATAL_ERROR;
        },
        (e, requestId) -> {
            LOG.error("Failed to refresh folder accounts for operation {} (requestId = {}) retry: {}",
                    operationId, requestId, e);
            return e.match(new ProviderError.Cases<RefreshResult>() {
                @Override
                public RefreshResult httpError(int statusCode) {
                    if (isRefreshNotImplemented(statusCode)) {
                        return RefreshResult.UNSUPPORTED;
                    }
                    if (isRefreshFatalError(statusCode)) {
                        return RefreshResult.FATAL_ERROR;
                    }
                    return RefreshResult.NON_FATAL_ERROR;
                }

                @Override
                public RefreshResult httpExtendedError(int statusCode, ErrorMessagesDto errors) {
                    if (isRefreshNotImplemented(statusCode)) {
                        return RefreshResult.UNSUPPORTED;
                    }
                    if (isRefreshFatalError(statusCode)) {
                        return RefreshResult.FATAL_ERROR;
                    }
                    return RefreshResult.NON_FATAL_ERROR;
                }

                @Override
                public RefreshResult grpcError(Status.Code statusCode, String message) {
                    if (isRefreshNotImplemented(statusCode)) {
                        return RefreshResult.UNSUPPORTED;
                    }
                    if (isRefreshFatalError(statusCode)) {
                        return RefreshResult.FATAL_ERROR;
                    }
                    return RefreshResult.NON_FATAL_ERROR;
                }

                @Override
                public RefreshResult grpcExtendedError(Status.Code statusCode, String message,
                                                       Map<String, String> badRequestDetails) {
                    if (isRefreshNotImplemented(statusCode)) {
                        return RefreshResult.UNSUPPORTED;
                    }
                    if (isRefreshFatalError(statusCode)) {
                        return RefreshResult.FATAL_ERROR;
                    }
                    return RefreshResult.NON_FATAL_ERROR;
                }
            });
        }), err -> {
            LOG.error("Failed to refresh folder accounts for operation {} retry: {}", operationId, err);
            return RefreshResult.NON_FATAL_ERROR;
        });
    }

    private boolean isRefreshNotImplemented(int statusCode) {
        return statusCode == HttpStatus.NOT_IMPLEMENTED.value()
                || statusCode == HttpStatus.METHOD_NOT_ALLOWED.value()
                || statusCode == HttpStatus.NOT_FOUND.value();
    }

    private boolean isRefreshNotImplemented(Status.Code statusCode) {
        return statusCode == Status.Code.UNIMPLEMENTED;
    }

    private boolean isRefreshFatalError(int statusCode) {
        return statusCode == HttpStatus.BAD_REQUEST.value()
                || statusCode == HttpStatus.UNPROCESSABLE_ENTITY.value();
    }

    private boolean isRefreshFatalError(Status.Code statusCode) {
        return statusCode == Status.Code.DATA_LOSS
                || statusCode == Status.Code.INVALID_ARGUMENT
                || statusCode == Status.Code.NOT_FOUND
                || statusCode == Status.Code.OUT_OF_RANGE
                || statusCode == Status.Code.UNKNOWN;
    }

    private boolean isNoRetryAfterRefresh(RefreshResult refreshResult, OperationCommonContext commonContext) {
        return isOperationAppliedAfterRefresh(refreshResult) ||
                isRetryUnsafeAfterRefresh(refreshResult, commonContext) ||
                isAnotherRoundAfterRefresh(refreshResult);
    }

    private boolean isOperationAppliedAfterRefresh(RefreshResult refreshResult) {
        return refreshResult == RefreshResult.OPERATION_APPLIED;
    }

    private boolean isRetryUnsafeAfterRefresh(RefreshResult refreshResult, OperationCommonContext commonContext) {
        boolean operationIdDeduplication = commonContext.getProvider()
                .getAccountsSettings().isOperationIdDeduplicationSupported();
        return (refreshResult == RefreshResult.FATAL_ERROR || refreshResult == RefreshResult.UNSUPPORTED
                || refreshResult == RefreshResult.UNDEFINED) && !operationIdDeduplication;
    }

    private Optional<String> getRefreshErrorDescription(CreateAccountOperationPostRefreshContext context,
                                                        RefreshResult refreshResult, Locale locale) {
        if (refreshResult == RefreshResult.OPERATION_APPLIED || refreshResult == RefreshResult.OPERATION_NOT_APPLIED) {
            return Optional.empty();
        }
        if (refreshResult == RefreshResult.UNDEFINED) {
            return Optional.of(messages.getMessage("errors.account.creation.check.is.not.possible", null, locale));
        }
        return context.getPostRefreshResult().match(
                response -> response.match(
                        (v, reqId) -> Optional.empty(),
                        e -> Optional.of(messages.getMessage("errors.unexpected.provider.communication.failure",
                                null, locale)),
                        (e, reqId) -> {
                            String prefix = refreshResult == RefreshResult.UNSUPPORTED
                                    ? messages.getMessage("errors.account.creation.check.is.not.possible",
                                            null, locale)
                                    : null;
                            return Optional.of(Errors.flattenProviderErrorResponse(e, prefix));
                        }),
                e -> Optional.of(Errors.flattenErrors(e)));
    }

    private Optional<String> getRetryErrorDescription(Result<Response<ReceivedAccount>> retryResponse,
                                                      RetryResult retryResult, Locale locale) {
        if (retryResult == RetryResult.SUCCESS) {
            return Optional.empty();
        }
        return retryResponse.match(
                response -> response.match(
                        (v, reqId) -> Optional.empty(),
                        e -> Optional.of(messages.getMessage("errors.unexpected.provider.communication.failure",
                                null, locale)),
                        (e, reqId) -> {
                            String prefix = retryResult == RetryResult.CONFLICT
                                    ? messages.getMessage("errors.accounts.quotas.out.of.sync.with.provider",
                                    null, locale)
                                    : null;
                            return Optional.of(Errors.flattenProviderErrorResponse(e, prefix));
                        }),
                e -> Optional.of(Errors.flattenErrors(e)));
    }

    private Optional<String> getPostRetryRefreshErrorDescription(
            Result<Response<List<ReceivedAccount>>> refreshResponse,
            PostRetryRefreshResult postRetryRefreshResult,
            Locale locale) {
        if (postRetryRefreshResult == PostRetryRefreshResult.SUCCESS) {
            return Optional.empty();
        }
        return refreshResponse.match(
                response -> response.match(
                        (v, reqId) -> Optional.empty(),
                        e -> Optional.of(messages.getMessage("errors.unexpected.provider.communication.failure",
                                null, locale)),
                        (e, reqId) -> {
                            String prefix = postRetryRefreshResult == PostRetryRefreshResult.UNSUPPORTED
                                    ? messages.getMessage("errors.account.creation.check.is.not.possible",
                                    null, locale)
                                    : null;
                            return Optional.of(Errors.flattenProviderErrorResponse(e, prefix));
                        }),
                e -> Optional.of(Errors.flattenErrors(e)));
    }

}
