package ru.yandex.intranet.d.services.operations;

import java.time.Instant;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.common.base.Objects;
import com.google.common.collect.Sets;
import io.grpc.Status;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.MessageSource;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.dao.Tenants;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.model.accounts.AccountModel;
import ru.yandex.intranet.d.model.accounts.AccountsQuotasModel;
import ru.yandex.intranet.d.model.accounts.AccountsQuotasOperationsModel;
import ru.yandex.intranet.d.model.accounts.OperationChangesModel;
import ru.yandex.intranet.d.model.accounts.OperationErrorKind;
import ru.yandex.intranet.d.model.accounts.OperationOrdersModel;
import ru.yandex.intranet.d.model.folders.AccountHistoryModel;
import ru.yandex.intranet.d.model.folders.AccountsHistoryModel;
import ru.yandex.intranet.d.model.folders.FolderModel;
import ru.yandex.intranet.d.model.folders.FolderOperationLogModel;
import ru.yandex.intranet.d.model.folders.FolderOperationType;
import ru.yandex.intranet.d.model.folders.OperationPhase;
import ru.yandex.intranet.d.model.folders.ProvisionHistoryModel;
import ru.yandex.intranet.d.model.folders.ProvisionsByResource;
import ru.yandex.intranet.d.model.folders.QuotasByAccount;
import ru.yandex.intranet.d.model.folders.QuotasByResource;
import ru.yandex.intranet.d.model.providers.AccountsSettingsModel;
import ru.yandex.intranet.d.model.quotas.QuotaModel;
import ru.yandex.intranet.d.services.integration.providers.ProviderError;
import ru.yandex.intranet.d.services.integration.providers.Response;
import ru.yandex.intranet.d.services.integration.providers.rest.model.ErrorMessagesDto;
import ru.yandex.intranet.d.services.operations.model.DeliverUpdateProvisionContext;
import ru.yandex.intranet.d.services.operations.model.DeliverUpdateProvisionOperationPostRefreshContext;
import ru.yandex.intranet.d.services.operations.model.DeliverUpdateProvisionOperationPreRefreshContext;
import ru.yandex.intranet.d.services.operations.model.DeliverUpdateProvisionOperationRefreshContext;
import ru.yandex.intranet.d.services.operations.model.DeliverUpdateProvisionOperationRetryContext;
import ru.yandex.intranet.d.services.operations.model.OperationCommonContext;
import ru.yandex.intranet.d.services.operations.model.OperationPostRefreshContext;
import ru.yandex.intranet.d.services.operations.model.OperationPreRefreshContext;
import ru.yandex.intranet.d.services.operations.model.OperationRefreshContext;
import ru.yandex.intranet.d.services.operations.model.OperationRetryContext;
import ru.yandex.intranet.d.services.operations.model.PostRetryRefreshResult;
import ru.yandex.intranet.d.services.operations.model.ReceivedAccount;
import ru.yandex.intranet.d.services.operations.model.ReceivedUpdatedProvision;
import ru.yandex.intranet.d.services.operations.model.RefreshResult;
import ru.yandex.intranet.d.services.operations.model.RetryResult;
import ru.yandex.intranet.d.services.operations.model.RetryableOperation;
import ru.yandex.intranet.d.services.operations.model.ValidatedReceivedAccount;
import ru.yandex.intranet.d.services.operations.model.ValidatedReceivedLastUpdate;
import ru.yandex.intranet.d.services.operations.model.ValidatedReceivedProvision;
import ru.yandex.intranet.d.services.operations.model.ValidatedReceivedUpdatedProvision;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.util.units.Units;
import ru.yandex.intranet.d.web.errors.Errors;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;

/**
 * Deliver and update provision operation retry service.
 *
 * @author Ruslan Kadriev <aqru@yandex-team.ru>
 */
@Component
public class DeliverUpdateProvisionOperationRetryService implements  BaseOperationRetryService {
    private static final Logger LOG = LoggerFactory.getLogger(DeliverUpdateProvisionOperationRetryService.class);

    private final OperationsRetryIntegrationService integrationService;
    private final OperationsRetryValidationService validationService;
    private final OperationsRetryStoreService storeService;
    private final OperationsObservabilityService operationsObservabilityService;
    private final MessageSource messages;
    private final long maxAsyncRetries;


    public DeliverUpdateProvisionOperationRetryService(
            OperationsRetryIntegrationService integrationService,
            OperationsRetryValidationService validationService,
            OperationsRetryStoreService storeService,
            OperationsObservabilityService operationsObservabilityService,
            @Qualifier("messageSource") MessageSource messages,
            @Value("${providers.client.maxAsyncRetries}") long maxAsyncRetries) {
        this.integrationService = integrationService;
        this.validationService = validationService;
        this.storeService = storeService;
        this.operationsObservabilityService = operationsObservabilityService;
        this.messages = messages;
        this.maxAsyncRetries = maxAsyncRetries;
    }

    @Override
    public Mono<Optional<DeliverUpdateProvisionOperationPreRefreshContext>> preRefreshOperation(
            YdbTxSession session,
            RetryableOperation operation,
            OperationCommonContext context,
            Instant now,
            Locale locale) {
        AccountsQuotasOperationsModel op = operation.getOperation();
        return storeService.loadDeliverUpdateProvisionContext(session, op).flatMap(deliverUpdateProvisionContext -> {
            if (isOperationAlreadyDone(deliverUpdateProvisionContext, operation)) {
                AccountsQuotasOperationsModel closedOperation = new AccountsQuotasOperationsModel.Builder(op)
                        .setUpdateDateTime(now)
                        .setRequestStatus(AccountsQuotasOperationsModel.RequestStatus.OK)
                        .setErrorKind(null)
                        .build();
                List<QuotaModel> unfreezedQuotas = unfreezeQuotas(op,
                        new ArrayList<>(deliverUpdateProvisionContext.getFolderQuotasByResourceId().values()));
                operationsObservabilityService.observeOperationFinished(closedOperation);
                return storeService.upsertQuotas(session, unfreezedQuotas)
                        .then(Mono.defer(() -> storeService.upsertOperationAndReturn(session, closedOperation)))
                        .then(Mono.defer(() -> storeService.deleteInProgressByOperationId(session,
                                op.getOperationId()))).thenReturn(Optional.empty());
            }
            return Mono.just(Optional.of(new DeliverUpdateProvisionOperationPreRefreshContext(context,
                    deliverUpdateProvisionContext)));
        });
    }

    @Override
    public Mono<Optional<DeliverUpdateProvisionOperationRefreshContext>> refreshOperation(
            RetryableOperation operation,
            OperationPreRefreshContext context,
            Instant now,
            Locale locale) {
        DeliverUpdateProvisionOperationPreRefreshContext preRefreshContext
                = (DeliverUpdateProvisionOperationPreRefreshContext) context;
        DeliverUpdateProvisionContext provisionContext = preRefreshContext.getDeliverUpdateProvisionContext();
        return integrationService.getAccountById(provisionContext.getFolder(), provisionContext.getAccount(),
                preRefreshContext.getCommonContext(), locale).map(r ->
                Optional.of(new DeliverUpdateProvisionOperationRefreshContext(preRefreshContext, r)));
    }

    @Override
    public Mono<Optional<DeliverUpdateProvisionOperationPostRefreshContext>> postRefreshOperation(
            YdbTxSession session,
            RetryableOperation operation,
            OperationRefreshContext context,
            Instant now,
            Locale locale) {
        DeliverUpdateProvisionOperationRefreshContext refreshContext =
                (DeliverUpdateProvisionOperationRefreshContext) context;
        return refreshContext.getRefreshResult().andThenMono(resp -> resp.match(
                        (acc, requestId) -> validationService.validateReceivedAccount(session, acc,
                                context.getCommonContext(), locale).map(r -> r.apply(a -> Response.success(a,
                                requestId))),
                        e -> Mono.just(Result.success(Response.<ValidatedReceivedAccount>failure(e))),
                        (e, requestId) -> Mono.just(Result.success(
                                Response.<ValidatedReceivedAccount>error(e, requestId)))))
                .map(v -> Optional.of(new DeliverUpdateProvisionOperationPostRefreshContext(refreshContext, v)));
    }

    @Override
    public Mono<Optional<DeliverUpdateProvisionOperationRetryContext>> retryOperation(
            RetryableOperation operation,
            OperationPostRefreshContext context,
            Instant now,
            Locale locale) {
        DeliverUpdateProvisionOperationPostRefreshContext postRefreshContext
                = (DeliverUpdateProvisionOperationPostRefreshContext) context;
        RefreshResult refreshResult = getRefreshResult(operation, postRefreshContext);
        if (isAnotherRoundAfterRefresh(refreshResult)) {
            return Mono.just(Optional.empty());
        }
        Optional<ValidatedReceivedAccount> alreadyUpdatedAccount = refreshResult == RefreshResult.OPERATION_APPLIED
                ? getRefreshedAccount(postRefreshContext) : Optional.empty();
        Optional<String> refreshError = getRefreshErrorDescription(postRefreshContext, refreshResult, locale);
        if (isOperationAppliedAfterRefresh(refreshResult)) {
            return Mono.just(Optional.of(new DeliverUpdateProvisionOperationRetryContext(postRefreshContext,
                    refreshResult, alreadyUpdatedAccount.orElse(null), refreshError.orElse(null), null,
                    null, null, null, null, null,
                    UUID.randomUUID().toString())));
        }
        DeliverUpdateProvisionContext deliverUpdateProvisionContext = postRefreshContext.getPreRefreshContext()
                .getDeliverUpdateProvisionContext();
        OperationCommonContext commonContext = postRefreshContext.getCommonContext();
        return integrationService.deliverUpdateProvision(deliverUpdateProvisionContext, commonContext,
                operation.getOperation(), locale).flatMap(updateResult -> {
            RetryResult retryResult = getRetryResult(operation, updateResult);
            Optional<ReceivedUpdatedProvision> updatedAccount = getUpdatedAccount(updateResult);
            Optional<String> retryError = getRetryErrorDescription(updateResult, retryResult, locale);
            if (retryResult != RetryResult.SUCCESS) {
                return integrationService.getAccountById(
                        deliverUpdateProvisionContext.getFolder(), deliverUpdateProvisionContext.getAccount(),
                        commonContext, locale).map(postRetryRefresh -> {
                    PostRetryRefreshResult postRetryRefreshResult
                            = getPostRetryRefreshResult(operation, postRetryRefresh);
                    Optional<ReceivedAccount> postRetryRefreshedAccounts
                            = getPostRetryRefreshedAccount(postRetryRefresh);
                    Optional<String> postRetryRefreshError = getPostRetryRefreshErrorDescription(
                            postRetryRefresh, postRetryRefreshResult, locale);
                    return Optional.of(new DeliverUpdateProvisionOperationRetryContext(postRefreshContext,
                            refreshResult, alreadyUpdatedAccount.orElse(null),
                            refreshError.orElse(null), retryResult,
                            updatedAccount.orElse(null), retryError.orElse(null),
                            postRetryRefreshResult, postRetryRefreshedAccounts.orElse(null),
                            postRetryRefreshError.orElse(null), UUID.randomUUID().toString()));
                });
            }
            return Mono.just(Optional.of(new DeliverUpdateProvisionOperationRetryContext(postRefreshContext,
                    refreshResult, alreadyUpdatedAccount.orElse(null), refreshError.orElse(null),
                    retryResult, updatedAccount.orElse(null), retryError.orElse(null),
                    null, null, null,
                    UUID.randomUUID().toString())));
        });
    }

    @Override
    public Mono<Void> postRetryOperation(YdbTxSession session,
                                         RetryableOperation operation,
                                         OperationRetryContext context,
                                         Instant now,
                                         Locale locale) {
        DeliverUpdateProvisionOperationRetryContext retryContext =
                (DeliverUpdateProvisionOperationRetryContext) context;
        if (isOperationAppliedAfterRefresh(retryContext.getRefreshResult())
                && retryContext.getAlreadyUpdatedAccount().isPresent()) {
            return completeOnRefreshAlreadyApplied(session, operation, retryContext, now);
        } else if (retryContext.getRetryResult().orElseThrow() == RetryResult.FATAL_FAILURE) {
            if (retryContext.getPostRetryRefreshResult().orElseThrow() == PostRetryRefreshResult.FATAL_FAILURE) {
                return abortOperation(session, operation, retryContext, now, OperationErrorKind.INVALID_ARGUMENT);
            } else if (retryContext.getPostRetryRefreshResult().get() == PostRetryRefreshResult.NON_FATAL_FAILURE) {
                return planToNextRetryOperation(session, operation, retryContext, now,
                        OperationErrorKind.INVALID_ARGUMENT);
            } else if (retryContext.getPostRetryRefreshResult().get() == PostRetryRefreshResult.UNSUPPORTED) {
                return abortOperation(session, operation, retryContext, now, OperationErrorKind.INVALID_ARGUMENT);
            } else if (retryContext.getPostRetryRefreshResult().get() == PostRetryRefreshResult.SUCCESS) {
                return onRetryNoSuccessFinish(session, operation, retryContext, now, locale,
                        OperationErrorKind.INVALID_ARGUMENT);
            } else {
                return Mono.error(new IllegalArgumentException("Unexpected operation retry state for operation "
                        + operation + ": " + context));
            }
        } else if (retryContext.getRetryResult().get() == RetryResult.NON_FATAL_FAILURE) {
            return planToNextRetryOperation(session, operation, retryContext, now, OperationErrorKind.EXPIRED);
        } else if (retryContext.getRetryResult().get() == RetryResult.SUCCESS) {
            return onRetrySuccess(session, operation, retryContext, now, locale);
        } else if (retryContext.getRetryResult().get() == RetryResult.CONFLICT) {
            if (retryContext.getPostRetryRefreshResult().orElseThrow() == PostRetryRefreshResult.FATAL_FAILURE) {
                return abortOperation(session, operation, retryContext, now, OperationErrorKind.FAILED_PRECONDITION);
            } else if (retryContext.getPostRetryRefreshResult().get() == PostRetryRefreshResult.NON_FATAL_FAILURE) {
                return planToNextRetryOperation(session, operation, retryContext, now,
                        OperationErrorKind.FAILED_PRECONDITION);
            } else if (retryContext.getPostRetryRefreshResult().get() == PostRetryRefreshResult.UNSUPPORTED) {
                return abortOperation(session, operation, retryContext, now, OperationErrorKind.FAILED_PRECONDITION);
            } else if (retryContext.getPostRetryRefreshResult().get() == PostRetryRefreshResult.SUCCESS) {
                return onRetryNoSuccessFinish(session, operation, retryContext, now, locale,
                        OperationErrorKind.FAILED_PRECONDITION);
            } else {
                return Mono.error(new IllegalArgumentException("Unexpected operation retry state for operation "
                        + operation + ": " + context));
            }
        } else {
            return Mono.error(new IllegalArgumentException("Unexpected operation retry state for operation "
                    + operation + ": " + context));
        }
    }

    @Override
    public Mono<Void> abortOperation(YdbTxSession session, AccountsQuotasOperationsModel operation, String comment,
                                     Instant now, YaUserDetails currentUser, Locale locale) {
        AccountsQuotasOperationsModel updatedOperation = new AccountsQuotasOperationsModel
                .Builder(operation)
                .setRequestStatus(AccountsQuotasOperationsModel.RequestStatus.ERROR)
                .setUpdateDateTime(now)
                .setErrorMessage(comment)
                .setErrorKind(OperationErrorKind.ABORTED)
                .build();
        return storeService.loadDeliverUpdateProvisionQuotas(session, operation).flatMap(quotas -> {
            List<QuotaModel> unfreezedQuotas = unfreezeQuotas(operation, quotas);
            operationsObservabilityService.observeOperationFinished(updatedOperation);
            return storeService.upsertQuotas(session, unfreezedQuotas)
                    .then(Mono.defer(() -> storeService.upsertOperationAndReturn(session, updatedOperation)))
                    .then(Mono.defer(() -> storeService.deleteInProgressByOperationId(session,
                            operation.getOperationId())));
        });
    }

    private boolean isOperationAlreadyDone(DeliverUpdateProvisionContext deliverUpdateProvisionContext,
                                           RetryableOperation operation) {
        String accountId = operation.getOperation().getRequestedChanges().getAccountId().orElseThrow();
        String operationId = operation.getOperation().getOperationId();
        return deliverUpdateProvisionContext.getProvisionsByAccountIdResourceId().getOrDefault(accountId, Map.of())
                .values().stream().anyMatch(provision -> provision.getLatestSuccessfulProvisionOperationId()
                        .map(operationId::equals).orElse(false));
    }

    private List<QuotaModel> unfreezeQuotas(AccountsQuotasOperationsModel operation, List<QuotaModel> currentQuotas) {
        List<QuotaModel> unfreezedQuotas = new ArrayList<>();
        List<OperationChangesModel.Provision> frozenProvisions = operation.getRequestedChanges()
                .getFrozenProvisions().orElse(List.of());
        Map<String, OperationChangesModel.Provision> frozenByResource = frozenProvisions.stream()
                .collect(Collectors.toMap(OperationChangesModel.Provision::getResourceId, Function.identity()));
        currentQuotas.forEach(quota -> {
            if (!frozenByResource.containsKey(quota.getResourceId())) {
                return;
            }
            OperationChangesModel.Provision frozen = frozenByResource.get(quota.getResourceId());
            if (frozen.getAmount() == 0L) {
                return;
            }
            if (frozen.getAmount() <= quota.getFrozenQuota()) {
                if (Units.add(quota.getBalance(), frozen.getAmount()).isEmpty()) {
                    LOG.error("Balance overflow ({} frozen + {} balance) while unfreezing "
                                    + "quota for resource {} in folder {}, operation {}", frozen.getAmount(),
                            quota.getBalance(), quota.getResourceId(), quota.getFolderId(), operation.getOperationId());
                    return;
                }
                QuotaModel unfreezedQuota = QuotaModel.builder(quota)
                        .frozenQuota(quota.getFrozenQuota() - frozen.getAmount())
                        .balance(quota.getBalance() + frozen.getAmount())
                        .build();
                unfreezedQuotas.add(unfreezedQuota);
            } else {
                LOG.error("Frozen quota {} for resource {} in folder {} is less then value {} stored for operation {}",
                        quota.getFrozenQuota(), quota.getResourceId(), quota.getFolderId(), frozen.getAmount(),
                        operation.getOperationId());
            }
        });
        return unfreezedQuotas;
    }

    private RefreshResult getRefreshResult(RetryableOperation operation,
                                           DeliverUpdateProvisionOperationPostRefreshContext postRefreshContext) {
        String operationId = operation.getOperation().getOperationId();
        String accountId = operation.getOperation().getRequestedChanges().getAccountId().orElseThrow();
        return postRefreshContext.getPostRefreshResult().match(resp -> resp.match((acc, requestId) -> {
                            boolean provisionChangeApplied = isProvisionChangeApplied(acc, operation.getOperation(),
                                    postRefreshContext.getPreRefreshContext().getDeliverUpdateProvisionContext(),
                                    postRefreshContext.getCommonContext());
                            if (provisionChangeApplied) {
                                return RefreshResult.OPERATION_APPLIED;
                            } else {
                                return RefreshResult.OPERATION_NOT_APPLIED;
                            }
                        },
                        e -> {
                            LOG.error("Failed to refresh folder account " + accountId + " for operation "
                                    + operationId + " retry", e);
                            return RefreshResult.NON_FATAL_ERROR;
                        },
                        (e, requestId) -> {
                            LOG.error("Failed to refresh folder account {} for operation {} (requestId = {}) retry: {}",
                                    accountId, operationId, requestId, e);
                            return e.match(new ProviderError.Cases<>() {
                                @Override
                                public RefreshResult httpError(int statusCode) {
                                    if (isRefreshNotImplemented(statusCode)) {
                                        return RefreshResult.UNSUPPORTED;
                                    }
                                    if (isRefreshFatalError(statusCode)) {
                                        return RefreshResult.FATAL_ERROR;
                                    }
                                    return RefreshResult.NON_FATAL_ERROR;
                                }

                                @Override
                                public RefreshResult httpExtendedError(int statusCode, ErrorMessagesDto errors) {
                                    if (isRefreshNotImplemented(statusCode)) {
                                        return RefreshResult.UNSUPPORTED;
                                    }
                                    if (isRefreshFatalError(statusCode)) {
                                        return RefreshResult.FATAL_ERROR;
                                    }
                                    return RefreshResult.NON_FATAL_ERROR;
                                }

                                @Override
                                public RefreshResult grpcError(Status.Code statusCode, String message) {
                                    if (isRefreshNotImplemented(statusCode)) {
                                        return RefreshResult.UNSUPPORTED;
                                    }
                                    if (isRefreshFatalError(statusCode)) {
                                        return RefreshResult.FATAL_ERROR;
                                    }
                                    return RefreshResult.NON_FATAL_ERROR;
                                }

                                @Override
                                public RefreshResult grpcExtendedError(Status.Code statusCode, String message,
                                                                       Map<String, String> badRequestDetails) {
                                    if (isRefreshNotImplemented(statusCode)) {
                                        return RefreshResult.UNSUPPORTED;
                                    }
                                    if (isRefreshFatalError(statusCode)) {
                                        return RefreshResult.FATAL_ERROR;
                                    }
                                    return RefreshResult.NON_FATAL_ERROR;
                                }
                            });
                        }),
                err -> {
                    LOG.error("Failed to refresh folder account {} for operation {} retry: {}",
                            accountId, operationId, err);
                    return RefreshResult.NON_FATAL_ERROR;
                });
    }

    private boolean isAnotherRoundAfterRefresh(RefreshResult refreshResult) {
        return refreshResult == RefreshResult.NON_FATAL_ERROR;
    }

    private boolean isRefreshNotImplemented(int statusCode) {
        return statusCode == HttpStatus.NOT_IMPLEMENTED.value()
                || statusCode == HttpStatus.METHOD_NOT_ALLOWED.value();
    }

    private boolean isRefreshNotImplemented(Status.Code statusCode) {
        return statusCode == Status.Code.UNIMPLEMENTED;
    }

    private boolean isRefreshFatalError(int statusCode) {
        return statusCode == HttpStatus.BAD_REQUEST.value()
                || statusCode == HttpStatus.UNPROCESSABLE_ENTITY.value()
                || statusCode == HttpStatus.NOT_FOUND.value();
    }

    private boolean isRefreshFatalError(Status.Code statusCode) {
        return statusCode == Status.Code.DATA_LOSS
                || statusCode == Status.Code.INVALID_ARGUMENT
                || statusCode == Status.Code.NOT_FOUND
                || statusCode == Status.Code.OUT_OF_RANGE
                || statusCode == Status.Code.UNKNOWN;
    }

    private boolean isProvisionChangeApplied(ValidatedReceivedAccount receivedAccount,
                                             AccountsQuotasOperationsModel operation,
                                             DeliverUpdateProvisionContext deliverUpdateProvisionContext,
                                             OperationCommonContext commonContext) {
        AccountsSettingsModel accountsSettings = commonContext.getProvider()
                .getAccountsSettings();
        boolean provisionOperationIdSupported = accountsSettings.isPerProvisionLastUpdateSupported();
        boolean accountAndProvisionsVersionedTogether = accountsSettings.isPerAccountVersionSupported()
                && !accountsSettings.isPerProvisionVersionSupported();
        boolean provisionsVersionedSeparately = accountsSettings.isPerProvisionVersionSupported();
        boolean hasProvisionOperationIds = receivedAccount.getProvisions().stream()
                .allMatch(p -> p.getLastUpdate().isPresent() && p.getLastUpdate().get().getOperationId().isPresent());
        boolean hasCommonVersion = receivedAccount.getAccountVersion().isPresent();
        boolean hasProvisionVersions = receivedAccount.getProvisions().stream()
                .allMatch(p -> p.getQuotaVersion().isPresent());
        if (provisionOperationIdSupported && hasProvisionOperationIds) {
            return hasOperationId(receivedAccount, operation, deliverUpdateProvisionContext,
                    accountAndProvisionsVersionedTogether, provisionsVersionedSeparately);
        }
        if (accountAndProvisionsVersionedTogether && hasCommonVersion) {
            return isExactValuesMatchAndFreshCommonVersion(receivedAccount, operation,
                    deliverUpdateProvisionContext);
        }
        if (provisionsVersionedSeparately && hasProvisionVersions) {
            return isExactValuesMatchAndFreshProvisionVersions(receivedAccount, operation,
                    deliverUpdateProvisionContext);
        }
        return isExactValuesMatch(receivedAccount, operation, deliverUpdateProvisionContext);
    }

    private boolean hasOperationId(ValidatedReceivedAccount receivedAccount,
                                   AccountsQuotasOperationsModel operation,
                                   DeliverUpdateProvisionContext deliverUpdateProvisionContext,
                                   boolean accountAndProvisionsVersionedTogether,
                                   boolean provisionsVersionedSeparately) {
        if (accountAndProvisionsVersionedTogether) {
            Optional<Long> knownVersionO = deliverUpdateProvisionContext.getAccount().getLastReceivedVersion();
            Optional<Long> receivedVersionO = receivedAccount.getAccountVersion();
            if (knownVersionO.isPresent() && receivedVersionO.isPresent()) {
                if (knownVersionO.get() >= receivedVersionO.get()) {
                    return false;
                }
            }
            if (isMismatchForRemainingResources(receivedAccount, operation, deliverUpdateProvisionContext)) {
                return false;
            }
        }
        Map<String, OperationChangesModel.Provision> targetProvisionByResourceId = operation.getRequestedChanges()
                .getUpdatedProvisions().orElse(List.of()).stream()
                .collect(Collectors.toMap(OperationChangesModel.Provision::getResourceId, Function.identity()));
        Set<String> targetResourceIds = targetProvisionByResourceId.keySet();
        Map<String, ValidatedReceivedProvision> receivedProvisionsByResourceId = receivedAccount.getProvisions()
                .stream().collect(Collectors.toMap(v -> v.getResource().getId(), Function.identity()));
        if (provisionsVersionedSeparately) {
            boolean versionMatch = targetResourceIds.stream().allMatch(resourceId -> {
                AccountsQuotasModel knownProvision = deliverUpdateProvisionContext.getProvisionsByAccountIdResourceId()
                        .getOrDefault(deliverUpdateProvisionContext.getAccount().getId(), Map.of())
                        .get(resourceId);
                ValidatedReceivedProvision receivedProvision = receivedProvisionsByResourceId.get(resourceId);
                Optional<Long> knownVersionO = knownProvision != null
                        ? knownProvision.getLastReceivedProvisionVersion() : Optional.empty();
                Optional<Long> receivedVersionO = receivedProvision != null
                        ? receivedProvision.getQuotaVersion() : Optional.empty();
                if (knownVersionO.isPresent() && receivedVersionO.isPresent()) {
                    return knownVersionO.get() < receivedVersionO.get();
                }
                return true;
            });
            if (!versionMatch) {
                return false;
            }
        }
        String knownFolderId = deliverUpdateProvisionContext.getFolder().getId();
        String receivedFolderId = receivedAccount.getFolder().getId();
        boolean foldersMatch = knownFolderId.equals(receivedFolderId);
        boolean knownDeletion = deliverUpdateProvisionContext.getAccount().isDeleted();
        boolean receivedDeletion = receivedAccount.isDeleted();
        boolean deletionMatch = knownDeletion == receivedDeletion;
        String targetOperationId = operation.getOperationId();
        boolean opIdMatch = targetResourceIds.stream().allMatch(resourceId -> {
            ValidatedReceivedProvision receivedProvision = receivedProvisionsByResourceId.get(resourceId);
            OperationChangesModel.Provision targetProvision = targetProvisionByResourceId.get(resourceId);
            if (receivedProvision == null) {
                return targetProvision.getAmount() == 0L;
            }
            return receivedProvision.getLastUpdate().flatMap(ValidatedReceivedLastUpdate::getOperationId)
                    .map(targetOperationId::equals).orElse(false);
        });
        return opIdMatch && foldersMatch && deletionMatch;
    }

    private boolean isMismatchForRemainingResources(ValidatedReceivedAccount receivedAccount,
                                                    AccountsQuotasOperationsModel operation,
                                                    DeliverUpdateProvisionContext deliverUpdateProvisionContext) {
        Map<String, ValidatedReceivedProvision> receivedProvisionsByResourceId = receivedAccount.getProvisions()
                .stream().collect(Collectors.toMap(v -> v.getResource().getId(), Function.identity()));
        Set<String> targetResourceIds = operation.getRequestedChanges().getUpdatedProvisions()
                .orElse(List.of()).stream().map(OperationChangesModel.Provision::getResourceId)
                .collect(Collectors.toSet());
        Set<String> receivedResourceIds = receivedProvisionsByResourceId.keySet();
        Set<String> knowResourceIds = deliverUpdateProvisionContext.getProvisionsByAccountIdResourceId()
                .getOrDefault(deliverUpdateProvisionContext.getAccount().getId(), Map.of()).keySet();
        Set<String> remainingResourceIds = Sets.difference(Sets.union(knowResourceIds, receivedResourceIds),
                targetResourceIds);
        return remainingResourceIds.stream().anyMatch(resourceId -> {
            ValidatedReceivedProvision receivedProvision = receivedProvisionsByResourceId.get(resourceId);
            AccountsQuotasModel knownProvision = deliverUpdateProvisionContext.getProvisionsByAccountIdResourceId()
                    .getOrDefault(deliverUpdateProvisionContext.getAccount().getId(), Map.of()).get(resourceId);
            long receivedAmount = receivedProvision != null ? receivedProvision.getProvidedAmount() : 0L;
            long knownAmount = knownProvision != null ? (knownProvision.getProvidedQuota() != null
                    ? knownProvision.getProvidedQuota() : 0L) : 0L;
            return receivedAmount != knownAmount;
        });
    }

    private boolean isExactValuesMatchAndFreshCommonVersion(
            ValidatedReceivedAccount receivedAccount,
            AccountsQuotasOperationsModel operation,
            DeliverUpdateProvisionContext deliverUpdateProvisionContext) {
        Optional<Long> receivedVersionO = receivedAccount.getAccountVersion();
        Optional<Long> knownVersionO = deliverUpdateProvisionContext.getAccount().getLastReceivedVersion();
        if (receivedVersionO.isPresent() && knownVersionO.isPresent()
                && receivedVersionO.get() <= knownVersionO.get()) {
            return false;
        }
        if (isMismatchForRemainingResources(receivedAccount, operation, deliverUpdateProvisionContext)) {
            return false;
        }
        return isExactValuesMatch(receivedAccount, operation, deliverUpdateProvisionContext);
    }

    private boolean isExactValuesMatch(ValidatedReceivedAccount receivedAccount,
                                       AccountsQuotasOperationsModel operation,
                                       DeliverUpdateProvisionContext deliverUpdateProvisionContext) {
        List<OperationChangesModel.Provision> targetProvisions = operation.getRequestedChanges()
                .getUpdatedProvisions().orElse(List.of());
        Map<String, ValidatedReceivedProvision> receivedProvisionsByResourceId = receivedAccount.getProvisions()
                .stream().collect(Collectors.toMap(v -> v.getResource().getId(), Function.identity()));
        boolean valuesMatch = targetProvisions.stream().allMatch(targetProvision -> {
            String targetResourceId = targetProvision.getResourceId();
            long targetAmount = targetProvision.getAmount();
            ValidatedReceivedProvision receivedProvision = receivedProvisionsByResourceId.get(targetResourceId);
            long receivedAmount = receivedProvision != null ? receivedProvision.getProvidedAmount() : 0L;
            return targetAmount == receivedAmount;
        });
        String knownFolderId = deliverUpdateProvisionContext.getFolder().getId();
        String receivedFolderId = receivedAccount.getFolder().getId();
        boolean foldersMatch = knownFolderId.equals(receivedFolderId);
        boolean knownDeletion = deliverUpdateProvisionContext.getAccount().isDeleted();
        boolean receivedDeletion = receivedAccount.isDeleted();
        boolean deletionMatch = knownDeletion == receivedDeletion;
        return valuesMatch && foldersMatch && deletionMatch;
    }

    private boolean isExactValuesMatchAndFreshProvisionVersions(
            ValidatedReceivedAccount receivedAccount,
            AccountsQuotasOperationsModel operation,
            DeliverUpdateProvisionContext deliverUpdateProvisionContext) {
        String accountId = deliverUpdateProvisionContext.getAccount().getId();
        Map<String, AccountsQuotasModel> knownProvisions = deliverUpdateProvisionContext
                .getProvisionsByAccountIdResourceId().getOrDefault(accountId, Map.of());
        boolean receivedVersionsAreFresh = receivedAccount.getProvisions().stream().allMatch(receivedProvision -> {
            Optional<Long> receivedVersionO = receivedProvision.getQuotaVersion();
            AccountsQuotasModel knownProvision = knownProvisions.get(receivedProvision.getResource().getId());
            Optional<Long> knownVersionO = knownProvision != null
                    ? knownProvision.getLastReceivedProvisionVersion() : Optional.empty();
            return receivedVersionO.isEmpty() || knownVersionO.isEmpty()
                    || receivedVersionO.get() > knownVersionO.get();
        });
        if (!receivedVersionsAreFresh) {
            return false;
        }
        return isExactValuesMatch(receivedAccount, operation, deliverUpdateProvisionContext);
    }

    private Optional<ValidatedReceivedAccount> getRefreshedAccount(
            DeliverUpdateProvisionOperationPostRefreshContext postRefreshContext) {
        return postRefreshContext.getPostRefreshResult().match(resp -> resp.match(
                        (acc, requestId) -> Optional.of(acc),
                        e -> Optional.empty(),
                        (e, requestId) -> Optional.empty()),
                err -> Optional.empty());
    }

    private Optional<String> getRefreshErrorDescription(DeliverUpdateProvisionOperationPostRefreshContext context,
                                                        RefreshResult refreshResult, Locale locale) {
        if (refreshResult == RefreshResult.OPERATION_APPLIED || refreshResult == RefreshResult.OPERATION_NOT_APPLIED) {
            return Optional.empty();
        }
        if (refreshResult == RefreshResult.UNDEFINED) {
            return Optional.of(messages.getMessage("errors.provision.update.check.is.not.possible", null, locale));
        }
        return context.getPostRefreshResult().match(
                response -> response.match(
                        (v, reqId) -> Optional.empty(),
                        e -> Optional.of(messages.getMessage("errors.unexpected.provider.communication.failure",
                                null, locale)),
                        (e, reqId) -> {
                            String prefix = refreshResult == RefreshResult.UNSUPPORTED
                                    ? messages.getMessage("errors.provision.update.check.is.not.possible",
                                    null, locale)
                                    : null;
                            return Optional.of(Errors.flattenProviderErrorResponse(e, prefix));
                        }),
                e -> Optional.of(Errors.flattenErrors(e)));
    }

    private boolean isOperationAppliedAfterRefresh(RefreshResult refreshResult) {
        return refreshResult == RefreshResult.OPERATION_APPLIED;
    }

    private RetryResult getRetryResult(RetryableOperation operation,
                                       Result<Response<ReceivedUpdatedProvision>> retryResult) {
        String operationId = operation.getOperation().getOperationId();
        return retryResult.match(
                resp -> resp.match(
                        (acc, requestId) -> RetryResult.SUCCESS,
                        e -> {
                            LOG.error("Failed to retry provision update for operation " + operationId, e);
                            return RetryResult.NON_FATAL_FAILURE;
                        },
                        (e, requestId) -> {
                            LOG.error("Failed to retry provision update for operation {} (requestId = {}): {}",
                                    operationId, requestId, e);
                            return e.match(new ProviderError.Cases<>() {
                                @Override
                                public RetryResult httpError(int statusCode) {
                                    if (isRetryConflict(statusCode)) {
                                        return RetryResult.CONFLICT;
                                    }
                                    if (isRetryFatalError(statusCode)) {
                                        return RetryResult.FATAL_FAILURE;
                                    }
                                    return RetryResult.NON_FATAL_FAILURE;
                                }

                                @Override
                                public RetryResult httpExtendedError(int statusCode, ErrorMessagesDto errors) {
                                    if (isRetryConflict(statusCode)) {
                                        return RetryResult.CONFLICT;
                                    }
                                    if (isRetryFatalError(statusCode)) {
                                        return RetryResult.FATAL_FAILURE;
                                    }
                                    return RetryResult.NON_FATAL_FAILURE;
                                }

                                @Override
                                public RetryResult grpcError(Status.Code statusCode, String message) {
                                    if (isRetryConflict(statusCode)) {
                                        return RetryResult.CONFLICT;
                                    }
                                    if (isRetryFatalError(statusCode)) {
                                        return RetryResult.FATAL_FAILURE;
                                    }
                                    return RetryResult.NON_FATAL_FAILURE;
                                }

                                @Override
                                public RetryResult grpcExtendedError(Status.Code statusCode, String message,
                                                                     Map<String, String> badRequestDetails) {
                                    if (isRetryConflict(statusCode)) {
                                        return RetryResult.CONFLICT;
                                    }
                                    if (isRetryFatalError(statusCode)) {
                                        return RetryResult.FATAL_FAILURE;
                                    }
                                    return RetryResult.NON_FATAL_FAILURE;
                                }
                            });
                        }),
                e -> {
                    LOG.error("Failed to retry provision update for operation {}: {}", operationId, e);
                    return RetryResult.NON_FATAL_FAILURE;
                });
    }

    private boolean isRetryConflict(Status.Code statusCode) {
        return statusCode == Status.Code.ALREADY_EXISTS
                || statusCode == Status.Code.FAILED_PRECONDITION;
    }

    private boolean isRetryConflict(int statusCode) {
        return statusCode == HttpStatus.CONFLICT.value()
                || statusCode == HttpStatus.PRECONDITION_FAILED.value();
    }

    private boolean isRetryFatalError(int statusCode) {
        return statusCode == HttpStatus.BAD_REQUEST.value()
                || statusCode == HttpStatus.UNPROCESSABLE_ENTITY.value()
                || statusCode == HttpStatus.NOT_FOUND.value()
                || statusCode == HttpStatus.NOT_IMPLEMENTED.value()
                || statusCode == HttpStatus.METHOD_NOT_ALLOWED.value();
    }

    private boolean isRetryFatalError(Status.Code statusCode) {
        return statusCode == Status.Code.DATA_LOSS
                || statusCode == Status.Code.INVALID_ARGUMENT
                || statusCode == Status.Code.NOT_FOUND
                || statusCode == Status.Code.OUT_OF_RANGE
                || statusCode == Status.Code.UNIMPLEMENTED
                || statusCode == Status.Code.UNKNOWN;
    }

    private Optional<ReceivedUpdatedProvision> getUpdatedAccount(
            Result<Response<ReceivedUpdatedProvision>> retryResult) {
        return retryResult.match(
                resp -> resp.match(
                        (acc, requestId) -> Optional.of(acc),
                        e -> Optional.empty(),
                        (e, requestId) -> Optional.empty()),
                e -> Optional.empty());
    }

    private Optional<String> getRetryErrorDescription(Result<Response<ReceivedUpdatedProvision>> retryResponse,
                                                      RetryResult retryResult, Locale locale) {
        if (retryResult == RetryResult.SUCCESS) {
            return Optional.empty();
        }
        return retryResponse.match(
                response -> response.match(
                        (v, reqId) -> Optional.empty(),
                        e -> Optional.of(messages.getMessage("errors.unexpected.provider.communication.failure",
                                null, locale)),
                        (e, reqId) -> {
                            String prefix = retryResult == RetryResult.CONFLICT
                                    ? messages.getMessage("errors.accounts.quotas.out.of.sync.with.provider",
                                    null, locale)
                                    : null;
                            return Optional.of(Errors.flattenProviderErrorResponse(e, prefix));
                        }),
                e -> Optional.of(Errors.flattenErrors(e)));
    }

    private PostRetryRefreshResult getPostRetryRefreshResult(RetryableOperation operation,
                                                             Result<Response<ReceivedAccount>> result) {
        String operationId = operation.getOperation().getOperationId();
        String accountId = operation.getOperation().getRequestedChanges().getAccountId().orElseThrow();
        return result.match(resp -> resp.match(
                        (acc, requestId) -> PostRetryRefreshResult.SUCCESS,
                        e -> {
                            LOG.error("Failed to refresh account " + accountId + " after provision update for " +
                                    "operation " + operationId, e);
                            return PostRetryRefreshResult.NON_FATAL_FAILURE;
                        },
                        (e, requestId) -> {
                            LOG.error("Failed to refresh account {} after provision update for operation " +
                                    "{} (requestId = {}): {}", accountId, operationId, requestId, e);
                            return e.match(new ProviderError.Cases<>() {
                                @Override
                                public PostRetryRefreshResult httpError(int statusCode) {
                                    if (isRefreshNotImplemented(statusCode)) {
                                        return PostRetryRefreshResult.UNSUPPORTED;
                                    }
                                    if (isRefreshFatalError(statusCode)) {
                                        return PostRetryRefreshResult.FATAL_FAILURE;
                                    }
                                    return PostRetryRefreshResult.NON_FATAL_FAILURE;
                                }

                                @Override
                                public PostRetryRefreshResult httpExtendedError(int statusCode,
                                                                                ErrorMessagesDto errors) {
                                    if (isRefreshNotImplemented(statusCode)) {
                                        return PostRetryRefreshResult.UNSUPPORTED;
                                    }
                                    if (isRefreshFatalError(statusCode)) {
                                        return PostRetryRefreshResult.FATAL_FAILURE;
                                    }
                                    return PostRetryRefreshResult.NON_FATAL_FAILURE;
                                }

                                @Override
                                public PostRetryRefreshResult grpcError(Status.Code statusCode, String message) {
                                    if (isRefreshNotImplemented(statusCode)) {
                                        return PostRetryRefreshResult.UNSUPPORTED;
                                    }
                                    if (isRefreshFatalError(statusCode)) {
                                        return PostRetryRefreshResult.FATAL_FAILURE;
                                    }
                                    return PostRetryRefreshResult.NON_FATAL_FAILURE;
                                }

                                @Override
                                public PostRetryRefreshResult grpcExtendedError(Status.Code statusCode, String message,
                                                                                Map<String, String> badRequestDetails) {
                                    if (isRefreshNotImplemented(statusCode)) {
                                        return PostRetryRefreshResult.UNSUPPORTED;
                                    }
                                    if (isRefreshFatalError(statusCode)) {
                                        return PostRetryRefreshResult.FATAL_FAILURE;
                                    }
                                    return PostRetryRefreshResult.NON_FATAL_FAILURE;
                                }
                            });
                        }),
                e -> {
                    LOG.error("Failed to refresh account {} after provision update for operation {}: {}",
                            accountId, operationId, e);
                    return PostRetryRefreshResult.NON_FATAL_FAILURE;
                });
    }

    private Optional<ReceivedAccount> getPostRetryRefreshedAccount(
            Result<Response<ReceivedAccount>> result) {
        return result.match(
                resp -> resp.match(
                        (acc, requestId) -> Optional.of(acc),
                        e -> Optional.empty(),
                        (e, requestId) -> Optional.empty()),
                e -> Optional.empty());
    }

    private Optional<String> getPostRetryRefreshErrorDescription(
            Result<Response<ReceivedAccount>> refreshResponse,
            PostRetryRefreshResult postRetryRefreshResult,
            Locale locale) {
        if (postRetryRefreshResult == PostRetryRefreshResult.SUCCESS) {
            return Optional.empty();
        }
        return refreshResponse.match(
                response -> response.match(
                        (v, reqId) -> Optional.empty(),
                        e -> Optional.of(messages.getMessage("errors.unexpected.provider.communication.failure",
                                null, locale)),
                        (e, reqId) -> {
                            String prefix = postRetryRefreshResult == PostRetryRefreshResult.UNSUPPORTED
                                    ? messages.getMessage("errors.provision.update.check.is.not.possible",
                                    null, locale)
                                    : null;
                            return Optional.of(Errors.flattenProviderErrorResponse(e, prefix));
                        }),
                e -> Optional.of(Errors.flattenErrors(e)));
    }

    private Mono<Void> completeOnRefreshAlreadyApplied(YdbTxSession session,
                                                       RetryableOperation operation,
                                                       DeliverUpdateProvisionOperationRetryContext retryContext,
                                                       Instant now) {
        if (isOperationComplete(operation)) {
            return Mono.empty();
        }
        return storeService.loadDeliverUpdateProvisionContext(session, operation.getOperation()).flatMap(context -> {
            if (!isProvisionChangeApplied(retryContext.getAlreadyUpdatedAccount().orElseThrow(),
                    operation.getOperation(), context, retryContext.getCommonContext())) {
                return Mono.empty();
            }
            return applyOperationOnRefresh(session, context, operation, retryContext, now);
        });
    }

    private Mono<Void> applyOperationOnRefresh(YdbTxSession session,
                                               DeliverUpdateProvisionContext updateContext,
                                               RetryableOperation operation,
                                               DeliverUpdateProvisionOperationRetryContext retryContext,
                                               Instant now) {
        return applyOperation(session, updateContext, operation, retryContext.getCommonContext(),
                retryContext.getAlreadyUpdatedAccount().orElseThrow(), retryContext.getPreGeneratedFolderOpLogId(),
                now);
    }

    private Mono<Void> applyOperation(YdbTxSession session,
                                      DeliverUpdateProvisionContext deliverUpdateProvisionContext,
                                      RetryableOperation operation,
                                      OperationCommonContext commonContext,
                                      ValidatedReceivedAccount receivedAccount,
                                      String folderOpLogId,
                                      Instant now) {
        String operationId = operation.getOperation().getOperationId();
        String knownFolderId = deliverUpdateProvisionContext.getFolder().getId();
        String knownAccountId = deliverUpdateProvisionContext.getAccount().getId();
        String knownProviderId = commonContext.getProvider().getId();
        AccountsSettingsModel accountsSettings = commonContext.getProvider().getAccountsSettings();
        boolean provisionsVersionedSeparately = accountsSettings.isPerProvisionVersionSupported();
        boolean accountAndProvisionsVersionedTogether = accountsSettings.isPerAccountVersionSupported()
                && !accountsSettings.isPerProvisionVersionSupported();
        List<OperationChangesModel.Provision> targetProvisions = operation.getOperation().getRequestedChanges()
                .getUpdatedProvisions().orElse(List.of());
        List<OperationChangesModel.Provision> frozenProvisions = operation.getOperation().getRequestedChanges()
                .getFrozenProvisions().orElse(List.of());
        Map<String, OperationChangesModel.Provision> frozenProvisionsByResourceId = frozenProvisions.stream()
                .collect(Collectors.toMap(OperationChangesModel.Provision::getResourceId, Function.identity()));
        Map<String, AccountsQuotasModel> knownProvisionsByResourceId = deliverUpdateProvisionContext
                .getProvisionsByAccountIdResourceId().getOrDefault(deliverUpdateProvisionContext.getAccount().getId(),
                        Map.of());
        Map<String, ValidatedReceivedProvision> receivedProvisionByResourceId = receivedAccount.getProvisions()
                .stream().collect(Collectors.toMap(v -> v.getResource().getId(), Function.identity()));
        List<AccountsQuotasModel> updatedProvisions = new ArrayList<>();
        List<QuotaModel> updatedQuotas = new ArrayList<>();
        FolderOperationLogModel.Builder opLogBuilder = prepareFolderOpLogBuilder(deliverUpdateProvisionContext, now,
                operationId, knownFolderId, commonContext, folderOpLogId);
        FolderModel updatedFolder = deliverUpdateProvisionContext.getFolder().toBuilder()
                .setNextOpLogOrder(deliverUpdateProvisionContext.getFolder().getNextOpLogOrder() + 1L)
                .build();
        Optional<AccountModel> updatedAccount = prepareUpdatedAccountOnRefresh(deliverUpdateProvisionContext,
                knownAccountId, receivedAccount, accountAndProvisionsVersionedTogether, opLogBuilder, accountsSettings);
        processUpdatedProvisions(deliverUpdateProvisionContext, now, operationId, knownFolderId, knownAccountId,
                knownProviderId, provisionsVersionedSeparately, targetProvisions, frozenProvisionsByResourceId,
                knownProvisionsByResourceId, receivedProvisionByResourceId, updatedProvisions, updatedQuotas,
                opLogBuilder);
        FolderOperationLogModel opLog = opLogBuilder.build();
        AccountsQuotasOperationsModel updatedOperation = new AccountsQuotasOperationsModel
                .Builder(operation.getOperation())
                .setRequestStatus(AccountsQuotasOperationsModel.RequestStatus.OK)
                .setUpdateDateTime(now)
                .setOrders(OperationOrdersModel.builder(operation.getOperation().getOrders())
                        .closeOrder(deliverUpdateProvisionContext.getFolder().getNextOpLogOrder())
                        .build())
                .setErrorKind(null)
                .build();
        operationsObservabilityService.observeOperationFinished(updatedOperation);
        return storeService.upsertOperationAndReturn(session, updatedOperation)
                .then(Mono.defer(() -> storeService.deleteOperationsInProgress(session, operation.getInProgress())))
                .then(Mono.defer(() -> storeService.upsertQuotas(session, updatedQuotas)))
                .then(Mono.defer(() -> storeService.upsertFolder(session, updatedFolder)))
                .then(Mono.defer(() -> storeService.upsertFolderOpLog(session, opLog)))
                .then(Mono.defer(() -> storeService.upsertProvisions(session, updatedProvisions)))
                .then(Mono.defer(() -> updatedAccount.isPresent()
                        ? storeService.upsertAccount(session, updatedAccount.get()) : Mono.empty()));
    }

    private FolderOperationLogModel.Builder prepareFolderOpLogBuilder(
            DeliverUpdateProvisionContext updateContext, Instant now, String operationId, String knownFolderId,
            OperationCommonContext commonContext, String opLogId) {
        return FolderOperationLogModel.builder()
                .setTenantId(Tenants.DEFAULT_TENANT_ID)
                .setFolderId(knownFolderId)
                .setOperationDateTime(now)
                .setId(opLogId)
                .setProviderRequestId(null)
                .setOperationType(FolderOperationType.DELIVER_PROVIDE_QUOTAS_TO_ACCOUNT)
                .setAuthorUserId(commonContext.getAuthor().getId())
                .setAuthorUserUid(commonContext.getAuthor().getPassportUid().orElse(null))
                .setAuthorProviderId(null)
                .setSourceFolderOperationsLogId(null)
                .setDestinationFolderOperationsLogId(null)
                .setOldFolderFields(null)
                .setNewFolderFields(null)
                .setOldQuotas(new QuotasByResource(Map.of()))
                .setNewQuotas(new QuotasByResource(Map.of()))
                .setAccountsQuotasOperationsId(operationId)
                .setQuotasDemandsId(null)
                .setOperationPhase(OperationPhase.CLOSE)
                .setOrder(updateContext.getFolder().getNextOpLogOrder());
    }

    private Optional<AccountModel> prepareUpdatedAccountOnRefresh(
            DeliverUpdateProvisionContext deliverUpdateProvisionContext,
            String knownAccountId,
            ValidatedReceivedAccount receivedAccount,
            boolean accountAndProvisionsVersionedTogether,
            FolderOperationLogModel.Builder opLogBuilder,
            AccountsSettingsModel accountsSettings) {
        if (accountAndProvisionsVersionedTogether && receivedAccount.getAccountVersion().isPresent()) {
            AccountModel.Builder updatedAccountBuilder =
                    new AccountModel.Builder(deliverUpdateProvisionContext.getAccount())
                            .setLastReceivedVersion(receivedAccount.getAccountVersion().get())
                            .setVersion(deliverUpdateProvisionContext.getAccount().getVersion() + 1L);
            AccountHistoryModel.Builder oldAccountBuilder = AccountHistoryModel.builder()
                    .lastReceivedVersion(deliverUpdateProvisionContext.getAccount()
                            .getLastReceivedVersion().orElse(null))
                    .version(deliverUpdateProvisionContext.getAccount().getVersion());
            AccountHistoryModel.Builder newAccountBuilder = AccountHistoryModel.builder()
                    .lastReceivedVersion(receivedAccount.getAccountVersion().get())
                    .version(deliverUpdateProvisionContext.getAccount().getVersion() + 1L);
            if (accountsSettings.isDisplayNameSupported()
                    && !deliverUpdateProvisionContext.getAccount()
                    .getDisplayName().equals(receivedAccount.getDisplayName())) {
                updatedAccountBuilder.setDisplayName(receivedAccount.getDisplayName().orElse(null));
                oldAccountBuilder.displayName(deliverUpdateProvisionContext.getAccount().getDisplayName().orElse(null));
                newAccountBuilder.displayName(receivedAccount.getDisplayName().orElse(null));
            }
            if (accountsSettings.isKeySupported()
                    && !deliverUpdateProvisionContext.getAccount()
                    .getOuterAccountKeyInProvider().equals(receivedAccount.getKey())) {
                updatedAccountBuilder.setOuterAccountKeyInProvider(receivedAccount.getKey().orElse(null));
                oldAccountBuilder.outerAccountKeyInProvider(deliverUpdateProvisionContext.getAccount()
                        .getOuterAccountKeyInProvider().orElse(null));
                newAccountBuilder.outerAccountKeyInProvider(receivedAccount.getKey().orElse(null));
            }
            Optional<AccountModel> updatedAccount = Optional.of(updatedAccountBuilder.build());
            opLogBuilder.setOldAccounts(new AccountsHistoryModel(Map.of(knownAccountId, oldAccountBuilder.build())));
            opLogBuilder.setNewAccounts(new AccountsHistoryModel(Map.of(knownAccountId, newAccountBuilder.build())));
            return updatedAccount;
        } else {
            return Optional.empty();
        }
    }

    @SuppressWarnings("ParameterNumber")
    private void processUpdatedProvisions(
            DeliverUpdateProvisionContext updateContext,
            Instant now,
            String operationId,
            String knownFolderId,
            String knownAccountId,
            String knownProviderId,
            boolean provisionsVersionedSeparately,
            List<OperationChangesModel.Provision> targetProvisions,
            Map<String, OperationChangesModel.Provision> frozenProvisionsByResourceId,
            Map<String, AccountsQuotasModel> knownProvisionsByResourceId,
            Map<String, ValidatedReceivedProvision> receivedProvisionByResourceId,
            List<AccountsQuotasModel> updatedProvisions,
            List<QuotaModel> updatedQuotas,
            FolderOperationLogModel.Builder opLogBuilder) {
        Map<String, ProvisionHistoryModel> oldProvisionsByResourceId = new HashMap<>();
        Map<String, ProvisionHistoryModel> newProvisionsByResourceId = new HashMap<>();
        Map<String, ProvisionHistoryModel> actualProvisionsByResourceId = new HashMap<>();
        Map<String, Long> oldBalanceByResourceId = new HashMap<>();
        Map<String, Long> newBalanceByResourceId = new HashMap<>();
        processUpdatedProvisions(updateContext, now, operationId, knownFolderId, knownAccountId,
                knownProviderId, provisionsVersionedSeparately, targetProvisions, frozenProvisionsByResourceId,
                knownProvisionsByResourceId, receivedProvisionByResourceId, updatedProvisions, updatedQuotas,
                oldProvisionsByResourceId, newProvisionsByResourceId, actualProvisionsByResourceId,
                oldBalanceByResourceId, newBalanceByResourceId);
        if (!oldProvisionsByResourceId.isEmpty() || !newProvisionsByResourceId.isEmpty()) {
            opLogBuilder.setOldProvisions(new QuotasByAccount(Map.of(knownAccountId,
                    new ProvisionsByResource(oldProvisionsByResourceId))));
            opLogBuilder.setNewProvisions(new QuotasByAccount(Map.of(knownAccountId,
                    new ProvisionsByResource(newProvisionsByResourceId))));
        } else {
            opLogBuilder.setOldProvisions(new QuotasByAccount(Map.of()));
            opLogBuilder.setNewProvisions(new QuotasByAccount(Map.of()));
        }
        if (!actualProvisionsByResourceId.isEmpty()) {
            opLogBuilder.setActuallyAppliedProvisions(new QuotasByAccount(Map.of(knownAccountId,
                    new ProvisionsByResource(actualProvisionsByResourceId))));
        }
        if (!oldBalanceByResourceId.isEmpty() || !newBalanceByResourceId.isEmpty()) {
            opLogBuilder.setOldBalance(new QuotasByResource(oldBalanceByResourceId));
            opLogBuilder.setNewBalance(new QuotasByResource(newBalanceByResourceId));
        } else {
            opLogBuilder.setOldBalance(new QuotasByResource(Map.of()));
            opLogBuilder.setNewBalance(new QuotasByResource(Map.of()));
        }
    }

    @SuppressWarnings("ParameterNumber")
    private void processUpdatedProvisions(
            DeliverUpdateProvisionContext updateContext,
            Instant now,
            String operationId,
            String knownFolderId,
            String knownAccountId,
            String knownProviderId,
            boolean provisionsVersionedSeparately,
            List<OperationChangesModel.Provision> targetProvisions,
            Map<String, OperationChangesModel.Provision> frozenProvisionsByResourceId,
            Map<String, AccountsQuotasModel> knownProvisionsByResourceId,
            Map<String, ValidatedReceivedProvision> receivedProvisionByResourceId,
            List<AccountsQuotasModel> updatedProvisions,
            List<QuotaModel> updatedQuotas,
            Map<String, ProvisionHistoryModel> oldProvisionsByResourceId,
            Map<String, ProvisionHistoryModel> newProvisionsByResourceId,
            Map<String, ProvisionHistoryModel> actualProvisionsByResourceId,
            Map<String, Long> oldBalanceByResourceId,
            Map<String, Long> newBalanceByResourceId) {
        targetProvisions.forEach(targetProvision -> {
            String resourceId = targetProvision.getResourceId();
            AccountsQuotasModel knownProvision = knownProvisionsByResourceId.get(resourceId);
            QuotaModel knownQuota = updateContext.getFolderQuotasByResourceId().get(resourceId);
            OperationChangesModel.Provision frozenProvision = frozenProvisionsByResourceId.get(resourceId);
            ValidatedReceivedProvision receivedProvision = receivedProvisionByResourceId.get(resourceId);
            Long receivedVersion = receivedProvision != null ? receivedProvision.getQuotaVersion().orElse(null) : null;
            Long knownVersion = knownProvision != null
                    ? knownProvision.getLastReceivedProvisionVersion().orElse(null) : null;
            long knownProvidedAmount = knownProvision != null ? (knownProvision.getProvidedQuota() != null
                    ? knownProvision.getProvidedQuota() : 0L) : 0L;
            long targetProvidedAmount = targetProvision.getAmount();
            long receivedProvidedAmount = receivedProvision != null ? receivedProvision.getProvidedAmount() : 0L;
            long receivedAllocatedAmount = receivedProvision != null ? receivedProvision.getAllocatedAmount() : 0L;
            long targetFrozenAmount = frozenProvision != null ? frozenProvision.getAmount() : 0L;
            long knownFrozenAmount = knownQuota != null ? knownQuota.getFrozenQuota() : 0L;
            long knownBalance = knownQuota != null
                    ? (knownQuota.getBalance() != null ? knownQuota.getBalance() : 0L) : 0L;
            processUpdatedProvidedAmounts(now, operationId, knownFolderId, knownAccountId, knownProviderId,
                    provisionsVersionedSeparately, updatedProvisions, oldProvisionsByResourceId,
                    newProvisionsByResourceId, actualProvisionsByResourceId, resourceId, knownProvision,
                    receivedVersion, knownVersion, knownProvidedAmount, targetProvidedAmount, receivedProvidedAmount,
                    receivedAllocatedAmount);
            processUpdatedBalance(updateContext, operationId, knownFolderId, knownAccountId, knownProviderId,
                    updatedQuotas, oldBalanceByResourceId, newBalanceByResourceId, resourceId, knownQuota,
                    receivedProvidedAmount, targetFrozenAmount, knownFrozenAmount, knownBalance);
        });
    }

    @SuppressWarnings("ParameterNumber")
    private void processUpdatedProvidedAmounts(
            Instant now,
            String operationId,
            String knownFolderId,
            String knownAccountId,
            String knownProviderId,
            boolean provisionsVersionedSeparately,
            List<AccountsQuotasModel> updatedProvisions,
            Map<String, ProvisionHistoryModel> oldProvisionsByResourceId,
            Map<String, ProvisionHistoryModel> newProvisionsByResourceId,
            Map<String, ProvisionHistoryModel> actualProvisionsByResourceId,
            String resourceId,
            AccountsQuotasModel knownProvision,
            Long receivedVersion,
            Long knownVersion,
            long knownProvidedAmount,
            long targetProvidedAmount,
            long receivedProvidedAmount,
            long receivedAllocatedAmount) {
        if (knownProvision != null) {
            AccountsQuotasModel updatedProvision = new AccountsQuotasModel.Builder(knownProvision)
                    .setProvidedQuota(receivedProvidedAmount)
                    .setAllocatedQuota(receivedAllocatedAmount)
                    .setLastProvisionUpdate(now)
                    .setLatestSuccessfulProvisionOperationId(operationId)
                    .setLastReceivedProvisionVersion(provisionsVersionedSeparately
                            ? receivedVersion
                            : knownProvision.getLastReceivedProvisionVersion().orElse(null))
                    .build();
            updatedProvisions.add(updatedProvision);
        } else {
            AccountsQuotasModel newProvision = new AccountsQuotasModel.Builder()
                    .setTenantId(Tenants.DEFAULT_TENANT_ID)
                    .setAccountId(knownAccountId)
                    .setResourceId(resourceId)
                    .setProvidedQuota(receivedProvidedAmount)
                    .setAllocatedQuota(receivedAllocatedAmount)
                    .setFolderId(knownFolderId)
                    .setProviderId(knownProviderId)
                    .setLastProvisionUpdate(now)
                    .setLatestSuccessfulProvisionOperationId(operationId)
                    .setLastReceivedProvisionVersion(provisionsVersionedSeparately
                            ? receivedVersion : null)
                    .build();
            updatedProvisions.add(newProvision);
        }
        if (receivedProvidedAmount != knownProvidedAmount || !Objects.equal(receivedVersion, knownVersion)) {
            oldProvisionsByResourceId.put(resourceId, new ProvisionHistoryModel(knownProvidedAmount,
                    knownVersion));
            actualProvisionsByResourceId.put(resourceId, new ProvisionHistoryModel(receivedProvidedAmount,
                    receivedVersion));
        }
        if (targetProvidedAmount != knownProvidedAmount) {
            oldProvisionsByResourceId.put(resourceId, new ProvisionHistoryModel(knownProvidedAmount,
                    knownVersion));
            newProvisionsByResourceId.put(resourceId, new ProvisionHistoryModel(targetProvidedAmount,
                    targetProvidedAmount == receivedProvidedAmount ? receivedVersion : knownVersion));
        }
    }

    @SuppressWarnings("ParameterNumber")
    private void processUpdatedBalance(
            DeliverUpdateProvisionContext updateContext,
            String operationId,
            String knownFolderId,
            String knownAccountId,
            String knownProviderId,
            List<QuotaModel> updatedQuotas,
            Map<String, Long> oldBalanceByResourceId,
            Map<String, Long> newBalanceByResourceId,
            String resourceId,
            QuotaModel knownQuota,
            long receivedProvidedAmount,
            long targetFrozenAmount,
            long knownFrozenAmount,
            long knownBalance) {
        long updatedFrozenAmount = knownQuota != null ? knownQuota.getFrozenQuota() : 0L;
        if (targetFrozenAmount != 0L) {
            if (targetFrozenAmount <= knownFrozenAmount && knownQuota != null) {
                updatedFrozenAmount = knownFrozenAmount - targetFrozenAmount;
            } else {
                LOG.error("Frozen quota {} for resource {} in folder {} is less then value {} stored for "
                                + "operation {}",
                        knownFrozenAmount, resourceId, knownFolderId, targetFrozenAmount, operationId);
            }
        }
        long updatedBalance = knownQuota != null ? knownQuota.getQuota() : 0L;
        for (Map<String, AccountsQuotasModel> provisionsByResourceId :
                updateContext.getProvisionsByAccountIdResourceId().values()) {
            AccountsQuotasModel currentProvision = provisionsByResourceId.get(resourceId);
            long currentProvidedAmount = currentProvision != null
                    ? (currentProvision.getProvidedQuota() != null ? currentProvision.getProvidedQuota() : 0L) : 0L;
            if (currentProvidedAmount == 0L) {
                continue;
            }
            if (currentProvision.getAccountId().equals(knownAccountId)) {
                continue;
            }
            Optional<Long> balanceUpdateO = Units.subtract(updatedBalance, currentProvidedAmount);
            if (balanceUpdateO.isPresent()) {
                updatedBalance = balanceUpdateO.get();
            } else {
                LOG.error("Underflow while updating balance of resource {} in folder {} for operation {}",
                        resourceId, knownFolderId, operationId);
            }
        }
        Optional<Long> balanceUpdateO = Units.subtract(updatedBalance, receivedProvidedAmount);
        if (balanceUpdateO.isPresent()) {
            updatedBalance = balanceUpdateO.get();
        } else {
            LOG.error("Underflow while updating balance of resource {} in folder {} for operation {}",
                    resourceId, knownFolderId, operationId);
        }
        Optional<Long> freezeUpdateO = Units.subtract(updatedBalance, updatedFrozenAmount);
        if (freezeUpdateO.isPresent()) {
            updatedBalance = freezeUpdateO.get();
        } else {
            LOG.error("Underflow while updating balance of resource {} in folder {} for operation {}",
                    resourceId, knownFolderId, operationId);
        }
        if (knownBalance != updatedBalance || knownFrozenAmount != updatedFrozenAmount) {
            if (knownQuota != null) {
                QuotaModel updatedQuota = QuotaModel.builder(knownQuota)
                        .frozenQuota(updatedFrozenAmount)
                        .balance(updatedBalance)
                        .build();
                updatedQuotas.add(updatedQuota);
            } else {
                QuotaModel newQuota = QuotaModel.builder()
                        .tenantId(Tenants.DEFAULT_TENANT_ID)
                        .folderId(knownFolderId)
                        .providerId(knownProviderId)
                        .resourceId(resourceId)
                        .quota(0L)
                        .balance(updatedBalance)
                        .frozenQuota(updatedFrozenAmount)
                        .build();
                updatedQuotas.add(newQuota);
            }
        }
        if (knownBalance != updatedBalance) {
            oldBalanceByResourceId.put(resourceId, knownBalance);
            newBalanceByResourceId.put(resourceId, updatedBalance);
        }
    }

    private boolean isOperationComplete(RetryableOperation operation) {
        return operation.getOperation().getRequestStatus().isPresent() && operation.getOperation()
                .getRequestStatus().get() != AccountsQuotasOperationsModel.RequestStatus.WAITING;
    }

    private Mono<Void> abortOperation(YdbTxSession session,
                                      RetryableOperation operation,
                                      DeliverUpdateProvisionOperationRetryContext retryContext,
                                      Instant now,
                                      OperationErrorKind errorKind) {
        if (isOperationComplete(operation)) {
            return Mono.empty();
        }
        return saveAbortOperation(session, operation, retryContext, now, errorKind);
    }

    private Mono<Void> saveAbortOperation(YdbTxSession session, RetryableOperation operation,
                                          DeliverUpdateProvisionOperationRetryContext retryContext, Instant now,
                                          OperationErrorKind errorKind) {
        String errorMessage = retryContext.getReceivedRetryError().orElse(retryContext
                .getReceivedPostRetryRefreshError().orElse(retryContext.getReceivedRefreshError().orElse(null)));
        AccountsQuotasOperationsModel updatedOperation = new AccountsQuotasOperationsModel
                .Builder(operation.getOperation())
                .setRequestStatus(AccountsQuotasOperationsModel.RequestStatus.ERROR)
                .setUpdateDateTime(now)
                .setErrorMessage(errorMessage)
                .setErrorKind(errorKind)
                .build();
        operationsObservabilityService.observeOperationFinished(updatedOperation);
        return storeService.loadDeliverUpdateProvisionQuotas(session, operation.getOperation()).flatMap(quotas -> {
            List<QuotaModel> unfreezedQuotas = unfreezeQuotas(operation.getOperation(), quotas);
            return storeService.upsertQuotas(session, unfreezedQuotas)
                    .then(Mono.defer(() -> storeService.finishDeliverOperation(session, operation.getInProgress(),
                            updatedOperation)));
        });
    }

    private Mono<Void> planToNextRetryOperation(
            YdbTxSession session,
            RetryableOperation operation,
            DeliverUpdateProvisionOperationRetryContext retryContext,
            Instant now,
            OperationErrorKind errorKind
    ) {
        Optional<Boolean> maxAsyncRetriesReached = operation.getInProgress().stream().findFirst().map(op ->
                op.getRetryCounter() >= maxAsyncRetries);
        if (maxAsyncRetriesReached.isPresent() && maxAsyncRetriesReached.get()) {
            return abortOperation(session, operation, retryContext, now, errorKind);
        }
        operationsObservabilityService.observeOperationTransientFailure(operation.getOperation());
        return storeService.incrementRetryCounter(session, operation.getInProgress());
    }

    private Mono<Void> onRetryNoSuccessFinish(YdbTxSession ts,
                                              RetryableOperation operation,
                                              DeliverUpdateProvisionOperationRetryContext retryContext,
                                              Instant now,
                                              Locale locale,
                                              OperationErrorKind errorKind) {
        if (isOperationComplete(operation)) {
            return Mono.empty();
        }
        ReceivedAccount receivedAccount = retryContext.getPostRetryRefreshedAccount().orElseThrow();
        OperationCommonContext commonContext = retryContext.getCommonContext();
        String operationId = operation.getOperation().getOperationId();
        return validationService.validateReceivedAccount(ts, receivedAccount, commonContext, locale)
                .flatMap(validatedR -> validatedR.match(validatedAccount -> storeService
                                .loadDeliverUpdateProvisionContext(ts, operation.getOperation()).flatMap(updCtx -> {
                                    if (!isProvisionChangeApplied(validatedAccount, operation.getOperation(),
                                            updCtx, retryContext.getCommonContext())) {
                                        return saveAbortOperation(ts, operation, updCtx, retryContext, now,
                                                errorKind);
                                    }
                                    return applyOperation(ts, updCtx, operation, retryContext.getCommonContext(),
                                            validatedAccount, retryContext.getPreGeneratedFolderOpLogId(), now);
                                }),
                        e -> {
                            LOG.error("Failed to process account refresh response after provision update " +
                                    "for operation {}: {}", operationId, e);
                            return Mono.empty();
                        }));

    }

    private Mono<Void> saveAbortOperation(YdbTxSession session,
                                          RetryableOperation operation,
                                          DeliverUpdateProvisionContext updateContext,
                                          DeliverUpdateProvisionOperationRetryContext retryContext,
                                          Instant now,
                                          OperationErrorKind errorKind) {
        String errorMessage = retryContext.getReceivedRetryError().orElse(retryContext
                .getReceivedPostRetryRefreshError().orElse(retryContext.getReceivedRefreshError().orElse(null)));
        AccountsQuotasOperationsModel updatedOperation = new AccountsQuotasOperationsModel
                .Builder(operation.getOperation())
                .setRequestStatus(AccountsQuotasOperationsModel.RequestStatus.ERROR)
                .setUpdateDateTime(now)
                .setErrorMessage(errorMessage)
                .setErrorKind(errorKind)
                .build();
        List<QuotaModel> unfreezedQuotas = unfreezeQuotas(operation.getOperation(),
                new ArrayList<>(updateContext.getFolderQuotasByResourceId().values()));
        operationsObservabilityService.observeOperationFinished(updatedOperation);
        return storeService.upsertQuotas(session, unfreezedQuotas)
                .then(Mono.defer(() -> storeService.finishDeliverOperation(session, operation.getInProgress(),
                        updatedOperation)));
    }

    private Mono<Void> onRetrySuccess(YdbTxSession session,
                                      RetryableOperation operation,
                                      DeliverUpdateProvisionOperationRetryContext retryContext,
                                      Instant now,
                                      Locale locale) {
        if (isOperationComplete(operation)) {
            return Mono.empty();
        }
        ReceivedUpdatedProvision updatedProvision = retryContext.getUpdatedAccount().orElseThrow();
        OperationCommonContext commonContext = retryContext.getCommonContext();
        String operationId = operation.getOperation().getOperationId();
        return validationService.validateReceivedUpdatedProvision(session, updatedProvision, commonContext, locale)
                .flatMap(validatedR -> validatedR.match(validated ->
                                storeService.loadDeliverUpdateProvisionContext(session, operation.getOperation())
                                        .flatMap(updCtx -> applyOperationOnRetry(session, updCtx, operation,
                                                retryContext, validated, now)),
                        e -> {
                            LOG.error("Failed to process provision update response for operation {}: {}",
                                    operationId, e);
                            return Mono.empty();
                        }));
    }

    private Mono<Void> applyOperationOnRetry(YdbTxSession session,
                                             DeliverUpdateProvisionContext updateContext,
                                             RetryableOperation operation,
                                             DeliverUpdateProvisionOperationRetryContext retryContext,
                                             ValidatedReceivedUpdatedProvision updatedProvision,
                                             Instant now) {
        String operationId = operation.getOperation().getOperationId();
        String knownFolderId = updateContext.getFolder().getId();
        String knownAccountId = updateContext.getAccount().getId();
        String knownProviderId = retryContext.getCommonContext().getProvider().getId();
        AccountsSettingsModel accountsSettings = retryContext.getCommonContext().getProvider().getAccountsSettings();
        boolean provisionsVersionedSeparately = accountsSettings.isPerProvisionVersionSupported();
        boolean accountAndProvisionsVersionedTogether = accountsSettings.isPerAccountVersionSupported()
                && !accountsSettings.isPerProvisionVersionSupported();
        List<OperationChangesModel.Provision> targetProvisions = operation.getOperation().getRequestedChanges()
                .getUpdatedProvisions().orElse(List.of());
        List<OperationChangesModel.Provision> frozenProvisions = operation.getOperation().getRequestedChanges()
                .getFrozenProvisions().orElse(List.of());
        Map<String, OperationChangesModel.Provision> frozenProvisionsByResourceId = frozenProvisions.stream()
                .collect(Collectors.toMap(OperationChangesModel.Provision::getResourceId, Function.identity()));
        Map<String, AccountsQuotasModel> knownProvisionsByResourceId = updateContext
                .getProvisionsByAccountIdResourceId().getOrDefault(updateContext.getAccount().getId(), Map.of());
        Map<String, ValidatedReceivedProvision> receivedProvisionByResourceId = updatedProvision.getProvisions()
                .stream().collect(Collectors.toMap(v -> v.getResource().getId(), Function.identity()));
        List<AccountsQuotasModel> updatedProvisions = new ArrayList<>();
        List<QuotaModel> updatedQuotas = new ArrayList<>();
        FolderOperationLogModel.Builder opLogBuilder = prepareFolderOpLogBuilder(updateContext, now, operationId,
                knownFolderId, retryContext.getCommonContext(), retryContext.getPreGeneratedFolderOpLogId());
        FolderModel updatedFolder = updateContext.getFolder().toBuilder()
                .setNextOpLogOrder(updateContext.getFolder().getNextOpLogOrder() + 1L)
                .build();
        Optional<AccountModel> updatedAccount = prepareUpdatedAccountOnRetry(updateContext, knownAccountId,
                updatedProvision, accountAndProvisionsVersionedTogether, opLogBuilder);
        processUpdatedProvisions(updateContext, now, operationId, knownFolderId, knownAccountId,
                knownProviderId, provisionsVersionedSeparately, targetProvisions, frozenProvisionsByResourceId,
                knownProvisionsByResourceId, receivedProvisionByResourceId, updatedProvisions, updatedQuotas,
                opLogBuilder);
        FolderOperationLogModel opLog = opLogBuilder.build();
        AccountsQuotasOperationsModel updatedOperation = new AccountsQuotasOperationsModel
                .Builder(operation.getOperation())
                .setRequestStatus(AccountsQuotasOperationsModel.RequestStatus.OK)
                .setUpdateDateTime(now)
                .setOrders(OperationOrdersModel.builder(operation.getOperation().getOrders())
                        .closeOrder(updateContext.getFolder().getNextOpLogOrder())
                        .build())
                .setErrorKind(null)
                .build();
        operationsObservabilityService.observeOperationFinished(updatedOperation);
        return storeService.upsertOperationAndReturn(session, updatedOperation)
                .then(Mono.defer(() -> storeService.deleteOperationsInProgress(session, operation.getInProgress())))
                .then(Mono.defer(() -> storeService.upsertQuotas(session, updatedQuotas)))
                .then(Mono.defer(() -> storeService.upsertFolder(session, updatedFolder)))
                .then(Mono.defer(() -> storeService.upsertFolderOpLog(session, opLog)))
                .then(Mono.defer(() -> storeService.upsertProvisions(session, updatedProvisions)))
                .then(Mono.defer(() -> updatedAccount.isPresent()
                        ? storeService.upsertAccount(session, updatedAccount.get()) : Mono.empty()));
    }

    private Optional<AccountModel> prepareUpdatedAccountOnRetry(
            DeliverUpdateProvisionContext updateContext,
            String knownAccountId,
            ValidatedReceivedUpdatedProvision updatedProvision,
            boolean accountAndProvisionsVersionedTogether,
            FolderOperationLogModel.Builder opLogBuilder) {
        if (accountAndProvisionsVersionedTogether && updatedProvision.getAccountVersion().isPresent()) {
            AccountModel.Builder updatedAccountBuilder = new AccountModel.Builder(updateContext.getAccount())
                    .setLastReceivedVersion(updatedProvision.getAccountVersion().get())
                    .setVersion(updateContext.getAccount().getVersion() + 1L);
            AccountHistoryModel.Builder oldAccountBuilder = AccountHistoryModel.builder()
                    .lastReceivedVersion(updateContext.getAccount().getLastReceivedVersion().orElse(null))
                    .version(updateContext.getAccount().getVersion());
            AccountHistoryModel.Builder newAccountBuilder = AccountHistoryModel.builder()
                    .lastReceivedVersion(updatedProvision.getAccountVersion().get())
                    .version(updateContext.getAccount().getVersion() + 1L);
            Optional<AccountModel> updatedAccount = Optional.of(updatedAccountBuilder.build());
            opLogBuilder.setOldAccounts(new AccountsHistoryModel(Map.of(knownAccountId, oldAccountBuilder.build())));
            opLogBuilder.setNewAccounts(new AccountsHistoryModel(Map.of(knownAccountId, newAccountBuilder.build())));
            return updatedAccount;
        } else {
            return Optional.empty();
        }
    }
}
