package ru.yandex.intranet.d.services.operations

import com.google.common.base.Objects
import kotlinx.coroutines.reactor.awaitSingle
import kotlinx.coroutines.reactor.awaitSingleOrNull
import mu.KotlinLogging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.beans.factory.annotation.Value
import org.springframework.context.MessageSource
import org.springframework.context.annotation.Lazy
import org.springframework.stereotype.Component
import reactor.core.publisher.Mono
import ru.yandex.intranet.d.dao.Tenants
import ru.yandex.intranet.d.datasource.model.YdbTxSession
import ru.yandex.intranet.d.i18n.Locales
import ru.yandex.intranet.d.kotlin.AccountId
import ru.yandex.intranet.d.kotlin.LoanId
import ru.yandex.intranet.d.kotlin.ResourceId
import ru.yandex.intranet.d.kotlin.awaitSingleOrNullWithMdc
import ru.yandex.intranet.d.kotlin.map
import ru.yandex.intranet.d.kotlin.mono
import ru.yandex.intranet.d.kotlin.withMDC
import ru.yandex.intranet.d.model.accounts.AccountModel
import ru.yandex.intranet.d.model.accounts.AccountsQuotasModel
import ru.yandex.intranet.d.model.accounts.AccountsQuotasOperationsModel
import ru.yandex.intranet.d.model.accounts.OperationChangesModel
import ru.yandex.intranet.d.model.accounts.OperationErrorKind
import ru.yandex.intranet.d.model.accounts.OperationInProgressModel
import ru.yandex.intranet.d.model.accounts.OperationOrdersModel
import ru.yandex.intranet.d.model.folders.AccountHistoryModel
import ru.yandex.intranet.d.model.folders.AccountsHistoryModel
import ru.yandex.intranet.d.model.folders.FolderModel
import ru.yandex.intranet.d.model.folders.FolderOperationLogModel
import ru.yandex.intranet.d.model.folders.FolderOperationType
import ru.yandex.intranet.d.model.folders.OperationPhase
import ru.yandex.intranet.d.model.folders.ProvisionHistoryModel
import ru.yandex.intranet.d.model.folders.ProvisionsByResource
import ru.yandex.intranet.d.model.folders.QuotasByAccount
import ru.yandex.intranet.d.model.folders.QuotasByResource
import ru.yandex.intranet.d.model.folders.TransferMetaHistoryModel
import ru.yandex.intranet.d.model.folders.TransferMetaHistoryModel.RoleInTransfer
import ru.yandex.intranet.d.model.providers.AccountsSettingsModel
import ru.yandex.intranet.d.model.quotas.QuotaModel
import ru.yandex.intranet.d.model.transfers.TransferRequestModel
import ru.yandex.intranet.d.services.integration.providers.Response
import ru.yandex.intranet.d.services.operations.OperationUtils.isAnotherRoundAfterRefresh
import ru.yandex.intranet.d.services.operations.OperationUtils.isOperationAppliedAfterRefresh
import ru.yandex.intranet.d.services.operations.OperationUtils.isOperationNotComplete
import ru.yandex.intranet.d.services.operations.OperationUtils.resolvePostRetryRefreshResult
import ru.yandex.intranet.d.services.operations.OperationUtils.resolveRefreshResult
import ru.yandex.intranet.d.services.operations.OperationUtils.resolveRetryResult
import ru.yandex.intranet.d.services.operations.model.MoveProvisionContext
import ru.yandex.intranet.d.services.operations.model.MoveProvisionOperationPostRefreshContext
import ru.yandex.intranet.d.services.operations.model.MoveProvisionOperationPreRefreshContext
import ru.yandex.intranet.d.services.operations.model.MoveProvisionOperationRefreshContext
import ru.yandex.intranet.d.services.operations.model.MoveProvisionOperationRetryContext
import ru.yandex.intranet.d.services.operations.model.OperationCommonContext
import ru.yandex.intranet.d.services.operations.model.OperationPostRefreshContext
import ru.yandex.intranet.d.services.operations.model.OperationPreRefreshContext
import ru.yandex.intranet.d.services.operations.model.OperationRefreshContext
import ru.yandex.intranet.d.services.operations.model.OperationRetryContext
import ru.yandex.intranet.d.services.operations.model.PostRetryRefreshResult
import ru.yandex.intranet.d.services.operations.model.ReceivedAccount
import ru.yandex.intranet.d.services.operations.model.ReceivedMoveProvision
import ru.yandex.intranet.d.services.operations.model.RefreshResult
import ru.yandex.intranet.d.services.operations.model.RetryResult
import ru.yandex.intranet.d.services.operations.model.RetryableOperation
import ru.yandex.intranet.d.services.operations.model.ValidatedReceivedAccount
import ru.yandex.intranet.d.services.operations.model.ValidatedReceivedMoveProvision
import ru.yandex.intranet.d.services.operations.model.ValidatedReceivedProvision
import ru.yandex.intranet.d.services.quotas.MoveProvisionLogicServiceImpl
import ru.yandex.intranet.d.services.quotas.MoveProvisionOperationResult
import ru.yandex.intranet.d.services.quotas.TransferRequestCloser
import ru.yandex.intranet.d.services.transfer.TransferLoansService
import ru.yandex.intranet.d.util.Details
import ru.yandex.intranet.d.util.MdcKey
import ru.yandex.intranet.d.util.result.ErrorCollection
import ru.yandex.intranet.d.util.result.ErrorType
import ru.yandex.intranet.d.util.result.LocalizedErrors
import ru.yandex.intranet.d.util.result.Result
import ru.yandex.intranet.d.util.result.TypedError
import ru.yandex.intranet.d.web.errors.Errors
import ru.yandex.intranet.d.web.security.model.YaUserDetails
import java.math.BigInteger
import java.time.Instant
import java.util.*
import kotlin.math.max
import kotlin.math.min

/**
 * MoveProvisionOperationsRetryService.
 *
 * @author Vladimir Zaytsev <vzay@yandex-team.ru>
 * @since 20-12-2021
 */
private val logger = KotlinLogging.logger {}

@Component
class MoveProvisionOperationsRetryService(
    private val integrationService: OperationsRetryIntegrationService,
    private val validationService: OperationsRetryValidationService,
    private val storeService: OperationsRetryStoreService,
    private val operationsObservabilityService: OperationsObservabilityService,
    private val transferLoansService: TransferLoansService,
    @Qualifier("messageSource") private val messages: MessageSource,
    @Value("\${providers.client.maxAsyncRetries}") private val maxAsyncRetries: Long,
) : BaseOperationRetryService {

    @Autowired
    @Lazy
    lateinit var transferRequestCloser: TransferRequestCloser

    override fun preRefreshOperation(
        session: YdbTxSession,
        operation: RetryableOperation,
        context: OperationCommonContext,
        now: Instant,
        locale: Locale
    ): Mono<Optional<MoveProvisionOperationPreRefreshContext>> = mono {
        preRefreshOperationSuspend(session, operation, context, now, locale)
    }

    suspend fun preRefreshOperationSuspend(
        session: YdbTxSession,
        operation: RetryableOperation,
        context: OperationCommonContext,
        now: Instant,
        locale: Locale
    ): Optional<MoveProvisionOperationPreRefreshContext> = withMDC(prepareOperationMDC(operation.operation)) {
        val op = operation.operation
        val moveProvisionContext = storeService.loadMoveProvisionContext(session, op).awaitSingle()
        return@withMDC Optional.of(
            MoveProvisionOperationPreRefreshContext(
                commonContext = context,
                moveProvisionContext = moveProvisionContext
            )
        )
    }

    override fun refreshOperation(
        operation: RetryableOperation,
        context: OperationPreRefreshContext,
        now: Instant,
        locale: Locale
    ): Mono<Optional<MoveProvisionOperationRefreshContext>> = mono {
        refreshOperationSuspend(operation, context, now, locale)
    }

    suspend fun refreshOperationSuspend(
        operation: RetryableOperation,
        context: OperationPreRefreshContext,
        now: Instant,
        locale: Locale
    ): Optional<MoveProvisionOperationRefreshContext> = withMDC(prepareOperationMDC(operation.operation)) {
        val preRefreshContext = context as MoveProvisionOperationPreRefreshContext
        val receivedAccountFrom = integrationService.getAccountById(
            preRefreshContext.moveProvisionContext.sourceFolder,
            preRefreshContext.moveProvisionContext.sourceAccount,
            preRefreshContext.commonContext,
            locale
        ).awaitSingleOrNullWithMdc()!!
        val receivedAccountTo = integrationService.getAccountById(
            preRefreshContext.moveProvisionContext.destinationFolder,
            preRefreshContext.moveProvisionContext.destinationAccount,
            preRefreshContext.commonContext,
            locale
        ).awaitSingleOrNullWithMdc()!!
        return@withMDC Optional.of(
            MoveProvisionOperationRefreshContext(
                preRefreshContext = preRefreshContext,
                receivedAccountFrom = receivedAccountFrom,
                receivedAccountTo = receivedAccountTo
            )
        )
    }

    override fun postRefreshOperation(
        session: YdbTxSession,
        operation: RetryableOperation,
        context: OperationRefreshContext,
        now: Instant,
        locale: Locale
    ): Mono<Optional<MoveProvisionOperationPostRefreshContext>> = mono {
        postRefreshOperationSuspend(session, operation, context, now, locale)
    }

    suspend fun postRefreshOperationSuspend(
        session: YdbTxSession,
        operation: RetryableOperation,
        context: OperationRefreshContext,
        now: Instant,
        locale: Locale
    ): Optional<MoveProvisionOperationPostRefreshContext> = withMDC(prepareOperationMDC(operation.operation)) {
        val refreshContext = context as MoveProvisionOperationRefreshContext
        val validatedAccountFrom = validateAccount(
            refreshContext.receivedAccountFrom, session, refreshContext.commonContext, locale
        )
        val validatedAccountTo = validateAccount(
            refreshContext.receivedAccountTo, session, refreshContext.commonContext, locale
        )
        return@withMDC Optional.of(
            MoveProvisionOperationPostRefreshContext(
                context = refreshContext,
                validatedAccountFrom = validatedAccountFrom,
                validatedAccountTo = validatedAccountTo,
            )
        )
    }

    override fun retryOperation(
        operation: RetryableOperation,
        postRefreshContext: OperationPostRefreshContext,
        now: Instant,
        locale: Locale
    ): Mono<Optional<OperationRetryContext>> = mono {
        retryOperationSuspend(operation, postRefreshContext, now, locale)
    }

    suspend fun retryOperationSuspend(
        operation: RetryableOperation,
        postRefreshContext: OperationPostRefreshContext,
        now: Instant,
        locale: Locale
    ): Optional<OperationRetryContext> = withMDC(prepareOperationMDC(operation.operation)) {
        val (context, validatedAccountFromResponse, validatedAccountToResponse) =
            postRefreshContext as MoveProvisionOperationPostRefreshContext
        val accountsQuotasOperationsModel = operation.operation
        val refreshResult = getRefreshResult(operation, postRefreshContext)
        if (isOperationAppliedAfterRefresh(refreshResult) || isAnotherRoundAfterRefresh(refreshResult)) {
            val validatedAccountFrom = extractValue(validatedAccountFromResponse)
            val validatedAccountTo = extractValue(validatedAccountToResponse)
            return@withMDC Optional.of(MoveProvisionOperationRetryContext(
                commonContext = postRefreshContext.commonContext,
                preRefreshContext = postRefreshContext.preRefreshContext,
                refreshContext = postRefreshContext.refreshContext,
                postRefreshContext = postRefreshContext,
                refreshResult = refreshResult,
                refreshedSourceAccount = validatedAccountFrom,
                refreshedDestinationAccount = validatedAccountTo,
                retryResult = null,
            ))
        }
        val refreshedSourceAccountErrors = getRefreshErrorDescription(validatedAccountFromResponse, refreshResult)
        val refreshedDestAccountErrors = getRefreshErrorDescription(validatedAccountToResponse, refreshResult)
        val moveProvisionResponse = integrationService.moveProvision(context, accountsQuotasOperationsModel, locale)
            .awaitSingleOrNullWithMdc()!!

        val operationId = operation.operation.operationId
        val retryResult = getRetryResult(moveProvisionResponse, operationId)
        val retryErrors = getRetryErrors(retryResult, moveProvisionResponse)
        if (retryResult != RetryResult.SUCCESS) {
            val moveProvisionContext = postRefreshContext.preRefreshContext.moveProvisionContext
            val receivedSourceAccountResponse = integrationService.getAccountById(moveProvisionContext.sourceFolder,
                moveProvisionContext.sourceAccount, postRefreshContext.commonContext, locale)
                .awaitSingleOrNullWithMdc()!!
            val receivedDestinationAccountResponse = integrationService.getAccountById(
                moveProvisionContext.destinationFolder, moveProvisionContext.destinationAccount,
                postRefreshContext.commonContext, locale)
                .awaitSingleOrNullWithMdc()!!
            val sourcePostRetryRefreshResult = getPostRetryRefreshResult(receivedSourceAccountResponse, operationId,
                moveProvisionContext.sourceAccount.id)
            val destinationPostRetryRefreshResult = getPostRetryRefreshResult(receivedDestinationAccountResponse,
                operationId, moveProvisionContext.destinationAccount.id)
            val postRetryRefreshResult = commonPostRetryRefreshResult(sourcePostRetryRefreshResult,
                destinationPostRetryRefreshResult)
            val postSourceAccountErrors = getPostRetryRefreshErrors(postRetryRefreshResult,
                receivedSourceAccountResponse)
            val postDestinationAccountErrors = getPostRetryRefreshErrors(postRetryRefreshResult,
                receivedDestinationAccountResponse)
            val receivedSourceAccount = extractValue(receivedSourceAccountResponse)
            val receivedDestinationAccount = extractValue(receivedDestinationAccountResponse)
            return@withMDC Optional.of(MoveProvisionOperationRetryContext(
                commonContext = postRefreshContext.commonContext,
                preRefreshContext = postRefreshContext.preRefreshContext,
                refreshContext = postRefreshContext.refreshContext,
                postRefreshContext = postRefreshContext,
                retryResult = retryResult,
                refreshResult = refreshResult,
                postRetryRefreshResult = postRetryRefreshResult,
                postRetrySourceAccount = receivedSourceAccount,
                postRetryDestinationAccount = receivedDestinationAccount,
                retryErrors = retryErrors,
                refreshErrors = refreshedSourceAccountErrors ?: refreshedDestAccountErrors,
                postRetryErrors = postSourceAccountErrors ?: postDestinationAccountErrors
            ))
        }
        val receivedMoveProvision = extractValue(moveProvisionResponse)
        return@withMDC Optional.of(MoveProvisionOperationRetryContext(
            commonContext = postRefreshContext.commonContext,
            preRefreshContext = postRefreshContext.preRefreshContext,
            refreshContext = postRefreshContext.refreshContext,
            postRefreshContext = postRefreshContext,
            retryResult = retryResult,
            receivedMoveProvision = receivedMoveProvision,
            refreshResult = refreshResult,
            retryErrors = retryErrors
        ))
    }

    override fun postRetryOperation(
        session: YdbTxSession,
        operation: RetryableOperation,
        retryContext: OperationRetryContext,
        now: Instant,
        locale: Locale
    ): Mono<Void> = mono {
        postRetryOperationSuspend(session, operation, retryContext, now, locale)
        return@mono null
    }

    suspend fun postRetryOperationSuspend(
        session: YdbTxSession,
        operation: RetryableOperation,
        retryContext: OperationRetryContext,
        now: Instant,
        locale: Locale
    ) = withMDC(prepareOperationMDC(operation.operation)) {
        val moveRetryContext = retryContext as MoveProvisionOperationRetryContext
        if (isAnotherRoundAfterRefresh(retryContext.refreshResult)) {
            planToNextRetryOperation(session, operation, moveRetryContext, RetryResult.NON_FATAL_FAILURE, now,
                OperationErrorKind.EXPIRED)
        } else if (isOperationAppliedAfterRefresh(moveRetryContext.refreshResult)
            && moveRetryContext.refreshedSourceAccount != null
            && moveRetryContext.refreshedDestinationAccount != null) {
            completeOnRefreshAlreadyApplied(session, operation, moveRetryContext.refreshedSourceAccount,
                moveRetryContext.refreshedDestinationAccount, moveRetryContext.commonContext, now)
        } else {
            when (val retryResult = moveRetryContext.retryResult.orElseThrow()) {
                RetryResult.SUCCESS -> {
                    onRetrySuccess(session, operation, retryContext, now, locale)
                }
                RetryResult.CONFLICT -> {
                    val errorKind = OperationErrorKind.FAILED_PRECONDITION
                    when (moveRetryContext.postRetryRefreshResult!!) {
                        PostRetryRefreshResult.SUCCESS -> {
                            onRetryNoSuccessFinish(session, operation, retryContext, retryResult, now, locale,
                                errorKind)
                        }
                        PostRetryRefreshResult.NON_FATAL_FAILURE -> {
                            planToNextRetryOperation(session, operation, moveRetryContext, retryResult, now,
                                errorKind)
                        }
                        PostRetryRefreshResult.FATAL_FAILURE -> {
                            abortOperation(session, operation, moveRetryContext, retryResult, now, errorKind)
                        }
                        PostRetryRefreshResult.UNSUPPORTED -> {
                            abortOperation(session, operation, moveRetryContext, retryResult, now, errorKind)
                        }
                    }
                }
                RetryResult.NON_FATAL_FAILURE -> {
                    planToNextRetryOperation(session, operation, moveRetryContext, retryResult, now,
                        OperationErrorKind.EXPIRED)
                }
                RetryResult.FATAL_FAILURE -> {
                    val errorKind = OperationErrorKind.INVALID_ARGUMENT
                    when (moveRetryContext.postRetryRefreshResult!!) {
                        PostRetryRefreshResult.SUCCESS -> {
                            onRetryNoSuccessFinish(session, operation, retryContext, retryResult, now, locale,
                                errorKind)
                        }
                        PostRetryRefreshResult.NON_FATAL_FAILURE -> {
                            planToNextRetryOperation(session, operation, moveRetryContext, retryResult, now,
                                errorKind)
                        }
                        PostRetryRefreshResult.FATAL_FAILURE -> {
                            abortOperation(session, operation, moveRetryContext, retryResult, now, errorKind)
                        }
                        PostRetryRefreshResult.UNSUPPORTED -> {
                            abortOperation(session, operation, moveRetryContext, retryResult, now, errorKind)
                        }
                    }
                }
            }
        }
    }

    override fun abortOperation(
        session: YdbTxSession,
        operation: AccountsQuotasOperationsModel,
        comment: String,
        now: Instant,
        currentUser: YaUserDetails,
        locale: Locale
    ): Mono<Void> = mono {
        abortOperationSuspend(session, operation, comment, now, locale)
        return@mono null
    }

    suspend fun abortOperationSuspend(
        session: YdbTxSession,
        operation: AccountsQuotasOperationsModel,
        comment: String,
        now: Instant,
        locale: Locale
    ) = withMDC(prepareOperationMDC(operation)) {
        if (isOperationNotComplete(operation)) {
            val errors = localizedErrors("errors.operation.was.aborted.by.user.request", ErrorType.CONFLICT,
                details = comment)
            val moveProvisionContext = storeService.loadMoveProvisionContext(session, operation).awaitSingleOrNull()!!
            val updatedOperation = AccountsQuotasOperationsModel.Builder(operation)
                .setRequestStatus(AccountsQuotasOperationsModel.RequestStatus.ERROR)
                .setUpdateDateTime(now)
                .setErrorMessage(comment)
                .setErrorKind(OperationErrorKind.ABORTED)
                .setOrders(OperationOrdersModel.builder(operation.orders)
                    .closeOrder(moveProvisionContext.sourceFolder.nextOpLogOrder)
                    .destinationCloseOrder(moveProvisionContext.destinationFolder.nextOpLogOrder)
                    .build())
                .build()

            finishUnsuccessfulOperation(session, moveProvisionContext, updatedOperation, listOf(),
                RetryResult.FATAL_FAILURE, errors)
            storeService.deleteInProgressByOperationId(session, operation.operationId).awaitSingleOrNull()
        }
    }

    private suspend fun onRetrySuccess(
        session: YdbTxSession,
        operation: RetryableOperation,
        retryContext: MoveProvisionOperationRetryContext,
        now: Instant,
        locale: Locale
    ) {
        if (!isOperationNotComplete(operation)) {
            return
        }
        val moveProvision = retryContext.receivedMoveProvision!!
        val commonContext = retryContext.commonContext
        val operationId = operation.operation.operationId
        val validationResult = validationService.validateReceivedMoveProvision(session,
            moveProvision, commonContext, locale).awaitSingleOrNull()!!
        validationResult.matchSuspend({
            val context = storeService.loadMoveProvisionContext(session, operation.operation).awaitSingle()
            applyOperationOnRetry(session, context, operation, it, commonContext, now)
        }, { e ->
            logger.error("Failed to process provision move response for operation {}: {}", operationId, e)
            planToNextRetryOperation(session, operation, retryContext, retryContext.retryResult.orElseThrow(), now,
                OperationErrorKind.INVALID_ARGUMENT)
        })
    }

    private suspend fun onRetryNoSuccessFinish(
        session: YdbTxSession,
        operation: RetryableOperation,
        retryContext: MoveProvisionOperationRetryContext,
        retryResult: RetryResult,
        now: Instant,
        locale: Locale,
        errorKind: OperationErrorKind
    ) {
        if (!isOperationNotComplete(operation)) {
            return
        }
        val commonContext = retryContext.commonContext
        val operationId = operation.operation.operationId
        val validatedSourceAccountRes = validationService.validateReceivedAccount(session,
            retryContext.postRetrySourceAccount!!, commonContext, locale).awaitSingleOrNull()!!
        val validatedDestinationAccountRes = validationService.validateReceivedAccount(session,
            retryContext.postRetryDestinationAccount!!, commonContext, locale).awaitSingleOrNull()!!
        val accountById = sequenceOf(validatedSourceAccountRes, validatedDestinationAccountRes)
            .map { res ->
                res.match({
                    it
                }, { e ->
                    logger.error("Failed to process account refresh response after provision update for operation {}: {}",
                        operationId, e)
                    null
                })
            }
            .filterNotNull()
            .associateBy { it.accountId }
        if (accountById.size != 2) {
            planToNextRetryOperation(session, operation, retryContext, retryResult, now, errorKind)
            return
        }
        val sourceAccount = accountById[retryContext.postRetrySourceAccount.accountId]!!
        val destinationAccount = accountById[retryContext.postRetryDestinationAccount.accountId]!!
        val context = storeService.loadMoveProvisionContext(session, operation.operation).awaitSingle()
        val requestedChanges = operation.operation.requestedChanges
        val sourceProvisionChangeApplied = isProvisionChangeApplied(sourceAccount, context.sourceAccount,
            context.sourceFolder, requestedChanges.updatedProvisions.orElse(listOf()),
            operation.operation, context, commonContext)
        val destinationProvisionChangeApplied = isProvisionChangeApplied(destinationAccount,
            context.destinationAccount, context.destinationFolder,
            requestedChanges.updatedDestinationProvisions.orElse(listOf()), operation.operation,
            context, commonContext)
        if (!(sourceProvisionChangeApplied && destinationProvisionChangeApplied)) {
            val errors = (retryContext.postRetryErrors ?: retryContext.retryErrors) ?: retryContext.refreshErrors!!
            saveAbortOperation(session, context, operation, retryResult, now, errors, errorKind)
        } else {
            applyOperationOnRefresh(session, context, operation, sourceAccount, destinationAccount, commonContext, now)
        }
    }

    private suspend fun completeOnRefreshAlreadyApplied(
        session: YdbTxSession,
        operation: RetryableOperation,
        refreshedSourceAccount: ValidatedReceivedAccount,
        refreshedDestinationAccount: ValidatedReceivedAccount,
        commonContext: OperationCommonContext,
        now: Instant
    ) {
        if (isOperationNotComplete(operation)) {
            val moveProvisionContext = storeService.loadMoveProvisionContext(session, operation.operation)
                .awaitSingle()
            val requestedChanges = operation.operation.requestedChanges
            if (isProvisionChangeApplied(refreshedSourceAccount, moveProvisionContext.sourceAccount,
                    moveProvisionContext.sourceFolder, requestedChanges.updatedProvisions.orElse(listOf()),
                    operation.operation, moveProvisionContext, commonContext)
                && isProvisionChangeApplied(refreshedDestinationAccount,
                    moveProvisionContext.destinationAccount, moveProvisionContext.destinationFolder,
                    requestedChanges.updatedDestinationProvisions.orElse(listOf()), operation.operation,
                    moveProvisionContext, commonContext)) {
                applyOperationOnRefresh(session, moveProvisionContext, operation, refreshedSourceAccount,
                    refreshedDestinationAccount, commonContext, now)
            }
        }
    }

    private suspend fun applyOperationOnRetry(
        session: YdbTxSession,
        moveProvisionContext: MoveProvisionContext,
        operation: RetryableOperation,
        receivedMoveProvision: ValidatedReceivedMoveProvision,
        commonContext: OperationCommonContext,
        now: Instant
    ) {
        val sourceFolderOpLogId = UUID.randomUUID().toString()
        val destinationFolderOpLogId = UUID.randomUUID().toString()
        val sourceAccountOpLogBuilder = prepareFolderOpLogBuilder(
            opLogId = sourceFolderOpLogId,
            operation = operation.operation,
            knownFolder = moveProvisionContext.sourceFolder,
            now = now,
            destinationFolderOpLogId = destinationFolderOpLogId,
            sourceFolder = moveProvisionContext.sourceFolder,
            sourceAccount = moveProvisionContext.sourceAccount,
            destinationFolder = moveProvisionContext.destinationFolder,
            destinationAccount = moveProvisionContext.destinationAccount,
        )
        val destinationAccountOpLogBuilder = prepareFolderOpLogBuilder(
            opLogId = destinationFolderOpLogId,
            operation = operation.operation,
            knownFolder = moveProvisionContext.destinationFolder,
            now = now,
            sourceFolderOpLogId = sourceFolderOpLogId,
            sourceFolder = moveProvisionContext.sourceFolder,
            sourceAccount = moveProvisionContext.sourceAccount,
            destinationFolder = moveProvisionContext.destinationFolder,
            destinationAccount = moveProvisionContext.destinationAccount,
        )
        val accountsSettings = commonContext.provider.accountsSettings
        val accountAndProvisionsVersionedTogether = accountsSettings.isPerAccountVersionSupported
            && !accountsSettings.isPerProvisionVersionSupported
        val updatedSourceAccount = prepareUpdatedAccountOnRetry(moveProvisionContext.sourceAccount,
            receivedMoveProvision.sourceAccountVersion, accountAndProvisionsVersionedTogether,
            sourceAccountOpLogBuilder)
        val updatedDestinationAccount = prepareUpdatedAccountOnRetry(moveProvisionContext.destinationAccount,
            receivedMoveProvision.destinationAccountVersion, accountAndProvisionsVersionedTogether,
            destinationAccountOpLogBuilder)
        applyOperation(session, moveProvisionContext, operation, commonContext, receivedMoveProvision.sourceProvisions,
            receivedMoveProvision.destinationProvisions, updatedSourceAccount, updatedDestinationAccount,
            sourceAccountOpLogBuilder, destinationAccountOpLogBuilder, now)
    }


    private suspend fun applyOperationOnRefresh(
        session: YdbTxSession,
        moveProvisionContext: MoveProvisionContext,
        operation: RetryableOperation,
        refreshedSourceAccount: ValidatedReceivedAccount,
        refreshedDestinationAccount: ValidatedReceivedAccount,
        commonContext: OperationCommonContext,
        now: Instant
    ) {
        val sourceFolderOpLogId = UUID.randomUUID().toString()
        val destinationFolderOpLogId = UUID.randomUUID().toString()
        val sourceAccountOpLogBuilder = prepareFolderOpLogBuilder(
            opLogId = sourceFolderOpLogId,
            operation = operation.operation,
            knownFolder = moveProvisionContext.sourceFolder,
            now = now,
            destinationFolderOpLogId = destinationFolderOpLogId,
            sourceFolder = moveProvisionContext.sourceFolder,
            sourceAccount = moveProvisionContext.sourceAccount,
            destinationFolder = moveProvisionContext.destinationFolder,
            destinationAccount = moveProvisionContext.destinationAccount,
        )
        val destinationAccountOpLogBuilder = prepareFolderOpLogBuilder(
            opLogId = destinationFolderOpLogId,
            operation = operation.operation,
            knownFolder = moveProvisionContext.destinationFolder,
            now = now,
            sourceFolderOpLogId = sourceFolderOpLogId,
            sourceFolder = moveProvisionContext.sourceFolder,
            sourceAccount = moveProvisionContext.sourceAccount,
            destinationFolder = moveProvisionContext.destinationFolder,
            destinationAccount = moveProvisionContext.destinationAccount,
        )
        val accountsSettings = commonContext.provider.accountsSettings
        val accountAndProvisionsVersionedTogether = accountsSettings.isPerAccountVersionSupported
            && !accountsSettings.isPerProvisionVersionSupported
        val updatedSourceAccount = prepareUpdatedAccountOnRefresh(moveProvisionContext.sourceAccount,
            refreshedSourceAccount, accountAndProvisionsVersionedTogether, sourceAccountOpLogBuilder, accountsSettings)
        val updatedDestinationAccount = prepareUpdatedAccountOnRefresh(moveProvisionContext.destinationAccount,
            refreshedDestinationAccount, accountAndProvisionsVersionedTogether, destinationAccountOpLogBuilder,
            accountsSettings)
        applyOperation(session, moveProvisionContext, operation, commonContext, refreshedSourceAccount.provisions,
            refreshedDestinationAccount.provisions, updatedSourceAccount, updatedDestinationAccount,
            sourceAccountOpLogBuilder, destinationAccountOpLogBuilder, now)
    }

    private suspend fun applyOperation(
        session: YdbTxSession,
        moveProvisionContext: MoveProvisionContext,
        operation: RetryableOperation,
        commonContext: OperationCommonContext,
        sourceProvisions: List<ValidatedReceivedProvision>,
        destinationProvisions: List<ValidatedReceivedProvision>,
        updatedSourceAccount: AccountModel?,
        updatedDestinationAccount: AccountModel?,
        sourceAccountOpLogBuilder: FolderOperationLogModel.Builder,
        destinationAccountOpLogBuilder: FolderOperationLogModel.Builder,
        now: Instant
    ) {
        val operationId = operation.operation.operationId
        val knownSourceFolder = moveProvisionContext.sourceFolder
        val knownDestinationFolder = moveProvisionContext.destinationFolder
        val knownSourceAccount = moveProvisionContext.sourceAccount
        val knownDestinationAccount = moveProvisionContext.destinationAccount
        val knownProviderId = commonContext.provider.id
        val accountsSettings = commonContext.provider.accountsSettings
        val provisionsVersionedSeparately = accountsSettings.isPerProvisionVersionSupported

        val requestedChanges = operation.operation.requestedChanges
        val targetSourceProvisions = requestedChanges.updatedProvisions.orElse(listOf())
        val targetDestinationProvisions = requestedChanges.updatedDestinationProvisions.orElse(listOf())
        val frozenSourceProvisions = requestedChanges.frozenProvisions.orElse(listOf())
        val frozenDestinationProvisions = requestedChanges.frozenDestinationProvisions.orElse(listOf())
        val frozenSourceProvisionsByResourceId = frozenSourceProvisions.associateBy { it.resourceId }
        val frozenDestinationProvisionsByResourceId = frozenDestinationProvisions.associateBy { it.resourceId }
        val knownSourceProvisionsByResourceId = moveProvisionContext
            .accountQuotaByAccountIdResourceId[knownSourceAccount.id] ?: mapOf()
        val knownDestinationProvisionsByResourceId = moveProvisionContext
            .accountQuotaByAccountIdResourceId[knownDestinationAccount.id] ?: mapOf()
        val receivedSourceProvisionByResourceId = sourceProvisions.associateBy { it.resource.id }
        val receivedDestinationProvisionByResourceId = destinationProvisions.associateBy { it.resource.id }
        val updatedSourceFolder = knownSourceFolder.toBuilder()
            .setNextOpLogOrder(knownSourceFolder.nextOpLogOrder + 1L)
            .build()
        val updatedDestinationFolder = knownDestinationFolder.toBuilder()
            .setNextOpLogOrder(knownDestinationFolder.nextOpLogOrder + 1L)
            .build()

        val sameFolderTransfer = knownSourceFolder.id == knownDestinationFolder.id
        val updatedProvisions = mutableListOf<AccountsQuotasModel>()
        val updatedQuotas = mutableListOf<QuotaModel>()
        processUpdatedProvisions(moveProvisionContext, now, operationId, knownSourceFolder.id, knownSourceAccount.id,
            knownProviderId, sameFolderTransfer, provisionsVersionedSeparately, targetSourceProvisions,
            frozenSourceProvisionsByResourceId, knownSourceProvisionsByResourceId, receivedSourceProvisionByResourceId,
            updatedProvisions, updatedQuotas, sourceAccountOpLogBuilder)
        processUpdatedProvisions(moveProvisionContext, now, operationId, knownDestinationFolder.id,
            knownDestinationAccount.id, knownProviderId, sameFolderTransfer, provisionsVersionedSeparately,
            targetDestinationProvisions, frozenDestinationProvisionsByResourceId,
            knownDestinationProvisionsByResourceId, receivedDestinationProvisionByResourceId, updatedProvisions,
            updatedQuotas, destinationAccountOpLogBuilder)
        val sourceOpLog = sourceAccountOpLogBuilder.build()
        val destinationOpLog = destinationAccountOpLogBuilder.build()
        val updatedOperation = AccountsQuotasOperationsModel.Builder(operation.operation)
            .setRequestStatus(AccountsQuotasOperationsModel.RequestStatus.OK)
            .setUpdateDateTime(now)
            .setOrders(OperationOrdersModel.builder(operation.operation.orders)
                .closeOrder(knownSourceFolder.nextOpLogOrder)
                .destinationCloseOrder(knownDestinationFolder.nextOpLogOrder)
                .build())
            .setErrorKind(null)
            .build()
        val opLogIdsByFolderId: Map<String, Set<String>> = if (sameFolderTransfer) {
            mapOf(knownSourceFolder.id to setOf(sourceOpLog.id, destinationOpLog.id))
        } else {
            mapOf(knownSourceFolder.id to setOf(sourceOpLog.id),
                knownDestinationFolder.id to setOf(destinationOpLog.id))
        }
        val transferRequestId = operation.operation.requestedChanges.transferRequestId.orElseThrow()
        val transferRequest = storeService.getTransferRequest(session, transferRequestId).awaitSingle()
            .orElseThrow { IllegalStateException("Transfer request $transferRequestId not found") }
        val borrowedLoanId = transferLoansService.processLoan(session, transferRequest, now,
            knownSourceAccount, knownDestinationAccount, knownSourceFolder, knownDestinationFolder,
            knownDestinationProvisionsByResourceId, receivedDestinationProvisionByResourceId,
            knownSourceProvisionsByResourceId, receivedSourceProvisionByResourceId)
        finishOperationForTransferRequest(session, operation.operation, RetryResult.SUCCESS, opLogIdsByFolderId,
            null, transferRequest, borrowedLoanId)
        storeService.upsertOperation(session, updatedOperation).awaitSingleOrNull()
        storeService.deleteOperationsInProgress(session, operation.inProgress).awaitSingleOrNull()
        storeService.upsertQuotas(session, updatedQuotas).awaitSingleOrNull()
        storeService.upsertFolders(session, listOf(updatedSourceFolder, updatedDestinationFolder)).awaitSingleOrNull()
        storeService.upsertFolderOpLogs(session, listOf(sourceOpLog, destinationOpLog)).awaitSingleOrNull()
        storeService.upsertProvisions(session, updatedProvisions).awaitSingleOrNull()
        val accounts = listOfNotNull(updatedSourceAccount, updatedDestinationAccount)
        if (accounts.isNotEmpty()) {
            storeService.upsertAccounts(session, accounts).awaitSingleOrNull()
        }
        operationsObservabilityService.observeOperationFinished(updatedOperation)
    }

    private suspend fun finishOperationForTransferRequest(
        session: YdbTxSession,
        operation: AccountsQuotasOperationsModel,
        retryResult: RetryResult,
        opLogIdsByFolderId: Map<String, Set<String>>,
        errors: LocalizedErrors?,
        transferRequest: TransferRequestModel,
        borrowedLoanId: LoanId?
    ) {
        val result = MoveProvisionOperationResult(retryResult, opLogIdsByFolderId,
            errors?.errorsRu ?: ErrorCollection.empty(), errors?.errorsEn ?: ErrorCollection.empty())
        val callback = transferRequestCloser.moveProvisionOperationFinished(session,
            transferRequest, operation.operationId, result, borrowedLoanId)
        callback()
    }

    private fun processUpdatedProvisions(
        moveProvisionContext: MoveProvisionContext,
        now: Instant,
        operationId: String,
        knownFolderId: String,
        knownAccountId: String,
        knownProviderId: String,
        isSameFolderTransfer: Boolean,
        provisionsVersionedSeparately: Boolean,
        targetProvisions: List<OperationChangesModel.Provision>,
        frozenProvisionsByResourceId: Map<ResourceId, OperationChangesModel.Provision>,
        knownProvisionsByResourceId: Map<ResourceId, AccountsQuotasModel>,
        receivedProvisionByResourceId: Map<ResourceId, ValidatedReceivedProvision>,
        updatedProvisions: MutableList<AccountsQuotasModel>,
        updatedQuotas: MutableList<QuotaModel>,
        opLogBuilder: FolderOperationLogModel.Builder
    ) {
        val oldProvisionsByResourceId = hashMapOf<String, ProvisionHistoryModel>()
        val newProvisionsByResourceId = hashMapOf<String, ProvisionHistoryModel>()
        val actualProvisionsByResourceId = hashMapOf<String, ProvisionHistoryModel>()
        val oldQuotaByResourceId = hashMapOf<String, Long>()
        val newQuotaByResourceId = hashMapOf<String, Long>()
        val oldBalanceByResourceId = hashMapOf<String, Long>()
        val newBalanceByResourceId = hashMapOf<String, Long>()
        processUpdatedProvisions(moveProvisionContext, now, operationId, knownFolderId, knownAccountId,
            knownProviderId, isSameFolderTransfer, provisionsVersionedSeparately, targetProvisions,
            frozenProvisionsByResourceId, knownProvisionsByResourceId, receivedProvisionByResourceId,
            updatedProvisions, updatedQuotas, oldProvisionsByResourceId, newProvisionsByResourceId,
            actualProvisionsByResourceId, oldQuotaByResourceId, newQuotaByResourceId, oldBalanceByResourceId,
            newBalanceByResourceId)
        if (oldProvisionsByResourceId.isNotEmpty() || newProvisionsByResourceId.isNotEmpty()) {
            opLogBuilder.setOldProvisions(QuotasByAccount(mapOf(knownAccountId to
                ProvisionsByResource(oldProvisionsByResourceId))))
            opLogBuilder.setNewProvisions(QuotasByAccount(mapOf(knownAccountId to
                ProvisionsByResource(newProvisionsByResourceId))))
        } else {
            opLogBuilder.setOldProvisions(QuotasByAccount(mapOf()))
            opLogBuilder.setNewProvisions(QuotasByAccount(mapOf()))
        }
        if (actualProvisionsByResourceId.isNotEmpty()) {
            opLogBuilder.setActuallyAppliedProvisions(QuotasByAccount(mapOf(knownAccountId to
                ProvisionsByResource(actualProvisionsByResourceId))))
        }
        if (oldQuotaByResourceId.isNotEmpty() || newQuotaByResourceId.isNotEmpty()) {
            opLogBuilder.setOldQuotas(QuotasByResource(oldQuotaByResourceId))
            opLogBuilder.setNewQuotas(QuotasByResource(newQuotaByResourceId))
        } else {
            opLogBuilder.setOldQuotas(QuotasByResource(mapOf()))
            opLogBuilder.setNewQuotas(QuotasByResource(mapOf()))
        }
        if (oldBalanceByResourceId.isNotEmpty() || newBalanceByResourceId.isNotEmpty()) {
            opLogBuilder.setOldBalance(QuotasByResource(oldBalanceByResourceId))
            opLogBuilder.setNewBalance(QuotasByResource(newBalanceByResourceId))
        } else {
            opLogBuilder.setOldBalance(QuotasByResource(mapOf()))
            opLogBuilder.setNewBalance(QuotasByResource(mapOf()))
        }
    }

    private fun processUpdatedProvisions(
        moveProvisionContext: MoveProvisionContext,
        now: Instant,
        operationId: String,
        knownFolderId: String,
        knownAccountId: String,
        knownProviderId: String,
        isSameFolderTransfer: Boolean,
        provisionsVersionedSeparately: Boolean,
        targetProvisions: List<OperationChangesModel.Provision>,
        frozenProvisionsByResourceId: Map<String, OperationChangesModel.Provision>,
        knownProvisionsByResourceId: Map<String, AccountsQuotasModel>,
        receivedProvisionByResourceId: Map<String, ValidatedReceivedProvision>,
        updatedProvisions: MutableList<AccountsQuotasModel>,
        updatedQuotas: MutableList<QuotaModel>,
        oldProvisionsByResourceId: MutableMap<String, ProvisionHistoryModel>,
        newProvisionsByResourceId: MutableMap<String, ProvisionHistoryModel>,
        actualProvisionsByResourceId: MutableMap<String, ProvisionHistoryModel>,
        oldQuotaByResourceId: MutableMap<String, Long>,
        newQuotaByResourceId: MutableMap<String, Long>,
        oldBalanceByResourceId: MutableMap<String, Long>,
        newBalanceByResourceId: MutableMap<String, Long>
    ) {
        targetProvisions.forEach { p ->
            val resourceId = p.resourceId
            val knownProvision = knownProvisionsByResourceId[resourceId]
            val knownQuota = moveProvisionContext.folderQuotaByFolderIdResourceId
                .getOrDefault(knownFolderId, mapOf())[resourceId]
            val frozenProvision = frozenProvisionsByResourceId[resourceId]
            val receivedProvision = receivedProvisionByResourceId[resourceId]
            val receivedVersion = receivedProvision?.quotaVersion?.orElse(null)
            val knownVersion = knownProvision?.lastReceivedProvisionVersion?.orElse(null)
            val knownProvidedAmount = knownProvision?.providedQuota ?: 0L
            val targetProvidedAmount = p.amount
            val receivedProvidedAmount = receivedProvision?.providedAmount ?: 0L
            val receivedAllocatedAmount = receivedProvision?.allocatedAmount ?: 0L
            val targetFrozenAmount = frozenProvision?.amount ?: 0L
            val quotaDelta = receivedProvidedAmount - knownProvidedAmount
            processUpdatedProvidedAmounts(now, operationId, knownFolderId, knownAccountId, knownProviderId,
                provisionsVersionedSeparately, updatedProvisions, oldProvisionsByResourceId,
                newProvisionsByResourceId, actualProvisionsByResourceId, resourceId, knownProvision,
                receivedVersion, knownVersion, knownProvidedAmount, targetProvidedAmount, receivedProvidedAmount,
                receivedAllocatedAmount, targetFrozenAmount)
            if (!isSameFolderTransfer) {
                processUpdatedQuota(moveProvisionContext, knownQuota, operationId, resourceId, knownFolderId,
                    knownProviderId, quotaDelta, updatedQuotas, oldQuotaByResourceId, newQuotaByResourceId,
                    oldBalanceByResourceId, newBalanceByResourceId)
            }
        }
    }

    private fun processUpdatedProvidedAmounts(
        now: Instant,
        operationId: String,
        knownFolderId: String,
        knownAccountId: String,
        knownProviderId: String,
        provisionsVersionedSeparately: Boolean,
        updatedProvisions: MutableList<AccountsQuotasModel>,
        oldProvisionsByResourceId: MutableMap<String, ProvisionHistoryModel>,
        newProvisionsByResourceId: MutableMap<String, ProvisionHistoryModel>,
        actualProvisionsByResourceId: MutableMap<String, ProvisionHistoryModel>,
        resourceId: String,
        knownProvision: AccountsQuotasModel?,
        receivedVersion: Long?,
        knownVersion: Long?,
        knownProvidedAmount: Long,
        targetProvidedAmount: Long,
        receivedProvidedAmount: Long,
        receivedAllocatedAmount: Long,
        knownFrozenProvision: Long
    ) {
        if (knownProvision != null) {
            val lastReceivedVersion = if (provisionsVersionedSeparately) receivedVersion else
                knownProvision.lastReceivedProvisionVersion.orElse(null)
            if (knownProvision.frozenProvidedQuota < knownFrozenProvision) {
                logger.error("Frozen provided quota {} for resource {} in account {} is less then value {} stored" +
                    " for operation {}", knownProvision.frozenProvidedQuota, resourceId, knownAccountId,
                    knownFrozenProvision, operationId)
            }
            val newFrozenQuota = max(knownProvision.frozenProvidedQuota - knownFrozenProvision, 0L)
            val updatedProvision = AccountsQuotasModel.Builder(knownProvision)
                .setProvidedQuota(receivedProvidedAmount)
                .setAllocatedQuota(receivedAllocatedAmount)
                .setFrozenProvidedQuota(newFrozenQuota)
                .setLastProvisionUpdate(now)
                .setLatestSuccessfulProvisionOperationId(operationId)
                .setLastReceivedProvisionVersion(lastReceivedVersion)
                .build()
            updatedProvisions.add(updatedProvision)
        } else {
            val newProvision = AccountsQuotasModel.Builder()
                .setTenantId(Tenants.DEFAULT_TENANT_ID)
                .setAccountId(knownAccountId)
                .setResourceId(resourceId)
                .setProvidedQuota(receivedProvidedAmount)
                .setAllocatedQuota(receivedAllocatedAmount)
                .setFolderId(knownFolderId)
                .setProviderId(knownProviderId)
                .setLastProvisionUpdate(now)
                .setLatestSuccessfulProvisionOperationId(operationId)
                .setLastReceivedProvisionVersion(if (provisionsVersionedSeparately) receivedVersion else null)
                .build()
            updatedProvisions.add(newProvision)
        }
        if (receivedProvidedAmount != knownProvidedAmount || !Objects.equal(receivedVersion, knownVersion)) {
            oldProvisionsByResourceId[resourceId] = ProvisionHistoryModel(knownProvidedAmount, knownVersion)
            actualProvisionsByResourceId[resourceId] = ProvisionHistoryModel(receivedProvidedAmount, receivedVersion)
        }
        if (targetProvidedAmount != knownProvidedAmount) {
            oldProvisionsByResourceId[resourceId] = ProvisionHistoryModel(knownProvidedAmount, knownVersion)
            newProvisionsByResourceId[resourceId] = ProvisionHistoryModel(targetProvidedAmount,
                if (targetProvidedAmount == receivedProvidedAmount) receivedVersion else knownVersion)
        }
    }

    private fun processUpdatedQuota(
        moveProvisionContext: MoveProvisionContext,
        knownQuota: QuotaModel?,
        operationId: String,
        resourceId: String,
        knownFolderId: String,
        knownProviderId: String,
        delta: Long,
        updatedQuotas: MutableList<QuotaModel>,
        oldQuotaByResourceId: MutableMap<String, Long>,
        newQuotaByResourceId: MutableMap<String, Long>,
        oldBalanceByResourceId: MutableMap<String, Long>,
        newBalanceByResourceId: MutableMap<String, Long>
    ) {
        val quota = knownQuota?.quota ?: 0L
        val updatedQuota = if (delta < 0) {
            quota.addExactOrNull(delta)?.coerceAtLeast(0L)
        } else {
            val oppositeFolderId = if (knownFolderId == moveProvisionContext.sourceFolder.id) {
                moveProvisionContext.destinationFolder.id
            } else {
                moveProvisionContext.sourceFolder.id
            }
            val knownSourceQuota = moveProvisionContext
                .folderQuotaByFolderIdResourceId[oppositeFolderId]?.get(resourceId)?.quota ?: delta
            quota.addExactOrNull(min(delta, knownSourceQuota))
        }
        val balance = knownQuota?.balance ?: 0L
        val updatedBalance = updatedQuota
            .addExactOrNull(-quota)
            ?.addExactOrNull(-delta)
            ?.addExactOrNull(balance)
        if (updatedQuota == null || updatedBalance == null) {
            logOverflowError(resourceId, knownFolderId, operationId)
            return
        }
        if (updatedQuota != quota) {
            if (knownQuota != null) {
                val updatedQuotaModel = QuotaModel.builder(knownQuota)
                    .balance(updatedBalance)
                    .quota(updatedQuota)
                    .build()
                updatedQuotas.add(updatedQuotaModel)
            } else {
                val newQuota = QuotaModel.builder()
                    .tenantId(Tenants.DEFAULT_TENANT_ID)
                    .folderId(knownFolderId)
                    .providerId(knownProviderId)
                    .resourceId(resourceId)
                    .quota(updatedQuota)
                    .balance(updatedBalance)
                    .frozenQuota(0L)
                    .build()
                updatedQuotas.add(newQuota)
            }
            oldQuotaByResourceId[resourceId] = quota
            newQuotaByResourceId[resourceId] = updatedQuota
            oldBalanceByResourceId[resourceId] = balance
            newBalanceByResourceId[resourceId] = updatedBalance
        }
    }

    private fun Long?.addExactOrNull(addend: Long?): Long? =
        if (this == null || addend == null) {
            null
        } else {
            val sum = BigInteger.valueOf(this).add(BigInteger.valueOf(addend))
            if (sum > BigInteger.valueOf(Long.MAX_VALUE) || sum < BigInteger.valueOf(Long.MIN_VALUE)) {
                null
            } else {
                sum.toLong()
            }
        }

    private fun logOverflowError(resourceId: String, knownFolderId: String, operationId: String) {
        logger.error(
            "Overflow while updating quota of resource {} in folder {} for operation {}",
            resourceId, knownFolderId, operationId
        )
    }

    private fun prepareUpdatedAccountOnRefresh(
        account: AccountModel,
        receivedAccount: ValidatedReceivedAccount,
        accountAndProvisionsVersionedTogether: Boolean,
        opLogBuilder: FolderOperationLogModel.Builder,
        accountsSettings: AccountsSettingsModel
    ): AccountModel? {
        return if (accountAndProvisionsVersionedTogether && receivedAccount.accountVersion.isPresent) {
            val updatedAccountBuilder = AccountModel.Builder(account)
                .setLastReceivedVersion(receivedAccount.accountVersion.get())
                .setVersion(account.version + 1L)
            val oldAccountBuilder = AccountHistoryModel.builder()
                .lastReceivedVersion(account.lastReceivedVersion.orElse(null))
                .version(account.version)
            val newAccountBuilder = AccountHistoryModel.builder()
                .lastReceivedVersion(receivedAccount.accountVersion.get())
                .version(account.version + 1L)
            if (accountsSettings.isDisplayNameSupported && account.displayName != receivedAccount.displayName) {
                updatedAccountBuilder.setDisplayName(receivedAccount.displayName.orElse(null))
                oldAccountBuilder.displayName(account.displayName.orElse(null))
                newAccountBuilder.displayName(receivedAccount.displayName.orElse(null))
            }
            if (accountsSettings.isKeySupported && account.outerAccountKeyInProvider != receivedAccount.key) {
                updatedAccountBuilder.setOuterAccountKeyInProvider(receivedAccount.key.orElse(null))
                oldAccountBuilder.outerAccountKeyInProvider(account.outerAccountKeyInProvider.orElse(null))
                newAccountBuilder.outerAccountKeyInProvider(receivedAccount.key.orElse(null))
            }
            opLogBuilder.setOldAccounts(AccountsHistoryModel(mapOf(account.id to oldAccountBuilder.build())))
            opLogBuilder.setNewAccounts(AccountsHistoryModel(mapOf(account.id to newAccountBuilder.build())))
            updatedAccountBuilder.build()
        } else {
            null
        }
    }

    private fun prepareUpdatedAccountOnRetry(
        account: AccountModel,
        receivedAccountVersion: Long?,
        accountAndProvisionsVersionedTogether: Boolean,
        opLogBuilder: FolderOperationLogModel.Builder
    ): AccountModel? {
        return if (accountAndProvisionsVersionedTogether && receivedAccountVersion != null) {
            val updatedAccountBuilder = AccountModel.Builder(account)
                .setLastReceivedVersion(receivedAccountVersion)
                .setVersion(account.version + 1L)
            val oldAccountBuilder = AccountHistoryModel.builder()
                .lastReceivedVersion(account.lastReceivedVersion.orElse(null))
                .version(account.version)
            val newAccountBuilder = AccountHistoryModel.builder()
                .lastReceivedVersion(receivedAccountVersion)
                .version(account.version + 1L)
            opLogBuilder.setOldAccounts(AccountsHistoryModel(mapOf(account.id to oldAccountBuilder.build())))
            opLogBuilder.setNewAccounts(AccountsHistoryModel(mapOf(account.id to newAccountBuilder.build())))
            updatedAccountBuilder.build()
        } else {
            null
        }
    }

    private fun prepareFolderOpLogBuilder(
        opLogId: String,
        operation: AccountsQuotasOperationsModel,
        knownFolder: FolderModel,
        now: Instant,
        sourceFolderOpLogId: String? = null,
        destinationFolderOpLogId: String? = null,
        sourceFolder: FolderModel,
        sourceAccount: AccountModel,
        destinationFolder: FolderModel,
        destinationAccount: AccountModel,
    ): FolderOperationLogModel.Builder {
        val anotherFolder: FolderModel
        val anotherAccount: AccountModel
        val roleInTransfer: RoleInTransfer
        if (sourceFolder.id == knownFolder.id) {
            roleInTransfer = RoleInTransfer.SOURCE
            anotherFolder = destinationFolder
            anotherAccount = destinationAccount
        } else {
            roleInTransfer = RoleInTransfer.DESTINATION
            anotherFolder = sourceFolder
            anotherAccount = sourceAccount
        }
        return FolderOperationLogModel.builder()
            .setTenantId(operation.tenantId)
            .setFolderId(knownFolder.id)
            .setOperationDateTime(now)
            .setId(opLogId)
            .setProviderRequestId(operation.lastRequestId.orElseThrow())
            .setOperationType(FolderOperationType.PROVISION_TRANSFER)
            .setAuthorUserId(operation.authorUserId)
            .setAuthorUserUid(operation.authorUserUid.orElse(null))
            .setAuthorProviderId(null)
            .setSourceFolderOperationsLogId(sourceFolderOpLogId)
            .setDestinationFolderOperationsLogId(destinationFolderOpLogId)
            .setOldFolderFields(null)
            .setAccountsQuotasOperationsId(operation.operationId)
            .setQuotasDemandsId(null)
            .setOperationPhase(OperationPhase.CLOSE)
            .setOrder(knownFolder.nextOpLogOrder)
            .setOldBalance(QuotasByResource(mapOf()))
            .setNewBalance(QuotasByResource(mapOf()))
            .setTransferMeta(TransferMetaHistoryModel(
                transferRequestId = operation.requestedChanges.transferRequestId.orElseThrow(),
                roleInTransfer = roleInTransfer,
                anotherParticipants = setOf(TransferMetaHistoryModel.Another(
                    anotherFolder.serviceId,
                    anotherFolder.id,
                    anotherAccount.id
                ))
            ))
    }

    private suspend fun planToNextRetryOperation(
        session: YdbTxSession,
        operation: RetryableOperation,
        retryContext: MoveProvisionOperationRetryContext,
        retryResult: RetryResult,
        now: Instant,
        errorKind: OperationErrorKind
    ) {
        val maxAsyncRetriesReached = operation.inProgress.any { it.retryCounter >= maxAsyncRetries }
        if (maxAsyncRetriesReached) {
            abortOperation(session, operation, retryContext, retryResult, now, errorKind)
        } else {
            operationsObservabilityService.observeOperationTransientFailure(operation.operation)
            storeService.incrementRetryCounter(session, operation.inProgress).awaitSingleOrNull()
        }
    }

    private suspend fun abortOperation(
        session: YdbTxSession,
        operation: RetryableOperation,
        retryContext: MoveProvisionOperationRetryContext,
        retryResult: RetryResult,
        now: Instant,
        errorKind: OperationErrorKind
    ) {
        if (isOperationNotComplete(operation)) {
            val moveProvisionContext = storeService.loadMoveProvisionContext(session, operation.operation)
                .awaitSingleOrNull()!!
            val errors = (retryContext.retryErrors ?: retryContext.postRetryErrors) ?: retryContext.refreshErrors!!
            saveAbortOperation(session, moveProvisionContext, operation, retryResult, now, errors, errorKind)
        }
    }

    private suspend fun saveAbortOperation(
        session: YdbTxSession,
        moveProvisionContext: MoveProvisionContext,
        operation: RetryableOperation,
        retryResult: RetryResult,
        now: Instant,
        errors: LocalizedErrors,
        errorKind: OperationErrorKind
    ) {
        val accountsOp = operation.operation
        val updatedOperation = AccountsQuotasOperationsModel.Builder(accountsOp)
            .setRequestStatus(AccountsQuotasOperationsModel.RequestStatus.ERROR)
            .setUpdateDateTime(now)
            .setErrorMessage(Errors.flattenErrors(errors.errorsEn))
            .setErrorKind(errorKind)
            .setOrders(OperationOrdersModel.builder(operation.operation.orders)
                .closeOrder(moveProvisionContext.sourceFolder.nextOpLogOrder)
                .destinationCloseOrder(moveProvisionContext.destinationFolder.nextOpLogOrder)
                .build())
            .build()
        finishUnsuccessfulOperation(session, moveProvisionContext, updatedOperation, operation.inProgress, retryResult,
            errors)
    }

    private suspend fun finishUnsuccessfulOperation(
        session: YdbTxSession,
        moveProvisionContext: MoveProvisionContext,
        updatedOperation: AccountsQuotasOperationsModel,
        operationsInProgress: List<OperationInProgressModel>,
        retryResult: RetryResult,
        errors: LocalizedErrors,
    ) {
        val requestedChanges = updatedOperation.requestedChanges
        val quotaByAccount = moveProvisionContext
                .accountQuotaByAccountIdResourceId
        val unfreezedSourceQuotas = unfreezeAccountQuotas(
                quotaByAccount[requestedChanges.accountId.orElseThrow()]?.values ?: listOf(),
                requestedChanges.frozenProvisions.orElse(listOf()), updatedOperation.operationId)
        val unfreezedDestinationQuotas = unfreezeAccountQuotas(
                quotaByAccount[requestedChanges.destinationAccountId.orElseThrow()]?.values ?: listOf(),
                requestedChanges.frozenDestinationProvisions.orElse(listOf()), updatedOperation.operationId)
        val folderOpLogs = unsuccessfulFolderOperationLogs(moveProvisionContext, updatedOperation)
        val opLogIdsByFolderId = folderOpLogs.groupBy { it.folderId }
                .mapValues { e -> e.value.map { it.id }.toSet() }
        val transferRequestId = updatedOperation.requestedChanges.transferRequestId.orElseThrow()
        val transferRequest = storeService.getTransferRequest(session, transferRequestId).awaitSingle()
            .orElseThrow { IllegalStateException("Transfer request $transferRequestId not found") }
        finishOperationForTransferRequest(session, updatedOperation, retryResult, opLogIdsByFolderId, errors,
            transferRequest, null)
        storeService.upsertAccountQuotas(session, unfreezedSourceQuotas + unfreezedDestinationQuotas)
                .awaitSingleOrNull()
        storeService.finishOperation(session, operationsInProgress, updatedOperation).awaitSingleOrNull()
        storeService.upsertFolderOpLogs(session, folderOpLogs).awaitSingleOrNull()
        storeService.upsertFolders(session, listOf(moveProvisionContext.sourceFolder.incrementOpLogOrder(),
                moveProvisionContext.destinationFolder.incrementOpLogOrder())).awaitSingleOrNull()
        operationsObservabilityService.observeOperationFinished(updatedOperation)
    }

    private fun FolderModel.incrementOpLogOrder(): FolderModel {
        return this.toBuilder()
            .setNextOpLogOrder(this.nextOpLogOrder + 1L).build()
    }

    private fun unsuccessfulFolderOperationLogs(
        moveProvisionContext: MoveProvisionContext,
        operation: AccountsQuotasOperationsModel,
    ): List<FolderOperationLogModel> {
        val sourceDeltaByResource = operation.requestedChanges.updatedProvisions.orElse(listOf())
            .associateBy({ it.resourceId }, {
                val knownQuota = moveProvisionContext
                    .accountQuotaByAccountIdResourceId[moveProvisionContext.sourceAccount.id]
                    ?.get(it.resourceId)
                    ?.providedQuota ?: 0L
                it.amount - knownQuota
            })
        val destinationDeltaByResource = operation.requestedChanges.updatedDestinationProvisions.orElse(listOf())
            .associateBy({ it.resourceId }, {
                val knownQuota = moveProvisionContext
                    .accountQuotaByAccountIdResourceId[moveProvisionContext.destinationAccount.id]
                    ?.get(it.resourceId)
                    ?.providedQuota ?: 0L
                it.amount - knownQuota
            })
        return MoveProvisionLogicServiceImpl.generateFolderOperationLogs(
            sourceAccountId = moveProvisionContext.sourceAccount.id,
            destinationAccountId = moveProvisionContext.destinationAccount.id,
            sourceFolderId = moveProvisionContext.sourceFolder.id,
            destinationFolderId = moveProvisionContext.destinationFolder.id,
            operation = operation,
            folderQuotaByFolderIdResourceId = moveProvisionContext.folderQuotaByFolderIdResourceId,
            accountQuotaByAccountIdResourceId = moveProvisionContext.accountQuotaByAccountIdResourceId,
            sourceDeltaByResourceId = sourceDeltaByResource,
            destinationDeltaByResourceId = destinationDeltaByResource,
            operationPhase = OperationPhase.FAIL,
            sourceFolderOpLogOrder = moveProvisionContext.sourceFolder.nextOpLogOrder,
            destinationFolderOpLogOrder = moveProvisionContext.destinationFolder.nextOpLogOrder,
            updatedAccountQuotas = mutableListOf()
        )
    }

    private fun commonPostRetryRefreshResult(
        resOne: PostRetryRefreshResult,
        resTwo: PostRetryRefreshResult,
    ): PostRetryRefreshResult {
        if (resOne == resTwo) {
            return resOne
        }
        return if (resOne == PostRetryRefreshResult.UNSUPPORTED || resTwo == PostRetryRefreshResult.UNSUPPORTED) {
            PostRetryRefreshResult.UNSUPPORTED
        } else if (resOne == PostRetryRefreshResult.FATAL_FAILURE || resTwo == PostRetryRefreshResult.FATAL_FAILURE) {
            PostRetryRefreshResult.FATAL_FAILURE
        } else if (resOne == PostRetryRefreshResult.NON_FATAL_FAILURE ||
            resTwo == PostRetryRefreshResult.NON_FATAL_FAILURE) {
            PostRetryRefreshResult.NON_FATAL_FAILURE
        } else {
            resOne
        }
    }

    private fun getPostRetryRefreshResult(
        result: Result<Response<ReceivedAccount>>,
        operationId: String,
        accountId: AccountId
    ): PostRetryRefreshResult {
        return result.match({ resp: Response<ReceivedAccount> ->
            resp.match({ acc, requestId -> PostRetryRefreshResult.SUCCESS },
                { e ->
                    logger.error("Failed to refresh account $accountId after provision update for operation $operationId",
                        e)
                    PostRetryRefreshResult.NON_FATAL_FAILURE
                }, { e, requestId ->
                logger.error("Failed to refresh account {} after provision update for operation " +
                    "{} (requestId = {}): {}", accountId, operationId, requestId, e)
                resolvePostRetryRefreshResult(e)
            })
        }
        ) { e: ErrorCollection? ->
            logger.error("Failed to refresh account {} after provision update for operation {}: {}",
                accountId, operationId, e)
            PostRetryRefreshResult.NON_FATAL_FAILURE
        }
    }

    private fun <T> extractValue(
        responseResult: Result<Response<T>>
    ): T? {
        return responseResult.match({ resp ->
            resp.match({ v, _ -> v }, { null }, { _, _ -> null })
        }, { null })
    }

    private fun getRefreshErrorDescription(
        validatedReceivedAccount: Result<Response<ValidatedReceivedAccount>>,
        refreshResult: RefreshResult
    ): LocalizedErrors? {
        if (refreshResult == RefreshResult.OPERATION_APPLIED || refreshResult == RefreshResult.OPERATION_NOT_APPLIED) {
            return null
        }
        return if (refreshResult == RefreshResult.UNDEFINED) {
            localizedErrors("errors.provision.update.check.is.not.possible", ErrorType.INVALID)
        } else validatedReceivedAccount.match({ response ->
            response.match({ account, _ ->
                null
            }, { throwable ->
                localizedErrors("errors.unexpected.provider.communication.failure", ErrorType.INVALID)
            }, { providerError, _ ->
                localizedErrors("errors.provision.update.check.is.not.possible", ErrorType.INVALID,
                    details = Errors.flattenProviderErrorResponse(providerError, null))
            })
        }, { errors ->
            LocalizedErrors(errors, errors)
        })
    }

    private fun getRefreshResult(operation: RetryableOperation,
                                 postRefreshContext: MoveProvisionOperationPostRefreshContext): RefreshResult {
        val operationId = operation.operation.operationId
        val moveProvisionContext = postRefreshContext.preRefreshContext.moveProvisionContext
        val folderById = mapOf(
            moveProvisionContext.sourceFolder.id to moveProvisionContext.sourceFolder,
            moveProvisionContext.destinationFolder.id to moveProvisionContext.destinationFolder
        )
        val requestedChanges = operation.operation.requestedChanges
        val updatedProvisionsByAccountId = mapOf(
            moveProvisionContext.sourceAccount.id to requestedChanges.updatedProvisions.orElse(listOf()),
            moveProvisionContext.destinationAccount.id to requestedChanges.updatedDestinationProvisions.orElse(listOf())
        )
        val refreshResults = sequenceOf(
            moveProvisionContext.sourceAccount to postRefreshContext.validatedAccountFrom,
            moveProvisionContext.destinationAccount to postRefreshContext.validatedAccountTo
        ).map { (account, res) ->
            res.match({ response ->
                response.match({ validatedAccount, _ ->
                    val provisions = updatedProvisionsByAccountId[account.id] ?: listOf()
                    if (isProvisionChangeApplied(validatedAccount, account, folderById[account.folderId]!!, provisions,
                            operation.operation, moveProvisionContext, postRefreshContext.commonContext)) {
                        RefreshResult.OPERATION_APPLIED
                    } else {
                        RefreshResult.OPERATION_NOT_APPLIED
                    }
                }, { throwable ->
                    logger.error("Failed to refresh folder account ${account.id} for operation $operationId retry",
                        throwable)
                    RefreshResult.NON_FATAL_ERROR
                }, { providerError, _ ->
                    resolveRefreshResult(providerError)
                })
            }, { errors ->
                logger.error("Failed to refresh account {} for operation {} retry: {}",
                    account, operationId, errors)
                RefreshResult.NON_FATAL_ERROR
            })
        }.toList()
        val distinct = refreshResults.distinct()
        return if (distinct.size == 1) {
            distinct[0]
        } else {
            if (refreshResults.any { it == RefreshResult.UNSUPPORTED }) {
                RefreshResult.UNSUPPORTED
            } else if (refreshResults.any { it == RefreshResult.FATAL_ERROR }) {
                RefreshResult.FATAL_ERROR
            } else if (refreshResults.any { it == RefreshResult.NON_FATAL_ERROR }) {
                RefreshResult.NON_FATAL_ERROR
            } else if (refreshResults.any { it == RefreshResult.OPERATION_NOT_APPLIED }) {
                RefreshResult.OPERATION_NOT_APPLIED
            } else {
                RefreshResult.UNDEFINED
            }
        }
    }

    private fun getRetryResult(
        resultResponse: Result<Response<ReceivedMoveProvision>>,
        operationId: String
    ): RetryResult {
        return resultResponse.match({ resp ->
            resp.match({ acc, reqId ->
                RetryResult.SUCCESS
            }, {
                logger.error("Failed to retry provision move for operation $operationId", it)
                RetryResult.NON_FATAL_FAILURE
            }, { e, reqId ->
                logger.error("Failed to retry provision move for operation {} (requestId = {}): {}", operationId, reqId, e)
                resolveRetryResult(e)
            })
        }, { e ->
            logger.error("Failed to retry provision move for operation {}: {}", operationId, e)
            RetryResult.NON_FATAL_FAILURE
        })
    }

    private fun getRetryErrors(
        retryResult: RetryResult,
        resultResponse: Result<Response<ReceivedMoveProvision>>,
    ): LocalizedErrors? {
        if (retryResult == RetryResult.SUCCESS) {
            return null
        }
        return resultResponse.match({ resp ->
            resp.match({ acc, reqId ->
                null
            }, {
                localizedErrors("errors.unexpected.provider.communication.failure", ErrorType.INVALID)
            }, { e, reqId ->
                val providerError = Errors.flattenProviderErrorResponse(e, null)
                if (retryResult == RetryResult.CONFLICT) {
                    localizedErrors("errors.accounts.quotas.out.of.sync.with.provider",
                        ErrorType.CONFLICT, details = providerError)
                } else {
                    localizedErrors("errors.unexpected.provider.response",
                        ErrorType.BAD_REQUEST, details = providerError)
                }
            })
        }, { e ->
            LocalizedErrors(e, e)
        })
    }

    private fun getPostRetryRefreshErrors(
        postRetryRefreshResult: PostRetryRefreshResult,
        resultResponse: Result<Response<ReceivedAccount>>,
    ): LocalizedErrors? {
        if (postRetryRefreshResult == PostRetryRefreshResult.SUCCESS) {
            return null
        }
        return resultResponse.match({ resp ->
            resp.match({ acc, reqId ->
                null
            }, {
                localizedErrors("errors.unexpected.provider.communication.failure", ErrorType.INVALID)
            }, { e, reqId ->
                val providerError = Errors.flattenProviderErrorResponse(e, null)
                if (postRetryRefreshResult == PostRetryRefreshResult.UNSUPPORTED) {
                    localizedErrors("errors.provision.update.check.is.not.possible",
                        ErrorType.BAD_REQUEST, details = providerError)
                } else {
                    localizedErrors("errors.unexpected.provider.response",
                        ErrorType.BAD_REQUEST, details = providerError)
                }
            })
        }, { e ->
            LocalizedErrors(e, e)
        })
    }

    private fun localizedErrors(
        errorCode: String,
        errorType: ErrorType,
        details: Any? = null
    ): LocalizedErrors {
        val errorRu = ErrorCollection.builder()
            .addError(TypedError.typedError(messages.getMessage(errorCode, null, errorCode, Locales.RUSSIAN),
                errorType))
        val errorEn = ErrorCollection.builder()
            .addError(TypedError.typedError(messages.getMessage(errorCode, null, errorCode, Locales.ENGLISH),
                errorType))
        if (details != null) {
            errorRu.addDetail(Details.ERROR_FROM_PROVIDER, details)
            errorEn.addDetail(Details.ERROR_FROM_PROVIDER, details)
        }
        return LocalizedErrors(errorRu.build(), errorEn.build())
    }

    private fun isProvisionChangeApplied(
        receivedAccount: ValidatedReceivedAccount,
        account: AccountModel,
        folder: FolderModel,
        updatedProvisions: List<OperationChangesModel.Provision>,
        operation: AccountsQuotasOperationsModel,
        moveProvisionContext: MoveProvisionContext,
        commonContext: OperationCommonContext
    ): Boolean {
        val accountsSettings = commonContext.provider.accountsSettings
        val provisionOperationIdSupported = accountsSettings.isPerProvisionLastUpdateSupported
        val accountAndProvisionsVersionedTogether = accountsSettings.isPerAccountVersionSupported
            && !accountsSettings.isPerProvisionVersionSupported
        val provisionsVersionedSeparately = accountsSettings.isPerProvisionVersionSupported
        val hasProvisionOperationIds = receivedAccount.provisions.all { p ->
            p.lastUpdate.isPresent && p.lastUpdate.get().operationId.isPresent
        }
        val hasCommonVersion = receivedAccount.accountVersion.isPresent
        val hasProvisionVersions = receivedAccount.provisions.all { p -> p.quotaVersion.isPresent }
        if (provisionOperationIdSupported && hasProvisionOperationIds) {
            return hasOperationId(receivedAccount, account, folder, updatedProvisions,
                operation.operationId, moveProvisionContext, accountAndProvisionsVersionedTogether,
                provisionsVersionedSeparately)
        }
        if (accountAndProvisionsVersionedTogether && hasCommonVersion) {
            return isExactValuesMatchAndFreshCommonVersion(receivedAccount, account, folder, moveProvisionContext,
                updatedProvisions)
        }
        return if (provisionsVersionedSeparately && hasProvisionVersions) {
            isExactValuesMatchAndFreshProvisionVersions(receivedAccount, account, folder, moveProvisionContext,
                updatedProvisions)
        } else isExactValuesMatch(receivedAccount, account, folder, updatedProvisions)
    }

    private fun isExactValuesMatch(
        receivedAccount: ValidatedReceivedAccount,
        account: AccountModel,
        folder: FolderModel,
        updatedProvisions: List<OperationChangesModel.Provision>,
    ): Boolean {
        val receivedProvisionsByResourceId = receivedAccount.provisions.associateBy { it.resource.id }
        val valuesMatch = updatedProvisions.all { targetProvision ->
            val targetResourceId = targetProvision.resourceId
            val targetAmount = targetProvision.amount
            val receivedProvision = receivedProvisionsByResourceId[targetResourceId]
            val receivedAmount = receivedProvision?.providedAmount ?: 0L
            return@all targetAmount == receivedAmount
        }
        val knownFolderId = folder.id
        val receivedFolderId = receivedAccount.folder.id
        val foldersMatch = knownFolderId == receivedFolderId
        val knownDeletion = account.isDeleted
        val receivedDeletion = receivedAccount.isDeleted
        val deletionMatch = knownDeletion == receivedDeletion
        return valuesMatch && foldersMatch && deletionMatch
    }

    private fun isExactValuesMatchAndFreshCommonVersion(
        receivedAccount: ValidatedReceivedAccount,
        account: AccountModel,
        folder: FolderModel,
        moveProvisionContext: MoveProvisionContext,
        updatedProvisions: List<OperationChangesModel.Provision>,
    ): Boolean {
        val receivedVersionO = receivedAccount.accountVersion
        val knownVersionO = account.lastReceivedVersion
        if (receivedVersionO.isPresent && knownVersionO.isPresent && receivedVersionO.get() <= knownVersionO.get()) {
            return false
        }
        return if (isMismatchForRemainingResources(receivedAccount, account, updatedProvisions, moveProvisionContext)) {
            false
        } else isExactValuesMatch(receivedAccount, account, folder, updatedProvisions)
    }

    private fun isExactValuesMatchAndFreshProvisionVersions(
        receivedAccount: ValidatedReceivedAccount,
        account: AccountModel,
        folder: FolderModel,
        moveProvisionContext: MoveProvisionContext,
        updatedProvisions: List<OperationChangesModel.Provision>,
    ): Boolean {
        val accountId = account.id
        val knownProvisions = moveProvisionContext.accountQuotaByAccountIdResourceId[accountId] ?: mapOf()
        val receivedVersionsAreFresh = receivedAccount.provisions.all { receivedProvision ->
            val receivedVersion = receivedProvision.quotaVersion.orElse(null)
            val knownVersion = knownProvisions[receivedProvision.resource.id]?.lastReceivedProvisionVersion
                ?.orElse(null)
            if (receivedVersion != null && knownVersion != null && receivedVersion <= knownVersion) {
                return@all false
            }
            return@all true
        }
        return if (!receivedVersionsAreFresh) {
            false
        } else isExactValuesMatch(receivedAccount, account, folder, updatedProvisions)
    }

    private fun hasOperationId(
        receivedAccount: ValidatedReceivedAccount,
        account: AccountModel,
        folder: FolderModel,
        updatedProvisions: List<OperationChangesModel.Provision>,
        operationId: String,
        moveProvisionContext: MoveProvisionContext,
        accountAndProvisionsVersionedTogether: Boolean,
        provisionsVersionedSeparately: Boolean
    ): Boolean {
        if (accountAndProvisionsVersionedTogether) {
            val knownVersionO = account.lastReceivedVersion
            val receivedVersionO = receivedAccount.accountVersion
            if (knownVersionO.isPresent && receivedVersionO.isPresent) {
                if (knownVersionO.get() >= receivedVersionO.get()) {
                    return false
                }
            }
            if (isMismatchForRemainingResources(receivedAccount, account, updatedProvisions, moveProvisionContext)) {
                return false
            }
        }
        val targetProvisionByResourceId = updatedProvisions.associateBy { it.resourceId }
        val targetResourceIds = targetProvisionByResourceId.keys
        val receivedProvisionsByResourceId = receivedAccount.provisions.associateBy { it.resource.id }
        if (provisionsVersionedSeparately) {
            val versionMatch = targetResourceIds.all { resourceId ->
                val knownProvision = moveProvisionContext.accountQuotaByAccountIdResourceId
                    .getOrDefault(account.id, mapOf())[resourceId]
                val receivedProvision = receivedProvisionsByResourceId[resourceId]
                val knownVersion = knownProvision?.lastReceivedProvisionVersion?.orElse(null)
                val receivedVersion = receivedProvision?.quotaVersion?.orElse(null)
                return@all if (knownVersion != null && receivedVersion != null) {
                    knownVersion < receivedVersion
                } else true
            }
            if (!versionMatch) {
                return false
            }
        }
        val knownFolderId = folder.id
        val receivedFolderId = receivedAccount.folder.id
        val foldersMatch = knownFolderId == receivedFolderId
        val knownDeletion = account.isDeleted
        val receivedDeletion = receivedAccount.isDeleted
        val deletionMatch = knownDeletion == receivedDeletion
        val opIdMatch = targetResourceIds.all { resourceId ->
            val receivedProvision = receivedProvisionsByResourceId[resourceId]
            val targetProvision = targetProvisionByResourceId[resourceId]
            if (receivedProvision == null) {
                return@all targetProvision!!.amount == 0L
            }
            return@all receivedProvision.lastUpdate
                .flatMap { it.operationId }
                .map { it == operationId }
                .orElse(false)
        }
        return opIdMatch && foldersMatch && deletionMatch
    }

    private fun isMismatchForRemainingResources(receivedAccount: ValidatedReceivedAccount,
                                                account: AccountModel,
                                                updatedProvisions: List<OperationChangesModel.Provision>,
                                                moveProvisionContext: MoveProvisionContext): Boolean {
        val receivedProvisionsByResourceId = receivedAccount.provisions.associateBy { it.resource.id }
        val targetResourceIds = updatedProvisions
            .map { it.resourceId }
            .toSet()
        val receivedResourceIds = receivedProvisionsByResourceId.keys
        val knowResourceIds = moveProvisionContext.accountQuotaByAccountIdResourceId.getOrDefault(account.id, mapOf())
            .keys
        val remainingResourceIds = (knowResourceIds + receivedResourceIds) - targetResourceIds
        return remainingResourceIds.any { resourceId ->
            val receivedProvision = receivedProvisionsByResourceId[resourceId]
            val knownProvision = moveProvisionContext.accountQuotaByAccountIdResourceId
                .getOrDefault(account.id, mapOf())[resourceId]
            val receivedAmount = receivedProvision?.providedAmount ?: 0L
            val knownAmount = knownProvision?.providedQuota ?: 0L
            return@any receivedAmount != knownAmount
        }
    }

    private suspend fun validateAccount(
        receivedAccountFrom: Result<Response<ReceivedAccount>>,
        session: YdbTxSession,
        commonContext: OperationCommonContext,
        locale: Locale
    ): Result<Response<ValidatedReceivedAccount>> {
        return receivedAccountFrom.map { acc: ReceivedAccount, requestId: String? ->
            validationService.validateReceivedAccount(session, acc, commonContext, locale)
                .map { it.apply { account -> Response.success(account, requestId) } }
                .awaitSingle()
        }
    }

    private fun unfreezeAccountQuotas(
        currentAccountQuotas: Collection<AccountsQuotasModel>,
        frozenProvisions: Collection<OperationChangesModel.Provision>,
        operationId: String,
    ): List<AccountsQuotasModel> {
        val updatedQuotas = mutableListOf<AccountsQuotasModel>()
        val frozenByResource = frozenProvisions.associateBy { it.resourceId }
        currentAccountQuotas.forEach { quota ->
            if (quota.resourceId !in frozenByResource) {
                return@forEach
            }
            val frozen = frozenByResource[quota.resourceId]!!
            if (frozen.amount == 0L) {
                return@forEach
            }
            if (frozen.amount <= quota.frozenProvidedQuota) {
                val updatedQuota = AccountsQuotasModel.Builder(quota)
                    .setFrozenProvidedQuota(quota.frozenProvidedQuota - frozen.amount)
                    .build()
                updatedQuotas.add(updatedQuota)
            } else {
                logger.error("Frozen provided quota {} for resource {} in account {} is less then value {} stored" +
                    " for operation {}", quota.frozenProvidedQuota, quota.resourceId, quota.accountId, frozen.amount,
                    operationId)
            }
        }
        return updatedQuotas
    }

    private fun prepareOperationMDC(
        operation: AccountsQuotasOperationsModel
    ): Map<String, String?> {
        return prepareOperationMDC(operation.operationId, operation.requestedChanges.transferRequestId.orElse(null))
    }

    private fun prepareOperationMDC(
        operationId: String,
        transferRequestId: String?
    ): Map<String, String?> {
        return mapOf(MdcKey.COMMON_TRANSFER_REQUEST_ID to transferRequestId, MdcKey.COMMON_OPERATION_ID to operationId)
    }
}
