package ru.yandex.intranet.d.services.operations

import io.grpc.Status
import org.springframework.http.HttpStatus
import ru.yandex.intranet.d.model.accounts.AccountsQuotasOperationsModel
import ru.yandex.intranet.d.services.integration.providers.ProviderError
import ru.yandex.intranet.d.services.integration.providers.rest.model.ErrorMessagesDto
import ru.yandex.intranet.d.services.operations.model.PostRetryRefreshResult
import ru.yandex.intranet.d.services.operations.model.RefreshResult
import ru.yandex.intranet.d.services.operations.model.RetryResult
import ru.yandex.intranet.d.services.operations.model.RetryableOperation

object OperationUtils {
    @JvmStatic
    fun isAnotherRoundAfterRefresh(refreshResult: RefreshResult): Boolean {
        return refreshResult == RefreshResult.NON_FATAL_ERROR
    }

    @JvmStatic
    fun isOperationAppliedAfterRefresh(refreshResult: RefreshResult): Boolean {
        return refreshResult == RefreshResult.OPERATION_APPLIED
    }

    @JvmStatic
    fun isRefreshNotImplemented(statusCode: Int): Boolean {
        return statusCode == HttpStatus.NOT_IMPLEMENTED.value()
            || statusCode == HttpStatus.METHOD_NOT_ALLOWED.value()
    }

    @JvmStatic
    fun isRefreshNotImplemented(statusCode: Status.Code): Boolean {
        return statusCode == Status.Code.UNIMPLEMENTED
    }

    @JvmStatic
    fun isRefreshFatalError(statusCode: Int): Boolean {
        return statusCode == HttpStatus.BAD_REQUEST.value()
            || statusCode == HttpStatus.UNPROCESSABLE_ENTITY.value() || statusCode == HttpStatus.NOT_FOUND.value()
    }

    @JvmStatic
    fun isRefreshFatalError(statusCode: Status.Code): Boolean {
        return statusCode == Status.Code.DATA_LOSS || statusCode == Status.Code.INVALID_ARGUMENT
            || statusCode == Status.Code.NOT_FOUND || statusCode == Status.Code.OUT_OF_RANGE
            || statusCode == Status.Code.UNKNOWN
    }

    @JvmStatic
    fun isRetryConflict(statusCode: Status.Code): Boolean {
        return (statusCode == Status.Code.ALREADY_EXISTS
            || statusCode == Status.Code.FAILED_PRECONDITION)
    }

    @JvmStatic
    fun isRetryConflict(statusCode: Int): Boolean {
        return statusCode == HttpStatus.CONFLICT.value()
            || statusCode == HttpStatus.PRECONDITION_FAILED.value()
    }

    @JvmStatic
    fun isRetryFatalError(statusCode: Int): Boolean {
        return statusCode == HttpStatus.BAD_REQUEST.value() || statusCode == HttpStatus.UNPROCESSABLE_ENTITY.value()
            || statusCode == HttpStatus.NOT_FOUND.value() || statusCode == HttpStatus.NOT_IMPLEMENTED.value()
            || statusCode == HttpStatus.METHOD_NOT_ALLOWED.value()
    }

    @JvmStatic
    fun isRetryFatalError(statusCode: Status.Code): Boolean {
        return statusCode == Status.Code.DATA_LOSS || statusCode == Status.Code.INVALID_ARGUMENT
            || statusCode == Status.Code.NOT_FOUND || statusCode == Status.Code.OUT_OF_RANGE
            || statusCode == Status.Code.UNIMPLEMENTED || statusCode == Status.Code.UNKNOWN
    }

    @JvmStatic
    fun isOperationNotComplete(operation: RetryableOperation): Boolean {
        return isOperationNotComplete(operation.operation)
    }

    @JvmStatic
    fun isOperationNotComplete(operation: AccountsQuotasOperationsModel): Boolean {
        return operation.requestStatus.isEmpty ||
            operation.requestStatus.get() == AccountsQuotasOperationsModel.RequestStatus.WAITING
    }

    @JvmStatic
    fun resolveRefreshResult(providerError: ProviderError): RefreshResult {
        return providerError.match(object : ProviderError.Cases<RefreshResult> {
            override fun httpError(statusCode: Int): RefreshResult {
                if (isRefreshNotImplemented(statusCode)) {
                    return RefreshResult.UNSUPPORTED
                }
                return if (isRefreshFatalError(statusCode)) {
                    RefreshResult.FATAL_ERROR
                } else RefreshResult.NON_FATAL_ERROR
            }

            override fun httpExtendedError(statusCode: Int, errors: ErrorMessagesDto?): RefreshResult {
                if (isRefreshNotImplemented(statusCode)) {
                    return RefreshResult.UNSUPPORTED
                }
                return if (isRefreshFatalError(statusCode)) {
                    RefreshResult.FATAL_ERROR
                } else RefreshResult.NON_FATAL_ERROR
            }

            override fun grpcError(statusCode: Status.Code, message: String?): RefreshResult {
                if (isRefreshNotImplemented(statusCode)) {
                    return RefreshResult.UNSUPPORTED
                }
                return if (isRefreshFatalError(statusCode)) {
                    RefreshResult.FATAL_ERROR
                } else RefreshResult.NON_FATAL_ERROR
            }

            override fun grpcExtendedError(statusCode: Status.Code, message: String?,
                                           badRequestDetails: Map<String, String>?): RefreshResult {
                if (isRefreshNotImplemented(statusCode)) {
                    return RefreshResult.UNSUPPORTED
                }
                return if (isRefreshFatalError(statusCode)) {
                    RefreshResult.FATAL_ERROR
                } else RefreshResult.NON_FATAL_ERROR
            }
        })
    }

    @JvmStatic
    fun resolveRetryResult(providerError: ProviderError): RetryResult {
        return providerError.match(object : ProviderError.Cases<RetryResult> {
            override fun httpError(statusCode: Int): RetryResult {
                if (isRetryConflict(statusCode)) {
                    return RetryResult.CONFLICT
                }
                return if (isRetryFatalError(statusCode)) {
                    RetryResult.FATAL_FAILURE
                } else RetryResult.NON_FATAL_FAILURE
            }

            override fun httpExtendedError(statusCode: Int, errors: ErrorMessagesDto?): RetryResult {
                if (isRetryConflict(statusCode)) {
                    return RetryResult.CONFLICT
                }
                return if (isRetryFatalError(statusCode)) {
                    RetryResult.FATAL_FAILURE
                } else RetryResult.NON_FATAL_FAILURE
            }

            override fun grpcError(statusCode: Status.Code, message: String?): RetryResult {
                if (isRetryConflict(statusCode)) {
                    return RetryResult.CONFLICT
                }
                return if (isRetryFatalError(statusCode)) {
                    RetryResult.FATAL_FAILURE
                } else RetryResult.NON_FATAL_FAILURE
            }

            override fun grpcExtendedError(statusCode: Status.Code, message: String?,
                                           badRequestDetails: Map<String, String>?): RetryResult {
                if (isRetryConflict(statusCode)) {
                    return RetryResult.CONFLICT
                }
                return if (isRetryFatalError(statusCode)) {
                    RetryResult.FATAL_FAILURE
                } else RetryResult.NON_FATAL_FAILURE
            }
        })
    }

    @JvmStatic
    fun resolvePostRetryRefreshResult(providerError: ProviderError): PostRetryRefreshResult {
        return providerError.match(object : ProviderError.Cases<PostRetryRefreshResult> {
            override fun httpError(statusCode: Int): PostRetryRefreshResult {
                if (isRefreshNotImplemented(statusCode)) {
                    return PostRetryRefreshResult.UNSUPPORTED
                }
                return if (isRefreshFatalError(statusCode)) {
                    PostRetryRefreshResult.FATAL_FAILURE
                } else PostRetryRefreshResult.NON_FATAL_FAILURE
            }

            override fun httpExtendedError(statusCode: Int, errors: ErrorMessagesDto?): PostRetryRefreshResult {
                if (isRefreshNotImplemented(statusCode)) {
                    return PostRetryRefreshResult.UNSUPPORTED
                }
                return if (isRefreshFatalError(statusCode)) {
                    PostRetryRefreshResult.FATAL_FAILURE
                } else PostRetryRefreshResult.NON_FATAL_FAILURE
            }

            override fun grpcError(statusCode: Status.Code, message: String?): PostRetryRefreshResult {
                if (isRefreshNotImplemented(statusCode)) {
                    return PostRetryRefreshResult.UNSUPPORTED
                }
                return if (isRefreshFatalError(statusCode)) {
                    PostRetryRefreshResult.FATAL_FAILURE
                } else PostRetryRefreshResult.NON_FATAL_FAILURE
            }

            override fun grpcExtendedError(statusCode: Status.Code, message: String?,
                                           badRequestDetails: Map<String, String>?): PostRetryRefreshResult {
                if (isRefreshNotImplemented(statusCode)) {
                    return PostRetryRefreshResult.UNSUPPORTED
                }
                return if (isRefreshFatalError(statusCode)) {
                    PostRetryRefreshResult.FATAL_FAILURE
                } else PostRetryRefreshResult.NON_FATAL_FAILURE
            }
        })
    }
}
