package ru.yandex.intranet.d.services.operations

import kotlinx.coroutines.CancellationException
import kotlinx.coroutines.reactor.awaitSingle
import mu.KotlinLogging
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.beans.factory.annotation.Value
import org.springframework.context.MessageSource
import org.springframework.stereotype.Component
import ru.yandex.intranet.d.dao.Tenants
import ru.yandex.intranet.d.i18n.Locales
import ru.yandex.intranet.d.loaders.providers.ProvidersLoader
import ru.yandex.intranet.d.model.accounts.AccountsQuotasOperationsModel
import ru.yandex.intranet.d.model.accounts.OperationErrorKind
import ru.yandex.intranet.d.model.providers.ProviderModel
import ru.yandex.intranet.d.services.integration.jns.JnsClient
import ru.yandex.intranet.d.services.integration.jns.JnsMessage
import ru.yandex.intranet.d.services.integration.jns.JnsResult
import ru.yandex.intranet.d.services.notifications.NotificationMailGenerator
import ru.yandex.intranet.d.util.dispatchers.CustomDispatcher
import ru.yandex.monlib.metrics.histogram.Histograms
import ru.yandex.monlib.metrics.labels.Labels
import ru.yandex.monlib.metrics.primitives.Histogram
import ru.yandex.monlib.metrics.primitives.Rate
import ru.yandex.monlib.metrics.registry.MetricRegistry
import java.time.Duration
import java.util.*
import java.util.concurrent.ConcurrentHashMap

private val logger = KotlinLogging.logger {}

/**
 * Operations observability service.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
class OperationsObservabilityService(private val providersLoader: ProvidersLoader,
                                     private val mailGenerator: NotificationMailGenerator,
                                     private val jnsClient: JnsClient,
                                     @Qualifier("observabilityDispatcher")
                                     private val observabilityDispatcher: CustomDispatcher,
                                     @Value("\${notifications.jns.targets.failedOperations.templateType}")
                                     private val jnsTemplateType: String,
                                     @Value("\${notifications.jns.project}")
                                     private val jnsProject: String,
                                     @Value("\${notifications.jns.targets.failedOperations.targetProject}")
                                     private val jnsTargetProject: String,
                                     @Value("\${notifications.jns.targets.failedOperations.channel}")
                                     private val jnsChannel: String,
                                     @Value("\${notifications.jns.targets.failedOperations.template}")
                                     private val jnsTemplate: String,
                                     @Qualifier("emailMessageSource") private val mailMessages: MessageSource,
) {

    private val providerTypedSubmitRate: ConcurrentHashMap<ProviderTypedOperationKey, Rate> = ConcurrentHashMap()
    private val providerTypedCompletionRate: ConcurrentHashMap<ProviderTypedOperationKey, Rate> = ConcurrentHashMap()
    private val providerTypedSuccessRate: ConcurrentHashMap<ProviderTypedOperationKey, Rate> = ConcurrentHashMap()
    private val providerTypedFailureRate: ConcurrentHashMap<ProviderTypedOperationKey, Rate> = ConcurrentHashMap()
    private val providerTypedFailureTypeRate: ConcurrentHashMap<ProviderTypedOperationFailureKey, Rate> = ConcurrentHashMap()
    private val providerTypedTransientFailureRate: ConcurrentHashMap<ProviderTypedOperationKey, Rate> = ConcurrentHashMap()
    private val providerTypedDuration: ConcurrentHashMap<ProviderTypedOperationKey, Histogram> = ConcurrentHashMap()
    private val providerSubmitRate: ConcurrentHashMap<ProviderOperationKey, Rate> = ConcurrentHashMap()
    private val providerCompletionRate: ConcurrentHashMap<ProviderOperationKey, Rate> = ConcurrentHashMap()
    private val providerSuccessRate: ConcurrentHashMap<ProviderOperationKey, Rate> = ConcurrentHashMap()
    private val providerFailureRate: ConcurrentHashMap<ProviderOperationKey, Rate> = ConcurrentHashMap()
    private val providerFailureTypeRate: ConcurrentHashMap<ProviderOperationFailureKey, Rate> = ConcurrentHashMap()
    private val providerTransientFailureRate: ConcurrentHashMap<ProviderOperationKey, Rate> = ConcurrentHashMap()
    private val providerDuration: ConcurrentHashMap<ProviderOperationKey, Histogram> = ConcurrentHashMap()
    private val typedSubmitRate: ConcurrentHashMap<TypedOperationKey, Rate> = ConcurrentHashMap()
    private val typedCompletionRate: ConcurrentHashMap<TypedOperationKey, Rate> = ConcurrentHashMap()
    private val typedSuccessRate: ConcurrentHashMap<TypedOperationKey, Rate> = ConcurrentHashMap()
    private val typedFailureRate: ConcurrentHashMap<TypedOperationKey, Rate> = ConcurrentHashMap()
    private val typedFailureTypeRate: ConcurrentHashMap<TypedOperationFailureKey, Rate> = ConcurrentHashMap()
    private val typedTransientFailureRate: ConcurrentHashMap<TypedOperationKey, Rate> = ConcurrentHashMap()
    private val typedDuration: ConcurrentHashMap<TypedOperationKey, Histogram> = ConcurrentHashMap()
    private val submitRate: Rate = MetricRegistry.root().rate("operations.rate",
        Labels.of("operation_phase", "submit", "provider", "any", "operation_type", "any"))
    private val completionRate: Rate = MetricRegistry.root().rate("operations.rate",
        Labels.of("operation_phase", "completion", "operation_result", "any",
            "provider", "any", "operation_type", "any"))
    private val successRate: Rate = MetricRegistry.root().rate("operations.rate",
        Labels.of("operation_phase", "completion", "operation_result", "success",
            "provider", "any", "operation_type", "any"))
    private val failureRate: Rate = MetricRegistry.root().rate("operations.rate",
        Labels.of("operation_phase", "completion", "operation_result", "failure",
            "provider", "any", "operation_type", "any", "failure_type", "any"))
    private val failureTypeRate: ConcurrentHashMap<OperationFailureKey, Rate> = ConcurrentHashMap()
    private val transientFailureRate: Rate = MetricRegistry.root().rate("operations.rate",
        Labels.of("operation_phase", "transient_failure", "provider", "any", "operation_type", "any"))
    private val duration: Histogram = MetricRegistry.root().histogramRate("operations.duration_millis",
        Labels.of("provider", "any", "operation_type", "any"), Histograms.exponential(22, 2.0, 1.0))

    fun observeOperationSubmitted(operation: AccountsQuotasOperationsModel) {
        launch {
            registerOperationSubmitted(operation)
            tryResolveProvider(operation)?.let { registerOperationSubmitted(operation, it) }
        }
    }

    fun observeOperationFinished(operation: AccountsQuotasOperationsModel) {
        launch {
            registerOperationFinished(operation)
            tryResolveProvider(operation)?.let { registerOperationFinished(operation, it) }
            if (operation.requestStatus.isPresent
                && operation.requestStatus.get() == AccountsQuotasOperationsModel.RequestStatus.ERROR) {
                notifyAboutFailure(operation)
            }
        }
    }

    fun observeOperationTransientFailure(operation: AccountsQuotasOperationsModel) {
        launch {
            registerOperationTransientFailure(operation)
            tryResolveProvider(operation)?.let { registerOperationTransientFailure(operation, it) }
        }
    }

    private fun launch(block: suspend () -> Unit) {
        try {
            observabilityDispatcher.launch {
                try {
                    block()
                } catch (e: Exception) {
                    logger.error(e) { "Error during operation metrics processing" }
                    if (e is CancellationException) {
                        throw e
                    }
                }
            }
        } catch (e: Exception) {
            logger.error(e) { "Error during operation metrics processing" }
        }
    }

    private suspend fun tryResolveProvider(operation: AccountsQuotasOperationsModel): ProviderModel? {
        return try {
            providersLoader.getProviderByIdImmediate(operation.providerId, Tenants.DEFAULT_TENANT_ID)
                .awaitSingle().orElse(null)
        } catch (e: Exception) {
            logger.error(e) { "Error during operation metrics processing" }
            if (e is CancellationException) {
                throw e
            }
            null
        }
    }

    private suspend fun notifyAboutFailure(operation: AccountsQuotasOperationsModel) {
        val message = formatMessage(operation, Locales.ENGLISH)
        val title = mailMessages.getMessage("notification.mail.operation.failure.subject", null, Locales.ENGLISH)
        val jnsMessage = JnsMessage(
            project = jnsProject,
            template = jnsTemplate,
            targetProject = jnsTargetProject,
            channel = jnsChannel,
            parameters = prepareTemplateParameters(title, message)
        )
        when (val result = jnsClient.send(jnsMessage)) {
            is JnsResult.Success -> {}
            is JnsResult.Failure -> {
                logger.error(result.error) { "Failed to send notification to JNS" }
            }
            is JnsResult.Error -> {
                logger.error { "Failed to send notification to JNS: $result" }
            }
        }
    }

    private fun prepareTemplateParameters(title: String, message: String): Map<String, String> {
        return when (jnsTemplateType) {
            "html" -> {
                mapOf("title" to title, "message" to message)
            }
            "text" -> {
                mapOf("message" to message)
            }
            else -> {
                mapOf("message" to message)
            }
        }
    }

    private fun formatMessage(operation: AccountsQuotasOperationsModel, locale: Locale): String {
        val transferId = operation.requestedChanges.transferRequestId.orElse(null)
        return when (jnsTemplateType) {
            "html" -> {
                mailGenerator.generateHtmlFailedOperationNotification(operation.operationId, transferId,
                    operation.errorKind.orElse(null), locale)
            }
            "text" -> {
                mailGenerator.generateTextFailedOperationNotification(operation.operationId, transferId,
                    operation.errorKind.orElse(null), locale)
            }
            else -> {
                mailGenerator.generateTextFailedOperationNotification(operation.operationId, transferId,
                    operation.errorKind.orElse(null), locale)
            }
        }
    }

    private fun registerOperationSubmitted(operation: AccountsQuotasOperationsModel) {
        submitRate.inc()
        typedSubmitRate.computeIfAbsent(TypedOperationKey(operation.operationType)) { k -> MetricRegistry
            .root().rate("operations.rate", Labels.of("operation_phase", "submit",
                "provider", "any", "operation_type", k.operationType.name)) }.inc()
    }

    private fun registerOperationFinished(operation: AccountsQuotasOperationsModel) {
        if (operation.requestStatus.isEmpty
            || operation.requestStatus.get() == AccountsQuotasOperationsModel.RequestStatus.WAITING) {
            logger.warn { "Operation ${operation.operationId} was incomplete while observed as finished" }
            return
        }
        val operationDuration = if (operation.updateDateTime.isPresent) {
            Duration.between(operation.createDateTime, operation.updateDateTime.get())
        } else {
            null
        }
        if (operationDuration != null && !operationDuration.isNegative && !operationDuration.isZero) {
            duration.record(operationDuration.toMillis())
            typedDuration.computeIfAbsent(TypedOperationKey(operation.operationType)) { k -> MetricRegistry.root()
                .histogramRate("operations.duration_millis", Labels.of("provider", "any",
                    "operation_type", k.operationType.name), Histograms.exponential(22, 2.0, 1.0)) }
                .record(operationDuration.toMillis())
        }
        completionRate.inc()
        typedCompletionRate.computeIfAbsent(TypedOperationKey(operation.operationType)) { k -> MetricRegistry.root()
            .rate("operations.rate", Labels.of("operation_phase", "completion", "operation_result", "any",
                "provider", "any", "operation_type", k.operationType.name)) }.inc()
        if (operation.requestStatus.get() == AccountsQuotasOperationsModel.RequestStatus.OK) {
            successRate.inc()
            typedSuccessRate.computeIfAbsent(TypedOperationKey(operation.operationType)) { k -> MetricRegistry.root()
                .rate("operations.rate", Labels.of("operation_phase", "completion", "operation_result", "success",
                    "provider", "any", "operation_type", k.operationType.name)) }.inc()
        } else {
            failureRate.inc()
            typedFailureRate.computeIfAbsent(TypedOperationKey(operation.operationType)) { k -> MetricRegistry.root()
                .rate("operations.rate", Labels.of("operation_phase", "completion", "operation_result", "failure",
                    "provider", "any", "operation_type", k.operationType.name, "failure_type", "any")) }.inc()
            if (operation.errorKind.isPresent) {
                failureTypeRate.computeIfAbsent(OperationFailureKey(operation.errorKind.get())) { k -> MetricRegistry.root()
                    .rate("operations.rate", Labels.of("operation_phase", "completion", "operation_result", "failure",
                        "provider", "any", "operation_type", "any", "failure_type", k.errorKind.name)) }.inc()
                typedFailureTypeRate.computeIfAbsent(TypedOperationFailureKey(operation.operationType, operation.errorKind.get()))
                    { k -> MetricRegistry.root().rate("operations.rate", Labels.of("operation_phase", "completion",
                        "operation_result", "failure", "provider", "any", "operation_type", k.operationType.name,
                        "failure_type", k.errorKind.name)) }.inc()
            }
        }
    }

    private fun registerOperationTransientFailure(operation: AccountsQuotasOperationsModel) {
        transientFailureRate.inc()
        typedTransientFailureRate.computeIfAbsent(TypedOperationKey(operation.operationType))
            { k -> MetricRegistry.root().rate("operations.rate", Labels.of("operation_phase", "transient_failure",
                "provider", "any", "operation_type", k.operationType.name)) }.inc()
    }

    private fun registerOperationSubmitted(operation: AccountsQuotasOperationsModel, provider: ProviderModel) {
        providerTypedSubmitRate.computeIfAbsent(ProviderTypedOperationKey(operation.operationType, provider.key))
            { k -> MetricRegistry.root().rate("operations.rate", Labels.of("operation_phase", "submit",
                "provider", k.providerKey, "operation_type", k.operationType.name)) }.inc()
        providerSubmitRate.computeIfAbsent(ProviderOperationKey(provider.key)) { k -> MetricRegistry
            .root().rate("operations.rate", Labels.of("operation_phase", "submit",
                "provider", k.providerKey, "operation_type", "any")) }.inc()
    }

    private fun registerOperationFinished(operation: AccountsQuotasOperationsModel, provider: ProviderModel) {
        if (operation.requestStatus.isEmpty
            || operation.requestStatus.get() == AccountsQuotasOperationsModel.RequestStatus.WAITING) {
            logger.warn { "Operation ${operation.operationId} was incomplete while observed as finished" }
            return
        }
        val operationDuration = if (operation.updateDateTime.isPresent) {
            Duration.between(operation.createDateTime, operation.updateDateTime.get())
        } else {
            null
        }
        if (operationDuration != null && !operationDuration.isNegative && !operationDuration.isZero) {
            providerTypedDuration.computeIfAbsent(ProviderTypedOperationKey(operation.operationType, provider.key))
                { k -> MetricRegistry.root().histogramRate("operations.duration_millis",
                    Labels.of("provider", k.providerKey, "operation_type", k.operationType.name),
                    Histograms.exponential(22, 2.0, 1.0)) }.record(operationDuration.toMillis())
            providerDuration.computeIfAbsent(ProviderOperationKey(provider.key))
            { k -> MetricRegistry.root().histogramRate("operations.duration_millis",
                Labels.of("provider", k.providerKey, "operation_type", "any"),
                Histograms.exponential(22, 2.0, 1.0)) }.record(operationDuration.toMillis())
        }
        providerTypedCompletionRate.computeIfAbsent(ProviderTypedOperationKey(operation.operationType, provider.key))
            { k -> MetricRegistry.root().rate("operations.rate", Labels.of("operation_phase", "completion",
                "operation_result", "any", "provider", k.providerKey, "operation_type", k.operationType.name)) }.inc()
        providerCompletionRate.computeIfAbsent(ProviderOperationKey(provider.key))
            { k -> MetricRegistry.root().rate("operations.rate", Labels.of("operation_phase", "completion",
                "operation_result", "any", "provider", k.providerKey, "operation_type", "any")) }.inc()
        if (operation.requestStatus.get() == AccountsQuotasOperationsModel.RequestStatus.OK) {
            providerTypedSuccessRate.computeIfAbsent(ProviderTypedOperationKey(operation.operationType, provider.key))
                { k -> MetricRegistry.root().rate("operations.rate", Labels.of("operation_phase", "completion",
                    "operation_result", "success", "provider", k.providerKey,
                    "operation_type", k.operationType.name)) }.inc()
            providerSuccessRate.computeIfAbsent(ProviderOperationKey(provider.key))
                { k -> MetricRegistry.root().rate("operations.rate", Labels.of("operation_phase", "completion",
                    "operation_result", "success", "provider", k.providerKey, "operation_type", "any")) }.inc()
        } else {
            providerTypedFailureRate.computeIfAbsent(ProviderTypedOperationKey(operation.operationType, provider.key))
                { k -> MetricRegistry.root().rate("operations.rate", Labels.of("operation_phase", "completion",
                    "operation_result", "failure", "provider", k.providerKey, "operation_type", k.operationType.name,
                    "failure_type", "any")) }.inc()
            providerFailureRate.computeIfAbsent(ProviderOperationKey(provider.key))
                { k -> MetricRegistry.root().rate("operations.rate", Labels.of("operation_phase", "completion",
                    "operation_result", "failure", "provider", k.providerKey, "operation_type", "any",
                    "failure_type", "any")) }.inc()
            if (operation.errorKind.isPresent) {
                providerTypedFailureTypeRate.computeIfAbsent(ProviderTypedOperationFailureKey(operation.operationType,
                    operation.errorKind.get(), provider.key)) { k -> MetricRegistry.root()
                    .rate("operations.rate", Labels.of("operation_phase", "completion", "operation_result", "failure",
                        "provider", k.providerKey, "operation_type", k.operationType.name,
                        "failure_type", k.errorKind.name)) }.inc()
                providerFailureTypeRate.computeIfAbsent(ProviderOperationFailureKey(operation.errorKind.get(), provider.key))
                    { k -> MetricRegistry.root().rate("operations.rate", Labels.of("operation_phase", "completion",
                        "operation_result", "failure", "provider", k.providerKey, "operation_type", "any",
                        "failure_type", k.errorKind.name)) }.inc()
            }
        }
    }

    private fun registerOperationTransientFailure(operation: AccountsQuotasOperationsModel, provider: ProviderModel) {
        providerTypedTransientFailureRate.computeIfAbsent(ProviderTypedOperationKey(operation.operationType, provider.key))
            { k -> MetricRegistry.root().rate("operations.rate", Labels.of("operation_phase", "transient_failure",
                "provider", k.providerKey, "operation_type", k.operationType.name)) }.inc()
        providerTransientFailureRate.computeIfAbsent(ProviderOperationKey(provider.key))
            { k -> MetricRegistry.root().rate("operations.rate", Labels.of("operation_phase", "transient_failure",
                "provider", k.providerKey, "operation_type", "any")) }.inc()
    }

    private data class TypedOperationKey(
        val operationType: AccountsQuotasOperationsModel.OperationType,
    )

    private data class TypedOperationFailureKey(
        val operationType: AccountsQuotasOperationsModel.OperationType,
        val errorKind: OperationErrorKind,
    )

    private data class ProviderTypedOperationKey(
        val operationType: AccountsQuotasOperationsModel.OperationType,
        val providerKey: String
    )

    private data class ProviderTypedOperationFailureKey(
        val operationType: AccountsQuotasOperationsModel.OperationType,
        val errorKind: OperationErrorKind,
        val providerKey: String
    )

    private data class ProviderOperationKey(
        val providerKey: String
    )

    private data class ProviderOperationFailureKey(
        val errorKind: OperationErrorKind,
        val providerKey: String
    )

    private data class OperationFailureKey(
        val errorKind: OperationErrorKind,
    )

}
