package ru.yandex.intranet.d.services.operations

import com.google.gson.Gson
import mu.KotlinLogging
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.stereotype.Component
import reactor.core.publisher.Mono
import ru.yandex.intranet.d.dao.accounts.AccountsQuotasOperationsRequestLogDao
import ru.yandex.intranet.d.datasource.Ydb
import ru.yandex.intranet.d.datasource.dbSessionRetryable
import ru.yandex.intranet.d.datasource.model.YdbTableClient
import ru.yandex.intranet.d.kotlin.mono
import ru.yandex.intranet.d.model.TenantId
import ru.yandex.intranet.d.model.accounts.AccountsQuotasOperationsRequestLogModel
import ru.yandex.intranet.d.util.dispatchers.CustomDispatcher
import java.time.Instant
import java.time.temporal.ChronoUnit

private val logger = KotlinLogging.logger {}

// Кол-во дней, по истечению которых, логи считаются устаревшими
private const val NUMBER_OF_DAYS_UNTIL_EXPIRY = 7L

/**
 *
 * @author Mustakayev Marat <mmarat248@yandex-team.ru>
 */
@Component
class OperationsRequestLogService (
    @Qualifier("operationRequestLogDispatcher")
    private val operationRequestLogDispatcher: CustomDispatcher,
    private val tableClient: YdbTableClient,
    private val accountsQuotasOperationRequestLogDao: AccountsQuotasOperationsRequestLogDao,
    private val gson: Gson
) {

    fun getAllByOperationIdMono(
        operationId: String, tenantId: TenantId): Mono<List<AccountsQuotasOperationsRequestLogModel>> {
        return mono {
            getAllByOperationId(operationId, tenantId)
        }
    }

    fun register(tenantId: TenantId, operationId: String, requestId: String, request: Any, response: Any) {
        val now = Instant.now()
        try {
            val operationLog = AccountsQuotasOperationsRequestLogModel(
                tenantId, operationId, requestId, now,
                gson.toJson(request),
                gson.toJson(response),
                now.plus(NUMBER_OF_DAYS_UNTIL_EXPIRY, ChronoUnit.DAYS)
            )
            operationRequestLogDispatcher.launch {
                create(operationLog)
            }
        } catch (e: Exception) {
            logger.error(e) { "Error during operation request logs processing" }
        }
    }

    fun registerError(tenantId: TenantId, operationId: String, requestId: String, request: Any, err: Throwable) {
        register(tenantId, operationId, requestId, request, Error(
            err.javaClass.name, err.message, err.stackTraceToString())
        )
    }

    private suspend fun getAllByOperationId(
        operationId: String, tenantId: TenantId): List<AccountsQuotasOperationsRequestLogModel> {
        return dbSessionRetryable(tableClient) {
            accountsQuotasOperationRequestLogDao.getAllByOperationId(
                roStaleSingleRetryableCommit(), operationId, tenantId, Ydb.MAX_RESPONSE_ROWS
            )
        }.orEmpty()
    }

    private suspend fun create(operationLog: AccountsQuotasOperationsRequestLogModel) {
        dbSessionRetryable(tableClient) {
            accountsQuotasOperationRequestLogDao.upsertOneRetryable(
                rwSingleRetryableCommit(), operationLog
            )
        }
    }

    private data class Error (
        val name: String? = "",
        val message: String? = "",
        val stacktrace: String? = ""
    )

}
