package ru.yandex.intranet.d.services.operations;

import java.math.BigDecimal;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Consumer;
import java.util.stream.Collectors;

import com.google.common.collect.Sets;
import com.google.common.collect.Streams;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuple2;
import reactor.util.function.Tuple3;
import reactor.util.function.Tuples;

import ru.yandex.intranet.d.model.accounts.AccountModel;
import ru.yandex.intranet.d.model.accounts.AccountSpaceModel;
import ru.yandex.intranet.d.model.accounts.AccountsQuotasModel;
import ru.yandex.intranet.d.model.accounts.AccountsQuotasOperationsModel;
import ru.yandex.intranet.d.model.accounts.OperationChangesModel;
import ru.yandex.intranet.d.model.folders.FolderModel;
import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.model.units.UnitModel;
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel;
import ru.yandex.intranet.d.services.integration.providers.ProvidersIntegrationService;
import ru.yandex.intranet.d.services.integration.providers.Response;
import ru.yandex.intranet.d.services.integration.providers.rest.model.AccountDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.AccountsSpaceKeyRequestDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.AccountsSpaceKeyResponseDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.CreateAccountRequestDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.GetAccountRequestDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.KnownAccountProvisionsDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.KnownProvisionDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.LastUpdateDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.ListAccountsByFolderRequestDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.MoveProvisionRequestDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.MoveProvisionResponseDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.ProvisionDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.ProvisionRequestDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.ResourceKeyRequestDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.ResourceKeyResponseDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.SegmentKeyRequestDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.SegmentKeyResponseDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.UpdateProvisionRequestDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.UpdateProvisionResponseDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.UserIdDto;
import ru.yandex.intranet.d.services.operations.model.CreateAccountContext;
import ru.yandex.intranet.d.services.operations.model.DeliverUpdateProvisionContext;
import ru.yandex.intranet.d.services.operations.model.ExpandedAccountSpace;
import ru.yandex.intranet.d.services.operations.model.ExpandedResource;
import ru.yandex.intranet.d.services.operations.model.MoveProvisionContext;
import ru.yandex.intranet.d.services.operations.model.MoveProvisionOperationPreRefreshContext;
import ru.yandex.intranet.d.services.operations.model.MoveProvisionOperationRefreshContext;
import ru.yandex.intranet.d.services.operations.model.OperationCommonContext;
import ru.yandex.intranet.d.services.operations.model.ReceivedAccount;
import ru.yandex.intranet.d.services.operations.model.ReceivedAccountsSpaceKey;
import ru.yandex.intranet.d.services.operations.model.ReceivedLastUpdate;
import ru.yandex.intranet.d.services.operations.model.ReceivedMoveProvision;
import ru.yandex.intranet.d.services.operations.model.ReceivedProvision;
import ru.yandex.intranet.d.services.operations.model.ReceivedResourceKey;
import ru.yandex.intranet.d.services.operations.model.ReceivedSegmentKey;
import ru.yandex.intranet.d.services.operations.model.ReceivedUpdatedProvision;
import ru.yandex.intranet.d.services.operations.model.ReceivedUserId;
import ru.yandex.intranet.d.services.operations.model.UpdateProvisionContext;
import ru.yandex.intranet.d.util.Uuids;
import ru.yandex.intranet.d.util.result.ErrorCollection;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.util.result.TypedError;
import ru.yandex.intranet.d.util.units.Units;

/**
 * Operations retry integration service implementation.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class OperationsRetryIntegrationService {

    private static final long REQUEST_PAGE_LIMIT = 1000L;

    private final ProvidersIntegrationService integrationService;

    private final MessageSource messages;

    public OperationsRetryIntegrationService(
            ProvidersIntegrationService integrationService,
            @Qualifier("messageSource") MessageSource messages) {
        this.integrationService = integrationService;
        this.messages = messages;
    }

    public Mono<Result<Response<ReceivedAccount>>> getAccountById(
            FolderModel folder, AccountModel account, OperationCommonContext commonContext, Locale locale
    ) {
        Boolean includeDeleted = commonContext.getProvider().getAccountsSettings().isDeleteSupported()
                && commonContext.getProvider().getAccountsSettings().isSoftDeleteSupported() ? true : null;
        String accountId = account.getOuterAccountIdInProvider();
        Optional<AccountsSpaceKeyRequestDto> accountsSpaceKeyO = createAccountsSpaceKey(commonContext
                .getAccountsSpace().orElse(null), commonContext);
        GetAccountRequestDto request = new GetAccountRequestDto(true, includeDeleted,
                folder.getId(), folder.getServiceId(), accountsSpaceKeyO.orElse(null));
        return integrationService.getAccount(accountId, commonContext.getProvider(), request, locale).map(r ->
                r.andThen(re -> re.match(
                        (a, requestId) -> validateAccount(a, locale).apply(b -> Response.success(b, requestId)),
                        e -> Result.success(Response.failure(e)),
                        (e, requestId) -> Result.success(Response.error(e, requestId)))));
    }

    public Mono<Result<Response<List<ReceivedAccount>>>> listAccountsByFolder(
            CreateAccountContext context, OperationCommonContext commonContext, Locale locale) {
        Boolean includeDeleted = commonContext.getProvider().getAccountsSettings().isDeleteSupported()
                && commonContext.getProvider().getAccountsSettings().isSoftDeleteSupported() ? true : null;
        Optional<AccountsSpaceKeyRequestDto> accountsSpaceKeyO = createAccountsSpaceKey(commonContext
                        .getAccountsSpace().orElse(null), commonContext);
        ListAccountsByFolderRequestDto firstPageRequest = new ListAccountsByFolderRequestDto(REQUEST_PAGE_LIMIT,
                null, true, includeDeleted, context.getFolder().getId(),
                context.getFolder().getServiceId(), accountsSpaceKeyO.orElse(null));
        return integrationService.listAccountsByFolder(commonContext.getProvider(), firstPageRequest, locale)
                .flatMap(firstPageResult -> firstPageResult.andThenMono(firstPageResponse -> firstPageResponse
                                .match((accounts, requestId) -> {
                                    if (accounts.getNextPageToken().isPresent()
                                            && !accounts.getNextPageToken().get().isEmpty()
                                            && accounts.getAccounts().isPresent()
                                            && !accounts.getAccounts().get().isEmpty()) {
                                        String nextPageToken = accounts.getNextPageToken().get();
                                        Result<Response<List<ReceivedAccount>>> folderAccounts = validateAccounts(
                                                accounts.getAccounts().orElse(List.of()), locale)
                                                .apply(r -> Response.success(r, requestId));
                                        return requestNextPage(commonContext.getProvider(),
                                                accountsSpaceKeyO.orElse(null),
                                                context.getFolder().getId(), context.getFolder().getServiceId(),
                                                nextPageToken, locale).expand(rs -> {
                                                    if (!rs.getT2()) {
                                                        return Mono.empty();
                                                    }
                                                    return rs.getT1().match(re -> re.match((t, reqId) -> {
                                                        if (t.getT2().isEmpty() || t.getT2().get().isEmpty()
                                                                || t.getT3()) {
                                                            return Mono.empty();
                                                        } else {
                                                            return requestNextPage(commonContext.getProvider(),
                                                                    accountsSpaceKeyO.orElse(null),
                                                                    context.getFolder().getId(),
                                                                    context.getFolder().getServiceId(),
                                                                    t.getT2().get(), locale);
                                                        }
                                                    },
                                                    e -> Mono.just(Tuples.of(Result.success(Response.failure(e)),
                                                            false)),
                                                    (e, reqId) -> Mono.just(Tuples.of(Result
                                                            .success(Response.error(e, reqId)), false))),
                                                    e -> Mono.just(Tuples.of(Result.failure(e), false)));
                                                }).map(Tuple2::getT1).reduce(folderAccounts, this::reducer);
                                    }
                                    return Mono.just(validateAccounts(accounts.getAccounts().orElse(List.of()),
                                            locale).apply(r -> Response.success(r, requestId)));
                                },
                                e -> Mono.just(Result.success(Response.failure(e))),
                                (e, requestId) -> Mono.just(Result.success(Response.error(e, requestId)))
                                )));
    }

    public Mono<Result<Response<ReceivedAccount>>> createAccount(CreateAccountContext context,
                                                                 OperationCommonContext commonContext,
                                                                 AccountsQuotasOperationsModel operation,
                                                                 Locale locale) {
        String key = operation.getRequestedChanges()
                .getAccountCreateParams().get().getKey().orElse(null);
        String displayName = operation.getRequestedChanges()
                .getAccountCreateParams().get().getDisplayName().orElse(null);
        String folderId = context.getFolder().getId();
        long serviceId = context.getFolder().getServiceId();
        UserIdDto author = new UserIdDto(commonContext.getAuthor().getPassportUid().orElse(null),
                commonContext.getAuthor().getPassportLogin().orElse(null));
        String operationId = operation.getOperationId();
        Optional<AccountsSpaceKeyRequestDto> accountsSpaceKeyO = createAccountsSpaceKey(commonContext
                .getAccountsSpace().orElse(null), commonContext);
        boolean freeTier = operation.getRequestedChanges().getAccountCreateParams().get()
                .getFreeTier().orElse(false);
        CreateAccountRequestDto request = new CreateAccountRequestDto(key, displayName, folderId, serviceId, author,
                operationId, accountsSpaceKeyO.orElse(null), freeTier, context.getAbcServiceSlug());
        return integrationService.createAccount(commonContext.getProvider(), request, locale)
                .map(r ->
                    r.andThen(re -> re.match(
                            (a, requestId) -> validateAccount(a, locale).apply(b -> Response.success(b, requestId)),
                            e -> Result.success(Response.failure(e)),
                            (e, requestId) -> Result.success(Response.error(e, requestId)))));
    }

    public Mono<Result<Response<ReceivedUpdatedProvision>>> updateProvision(UpdateProvisionContext context,
                                                                            OperationCommonContext commonContext,
                                                                            AccountsQuotasOperationsModel operation,
                                                                            Locale locale) {
        String accountId = context.getAccount().getOuterAccountIdInProvider();
        String folderId = context.getFolder().getId();
        long serviceId = context.getFolder().getServiceId();
        UserIdDto author = new UserIdDto(commonContext.getAuthor().getPassportUid().orElse(null),
                commonContext.getAuthor().getPassportLogin().orElse(null));
        String operationId = operation.getOperationId();
        Optional<AccountsSpaceKeyRequestDto> accountsSpaceKeyO = createAccountsSpaceKey(commonContext
                .getAccountsSpace().orElse(null), commonContext);
        List<ProvisionRequestDto> updatedProvisions = prepareUpdatedProvisions(context, commonContext, operation);
        List<KnownAccountProvisionsDto> knownProvisions = prepareKnownProvisions(
                context.getAccount(),
                context.getProvisionsByAccountIdResourceId(),
                context.getAccountsById(),
                commonContext,
                operation
        );
        UpdateProvisionRequestDto request = new UpdateProvisionRequestDto(folderId, serviceId, updatedProvisions,
                knownProvisions, author, operationId, accountsSpaceKeyO.orElse(null));
        return integrationService.updateProvision(accountId, commonContext.getProvider(), request, locale)
                .map(r ->
                    r.andThen(re -> re.match(
                            (a, requestId) -> validateUpdatedProvision(a, locale)
                                    .apply(b -> Response.success(b, requestId)),
                            e -> Result.success(Response.failure(e)),
                            (e, requestId) -> Result.success(Response.error(e, requestId)))));
    }

    public Mono<Result<Response<ReceivedUpdatedProvision>>> deliverUpdateProvision(
            DeliverUpdateProvisionContext context,
            OperationCommonContext commonContext,
            AccountsQuotasOperationsModel operation,
            Locale locale) {
        String accountId = context.getAccount().getOuterAccountIdInProvider();
        String folderId = context.getFolder().getId();
        long serviceId = context.getFolder().getServiceId();
        UserIdDto author = new UserIdDto(commonContext.getAuthor().getPassportUid().orElse(null),
                commonContext.getAuthor().getPassportLogin().orElse(null));
        String operationId = operation.getOperationId();
        Optional<AccountsSpaceKeyRequestDto> accountsSpaceKeyO = createAccountsSpaceKey(commonContext
                .getAccountsSpace().orElse(null), commonContext);
        List<ProvisionRequestDto> updatedProvisions = prepareDeliverUpdatedProvisions(context, commonContext,
                operation);
        List<KnownAccountProvisionsDto> knownProvisions = prepareKnownProvisions(
                context.getAccount(),
                context.getProvisionsByAccountIdResourceId(),
                context.getAccountsById(),
                commonContext,
                operation
        );
        UpdateProvisionRequestDto request = new UpdateProvisionRequestDto(folderId, serviceId, updatedProvisions,
                knownProvisions, author, operationId, accountsSpaceKeyO.orElse(null));
        return integrationService.updateProvision(accountId, commonContext.getProvider(), request, locale).map(r ->
                r.andThen(re -> re.match(
                        (a, requestId) -> validateUpdatedProvision(a, locale)
                                .apply(b -> Response.success(b, requestId)),
                        e -> Result.success(Response.failure(e)),
                        (e, requestId) -> Result.success(Response.error(e, requestId)))));
    }

    public Mono<Result<Response<ReceivedMoveProvision>>> moveProvision(
            MoveProvisionOperationRefreshContext context,
            AccountsQuotasOperationsModel operation,
            Locale locale
    ) {
        OperationCommonContext commonContext = context.getCommonContext();
        UserIdDto author = new UserIdDto(commonContext.getAuthor().getPassportUid().orElse(null),
                commonContext.getAuthor().getPassportLogin().orElse(null));
        String operationId = operation.getOperationId();
        Optional<AccountsSpaceKeyRequestDto> accountsSpaceKeyO = createAccountsSpaceKey(commonContext
                .getAccountsSpace().orElse(null), commonContext);

        MoveProvisionOperationPreRefreshContext preRefreshContext = context.getPreRefreshContext();
        MoveProvisionContext moveProvisionContext = preRefreshContext.getMoveProvisionContext();

        AccountModel sourceAccount = moveProvisionContext.getSourceAccount();
        AccountModel destinationAccount = moveProvisionContext.getDestinationAccount();
        var sourceProvisionsByResourceId = moveProvisionContext.getAccountQuotaByAccountIdResourceId()
                .getOrDefault(sourceAccount.getId(), Collections.emptyMap());
        var destinationProvisionsByResourceId = moveProvisionContext.getAccountQuotaByAccountIdResourceId()
                .getOrDefault(destinationAccount.getId(), Collections.emptyMap());
        var sourceUpdatedProvisions = operation.getRequestedChanges().getUpdatedProvisions().orElse(List.of());
        var destinationUpdatedProvisions =
                operation.getRequestedChanges().getUpdatedDestinationProvisions().orElse(List.of());
        Set<String> targetResourceIds = extractTargetResourceIds(
                sourceProvisionsByResourceId, destinationProvisionsByResourceId,
                sourceUpdatedProvisions, destinationUpdatedProvisions,
                commonContext.getResourceIndex()
        );

        List<KnownAccountProvisionsDto> knownSourceProvisions = prepareKnownProvisionsForMove(
                sourceAccount,
                commonContext,
                sourceProvisionsByResourceId,
                targetResourceIds
        );
        List<KnownAccountProvisionsDto> knownDestinationProvisions = prepareKnownProvisionsForMove(
                destinationAccount,
                commonContext,
                destinationProvisionsByResourceId,
                targetResourceIds
        );
        List<ProvisionRequestDto> updatedSourceProvisions = prepareUpdatedProvisionsForMove(
                commonContext,
                sourceUpdatedProvisions,
                sourceProvisionsByResourceId,
                targetResourceIds
        );
        List<ProvisionRequestDto> updatedDestinationProvisions = prepareUpdatedProvisionsForMove(
                commonContext,
                destinationUpdatedProvisions,
                destinationProvisionsByResourceId,
                targetResourceIds
        );

        MoveProvisionRequestDto request = new MoveProvisionRequestDto(
                moveProvisionContext.getDestinationAccount().getOuterAccountIdInProvider(), // destinationAccountId
                moveProvisionContext.getSourceFolder().getId(), // sourceFolderId
                moveProvisionContext.getDestinationFolder().getId(), // destinationFolderId
                moveProvisionContext.getSourceFolder().getServiceId(), // sourceAbcServiceId
                moveProvisionContext.getDestinationFolder().getServiceId(), // destinationAbcServiceId
                knownSourceProvisions, // knownSourceProvisions
                knownDestinationProvisions, // knownDestinationProvisions
                updatedSourceProvisions, // updatedSourceProvisions
                updatedDestinationProvisions, // updatedDestinationProvisions
                author, // author
                operationId, // operationId
                accountsSpaceKeyO.orElse(null) // accountsSpaceKey
        );
        return integrationService.moveProvision(
                moveProvisionContext.getSourceAccount().getOuterAccountIdInProvider(),
                commonContext.getProvider(),
                request,
                locale
        ).map(r -> r.andThen(re -> re.match(
            (mp, requestId) -> validateMoveProvision(mp, locale).apply(rmp -> Response.success(rmp, requestId)),
            e -> Result.success(Response.failure(e)),
            (e, requestId) -> Result.success(Response.error(e, requestId))
        )));
    }

    private List<ProvisionRequestDto> prepareUpdatedProvisions(UpdateProvisionContext context,
                                                               OperationCommonContext commonContext,
                                                               AccountsQuotasOperationsModel operation) {
        List<ProvisionRequestDto> result = new ArrayList<>();
        Set<String> updatedResourceIds = new HashSet<>();
        operation.getRequestedChanges().getUpdatedProvisions()
                .ifPresent(provisions -> provisions.forEach(provision -> {
                    updatedResourceIds.add(provision.getResourceId());
                    ResourceModel resource = commonContext.getResourceIndex().get(provision.getResourceId())
                            .getResource();
                    UnitsEnsembleModel unitsEnsemble = commonContext.getResourceIndex().get(provision.getResourceId())
                            .getUnitsEnsemble();
                    ResourceKeyRequestDto resourceKey = prepareResourceKey(resource,
                            commonContext.getAccountsSpace().orElse(null), commonContext);
                    Tuple2<BigDecimal, UnitModel> valueAndUnit = Units.convertToApi(provision.getAmount(),
                            resource, unitsEnsemble);
                    result.add(new ProvisionRequestDto(resourceKey, valueAndUnit.getT1().longValue(),
                            valueAndUnit.getT2().getKey()));
                }));

        Map<String, AccountsQuotasModel> accountProvisions = context.getProvisionsByAccountIdResourceId()
                .getOrDefault(context.getAccount().getId(), Collections.emptyMap());
        accountProvisions.forEach((resourceId, provision) -> {
            long providedAmount = provision.getProvidedQuota() != null ? provision.getProvidedQuota() : 0L;
            if (updatedResourceIds.contains(resourceId) || providedAmount == 0L) {
                return;
            }
            ResourceModel resource = commonContext.getResourceIndex().get(resourceId).getResource();
            if (!resource.isManaged() || resource.isReadOnly() || resource.isDeleted()) {
                return;
            }
            UnitsEnsembleModel unitsEnsemble = commonContext.getResourceIndex().get(resourceId).getUnitsEnsemble();
            ResourceKeyRequestDto resourceKey = prepareResourceKey(resource,
                    commonContext.getAccountsSpace().orElse(null), commonContext);
            Tuple2<BigDecimal, UnitModel> valueAndUnit = Units.convertToApi(providedAmount, resource, unitsEnsemble);
            result.add(new ProvisionRequestDto(resourceKey, valueAndUnit.getT1().longValue(),
                    valueAndUnit.getT2().getKey()));
        });
        return result;
    }

    private List<ProvisionRequestDto> prepareDeliverUpdatedProvisions(DeliverUpdateProvisionContext context,
                                                                      OperationCommonContext commonContext,
                                                                      AccountsQuotasOperationsModel operation) {
        List<ProvisionRequestDto> result = new ArrayList<>();
        Set<String> updatedResourceIds = new HashSet<>();
        operation.getRequestedChanges().getUpdatedProvisions()
                .ifPresent(provisions -> provisions.forEach(provision -> {
                    updatedResourceIds.add(provision.getResourceId());
                    ResourceModel resource = commonContext.getResourceIndex().get(provision.getResourceId())
                            .getResource();
                    UnitsEnsembleModel unitsEnsemble = commonContext.getResourceIndex().get(provision.getResourceId())
                            .getUnitsEnsemble();
                    ResourceKeyRequestDto resourceKey = prepareResourceKey(resource,
                            commonContext.getAccountsSpace().orElse(null), commonContext);
                    Tuple2<BigDecimal, UnitModel> valueAndUnit = Units.convertToApi(provision.getAmount(),
                            resource, unitsEnsemble);
                    result.add(new ProvisionRequestDto(resourceKey, valueAndUnit.getT1().longValue(),
                            valueAndUnit.getT2().getKey()));
                }));

        Map<String, AccountsQuotasModel> accountProvisions = context.getProvisionsByAccountIdResourceId()
                .getOrDefault(context.getAccount().getId(), Collections.emptyMap());
        accountProvisions.forEach((resourceId, provision) -> {
            long providedAmount = provision.getProvidedQuota() != null ? provision.getProvidedQuota() : 0L;
            if (updatedResourceIds.contains(resourceId) || providedAmount == 0L) {
                return;
            }
            ResourceModel resource = commonContext.getResourceIndex().get(resourceId).getResource();
            if (!resource.isManaged() || resource.isReadOnly() || resource.isDeleted()) {
                return;
            }
            UnitsEnsembleModel unitsEnsemble = commonContext.getResourceIndex().get(resourceId).getUnitsEnsemble();
            ResourceKeyRequestDto resourceKey = prepareResourceKey(resource,
                    commonContext.getAccountsSpace().orElse(null), commonContext);
            Tuple2<BigDecimal, UnitModel> valueAndUnit = Units.convertToApi(providedAmount, resource, unitsEnsemble);
            result.add(new ProvisionRequestDto(resourceKey, valueAndUnit.getT1().longValue(),
                    valueAndUnit.getT2().getKey()));
        });
        return result;
    }

    private List<ProvisionRequestDto> prepareUpdatedProvisionsForMove(
            OperationCommonContext commonContext,
            List<OperationChangesModel.Provision> updatedProvisions,
            Map<String, AccountsQuotasModel> currentProvisionsByResourceId,
            Set<String> targetResourceIds
    ) {
        List<ProvisionRequestDto> result = new ArrayList<>();
        Set<String> collectedResourceIds = new HashSet<>();
        Set<String> updatedResourceIds = new HashSet<>();
        updatedProvisions.forEach(provision -> {
            updatedResourceIds.add(provision.getResourceId());
            collectedResourceIds.add(provision.getResourceId());
            ResourceModel resource = commonContext.getResourceIndex().get(provision.getResourceId())
                    .getResource();
            UnitsEnsembleModel unitsEnsemble = commonContext.getResourceIndex().get(provision.getResourceId())
                    .getUnitsEnsemble();
            ResourceKeyRequestDto resourceKey = prepareResourceKey(resource,
                    commonContext.getAccountsSpace().orElse(null), commonContext);
            Tuple2<BigDecimal, UnitModel> valueAndUnit = Units.convertToApi(provision.getAmount(),
                    resource, unitsEnsemble);
            result.add(new ProvisionRequestDto(resourceKey, valueAndUnit.getT1().longValue(),
                    valueAndUnit.getT2().getKey()));
        });

        currentProvisionsByResourceId.forEach((resourceId, provision) -> {
            ResourceModel resource = commonContext.getResourceIndex().get(resourceId).getResource();
            long providedAmount = provision.getProvidedQuota() != null ? provision.getProvidedQuota() : 0L;
            if (updatedResourceIds.contains(resourceId) ||  !isSuitable(resource, provision.getProvidedQuota())) {
                return;
            }
            UnitsEnsembleModel unitsEnsemble = commonContext.getResourceIndex().get(resourceId).getUnitsEnsemble();
            ResourceKeyRequestDto resourceKey = prepareResourceKey(resource,
                    commonContext.getAccountsSpace().orElse(null), commonContext);
            Tuple2<BigDecimal, UnitModel> valueAndUnit = Units.convertToApi(providedAmount, resource, unitsEnsemble);
            result.add(new ProvisionRequestDto(resourceKey, valueAndUnit.getT1().longValue(),
                    valueAndUnit.getT2().getKey()));
            collectedResourceIds.add(resourceId);
        });

        Set<String> remainingResourceIds = Sets.difference(targetResourceIds, collectedResourceIds);
        remainingResourceIds.forEach(resourceId -> {
            ResourceModel resource = commonContext.getResourceIndex().get(resourceId).getResource();
            UnitsEnsembleModel unitsEnsemble = commonContext.getResourceIndex().get(resourceId).getUnitsEnsemble();
            ResourceKeyRequestDto resourceKey = prepareResourceKey(resource,
                    commonContext.getAccountsSpace().orElse(null), commonContext);
            Tuple2<BigDecimal, UnitModel> valueAndUnit = Units.convertToApi(0L, resource, unitsEnsemble);
            result.add(new ProvisionRequestDto(resourceKey, valueAndUnit.getT1().longValue(),
                    valueAndUnit.getT2().getKey()));
        });

        return result;
    }

    private List<KnownAccountProvisionsDto> prepareKnownProvisionsForMove(
            AccountModel accountModel,
            OperationCommonContext commonContext,
            Map<String, AccountsQuotasModel> provisionsByResourceId,
            Set<String> targetResourceIds
    ) {
        Set<String> collectedResourceIds = new HashSet<>();
        List<KnownProvisionDto> knownProvisions = new ArrayList<>();
        provisionsByResourceId.forEach((resourceId, provision) -> {
            long providedAmount = provision.getProvidedQuota() != null ? provision.getProvidedQuota() : 0L;
            collectedResourceIds.add(resourceId);
            if (providedAmount == 0 && !targetResourceIds.contains(resourceId)) {
                return;
            }
            ResourceModel resource = commonContext.getResourceIndex().get(resourceId).getResource();
            UnitsEnsembleModel unitsEnsemble = commonContext.getResourceIndex().get(resourceId).getUnitsEnsemble();
            ResourceKeyRequestDto resourceKey = prepareResourceKey(resource,
                    commonContext.getAccountsSpace().orElse(null), commonContext);
            Tuple2<BigDecimal, UnitModel> valueAndUnit = Units.convertToApi(providedAmount,
                    resource, unitsEnsemble);
            knownProvisions.add(new KnownProvisionDto(resourceKey, valueAndUnit.getT1().longValue(),
                    valueAndUnit.getT2().getKey()));
        });
        Set<String> remainingResourceIds = Sets.difference(targetResourceIds, collectedResourceIds);
        remainingResourceIds.forEach(resourceId -> {
            ResourceModel resource = commonContext.getResourceIndex().get(resourceId).getResource();
            UnitsEnsembleModel unitsEnsemble = commonContext.getResourceIndex().get(resourceId).getUnitsEnsemble();
            ResourceKeyRequestDto resourceKey = prepareResourceKey(resource,
                    commonContext.getAccountsSpace().orElse(null), commonContext);
            Tuple2<BigDecimal, UnitModel> valueAndUnit = Units.convertToApi(0L, resource, unitsEnsemble);
            knownProvisions.add(new KnownProvisionDto(resourceKey, valueAndUnit.getT1().longValue(),
                    valueAndUnit.getT2().getKey()));
        });

        return List.of(new KnownAccountProvisionsDto(accountModel.getOuterAccountIdInProvider(), knownProvisions));
    }

    private Set<String> extractTargetResourceIds(
            Map<String, AccountsQuotasModel> sourceProvisionsByResourceId,
            Map<String, AccountsQuotasModel> destinationProvisionsByResourceId,
            List<OperationChangesModel.Provision> sourceUpdatedProvisions,
            List<OperationChangesModel.Provision> destinationUpdatedProvisions,
            Map<String, ExpandedResource> resourceIndex
    ) {
        Set<String> targetResourceIds = new HashSet<>();
        sourceUpdatedProvisions.forEach(provision -> targetResourceIds.add(provision.getResourceId()));
        destinationUpdatedProvisions.forEach(provision -> targetResourceIds.add(provision.getResourceId()));
        sourceProvisionsByResourceId.forEach((resourceId, provision) -> {
            if (!isSuitable(resourceIndex.get(resourceId).getResource(), provision.getProvidedQuota())) {
                return;
            }
            targetResourceIds.add(resourceId);
        });
        destinationProvisionsByResourceId.forEach((resourceId, provision) -> {
            if (!isSuitable(resourceIndex.get(resourceId).getResource(), provision.getProvidedQuota())) {
                return;
            }
            targetResourceIds.add(resourceId);
        });
        return targetResourceIds;
    }

    private boolean isSuitable(ResourceModel resource, Long provision) {
        return resource.isManaged() &&
                !resource.isReadOnly() &&
                !resource.isDeleted() &&
                provision != null &&
                provision != 0L;
    }

    private List<KnownAccountProvisionsDto> prepareKnownProvisions(
            AccountModel accountModel,
            Map<String, Map<String, AccountsQuotasModel>> provisionsByAccountIdResourceId,
            Map<String, AccountModel> accountsById,
            OperationCommonContext commonContext,
            AccountsQuotasOperationsModel operation
    ) {
        Set<String> targetResourceIds = new HashSet<>();
        operation.getRequestedChanges().getUpdatedProvisions().ifPresent(provisions -> provisions
                .forEach(provision -> targetResourceIds.add(provision.getResourceId())));
        Map<String, AccountsQuotasModel> accountProvisions = provisionsByAccountIdResourceId
                .getOrDefault(accountModel.getId(), Collections.emptyMap());
        accountProvisions.forEach((resourceId, provision) -> {
            ResourceModel resource = commonContext.getResourceIndex().get(resourceId).getResource();
            if (!resource.isManaged() || resource.isReadOnly() || resource.isDeleted()) {
                return;
            }
            if ((provision.getProvidedQuota() != null ? provision.getProvidedQuota() : 0L) != 0L) {
                targetResourceIds.add(resourceId);
            }
        });
        List<KnownAccountProvisionsDto> result = new ArrayList<>();
        accountsById.forEach((accountId, account) -> {
            Map<String, AccountsQuotasModel> provisionsByResourceId = provisionsByAccountIdResourceId
                    .getOrDefault(accountId, Collections.emptyMap());
            Set<String> collectedResourceIds = new HashSet<>();
            List<KnownProvisionDto> knownProvisions = new ArrayList<>();
            provisionsByResourceId.forEach((resourceId, provision) -> {
                long providedAmount = provision.getProvidedQuota() != null ? provision.getProvidedQuota() : 0L;
                collectedResourceIds.add(resourceId);
                if (providedAmount == 0 && !targetResourceIds.contains(resourceId)) {
                    return;
                }
                ResourceModel resource = commonContext.getResourceIndex().get(resourceId).getResource();
                UnitsEnsembleModel unitsEnsemble = commonContext.getResourceIndex().get(resourceId).getUnitsEnsemble();
                ResourceKeyRequestDto resourceKey = prepareResourceKey(resource,
                        commonContext.getAccountsSpace().orElse(null), commonContext);
                Tuple2<BigDecimal, UnitModel> valueAndUnit = Units.convertToApi(providedAmount,
                        resource, unitsEnsemble);
                knownProvisions.add(new KnownProvisionDto(resourceKey, valueAndUnit.getT1().longValue(),
                        valueAndUnit.getT2().getKey()));
            });
            Set<String> remainingResourceIds = Sets.difference(targetResourceIds, collectedResourceIds);
            remainingResourceIds.forEach(resourceId -> {
                ResourceModel resource = commonContext.getResourceIndex().get(resourceId).getResource();
                UnitsEnsembleModel unitsEnsemble = commonContext.getResourceIndex().get(resourceId).getUnitsEnsemble();
                ResourceKeyRequestDto resourceKey = prepareResourceKey(resource,
                        commonContext.getAccountsSpace().orElse(null), commonContext);
                Tuple2<BigDecimal, UnitModel> valueAndUnit = Units.convertToApi(0L, resource, unitsEnsemble);
                knownProvisions.add(new KnownProvisionDto(resourceKey, valueAndUnit.getT1().longValue(),
                        valueAndUnit.getT2().getKey()));
            });
            result.add(new KnownAccountProvisionsDto(account.getOuterAccountIdInProvider(), knownProvisions));
        });
        return result;
    }

    private ResourceKeyRequestDto prepareResourceKey(ResourceModel resource,
                                                     AccountSpaceModel accountsSpace,
                                                     OperationCommonContext commonContext) {
        ExpandedResource syncResource = commonContext.getResourceIndex().get(resource.getId());
        String resourceTypeKey = syncResource.getResourceType().getKey();
        Map<String, String> segmentationKeys = new HashMap<>();
        Map<String, String> segmentKeys = new HashMap<>();
        syncResource.getResourceSegments().forEach(s -> {
            segmentationKeys.put(s.getSegmentation().getId(), s.getSegmentation().getKey());
            segmentKeys.put(s.getSegment().getId(), s.getSegment().getKey());
        });
        Set<String> accountSpaceSegmentationIds = new HashSet<>();
        if (accountsSpace != null) {
            accountsSpace.getSegments().forEach(s -> accountSpaceSegmentationIds.add(s.getSegmentationId()));
        }
        List<SegmentKeyRequestDto> segmentation = new ArrayList<>();
        resource.getSegments().forEach(resourceSegment -> {
            if (accountSpaceSegmentationIds.contains(resourceSegment.getSegmentationId())) {
                return;
            }
            String segmentationKey = segmentationKeys.get(resourceSegment.getSegmentationId());
            String segmentKey = segmentKeys.get(resourceSegment.getSegmentId());
            segmentation.add(new SegmentKeyRequestDto(segmentationKey, segmentKey));
        });
        return new ResourceKeyRequestDto(resourceTypeKey, segmentation);
    }

    private Result<Response<List<ReceivedAccount>>> reducer(
            Result<Response<List<ReceivedAccount>>> reduced,
            Result<Response<Tuple3<List<ReceivedAccount>, Optional<String>, Boolean>>> next) {
        return next.andThen(a -> a.match(
                (b, requestId) -> reduced.apply(c -> c.match(
                        (d, reqId) -> Response.success(Streams.concat(b.getT1().stream(), d.stream())
                                .collect(Collectors.toList()), reqId),
                        Response::failure,
                        Response::error)),
                e -> Result.success(Response.failure(e)),
                (e, requestId) -> Result.success(Response.error(e, requestId))));
    }

    private Optional<AccountsSpaceKeyRequestDto> createAccountsSpaceKey(AccountSpaceModel accountsSpace,
                                                                        OperationCommonContext context) {
        if (accountsSpace == null) {
            return Optional.empty();
        }
        ExpandedAccountSpace syncAccountSpace = context.getAccountsSpaceIndex().get(accountsSpace.getId());
        List<SegmentKeyRequestDto> keys = syncAccountSpace.getResourceSegments().stream()
                .map(v -> new SegmentKeyRequestDto(v.getSegmentation().getKey(), v.getSegment().getKey()))
                .collect(Collectors.toList());
        return Optional.of(new AccountsSpaceKeyRequestDto(keys));
    }

    private Mono<Tuple2<Result<Response<Tuple3<List<ReceivedAccount>, Optional<String>, Boolean>>>, Boolean>>
    requestNextPage(
            ProviderModel provider, AccountsSpaceKeyRequestDto accountsSpaceKey,
            String folderId, long serviceId, String nextPageToken, Locale locale) {
        Boolean includeDeleted = provider.getAccountsSettings().isDeleteSupported()
                && provider.getAccountsSettings().isSoftDeleteSupported() ? true : null;
        ListAccountsByFolderRequestDto nextPageRequest = new ListAccountsByFolderRequestDto(REQUEST_PAGE_LIMIT,
                nextPageToken, false, includeDeleted, folderId, serviceId, accountsSpaceKey);
        return integrationService.listAccountsByFolder(provider, nextPageRequest, locale).map(nextPageResult ->
                Tuples.of(nextPageResult.andThen(nextPageResponse -> nextPageResponse.match((accounts, requestId) ->
                                validateAccounts(accounts.getAccounts().orElse(List.of()), locale)
                                        .apply(l -> Response.success(Tuples.of(l, accounts.getNextPageToken(),
                                                accounts.getAccounts().isEmpty()
                                                        || accounts.getAccounts().get().isEmpty()), requestId)),
                        e -> Result.success(Response.failure(e)),
                        (e, requestId) -> Result.success(Response.error(e, requestId)))), true));
    }

    private Result<List<ReceivedAccount>> validateAccounts(List<AccountDto> accounts, Locale locale) {
        List<ReceivedAccount> result = new ArrayList<>();
        ErrorCollection.Builder errorsBuilder = ErrorCollection.builder();
        for (int i = 0; i < accounts.size(); i++) {
            AccountDto account = accounts.get(i);
            ReceivedAccount.Builder accountBuilder = ReceivedAccount.builder();
            ErrorCollection.Builder accountErrorsBuilder = ErrorCollection.builder();
            validateAccount(account, accountBuilder, accountErrorsBuilder, "accounts." + i + ".", locale);
            if (accountErrorsBuilder.hasAnyErrors()) {
                errorsBuilder.add(accountErrorsBuilder);
            } else {
                result.add(accountBuilder.build());
            }
        }
        if (errorsBuilder.hasAnyErrors()) {
            return Result.failure(errorsBuilder.build());
        }
        return Result.success(result);
    }

    private Result<ReceivedMoveProvision> validateMoveProvision(MoveProvisionResponseDto response,
                                                                Locale locale) {
        ReceivedMoveProvision.Builder builder = ReceivedMoveProvision.builder();
        ErrorCollection.Builder errorsBuilder = ErrorCollection.builder();
        response.getSourceProvisions().ifPresent(provisions -> {
            for (int i = 0; i < provisions.size(); i++) {
                ReceivedProvision.Builder provisionBuilder = ReceivedProvision.builder();
                ErrorCollection.Builder provisionErrorsBuilder = ErrorCollection.builder();
                validateProvision(provisions.get(i), provisionBuilder, provisionErrorsBuilder,
                        "sourceProvisions." + i + ".", locale);
                if (provisionErrorsBuilder.hasAnyErrors()) {
                    errorsBuilder.add(provisionErrorsBuilder);
                } else {
                    builder.addSourceProvision(provisionBuilder.build());
                }
            }
        });
        response.getDestinationProvisions().ifPresent(provisions -> {
            for (int i = 0; i < provisions.size(); i++) {
                ReceivedProvision.Builder provisionBuilder = ReceivedProvision.builder();
                ErrorCollection.Builder provisionErrorsBuilder = ErrorCollection.builder();
                validateProvision(provisions.get(i), provisionBuilder, provisionErrorsBuilder,
                        "destinationProvisions." + i + ".", locale);
                if (provisionErrorsBuilder.hasAnyErrors()) {
                    errorsBuilder.add(provisionErrorsBuilder);
                } else {
                    builder.addDestinationProvision(provisionBuilder.build());
                }
            }
        });
        response.getSourceAccountVersion().ifPresent(builder::sourceAccountVersion);
        response.getDestinationAccountVersion().ifPresent(builder::destinationAccountVersion);
        response.getAccountsSpaceKey().ifPresent(accountsSpaceKey -> {
            ReceivedAccountsSpaceKey.Builder accountsSpaceKeyBuilder = ReceivedAccountsSpaceKey.builder();
            ErrorCollection.Builder accountsSpaceKeyErrorsBuilder = ErrorCollection.builder();
            validateAccountsSpaceKey(accountsSpaceKey, accountsSpaceKeyBuilder, accountsSpaceKeyErrorsBuilder,
                    "accountsSpaceKey.", locale);
            if (accountsSpaceKeyErrorsBuilder.hasAnyErrors()) {
                errorsBuilder.add(accountsSpaceKeyErrorsBuilder);
            } else {
                accountsSpaceKey.getSegmentation().ifPresent(s ->
                        builder.accountSpaceKey(accountsSpaceKeyBuilder.build()));
            }
        });
        if (errorsBuilder.hasAnyErrors()) {
            return Result.failure(errorsBuilder.build());
        }
        return Result.success(builder.build());
    }

    private Result<ReceivedUpdatedProvision> validateUpdatedProvision(UpdateProvisionResponseDto response,
                                                                      Locale locale) {
        ReceivedUpdatedProvision.Builder builder = ReceivedUpdatedProvision.builder();
        ErrorCollection.Builder errorsBuilder = ErrorCollection.builder();
        response.getProvisions().ifPresent(provisions -> {
            for (int i = 0; i < provisions.size(); i++) {
                ReceivedProvision.Builder provisionBuilder = ReceivedProvision.builder();
                ErrorCollection.Builder provisionErrorsBuilder = ErrorCollection.builder();
                validateProvision(provisions.get(i), provisionBuilder, provisionErrorsBuilder,
                        "provisions." + i + ".", locale);
                if (provisionErrorsBuilder.hasAnyErrors()) {
                    errorsBuilder.add(provisionErrorsBuilder);
                } else {
                    builder.addProvision(provisionBuilder.build());
                }
            }
        });
        response.getAccountVersion().ifPresent(builder::accountVersion);
        response.getAccountsSpaceKey().ifPresent(accountsSpaceKey -> {
            ReceivedAccountsSpaceKey.Builder accountsSpaceKeyBuilder = ReceivedAccountsSpaceKey.builder();
            ErrorCollection.Builder accountsSpaceKeyErrorsBuilder = ErrorCollection.builder();
            validateAccountsSpaceKey(accountsSpaceKey, accountsSpaceKeyBuilder, accountsSpaceKeyErrorsBuilder,
                    "accountsSpaceKey.", locale);
            if (accountsSpaceKeyErrorsBuilder.hasAnyErrors()) {
                errorsBuilder.add(accountsSpaceKeyErrorsBuilder);
            } else {
                builder.accountsSpaceKey(accountsSpaceKeyBuilder.build());
            }
        });
        if (errorsBuilder.hasAnyErrors()) {
            return Result.failure(errorsBuilder.build());
        }
        return Result.success(builder.build());
    }

    private Result<ReceivedAccount> validateAccount(AccountDto account, Locale locale) {
        ReceivedAccount.Builder accountBuilder = ReceivedAccount.builder();
        ErrorCollection.Builder errorsBuilder = ErrorCollection.builder();
        validateAccount(account, accountBuilder, errorsBuilder, "", locale);
        if (errorsBuilder.hasAnyErrors()) {
            return Result.failure(errorsBuilder.build());
        }
        return Result.success(accountBuilder.build());
    }

    private void validateAccount(AccountDto account, ReceivedAccount.Builder accountBuilder,
                                 ErrorCollection.Builder errorsBuilder, String keyPrefix, Locale locale) {
        if (account.getAccountId().isEmpty() || account.getAccountId().get().isEmpty()) {
            errorsBuilder.addError(keyPrefix + "accountId", TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        } else {
            accountBuilder.accountId(account.getAccountId().get());
        }
        account.getKey().ifPresent(accountBuilder::key);
        account.getDisplayName().ifPresent(accountBuilder::displayName);
        if (account.getFolderId().isEmpty() || account.getFolderId().get().isEmpty()) {
            errorsBuilder.addError(keyPrefix + "folderId", TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        } else {
            if (!Uuids.isValidUuid(account.getFolderId().get())) {
                errorsBuilder.addError(keyPrefix + "folderId", TypedError.invalid(messages
                        .getMessage("errors.folder.not.found", null, locale)));
            } else {
                accountBuilder.folderId(account.getFolderId().get());
            }
        }
        accountBuilder.deleted(account.getDeleted().orElse(false));
        account.getProvisions().ifPresent(provisions -> {
            for (int i = 0; i < provisions.size(); i++) {
                ReceivedProvision.Builder provisionBuilder = ReceivedProvision.builder();
                ErrorCollection.Builder provisionErrorsBuilder = ErrorCollection.builder();
                validateProvision(provisions.get(i), provisionBuilder, provisionErrorsBuilder,
                        keyPrefix + "provisions." + i + ".", locale);
                if (provisionErrorsBuilder.hasAnyErrors()) {
                    errorsBuilder.add(provisionErrorsBuilder);
                } else {
                    accountBuilder.addProvision(provisionBuilder.build());
                }
            }
        });
        account.getAccountVersion().ifPresent(accountBuilder::accountVersion);
        account.getLastUpdate().ifPresent(lastUpdate -> {
            ReceivedLastUpdate.Builder lastUpdateBuilder = ReceivedLastUpdate.builder();
            ErrorCollection.Builder lastUpdateErrorsBuilder = ErrorCollection.builder();
            validateLastUpdate(lastUpdate, lastUpdateBuilder, lastUpdateErrorsBuilder,
                    keyPrefix + "lastUpdate.", locale);
            if (lastUpdateErrorsBuilder.hasAnyErrors()) {
                errorsBuilder.add(lastUpdateErrorsBuilder);
            } else {
                accountBuilder.lastUpdate(lastUpdateBuilder.build());
            }
        });
        account.getAccountsSpaceKey().ifPresent(accountsSpaceKey -> {
            ReceivedAccountsSpaceKey.Builder accountsSpaceKeyBuilder = ReceivedAccountsSpaceKey.builder();
            ErrorCollection.Builder accountsSpaceKeyErrorsBuilder = ErrorCollection.builder();
            validateAccountsSpaceKey(accountsSpaceKey, accountsSpaceKeyBuilder, accountsSpaceKeyErrorsBuilder,
                    keyPrefix + "accountsSpaceKey.", locale);
            if (accountsSpaceKeyErrorsBuilder.hasAnyErrors()) {
                errorsBuilder.add(accountsSpaceKeyErrorsBuilder);
            } else {
                accountBuilder.accountsSpaceKey(accountsSpaceKeyBuilder.build());
            }
        });
    }

    private void validateProvision(ProvisionDto provision, ReceivedProvision.Builder provisionBuilder,
                                   ErrorCollection.Builder errorsBuilder, String keyPrefix, Locale locale) {
        if (provision.getResourceKey().isEmpty()) {
            errorsBuilder.addError(keyPrefix + "resourceKey", TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        } else {
            ReceivedResourceKey.Builder resourceKeyBuilder = ReceivedResourceKey.builder();
            ErrorCollection.Builder resourceKeyErrorsBuilder = ErrorCollection.builder();
            validateResourceKey(provision.getResourceKey().get(), resourceKeyBuilder,
                    resourceKeyErrorsBuilder, keyPrefix + "resourceKey.", locale);
            if (resourceKeyErrorsBuilder.hasAnyErrors()) {
                errorsBuilder.add(resourceKeyErrorsBuilder);
            } else {
                provisionBuilder.resourceKey(resourceKeyBuilder.build());
            }
        }
        if (provision.getProvidedAmount().isEmpty()) {
            errorsBuilder.addError(keyPrefix + "providedAmount", TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        } else {
            provisionBuilder.providedAmount(provision.getProvidedAmount().get());
        }
        if (provision.getAllocatedAmount().isEmpty()) {
            errorsBuilder.addError(keyPrefix + "allocatedAmount", TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        } else {
            provisionBuilder.allocatedAmount(provision.getAllocatedAmount().get());
        }
        if (provision.getProvidedAmountUnitKey().isEmpty()) {
            errorsBuilder.addError(keyPrefix + "providedAmountUnitKey", TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        } else {
            provisionBuilder.providedAmountUnitKey(provision.getProvidedAmountUnitKey().get());
        }
        if (provision.getAllocatedAmountUnitKey().isEmpty()) {
            errorsBuilder.addError(keyPrefix + "allocatedAmountUnitKey", TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        } else {
            provisionBuilder.allocatedAmountUnitKey(provision.getAllocatedAmountUnitKey().get());
        }
        provision.getQuotaVersion().ifPresent(provisionBuilder::quotaVersion);
        provision.getLastUpdate().ifPresent(lastUpdate -> {
            ReceivedLastUpdate.Builder lastUpdateBuilder = ReceivedLastUpdate.builder();
            ErrorCollection.Builder lastUpdateErrorsBuilder = ErrorCollection.builder();
            validateLastUpdate(lastUpdate, lastUpdateBuilder, lastUpdateErrorsBuilder,
                    keyPrefix + "lastUpdate.", locale);
            if (lastUpdateErrorsBuilder.hasAnyErrors()) {
                errorsBuilder.add(lastUpdateErrorsBuilder);
            } else {
                provisionBuilder.lastUpdate(lastUpdateBuilder.build());
            }
        });
    }

    private void validateResourceKey(ResourceKeyResponseDto resourceKey,
                                     ReceivedResourceKey.Builder resourceKeyBuilder,
                                     ErrorCollection.Builder errorsBuilder, String keyPrefix, Locale locale) {
        if (resourceKey.getResourceTypeKey().isEmpty() || resourceKey.getResourceTypeKey().get().isEmpty()) {
            errorsBuilder.addError(keyPrefix + "resourceTypeKey", TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        } else {
            resourceKeyBuilder.resourceTypeKey(resourceKey.getResourceTypeKey().get());
        }
        resourceKey.getSegmentation().ifPresent(segmentation -> {
            for (int i = 0; i < segmentation.size(); i++) {
                SegmentKeyResponseDto segmentKey = segmentation.get(i);
                ReceivedSegmentKey.Builder segmentKeyBuilder = ReceivedSegmentKey.builder();
                ErrorCollection.Builder segmentKeyErrorsBuilder = ErrorCollection.builder();
                validateSegmentKey(segmentKey, segmentKeyBuilder, segmentKeyErrorsBuilder,
                        keyPrefix + "segmentation." + i + ".", locale);
                if (segmentKeyErrorsBuilder.hasAnyErrors()) {
                    errorsBuilder.add(segmentKeyErrorsBuilder);
                } else {
                    resourceKeyBuilder.addSegment(segmentKeyBuilder.build());
                }
            }
        });
    }

    private void validateSegmentKey(SegmentKeyResponseDto segmentKey,
                                    ReceivedSegmentKey.Builder segmentKeyBuilder,
                                    ErrorCollection.Builder errorsBuilder, String keyPrefix, Locale locale) {
        if (segmentKey.getSegmentKey().isEmpty() || segmentKey.getSegmentKey().get().isEmpty()) {
            errorsBuilder.addError(keyPrefix + "segmentKey", TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        } else {
            segmentKeyBuilder.segmentKey(segmentKey.getSegmentKey().get());
        }
        if (segmentKey.getSegmentationKey().isEmpty() || segmentKey.getSegmentationKey().get().isEmpty()) {
            errorsBuilder.addError(keyPrefix + "segmentationKey", TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        } else {
            segmentKeyBuilder.segmentationKey(segmentKey.getSegmentationKey().get());
        }
    }

    private void validateLastUpdate(LastUpdateDto lastUpdate,
                                    ReceivedLastUpdate.Builder lastUpdateBuilder,
                                    ErrorCollection.Builder errorsBuilder, String keyPrefix, Locale locale) {
        lastUpdate.getTimestamp().ifPresent(t -> lastUpdateBuilder.timestamp(Instant.ofEpochMilli(t)));
        validateStringValue(lastUpdate.getOperationId(), lastUpdateBuilder::operationId);
        lastUpdate.getAuthor().ifPresent(userId -> {
            ReceivedUserId.Builder userIdBuilder = ReceivedUserId.builder();
            ErrorCollection.Builder userIdErrorsBuilder = ErrorCollection.builder();
            validateUserId(userId, userIdBuilder, userIdErrorsBuilder, keyPrefix + "author", locale);
            if (userIdErrorsBuilder.hasAnyErrors()) {
                errorsBuilder.add(userIdErrorsBuilder);
            } else {
                lastUpdateBuilder.author(userIdBuilder.build());
            }
        });
    }

    private void validateUserId(UserIdDto userId,
                                ReceivedUserId.Builder userIdBuilder,
                                ErrorCollection.Builder errorsBuilder, String keyPrefix, Locale locale) {
        validateStringValue(userId.getStaffLogin(), userIdBuilder::staffLogin);
        validateStringValue(userId.getPassportUid(), userIdBuilder::passportUid);
    }

    private static void validateStringValue(Optional<String> value, Consumer<String> setter) {
        value.filter(s -> !s.isEmpty())
                .ifPresent(setter);
    }

    private void validateAccountsSpaceKey(AccountsSpaceKeyResponseDto accountsSpaceKey,
                                          ReceivedAccountsSpaceKey.Builder accountsSpaceKeyBuilder,
                                          ErrorCollection.Builder errorsBuilder, String keyPrefix, Locale locale) {
        accountsSpaceKey.getSegmentation().ifPresent(segmentation -> {
            for (int i = 0; i < segmentation.size(); i++) {
                SegmentKeyResponseDto segmentKey = segmentation.get(i);
                ReceivedSegmentKey.Builder segmentKeyBuilder = ReceivedSegmentKey.builder();
                ErrorCollection.Builder segmentKeyErrorsBuilder = ErrorCollection.builder();
                validateSegmentKey(segmentKey, segmentKeyBuilder, segmentKeyErrorsBuilder,
                        keyPrefix + "segmentation." + i + ".", locale);
                if (segmentKeyErrorsBuilder.hasAnyErrors()) {
                    errorsBuilder.add(segmentKeyErrorsBuilder);
                } else {
                    accountsSpaceKeyBuilder.addSegment(segmentKeyBuilder.build());
                }
            }
        });
    }

}
