package ru.yandex.intranet.d.services.operations;

import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.UUID;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.yandex.ydb.table.transaction.TransactionMode;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.datasource.model.YdbTableClient;
import ru.yandex.intranet.d.model.accounts.AccountsQuotasOperationsModel;
import ru.yandex.intranet.d.model.accounts.OperationInProgressModel;
import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.services.operations.model.OperationPostRefreshContext;
import ru.yandex.intranet.d.services.operations.model.OperationPreRefreshContext;
import ru.yandex.intranet.d.services.operations.model.OperationRefreshContext;
import ru.yandex.intranet.d.services.operations.model.OperationRetryContext;
import ru.yandex.intranet.d.services.operations.model.RetryResult;
import ru.yandex.intranet.d.services.operations.model.RetryableOperation;
import ru.yandex.intranet.d.util.Uuids;
import ru.yandex.intranet.d.util.result.ErrorCollection;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.util.result.TypedError;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;

/**
 * Operations retry service implementation.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class OperationsRetryService {

    private static final Logger LOG = LoggerFactory.getLogger(OperationsRetryService.class);

    private static final Duration OPERATION_RETRY_COOLDOWN = Duration.ofMinutes(1);
    private static final int ABORT_COMMENT_LENGTH_LIMIT = 4096;

    private final YdbTableClient tableClient;
    private final OperationsRetryStoreService storeService;
    private final UpdateProvisionOperationsRetryService updateProvisionsRetryService;
    private final CreateAccountOperationsRetryService createAccountRetryService;
    private final MoveProvisionOperationsRetryService moveProvisionOperationsRetryService;
    private final DeliverUpdateProvisionOperationRetryService deliverUpdateProvisionOperationRetryService;
    private final ProvideReserveOperationRetryService provideReserveOperationRetryService;
    private final MessageSource messages;

    @SuppressWarnings("ParameterNumber")
    public OperationsRetryService(
            YdbTableClient tableClient,
            OperationsRetryStoreService storeService,
            UpdateProvisionOperationsRetryService updateProvisionsRetryService,
            CreateAccountOperationsRetryService createAccountRetryService,
            MoveProvisionOperationsRetryService moveProvisionOperationsRetryService,
            DeliverUpdateProvisionOperationRetryService deliverUpdateProvisionOperationRetryService,
            ProvideReserveOperationRetryService provideReserveOperationRetryService,
            @Qualifier("messageSource") MessageSource messages
    ) {
        this.tableClient = tableClient;
        this.storeService = storeService;
        this.updateProvisionsRetryService = updateProvisionsRetryService;
        this.createAccountRetryService = createAccountRetryService;
        this.moveProvisionOperationsRetryService = moveProvisionOperationsRetryService;
        this.deliverUpdateProvisionOperationRetryService = deliverUpdateProvisionOperationRetryService;
        this.provideReserveOperationRetryService = provideReserveOperationRetryService;
        this.messages = messages;
    }

    public Mono<Void> retryOperations(Clock clock, Locale locale) {
        Instant now = Instant.now(clock);
        return getEligibleOperations(now).flatMap(eligibleOps -> {
            Map<String, ProviderModel> providers = eligibleOps.getProviders();
            return Mono.just(eligibleOps.getOperations()).flatMapIterable(Function.identity()).concatMap(operation ->
                    retryOperation(operation, providers, clock, locale)
                            .doOnError(e -> LOG.error("Failed to retry operation "
                                    + operation.getOperation().getOperationId(), e))
                            .onErrorResume(v -> Mono.empty()))
                    .collectList().then();
        });
    }

    public Mono<RetryResult> retryOneOperation(
            RetryableOperation operation, Map<String, ProviderModel> providers
    ) {
        return retryOperation(operation, providers, Clock.systemUTC(), Locale.ENGLISH);
    }

    private Mono<RetryResult> retryOperation(
            RetryableOperation operation, Map<String, ProviderModel> providers, Clock clock, Locale locale
    ) {
        return preRefreshOperation(operation, providers, Instant.now(clock), locale)
                .flatMap(preRefreshContextWithOp -> {
            if (preRefreshContextWithOp.isEmpty()) {
                return Mono.empty();
            }
            RetryableOperation preRefreshOperation = preRefreshContextWithOp.get().getOperation();
            OperationPreRefreshContext preRefreshContext = preRefreshContextWithOp.get().getValue();
            return refreshOperation(preRefreshOperation, preRefreshContext, Instant.now(clock), locale)
                    .flatMap(refreshContext -> {
                if (refreshContext.isEmpty()) {
                    return Mono.empty();
                }
                return postRefreshOperation(preRefreshOperation, refreshContext.get(), Instant.now(clock), locale)
                    .flatMap(postRefreshContext -> {
                        if (postRefreshContext.isEmpty()) {
                            return Mono.empty();
                        }
                        return retryOperation(
                            preRefreshOperation, postRefreshContext.get(), Instant.now(clock), locale
                        ).flatMap(retryContext -> {
                            if (retryContext.isEmpty()) {
                                return Mono.empty();
                            }
                            return postRetryOperation(
                                    preRefreshOperation, retryContext.get(), Instant.now(clock), locale
                            ).thenReturn(retryContext.get().getRetryResult()
                                    .orElse(RetryResult.NON_FATAL_FAILURE)
                            );
                        });
                    });
            });
        });
    }

    public Mono<Result<Void>> abortOperation(String operationId, String comment, YaUserDetails currentUser,
                                             Locale locale) {
        if (!Uuids.isValidUuid(operationId)) {
            return Mono.just(Result.failure(ErrorCollection.builder().addError(TypedError.notFound(messages
                    .getMessage("errors.operation.not.found", null, locale))).build()));
        }
        if (comment == null || comment.isBlank()) {
            return Mono.just(Result.failure(ErrorCollection.builder().addError("comment",
                    TypedError.invalid(messages.getMessage("errors.field.is.required", null, locale)))
                    .build()));
        }
        if (comment.length() > ABORT_COMMENT_LENGTH_LIMIT) {
            return Mono.just(Result.failure(ErrorCollection.builder().addError("comment",
                    TypedError.invalid(messages.getMessage("errors.text.is.too.long", null, locale)))
                    .build()));
        }
        return tableClient.usingSessionMonoRetryable(session -> session.usingTxMonoRetryable(
                TransactionMode.SERIALIZABLE_READ_WRITE, txSession ->
                        storeService.getOperations(txSession, List.of(operationId)).flatMap(ops -> {
                    if (ops.isEmpty()) {
                        return Mono.just(Result.failure(ErrorCollection.builder().addError(TypedError
                                .notFound(messages.getMessage("errors.operation.not.found", null,
                                        locale))).build()));
                    }
                    AccountsQuotasOperationsModel operation = ops.get(0);
                    if (operation.getRequestStatus().isPresent() && operation.getRequestStatus().get()
                            != AccountsQuotasOperationsModel.RequestStatus.WAITING) {
                        return Mono.just(Result.failure(ErrorCollection.builder().addError(TypedError
                                .invalid(messages.getMessage("errors.operation.already.finished", null,
                                        locale))).build()));
                    }
                    BaseOperationRetryService service = selectImpl(operation.getOperationType());
                    return service.abortOperation(txSession, operation, comment, Instant.now(), currentUser, locale)
                            .thenReturn(Result.success(null));
                })));
    }

    public Mono<Result<List<AccountsQuotasOperationsModel>>> getInProgressOperations() {
        return tableClient.usingSessionMonoRetryable(session -> session.usingTxMonoRetryable(
                TransactionMode.SERIALIZABLE_READ_WRITE, txSession ->
                        storeService.getInProgressOperations(txSession).flatMap(inProgressOperations -> {
                            List<String> operationIds = inProgressOperations.stream()
                                    .map(OperationInProgressModel::getOperationId).distinct()
                                    .collect(Collectors.toList());
                            return storeService.getOperations(txSession, operationIds).map(Result::success);
                        })));
    }

    private Mono<Optional<WithOperation<? extends OperationPreRefreshContext>>> preRefreshOperation(
            RetryableOperation operation, Map<String, ProviderModel> providers, Instant now, Locale locale) {
        return tableClient.usingSessionMonoRetryable(session -> session.usingTxMonoRetryable(
                TransactionMode.SERIALIZABLE_READ_WRITE, txSession ->
                        storeService.reloadOperation(txSession, operation.getOperation()).flatMap(reloadedOperation -> {
                            if (reloadedOperation.isEmpty()) {
                                LOG.error("Operation {} vanished unexpectedly before retry",
                                        operation.getOperation().getOperationId());
                                return Mono.just(Optional.empty());
                            }
                            if (!isOperationInEligibleStatus(reloadedOperation.get())) {
                                return Mono.just(Optional.empty());
                            }
                            RetryableOperation currentOperation = operation.withOperation(reloadedOperation.get());
                            return storeService.loadCommonContext(txSession, currentOperation.getOperation())
                                    .flatMap(commonContext -> {
                                        BaseOperationRetryService service = selectImpl(currentOperation.getOperation()
                                                .getOperationType());
                                        return service.preRefreshOperation(txSession, currentOperation,
                                                commonContext, now, locale)
                                                .map(v -> v.map(r -> new WithOperation<>(r, currentOperation)));
                                    });
                        })));
    }

    private Mono<? extends Optional<? extends OperationRefreshContext>> refreshOperation(
            RetryableOperation operation,
            OperationPreRefreshContext context,
            Instant now,
            Locale locale) {
        BaseOperationRetryService service = selectImpl(operation.getOperation().getOperationType());
        return service.refreshOperation(operation, context, now, locale);
    }

    private Mono<? extends Optional<? extends OperationPostRefreshContext>> postRefreshOperation(
            RetryableOperation operation,
            OperationRefreshContext context,
            Instant now,
            Locale locale) {
        return tableClient.usingSessionMonoRetryable(session -> session.usingTxMonoRetryable(
                TransactionMode.SERIALIZABLE_READ_WRITE, txSession -> {
                    BaseOperationRetryService service = selectImpl(operation.getOperation().getOperationType());
                    return service.postRefreshOperation(txSession, operation, context, now, locale);
                }));
    }

    private Mono<? extends Optional<? extends OperationRetryContext>> retryOperation(
            RetryableOperation operation,
            OperationPostRefreshContext context,
            Instant now,
            Locale locale) {
        return updateOperationLastRequestId(operation)
                .flatMap(retryableOperation -> {
                    if (retryableOperation.isEmpty()) {
                        LOG.error("Operation {} vanished unexpectedly before retry",
                                operation.getOperation().getOperationId());
                        return Mono.just(Optional.empty());
                    }
                    BaseOperationRetryService service = selectImpl(
                            retryableOperation.get().getOperation().getOperationType());
                    return service.retryOperation(retryableOperation.get(), context, now, locale);
                });
    }

    private Mono<Optional<RetryableOperation>> updateOperationLastRequestId(RetryableOperation operation) {
        return tableClient.usingSessionMonoRetryable(session -> session.usingTxMonoRetryable(
                TransactionMode.SERIALIZABLE_READ_WRITE, txSession ->
                        storeService.reloadOperation(txSession, operation.getOperation())
                                .flatMap(reloadedOperation -> {
                                    if (reloadedOperation.isEmpty()) {
                                        return Mono.just(Optional.empty());
                                    }
                                    AccountsQuotasOperationsModel currentOperation =
                                            new AccountsQuotasOperationsModel.Builder(reloadedOperation.get())
                                                    .setLastRequestId(UUID.randomUUID().toString())
                                                    .build();
                                    return storeService.upsertOperation(txSession, currentOperation)
                                            .thenReturn(Optional.of(operation.withOperation(currentOperation)));
                                })
        ));
    }

    private Mono<WithOperation<Void>> postRetryOperation(RetryableOperation operation,
                                                         OperationRetryContext context,
                                                         Instant now,
                                                         Locale locale) {
        return tableClient.usingSessionMonoRetryable(session -> session.usingTxMonoRetryable(
                TransactionMode.SERIALIZABLE_READ_WRITE, txSession ->
                        storeService.reloadOperation(txSession, operation.getOperation()).flatMap(reloadedOperation -> {
                            if (reloadedOperation.isEmpty()) {
                                LOG.error("Operation {} vanished unexpectedly after retry",
                                        operation.getOperation().getOperationId());
                                return Mono.empty();
                            }
                            BaseOperationRetryService service = selectImpl(operation.getOperation().getOperationType());
                            RetryableOperation currentOperation = operation.withOperation(reloadedOperation.get());
                            return service.postRetryOperation(txSession, currentOperation, context, now, locale)
                                    .thenReturn(new WithOperation<>(null, currentOperation));
                        })));
    }

    private Mono<EligibleOperations> getEligibleOperations(Instant now) {
        return tableClient.usingSessionMonoRetryable(session -> session.usingTxMonoRetryable(
                TransactionMode.SERIALIZABLE_READ_WRITE, txSession ->
                        storeService.getInProgressOperations(txSession).flatMap(inProgressOperations -> {
                            Map<String, List<OperationInProgressModel>> opsById =  inProgressOperations.stream()
                                    .collect(Collectors.groupingBy(OperationInProgressModel::getOperationId));
                            List<String> operationIds = new ArrayList<>(opsById.keySet());
                            return storeService.getOperations(txSession, operationIds).flatMap(ops ->
                                    storeService.getOperationsProviders(txSession, ops).map(providers ->
                                            new EligibleOperations(ops.stream()
                                                    .filter(op -> isOperationNotInCoolDown(op, now, providers)
                                                            && isOperationInEligibleStatus(op))
                                                    .map(op -> new RetryableOperation(op,
                                                            opsById.getOrDefault(op.getOperationId(), List.of())))
                                                    .collect(Collectors.toList()), providers)));
                })));
    }

    private boolean isOperationNotInCoolDown(AccountsQuotasOperationsModel operation, Instant now,
                                             Map<String, ProviderModel> providers) {
        boolean coolDownDisabled = providers.get(operation.getProviderId()).getAccountsSettings()
                .isRetryCoolDownDisabled();
        Instant operationTimestamp = operation.getUpdateDateTime().orElseGet(operation::getCreateDateTime);
        boolean coolDown = (operationTimestamp.isAfter(now)
                || Duration.between(operationTimestamp, now).compareTo(OPERATION_RETRY_COOLDOWN) <= 0);
        return !coolDown || coolDownDisabled;
    }

    private boolean isOperationInEligibleStatus(AccountsQuotasOperationsModel operation) {
        return operation.getRequestStatus().isEmpty()
                || AccountsQuotasOperationsModel.RequestStatus.WAITING.equals(operation.getRequestStatus().get());
    }

    private BaseOperationRetryService selectImpl(AccountsQuotasOperationsModel.OperationType type) {
        return switch (type) {
            case UPDATE_PROVISION -> updateProvisionsRetryService;
            case CREATE_ACCOUNT -> createAccountRetryService;
            case MOVE_PROVISION -> moveProvisionOperationsRetryService;
            case DELIVER_AND_UPDATE_PROVISION -> deliverUpdateProvisionOperationRetryService;
            case PROVIDE_RESERVE -> provideReserveOperationRetryService;
            default -> throw new IllegalArgumentException("Unsupported operation type: " + type);
        };
    }

    private static final class WithOperation<T> {

        private final T value;
        private final RetryableOperation operation;

        private WithOperation(T value, RetryableOperation operation) {
            this.value = value;
            this.operation = operation;
        }

        public T getValue() {
            return value;
        }

        public RetryableOperation getOperation() {
            return operation;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            WithOperation<?> that = (WithOperation<?>) o;
            return Objects.equals(value, that.value) &&
                    Objects.equals(operation, that.operation);
        }

        @Override
        public int hashCode() {
            return Objects.hash(value, operation);
        }

        @Override
        public String toString() {
            return "WithOperation{" +
                    "value=" + value +
                    ", operation=" + operation +
                    '}';
        }

    }

    private static final class EligibleOperations {

        private final List<RetryableOperation> operations;
        private final Map<String, ProviderModel> providers;

        private EligibleOperations(List<RetryableOperation> operations, Map<String, ProviderModel> providers) {
            this.operations = operations;
            this.providers = providers;
        }

        public List<RetryableOperation> getOperations() {
            return operations;
        }

        public Map<String, ProviderModel> getProviders() {
            return providers;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            EligibleOperations that = (EligibleOperations) o;
            return Objects.equals(operations, that.operations) &&
                    Objects.equals(providers, that.providers);
        }

        @Override
        public int hashCode() {
            return Objects.hash(operations, providers);
        }

        @Override
        public String toString() {
            return "EligibleOperations{" +
                    "operations=" + operations +
                    ", providers=" + providers +
                    '}';
        }

    }

}
