package ru.yandex.intranet.d.services.operations;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuple2;
import reactor.util.function.Tuples;
import ru.yandex.intranet.d.dao.Tenants;
import ru.yandex.intranet.d.dao.accounts.AccountsDao;
import ru.yandex.intranet.d.dao.accounts.AccountsQuotasDao;
import ru.yandex.intranet.d.dao.accounts.AccountsQuotasOperationsDao;
import ru.yandex.intranet.d.dao.accounts.AccountsSpacesDao;
import ru.yandex.intranet.d.dao.accounts.OperationsInProgressDao;
import ru.yandex.intranet.d.dao.delivery.DeliveriesAndProvidedRequestsDao;
import ru.yandex.intranet.d.dao.folders.FolderDao;
import ru.yandex.intranet.d.dao.folders.FolderOperationLogDao;
import ru.yandex.intranet.d.dao.providers.ProvidersDao;
import ru.yandex.intranet.d.dao.quotas.QuotasDao;
import ru.yandex.intranet.d.dao.resources.ResourcesDao;
import ru.yandex.intranet.d.dao.resources.segmentations.ResourceSegmentationsDao;
import ru.yandex.intranet.d.dao.resources.segments.ResourceSegmentsDao;
import ru.yandex.intranet.d.dao.resources.types.ResourceTypesDao;
import ru.yandex.intranet.d.dao.services.ServicesDao;
import ru.yandex.intranet.d.dao.transfers.TransferRequestsDao;
import ru.yandex.intranet.d.dao.units.UnitsEnsemblesDao;
import ru.yandex.intranet.d.dao.users.UsersDao;
import ru.yandex.intranet.d.datasource.model.WithTxId;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.WithTenant;
import ru.yandex.intranet.d.model.accounts.AccountModel;
import ru.yandex.intranet.d.model.accounts.AccountSpaceModel;
import ru.yandex.intranet.d.model.accounts.AccountsQuotasModel;
import ru.yandex.intranet.d.model.accounts.AccountsQuotasOperationsModel;
import ru.yandex.intranet.d.model.accounts.OperationChangesModel;
import ru.yandex.intranet.d.model.accounts.OperationInProgressModel;
import ru.yandex.intranet.d.model.delivery.provide.DeliveryAndProvideModel;
import ru.yandex.intranet.d.model.delivery.provide.DeliveryAndProvideOperationListModel;
import ru.yandex.intranet.d.model.delivery.provide.DeliveryAndProvideOperationModel;
import ru.yandex.intranet.d.model.folders.FolderModel;
import ru.yandex.intranet.d.model.folders.FolderOperationLogModel;
import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.model.quotas.QuotaModel;
import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.model.resources.ResourceSegmentSettingsModel;
import ru.yandex.intranet.d.model.resources.segmentations.ResourceSegmentationModel;
import ru.yandex.intranet.d.model.resources.segments.ResourceSegmentModel;
import ru.yandex.intranet.d.model.resources.types.ResourceTypeModel;
import ru.yandex.intranet.d.model.transfers.TransferRequestModel;
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel;
import ru.yandex.intranet.d.model.users.UserModel;
import ru.yandex.intranet.d.services.operations.model.CreateAccountApplicationContext;
import ru.yandex.intranet.d.services.operations.model.CreateAccountContext;
import ru.yandex.intranet.d.services.operations.model.DeliverUpdateProvisionContext;
import ru.yandex.intranet.d.services.operations.model.ExpandedAccountSpace;
import ru.yandex.intranet.d.services.operations.model.ExpandedResource;
import ru.yandex.intranet.d.services.operations.model.ExpandedSegment;
import ru.yandex.intranet.d.services.operations.model.ExternalAccountsSpaceKey;
import ru.yandex.intranet.d.services.operations.model.ExternalResourceKey;
import ru.yandex.intranet.d.services.operations.model.ExternalSegmentKey;
import ru.yandex.intranet.d.services.operations.model.MoveProvisionContext;
import ru.yandex.intranet.d.services.operations.model.OperationCommonContext;
import ru.yandex.intranet.d.services.operations.model.UpdateProvisionContext;

/**
 * Operations retry store service implementation.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class OperationsRetryStoreService {

    private final OperationsInProgressDao operationsInProgressDao;
    private final AccountsQuotasOperationsDao accountsQuotasOperationsDao;
    private final ProvidersDao providersDao;
    private final AccountsSpacesDao accountsSpacesDao;
    private final UsersDao usersDao;
    private final AccountsDao accountsDao;
    private final FolderDao folderDao;
    private final QuotasDao quotasDao;
    private final AccountsQuotasDao accountsQuotasDao;
    private final UnitsEnsemblesDao unitsEnsemblesDao;
    private final ResourcesDao resourcesDao;
    private final ResourceTypesDao resourceTypesDao;
    private final ResourceSegmentationsDao resourceSegmentationsDao;
    private final ResourceSegmentsDao resourceSegmentsDao;
    private final FolderOperationLogDao folderOperationLogDao;
    private final ServicesDao servicesDao;
    private final DeliveriesAndProvidedRequestsDao deliveriesAndProvidedRequestsDao;
    private final TransferRequestsDao transferRequestsDao;

    @SuppressWarnings("ParameterNumber")
    public OperationsRetryStoreService(OperationsInProgressDao operationsInProgressDao,
                                       AccountsQuotasOperationsDao accountsQuotasOperationsDao,
                                       ProvidersDao providersDao,
                                       AccountsSpacesDao accountsSpacesDao,
                                       UsersDao usersDao,
                                       AccountsDao accountsDao,
                                       FolderDao folderDao,
                                       QuotasDao quotasDao,
                                       AccountsQuotasDao accountsQuotasDao,
                                       UnitsEnsemblesDao unitsEnsemblesDao,
                                       ResourcesDao resourcesDao,
                                       ResourceTypesDao resourceTypesDao,
                                       ResourceSegmentationsDao resourceSegmentationsDao,
                                       ResourceSegmentsDao resourceSegmentsDao,
                                       FolderOperationLogDao folderOperationLogDao,
                                       ServicesDao servicesDao,
                                       DeliveriesAndProvidedRequestsDao deliveriesAndProvidedRequestsDao,
                                       TransferRequestsDao transferRequestsDao) {
        this.operationsInProgressDao = operationsInProgressDao;
        this.accountsQuotasOperationsDao = accountsQuotasOperationsDao;
        this.providersDao = providersDao;
        this.accountsSpacesDao = accountsSpacesDao;
        this.usersDao = usersDao;
        this.accountsDao = accountsDao;
        this.folderDao = folderDao;
        this.quotasDao = quotasDao;
        this.accountsQuotasDao = accountsQuotasDao;
        this.unitsEnsemblesDao = unitsEnsemblesDao;
        this.resourcesDao = resourcesDao;
        this.resourceTypesDao = resourceTypesDao;
        this.resourceSegmentationsDao = resourceSegmentationsDao;
        this.resourceSegmentsDao = resourceSegmentsDao;
        this.folderOperationLogDao = folderOperationLogDao;
        this.servicesDao = servicesDao;
        this.deliveriesAndProvidedRequestsDao = deliveriesAndProvidedRequestsDao;
        this.transferRequestsDao = transferRequestsDao;
    }

    public Mono<List<OperationInProgressModel>> getInProgressOperations(YdbTxSession session) {
        return operationsInProgressDao.getAllByTenant(session, Tenants.DEFAULT_TENANT_ID);
    }

    public Mono<List<AccountsQuotasOperationsModel>> getOperations(YdbTxSession session, List<String> ids) {
        if (ids.isEmpty()) {
            return Mono.just(List.of());
        }
        return Flux.fromIterable(Lists.partition(ids, 500))
                .concatMap(v -> accountsQuotasOperationsDao.getByIds(session, v, Tenants.DEFAULT_TENANT_ID))
                .collectList().map(l -> l.stream().flatMap(Collection::stream).collect(Collectors.toList()));
    }

    public Mono<Optional<AccountsQuotasOperationsModel>> reloadOperation(YdbTxSession session,
                                                                         AccountsQuotasOperationsModel operation) {
        return accountsQuotasOperationsDao.getById(session, operation.getOperationId(), operation.getTenantId());
    }

    public Mono<Void> upsertOperation(YdbTxSession session, AccountsQuotasOperationsModel operation) {
        return accountsQuotasOperationsDao.upsertOneRetryable(session, operation).then();
    }

    public Mono<AccountsQuotasOperationsModel> upsertOperationAndReturn(YdbTxSession session,
                                                                        AccountsQuotasOperationsModel operation) {
        return accountsQuotasOperationsDao.upsertOneRetryable(session, operation);
    }

    public Mono<Optional<DeliveryAndProvideModel>> getDelivery(YdbTxSession session, String deliveryId) {
        return deliveriesAndProvidedRequestsDao.getById(session, deliveryId, Tenants.DEFAULT_TENANT_ID);
    }

    public Mono<Void> upsertDelivery(YdbTxSession session, AccountsQuotasOperationsModel operation) {
        String accountId = operation.getRequestedChanges().getAccountId().orElse(null);
        if (accountId == null) {
            return Mono.empty();
        }

        String deliveryId = operation.getRequestedChanges().getDeliveryId().orElse(null);

        return deliveryId != null ? getDelivery(session, deliveryId)
                .flatMap(deliveryO -> {
                    if (deliveryO.isEmpty()) {
                        return Mono.empty();
                    }

                    DeliveryAndProvideModel delivery = deliveryO.get();

                    List<DeliveryAndProvideOperationListModel> otherModels = new ArrayList<>();
                    DeliveryAndProvideOperationListModel listModel = null;
                    for (DeliveryAndProvideOperationListModel current: delivery.getOperations()) {
                        if (current.getAccountId().equals(accountId)) {
                            listModel = current;
                        } else {
                            otherModels.add(current);
                        }
                    }

                    if (listModel == null) {
                        return Mono.empty();
                    }

                    List<DeliveryAndProvideOperationModel> operations = new ArrayList<>(listModel.getOperations());

                    int max = operations.stream()
                            .mapToInt(DeliveryAndProvideOperationModel::getVersion)
                            .max()
                            .orElse(-1);

                    operations.add(new DeliveryAndProvideOperationModel(max + 1,
                            operation.getOperationId()));

                    DeliveryAndProvideModel deliveryAndProvideModel = new DeliveryAndProvideModel.Builder()
                            .deliveryId(deliveryId)
                            .tenantId(delivery.getTenantId())
                            .request(delivery.getRequest())
                            .addOperations(otherModels)
                            .addOperation(new DeliveryAndProvideOperationListModel(listModel.getServiceId(),
                                    listModel.getFolderId(), listModel.getAccountId(), operations))
                            .build();

                    return deliveriesAndProvidedRequestsDao.upsertOneRetryable(session, deliveryAndProvideModel);

                }).then() : Mono.empty();
    }

    public Mono<Void> upsertQuotas(YdbTxSession session, List<QuotaModel> quotas) {
        if (quotas.isEmpty()) {
            return Mono.empty();
        }
        return quotasDao.upsertAllRetryable(session, quotas);
    }

    public Mono<Void> upsertAccountQuotas(YdbTxSession session, List<AccountsQuotasModel> quotas) {
        if (quotas.isEmpty()) {
            return Mono.empty();
        }
        return accountsQuotasDao.upsertAllRetryable(session, quotas);
    }

    public Mono<Void> deleteOperationsInProgress(YdbTxSession session, List<OperationInProgressModel> inProgress) {
        return operationsInProgressDao.deleteManyRetryable(session, inProgress.stream()
                .map(op -> new WithTenant<>(op.getTenantId(), op.getKey())).collect(Collectors.toList()));
    }

    public Mono<Void> deleteInProgressByOperationId(YdbTxSession session, String operationId) {
        return operationsInProgressDao.deleteByOperationIdRetryable(session, Tenants.DEFAULT_TENANT_ID, operationId);
    }

    public Mono<OperationCommonContext> loadCommonContext(YdbTxSession session,
                                                          AccountsQuotasOperationsModel operation) {
        return providersDao.getById(session, operation.getProviderId(), operation.getTenantId()).flatMap(providerO ->
                usersDao.getById(session, operation.getAuthorUserId(), operation.getTenantId()).flatMap(userO -> {
            if (providerO.isEmpty()) {
                return Mono.error(new IllegalArgumentException("Operation " + operation.getOperationId()
                        + "references missing provider " + operation.getProviderId()));
            }
            if (userO.isEmpty()) {
                return Mono.error(new IllegalArgumentException("Operation " + operation.getOperationId()
                        + "references missing user " + operation.getAuthorUserId()));
            }
            return getResources(session, providerO.get()).flatMap(resources ->
                    getAccountsSpacesIfPresent(session, providerO.get()).flatMap(accountsSpaces -> {
                Map<ExternalResourceKey, ExpandedResource> externalResourceIndex = prepareExternalIndex(resources);
                Map<String, ExpandedResource> resourceIndex = resources.stream()
                        .collect(Collectors.toMap(r -> r.getResource().getId(), Function.identity()));
                Map<ExternalAccountsSpaceKey, ExpandedAccountSpace> externalAccountsSpaceIndex
                        = prepareExternalAccountsSpaceIndex(accountsSpaces);
                Map<String, ExpandedAccountSpace> accountsSpaceIndex = accountsSpaces.stream()
                        .collect(Collectors.toMap(s -> s.getAccountSpace().getId(), Function.identity()));
                if (operation.getAccountsSpaceId().isEmpty()) {
                    return Mono.just(new OperationCommonContext(providerO.get(), null,
                            userO.get(), externalResourceIndex, resourceIndex,
                            externalAccountsSpaceIndex, accountsSpaceIndex));
                }
                ExpandedAccountSpace accountsSpace = accountsSpaceIndex.get(operation.getAccountsSpaceId().get());
                if (accountsSpace == null) {
                    return Mono.error(new IllegalArgumentException("Operation " + operation.getOperationId()
                            + "references missing accounts space " + operation.getAccountsSpaceId().get()));
                }
                return Mono.just(new OperationCommonContext(providerO.get(), accountsSpace.getAccountSpace(),
                        userO.get(), externalResourceIndex, resourceIndex, externalAccountsSpaceIndex,
                        accountsSpaceIndex));
            }));
        }));
    }

    public Mono<MoveProvisionContext> loadMoveProvisionContext(YdbTxSession session,
                                                               AccountsQuotasOperationsModel operation) {
        OperationChangesModel requestedChanges = operation.getRequestedChanges();

        List<String> accountIds = Stream.of(requestedChanges.getAccountId(),
                requestedChanges.getDestinationAccountId())
                .filter(Optional::isPresent)
                .map(Optional::get)
                .collect(Collectors.toList());
        return accountsDao.getAllByIdsWithDeleted(session, accountIds, Tenants.DEFAULT_TENANT_ID).flatMap(accounts -> {
            if (accounts.size() < 2) {
                Set<String> missingIds = missingIds(accounts, accountIds, AccountModel::getId);
                return Mono.error(new IllegalArgumentException("Operation references missing account " + missingIds));
            }
            List<String> folderIds = accounts.stream()
                    .map(AccountModel::getFolderId)
                    .distinct()
                    .collect(Collectors.toList());
            return folderDao.getByIds(session, folderIds, Tenants.DEFAULT_TENANT_ID).flatMap(folders -> {
                Set<String> missingIds = missingIds(folders, folderIds, FolderModel::getId);
                if (!missingIds.isEmpty()) {
                    return Mono.error(new IllegalArgumentException("Operation references missing folders "
                            + missingIds));
                }
                return loadMoveProvisionContext(session, operation, accounts, folders);
            });
        });
    }

    private static <T> Set<String> missingIds(Collection<T> items,
                                              Collection<String> ids,
                                              Function<T, String> idExtractor) {
        Set<String> loadedIds = items.stream().map(idExtractor).collect(Collectors.toSet());
        Set<String> expectedIds = new HashSet<>(ids);
        return Sets.difference(expectedIds, loadedIds);
    }

    public Mono<UpdateProvisionContext> loadUpdateProvisionContext(YdbTxSession session,
                                                                   AccountsQuotasOperationsModel operation) {
        return accountsDao.getByIdWithDeleted(session, operation.getRequestedChanges().getAccountId().get(),
                Tenants.DEFAULT_TENANT_ID).flatMap(accountO -> {
                    if (accountO.isEmpty()) {
                        return Mono.error(new IllegalArgumentException("Operation references missing account "
                                + operation.getRequestedChanges().getAccountId().get()));
                    }
                    return folderDao.getById(session, accountO.get().getFolderId(), Tenants.DEFAULT_TENANT_ID)
                            .flatMap(folderO -> {
                                if (folderO.isEmpty()) {
                                    return Mono.error(new IllegalArgumentException("Operation references missing "
                                            + "folder " + accountO.get().getFolderId()));
                                }
                                return loadUpdateProvisionContext(session, operation, accountO.get(), folderO.get());
                            });
                });
    }

    public Mono<DeliverUpdateProvisionContext> loadDeliverUpdateProvisionContext(
            YdbTxSession session, AccountsQuotasOperationsModel operation) {
        return accountsDao.getByIdWithDeleted(session, operation.getRequestedChanges().getAccountId().orElseThrow(),
                Tenants.DEFAULT_TENANT_ID).flatMap(accountO -> {
            if (accountO.isEmpty()) {
                return Mono.error(new IllegalArgumentException("Operation references missing account "
                        + operation.getRequestedChanges().getAccountId().get()));
            }
            return folderDao.getById(session, accountO.get().getFolderId(), Tenants.DEFAULT_TENANT_ID)
                    .flatMap(folderO -> {
                        if (folderO.isEmpty()) {
                            return Mono.error(new IllegalArgumentException("Operation references missing "
                                    + "folder " + accountO.get().getFolderId()));
                        }
                        return loadDeliverUpdateProvisionContext(session, operation, accountO.get(), folderO.get());
                    });
        });
    }

    public Mono<CreateAccountContext> loadCreateAccountContext(YdbTxSession session,
                                                               AccountsQuotasOperationsModel operation) {
        return folderDao.getById(session, operation.getRequestedChanges()
                .getAccountCreateParams().get().getFolderId(), Tenants.DEFAULT_TENANT_ID).flatMap(folderO -> {
                    if (folderO.isEmpty()) {
                        return Mono.error(new IllegalArgumentException("Operation references missing folder "
                                + operation.getRequestedChanges().getAccountCreateParams().get().getFolderId()));
                    }
                    FolderModel folder = folderO.get();
                    return accountsDao.getByFoldersForProvider(session, Tenants.DEFAULT_TENANT_ID,
                            operation.getProviderId(), Set.of(folder.getId()),
                            operation.getAccountsSpaceId().orElse(null)).flatMap(accounts -> {
                        Map<String, AccountModel> accountsById = accounts.stream().collect(Collectors
                                .toMap(AccountModel::getId, Function.identity()));
                        String targetAccountId = operation.getRequestedChanges()
                                .getAccountCreateParams().get().getAccountId();
                        return accountsDao.getById(session, targetAccountId, Tenants.DEFAULT_TENANT_ID)
                                .flatMap(targetAccountO -> servicesDao.getByIdMinimal(session, folder.getServiceId())
                                        .map(service -> new CreateAccountContext(folder, accountsById,
                                                targetAccountO.orElse(null), service.get().orElseThrow().getSlug())));
                    });
        });
    }

    public Mono<CreateAccountApplicationContext> loadCreateAccountApplicationContext(
            YdbTxSession session, AccountsQuotasOperationsModel operation, String externalAccountId) {
        return folderDao.getById(session, operation.getRequestedChanges()
                .getAccountCreateParams().get().getFolderId(), Tenants.DEFAULT_TENANT_ID).flatMap(folderO -> {
            if (folderO.isEmpty()) {
                return Mono.error(new IllegalArgumentException("Operation references missing folder "
                        + operation.getRequestedChanges().getAccountCreateParams().get().getFolderId()));
            }
            FolderModel folder = folderO.get();
            String targetAccountId = operation.getRequestedChanges().getAccountCreateParams().get().getAccountId();
            return accountsDao.getById(session, targetAccountId, Tenants.DEFAULT_TENANT_ID).flatMap(targetAccountO -> {
                WithTenant<AccountModel.ExternalId> externalId = new WithTenant<>(Tenants.DEFAULT_TENANT_ID,
                        new AccountModel.ExternalId(operation.getProviderId(), externalAccountId,
                                operation.getAccountsSpaceId().orElse(null)));
                return accountsDao.getAllByExternalId(session, externalId).map(targetExternalAccountO ->
                        new CreateAccountApplicationContext(folder, targetAccountO.orElse(null),
                                targetExternalAccountO.orElse(null)));
            });
        });
    }

    public Mono<List<ExpandedResource>> getResources(YdbTxSession session, ProviderModel provider) {
        //deleted == true because of DISPENSER-5324
        return resourcesDao.getAllByProvider(session, provider.getId(), Tenants.DEFAULT_TENANT_ID, true)
                .flatMap(resources -> expandResources(session, resources));
    }

    public Mono<List<ExpandedAccountSpace>> getAccountSpaces(YdbTxSession session, ProviderModel provider) {
        return accountsSpacesDao.getAllByProvider(session, Tenants.DEFAULT_TENANT_ID, provider.getId())
                .map(WithTxId::get).flatMap(accountSpaces -> expandAccountSpaces(session, accountSpaces));
    }

    public Mono<Optional<FolderModel>> loadFolder(YdbTxSession session, String folderId) {
        return folderDao.getById(session, folderId, Tenants.DEFAULT_TENANT_ID);
    }

    public Mono<List<FolderModel>> loadFolders(YdbTxSession session, Set<String> folderIds) {
        if (folderIds.isEmpty()) {
            return Mono.just(List.of());
        }
        return Flux.fromIterable(Lists.partition(new ArrayList<>(folderIds), 500))
                .concatMap(v -> folderDao.getByIds(session, v, Tenants.DEFAULT_TENANT_ID))
                .collectList().map(l -> l.stream().flatMap(Collection::stream).collect(Collectors.toList()));
    }

    public Mono<Optional<AccountModel>> loadAccountByExternalId(YdbTxSession session, String providerId,
                                                                 String accountsSpaceId, String externalAccountId) {
        return accountsDao.getAllByExternalId(session, new WithTenant<>(Tenants.DEFAULT_TENANT_ID,
                new AccountModel.ExternalId(providerId, externalAccountId, accountsSpaceId)));
    }

    public Mono<List<AccountModel>> loadAccountsByExternalIds(YdbTxSession session,
                                                               String providerId,
                                                               String accountsSpaceId,
                                                               Set<String> externalAccountIds) {
        if (externalAccountIds.isEmpty()) {
            return Mono.just(List.of());
        }
        List<WithTenant<AccountModel.ExternalId>> ids = externalAccountIds.stream().map(id ->
                new WithTenant<>(Tenants.DEFAULT_TENANT_ID, new AccountModel
                        .ExternalId(providerId, id, accountsSpaceId))).collect(Collectors.toList());
        return Flux.fromIterable(Lists.partition(ids, 500))
                .concatMap(v -> accountsDao.getAllByExternalIds(session, v))
                .collectList().map(l -> l.stream().flatMap(Collection::stream).collect(Collectors.toList()));
    }

    public Mono<List<UserModel>> loadUsers(YdbTxSession session, Set<String> uids, Set<String> logins) {
        if (logins.isEmpty() && uids.isEmpty()) {
            return Mono.just(List.of());
        }
        List<List<String>> partitionedLogins = Lists.partition(new ArrayList<>(logins), 250);
        List<List<String>> partitionedUids = Lists.partition(new ArrayList<>(uids), 250);
        List<Tuple2<List<Tuple2<String, TenantId>>, List<Tuple2<String, TenantId>>>> zipped = new ArrayList<>();
        for (int i = 0; i < Math.max(partitionedLogins.size(), partitionedUids.size()); i++) {
            List<Tuple2<String, TenantId>> loginPairs = i < partitionedLogins.size()
                    ? partitionedLogins.get(i).stream().map(v -> Tuples.of(v, Tenants.DEFAULT_TENANT_ID))
                    .collect(Collectors.toList())
                    : List.of();
            List<Tuple2<String, TenantId>> uidPairs = i < partitionedUids.size()
                    ? partitionedUids.get(i).stream().map(v -> Tuples.of(v, Tenants.DEFAULT_TENANT_ID))
                    .collect(Collectors.toList())
                    : List.of();
            zipped.add(Tuples.of(uidPairs, loginPairs));
        }
        return Flux.fromIterable(zipped)
                .concatMap(v -> usersDao.getByExternalIds(session, v.getT1(), v.getT2(), List.of()))
                .collectList().map(l -> l.stream().flatMap(Collection::stream).collect(Collectors.toList()));
    }

    public Mono<List<AccountsQuotasOperationsModel>> loadOperations(YdbTxSession session, Set<String> operationIds) {
        if (operationIds.isEmpty()) {
            return Mono.just(List.of());
        }
        List<WithTenant<String>> unitsEnsembleIdsToLoad = operationIds.stream()
                .map(id -> new WithTenant<>(Tenants.DEFAULT_TENANT_ID, id)).collect(Collectors.toList());
        return Flux.fromIterable(Lists.partition(unitsEnsembleIdsToLoad, 500))
                .concatMap(v -> accountsQuotasOperationsDao.getByIds(session, v))
                .collectList().map(l -> l.stream().flatMap(Collection::stream).collect(Collectors.toList()));
    }

    public Mono<Void> finishOperation(YdbTxSession session,
                                      List<OperationInProgressModel> operationsInProgressToDelete,
                                      AccountsQuotasOperationsModel updatedOperation) {
        List<WithTenant<OperationInProgressModel.Key>> inProgressKeysToDelete = operationsInProgressToDelete.stream()
                .map(v -> new WithTenant<>(v.getTenantId(), v.getKey())).collect(Collectors.toList());
        return operationsInProgressDao.deleteManyRetryable(session, inProgressKeysToDelete)
                .then(Mono.defer(() -> accountsQuotasOperationsDao.upsertOneRetryable(session, updatedOperation)
                        .then()));
    }

    public Mono<Void> finishDeliverOperation(YdbTxSession session,
                                             List<OperationInProgressModel> operationsInProgressToDelete,
                                             AccountsQuotasOperationsModel updatedOperation) {
        List<WithTenant<OperationInProgressModel.Key>> inProgressKeysToDelete = operationsInProgressToDelete.stream()
                .map(v -> new WithTenant<>(v.getTenantId(), v.getKey())).collect(Collectors.toList());
        return operationsInProgressDao.deleteManyRetryable(session, inProgressKeysToDelete)
                .then(Mono.defer(() -> accountsQuotasOperationsDao.upsertOneRetryable(session, updatedOperation)
                        .then()));
    }

    public Mono<Void> createAccount(
            YdbTxSession session,
            AccountModel accountToUpsert,
            FolderModel updatedFolder,
            FolderOperationLogModel folderOpLogToUpsert,
            List<QuotaModel> updatedQuotas,
            List<AccountsQuotasModel> accountProvisions
    ) {
        return accountsDao.upsertOneRetryable(session, accountToUpsert)
                .then(Mono.defer(() -> folderDao.upsertOneRetryable(session, updatedFolder)))
                .then(Mono.defer(() -> upsertQuotas(session, updatedQuotas)))
                .then(Mono.defer(() -> upsertProvisions(session, accountProvisions)))
                .then(Mono.defer(() -> folderOperationLogDao.upsertOneRetryable(session, folderOpLogToUpsert).then()));
    }

    public Mono<List<QuotaModel>> loadUpdateProvisionQuotas(YdbTxSession session,
                                                            AccountsQuotasOperationsModel operation) {
        String accountId = operation.getRequestedChanges().getAccountId().get();
        return accountsDao.getByIdWithDeleted(session, accountId, Tenants.DEFAULT_TENANT_ID).flatMap(accountO -> {
            if (accountO.isEmpty()) {
                return Mono.error(new IllegalArgumentException("Operation references missing account " + accountId));
            }
            return quotasDao.getByFoldersAndProvider(session, List.of(accountO.get().getFolderId()),
                    Tenants.DEFAULT_TENANT_ID, operation.getProviderId());
        });
    }

    public Mono<List<QuotaModel>> loadDeliverUpdateProvisionQuotas(YdbTxSession session,
                                                                   AccountsQuotasOperationsModel operation) {
        String accountId = operation.getRequestedChanges().getAccountId().orElseThrow();
        return accountsDao.getByIdWithDeleted(session, accountId, Tenants.DEFAULT_TENANT_ID).flatMap(accountO -> {
            if (accountO.isEmpty()) {
                return Mono.error(new IllegalArgumentException("Operation references missing account " + accountId));
            }
            return quotasDao.getByFoldersAndProvider(session, List.of(accountO.get().getFolderId()),
                    Tenants.DEFAULT_TENANT_ID, operation.getProviderId());
        });
    }

    public Mono<List<AccountsQuotasModel>> loadMoveProvisionQuotas(YdbTxSession session,
                                                                   AccountsQuotasOperationsModel operation) {
        String sourceAccountId = operation.getRequestedChanges().getAccountId().orElseThrow();
        String destinationAccountId = operation.getRequestedChanges().getDestinationAccountId().orElseThrow();
        List<String> accountIds = List.of(sourceAccountId, destinationAccountId);
        return accountsDao.getAllByIdsWithDeleted(session, accountIds,
                Tenants.DEFAULT_TENANT_ID).flatMap(accounts -> {
            if (accounts.size() < 2) {
                Set<String> missingIds = missingIds(accounts, accountIds, AccountModel::getId);
                return Mono.error(new IllegalArgumentException("Operation references missing accounts " + missingIds));
            }
            Set<String> notRemovedAccountIds = accounts.stream()
                    .filter(a -> !a.isDeleted())
                    .map(AccountModel::getId)
                    .collect(Collectors.toSet());
            return accountsQuotasDao.getAllByAccountIds(session, Tenants.DEFAULT_TENANT_ID, notRemovedAccountIds);
        });
    }

    public Mono<List<QuotaModel>> loadFolderQuotas(YdbTxSession session, String folderId, String providerId) {
        return quotasDao.getByFoldersAndProvider(session, List.of(folderId), Tenants.DEFAULT_TENANT_ID, providerId);
    }

    public Mono<Void> upsertFolder(YdbTxSession session, FolderModel folder) {
        return folderDao.upsertOneRetryable(session, folder).then();
    }

    public Mono<Void> upsertFolders(YdbTxSession session, List<FolderModel> folders) {
        return folderDao.upsertAllRetryable(session, folders);
    }

    public Mono<Void> upsertFolderOpLog(YdbTxSession session, FolderOperationLogModel opLog) {
        return folderOperationLogDao.upsertOneRetryable(session, opLog).then();
    }

    public Mono<Void> upsertFolderOpLogs(YdbTxSession session, List<FolderOperationLogModel> opLogs) {
        return folderOperationLogDao.upsertAllRetryable(session, opLogs);
    }

    public Mono<Void> upsertProvisions(YdbTxSession session, List<AccountsQuotasModel> provisions) {
        return accountsQuotasDao.upsertAllRetryable(session, provisions);
    }

    public Mono<Void> upsertAccount(YdbTxSession session, AccountModel account) {
        return accountsDao.upsertOneRetryable(session, account).then();
    }

    public Mono<Void> upsertAccounts(YdbTxSession session, List<AccountModel> accounts) {
        return accountsDao.upsertAllRetryable(session, accounts);
    }

    public Mono<Map<String, ProviderModel>> getOperationsProviders(YdbTxSession session,
                                                                   List<AccountsQuotasOperationsModel> operations) {
        if (operations.isEmpty()) {
            return Mono.just(Map.of());
        }
        List<Tuple2<String, TenantId>> providerIds = operations.stream()
                .map(AccountsQuotasOperationsModel::getProviderId).distinct()
                .map(id -> Tuples.of(id, Tenants.DEFAULT_TENANT_ID)).collect(Collectors.toList());
        return Flux.fromIterable(Lists.partition(new ArrayList<>(providerIds), 500))
                .concatMap(v -> providersDao.getByIds(session, v))
                .collectList().map(l -> l.stream().flatMap(Collection::stream)
                        .collect(Collectors.toMap(ProviderModel::getId, Function.identity())));
    }

    public Mono<Optional<TransferRequestModel>> getTransferRequest(YdbTxSession session, String transferRequestId) {
        return transferRequestsDao.getById(session, transferRequestId, Tenants.DEFAULT_TENANT_ID);
    }

    private Mono<UpdateProvisionContext> loadUpdateProvisionContext(YdbTxSession session,
                                                                    AccountsQuotasOperationsModel operation,
                                                                    AccountModel account,
                                                                    FolderModel folder) {
        return accountsDao.getByFoldersForProvider(session, Tenants.DEFAULT_TENANT_ID,
                operation.getProviderId(), Set.of(folder.getId()),
                operation.getAccountsSpaceId().orElse(null)).flatMap(accounts -> {
            Map<String, AccountModel> accountsById = new HashMap<>();
            accounts.forEach(acc -> accountsById.put(acc.getId(), acc));
            accountsById.put(account.getId(), account);
            return quotasDao.getByFoldersAndProvider(session, List.of(folder.getId()),
                    Tenants.DEFAULT_TENANT_ID, operation.getProviderId()).flatMap(quotas -> {
                Map<String, QuotaModel> quotasByResourceId = quotas.stream().collect(
                        Collectors.toMap(QuotaModel::getResourceId, Function.identity()));
                Set<String> notRemovedAccountIds = accountsById.values().stream()
                        .filter(acc -> !acc.isDeleted()).map(AccountModel::getId)
                        .collect(Collectors.toSet());
                return accountsQuotasDao.getAllByAccountIds(session, Tenants.DEFAULT_TENANT_ID, notRemovedAccountIds)
                        .map(provisions -> {
                            Map<String, Map<String, AccountsQuotasModel>> provisionsByAccountIdResourceId
                                    = new HashMap<>();
                            provisions.forEach(p -> provisionsByAccountIdResourceId.computeIfAbsent(p.getAccountId(),
                                    k -> new HashMap<>()).put(p.getResourceId(), p));
                            return new UpdateProvisionContext(account, folder, quotasByResourceId,
                                    provisionsByAccountIdResourceId, accountsById);

                        });
            });
        });
    }

    private Mono<DeliverUpdateProvisionContext> loadDeliverUpdateProvisionContext(
            YdbTxSession session,
            AccountsQuotasOperationsModel operation,
            AccountModel account,
            FolderModel folder) {
        return accountsDao.getByFoldersForProvider(session, Tenants.DEFAULT_TENANT_ID,
                operation.getProviderId(), Set.of(folder.getId()),
                operation.getAccountsSpaceId().orElse(null)).flatMap(accounts -> {
            Map<String, AccountModel> accountsById = new HashMap<>();
            accounts.forEach(acc -> accountsById.put(acc.getId(), acc));
            accountsById.put(account.getId(), account);
            return quotasDao.getByFoldersAndProvider(session, List.of(folder.getId()),
                    Tenants.DEFAULT_TENANT_ID, operation.getProviderId()).flatMap(quotas -> {
                Map<String, QuotaModel> quotasByResourceId = quotas.stream().collect(
                        Collectors.toMap(QuotaModel::getResourceId, Function.identity()));
                Set<String> notRemovedAccountIds = accountsById.values().stream()
                        .filter(acc -> !acc.isDeleted()).map(AccountModel::getId)
                        .collect(Collectors.toSet());
                return accountsQuotasDao.getAllByAccountIds(session, Tenants.DEFAULT_TENANT_ID, notRemovedAccountIds)
                        .map(provisions -> {
                            Map<String, Map<String, AccountsQuotasModel>> provisionsByAccountIdResourceId
                                    = new HashMap<>();
                            provisions.forEach(p -> provisionsByAccountIdResourceId.computeIfAbsent(p.getAccountId(),
                                    k -> new HashMap<>()).put(p.getResourceId(), p));
                            return new DeliverUpdateProvisionContext(account, folder, quotasByResourceId,
                                    provisionsByAccountIdResourceId, accountsById);

                        });
            });
        });
    }

    private Mono<MoveProvisionContext> loadMoveProvisionContext(YdbTxSession session,
                                                                  AccountsQuotasOperationsModel operation,
                                                                  List<AccountModel> accounts,
                                                                  List<FolderModel> folders) {
        Map<String, AccountModel> accountsById = new HashMap<>();
        accounts.forEach(acc -> accountsById.put(acc.getId(), acc));
        List<String> folderIds = folders.stream().map(FolderModel::getId).collect(Collectors.toList());
        return quotasDao.getByFoldersAndProvider(session, folderIds,
                Tenants.DEFAULT_TENANT_ID, operation.getProviderId()).flatMap(quotas -> {
            Map<String, Map<String, QuotaModel>> quotasByFolderIdResourceId = new HashMap<>();
            quotas.forEach(q -> quotasByFolderIdResourceId.computeIfAbsent(q.getFolderId(), k -> new HashMap<>())
                    .put(q.getResourceId(), q));
            Set<String> notRemovedAccountIds = accountsById.values().stream()
                    .filter(acc -> !acc.isDeleted()).map(AccountModel::getId)
                    .collect(Collectors.toSet());
            return accountsQuotasDao.getAllByAccountIds(session, Tenants.DEFAULT_TENANT_ID, notRemovedAccountIds)
                    .map(provisions -> {
                        Map<String, Map<String, AccountsQuotasModel>> provisionsByAccountIdResourceId
                                = new HashMap<>();
                        provisions.forEach(p -> provisionsByAccountIdResourceId.computeIfAbsent(p.getAccountId(),
                                k -> new HashMap<>()).put(p.getResourceId(), p));
                        OperationChangesModel requestedChanges = operation.getRequestedChanges();
                        Map<String, FolderModel> folderById =
                                folders.stream().collect(Collectors.toMap(FolderModel::getId, Function.identity()));
                        AccountModel sourceAccount = accountsById.get(requestedChanges.getAccountId().orElseThrow());
                        AccountModel destinationAccount = accountsById.get(
                                requestedChanges.getDestinationAccountId().orElseThrow());
                        FolderModel sourceFolder = folderById.get(sourceAccount.getFolderId());
                        FolderModel destinationFolder = folderById.get(destinationAccount.getFolderId());
                        return new MoveProvisionContext(sourceAccount, destinationAccount, sourceFolder,
                                destinationFolder, quotasByFolderIdResourceId, provisionsByAccountIdResourceId);
                    });
        });
    }

    private Mono<List<ExpandedAccountSpace>> getAccountsSpacesIfPresent(YdbTxSession session, ProviderModel provider) {
        if (provider.isAccountsSpacesSupported()) {
            return getAccountSpaces(session, provider);
        }
        return Mono.just(List.of());
    }

    private Mono<List<ExpandedResource>> expandResources(YdbTxSession session, List<ResourceModel> resources) {
        return getResourceTypes(session, resources).flatMap(resourceTypes -> getResourceSegmentations(session,
                resources).flatMap(resourceSegmentations -> getResourceSegments(session, resources)
                .flatMap(resourceSegments -> getUnitsEnsembles(session, resources).map(unitsEnsembles -> {
                    List<ExpandedResource> result = new ArrayList<>();
                    Map<String, ResourceTypeModel> resourceTypesById = resourceTypes.stream()
                            .collect(Collectors.toMap(ResourceTypeModel::getId, Function.identity()));
                    Map<String, ResourceSegmentationModel> segmentationsById = resourceSegmentations.stream()
                            .collect(Collectors.toMap(ResourceSegmentationModel::getId, Function.identity()));
                    Map<String, ResourceSegmentModel> segmentById = resourceSegments.stream()
                            .collect(Collectors.toMap(ResourceSegmentModel::getId, Function.identity()));
                    Map<String, UnitsEnsembleModel> ensemblesById = unitsEnsembles.stream()
                            .collect(Collectors.toMap(UnitsEnsembleModel::getId, Function.identity()));
                    resources.forEach(resource -> {
                        ResourceTypeModel resourceType = resourceTypesById.get(resource.getResourceTypeId());
                        Set<ExpandedSegment> segments = new HashSet<>();
                        Optional.ofNullable(resource.getSegments()).orElse(Collections.emptySet()).forEach(segment ->
                                segments.add(new ExpandedSegment(segmentationsById.get(segment.getSegmentationId()),
                                        segmentById.get(segment.getSegmentId()))));
                        UnitsEnsembleModel unitsEnsemble = ensemblesById.get(resource.getUnitsEnsembleId());
                        result.add(new ExpandedResource(resource, resourceType, segments, unitsEnsemble));
                    });
                    return result;
                }))));
    }

    private Mono<List<ExpandedAccountSpace>> expandAccountSpaces(YdbTxSession session,
                                                             List<AccountSpaceModel> accountSpaces) {
        return getAccountSpacesSegmentations(session, accountSpaces).flatMap(resourceSegmentations ->
                getAccountSpacesSegments(session, accountSpaces).map(resourceSegments -> {
                    List<ExpandedAccountSpace> result = new ArrayList<>();
                    Map<String, ResourceSegmentationModel> segmentationsById = resourceSegmentations.stream()
                            .collect(Collectors.toMap(ResourceSegmentationModel::getId, Function.identity()));
                    Map<String, ResourceSegmentModel> segmentById = resourceSegments.stream()
                            .collect(Collectors.toMap(ResourceSegmentModel::getId, Function.identity()));
                    accountSpaces.forEach(accountsSpace -> {
                        Set<ExpandedSegment> segments = new HashSet<>();
                        Optional.ofNullable(accountsSpace.getSegments()).orElse(Collections.emptySet())
                                .forEach(segment -> segments.add(new ExpandedSegment(segmentationsById
                                        .get(segment.getSegmentationId()), segmentById.get(segment.getSegmentId()))));
                        result.add(new ExpandedAccountSpace(accountsSpace, segments));
                    });
                    return result;
                }));
    }

    private Mono<List<ResourceTypeModel>> getResourceTypes(YdbTxSession session, List<ResourceModel> resources) {
        List<String> resourceTypeIds = resources.stream().flatMap(r -> Optional.ofNullable(r.getResourceTypeId())
                .stream()).distinct().collect(Collectors.toList());
        List<Tuple2<String, TenantId>> ids = resourceTypeIds.stream().map(id -> Tuples.of(id,
                Tenants.DEFAULT_TENANT_ID)).collect(Collectors.toList());
        return Flux.fromIterable(Lists.partition(ids, 500))
                .concatMap(v -> resourceTypesDao.getByIds(session, v))
                .collectList().map(l -> l.stream().flatMap(Collection::stream).collect(Collectors.toList()));
    }

    private Mono<List<ResourceSegmentationModel>> getResourceSegmentations(YdbTxSession session,
                                                                           List<ResourceModel> resources) {
        List<String> resourceSegmentationIds = resources.stream().flatMap(r -> Optional.ofNullable(r.getSegments())
                .stream().flatMap(m -> m.stream().map(ResourceSegmentSettingsModel::getSegmentationId)))
                .distinct().collect(Collectors.toList());
        List<Tuple2<String, TenantId>> ids = resourceSegmentationIds.stream().map(id -> Tuples.of(id,
                Tenants.DEFAULT_TENANT_ID)).collect(Collectors.toList());
        return Flux.fromIterable(Lists.partition(ids, 500))
                .concatMap(v -> resourceSegmentationsDao.getByIds(session, v))
                .collectList().map(l -> l.stream().flatMap(Collection::stream).collect(Collectors.toList()));
    }

    private Mono<List<ResourceSegmentationModel>> getAccountSpacesSegmentations(
            YdbTxSession session, List<AccountSpaceModel> accountSpaces) {
        List<String> resourceSegmentationIds = accountSpaces.stream().flatMap(r -> Optional.ofNullable(r.getSegments())
                .stream().flatMap(m -> m.stream().map(ResourceSegmentSettingsModel::getSegmentationId)))
                .distinct().collect(Collectors.toList());
        List<Tuple2<String, TenantId>> ids = resourceSegmentationIds.stream().map(id -> Tuples.of(id,
                Tenants.DEFAULT_TENANT_ID)).collect(Collectors.toList());
        return Flux.fromIterable(Lists.partition(ids, 500))
                .concatMap(v -> resourceSegmentationsDao.getByIds(session, v))
                .collectList().map(l -> l.stream().flatMap(Collection::stream).collect(Collectors.toList()));
    }

    private Mono<List<ResourceSegmentModel>> getResourceSegments(YdbTxSession session, List<ResourceModel> resources) {
        List<String> resourceSegmentationIds = resources.stream().flatMap(r -> Optional.ofNullable(r.getSegments())
                .stream().flatMap(m -> m.stream().map(ResourceSegmentSettingsModel::getSegmentId)))
                .distinct().collect(Collectors.toList());
        List<Tuple2<String, TenantId>> ids = resourceSegmentationIds.stream().map(id -> Tuples.of(id,
                Tenants.DEFAULT_TENANT_ID)).collect(Collectors.toList());
        return Flux.fromIterable(Lists.partition(ids, 500))
                .concatMap(v -> resourceSegmentsDao.getByIds(session, v))
                .collectList().map(l -> l.stream().flatMap(Collection::stream).collect(Collectors.toList()));
    }

    private Mono<List<ResourceSegmentModel>> getAccountSpacesSegments(YdbTxSession session,
                                                                      List<AccountSpaceModel> accountSpaces) {
        List<String> resourceSegmentationIds = accountSpaces.stream().flatMap(r -> Optional.ofNullable(r.getSegments())
                .stream().flatMap(m -> m.stream().map(ResourceSegmentSettingsModel::getSegmentId)))
                .distinct().collect(Collectors.toList());
        List<Tuple2<String, TenantId>> ids = resourceSegmentationIds.stream().map(id -> Tuples.of(id,
                Tenants.DEFAULT_TENANT_ID)).collect(Collectors.toList());
        return Flux.fromIterable(Lists.partition(ids, 500))
                .concatMap(v -> resourceSegmentsDao.getByIds(session, v))
                .collectList().map(l -> l.stream().flatMap(Collection::stream).collect(Collectors.toList()));
    }

    private Mono<List<UnitsEnsembleModel>> getUnitsEnsembles(YdbTxSession session, List<ResourceModel> resources) {
        List<String> unitsEnsembleIds = resources.stream().map(ResourceModel::getUnitsEnsembleId)
                .distinct().collect(Collectors.toList());
        List<Tuple2<String, TenantId>> ids = unitsEnsembleIds.stream().map(id -> Tuples.of(id,
                Tenants.DEFAULT_TENANT_ID)).collect(Collectors.toList());
        return Flux.fromIterable(Lists.partition(ids, 500))
                .concatMap(v -> unitsEnsemblesDao.getByIds(session, v))
                .collectList().map(l -> l.stream().flatMap(Collection::stream).collect(Collectors.toList()));
    }

    private Map<ExternalResourceKey, ExpandedResource> prepareExternalIndex(List<ExpandedResource> resources) {
        Map<ExternalResourceKey, ExpandedResource> result = new HashMap<>();
        resources.forEach(resource -> {
            String resourceTypeKey = resource.getResourceType().getKey();
            Set<ExternalSegmentKey> segments = resource.getResourceSegments()
                    .stream().map(s -> new ExternalSegmentKey(s.getSegmentation()
                            .getKey(), s.getSegment().getKey())).collect(Collectors.toSet());
            ExternalResourceKey externalKey = new ExternalResourceKey(resourceTypeKey, segments);
            result.put(externalKey, resource);
        });
        return result;
    }

    private Map<ExternalAccountsSpaceKey, ExpandedAccountSpace> prepareExternalAccountsSpaceIndex(
            List<ExpandedAccountSpace> accountSpaces) {
        Map<ExternalAccountsSpaceKey, ExpandedAccountSpace> result = new HashMap<>();
        accountSpaces.forEach(accountSpace -> {
            Set<ExternalSegmentKey> segments = accountSpace.getResourceSegments()
                    .stream().map(s -> new ExternalSegmentKey(s.getSegmentation()
                            .getKey(), s.getSegment().getKey())).collect(Collectors.toSet());
            ExternalAccountsSpaceKey externalKey = new ExternalAccountsSpaceKey(segments);
            result.put(externalKey, accountSpace);
        });
        return result;
    }

    public Mono<Void> incrementRetryCounter(YdbTxSession txSession, List<OperationInProgressModel> inProgress) {
        long retryCounter = inProgress.stream().findFirst().map(OperationInProgressModel::getRetryCounter).orElse(0L);
        var ids = inProgress.stream().map(op ->
                        new WithTenant<>(op.getTenantId(), op.getKey())).collect(Collectors.toList());
        return operationsInProgressDao.incrementRetryCounterRetryable(txSession, ids, retryCounter).then();
    }
}
