package ru.yandex.intranet.d.services.operations;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.model.accounts.AccountModel;
import ru.yandex.intranet.d.model.accounts.AccountSpaceModel;
import ru.yandex.intranet.d.model.accounts.AccountsQuotasOperationsModel;
import ru.yandex.intranet.d.model.folders.FolderModel;
import ru.yandex.intranet.d.model.units.UnitModel;
import ru.yandex.intranet.d.model.users.UserModel;
import ru.yandex.intranet.d.services.operations.model.ExpandedAccountSpace;
import ru.yandex.intranet.d.services.operations.model.ExpandedResource;
import ru.yandex.intranet.d.services.operations.model.ExternalAccountsSpaceKey;
import ru.yandex.intranet.d.services.operations.model.ExternalResourceKey;
import ru.yandex.intranet.d.services.operations.model.ExternalSegmentKey;
import ru.yandex.intranet.d.services.operations.model.OperationCommonContext;
import ru.yandex.intranet.d.services.operations.model.ReceivedAccount;
import ru.yandex.intranet.d.services.operations.model.ReceivedAccountsSpaceKey;
import ru.yandex.intranet.d.services.operations.model.ReceivedLastUpdate;
import ru.yandex.intranet.d.services.operations.model.ReceivedMoveProvision;
import ru.yandex.intranet.d.services.operations.model.ReceivedProvision;
import ru.yandex.intranet.d.services.operations.model.ReceivedSegmentKey;
import ru.yandex.intranet.d.services.operations.model.ReceivedUpdatedProvision;
import ru.yandex.intranet.d.services.operations.model.ValidatedReceivedAccount;
import ru.yandex.intranet.d.services.operations.model.ValidatedReceivedLastUpdate;
import ru.yandex.intranet.d.services.operations.model.ValidatedReceivedMoveProvision;
import ru.yandex.intranet.d.services.operations.model.ValidatedReceivedProvision;
import ru.yandex.intranet.d.services.operations.model.ValidatedReceivedUpdatedProvision;
import ru.yandex.intranet.d.util.result.ErrorCollection;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.util.result.TypedError;
import ru.yandex.intranet.d.util.units.Units;

/**
 * Operations retry validation service implementation.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class OperationsRetryValidationService {

    private final MessageSource messages;
    private final OperationsRetryStoreService storeService;

    @SuppressWarnings("ParameterNumber")
    public OperationsRetryValidationService(@Qualifier("messageSource") MessageSource messages,
                                            OperationsRetryStoreService storeService) {
        this.messages = messages;
        this.storeService = storeService;
    }

    public Mono<Result<ValidatedReceivedAccount>> validateReceivedAccount(
            YdbTxSession session, ReceivedAccount account, OperationCommonContext commonContext, Locale locale) {
        String externalAccountIdToLoad = account.getAccountId();
        String folderIdToLoad = account.getFolderId();
        Set<String> userUidsToLoad = new HashSet<>();
        Set<String> userLoginsToLoad = new HashSet<>();
        Set<String> operationIdsToLoad = new HashSet<>();
        account.getLastUpdate().flatMap(ReceivedLastUpdate::getAuthor).ifPresent(a -> {
            a.getPassportUid().ifPresent(userUidsToLoad::add);
            a.getStaffLogin().ifPresent(userLoginsToLoad::add);
        });
        account.getProvisions().forEach(p -> p.getLastUpdate().flatMap(ReceivedLastUpdate::getAuthor).ifPresent(a -> {
            a.getPassportUid().ifPresent(userUidsToLoad::add);
            a.getStaffLogin().ifPresent(userLoginsToLoad::add);
        }));
        account.getLastUpdate().flatMap(ReceivedLastUpdate::getOperationId).ifPresent(operationIdsToLoad::add);
        account.getProvisions().forEach(p -> p.getLastUpdate().flatMap(ReceivedLastUpdate::getOperationId)
                .ifPresent(operationIdsToLoad::add));
        String accountsSpaceId = commonContext.getAccountsSpace().map(AccountSpaceModel::getId).orElse(null);
        return storeService.loadUsers(session, userUidsToLoad, userLoginsToLoad).flatMap(users ->
                storeService.loadOperations(session, operationIdsToLoad).flatMap(operations ->
                        storeService.loadFolder(session, folderIdToLoad).flatMap(folderO ->
                                storeService.loadAccountByExternalId(session, commonContext.getProvider().getId(),
                                        accountsSpaceId, externalAccountIdToLoad)
                                        .map(accountO -> validateReceivedAccount(account, commonContext, users,
                                                operations, folderO.orElse(null), accountO.orElse(null),
                                                locale)))));
    }

    public Mono<Result<List<ValidatedReceivedAccount>>> validateReceivedAccounts(
            YdbTxSession session, List<ReceivedAccount> accounts, OperationCommonContext commonContext,
            Locale locale) {
        Set<String> externalAccountIdsToLoad = accounts.stream()
                .map(ReceivedAccount::getAccountId).collect(Collectors.toSet());
        Set<String> folderIdsToLoad = accounts.stream().map(ReceivedAccount::getFolderId).collect(Collectors.toSet());
        Set<String> userUidsToLoad = new HashSet<>();
        Set<String> userLoginsToLoad = new HashSet<>();
        Set<String> operationIdsToLoad = new HashSet<>();
        accounts.forEach(account -> account.getLastUpdate().flatMap(ReceivedLastUpdate::getAuthor).ifPresent(a -> {
            a.getPassportUid().ifPresent(userUidsToLoad::add);
            a.getStaffLogin().ifPresent(userLoginsToLoad::add);
        }));
        accounts.forEach(account -> account.getProvisions().forEach(p -> p.getLastUpdate()
                .flatMap(ReceivedLastUpdate::getAuthor).ifPresent(a -> {
                    a.getPassportUid().ifPresent(userUidsToLoad::add);
                    a.getStaffLogin().ifPresent(userLoginsToLoad::add);
        })));
        accounts.forEach(account -> account.getLastUpdate().flatMap(ReceivedLastUpdate::getOperationId)
                .ifPresent(operationIdsToLoad::add));
        accounts.forEach(account -> account.getProvisions().forEach(p -> p.getLastUpdate()
                .flatMap(ReceivedLastUpdate::getOperationId).ifPresent(operationIdsToLoad::add)));
        String accountsSpaceId = commonContext.getAccountsSpace().map(AccountSpaceModel::getId).orElse(null);
        return storeService.loadUsers(session, userUidsToLoad, userLoginsToLoad).flatMap(users ->
                storeService.loadOperations(session, operationIdsToLoad).flatMap(operations ->
                        storeService.loadFolders(session, folderIdsToLoad).flatMap(foundFolders ->
                                storeService.loadAccountsByExternalIds(session, commonContext.getProvider().getId(),
                                        accountsSpaceId, externalAccountIdsToLoad).map(foundAccounts ->
                                            validateReceivedAccounts(accounts, commonContext, users, operations,
                                                    foundFolders, foundAccounts, locale)))));
    }

    public Mono<Result<ValidatedReceivedUpdatedProvision>> validateReceivedUpdatedProvision(
            YdbTxSession session, ReceivedUpdatedProvision provision, OperationCommonContext commonContext,
            Locale locale) {
        Set<String> userUidsToLoad = new HashSet<>();
        Set<String> userLoginsToLoad = new HashSet<>();
        Set<String> operationIdsToLoad = new HashSet<>();
        provision.getProvisions().forEach(p -> p.getLastUpdate().flatMap(ReceivedLastUpdate::getAuthor).ifPresent(a -> {
            a.getPassportUid().ifPresent(userUidsToLoad::add);
            a.getStaffLogin().ifPresent(userLoginsToLoad::add);
        }));
        provision.getProvisions().forEach(p -> p.getLastUpdate().flatMap(ReceivedLastUpdate::getOperationId)
                .ifPresent(operationIdsToLoad::add));
        return storeService.loadUsers(session, userUidsToLoad, userLoginsToLoad).flatMap(users ->
                storeService.loadOperations(session, operationIdsToLoad).map(operations ->
                        validateReceivedUpdatedProvision(provision, commonContext, users, operations, locale)));
    }

    public Mono<Result<ValidatedReceivedMoveProvision>> validateReceivedMoveProvision(
            YdbTxSession session, ReceivedMoveProvision receivedMoveProvision,
            OperationCommonContext commonContext, Locale locale
    ) {
        Set<String> userUidsToLoad = new HashSet<>();
        Set<String> userLoginsToLoad = new HashSet<>();
        Set<String> operationIdsToLoad = new HashSet<>();
        receivedMoveProvision.getSourceProvisions().forEach(p -> p.getLastUpdate()
                .flatMap(ReceivedLastUpdate::getAuthor).ifPresent(a -> {
            a.getPassportUid().ifPresent(userUidsToLoad::add);
            a.getStaffLogin().ifPresent(userLoginsToLoad::add);
        }));
        receivedMoveProvision.getDestinationProvisions().forEach(p -> p.getLastUpdate()
                .flatMap(ReceivedLastUpdate::getAuthor).ifPresent(a -> {
                    a.getPassportUid().ifPresent(userUidsToLoad::add);
                    a.getStaffLogin().ifPresent(userLoginsToLoad::add);
                }));
        receivedMoveProvision.getSourceProvisions().forEach(p -> p.getLastUpdate()
                .flatMap(ReceivedLastUpdate::getOperationId).ifPresent(operationIdsToLoad::add));
        receivedMoveProvision.getDestinationProvisions().forEach(p -> p.getLastUpdate()
                .flatMap(ReceivedLastUpdate::getOperationId).ifPresent(operationIdsToLoad::add));
        return storeService.loadUsers(session, userUidsToLoad, userLoginsToLoad).flatMap(users ->
                storeService.loadOperations(session, operationIdsToLoad).map(operations ->
                        validateReceivedMoveProvision(receivedMoveProvision, commonContext, users, operations,
                                locale)));
    }

    private Result<ValidatedReceivedMoveProvision> validateReceivedMoveProvision(
            ReceivedMoveProvision receivedMoveProvision,
            OperationCommonContext commonContext,
            List<UserModel> users,
            List<AccountsQuotasOperationsModel> operations,
            Locale locale
    ) {
        Map<String, UserModel> usersByUid = users.stream()
                .collect(Collectors.toMap(u -> u.getPassportUid().orElseThrow(), Function.identity()));
        Map<String, UserModel> usersByLogin = users.stream()
                .collect(Collectors.toMap(u -> u.getPassportLogin().orElseThrow(), Function.identity()));
        Map<String, AccountsQuotasOperationsModel> operationsById = operations.stream()
                .collect(Collectors.toMap(AccountsQuotasOperationsModel::getOperationId, Function.identity()));
        ValidatedReceivedMoveProvision.Builder builder = ValidatedReceivedMoveProvision.builder();
        ErrorCollection.Builder errors = ErrorCollection.builder();
        validateReceivedMoveProvision(receivedMoveProvision, commonContext, usersByUid, usersByLogin, operationsById,
                builder, errors, "", locale);
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build());
        } else {
            return Result.success(builder.build());
        }
    }

    @SuppressWarnings("ParameterNumber")
    private void validateReceivedMoveProvision(
            ReceivedMoveProvision receivedMoveProvision,
            OperationCommonContext commonContext,
            Map<String, UserModel> usersByUid,
            Map<String, UserModel> usersByLogin,
            Map<String, AccountsQuotasOperationsModel> operationsById,
            ValidatedReceivedMoveProvision.Builder builder,
            ErrorCollection.Builder errors,
            String fieldKeyPrefix,
            Locale locale) {
        receivedMoveProvision.getSourceAccountVersion().ifPresent(builder::sourceAccountVersion);
        receivedMoveProvision.getDestinationAccountVersion().ifPresent(builder::destinationAccountVersion);
        validateAccountsSpace(receivedMoveProvision::getAccountSpaceKey, commonContext, builder::accountSpace, errors,
                fieldKeyPrefix, locale);
        for (int i = 0; i < receivedMoveProvision.getSourceProvisions().size(); i++) {
            ReceivedProvision provision = receivedMoveProvision.getSourceProvisions().get(i);
            ValidatedReceivedProvision.Builder provisionBuilder = ValidatedReceivedProvision.builder();
            ErrorCollection.Builder provisionErrors = ErrorCollection.builder();
            validateProvision(receivedMoveProvision.getAccountSpaceKey().orElse(null), provision, usersByUid,
                    usersByLogin, operationsById, commonContext, provisionBuilder, provisionErrors,
                    fieldKeyPrefix + "sourceProvisions." + i + ".", locale);
            if (provisionErrors.hasAnyErrors()) {
                errors.add(provisionErrors);
            } else {
                builder.addSourceProvision(provisionBuilder.build());
            }
        }
        for (int i = 0; i < receivedMoveProvision.getDestinationProvisions().size(); i++) {
            ReceivedProvision provision = receivedMoveProvision.getDestinationProvisions().get(i);
            ValidatedReceivedProvision.Builder provisionBuilder = ValidatedReceivedProvision.builder();
            ErrorCollection.Builder provisionErrors = ErrorCollection.builder();
            validateProvision(receivedMoveProvision.getAccountSpaceKey().orElse(null), provision, usersByUid,
                    usersByLogin, operationsById, commonContext, provisionBuilder, provisionErrors,
                    fieldKeyPrefix + "destinationProvisions." + i + ".", locale);
            if (provisionErrors.hasAnyErrors()) {
                errors.add(provisionErrors);
            } else {
                builder.addDestinationProvision(provisionBuilder.build());
            }
        }
    }

    private Result<ValidatedReceivedUpdatedProvision> validateReceivedUpdatedProvision(
            ReceivedUpdatedProvision provision,
            OperationCommonContext commonContext,
            List<UserModel> users,
            List<AccountsQuotasOperationsModel> operations,
            Locale locale) {
        Map<String, UserModel> usersByUid = users.stream()
                .collect(Collectors.toMap(u -> u.getPassportUid().get(), Function.identity()));
        Map<String, UserModel> usersByLogin = users.stream()
                .collect(Collectors.toMap(u -> u.getPassportLogin().get(), Function.identity()));
        Map<String, AccountsQuotasOperationsModel> operationsById = operations.stream()
                .collect(Collectors.toMap(AccountsQuotasOperationsModel::getOperationId, Function.identity()));
        ValidatedReceivedUpdatedProvision.Builder builder = ValidatedReceivedUpdatedProvision.builder();
        ErrorCollection.Builder errors = ErrorCollection.builder();
        validateReceivedUpdatedProvision(provision, commonContext, usersByUid, usersByLogin, operationsById, builder,
                errors, "", locale);
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build());
        } else {
            return Result.success(builder.build());
        }
    }

    @SuppressWarnings("ParameterNumber")
    private void validateReceivedUpdatedProvision(
            ReceivedUpdatedProvision receivedProvision,
            OperationCommonContext commonContext,
            Map<String, UserModel> usersByUid,
            Map<String, UserModel> usersByLogin,
            Map<String, AccountsQuotasOperationsModel> operationsById,
            ValidatedReceivedUpdatedProvision.Builder builder,
            ErrorCollection.Builder errors,
            String fieldKeyPrefix,
            Locale locale) {
        receivedProvision.getAccountVersion().ifPresent(builder::accountVersion);
        validateAccountsSpace(receivedProvision::getAccountsSpaceKey, commonContext, builder::accountsSpace, errors,
                fieldKeyPrefix, locale);
        for (int i = 0; i < receivedProvision.getProvisions().size(); i++) {
            ReceivedProvision provision = receivedProvision.getProvisions().get(i);
            ValidatedReceivedProvision.Builder provisionBuilder = ValidatedReceivedProvision.builder();
            ErrorCollection.Builder provisionErrors = ErrorCollection.builder();
            validateProvision(receivedProvision.getAccountsSpaceKey().orElse(null), provision, usersByUid,
                    usersByLogin, operationsById, commonContext, provisionBuilder, provisionErrors,
                    fieldKeyPrefix + "provisions." + i + ".", locale);
            if (provisionErrors.hasAnyErrors()) {
                errors.add(provisionErrors);
            } else {
                builder.addProvision(provisionBuilder.build());
            }
        }
    }

    @SuppressWarnings("ParameterNumber")
    private Result<List<ValidatedReceivedAccount>> validateReceivedAccounts(
            List<ReceivedAccount> receivedAccounts,
            OperationCommonContext commonContext,
            List<UserModel> users,
            List<AccountsQuotasOperationsModel> operations,
            List<FolderModel> folders,
            List<AccountModel> accounts,
            Locale locale) {
        List<ValidatedReceivedAccount> result = new ArrayList<>();
        ErrorCollection.Builder errors = ErrorCollection.builder();
        Map<String, UserModel> usersByUid = users.stream()
                .collect(Collectors.toMap(u -> u.getPassportUid().get(), Function.identity()));
        Map<String, UserModel> usersByLogin = users.stream()
                .collect(Collectors.toMap(u -> u.getPassportLogin().get(), Function.identity()));
        Map<String, AccountsQuotasOperationsModel> operationsById = operations.stream()
                .collect(Collectors.toMap(AccountsQuotasOperationsModel::getOperationId, Function.identity()));
        Map<String, FolderModel> foldersById = folders.stream()
                .collect(Collectors.toMap(FolderModel::getId, Function.identity()));
        Map<String, AccountModel> accountsByExternalId = accounts.stream()
                .collect(Collectors.toMap(AccountModel::getOuterAccountIdInProvider, Function.identity()));
        for (int i = 0; i < receivedAccounts.size(); i++) {
            ReceivedAccount receivedAccount = receivedAccounts.get(i);
            ValidatedReceivedAccount.Builder accountBuilder = ValidatedReceivedAccount.builder();
            ErrorCollection.Builder accountErrors = ErrorCollection.builder();
            FolderModel folder = foldersById.get(receivedAccount.getFolderId());
            AccountModel account = accountsByExternalId.get(receivedAccount.getAccountId());
            validateReceivedAccount(receivedAccount, commonContext, usersByUid, usersByLogin, operationsById, folder,
                    account, accountBuilder, accountErrors, "accounts." + i + ".", locale);
            if (accountErrors.hasAnyErrors()) {
                errors.add(accountErrors);
            } else {
                result.add(accountBuilder.build());
            }
        }
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build());
        } else {
            return Result.success(result);
        }
    }

    @SuppressWarnings("ParameterNumber")
    private Result<ValidatedReceivedAccount> validateReceivedAccount(ReceivedAccount receivedAccount,
                                                                     OperationCommonContext commonContext,
                                                                     List<UserModel> users,
                                                                     List<AccountsQuotasOperationsModel> operations,
                                                                     @Nullable FolderModel folder,
                                                                     @Nullable AccountModel account,
                                                                     Locale locale) {
        Map<String, UserModel> usersByUid = users.stream()
                .collect(Collectors.toMap(u -> u.getPassportUid().get(), Function.identity()));
        Map<String, UserModel> usersByLogin = users.stream()
                .collect(Collectors.toMap(u -> u.getPassportLogin().get(), Function.identity()));
        Map<String, AccountsQuotasOperationsModel> operationsById = operations.stream()
                .collect(Collectors.toMap(AccountsQuotasOperationsModel::getOperationId, Function.identity()));
        ValidatedReceivedAccount.Builder builder = ValidatedReceivedAccount.builder();
        ErrorCollection.Builder errors = ErrorCollection.builder();
        validateReceivedAccount(receivedAccount, commonContext, usersByUid, usersByLogin, operationsById, folder,
                account, builder, errors, "", locale);
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build());
        } else {
            return Result.success(builder.build());
        }
    }

    @SuppressWarnings("ParameterNumber")
    private void validateReceivedAccount(
            ReceivedAccount receivedAccount,
            OperationCommonContext commonContext,
            Map<String, UserModel> usersByUid,
            Map<String, UserModel> usersByLogin,
            Map<String, AccountsQuotasOperationsModel> operationsById,
            @Nullable FolderModel folder,
            @Nullable AccountModel account,
            ValidatedReceivedAccount.Builder builder,
            ErrorCollection.Builder errors,
            String fieldKeyPrefix,
            Locale locale) {
        builder.accountId(receivedAccount.getAccountId());
        receivedAccount.getKey().ifPresent(builder::key);
        receivedAccount.getDisplayName().ifPresent(builder::displayName);
        builder.deleted(receivedAccount.isDeleted());
        if (folder == null) {
            errors.addError(fieldKeyPrefix + "folderId",  TypedError.invalid(messages
                    .getMessage("errors.folder.not.found", null, locale)));
        } else {
            builder.folder(folder);
        }
        receivedAccount.getAccountVersion().ifPresent(builder::accountVersion);
        if (account != null) {
            builder.account(account);
        }
        validateAccountsSpace(receivedAccount::getAccountsSpaceKey, commonContext, builder::accountsSpace, errors,
                fieldKeyPrefix, locale);
        for (int i = 0; i < receivedAccount.getProvisions().size(); i++) {
            ReceivedProvision provision = receivedAccount.getProvisions().get(i);
            ValidatedReceivedProvision.Builder provisionBuilder = ValidatedReceivedProvision.builder();
            ErrorCollection.Builder provisionErrors = ErrorCollection.builder();
            validateProvision(receivedAccount.getAccountsSpaceKey().orElse(null), provision, usersByUid,
                    usersByLogin, operationsById, commonContext, provisionBuilder, provisionErrors,
                    fieldKeyPrefix + "provisions." + i + ".", locale);
            if (provisionErrors.hasAnyErrors()) {
                errors.add(provisionErrors);
            } else {
                builder.addProvision(provisionBuilder.build());
            }
        }
        if (receivedAccount.getLastUpdate().isPresent()) {
            ValidatedReceivedLastUpdate.Builder lastUpdateBuilder = ValidatedReceivedLastUpdate.builder();
            ErrorCollection.Builder lastUpdateErrors = ErrorCollection.builder();
            validateLastUpdate(receivedAccount.getLastUpdate().get(), usersByUid, usersByLogin, operationsById,
                    lastUpdateBuilder, lastUpdateErrors, fieldKeyPrefix + "lastUpdate.", locale);
            if (lastUpdateErrors.hasAnyErrors()) {
                errors.add(lastUpdateErrors);
            } else {
                builder.lastUpdate(lastUpdateBuilder.build());
            }
        }
        builder.freeTier(receivedAccount.isFreeTier());
    }

    private void validateAccountsSpace(Supplier<Optional<ReceivedAccountsSpaceKey>> accountsSpaceKeySupplier,
                                       OperationCommonContext commonContext,
                                       Consumer<AccountSpaceModel> builder,
                                       ErrorCollection.Builder errors,
                                       String fieldKeyPrefix,
                                       Locale locale) {
        Optional<ReceivedAccountsSpaceKey> accountsSpaceO = accountsSpaceKeySupplier.get();
        if (accountsSpaceO.isPresent()) {
            if (commonContext.getProvider().isAccountsSpacesSupported()) {
                Optional<ExpandedAccountSpace> accountsSpace = getTargetAccountsSpace(
                        accountsSpaceO.get(), commonContext.getExternalAccountsSpaceIndex());
                if (accountsSpace.isPresent()) {
                    if (accountsSpace.get().getAccountSpace().getId()
                            .equals(commonContext.getAccountsSpace().get().getId())) {
                        builder.accept(accountsSpace.get().getAccountSpace());
                    } else {
                        errors.addError(fieldKeyPrefix + "accountsSpaceKey", TypedError
                                .invalid(messages.getMessage("errors.accounts.space.mismatch", null, locale)));
                    }
                } else {
                    errors.addError(fieldKeyPrefix + "accountsSpaceKey", TypedError
                            .invalid(messages.getMessage("errors.accounts.space.not.found", null, locale)));
                }
            } else {
                errors.addError(fieldKeyPrefix + "accountsSpaceKey", TypedError
                        .invalid(messages.getMessage("errors.accounts.spaces.is.not.supported", null, locale)));
            }
        } else {
            if (commonContext.getProvider().isAccountsSpacesSupported()) {
                errors.addError(fieldKeyPrefix + "accountsSpaceKey",
                        TypedError.invalid(messages.getMessage("errors.field.is.required", null, locale)));
            }
        }
    }

    @SuppressWarnings("ParameterNumber")
    private void validateLastUpdate(ReceivedLastUpdate lastUpdate,
                                    Map<String, UserModel> usersByUid,
                                    Map<String, UserModel> usersByLogin,
                                    Map<String, AccountsQuotasOperationsModel> operationsById,
                                    ValidatedReceivedLastUpdate.Builder lastUpdateBuilder,
                                    ErrorCollection.Builder lastUpdateErrors,
                                    String fieldKeyPrefix,
                                    Locale locale) {
        lastUpdate.getTimestamp().ifPresent(lastUpdateBuilder::timestamp);
        lastUpdate.getOperationId().ifPresent(lastUpdateBuilder::operationId);
        if (lastUpdate.getOperationId().isPresent()) {
            AccountsQuotasOperationsModel operation = operationsById.get(lastUpdate.getOperationId().get());
            if (operation != null) {
                lastUpdateBuilder.operation(operation);
            }
        }
        if (lastUpdate.getAuthor().isPresent() && (lastUpdate.getAuthor().get().getPassportUid().isPresent()
                || lastUpdate.getAuthor().get().getStaffLogin().isPresent())) {
            UserModel authorByUid = null;
            if (lastUpdate.getAuthor().get().getPassportUid().isPresent()) {
                authorByUid = usersByUid.get(lastUpdate.getAuthor().get().getPassportUid().get());
            }
            UserModel authorByLogin = null;
            if (lastUpdate.getAuthor().get().getStaffLogin().isPresent()) {
                authorByLogin = usersByLogin.get(lastUpdate.getAuthor().get().getStaffLogin().get());
            }
            if ((authorByUid == null && authorByLogin == null) || (authorByUid != null
                    && authorByLogin != null && !authorByUid.getId().equals(authorByLogin.getId()))) {
                lastUpdateErrors.addError(fieldKeyPrefix + "author", TypedError
                        .invalid(messages.getMessage("errors.user.not.found", null, locale)));
            } else {
                lastUpdateBuilder.author(authorByUid != null ? authorByUid : authorByLogin);
            }
        }
    }

    @SuppressWarnings("ParameterNumber")
    private void validateProvision(@Nullable ReceivedAccountsSpaceKey accountsSpaceKey,
                                   ReceivedProvision provision,
                                   Map<String, UserModel> usersByUid,
                                   Map<String, UserModel> usersByLogin,
                                   Map<String, AccountsQuotasOperationsModel> operationsById,
                                   OperationCommonContext commonContext,
                                   ValidatedReceivedProvision.Builder provisionBuilder,
                                   ErrorCollection.Builder provisionErrors,
                                   String fieldKeyPrefix,
                                   Locale locale) {
        provision.getQuotaVersion().ifPresent(provisionBuilder::quotaVersion);
        if (provision.getLastUpdate().isPresent()) {
            ValidatedReceivedLastUpdate.Builder lastUpdateBuilder = ValidatedReceivedLastUpdate.builder();
            ErrorCollection.Builder lastUpdateErrors = ErrorCollection.builder();
            validateLastUpdate(provision.getLastUpdate().get(), usersByUid, usersByLogin, operationsById,
                    lastUpdateBuilder, lastUpdateErrors, fieldKeyPrefix + "lastUpdate.", locale);
            if (lastUpdateErrors.hasAnyErrors()) {
                provisionErrors.add(lastUpdateErrors);
            } else {
                provisionBuilder.lastUpdate(lastUpdateBuilder.build());
            }
        }
        Optional<ExpandedResource> resource = getTargetResource(accountsSpaceKey, provision,
                commonContext.getExternalResourceIndex());
        if (resource.isEmpty()) {
            provisionErrors.addError(fieldKeyPrefix + "resourceKey",  TypedError.invalid(messages
                    .getMessage("errors.resource.not.found", null, locale)));
            return;
        } else {
            provisionBuilder.resource(resource.get().getResource());
        }
        Optional<UnitModel> providedUnit = resource.get().getUnitsEnsemble()
                .unitByKey(provision.getProvidedAmountUnitKey());
        if (providedUnit.isEmpty()) {
            provisionErrors.addError(fieldKeyPrefix + "providedAmountUnitKey",  TypedError.invalid(messages
                    .getMessage("errors.unit.not.found", null, locale)));
        } else {
            Optional<Long> providedAmount = Units.convertFromApi(provision.getProvidedAmount(),
                    resource.get().getResource(), resource.get().getUnitsEnsemble(), providedUnit.get());
            if (providedAmount.isEmpty()) {
                provisionErrors.addError(fieldKeyPrefix + "providedAmount",  TypedError.invalid(messages
                        .getMessage("errors.value.can.not.be.converted.to.base.unit", null, locale)));
            } else {
                provisionBuilder.providedAmount(providedAmount.get());
            }
        }
        Optional<UnitModel> allocatedUnit = resource.get().getUnitsEnsemble()
                .unitByKey(provision.getAllocatedAmountUnitKey());
        if (allocatedUnit.isEmpty()) {
            provisionErrors.addError(fieldKeyPrefix + "allocatedAmountUnitKey",  TypedError.invalid(messages
                    .getMessage("errors.unit.not.found", null, locale)));
        } else {
            Optional<Long> allocatedAmount = Units.convertFromApi(provision.getAllocatedAmount(),
                    resource.get().getResource(), resource.get().getUnitsEnsemble(), allocatedUnit.get());
            if (allocatedAmount.isEmpty()) {
                provisionErrors.addError(fieldKeyPrefix + "allocatedAmount",  TypedError.invalid(messages
                        .getMessage("errors.value.can.not.be.converted.to.base.unit", null, locale)));
            } else {
                provisionBuilder.allocatedAmount(allocatedAmount.get());
            }
        }
    }

    private Optional<ExpandedResource> getTargetResource(@Nullable ReceivedAccountsSpaceKey accountsSpaceKey,
                                                         ReceivedProvision provision,
                                                         Map<ExternalResourceKey, ExpandedResource> externalIndex) {
        String resourceTypeKey = provision.getResourceKey().getResourceTypeKey();
        List<ReceivedSegmentKey> segments = new ArrayList<>();
        if (accountsSpaceKey != null) {
            segments.addAll(accountsSpaceKey.getSegmentation());
        }
        segments.addAll(provision.getResourceKey().getSegmentation());
        Map<String, List<String>> segmentsByKey = segments
                .stream().collect(Collectors.groupingBy(ReceivedSegmentKey::getSegmentationKey,
                        Collectors.mapping(ReceivedSegmentKey::getSegmentKey, Collectors.toList())));
        boolean duplicateSegments = segmentsByKey.values().stream().anyMatch(v -> v.size() > 1);
        if (duplicateSegments) {
            return Optional.empty();
        }
        Set<ExternalSegmentKey> segmentKeys = segments.stream()
                .map(s -> new ExternalSegmentKey(s.getSegmentationKey(), s.getSegmentKey()))
                .collect(Collectors.toSet());
        ExternalResourceKey externalKey = new ExternalResourceKey(resourceTypeKey, segmentKeys);
        if (!externalIndex.containsKey(externalKey)) {
            return Optional.empty();
        }
        return Optional.of(externalIndex.get(externalKey));
    }

    private Optional<ExpandedAccountSpace> getTargetAccountsSpace(
            ReceivedAccountsSpaceKey accountsSpaceKey,
            Map<ExternalAccountsSpaceKey, ExpandedAccountSpace> externalAccountsSpaceIndex) {
        List<ReceivedSegmentKey> segments = accountsSpaceKey.getSegmentation();
        Map<String, List<String>> segmentsByKey = segments
                .stream().collect(Collectors.groupingBy(ReceivedSegmentKey::getSegmentationKey,
                        Collectors.mapping(ReceivedSegmentKey::getSegmentKey, Collectors.toList())));
        boolean duplicateSegments = segmentsByKey.values().stream().anyMatch(v -> v.size() > 1);
        if (duplicateSegments) {
            return Optional.empty();
        }
        Set<ExternalSegmentKey> segmentKeys = segments.stream()
                .map(s -> new ExternalSegmentKey(s.getSegmentationKey(),
                        s.getSegmentKey())).collect(Collectors.toSet());
        ExternalAccountsSpaceKey externalKey = new ExternalAccountsSpaceKey(segmentKeys);
        if (!externalAccountsSpaceIndex.containsKey(externalKey)) {
            return Optional.empty();
        }
        return Optional.of(externalAccountsSpaceIndex.get(externalKey));
    }

}
