package ru.yandex.intranet.d.services.operations;

import java.util.Locale;

import com.yandex.ydb.table.transaction.TransactionMode;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.dao.Tenants;
import ru.yandex.intranet.d.dao.accounts.AccountsQuotasOperationsDao;
import ru.yandex.intranet.d.datasource.model.YdbSession;
import ru.yandex.intranet.d.datasource.model.YdbTableClient;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.model.accounts.AccountsQuotasOperationsModel;
import ru.yandex.intranet.d.services.security.SecurityManagerService;
import ru.yandex.intranet.d.util.Uuids;
import ru.yandex.intranet.d.util.result.ErrorCollection;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.util.result.TypedError;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;

/**
 * Operations service.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class OperationsService {

    private final AccountsQuotasOperationsDao accountsQuotasOperationsDao;
    private  final OperationsRequestLogService operationsRequestsLogService;
    private final SecurityManagerService securityManagerService;
    private final YdbTableClient tableClient;
    private final MessageSource messages;

    public OperationsService(
            AccountsQuotasOperationsDao accountsQuotasOperationsDao,
            OperationsRequestLogService operationsRequestsLogService,
            SecurityManagerService securityManagerService,
            YdbTableClient tableClient,
            @Qualifier("messageSource") MessageSource messages
    ) {
        this.accountsQuotasOperationsDao = accountsQuotasOperationsDao;
        this.operationsRequestsLogService = operationsRequestsLogService;
        this.securityManagerService = securityManagerService;
        this.tableClient = tableClient;
        this.messages = messages;
    }

    public Mono<Result<AccountsQuotasOperationsModel>> getById(String id, YaUserDetails currentUser, Locale locale) {
        return securityManagerService.checkReadPermissions(currentUser, locale)
            .flatMap(res -> res.andThen(v -> validateId(id, locale)).andThenMono(u ->
                tableClient.usingSessionMonoRetryable(session ->
                    accountsQuotasOperationsDao.getById(immediateTx(session), id, Tenants.getTenantId(currentUser))
                ).flatMap(operation -> {
                    if (operation.isEmpty()) {
                        ErrorCollection error = ErrorCollection.builder()
                            .addError(TypedError.notFound(
                                    messages.getMessage("errors.operation.not.found", null, locale))
                            ).build();
                        return Mono.just(Result.failure(error));
                    }
                    AccountsQuotasOperationsModel o = operation.get();
                    return operationsRequestsLogService
                        .getAllByOperationIdMono(o.getOperationId(), o.getTenantId())
                        .map(logs -> Result.success(
                            new AccountsQuotasOperationsModel.Builder(o).setLogs(logs).build()
                        ));
                })
            ));
    }

    private Result<Void> validateId(String resourceId, Locale locale) {
        if (!Uuids.isValidUuid(resourceId)) {
            ErrorCollection error = ErrorCollection.builder().addError(TypedError.notFound(messages
                            .getMessage("errors.operation.not.found", null, locale)))
                    .build();
            return Result.failure(error);
        }
        return Result.success(null);
    }

    private YdbTxSession immediateTx(YdbSession session) {
        return session.asTxCommitRetryable(TransactionMode.STALE_READ_ONLY);
    }

}
