package ru.yandex.intranet.d.services.operations

import kotlinx.coroutines.reactor.awaitSingle
import kotlinx.coroutines.reactor.awaitSingleOrNull
import mu.KotlinLogging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.beans.factory.annotation.Value
import org.springframework.context.MessageSource
import org.springframework.context.annotation.Lazy
import org.springframework.stereotype.Component
import reactor.core.publisher.Mono
import ru.yandex.intranet.d.dao.Tenants
import ru.yandex.intranet.d.dao.accounts.AccountsDao
import ru.yandex.intranet.d.dao.accounts.OperationsInProgressDao
import ru.yandex.intranet.d.dao.folders.FolderDao
import ru.yandex.intranet.d.dao.quotas.QuotasDao
import ru.yandex.intranet.d.dao.resources.ResourcesDao
import ru.yandex.intranet.d.datasource.dbSessionRetryable
import ru.yandex.intranet.d.datasource.model.YdbTableClient
import ru.yandex.intranet.d.datasource.model.YdbTxSession
import ru.yandex.intranet.d.i18n.Locales
import ru.yandex.intranet.d.kotlin.ResourceId
import ru.yandex.intranet.d.kotlin.awaitSingleWithMdc
import ru.yandex.intranet.d.kotlin.elapsed
import ru.yandex.intranet.d.kotlin.mono
import ru.yandex.intranet.d.kotlin.withMDC
import ru.yandex.intranet.d.loaders.providers.ProvidersLoader
import ru.yandex.intranet.d.model.accounts.AccountModel
import ru.yandex.intranet.d.model.accounts.AccountsQuotasOperationsModel
import ru.yandex.intranet.d.model.accounts.OperationErrorCollections
import ru.yandex.intranet.d.model.accounts.OperationErrorKind
import ru.yandex.intranet.d.model.accounts.OperationInProgressModel
import ru.yandex.intranet.d.model.folders.FolderModel
import ru.yandex.intranet.d.model.providers.ProviderModel
import ru.yandex.intranet.d.model.quotas.QuotaModel
import ru.yandex.intranet.d.model.resources.ResourceModel
import ru.yandex.intranet.d.model.transfers.ProvisionTransfer
import ru.yandex.intranet.d.model.transfers.TransferRequestModel
import ru.yandex.intranet.d.services.integration.providers.Response
import ru.yandex.intranet.d.services.integration.providers.isProvisionChangesApplied
import ru.yandex.intranet.d.services.operations.OperationUtils.resolvePostRetryRefreshResult
import ru.yandex.intranet.d.services.operations.OperationUtils.resolveRefreshResult
import ru.yandex.intranet.d.services.operations.OperationUtils.resolveRetryResult
import ru.yandex.intranet.d.services.operations.model.OperationCommonContext
import ru.yandex.intranet.d.services.operations.model.OperationPostRefreshContext
import ru.yandex.intranet.d.services.operations.model.OperationPreRefreshContext
import ru.yandex.intranet.d.services.operations.model.OperationRefreshContext
import ru.yandex.intranet.d.services.operations.model.OperationRetryContext
import ru.yandex.intranet.d.services.operations.model.PostRetryRefreshResult
import ru.yandex.intranet.d.services.operations.model.ProvideReserveOperationPostRefreshContext
import ru.yandex.intranet.d.services.operations.model.ProvideReserveOperationPreRefreshContext
import ru.yandex.intranet.d.services.operations.model.ProvideReserveOperationRefreshContext
import ru.yandex.intranet.d.services.operations.model.ProvideReserveOperationRetryContext
import ru.yandex.intranet.d.services.operations.model.RefreshResult
import ru.yandex.intranet.d.services.operations.model.RetryResult
import ru.yandex.intranet.d.services.operations.model.RetryableOperation
import ru.yandex.intranet.d.services.operations.model.UpdateProvisionContext
import ru.yandex.intranet.d.services.operations.model.ValidatedReceivedAccount
import ru.yandex.intranet.d.services.operations.model.ValidatedReceivedProvision
import ru.yandex.intranet.d.services.provisions.ReserveProvisionCompletionResult
import ru.yandex.intranet.d.services.provisions.ReserveProvisionsService
import ru.yandex.intranet.d.services.quotas.MoveProvisionOperationResult
import ru.yandex.intranet.d.services.quotas.TransferRequestCloser
import ru.yandex.intranet.d.services.transfer.TransferLoansService
import ru.yandex.intranet.d.util.Details
import ru.yandex.intranet.d.util.MdcKey
import ru.yandex.intranet.d.util.result.ErrorCollection
import ru.yandex.intranet.d.util.result.ErrorType
import ru.yandex.intranet.d.util.result.LocalizedErrors
import ru.yandex.intranet.d.util.result.Result
import ru.yandex.intranet.d.util.result.TypedError
import ru.yandex.intranet.d.web.errors.Errors
import ru.yandex.intranet.d.web.security.model.YaUserDetails
import java.time.Instant
import java.util.*

private val logger = KotlinLogging.logger {}

/**
 * Provide reserve operation retry service interface.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
class ProvideReserveOperationRetryService(
    private val accountsDao: AccountsDao,
    private val folderDao: FolderDao,
    private val quotasDao: QuotasDao,
    private val operationsInProgressDao: OperationsInProgressDao,
    private val resourcesDao: ResourcesDao,
    private val providersLoader: ProvidersLoader,
    private val reserveProvisionsService: ReserveProvisionsService,
    private val retryIntegrationService: OperationsRetryIntegrationService,
    private val storeService: OperationsRetryStoreService,
    private val validationService: OperationsRetryValidationService,
    private val operationsObservabilityService: OperationsObservabilityService,
    private val transferLoansService: TransferLoansService,
    private val tableClient: YdbTableClient,
    @Qualifier("messageSource") private val messages: MessageSource,
    @Value("\${providers.client.maxAsyncRetries}") private val maxAsyncRetries: Long,
): BaseOperationRetryService {

    @Autowired
    @Lazy
    lateinit var transferRequestCloser: TransferRequestCloser

    override fun preRefreshOperation(session: YdbTxSession,
                                     operation: RetryableOperation,
                                     context: OperationCommonContext,
                                     now: Instant,
                                     locale: Locale): Mono<Optional<ProvideReserveOperationPreRefreshContext>> {
        return mono {
            val account = accountsDao.getByIdWithDeleted(session, operation.operation.requestedChanges.accountId.get(),
                Tenants.DEFAULT_TENANT_ID).awaitSingle().orElseThrow()
            val folder = folderDao.getById(session, account.folderId, Tenants.DEFAULT_TENANT_ID).awaitSingle()
                .orElseThrow()
            Optional.of(ProvideReserveOperationPreRefreshContext(context, account, folder))
        }
    }

    override fun refreshOperation(operation: RetryableOperation,
                                  context: OperationPreRefreshContext,
                                  now: Instant,
                                  locale: Locale): Mono<Optional<ProvideReserveOperationRefreshContext>> {
        return mono {
            refreshOperationSuspend(operation, context, locale)
        }
    }

    private suspend fun refreshOperationSuspend(
        operation: RetryableOperation,
        context: OperationPreRefreshContext,
        locale: Locale
    ) = withMDC(prepareOperationMDC(operation.operation)) {
        val preRefreshContext = context as ProvideReserveOperationPreRefreshContext
        val receivedAccount = meter({
            retryIntegrationService.getAccountById(
                preRefreshContext.folder, preRefreshContext.account,
                context.commonContext, locale
            ).awaitSingleWithMdc()
        }, "Provide reserve background retry, get account from provider")
        Optional.of(ProvideReserveOperationRefreshContext(preRefreshContext, receivedAccount))
    }

    override fun postRefreshOperation(session: YdbTxSession,
                                      operation: RetryableOperation,
                                      context: OperationRefreshContext,
                                      now: Instant,
                                      locale: Locale): Mono<Optional<ProvideReserveOperationPostRefreshContext>> {
        return mono {
            postRefreshOperationSuspend(session, operation, context, now, locale)
        }
    }

    private suspend fun postRefreshOperationSuspend(
        session: YdbTxSession,
        operation: RetryableOperation,
        context: OperationRefreshContext,
        now: Instant,
        locale: Locale
    ) = withMDC(prepareOperationMDC(operation.operation)) {
        val refreshContext = context as ProvideReserveOperationRefreshContext
        val accountId = refreshContext.preRefreshContext.account.id
        val operationId = operation.operation.operationId
        val updateProvisionContext = storeService.loadUpdateProvisionContext(session, operation.operation)
            .awaitSingle()
        val (refreshResult, validatedAccount, errors) = refreshContext.refreshedAccount.matchSuspend({ response ->
            response.matchSuspend({ receivedAccount, requestId ->
                val validated = validationService.validateReceivedAccount(
                    session, receivedAccount,
                    context.commonContext, locale
                ).awaitSingle()
                validated.matchSuspend({ validatedAccount ->
                    if (isProvisionChangesApplied(
                            validatedAccount,
                            operation.operation,
                            refreshContext.commonContext.provider,
                            updateProvisionContext.account,
                            updateProvisionContext.folder,
                            updateProvisionContext.provisionsByAccountIdResourceId
                        )
                    ) {
                        Triple(RefreshResult.OPERATION_APPLIED, validatedAccount, null)
                    } else {
                        Triple(RefreshResult.OPERATION_NOT_APPLIED, validatedAccount, null)
                    }
                }, { errors ->
                    logger.error {
                        "Failed to validate received account $accountId for operation $operationId"
                    }
                    Triple(RefreshResult.NON_FATAL_ERROR, null, LocalizedErrors(errors, errors))
                })
            }, { ex ->
                logger.error(ex) {
                    "Failed to refresh account $accountId for operation $operationId"
                }
                Triple(
                    RefreshResult.NON_FATAL_ERROR, null,
                    localizedErrors("errors.unexpected.provider.communication.failure", ErrorType.INVALID)
                )
            }, { error, requestId ->
                val flattenError = Errors.flattenProviderErrorResponse(error, null)
                logger.error {
                    "Failed to refresh account $accountId for operation $operationId with response: $flattenError"
                }
                Triple(
                    resolveRefreshResult(error), null, localizedErrors(
                        "errors.provision.update.check.is.not.possible", ErrorType.INVALID,
                        details = flattenError
                    )
                )
            })
        }, { errors ->
            logger.error {
                "Failed to refresh account $accountId for operation $operationId with errors: $errors"
            }
            Triple(RefreshResult.NON_FATAL_ERROR, null, LocalizedErrors(errors, errors))
        })
        if (refreshResult == RefreshResult.OPERATION_APPLIED) {
            completeOperationAfterRefresh(
                session, operation, context.commonContext, updateProvisionContext,
                validatedAccount!!, now
            )
            return@withMDC Optional.empty()
        } else if (refreshResult != RefreshResult.OPERATION_NOT_APPLIED) {
            planToNextRetryOperation(session, operation, errors, now, OperationErrorKind.EXPIRED)
            return@withMDC Optional.empty()
        }
        Optional.of(ProvideReserveOperationPostRefreshContext(refreshContext, updateProvisionContext))
    }

    override fun retryOperation(operation: RetryableOperation,
                                context: OperationPostRefreshContext,
                                now: Instant,
                                locale: Locale): Mono<Optional<ProvideReserveOperationRetryContext>> {
        return mono {
            retryOperationSuspend(operation, context, locale)
        }
    }

    private suspend fun retryOperationSuspend(
        operation: RetryableOperation,
        context: OperationPostRefreshContext,
        locale: Locale
    ) = withMDC(prepareOperationMDC(operation.operation)) {
        val accountOperation = operation.operation
        val postRefreshContext = context as ProvideReserveOperationPostRefreshContext

        val updateProvisionResponse = retryIntegrationService.updateProvision(
            postRefreshContext.updateProvisionContext,
            context.commonContext, accountOperation, locale
        ).awaitSingleWithMdc()

        val (retryResult, receivedUpdatedProvision, retryErrors) = getRetryResult(updateProvisionResponse)

        if (retryResult != RetryResult.SUCCESS) {
            val preRefreshContext = postRefreshContext.preRefreshContext
            val postRetryAccount = retryIntegrationService.getAccountById(
                preRefreshContext.folder,
                preRefreshContext.account, context.commonContext, locale
            ).awaitSingleWithMdc()
            val (postRetryResult, receivedAccount, postRetryErrors) = getPostRetryRefreshResult(postRetryAccount)
            return@withMDC Optional.of(
                ProvideReserveOperationRetryContext(
                    postRefreshContext = postRefreshContext,
                    retryResult = retryResult,
                    updatedProvision = receivedUpdatedProvision,
                    retryErrors = retryErrors,
                    postRetryRefreshAccount = receivedAccount,
                    postRetryRefreshResult = postRetryResult,
                    postRetryRefreshErrors = postRetryErrors,
                )
            )
        }
        return@withMDC Optional.of(
            ProvideReserveOperationRetryContext(
                postRefreshContext = postRefreshContext,
                retryResult = retryResult,
                updatedProvision = receivedUpdatedProvision,
                retryErrors = retryErrors,
            )
        )
    }

    override fun postRetryOperation(
        session: YdbTxSession,
        operation: RetryableOperation,
        context: OperationRetryContext,
        now: Instant,
        locale: Locale
    ): Mono<Void?> {
        return mono {
            postRetryOperationSuspend(session, operation, context, now, locale)
        }
    }

    private suspend fun postRetryOperationSuspend(
        session: YdbTxSession,
        operation: RetryableOperation,
        context: OperationRetryContext,
        now: Instant,
        locale: Locale
    ) = withMDC(prepareOperationMDC(operation.operation)) {
        val retryContext = context as ProvideReserveOperationRetryContext
        val errors = retryContext.postRetryRefreshErrors ?: retryContext.retryErrors
        if (retryContext.retryResult.isPresent) {
            when (retryContext.retryResult.get()) {
                RetryResult.SUCCESS -> {
                    onRetrySuccess(session, operation, retryContext, now, locale)
                }
                RetryResult.CONFLICT -> {
                    resolvePostRetryResult(
                        session, operation, retryContext, OperationErrorKind.FAILED_PRECONDITION,
                        errors, now, locale
                    )
                }
                RetryResult.NON_FATAL_FAILURE -> {
                    planToNextRetryOperation(session, operation, errors, now, OperationErrorKind.EXPIRED)
                }
                RetryResult.FATAL_FAILURE -> {
                    resolvePostRetryResult(
                        session, operation, retryContext, OperationErrorKind.INVALID_ARGUMENT,
                        errors, now, locale
                    )
                }
            }
        }
        null
    }

    private suspend fun completeOperationAfterRefresh(
        session: YdbTxSession,
        operation: RetryableOperation,
        context: OperationCommonContext,
        updateProvisionContext: UpdateProvisionContext,
        validatedAccount: ValidatedReceivedAccount,
        now: Instant
    ) {
        val operationInProgress = operation.inProgress.firstOrNull()
        val resources = context.resourceIndex.mapValues { it.value.resource }
        val unitsEnsembles = context.resourceIndex.values
            .map { it.unitsEnsemble }
            .associateBy { it.id }
        val transferRequestContext = loadTransferRequestContext(session, operation.operation, updateProvisionContext)
        val completionResult = reserveProvisionsService.completeOperationAfterRefresh(
            session, operation.operation, operationInProgress, updateProvisionContext.account,
            updateProvisionContext.folder, updateProvisionContext.provisionsByAccountIdResourceId,
            updateProvisionContext.folderQuotasByResourceId, validatedAccount,
            context.provider, resources, unitsEnsembles, now, null
        )
        if (transferRequestContext != null) {
            finishOperationWithTransferRequest(session, updateProvisionContext, transferRequestContext,
                validatedAccount.provisions, completionResult, operation.operation, now)
        }
        if (completionResult.updatedOperation != null) {
            operationsObservabilityService.observeOperationFinished(completionResult.updatedOperation)
        }
    }

    private fun <T> getRetryResult(result: Result<Response<T>>): Triple<RetryResult, T?, LocalizedErrors?> {
        return result.match({ response ->
            response.match({ res, reqId ->
                Triple(RetryResult.SUCCESS, res, null)
            }, {
                Triple(RetryResult.NON_FATAL_FAILURE, null,
                    localizedErrors("errors.unexpected.provider.communication.failure", ErrorType.INVALID))
            }, { error, reqId ->
                val providerError = Errors.flattenProviderErrorResponse(error, null)
                Triple(resolveRetryResult(error), null,
                    localizedErrors("errors.unexpected.provider.response", ErrorType.BAD_REQUEST,
                        details = providerError))
            })
        }, { errors ->
            Triple(RetryResult.NON_FATAL_FAILURE, null, LocalizedErrors(errors, errors))
        })
    }

    private fun <T> getPostRetryRefreshResult(
        result: Result<Response<T>>
    ): Triple<PostRetryRefreshResult, T?, LocalizedErrors?> {
        return result.match({ response ->
            response.match({ res, reqId ->
                Triple(PostRetryRefreshResult.SUCCESS, res, null)
            }, {
                Triple(PostRetryRefreshResult.NON_FATAL_FAILURE, null,
                    localizedErrors("errors.unexpected.provider.communication.failure", ErrorType.INVALID))
            }, { error, reqId ->
                val providerError = Errors.flattenProviderErrorResponse(error, null)
                Triple(resolvePostRetryRefreshResult(error), null, localizedErrors(
                    "errors.unexpected.provider.response",
                    ErrorType.BAD_REQUEST, details = providerError))
            })
        }, { errors ->
            Triple(PostRetryRefreshResult.NON_FATAL_FAILURE, null, LocalizedErrors(errors, errors))
        })
    }

    private fun localizedErrors(
        errorCode: String,
        errorType: ErrorType,
        details: Any? = null
    ): LocalizedErrors {
        val errorRu = ErrorCollection.builder()
            .addError(TypedError.typedError(messages.getMessage(errorCode, null, errorCode, Locales.RUSSIAN),
                errorType))
        val errorEn = ErrorCollection.builder()
            .addError(TypedError.typedError(messages.getMessage(errorCode, null, errorCode, Locales.ENGLISH),
                errorType))
        if (details != null) {
            errorRu.addDetail(Details.ERROR_FROM_PROVIDER, details)
            errorEn.addDetail(Details.ERROR_FROM_PROVIDER, details)
        }
        return LocalizedErrors(errorRu.build(), errorEn.build())
    }

    private suspend fun resolvePostRetryResult(
        session: YdbTxSession,
        operation: RetryableOperation,
        retryContext: ProvideReserveOperationRetryContext,
        errorKind: OperationErrorKind,
        errors: LocalizedErrors?,
        now: Instant,
        locale: Locale
    ) {
        when (retryContext.postRetryRefreshResult!!) {
            PostRetryRefreshResult.SUCCESS -> {
                onRetryNoSuccessFinish(session, operation, retryContext, now, locale, errorKind)
            }
            PostRetryRefreshResult.NON_FATAL_FAILURE -> {
                planToNextRetryOperation(session, operation, errors, now, errorKind)
            }
            PostRetryRefreshResult.FATAL_FAILURE, PostRetryRefreshResult.UNSUPPORTED -> {
                saveAbortOperation(session, operation.operation, operation.inProgress.firstOrNull(), null, errors,
                    now, null, errorKind)
            }
        }
    }

    private suspend fun onRetrySuccess(
        session: YdbTxSession,
        operation: RetryableOperation,
        retryContext: ProvideReserveOperationRetryContext,
        now: Instant,
        locale: Locale
    ) {
        val accountOperation = operation.operation
        val validatedUpdatedProvision = validationService.validateReceivedUpdatedProvision(
            session, retryContext.updatedProvision, retryContext.commonContext, locale)
            .awaitSingleWithMdc()
        validatedUpdatedProvision.matchSuspend({ response ->
            val commonContext = storeService.loadCommonContext(session, accountOperation).awaitSingle()
            val updateContext = storeService.loadUpdateProvisionContext(session, accountOperation)
                .awaitSingle()
            val requestContext = loadTransferRequestContext(session, accountOperation, updateContext)
            val resources = commonContext.resourceIndex.mapValues { e -> e.value.resource }
            val unitsEnsembles = commonContext.resourceIndex.values
                .map { it.unitsEnsemble }
                .associateBy { it.id }

            val completionResult = reserveProvisionsService.completeOperation(
                session, accountOperation, operation.inProgress.firstOrNull(),
                updateContext.folder, updateContext.account, updateContext.folderQuotasByResourceId,
                updateContext.provisionsByAccountIdResourceId, response, commonContext.provider,
                resources, unitsEnsembles, now, null
            )
            if (requestContext != null) {
                finishOperationWithTransferRequest(session, updateContext, requestContext,
                    response.provisions, completionResult, accountOperation, now)
            }
        }, { error ->
            logger.error {
                "Failed to process provision update response for operation ${accountOperation.operationId}: $error"
            }
            val errors = LocalizedErrors(error, error)
            planToNextRetryOperation(session, operation, errors, now, OperationErrorKind.INVALID_ARGUMENT)
        })
    }

    private suspend fun finishOperationWithTransferRequest(
        session: YdbTxSession,
        updateContext: UpdateProvisionContext,
        transferRequestContext: TransferRequestContext,
        receivedProvisions: List<ValidatedReceivedProvision>,
        completionResult: ReserveProvisionCompletionResult,
        accountOperation: AccountsQuotasOperationsModel,
        now: Instant
    ) {
        val accountId = updateContext.account.id
        val transferRequest = transferRequestContext.transferRequest
        val provisionTransfer = transferRequestContext.provisionTransfer
        val payOff = provisionTransfer.sourceAccountId == accountId
        val loanMeta = transferRequest.loanMeta.orElseThrow()
        val loanId = if (payOff) {
            transferLoansService.processLoanPayOff(
                session, transferRequest, loanMeta, updateContext.account, transferRequestContext.destinationAccount,
                updateContext.provisionsByAccountIdResourceId[updateContext.account.id] ?: mapOf(),
                receivedProvisions.associateBy { it.resource.id }, now
            )
            null
        } else {
            val loanId = transferLoansService.processLoanBorrow(session, transferRequest,
                loanMeta, now, transferRequestContext.sourceAccount, updateContext.account,
                transferRequestContext.sourceFolder, updateContext.folder,
                updateContext.provisionsByAccountIdResourceId[updateContext.account.id] ?: mapOf(),
                receivedProvisions.associateBy { it.resource.id }
            )
            loanId
        }
        val finishTransferRequest = transferRequestCloser.moveProvisionOperationFinished(
            session, transferRequest,
            accountOperation.operationId, MoveProvisionOperationResult(
                RetryResult.SUCCESS,
                mapOf(updateContext.folder.id to setOf(completionResult.operationLogId!!)),
                ErrorCollection.empty(), ErrorCollection.empty()
            ), loanId
        )
        finishTransferRequest()
    }

    private suspend fun loadTransferRequestContext(
        session: YdbTxSession,
        operation: AccountsQuotasOperationsModel,
        updateContext: UpdateProvisionContext,
    ): TransferRequestContext? {
        if (operation.requestedChanges.transferRequestId.isEmpty) {
            return null
        }
        val transferRequestId = operation.requestedChanges.transferRequestId.orElseThrow()
        val transferRequest = meter({
            storeService.getTransferRequest(session, transferRequestId)
                .awaitSingle()
                .orElseThrow()
        }, "Provide reserve background retry, load transfer request")
        val accountId = updateContext.account.id
        val provisionTransfer = transferRequest.parameters.provisionTransfers.find {
            it.sourceAccountId == accountId || it.destinationAccountId == accountId
        }!!
        val sourceAccount = provisionTransfer.sourceAccountId == accountId
        val (anotherAccountId, anotherFolderId) = if (sourceAccount) {
            Pair(provisionTransfer.destinationAccountId, provisionTransfer.destinationFolderId)
        } else {
            Pair(provisionTransfer.sourceAccountId, provisionTransfer.sourceFolderId)
        }
        val anotherAccount = meter({
            accountsDao.getByIdWithDeleted(session, anotherAccountId,
                Tenants.DEFAULT_TENANT_ID).awaitSingle().orElseThrow()
        }, "Provide reserve background retry, load another account")
        val anotherFolder = meter({
            folderDao.getById(session, anotherFolderId,
                Tenants.DEFAULT_TENANT_ID).awaitSingle().orElseThrow()
        }, "Provide reserve background retry, load another folder")
        return if (sourceAccount) {
            TransferRequestContext(transferRequest, provisionTransfer, updateContext.account, anotherAccount,
                updateContext.folder, anotherFolder)
        } else {
            TransferRequestContext(transferRequest, provisionTransfer, anotherAccount, updateContext.account,
                anotherFolder, updateContext.folder)
        }
    }

    private suspend fun onRetryNoSuccessFinish(
        session: YdbTxSession,
        operation: RetryableOperation,
        retryContext: ProvideReserveOperationRetryContext,
        now: Instant,
        locale: Locale,
        errorKind: OperationErrorKind
    ) {
        val validatedRefreshedAccount = validationService.validateReceivedAccount(session,
            retryContext.postRetryRefreshAccount!!, retryContext.commonContext, locale).awaitSingle()
        validatedRefreshedAccount.matchSuspend({ validated ->
            val updateProvisionContext = storeService.loadUpdateProvisionContext(session, operation.operation)
                .awaitSingle()
            val provisionChangesApplied = isProvisionChangesApplied(
                validated, operation.operation, retryContext.commonContext.provider,
                updateProvisionContext.account, updateProvisionContext.folder,
                updateProvisionContext.provisionsByAccountIdResourceId
            )
            if (provisionChangesApplied) {
                completeOperationAfterRefresh(session, operation, retryContext.commonContext, updateProvisionContext, validated, now)
            } else {
                val errors = retryContext.postRetryRefreshErrors ?: retryContext.retryErrors
                saveAbortOperation(session, operation.operation, operation.inProgress.firstOrNull(),
                    updateProvisionContext.folder, updateProvisionContext.folderQuotasByResourceId, null,
                    errors, now, null, errorKind)
            }
        }, { error ->
            logger.error {
                "Failed to process account ${retryContext.preRefreshContext.account.id} refresh response after" +
                    " provision update for operation ${operation.operation.operationId}: ${error}"
            }
            planToNextRetryOperation(session, operation, LocalizedErrors(error, error), now,
                OperationErrorKind.INVALID_ARGUMENT)
        })
    }

    private suspend fun planToNextRetryOperation(
        session: YdbTxSession,
        operation: RetryableOperation,
        errors: LocalizedErrors?,
        now: Instant,
        errorKind: OperationErrorKind
    ) {
        val maxAsyncRetriesReached = operation.inProgress.any { it.retryCounter >= maxAsyncRetries }
        if (maxAsyncRetriesReached) {
            saveAbortOperation(session, operation.operation, operation.inProgress.firstOrNull(), null, errors, now,
                null, errorKind)
        } else {
            operationsObservabilityService.observeOperationTransientFailure(operation.operation)
            storeService.incrementRetryCounter(session, operation.inProgress).awaitSingleOrNull()
        }
    }

    override fun abortOperation(session: YdbTxSession,
                                operation: AccountsQuotasOperationsModel,
                                comment: String,
                                now: Instant,
                                currentUser: YaUserDetails,
                                locale: Locale): Mono<Void?> {
        return mono {
            abortOperationSuspend(operation, comment, session, now, currentUser)
            return@mono null
        }
    }

    private suspend fun abortOperationSuspend(
        operation: AccountsQuotasOperationsModel,
        comment: String,
        session: YdbTxSession,
        now: Instant,
        currentUser: YaUserDetails
    ) = withMDC(prepareOperationMDC(operation)) {
        val errors = ErrorCollection.builder().addError(TypedError.unexpected(comment)).build()
        val localizedErrors = LocalizedErrors(errors, errors)
        val context = prepareBasicOperationContext(session, operation)
        val operationInProgress = meter({
            operationsInProgressDao.getById(
                session,
                OperationInProgressModel.Key(operation.operationId, context.folder.id),
                Tenants.DEFAULT_TENANT_ID
            )
                .awaitSingle().orElse(null)
        }, "Provide reserve background retry, load operation in progress")

        saveAbortOperation(
            session, operation, operationInProgress, context.folder, context.folderQuotas,
            comment, localizedErrors, now, currentUser, OperationErrorKind.ABORTED
        )
    }

    private suspend fun saveAbortOperation(
        session: YdbTxSession,
        operation: AccountsQuotasOperationsModel,
        operationInProgress: OperationInProgressModel?,
        comment: String?,
        errors: LocalizedErrors?,
        now: Instant,
        currentUser: YaUserDetails?,
        errorKind: OperationErrorKind,
    ) {
        val basicContext = prepareBasicOperationContext(session, operation)
        saveAbortOperation(session, operation, operationInProgress, basicContext.folder, basicContext.folderQuotas,
            comment, errors, now, currentUser, errorKind)
    }

    private suspend fun saveAbortOperation(
        session: YdbTxSession,
        operation: AccountsQuotasOperationsModel,
        operationInProgress: OperationInProgressModel?,
        folder: FolderModel,
        folderQuotas: Map<ResourceId, QuotaModel>,
        comment: String?,
        errors: LocalizedErrors?,
        now: Instant,
        currentUser: YaUserDetails?,
        errorKind: OperationErrorKind,
    ) {
        val errorsBuilder = OperationErrorCollections.builder()
        if (errors != null) {
            errorsBuilder
                .addErrorCollection(Locales.RUSSIAN, errors.errorsRu)
                .addErrorCollection(Locale.ENGLISH, errors.errorsEn)
        }
        val rollbackResult = reserveProvisionsService.rollbackOperation(session, operation, operationInProgress,
            folderQuotas, folder, now, comment, errorsBuilder.build(), errorKind, currentUser)!!
        if (operation.requestedChanges.transferRequestId.isPresent) {
            val transferRequestId = operation.requestedChanges.transferRequestId.orElseThrow()
            val transferRequestModel = storeService.getTransferRequest(session, transferRequestId).awaitSingle()
                .orElseThrow { IllegalStateException("Transfer request $transferRequestId not found") }
            val finishTransferRequest = transferRequestCloser.moveProvisionOperationFinished(
                session, transferRequestModel,
                operation.operationId, MoveProvisionOperationResult(
                    RetryResult.NON_FATAL_FAILURE,
                    mapOf(folder.id to setOf(rollbackResult.folderOperationLog.id)),
                    errors?.errorsRu ?: ErrorCollection.empty(),
                    errors?.errorsEn ?: ErrorCollection.empty()
                ),
                null
            )
            finishTransferRequest()
        }
        operationsObservabilityService.observeOperationFinished(rollbackResult.updatedOperation)
    }

    private suspend fun prepareBasicOperationContext(session: YdbTxSession,
                                                     operation: AccountsQuotasOperationsModel): BasicOperationContext {
        val provider = meter({ providersLoader.getProviderByIdImmediate(operation.providerId, Tenants.DEFAULT_TENANT_ID)
            .awaitSingle().orElseThrow() }, "Provide reserve background retry, load provider")
        val account = meter({ accountsDao.getByIdWithDeleted(session, operation.requestedChanges.accountId.get(),
            Tenants.DEFAULT_TENANT_ID).awaitSingle().orElseThrow() }, "Provide reserve background retry, load account")
        val folder = meter({ folderDao.getById(session, account.folderId, Tenants.DEFAULT_TENANT_ID).awaitSingle()
            .orElseThrow() }, "Provide reserve background retry, load folder")
        val accountsSpaceId = account.accountsSpacesId.orElse(null)
        val resources = dbSessionRetryable(tableClient) {
            if (accountsSpaceId == null) {
                meter({ resourcesDao.getAllByProvider(roStaleSingleRetryableCommit(), provider.id, provider.tenantId,
                    true).awaitSingle().filter { it.accountsSpacesId == null } },
                    "Provide reserve background retry, load resources")
            } else {
                meter({ resourcesDao.getAllByProviderAccountsSpace(roStaleSingleRetryableCommit(), provider.id, accountsSpaceId,
                    provider.tenantId, true).awaitSingle() }, "Provide reserve background retry, load resources")
            }
        }!!
        val resourceIds = resources.map { it.id }.toSet()
        val folderQuotas = if (accountsSpaceId == null) {
            meter({ quotasDao.getByFoldersAndProvider(session, listOf(account.folderId), account.tenantId,
                account.providerId).awaitSingle().filter { resourceIds.contains(it.resourceId) }
                .associateBy { it.resourceId } }, "Provide reserve background retry, load quotas")
        } else {
            meter({ quotasDao.getByProviderFoldersResources(session, account.tenantId, setOf(account.folderId),
                account.providerId, resourceIds).awaitSingle().associateBy { it.resourceId } },
                "Provide reserve background retry, load quotas")
        }
        return BasicOperationContext(provider, folder, account, resources, folderQuotas)
    }

    private fun prepareOperationMDC(
        operation: AccountsQuotasOperationsModel
    ): Map<String, String?> {
        return prepareOperationMDC(operation.operationId, operation.requestedChanges.transferRequestId.orElse(null))
    }

    private fun prepareOperationMDC(
        operationId: String,
        transferRequestId: String?
    ): Map<String, String?> {
        return mapOf(MdcKey.COMMON_TRANSFER_REQUEST_ID to transferRequestId, MdcKey.COMMON_OPERATION_ID to operationId)
    }

    private inline fun <T> meter(statement: () -> T,
                                 label: String): T {
        return elapsed(statement)
        { millis, success -> logger.info { "$label: duration = $millis ms, success = $success" } }
    }

    private data class BasicOperationContext(
        val provider: ProviderModel,
        val folder: FolderModel,
        val account: AccountModel,
        val resources: List<ResourceModel>,
        val folderQuotas: Map<ResourceId, QuotaModel>
    )

    private data class TransferRequestContext(
        val transferRequest: TransferRequestModel,
        val provisionTransfer: ProvisionTransfer,
        val sourceAccount: AccountModel,
        val destinationAccount: AccountModel,
        val sourceFolder: FolderModel,
        val destinationFolder: FolderModel,
    )
}
