package ru.yandex.intranet.d.services.operations.model;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

import com.google.common.base.Preconditions;

/**
 * Received account.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public final class ReceivedAccount {

    private final String accountId;
    private final String key;
    private final String displayName;
    private final String folderId;
    private final boolean deleted;
    private final List<ReceivedProvision> provisions;
    private final Long accountVersion;
    private final ReceivedAccountsSpaceKey accountsSpaceKey;
    private final ReceivedLastUpdate lastUpdate;
    private final boolean freeTier;

    @SuppressWarnings("ParameterNumber")
    public ReceivedAccount(String accountId,
                           String key,
                           String displayName,
                           String folderId,
                           boolean deleted,
                           List<ReceivedProvision> provisions,
                           Long accountVersion,
                           ReceivedAccountsSpaceKey accountsSpaceKey,
                           ReceivedLastUpdate lastUpdate,
                           boolean freeTier) {
        this.accountId = accountId;
        this.key = key;
        this.displayName = displayName;
        this.folderId = folderId;
        this.deleted = deleted;
        this.provisions = provisions;
        this.accountVersion = accountVersion;
        this.accountsSpaceKey = accountsSpaceKey;
        this.lastUpdate = lastUpdate;
        this.freeTier = freeTier;
    }

    public static Builder builder() {
        return new Builder();
    }

    public String getAccountId() {
        return accountId;
    }

    public Optional<String> getKey() {
        return Optional.ofNullable(key);
    }

    public Optional<String> getDisplayName() {
        return Optional.ofNullable(displayName);
    }

    public String getFolderId() {
        return folderId;
    }

    public boolean isDeleted() {
        return deleted;
    }

    public List<ReceivedProvision> getProvisions() {
        return provisions;
    }

    public Optional<Long> getAccountVersion() {
        return Optional.ofNullable(accountVersion);
    }

    public Optional<ReceivedAccountsSpaceKey> getAccountsSpaceKey() {
        return Optional.ofNullable(accountsSpaceKey);
    }

    public Optional<ReceivedLastUpdate> getLastUpdate() {
        return Optional.ofNullable(lastUpdate);
    }

    public boolean isFreeTier() {
        return freeTier;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        ReceivedAccount that = (ReceivedAccount) o;
        return deleted == that.deleted && Objects.equals(accountId, that.accountId) &&
                Objects.equals(key, that.key) &&
                Objects.equals(displayName, that.displayName) &&
                Objects.equals(folderId, that.folderId) &&
                Objects.equals(provisions, that.provisions) &&
                Objects.equals(accountVersion, that.accountVersion) &&
                Objects.equals(accountsSpaceKey, that.accountsSpaceKey) &&
                Objects.equals(lastUpdate, that.lastUpdate);
    }

    @Override
    public int hashCode() {
        return Objects.hash(accountId, key, displayName, folderId, deleted, provisions, accountVersion,
                accountsSpaceKey, lastUpdate);
    }

    @Override
    public String toString() {
        return "ReceivedAccount{" +
                "accountId='" + accountId + '\'' +
                ", key='" + key + '\'' +
                ", displayName='" + displayName + '\'' +
                ", folderId='" + folderId + '\'' +
                ", deleted=" + deleted +
                ", provisions=" + provisions +
                ", accountVersion=" + accountVersion +
                ", accountsSpaceKey=" + accountsSpaceKey +
                ", lastUpdate=" + lastUpdate +
                '}';
    }

    public static final class Builder {

        private final List<ReceivedProvision> provisions = new ArrayList<>();

        private String accountId;
        private String key;
        private String displayName;
        private String folderId;
        private boolean deleted = false;

        private Long accountVersion;
        private ReceivedAccountsSpaceKey accountsSpaceKey;
        private ReceivedLastUpdate lastUpdate;
        private boolean freeTier;

        private Builder() {
        }

        public Builder accountId(String accountId) {
            this.accountId = accountId;
            return this;
        }

        public Builder key(String key) {
            this.key = key;
            return this;
        }

        public Builder displayName(String displayName) {
            this.displayName = displayName;
            return this;
        }

        public Builder folderId(String folderId) {
            this.folderId = folderId;
            return this;
        }

        public Builder deleted(boolean deleted) {
            this.deleted = deleted;
            return this;
        }

        public Builder addProvision(ReceivedProvision provision) {
            this.provisions.add(provision);
            return this;
        }

        public Builder addProvisions(Collection<? extends ReceivedProvision> provisions) {
            this.provisions.addAll(provisions);
            return this;
        }

        public Builder accountVersion(long accountVersion) {
            this.accountVersion = accountVersion;
            return this;
        }

        public Builder accountsSpaceKey(ReceivedAccountsSpaceKey accountsSpaceKey) {
            this.accountsSpaceKey = accountsSpaceKey;
            return this;
        }

        public Builder lastUpdate(ReceivedLastUpdate lastUpdate) {
            this.lastUpdate = lastUpdate;
            return this;
        }

        public Builder freeTier(boolean freeTier) {
            this.freeTier = freeTier;
            return this;
        }

        public ReceivedAccount build() {
            Preconditions.checkNotNull(accountId, "AccountId is required");
            Preconditions.checkNotNull(folderId, "FolderId is required");
            return new ReceivedAccount(accountId, key, displayName, folderId, deleted, provisions, accountVersion,
                    accountsSpaceKey, lastUpdate, freeTier);
        }

    }

}
