package ru.yandex.intranet.d.services.operations.model;

import java.util.Objects;
import java.util.Optional;

import com.google.common.base.Preconditions;

/**
 * Received provision.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public final class ReceivedProvision {

    private final ReceivedResourceKey resourceKey;
    private final long providedAmount;
    private final String providedAmountUnitKey;
    private final long allocatedAmount;
    private final String allocatedAmountUnitKey;
    private final ReceivedLastUpdate lastUpdate;
    private final Long quotaVersion;

    @SuppressWarnings("ParameterNumber")
    public ReceivedProvision(ReceivedResourceKey resourceKey,
                             long providedAmount,
                             String providedAmountUnitKey,
                             long allocatedAmount,
                             String allocatedAmountUnitKey,
                             ReceivedLastUpdate lastUpdate,
                             Long quotaVersion) {
        this.resourceKey = resourceKey;
        this.providedAmount = providedAmount;
        this.providedAmountUnitKey = providedAmountUnitKey;
        this.allocatedAmount = allocatedAmount;
        this.allocatedAmountUnitKey = allocatedAmountUnitKey;
        this.lastUpdate = lastUpdate;
        this.quotaVersion = quotaVersion;
    }

    public static Builder builder() {
        return new Builder();
    }

    public ReceivedResourceKey getResourceKey() {
        return resourceKey;
    }

    public long getProvidedAmount() {
        return providedAmount;
    }

    public String getProvidedAmountUnitKey() {
        return providedAmountUnitKey;
    }

    public long getAllocatedAmount() {
        return allocatedAmount;
    }

    public String getAllocatedAmountUnitKey() {
        return allocatedAmountUnitKey;
    }

    public Optional<ReceivedLastUpdate> getLastUpdate() {
        return Optional.ofNullable(lastUpdate);
    }

    public Optional<Long> getQuotaVersion() {
        return Optional.ofNullable(quotaVersion);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        ReceivedProvision provision = (ReceivedProvision) o;
        return providedAmount == provision.providedAmount &&
                allocatedAmount == provision.allocatedAmount &&
                Objects.equals(resourceKey, provision.resourceKey) &&
                Objects.equals(providedAmountUnitKey, provision.providedAmountUnitKey) &&
                Objects.equals(allocatedAmountUnitKey, provision.allocatedAmountUnitKey) &&
                Objects.equals(lastUpdate, provision.lastUpdate) &&
                Objects.equals(quotaVersion, provision.quotaVersion);
    }

    @Override
    public int hashCode() {
        return Objects.hash(resourceKey, providedAmount, providedAmountUnitKey, allocatedAmount,
                allocatedAmountUnitKey, lastUpdate, quotaVersion);
    }

    @Override
    public String toString() {
        return "ReceivedProvision{" +
                "resourceKey=" + resourceKey +
                ", providedAmount=" + providedAmount +
                ", providedAmountUnitKey='" + providedAmountUnitKey + '\'' +
                ", allocatedAmount=" + allocatedAmount +
                ", allocatedAmountUnitKey='" + allocatedAmountUnitKey + '\'' +
                ", lastUpdate=" + lastUpdate +
                ", quotaVersion=" + quotaVersion +
                '}';
    }

    public static final class Builder {

        private ReceivedResourceKey resourceKey;
        private Long providedAmount;
        private String providedAmountUnitKey;
        private Long allocatedAmount;
        private String allocatedAmountUnitKey;
        private ReceivedLastUpdate lastUpdate;
        private Long quotaVersion;

        private Builder() {
        }

        public Builder resourceKey(ReceivedResourceKey resourceKey) {
            this.resourceKey = resourceKey;
            return this;
        }

        public Builder providedAmount(long providedAmount) {
            this.providedAmount = providedAmount;
            return this;
        }

        public Builder providedAmountUnitKey(String providedAmountUnitKey) {
            this.providedAmountUnitKey = providedAmountUnitKey;
            return this;
        }

        public Builder allocatedAmount(long allocatedAmount) {
            this.allocatedAmount = allocatedAmount;
            return this;
        }

        public Builder allocatedAmountUnitKey(String allocatedAmountUnitKey) {
            this.allocatedAmountUnitKey = allocatedAmountUnitKey;
            return this;
        }

        public Builder lastUpdate(ReceivedLastUpdate lastUpdate) {
            this.lastUpdate = lastUpdate;
            return this;
        }

        public Builder quotaVersion(long quotaVersion) {
            this.quotaVersion = quotaVersion;
            return this;
        }

        public ReceivedProvision build() {
            Preconditions.checkNotNull(resourceKey, "ResourceKey is required");
            Preconditions.checkNotNull(providedAmount, "ProvidedAmount is required");
            Preconditions.checkNotNull(providedAmountUnitKey, "ProvidedAmountUnitKey is required");
            Preconditions.checkNotNull(allocatedAmount, "AllocatedAmount is required");
            Preconditions.checkNotNull(allocatedAmountUnitKey, "AllocatedAmountUnitKey is required");
            return new ReceivedProvision(resourceKey, providedAmount, providedAmountUnitKey, allocatedAmount,
                    allocatedAmountUnitKey, lastUpdate, quotaVersion);
        }

    }

}
