package ru.yandex.intranet.d.services.operations.model;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

import com.google.common.base.Preconditions;

import ru.yandex.intranet.d.model.accounts.AccountModel;
import ru.yandex.intranet.d.model.accounts.AccountSpaceModel;
import ru.yandex.intranet.d.model.folders.FolderModel;

/**
 * Validated received account.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public final class ValidatedReceivedAccount {

    private final String accountId;
    private final String key;
    private final String displayName;
    private final boolean deleted;
    private final FolderModel folder;
    private final List<ValidatedReceivedProvision> provisions;
    private final Long accountVersion;
    private final AccountSpaceModel accountsSpace;
    private final ValidatedReceivedLastUpdate lastUpdate;
    private final AccountModel account;
    private final boolean freeTier;

    @SuppressWarnings("ParameterNumber")
    public ValidatedReceivedAccount(String accountId,
                                    String key,
                                    String displayName,
                                    boolean deleted,
                                    FolderModel folder,
                                    List<ValidatedReceivedProvision> provisions,
                                    Long accountVersion,
                                    AccountSpaceModel accountsSpace,
                                    ValidatedReceivedLastUpdate lastUpdate,
                                    AccountModel account, boolean freeTier) {
        this.accountId = accountId;
        this.key = key;
        this.displayName = displayName;
        this.deleted = deleted;
        this.folder = folder;
        this.provisions = provisions;
        this.accountVersion = accountVersion;
        this.accountsSpace = accountsSpace;
        this.lastUpdate = lastUpdate;
        this.account = account;
        this.freeTier = freeTier;
    }

    public static Builder builder() {
        return new Builder();
    }

    public String getAccountId() {
        return accountId;
    }

    public Optional<String> getKey() {
        return Optional.ofNullable(key);
    }

    public Optional<String> getDisplayName() {
        return Optional.ofNullable(displayName);
    }

    public boolean isDeleted() {
        return deleted;
    }

    public FolderModel getFolder() {
        return folder;
    }

    public List<ValidatedReceivedProvision> getProvisions() {
        return provisions;
    }

    public Optional<Long> getAccountVersion() {
        return Optional.ofNullable(accountVersion);
    }

    public Optional<AccountSpaceModel> getAccountsSpace() {
        return Optional.ofNullable(accountsSpace);
    }

    public Optional<ValidatedReceivedLastUpdate> getLastUpdate() {
        return Optional.ofNullable(lastUpdate);
    }

    public Optional<AccountModel> getAccount() {
        return Optional.ofNullable(account);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        ValidatedReceivedAccount that = (ValidatedReceivedAccount) o;
        return deleted == that.deleted &&
                freeTier == that.freeTier &&
                Objects.equals(accountId, that.accountId) &&
                Objects.equals(key, that.key) &&
                Objects.equals(displayName, that.displayName) &&
                Objects.equals(folder, that.folder) &&
                Objects.equals(provisions, that.provisions) &&
                Objects.equals(accountVersion, that.accountVersion) &&
                Objects.equals(accountsSpace, that.accountsSpace) &&
                Objects.equals(lastUpdate, that.lastUpdate) &&
                Objects.equals(account, that.account);
    }

    @Override
    public int hashCode() {
        return Objects.hash(accountId, key, displayName, deleted, folder, provisions, accountVersion, accountsSpace,
                lastUpdate, account, freeTier);
    }

    @Override
    public String toString() {
        return "ValidatedReceivedAccount{" +
                "accountId='" + accountId + '\'' +
                ", key='" + key + '\'' +
                ", displayName='" + displayName + '\'' +
                ", deleted=" + deleted +
                ", folder=" + folder +
                ", provisions=" + provisions +
                ", accountVersion=" + accountVersion +
                ", accountsSpace=" + accountsSpace +
                ", lastUpdate=" + lastUpdate +
                ", account=" + account +
                ", freeTier=" + freeTier +
                '}';
    }

    public static final class Builder {

        private final List<ValidatedReceivedProvision> provisions = new ArrayList<>();

        private String accountId;
        private String key;
        private String displayName;
        private Boolean deleted;
        private FolderModel folder;
        private Long accountVersion;
        private AccountSpaceModel accountsSpace;
        private ValidatedReceivedLastUpdate lastUpdate;
        private AccountModel account;
        private boolean freeTier;

        public Builder accountId(String accountId) {
            this.accountId = accountId;
            return this;
        }

        public Builder key(String key) {
            this.key = key;
            return this;
        }

        public Builder displayName(String displayName) {
            this.displayName = displayName;
            return this;
        }

        public Builder deleted(boolean deleted) {
            this.deleted = deleted;
            return this;
        }

        public Builder folder(FolderModel folder) {
            this.folder = folder;
            return this;
        }

        public Builder accountVersion(Long accountVersion) {
            this.accountVersion = accountVersion;
            return this;
        }

        public Builder accountsSpace(AccountSpaceModel accountsSpace) {
            this.accountsSpace = accountsSpace;
            return this;
        }

        public Builder lastUpdate(ValidatedReceivedLastUpdate lastUpdate) {
            this.lastUpdate = lastUpdate;
            return this;
        }

        public Builder account(AccountModel account) {
            this.account = account;
            return this;
        }

        public Builder addProvision(ValidatedReceivedProvision provision) {
            this.provisions.add(provision);
            return this;
        }

        public Builder addProvisions(Collection<? extends ValidatedReceivedProvision> provisions) {
            this.provisions.addAll(provisions);
            return this;
        }

        public Builder freeTier(boolean freeTier) {
            this.freeTier = freeTier;
            return this;
        }

        public ValidatedReceivedAccount build() {
            Preconditions.checkNotNull(accountId, "AccountId is required");
            Preconditions.checkNotNull(folder, "Folder is required");
            Preconditions.checkNotNull(deleted, "Deleted is required");
            return new ValidatedReceivedAccount(accountId, key, displayName, deleted, folder, provisions,
                    accountVersion, accountsSpace, lastUpdate, account, freeTier);
        }

    }

}
