package ru.yandex.intranet.d.services.operations.model;

import java.util.Objects;
import java.util.Optional;

import com.google.common.base.Preconditions;

import ru.yandex.intranet.d.model.resources.ResourceModel;

/**
 * Validated received provision.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public final class ValidatedReceivedProvision {

    private final ResourceModel resource;
    private final long providedAmount;
    private final long allocatedAmount;
    private final ValidatedReceivedLastUpdate lastUpdate;
    private final Long quotaVersion;

    public ValidatedReceivedProvision(ResourceModel resource,
                                      long providedAmount,
                                      long allocatedAmount,
                                      ValidatedReceivedLastUpdate lastUpdate,
                                      Long quotaVersion) {
        this.resource = resource;
        this.providedAmount = providedAmount;
        this.allocatedAmount = allocatedAmount;
        this.lastUpdate = lastUpdate;
        this.quotaVersion = quotaVersion;
    }

    public static Builder builder() {
        return new Builder();
    }

    public ResourceModel getResource() {
        return resource;
    }

    public long getProvidedAmount() {
        return providedAmount;
    }

    public long getAllocatedAmount() {
        return allocatedAmount;
    }

    public Optional<ValidatedReceivedLastUpdate> getLastUpdate() {
        return Optional.ofNullable(lastUpdate);
    }

    public Optional<Long> getQuotaVersion() {
        return Optional.ofNullable(quotaVersion);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        ValidatedReceivedProvision that = (ValidatedReceivedProvision) o;
        return providedAmount == that.providedAmount &&
                allocatedAmount == that.allocatedAmount &&
                Objects.equals(resource, that.resource) &&
                Objects.equals(lastUpdate, that.lastUpdate) &&
                Objects.equals(quotaVersion, that.quotaVersion);
    }

    @Override
    public int hashCode() {
        return Objects.hash(resource, providedAmount, allocatedAmount, lastUpdate, quotaVersion);
    }

    @Override
    public String toString() {
        return "ValidatedReceivedProvision{" +
                "resource=" + resource +
                ", providedAmount=" + providedAmount +
                ", allocatedAmount=" + allocatedAmount +
                ", lastUpdate=" + lastUpdate +
                ", quotaVersion=" + quotaVersion +
                '}';
    }

    public static final class Builder {

        private ResourceModel resource;
        private Long providedAmount;
        private Long allocatedAmount;
        private ValidatedReceivedLastUpdate lastUpdate;
        private Long quotaVersion;

        private Builder() {
        }

        public Builder resource(ResourceModel resource) {
            this.resource = resource;
            return this;
        }

        public Builder providedAmount(long providedAmount) {
            this.providedAmount = providedAmount;
            return this;
        }

        public Builder allocatedAmount(long allocatedAmount) {
            this.allocatedAmount = allocatedAmount;
            return this;
        }

        public Builder lastUpdate(ValidatedReceivedLastUpdate lastUpdate) {
            this.lastUpdate = lastUpdate;
            return this;
        }

        public Builder quotaVersion(long quotaVersion) {
            this.quotaVersion = quotaVersion;
            return this;
        }

        public ValidatedReceivedProvision build() {
            Preconditions.checkNotNull(resource, "Resource is required");
            Preconditions.checkNotNull(providedAmount, "ProvidedAmount is required");
            Preconditions.checkNotNull(allocatedAmount, "AllocatedAmount is required");
            return new ValidatedReceivedProvision(resource, providedAmount, allocatedAmount,
                    lastUpdate, quotaVersion);
        }

    }

}
