package ru.yandex.intranet.d.services.providers;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import com.google.common.collect.Lists;
import com.yandex.ydb.table.transaction.TransactionMode;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuple2;
import reactor.util.function.Tuples;

import ru.yandex.intranet.d.dao.Tenants;
import ru.yandex.intranet.d.dao.providers.ProvidersDao;
import ru.yandex.intranet.d.dao.resources.ResourcesDao;
import ru.yandex.intranet.d.datasource.model.YdbTableClient;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.loaders.providers.ProvidersLoader;
import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.model.providers.RelatedCoefficient;
import ru.yandex.intranet.d.model.providers.RelatedResourceMapping;
import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.services.security.SecurityManagerService;
import ru.yandex.intranet.d.util.Uuids;
import ru.yandex.intranet.d.util.result.ErrorCollection;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.util.result.TypedError;
import ru.yandex.intranet.d.web.model.providers.PutProviderRelatedResourcesSettingsDto;
import ru.yandex.intranet.d.web.model.providers.PutProviderRelatedResourcesSettingsRequestDto;
import ru.yandex.intranet.d.web.model.providers.PutRelatedResourceDto;
import ru.yandex.intranet.d.web.model.providers.PutRelatedResourcesForResourceDto;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;

/**
 * Provider related resources service implementation.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class ProviderRelatedResourcesService {

    private final ProvidersDao providersDao;
    private final ResourcesDao resourcesDao;
    private final ProvidersLoader providersLoader;
    private final MessageSource messages;
    private final SecurityManagerService securityManagerService;
    private final YdbTableClient tableClient;

    public ProviderRelatedResourcesService(ProvidersDao providersDao,
                                           ResourcesDao resourcesDao,
                                           ProvidersLoader providersLoader,
                                           @Qualifier("messageSource") MessageSource messages,
                                           SecurityManagerService securityManagerService,
                                           YdbTableClient tableClient) {
        this.providersDao = providersDao;
        this.resourcesDao = resourcesDao;
        this.providersLoader = providersLoader;
        this.messages = messages;
        this.securityManagerService = securityManagerService;
        this.tableClient = tableClient;
    }

    public Mono<Result<ProviderModel>> put(String providerId, PutProviderRelatedResourcesSettingsRequestDto putDto,
                                           YaUserDetails currentUser, Locale locale) {
        return securityManagerService.checkWritePermissionsForProvider(currentUser, locale).andThen(v ->
                validateId(providerId, locale)).andThenMono(u -> tableClient.usingSessionMonoRetryable(session ->
                        session.usingCompTxMonoRetryable(TransactionMode.SERIALIZABLE_READ_WRITE,
                                ts -> providersDao.getByIdStartTx(ts, providerId, Tenants.DEFAULT_TENANT_ID),
                                (ts, r) -> validateExists(r.orElse(null), locale)
                                        .andThenMono(p -> securityManagerService.checkWritePermissionsForProvider(
                                                providerId, currentUser, locale, p
                                        )).flatMap(pr -> pr.andThenMono(p -> validatePut(putDto, p, locale, ts))),
                                (ts, r) -> r.match(
                                        m -> {
                                            if (m.getT2()) {
                                                return providersDao.updateProviderRetryable(ts, m.getT1())
                                                        .doOnSuccess(v -> providersLoader.update(m.getT1()))
                                                        .thenReturn(Result.success(m.getT1()));
                                            } else {
                                                return ts.commitTransaction().thenReturn(Result.success(m.getT1()));
                                            }
                                        },
                                        e -> ts.commitTransaction().thenReturn(Result.failure(e))
                                )
                        )));
    }

    private Result<Void> validateId(String providerId, Locale locale) {
        if (!Uuids.isValidUuid(providerId)) {
            ErrorCollection error = ErrorCollection.builder().addError(TypedError.notFound(messages
                    .getMessage("errors.provider.not.found", null, locale)))
                    .build();
            return Result.failure(error);
        }
        return Result.success(null);
    }

    private Result<ProviderModel> validateExists(ProviderModel provider, Locale locale) {
        if (provider == null || provider.isDeleted()) {
            ErrorCollection error = ErrorCollection.builder().addError(TypedError.notFound(messages
                    .getMessage("errors.provider.not.found", null, locale)))
                    .build();
            return Result.failure(error);
        }
        return Result.success(provider);
    }

    private Mono<Result<Tuple2<ProviderModel, Boolean>>> validatePut(
            PutProviderRelatedResourcesSettingsRequestDto putDto, ProviderModel existingProvider,
            Locale locale, YdbTxSession session) {
        Set<String> referencedResourceIds = collectResourceIds(putDto);
        return getResources(session, referencedResourceIds).map(referencedResources -> {
            Map<String, ResourceModel> resourceById = referencedResources.stream()
                    .collect(Collectors.toMap(ResourceModel::getId, Function.identity()));
            ErrorCollection.Builder errors = ErrorCollection.builder();
            ProviderModel.Builder builder = ProviderModel.builder(existingProvider);
            validateVersion(putDto::getProviderVersion, existingProvider::getVersion, builder::version, errors,
                    "providerVersion", locale);
            validateSettings(putDto::getSettings, builder::relatedResourcesByResourceId, errors,
                    "settings", locale, resourceById, existingProvider.getId());
            if (errors.hasAnyErrors()) {
                return Result.failure(errors.build());
            }
            if (builder.hasChanges(existingProvider)) {
                return Result.success(Tuples.of(builder.build(), true));
            }
            return Result.success(Tuples.of(existingProvider, false));
        });
    }

    private void validateSettings(Supplier<Optional<PutProviderRelatedResourcesSettingsDto>> getter,
                                  Consumer<Map<String, RelatedResourceMapping>> setter, ErrorCollection.Builder errors,
                                  String fieldKey, Locale locale, Map<String, ResourceModel> resourceById,
                                  String providerId) {
        Optional<PutProviderRelatedResourcesSettingsDto> valueO = getter.get();
        if (valueO.isEmpty()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        } else {
            PutProviderRelatedResourcesSettingsDto value = valueO.get();
            validateRelatedResourcesByResource(value::getRelatedResourcesByResource, setter, errors,
                    fieldKey + ".relatedResourcesByResource", locale, resourceById, providerId);
        }
    }

    private void validateRelatedResourcesByResource(Supplier<Optional<List<PutRelatedResourcesForResourceDto>>> getter,
                                                    Consumer<Map<String, RelatedResourceMapping>> setter,
                                                    ErrorCollection.Builder errors, String fieldKey, Locale locale,
                                                    Map<String, ResourceModel> resourceById, String providerId) {
        Optional<List<PutRelatedResourcesForResourceDto>> valueO = getter.get();
        if (valueO.isEmpty()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        } else {
            List<PutRelatedResourcesForResourceDto> value = valueO.get();
            boolean hasErrors = false;
            List<String> keyList = value.stream().filter(Objects::nonNull)
                    .flatMap(v -> v.getResourceId().stream()).collect(Collectors.toList());
            Set<String> keySet = new HashSet<>(keyList);
            if (keySet.size() != keyList.size()) {
                errors.addError(fieldKey, TypedError.invalid(messages
                        .getMessage("errors.resource.id.duplicate", null, locale)));
                hasErrors = true;
            }
            Map<String, RelatedResourceMapping> relatedResourcesResult = new HashMap<>();
            for (int i = 0; i < value.size(); i++) {
                PutRelatedResourcesForResourceDto relatedResourcesForResource = value.get(i);
                if (relatedResourcesForResource == null) {
                    errors.addError(fieldKey, TypedError.invalid(messages
                            .getMessage("errors.field.is.required" + "." + i, null, locale)));
                    hasErrors = true;
                    continue;
                }
                Optional<String> resourceId = validateResourceId(relatedResourcesForResource::getResourceId, errors,
                        fieldKey + "." + i + ".resourceId", locale, providerId, resourceById);
                Optional<RelatedResourceMapping> relatedResources = validateRelatedResources(
                        relatedResourcesForResource::getRelatedResources, errors,
                        fieldKey + "." + i + ".relatedResources", locale, providerId, resourceById);
                if (resourceId.isPresent() && relatedResources.isPresent()) {
                    relatedResourcesResult.put(resourceId.get(), relatedResources.get());
                } else {
                    hasErrors = true;
                }
            }
            boolean noSelfLoop = validateNoSelfLoop(relatedResourcesResult, errors, locale);
            if (!noSelfLoop) {
                hasErrors = true;
            }
            if (!hasErrors) {
                setter.accept(relatedResourcesResult);
            }
        }
    }

    private boolean validateNoSelfLoop(Map<String, RelatedResourceMapping> relatedResources,
                                       ErrorCollection.Builder errors, Locale locale) {
        boolean hasSelfLoop = relatedResources.entrySet().stream().anyMatch(mapping ->
                mapping.getValue().getRelatedCoefficientMap().containsKey(mapping.getKey()));
        if (hasSelfLoop) {
            errors.addError(TypedError.invalid(messages
                    .getMessage("errors.related.resources.can.not.have.loop", null, locale)));
        }
        return !hasSelfLoop;
    }

    private void validateVersion(Supplier<Optional<Long>> getter, Supplier<Long> existingGetter,
                                 Consumer<Long> setter, ErrorCollection.Builder errors, String fieldKey,
                                 Locale locale) {
        Optional<Long> value = getter.get();
        Long existingValue = existingGetter.get();
        if (value.isEmpty()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        } else if (!Objects.equals(value.get(), existingValue)) {
            errors.addError(fieldKey, TypedError.versionMismatch(messages
                    .getMessage("errors.version.mismatch", null, locale)));
        } else {
            setter.accept(existingValue + 1L);
        }
    }

    private Optional<String> validateResourceId(Supplier<Optional<String>> getter, ErrorCollection.Builder errors,
                                                String fieldKey, Locale locale, String providerId,
                                                Map<String, ResourceModel> resourceById) {
        Optional<String> valueO = getter.get();
        if (valueO.isEmpty()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
            return Optional.empty();
        }
        String resourceId = valueO.get();
        if (!resourceById.containsKey(resourceId)) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.resource.not.found", null, locale)));
            return Optional.empty();
        }
        ResourceModel resource = resourceById.get(resourceId);
        if (resource.isDeleted() || !resource.getProviderId().equals(providerId)) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.resource.not.found", null, locale)));
            return Optional.empty();
        }
        return Optional.of(resourceId);
    }

    private Optional<RelatedResourceMapping> validateRelatedResources(
            Supplier<Optional<List<PutRelatedResourceDto>>> getter, ErrorCollection.Builder errors,
            String fieldKey, Locale locale, String providerId, Map<String, ResourceModel> resourceById) {
        Optional<List<PutRelatedResourceDto>> valueO = getter.get();
        if (valueO.isEmpty() || valueO.get().isEmpty()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
            return Optional.empty();
        }
        List<PutRelatedResourceDto> value = valueO.get();
        boolean hasErrors = false;
        List<String> keyList = value.stream().filter(Objects::nonNull)
                .flatMap(v -> v.getResourceId().stream()).collect(Collectors.toList());
        Set<String> keySet = new HashSet<>(keyList);
        if (keySet.size() != keyList.size()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.resource.id.duplicate", null, locale)));
            hasErrors = true;
        }
        Map<String, RelatedCoefficient> relatedCoefficientMap = new HashMap<>();
        for (int i = 0; i < value.size(); i++) {
            PutRelatedResourceDto relatedResource = value.get(i);
            if (relatedResource == null) {
                errors.addError(fieldKey, TypedError.invalid(messages
                        .getMessage("errors.field.is.required" + "." + i, null, locale)));
                hasErrors = true;
                continue;
            }
            Optional<String> resourceId = validateResourceId(relatedResource::getResourceId, errors,
                    fieldKey + "." + i + ".resourceId", locale, providerId, resourceById);
            Optional<Long> numerator = validateNumber(relatedResource::getNumerator, errors,
                    fieldKey + "." + i + ".numerator", locale, true);
            Optional<Long> denominator = validateNumber(relatedResource::getDenominator, errors,
                    fieldKey + "." + i + ".denominator", locale, false);
            if (resourceId.isPresent() && numerator.isPresent() && denominator.isPresent()) {
                relatedCoefficientMap.put(resourceId.get(), new RelatedCoefficient(numerator.get(), denominator.get()));
            } else {
                hasErrors = true;
            }
        }
        if (!hasErrors) {
            return Optional.of(new RelatedResourceMapping(relatedCoefficientMap));
        }
        return Optional.empty();
    }

    private Optional<Long> validateNumber(Supplier<Optional<Long>> getter, ErrorCollection.Builder errors,
                                          String fieldKey, Locale locale, boolean zeroAllowed) {
        Optional<Long> valueO = getter.get();
        if (valueO.isEmpty()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
            return Optional.empty();
        }
        long value = valueO.get();
        if (zeroAllowed && value < 0L) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.number.must.be.non.negative", null, locale)));
            return Optional.empty();
        }
        if (!zeroAllowed && value <= 0L) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.number.must.be.positive", null, locale)));
            return Optional.empty();
        }
        return Optional.of(value);
    }

    private Set<String> collectResourceIds(PutProviderRelatedResourcesSettingsRequestDto putDto) {
        Set<String> resourceIds = new HashSet<>();
        putDto.getSettings().flatMap(PutProviderRelatedResourcesSettingsDto::getRelatedResourcesByResource)
                .ifPresent(byResource -> byResource.forEach(forResource -> {
                    if (forResource != null) {
                        forResource.getResourceId().filter(Uuids::isValidUuid).ifPresent(resourceIds::add);
                        forResource.getRelatedResources().ifPresent(resources -> resources.forEach(resource -> {
                            if (resource != null) {
                                resource.getResourceId().filter(Uuids::isValidUuid).ifPresent(resourceIds::add);
                            }
                        }));
                    }
                }));
        return resourceIds;
    }

    private Mono<List<ResourceModel>> getResources(YdbTxSession session, Set<String> resourceIds) {
        List<Tuple2<String, TenantId>> ids = resourceIds.stream().map(id -> Tuples.of(id,
                Tenants.DEFAULT_TENANT_ID)).collect(Collectors.toList());
        return Flux.fromIterable(Lists.partition(ids, 500))
                        .concatMap(v -> resourcesDao.getByIds(session, v))
                        .collectList().map(l -> l.stream().flatMap(Collection::stream).collect(Collectors.toList()));
    }

}
