package ru.yandex.intranet.d.services.providers;

import java.net.URI;
import java.time.Clock;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.RejectedExecutionException;
import java.util.function.Consumer;
import java.util.function.Supplier;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import com.fasterxml.jackson.databind.ObjectReader;
import com.fasterxml.jackson.databind.ObjectWriter;
import com.google.common.net.HostAndPort;
import com.yandex.ydb.table.transaction.TransactionMode;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuple2;
import reactor.util.function.Tuples;

import ru.yandex.intranet.d.dao.Tenants;
import ru.yandex.intranet.d.dao.accounts.AccountsQuotasDao;
import ru.yandex.intranet.d.dao.folders.FolderDao;
import ru.yandex.intranet.d.dao.folders.FolderOperationLogDao;
import ru.yandex.intranet.d.dao.providers.ProvidersDao;
import ru.yandex.intranet.d.dao.quotas.QuotasDao;
import ru.yandex.intranet.d.dao.services.ServicesDao;
import ru.yandex.intranet.d.dao.sync.ProvidersSyncErrorsDao;
import ru.yandex.intranet.d.dao.sync.ProvidersSyncStatusDao;
import ru.yandex.intranet.d.datasource.model.WithTxId;
import ru.yandex.intranet.d.datasource.model.YdbSession;
import ru.yandex.intranet.d.datasource.model.YdbTableClient;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.loaders.providers.ProvidersLoader;
import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.WithTenant;
import ru.yandex.intranet.d.model.accounts.AccountsQuotasModel;
import ru.yandex.intranet.d.model.folders.FolderModel;
import ru.yandex.intranet.d.model.folders.FolderOperationLogModel;
import ru.yandex.intranet.d.model.folders.QuotasByAccount;
import ru.yandex.intranet.d.model.folders.QuotasByResource;
import ru.yandex.intranet.d.model.providers.AccountsSettingsModel;
import ru.yandex.intranet.d.model.providers.BillingMeta;
import ru.yandex.intranet.d.model.providers.ExternalAccountUrlTemplate;
import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.model.providers.ProviderUISettings;
import ru.yandex.intranet.d.model.quotas.QuotaModel;
import ru.yandex.intranet.d.model.services.ServiceMinimalModel;
import ru.yandex.intranet.d.model.services.ServiceState;
import ru.yandex.intranet.d.model.sync.ProvidersSyncErrorsModel;
import ru.yandex.intranet.d.model.sync.ProvidersSyncStatusModel;
import ru.yandex.intranet.d.services.folders.FolderService;
import ru.yandex.intranet.d.services.security.SecurityManagerService;
import ru.yandex.intranet.d.services.sync.AccountsSyncService;
import ru.yandex.intranet.d.util.AggregationAlgorithmHelper;
import ru.yandex.intranet.d.util.AggregationSettingsHelper;
import ru.yandex.intranet.d.util.FeatureStateHelper;
import ru.yandex.intranet.d.util.ObjectMapperHolder;
import ru.yandex.intranet.d.util.Uuids;
import ru.yandex.intranet.d.util.paging.ContinuationTokens;
import ru.yandex.intranet.d.util.paging.Page;
import ru.yandex.intranet.d.util.paging.PageRequest;
import ru.yandex.intranet.d.util.result.ErrorCollection;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.util.result.TypedError;
import ru.yandex.intranet.d.web.model.providers.ExternalAccountUrlTemplateDto;
import ru.yandex.intranet.d.web.model.providers.ProviderCreateDto;
import ru.yandex.intranet.d.web.model.providers.ProviderPutDto;
import ru.yandex.intranet.d.web.model.providers.ProviderSyncErrorDto;
import ru.yandex.intranet.d.web.model.providers.ProviderSyncStatusDto;
import ru.yandex.intranet.d.web.model.providers.ProviderUISettingsDto;
import ru.yandex.intranet.d.web.model.quotas.QuotaSetDto;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;

import static ru.yandex.intranet.d.dao.Tenants.getTenantId;
import static ru.yandex.intranet.d.model.folders.FolderOperationType.PROVISION_AS_QUOTA_BY_ADMIN;
import static ru.yandex.intranet.d.web.model.quotas.QuotaSetDto.Status.OK;

/**
 * Provider service implementation.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class ProvidersService {

    private static final Logger LOG = LoggerFactory.getLogger(ProvidersService.class);
    private static final Set<ServiceState> ALLOWED_SERVICE_STATES = Set.of(ServiceState.DEVELOP,
            ServiceState.SUPPORTED, ServiceState.NEED_INFO);
    private static final int MAX_NAME_LENGTH = 256;
    private static final int MAX_DESCRIPTION_LENGTH = 1024;
    private static final int MAX_URI_LENGTH = 4096;
    private static final int MAX_KEY_LENGTH = 256;
    private static final Pattern PATTERN_COMMA = Pattern.compile(",");
    private static final int FOLDERS_IN_PREFETCH = 100;

    private final ProvidersDao providersDao;
    private final ServicesDao servicesDao;
    private final QuotasDao quotasDao;
    private final AccountsQuotasDao accountsQuotasDao;
    private final FolderOperationLogDao folderOperationLogDao;
    private final YdbTableClient tableClient;
    private final ProvidersLoader providersLoader;
    private final MessageSource messages;
    private final SecurityManagerService securityManagerService;
    private final ObjectReader continuationTokenReader;
    private final ObjectWriter continuationTokenWriter;
    private final AccountsSyncService accountsSyncService;
    private final ProvidersSyncStatusDao providersSyncStatusDao;
    private final ProvidersSyncErrorsDao providersSyncErrorsDao;
    private final FolderDao folderDao;
    private final FolderService folderService;
    private final ThreadPoolTaskExecutor putProvisionsAsQuotasExecutor;

    @SuppressWarnings("ParameterNumber")
    public ProvidersService(ProvidersDao providersDao,
                            ServicesDao servicesDao,
                            QuotasDao quotasDao,
                            AccountsQuotasDao accountsQuotasDao,
                            FolderOperationLogDao folderOperationLogDao,
                            YdbTableClient tableClient,
                            ProvidersLoader providersLoader,
                            @Qualifier("messageSource") MessageSource messages,
                            SecurityManagerService securityManagerService,
                            @Qualifier("continuationTokensJsonObjectMapper") ObjectMapperHolder objectMapper,
                            AccountsSyncService accountsSyncService,
                            ProvidersSyncStatusDao providersSyncStatusDao,
                            ProvidersSyncErrorsDao providersSyncErrorsDao,
                            FolderDao folderDao,
                            FolderService folderService,
                            @Qualifier("putProvisionsAsQuotasExecutor")
                                        ThreadPoolTaskExecutor putProvisionsAsQuotasExecutor) {
        this.providersDao = providersDao;
        this.servicesDao = servicesDao;
        this.quotasDao = quotasDao;
        this.accountsQuotasDao = accountsQuotasDao;
        this.folderOperationLogDao = folderOperationLogDao;
        this.tableClient = tableClient;
        this.providersLoader = providersLoader;
        this.messages = messages;
        this.securityManagerService = securityManagerService;
        this.continuationTokenReader = objectMapper.getObjectMapper().readerFor(ProviderContinuationToken.class);
        this.continuationTokenWriter = objectMapper.getObjectMapper().writerFor(ProviderContinuationToken.class);
        this.accountsSyncService = accountsSyncService;
        this.providersSyncStatusDao = providersSyncStatusDao;
        this.providersSyncErrorsDao = providersSyncErrorsDao;
        this.folderDao = folderDao;
        this.folderService = folderService;
        this.putProvisionsAsQuotasExecutor = putProvisionsAsQuotasExecutor;
    }

    public Mono<Result<ProviderModel>> getById(String id, YaUserDetails currentUser, Locale locale) {
        return securityManagerService.checkReadPermissions(currentUser, locale).flatMap(res -> res.andThen(v ->
                validateId(id, locale)).andThenMono(u ->
                tableClient.usingSessionMonoRetryable(session -> providersDao
                        .getById(immediateTx(session), id, Tenants.getTenantId(currentUser))
                        .map(r -> validateExists(r.orElse(null), locale))
                )
        ));
    }

    public Mono<Result<Page<ProviderModel>>> getPage(PageRequest pageRequest,
                                                     YaUserDetails currentUser, Locale locale) {
        return securityManagerService.checkReadPermissions(currentUser, locale).flatMap(res -> res.andThenMono(u -> {
            Result<PageRequest.Validated<ProviderContinuationToken>> pageValidation
                    = pageRequest.validate(continuationTokenReader, messages, locale);
            return pageValidation.andThenDo(p -> validateContinuationToken(p, locale)).andThenMono(p -> {
                int limit = p.getLimit();
                String fromId = p.getContinuationToken().map(ProviderContinuationToken::getId).orElse(null);
                return tableClient.usingSessionMonoRetryable(session ->
                        providersDao
                                .getByTenant(immediateTx(session),
                                        Tenants.DEFAULT_TENANT_ID, fromId, limit + 1, false)
                                .map(values -> values.size() > limit
                                        ? Page.page(values.subList(0, limit),
                                        prepareToken(values.get(limit - 1)))
                                        : Page.lastPage(values))
                                .map(Result::success));
            });
        }));
    }

    public Mono<Result<List<ProviderModel>>> getAllProviders(
            boolean withDefaultQuotas, YaUserDetails currentUser, Locale locale
    ) {
        TenantId tenantId = getTenantId(currentUser);
        return securityManagerService.checkReadPermissions(currentUser, locale).flatMap(res -> res.andThenMono(u ->
                tableClient.usingSessionMonoRetryable(session ->
                        providersDao.getAllByTenant(immediateTx(session), tenantId, false)
                                .map(providersWithTxId -> {
                                    var providers = providersWithTxId.get();
                                    providers = withDefaultQuotas ? providers.stream()
                                            .filter(ProviderModel::hasDefaultQuotas).collect(Collectors.toList()) :
                                            providers;
                                    return Result.success(providers);
                                })
                )
        ));
    }

    public Mono<Result<ProviderModel>> setReadOnly(String id, boolean readOnly,
                                                   YaUserDetails currentUser, Locale locale) {
        return securityManagerService.checkWritePermissionsForProvider(currentUser, locale).andThen(v ->
                validateId(id, locale)).andThenMono(u ->
                tableClient.usingSessionMonoRetryable(session ->
                        session.usingCompTxMonoRetryable(TransactionMode.SERIALIZABLE_READ_WRITE,
                                ts -> providersDao.getByIdStartTx(ts, id, Tenants.DEFAULT_TENANT_ID),
                                (ts, r) -> validateExists(r.orElse(null), locale)
                                        .andThenMono(p -> securityManagerService.checkWritePermissionsForProvider(
                                                id, currentUser, locale, p
                                        ))
                                        .map(result -> result.andThen(p -> validateSetReadOnly(p, readOnly, locale))),
                                (ts, r) -> r.match(
                                        m -> {
                                            if (m.getT2()) {
                                                return providersDao.updateProviderRetryable(ts, m.getT1())
                                                        .doOnSuccess(v -> providersLoader.update(m.getT1()))
                                                        .thenReturn(Result.success(m.getT1()));
                                            } else {
                                                return ts.commitTransaction().thenReturn(Result.success(m.getT1()));
                                            }
                                        },
                                        e -> ts.commitTransaction().thenReturn(Result.failure(e))
                                )
                        )
                )
        );
    }

    public Mono<Result<ProviderModel>> create(ProviderCreateDto provider,
                                              YaUserDetails currentUser, Locale locale) {
        return securityManagerService.checkDictionaryWritePermissions(currentUser, locale).andThenMono(u -> {
            String newId = UUID.randomUUID().toString();
            String newReserveFolderId = UUID.randomUUID().toString();
            return tableClient.usingSessionMonoRetryable(session ->
                    session.usingTxMonoRetryable(TransactionMode.SERIALIZABLE_READ_WRITE, ts ->
                            validateCreation(provider, newId, locale, ts)
                                    .flatMap(r -> r.andThenMono(validated ->
                                            addReserveFolderToProviderModel(validated, newReserveFolderId, ts, locale)))
                                    .flatMap(r -> r.andThenMono(validated ->
                                                    providersDao.upsertProviderRetryable(ts, validated)
                                                            .doOnSuccess(v -> providersLoader.update(validated))
                                                            .thenReturn(Result.success(validated))
                                            )
                                    )
                    )
            );
        });
    }

    public Mono<Result<ProviderModel>> put(String id, Long version, ProviderPutDto providerPutDto,
                                           YaUserDetails currentUser, Locale locale) {
        String newReserveFolderId = UUID.randomUUID().toString();
        return securityManagerService.checkDictionaryWritePermissions(currentUser, locale).andThen(v ->
                validateId(id, locale)).andThenMono(u ->
                tableClient.usingSessionMonoRetryable(session ->
                        session.usingCompTxMonoRetryable(TransactionMode.SERIALIZABLE_READ_WRITE,
                                ts -> providersDao.getByIdStartTx(ts, id, Tenants.DEFAULT_TENANT_ID),
                                (ts, r) -> validateExists(r.orElse(null), locale)
                                        .andThenMono(p -> validatePut(p, providerPutDto, version, locale, ts))
                                        .flatMap(res -> res.andThenMono(t ->
                                                addReserveFolderToProviderModel(t.getT1(), newReserveFolderId, ts,
                                                        locale).map(res2 -> res2.andThen(p ->
                                                                Result.success(Tuples.of(p, t.getT2())))))),
                                (ts, r) -> r.match(
                                        m -> {
                                            if (m.getT2()) {
                                                return providersDao.updateProviderRetryable(ts, m.getT1())
                                                        .doOnSuccess(v -> providersLoader.update(m.getT1()))
                                                        .thenReturn(Result.success(m.getT1()));
                                            } else {
                                                return ts.commitTransaction().thenReturn(Result.success(m.getT1()));
                                            }
                                        },
                                        e -> ts.commitTransaction().thenReturn(Result.failure(e))
                                )
                        )
                )
        );
    }

    public Mono<Result<ProviderModel>> mergeUiSettings(
            String providerId, Long version, ProviderUISettings uiSettings,
            YaUserDetails currentUser, Locale locale
    ) {
        String newReserveFolderId = UUID.randomUUID().toString();
        return securityManagerService.checkWritePermissionsForProvider(currentUser, locale).andThen(v ->
                validateId(providerId, locale)).andThenMono(u ->
                tableClient.usingSessionMonoRetryable(session ->
                        session.usingCompTxMonoRetryable(TransactionMode.SERIALIZABLE_READ_WRITE,
                                ts -> providersDao.getByIdStartTx(ts, providerId, Tenants.DEFAULT_TENANT_ID),
                                (ts, r) -> validateExists(r.orElse(null), locale)
                                        .andThen(p -> mergeUISettings(p, uiSettings, version, locale))
                                        .andThenMono(t ->
                                                addReserveFolderToProviderModel(t.getT1(), newReserveFolderId, ts,
                                                        locale).map(res2 -> res2.andThen(p ->
                                                                Result.success(Tuples.of(p, t.getT2()))))),
                                (ts, r) -> r.match(
                                        m -> {
                                            if (m.getT2()) {
                                                return providersDao.updateProviderRetryable(ts, m.getT1())
                                                        .doOnSuccess(v -> providersLoader.update(m.getT1()))
                                                        .thenReturn(Result.success(m.getT1()));
                                            } else {
                                                return ts.commitTransaction().thenReturn(Result.success(m.getT1()));
                                            }
                                        },
                                        e -> ts.commitTransaction().thenReturn(Result.failure(e))
                                )
                        )
                )
        );
    }

    public Mono<Result<ProviderModel>> mergeExternalAccountUrlTemplate(
            String providerId, Long version, List<ExternalAccountUrlTemplateDto> externalAccountUrlTemplateDto,
            YaUserDetails currentUser, Locale locale
    ) {
        return securityManagerService.checkWritePermissionsForProvider(currentUser, locale).andThen(v ->
                validateId(providerId, locale)).andThenMono(u ->
                tableClient.usingSessionMonoRetryable(session ->
                        session.usingCompTxMonoRetryable(TransactionMode.SERIALIZABLE_READ_WRITE,
                                ts -> providersDao.getByIdStartTx(ts, providerId, Tenants.DEFAULT_TENANT_ID),
                                (ts, r) -> validateExists(r.orElse(null), locale)
                                        .andThenMono(p -> Mono.just(mergeExternalAccountUrlTemplate(
                                                p, externalAccountUrlTemplateDto, version, locale))),
                                (ts, r) -> r.match(
                                        m -> {
                                            if (m.getT2()) {
                                                return providersDao.updateProviderRetryable(ts, m.getT1())
                                                        .doOnSuccess(v -> providersLoader.update(m.getT1()))
                                                        .thenReturn(Result.success(m.getT1()));
                                            } else {
                                                return ts.commitTransaction().thenReturn(Result.success(m.getT1()));
                                            }
                                        },
                                        e -> ts.commitTransaction().thenReturn(Result.failure(e))
                                )
                        )
                )
        );
    }

    public Mono<Result<FolderModel>> createReserveFolder(String providerId, YaUserDetails currentUser, Locale locale) {
        String newFolderId = UUID.randomUUID().toString();
        return securityManagerService.checkDictionaryWritePermissions(currentUser, locale).andThenMono(u ->
                tableClient.usingSessionMonoRetryable(session ->
                        session.usingCompTxMonoRetryable(TransactionMode.SERIALIZABLE_READ_WRITE,
                                ts -> providersDao.getByIdStartTx(ts, providerId, Tenants.DEFAULT_TENANT_ID),
                                (ts, r) -> validateExists(r.orElse(null), locale)
                                        .andThenMono(provider -> {
                                            if (!provider.isManaged()) {
                                                return Mono.just(Result.failure(ErrorCollection.builder()
                                                        .addError(TypedError.invalid(messages.getMessage(
                                                                "errors.provider.is.not.managed",
                                                                null, locale))).build()));
                                            }
                                            if (provider.getReserveFolderId().isPresent()) {
                                                return Mono.just(Result.failure(ErrorCollection.builder()
                                                        .addError(TypedError.invalid(messages.getMessage(
                                                                "errors.provider.reserved.folder.already.exists",
                                                                null, locale))).build()));
                                            }
                                            return Mono.just(Result.success(provider));
                                        })
                                        .flatMap(result -> result.andThenMono(provider ->
                                                folderService.createReservedFolder(newFolderId, provider.getServiceId(),
                                                        provider.getTenantId(), ts, locale)
                                            .flatMap(result2 -> result2.andThenMono(folder -> {
                                                ProviderModel updatedProvider = ProviderModel.builder(provider)
                                                        .reserveFolderId(folder.getId()).build();
                                                return providersDao.updateProviderRetryable(ts, updatedProvider)
                                                        .doOnSuccess(v -> providersLoader.update(updatedProvider))
                                                        .thenReturn(Result.success(folder));
                                        })))),
                                (ts, r) -> r.match(s -> ts.commitTransaction().thenReturn(Result.success(s)),
                                        e -> ts.commitTransaction().thenReturn(Result.failure(e)))
                        )
                )
        );
    }

    private Mono<Result<ProviderModel>> addReserveFolderToProviderModel(ProviderModel provider,
                                                                        String newReserveFolderId,
                                                                        YdbTxSession ts,
                                                                        Locale locale) {
        if (!provider.isManaged() || provider.getReserveFolderId().isPresent()) {
            return Mono.just(Result.success(provider));
        }
        TenantId tenantId = provider.getTenantId();
        long serviceId = provider.getServiceId();

        return folderDao.getAllReservedFoldersByServiceIds(ts,
                Collections.singletonList(new WithTenant<>(tenantId, serviceId)))
                .flatMap(folders -> {
                    if (folders.isEmpty()) {
                        return folderService.createReservedFolder(newReserveFolderId, serviceId, tenantId, ts, locale);
                    }
                    return Mono.just(Result.success(folders.get(0)));
                })
                .map(r -> r.andThen(f -> Result.success(
                        ProviderModel.builder(provider).reserveFolderId(f.getId()).build())));
    }

    private String prepareToken(ProviderModel lastItem) {
        return ContinuationTokens.encode(new ProviderContinuationToken(lastItem.getId()),
                continuationTokenWriter);
    }

    private Result<Void> validateId(String resourceId, Locale locale) {
        if (!Uuids.isValidUuid(resourceId)) {
            ErrorCollection error = ErrorCollection.builder().addError(TypedError.notFound(messages
                    .getMessage("errors.provider.not.found", null, locale)))
                    .build();
            return Result.failure(error);
        }
        return Result.success(null);
    }

    private Result<Void> validateContinuationToken(PageRequest.Validated<ProviderContinuationToken> pageRequest,
                                                   Locale locale) {
        if (pageRequest.getContinuationToken().isEmpty()) {
            return Result.success(null);
        }
        return validateId(pageRequest.getContinuationToken().get().getId(), locale);
    }

    private Result<ProviderModel> validateExists(ProviderModel provider, Locale locale) {
        if (provider == null || provider.isDeleted()) {
            ErrorCollection error = ErrorCollection.builder().addError(TypedError.notFound(messages
                    .getMessage("errors.provider.not.found", null, locale)))
                    .build();
            return Result.failure(error);
        }
        return Result.success(provider);
    }

    private Result<Tuple2<ProviderModel, Boolean>> validateSetReadOnly(ProviderModel provider,
                                                                       boolean readOnly, Locale locale) {
        if (!provider.isManaged() && !readOnly) {
            ErrorCollection error = ErrorCollection.builder().addError("abcServiceId", TypedError.invalid(messages
                    .getMessage("errors.non.managed.provider.must.be.read.only", null, locale)))
                    .build();
            return Result.failure(error);
        }
        if (provider.isReadOnly() == readOnly) {
            return Result.success(Tuples.of(provider, false));
        }
        return Result.success(Tuples.of(ProviderModel.builder(provider)
                .readOnly(readOnly)
                .version(provider.getVersion() + 1L)
                .build(), true));
    }

    private YdbTxSession immediateTx(YdbSession session) {
        return session.asTxCommitRetryable(TransactionMode.STALE_READ_ONLY);
    }

    private Mono<Result<ProviderModel>> validateCreation(ProviderCreateDto provider, String newId,
                                                         Locale locale, YdbTxSession session) {
        return validateAbcService(provider::getAbcServiceId, locale, session).flatMap(serviceR ->
                validateKey(session, provider::getKey, "key", locale).map(keyR -> {
                    ProviderModel.Builder builder = ProviderModel.builder();
                    AccountsSettingsModel.Builder accountSettingsBuilder = AccountsSettingsModel.builder();
                    BillingMeta.Builder billingMetaBuilder = BillingMeta.builder();
                    ErrorCollection.Builder errors = ErrorCollection.builder();
                    builder.id(newId);
                    builder.version(0L);
                    builder.tenantId(Tenants.DEFAULT_TENANT_ID);
                    builder.deleted(false);
                    serviceR.doOnFailure(errors::add);
                    serviceR.doOnSuccess(s -> builder.serviceId(s.getId()));
                    keyR.doOnFailure(errors::add);
                    keyR.doOnSuccess(builder::key);
                    validateText(provider::getNameEn, builder::nameEn, errors, "nameEn",
                            MAX_NAME_LENGTH, locale);
                    validateText(provider::getNameRu, builder::nameRu, errors, "nameRu",
                            MAX_NAME_LENGTH, locale);
                    validateText(provider::getDescriptionEn, builder::descriptionEn, errors, "descriptionEn",
                            MAX_DESCRIPTION_LENGTH, locale);
                    validateText(provider::getDescriptionRu, builder::descriptionRu, errors, "descriptionRu",
                            MAX_DESCRIPTION_LENGTH, locale);
                    validateRestUri(provider::getRestApiUri, builder::restApiUri, errors, "restApiUri",
                            MAX_URI_LENGTH, locale);
                    validateGrpcUri(provider::getGrpcApiUri, builder::grpcApiUri, errors, "grpcApiUri",
                            MAX_URI_LENGTH, locale);
                    validateTvmId(provider::getSourceTvmId, builder::sourceTvmId, errors,
                            "sourceTvmId", locale);
                    validateTvmId(provider::getDestinationTvmId, builder::destinationTvmId, errors,
                            "destinationTvmId", locale);
                    validateRequired(provider::getReadOnly, builder::readOnly, errors, "readOnly", locale);
                    validateRequired(provider::getMultipleAccountsPerFolder, builder::multipleAccountsPerFolder,
                            errors, "multipleAccountsPerFolder", locale);
                    validateRequired(provider::getAccountTransferWithQuota, builder::accountTransferWithQuota,
                            errors, "accountTransferWithQuota", locale);
                    validateRequired(provider::getManaged, builder::managed, errors,
                            "managed", locale);
                    validateRequired(provider::getAccountsDisplayNameSupported,
                            accountSettingsBuilder::displayNameSupported, errors, "accountsDisplayNameSupported",
                            locale);
                    validateRequired(provider::getAccountsKeySupported, accountSettingsBuilder::keySupported,
                            errors, "accountsKeySupported", locale);
                    validateRequired(provider::getAccountsDeleteSupported, accountSettingsBuilder::deleteSupported,
                            errors, "accountsDeleteSupported", locale);
                    validateRequired(provider::getAccountsSoftDeleteSupported,
                            accountSettingsBuilder::softDeleteSupported, errors, "accountsSoftDeleteSupported",
                            locale);
                    validateRequired(provider::getAccountsMoveSupported, accountSettingsBuilder::moveSupported,
                            errors, "accountsMoveSupported", locale);
                    validateRequired(provider::getAccountsRenameSupported, accountSettingsBuilder::renameSupported,
                            errors, "accountsRenameSupported", locale);
                    validateRequired(provider::getImportAllowed, builder::importAllowed, errors,
                            "importAllowed", locale);
                    validateRequired(provider::getAccountsSpacesSupported, builder::accountsSpacesSupported, errors,
                            "accountsSpacesSupported", locale);
                    validateRequired(provider::getPerAccountVersionSupported,
                            accountSettingsBuilder::perAccountVersionSupported, errors, "perAccountVersionSupported",
                            locale);
                    validateRequired(provider::getPerProvisionVersionSupported,
                            accountSettingsBuilder::perProvisionVersionSupported, errors,
                            "perProvisionVersionSupported", locale);
                    validateRequired(provider::getPerAccountLastUpdateSupported,
                            accountSettingsBuilder::perAccountLastUpdateSupported, errors,
                            "perAccountLastUpdateSupported", locale);
                    validateRequired(provider::getPerProvisionLastUpdateSupported,
                            accountSettingsBuilder::perProvisionLastUpdateSupported, errors,
                            "perProvisionLastUpdateSupported", locale);
                    validateRequired(provider::getOperationIdDeduplicationSupported,
                            accountSettingsBuilder::operationIdDeduplicationSupported, errors,
                            "operationIdDeduplicationSupported", locale);
                    validateRequired(provider::getSyncCoolDownDisabled,
                            accountSettingsBuilder::syncCoolDownDisabled, errors,
                            "syncCoolDownDisabled", locale);
                    validateRequired(provider::getRetryCoolDownDisabled,
                            accountSettingsBuilder::retryCoolDownDisabled, errors,
                            "retryCoolDownDisabled", locale);
                    validateAccountSyncPageSize(provider::getAccountsSyncPageSize,
                            accountSettingsBuilder::accountsSyncPageSize, errors,
                            "accountsSyncPageSize", locale);
                    validateMoveProvisionSupported(provider::getMoveProvisionSupported,
                            accountSettingsBuilder::moveProvisionSupported, errors,
                            "moveProvisionSupported", locale);
                    validateExternalAccountUrlTemplates(provider::getExternalAccountUrlTemplates,
                            accountSettingsBuilder::externalAccountUrlTemplates, errors,
                            "externalAccountUrlTemplates", locale);
                    validateRequired(provider::getSyncEnabled, builder::syncEnabled, errors, "syncEnabled", locale);
                    validateRequired(provider::getGrpcTlsOn, builder::grpcTlsOn, errors, "grpcTlsOn", locale);
                    validateOptionalText(provider::getMeteringKey, billingMetaBuilder::meteringKey, errors,
                            "meteringKey", MAX_KEY_LENGTH, locale);
                    validateRequired(provider::getTrackerComponentId, builder::trackerComponentId, errors,
                            "trackerComponentId", locale);
                    if (builder.getGrpcApiUri().isEmpty() && builder.getRestApiUri().isEmpty()) {
                        errors.addError(TypedError.invalid(messages
                                .getMessage("errors.provider.api.uri.is.required", null, locale)));
                    }
                    if (builder.getManaged().isPresent() && builder.getReadOnly().isPresent()
                            && !builder.getManaged().get() && !builder.getReadOnly().get()) {
                        errors.addError(TypedError.invalid(messages
                                .getMessage("errors.non.managed.provider.must.be.read.only", null, locale)));
                    }
                    if (accountSettingsBuilder.getSoftDeleteSupported().isPresent()
                            && accountSettingsBuilder.getDeleteSupported().isPresent()
                            && accountSettingsBuilder.getSoftDeleteSupported().get()
                            && !accountSettingsBuilder.getDeleteSupported().get()) {
                        errors.addError(TypedError.invalid(messages
                                .getMessage("errors.soft.delete.without.delete.is.not.allowed", null, locale)));
                    }
                    if (accountSettingsBuilder.getDisplayNameSupported().isPresent()
                            && accountSettingsBuilder.getRenameSupported().isPresent()
                            && accountSettingsBuilder.getRenameSupported().get()
                            && !accountSettingsBuilder.getDisplayNameSupported().get()) {
                        errors.addError(TypedError.invalid(messages
                                .getMessage("errors.rename.without.names.is.not.allowed", null, locale)));
                    }
                    FeatureStateHelper.validateFeatureState(provider::getMultipleReservesAllowed,
                            accountSettingsBuilder::multipleReservesAllowed, errors, "multipleReservesAllowed",
                            locale, messages);
                    if (errors.hasAnyErrors()) {
                        return Result.failure(errors.build());
                    }
                    builder.accountsSettings(accountSettingsBuilder.build());
                    builder.billingMeta(billingMetaBuilder.build());
                    FeatureStateHelper.validateFeatureState(provider::getAllocatedSupported,
                            builder::allocatedSupported, errors, "allocatedSupported", locale, messages);
                    AggregationSettingsHelper.validateAggregationSettings(provider::getAggregationSettings,
                            builder::aggregationSettings, errors, "aggregationSettings", locale, messages);
                    AggregationAlgorithmHelper.validateAggregationAlgorithm(provider::getAggregationAlgorithm,
                            builder::aggregationAlgorithm, errors, "aggregationAlgorithm", locale, messages);
                    return Result.success(builder.build());
                }));
    }

    private Result<Tuple2<ProviderModel, Boolean>> mergeUISettings(
            ProviderModel provider,
            ProviderUISettings uiSettings,
            Long version,
            Locale locale
    ) {
        ProviderModel.Builder builder = ProviderModel.builder(provider);
        ErrorCollection.Builder errors = ErrorCollection.builder();
        validateVersion(() -> Optional.ofNullable(version), provider::getVersion, builder::version, errors,
                "version", locale);
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build());
        }
        builder.uiSettings(uiSettings);
        if (builder.hasChanges(provider)) {
            return Result.success(Tuples.of(builder.build(), true));
        }
        return Result.success(Tuples.of(provider, false));
    }

    private Result<Tuple2<ProviderModel, Boolean>> mergeExternalAccountUrlTemplate(
            ProviderModel provider,
            List<ExternalAccountUrlTemplateDto> externalAccountUrlTemplateDto,
            Long version,
            Locale locale
    ) {
        ProviderModel.Builder builder = ProviderModel.builder(provider);
        AccountsSettingsModel.Builder accountSettingsBuilder =
                AccountsSettingsModel.builder(provider.getAccountsSettings());
        ErrorCollection.Builder errors = ErrorCollection.builder();
        validateVersion(() -> Optional.ofNullable(version), provider::getVersion, builder::version, errors,
                "version", locale);
        validateExternalAccountUrlTemplates(() -> Optional.ofNullable(externalAccountUrlTemplateDto),
                accountSettingsBuilder::externalAccountUrlTemplates, errors,
                "externalAccountUrlTemplates", locale);
        builder.accountsSettings(accountSettingsBuilder.build());
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build());
        }
        if (builder.hasChanges(provider)) {
            return Result.success(Tuples.of(builder.build(), true));
        }
        return Result.success(Tuples.of(provider, false));
    }

    private Mono<Result<Tuple2<ProviderModel, Boolean>>> validatePut(ProviderModel provider,
                                                                     ProviderPutDto providerPutDto, Long version,
                                                                     Locale locale, YdbTxSession session) {
        return validateAbcService(providerPutDto::getAbcServiceId, locale, session).map(serviceR -> {
            ProviderModel.Builder builder = ProviderModel.builder(provider);
            AccountsSettingsModel.Builder accountSettingsBuilder = AccountsSettingsModel.builder();
            BillingMeta.Builder billingMetaBuilder = BillingMeta.builder();
            ErrorCollection.Builder errors = ErrorCollection.builder();
            serviceR.doOnFailure(errors::add);
            serviceR.doOnSuccess(s -> builder.serviceId(s.getId()));
            validateText(providerPutDto::getNameEn, builder::nameEn, errors, "nameEn", MAX_NAME_LENGTH, locale);
            validateText(providerPutDto::getNameRu, builder::nameRu, errors, "nameRu", MAX_NAME_LENGTH, locale);
            validateText(providerPutDto::getDescriptionEn, builder::descriptionEn, errors, "descriptionEn",
                    MAX_DESCRIPTION_LENGTH, locale);
            validateText(providerPutDto::getDescriptionRu, builder::descriptionRu, errors, "descriptionRu",
                    MAX_DESCRIPTION_LENGTH, locale);
            validateRestUri(providerPutDto::getRestApiUri, builder::restApiUri, errors, "restApiUri",
                    MAX_URI_LENGTH, locale);
            validateGrpcUri(providerPutDto::getGrpcApiUri, builder::grpcApiUri, errors, "grpcApiUri",
                    MAX_URI_LENGTH, locale);
            validateTvmId(providerPutDto::getSourceTvmId, builder::sourceTvmId, errors,
                    "sourceTvmId", locale);
            validateTvmId(providerPutDto::getDestinationTvmId, builder::destinationTvmId, errors,
                    "destinationTvmId", locale);
            validateRequired(providerPutDto::getReadOnly, builder::readOnly, errors, "readOnly", locale);
            validateRequired(providerPutDto::getMultipleAccountsPerFolder, builder::multipleAccountsPerFolder, errors,
                    "multipleAccountsPerFolder", locale);
            validateRequired(providerPutDto::getAccountTransferWithQuota, builder::accountTransferWithQuota, errors,
                    "accountTransferWithQuota", locale);
            validateRequired(providerPutDto::getManaged, builder::managed, errors,
                    "accountTransferWithQuota", locale);
            validateVersion(() -> Optional.ofNullable(version), provider::getVersion, builder::version, errors,
                    "version", locale);
            validateRequired(providerPutDto::getAccountsDisplayNameSupported,
                    accountSettingsBuilder::displayNameSupported, errors, "accountsDisplayNameSupported",
                    locale);
            validateRequired(providerPutDto::getAccountsKeySupported, accountSettingsBuilder::keySupported,
                    errors, "accountsKeySupported", locale);
            validateRequired(providerPutDto::getAccountsDeleteSupported, accountSettingsBuilder::deleteSupported,
                    errors, "accountsDeleteSupported", locale);
            validateRequired(providerPutDto::getAccountsSoftDeleteSupported,
                    accountSettingsBuilder::softDeleteSupported, errors, "accountsSoftDeleteSupported",
                    locale);
            validateRequired(providerPutDto::getAccountsMoveSupported, accountSettingsBuilder::moveSupported,
                    errors, "accountsMoveSupported", locale);
            validateRequired(providerPutDto::getAccountsRenameSupported, accountSettingsBuilder::renameSupported,
                    errors, "accountsRenameSupported", locale);
            validateRequired(providerPutDto::getImportAllowed, builder::importAllowed, errors,
                    "importAllowed", locale);
            validateRequired(providerPutDto::getAccountsSpacesSupported, builder::accountsSpacesSupported, errors,
                    "accountsSpacesSupported", locale);
            // TODO More validation for 'accounts spaces' flag update
            validateRequired(providerPutDto::getPerAccountVersionSupported,
                    accountSettingsBuilder::perAccountVersionSupported, errors, "perAccountVersionSupported",
                    locale);
            validateRequired(providerPutDto::getPerProvisionVersionSupported,
                    accountSettingsBuilder::perProvisionVersionSupported, errors,
                    "perProvisionVersionSupported", locale);
            validateRequired(providerPutDto::getPerAccountLastUpdateSupported,
                    accountSettingsBuilder::perAccountLastUpdateSupported, errors,
                    "perAccountLastUpdateSupported", locale);
            validateRequired(providerPutDto::getPerProvisionLastUpdateSupported,
                    accountSettingsBuilder::perProvisionLastUpdateSupported, errors,
                    "perProvisionLastUpdateSupported", locale);
            validateRequired(providerPutDto::getOperationIdDeduplicationSupported,
                    accountSettingsBuilder::operationIdDeduplicationSupported, errors,
                    "operationIdDeduplicationSupported", locale);
            validateRequired(providerPutDto::getSyncCoolDownDisabled,
                    accountSettingsBuilder::syncCoolDownDisabled, errors,
                    "syncCoolDownDisabled", locale);
            validateRequired(providerPutDto::getRetryCoolDownDisabled,
                    accountSettingsBuilder::retryCoolDownDisabled, errors,
                    "retryCoolDownDisabled", locale);
            validateAccountSyncPageSize(providerPutDto::getAccountsSyncPageSize,
                    accountSettingsBuilder::accountsSyncPageSize, errors,
                    "accountsSyncPageSize", locale);
            validateMoveProvisionSupported(providerPutDto::getMoveProvisionSupported,
                    accountSettingsBuilder::moveProvisionSupported, errors,
                    "moveProvisionSupported", locale);
            validateRequired(providerPutDto::getSyncEnabled, builder::syncEnabled, errors, "syncEnabled", locale);
            validateRequired(providerPutDto::getGrpcTlsOn, builder::grpcTlsOn, errors, "grpcTlsOn", locale);
            validateOptionalText(providerPutDto::getMeteringKey, billingMetaBuilder::meteringKey, errors,
                    "meteringKey", MAX_KEY_LENGTH, locale);
            validateRequired(providerPutDto::getTrackerComponentId, builder::trackerComponentId, errors,
                    "trackerComponentId", locale);
            if (builder.getGrpcApiUri().isEmpty() && builder.getRestApiUri().isEmpty()) {
                errors.addError(TypedError.invalid(messages
                        .getMessage("errors.provider.api.uri.is.required", null, locale)));
            }
            if (builder.getManaged().isPresent() && builder.getReadOnly().isPresent()
                    && !builder.getManaged().get() && !builder.getReadOnly().get()) {
                errors.addError(TypedError.invalid(messages
                        .getMessage("errors.non.managed.provider.must.be.read.only", null, locale)));
            }
            if (accountSettingsBuilder.getSoftDeleteSupported().isPresent()
                    && accountSettingsBuilder.getDeleteSupported().isPresent()
                    && accountSettingsBuilder.getSoftDeleteSupported().get()
                    && !accountSettingsBuilder.getDeleteSupported().get()) {
                errors.addError(TypedError.invalid(messages
                        .getMessage("errors.soft.delete.without.delete.is.not.allowed", null, locale)));
            }
            if (accountSettingsBuilder.getDisplayNameSupported().isPresent()
                    && accountSettingsBuilder.getRenameSupported().isPresent()
                    && accountSettingsBuilder.getRenameSupported().get()
                    && !accountSettingsBuilder.getDisplayNameSupported().get()) {
                errors.addError(TypedError.invalid(messages
                        .getMessage("errors.rename.without.names.is.not.allowed", null, locale)));
            }
            FeatureStateHelper.validateFeatureState(providerPutDto::getAllocatedSupported, builder::allocatedSupported,
                    errors, "allocatedSupported", locale, messages);
            AggregationSettingsHelper.validateAggregationSettings(providerPutDto::getAggregationSettings,
                    builder::aggregationSettings, errors, "aggregationSettings", locale, messages);
            AggregationAlgorithmHelper.validateAggregationAlgorithm(providerPutDto::getAggregationAlgorithm,
                    builder::aggregationAlgorithm, errors, "aggregationAlgorithm", locale, messages);
            FeatureStateHelper.validateFeatureState(providerPutDto::getMultipleReservesAllowed,
                    accountSettingsBuilder::multipleReservesAllowed, errors, "multipleReservesAllowed",
                    locale, messages);
            if (errors.hasAnyErrors()) {
                return Result.failure(errors.build());
            }
            builder.accountsSettings(accountSettingsBuilder.build());
            builder.billingMeta(billingMetaBuilder.build());
            if (builder.hasChanges(provider)) {
                return Result.success(Tuples.of(builder.build(), true));
            }
            return Result.success(Tuples.of(provider, false));
        });
    }

    private Mono<Result<ServiceMinimalModel>> validateAbcService(Supplier<Optional<Long>> serviceIdSupplier,
                                                                 Locale locale, YdbTxSession session) {
        Optional<Long> serviceId = serviceIdSupplier.get();
        if (serviceId.isEmpty()) {
            ErrorCollection error = ErrorCollection.builder().addError("abcServiceId", TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)))
                    .build();
            return Mono.just(Result.failure(error));
        }
        return servicesDao.getByIdMinimal(session, serviceId.get()).map(WithTxId::get).map(service -> {
            if (service.isEmpty()) {
                ErrorCollection error = ErrorCollection.builder().addError("abcServiceId", TypedError.invalid(messages
                        .getMessage("errors.service.not.found", null, locale)))
                        .build();
                return Result.failure(error);
            }
            if (!service.get().isExportable()) {
                ErrorCollection error = ErrorCollection.builder().addError("abcServiceId", TypedError.invalid(messages
                        .getMessage("errors.service.is.non.exportable", null, locale)))
                        .build();
                return Result.failure(error);
            }
            if (!ALLOWED_SERVICE_STATES.contains(service.get().getState())) {
                ErrorCollection error = ErrorCollection.builder().addError("abcServiceId", TypedError.invalid(messages
                        .getMessage("errors.service.bad.status", null, locale)))
                        .build();
                return Result.failure(error);
            }
            return Result.success(service.get());
        });
    }

    private void validateText(Supplier<Optional<String>> getter, Consumer<String> setter,
                              ErrorCollection.Builder errors, String fieldKey, int maxLength, Locale locale) {
        Optional<String> text = getter.get();
        if (text.isEmpty()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        } else if (text.get().isBlank()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.non.blank.text.is.required", null, locale)));
        } else if (text.get().length() > maxLength) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.text.is.too.long", null, locale)));
        } else {
            setter.accept(text.get());
        }
    }

    private void validateOptionalText(Supplier<Optional<String>> getter, Consumer<String> setter,
                                      ErrorCollection.Builder errors, String fieldKey, int maxLength, Locale locale) {
        Optional<String> text = getter.get();
        if (text.isEmpty()) {
            setter.accept(null);
        } else if (text.get().isBlank()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.non.blank.text.is.required", null, locale)));
        } else if (text.get().length() > maxLength) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.text.is.too.long", null, locale)));
        } else {
            setter.accept(text.get());
        }
    }

    private void validateRestUri(Supplier<Optional<String>> getter, Consumer<String> setter,
                                 ErrorCollection.Builder errors, String fieldKey, int maxLength, Locale locale) {
        Optional<String> uri = getter.get();
        if (uri.isEmpty()) {
            return;
        }
        if (uri.get().isBlank()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.non.blank.text.is.required", null, locale)));
        } else if (uri.get().length() > maxLength) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.text.is.too.long", null, locale)));
        } else {
            Optional<URI> parsedUri = tryParseUri(uri.get());
            if (parsedUri.isEmpty()) {
                errors.addError(fieldKey, TypedError.invalid(messages
                        .getMessage("errors.invalid.uri", null, locale)));
            } else {
                URI validUri = parsedUri.get();
                boolean hasInvalidComponents = false;
                if (!"http".equals(validUri.getScheme()) && !"https".equals(validUri.getScheme())) {
                    errors.addError(fieldKey, TypedError.invalid(messages
                            .getMessage("errors.invalid.uri.scheme", null, locale)));
                    hasInvalidComponents = true;
                }
                if (!validUri.isAbsolute()) {
                    errors.addError(fieldKey, TypedError.invalid(messages
                            .getMessage("errors.non.absolute.uri", null, locale)));
                    hasInvalidComponents = true;
                }
                if (validUri.isOpaque()) {
                    errors.addError(fieldKey, TypedError.invalid(messages
                            .getMessage("errors.opaque.uri", null, locale)));
                    hasInvalidComponents = true;
                }
                if (validUri.getHost() == null || validUri.getHost().isBlank()) {
                    errors.addError(fieldKey, TypedError.invalid(messages
                            .getMessage("errors.uri.host.is.required", null, locale)));
                    hasInvalidComponents = true;
                }
                if (validUri.getQuery() != null) {
                    errors.addError(fieldKey, TypedError.invalid(messages
                            .getMessage("errors.uri.query.is.not.supported", null, locale)));
                    hasInvalidComponents = true;
                }
                if (validUri.getFragment() != null) {
                    errors.addError(fieldKey, TypedError.invalid(messages
                            .getMessage("errors.uri.fragment.is.not.supported", null, locale)));
                    hasInvalidComponents = true;
                }
                if (validUri.getUserInfo() != null) {
                    errors.addError(fieldKey, TypedError.invalid(messages
                            .getMessage("errors.uri.user.info.is.not.supported", null, locale)));
                    hasInvalidComponents = true;
                }
                if (!hasInvalidComponents) {
                    setter.accept(uri.get());
                }
            }
        }
    }

    private void validateGrpcUri(Supplier<Optional<String>> getter, Consumer<String> setter,
                                 ErrorCollection.Builder errors, String fieldKey, int maxLength, Locale locale) {
        Optional<String> uri = getter.get();
        if (uri.isEmpty()) {
            return;
        }
        if (uri.get().isBlank()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.non.blank.text.is.required", null, locale)));
        } else if (uri.get().length() > maxLength) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.text.is.too.long", null, locale)));
        } else {
            Optional<URI> parsedUri = tryParseUri(uri.get());
            if (parsedUri.isEmpty()) {
                errors.addError(fieldKey, TypedError.invalid(messages
                        .getMessage("errors.invalid.uri", null, locale)));
            } else {
                URI validUri = parsedUri.get();
                boolean hasInvalidComponents = false;
                if (!"dns".equals(validUri.getScheme()) && !"static".equals(validUri.getScheme())) {
                    errors.addError(fieldKey, TypedError.invalid(messages
                            .getMessage("errors.invalid.uri.scheme", null, locale)));
                    hasInvalidComponents = true;
                }
                boolean dnsScheme = "dns".equals(validUri.getScheme());
                boolean staticScheme = "static".equals(validUri.getScheme());
                if (!validUri.isAbsolute()) {
                    errors.addError(fieldKey, TypedError.invalid(messages
                            .getMessage("errors.non.absolute.uri", null, locale)));
                    hasInvalidComponents = true;
                }
                if (validUri.isOpaque()) {
                    errors.addError(fieldKey, TypedError.invalid(messages
                            .getMessage("errors.opaque.uri", null, locale)));
                    hasInvalidComponents = true;
                }
                if (dnsScheme && (validUri.getHost() == null || validUri.getHost().isBlank())) {
                    errors.addError(fieldKey, TypedError.invalid(messages
                            .getMessage("errors.uri.host.is.required", null, locale)));
                    hasInvalidComponents = true;
                }
                if (dnsScheme && validUri.getPort() < 0) {
                    errors.addError(fieldKey, TypedError.invalid(messages
                            .getMessage("errors.uri.port.is.required", null, locale)));
                    hasInvalidComponents = true;
                }
                if (staticScheme && (validUri.getAuthority() == null || validUri.getAuthority().isBlank())) {
                    errors.addError(fieldKey, TypedError.invalid(messages
                            .getMessage("errors.uri.authority.is.required", null, locale)));
                    hasInvalidComponents = true;
                }
                if (staticScheme && validUri.getAuthority() != null && !validUri.getAuthority().isBlank()) {
                    String[] hosts = PATTERN_COMMA.split(validUri.getAuthority());
                    boolean hasInvalidHost = false;
                    for (String host : hosts) {
                        try {
                            HostAndPort.fromString(host);
                        } catch (IllegalArgumentException e) {
                            hasInvalidHost = true;
                        }
                    }
                    if (hasInvalidHost) {
                        errors.addError(fieldKey, TypedError.invalid(messages
                                .getMessage("errors.uri.invalid.authority", null, locale)));
                        hasInvalidComponents = true;
                    }
                }
                if (validUri.getQuery() != null) {
                    errors.addError(fieldKey, TypedError.invalid(messages
                            .getMessage("errors.uri.query.is.not.supported", null, locale)));
                    hasInvalidComponents = true;
                }
                if (validUri.getFragment() != null) {
                    errors.addError(fieldKey, TypedError.invalid(messages
                            .getMessage("errors.uri.fragment.is.not.supported", null, locale)));
                    hasInvalidComponents = true;
                }
                if (validUri.getUserInfo() != null) {
                    errors.addError(fieldKey, TypedError.invalid(messages
                            .getMessage("errors.uri.user.info.is.not.supported", null, locale)));
                    hasInvalidComponents = true;
                }
                if (validUri.getPath() != null && !validUri.getPath().isEmpty()) {
                    errors.addError(fieldKey, TypedError.invalid(messages
                            .getMessage("errors.uri.path.is.not.supported", null, locale)));
                    hasInvalidComponents = true;
                }
                if (!hasInvalidComponents) {
                    setter.accept(uri.get());
                }
            }
        }
    }

    private Optional<URI> tryParseUri(String value) {
        try {
            return Optional.of(URI.create(value));
        } catch (Exception e) {
            return Optional.empty();
        }
    }

    private void validateTvmId(Supplier<Optional<Long>> getter, Consumer<Long> setter,
                               ErrorCollection.Builder errors, String fieldKey, Locale locale) {
        Optional<Long> tvmId = getter.get();
        if (tvmId.isEmpty()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        } else if (tvmId.get() <= 0L) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.number.must.be.positive", null, locale)));
        } else {
            setter.accept(tvmId.get());
        }
    }

    private <T> void validateRequired(Supplier<Optional<T>> getter, Consumer<T> setter,
                                      ErrorCollection.Builder errors, String fieldKey, Locale locale) {
        Optional<T> value = getter.get();
        if (value.isEmpty()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        } else {
            setter.accept(value.get());
        }
    }

    private void validateAccountSyncPageSize(Supplier<Optional<Long>> getter, Consumer<Long> setter,
                                      ErrorCollection.Builder errors, String fieldKey, Locale locale) {
        Optional<Long> value = getter.get();
        if (value.isEmpty() || value.get() == 0) {
            //backward compatibility
            setter.accept(AccountsSettingsModel.DEFAULT_ACCOUNTS_SYNC_PAGE_SIZE);
        } else if (value.get() < 0) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.number.must.be.positive", null, locale)));
        } else {
            setter.accept(value.get());
        }
    }

    private void validateMoveProvisionSupported(Supplier<Optional<Boolean>> getter, Consumer<Boolean> setter,
                                             ErrorCollection.Builder errors, String fieldKey, Locale locale) {
        Optional<Boolean> value = getter.get();
        if (value.isEmpty()) {
            //backward compatibility
            setter.accept(Boolean.FALSE);
        } else {
            setter.accept(value.get());
        }
    }

    private void validateExternalAccountUrlTemplates(Supplier<Optional<List<ExternalAccountUrlTemplateDto>>> getter,
                                                     Consumer<Set<ExternalAccountUrlTemplate>> setter,
                                                     ErrorCollection.Builder errors, String fieldKey, Locale locale) {
        Optional<List<ExternalAccountUrlTemplateDto>> listTemplatesOptional = getter.get();
        if (listTemplatesOptional.isEmpty()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        } else {
            List<ExternalAccountUrlTemplateDto> listTemplates = listTemplatesOptional.get();
            boolean successfulValidation = true;
            for (int i = 0; i < listTemplates.size(); i++) {
                ExternalAccountUrlTemplateDto dto = listTemplates.get(i);
                if (dto.getDefaultTemplate() == null) {
                    errors.addError(fieldKey + "." + i + ".defaultTemplate", TypedError.invalid(messages
                            .getMessage("errors.field.is.required", null, locale)));
                    successfulValidation = false;
                }
                if (dto.getUrlsForSegments() == null) {
                    errors.addError(fieldKey + "." + i + ".urlsForSegments", TypedError.invalid(messages
                            .getMessage("errors.field.is.required", null, locale)));
                    successfulValidation = false;
                }
                if (dto.getUrlTemplates() == null) {
                    errors.addError(fieldKey + "." + i + ".urlTemplates", TypedError.invalid(messages
                            .getMessage("errors.field.is.required", null, locale)));
                    successfulValidation = false;
                } else {
                    for (var entry : dto.getUrlTemplates().entrySet()) {
                        if (entry.getValue() == null) {
                            errors.addError(fieldKey + "." + i + ".urlTemplates." + entry.getKey(),
                                    TypedError.invalid(messages.getMessage("errors.field.is.required", null, locale)));
                            successfulValidation = false;
                        }
                    }
                }
                if (dto.getSegments() == null) {
                    errors.addError(fieldKey + "." + i + ".segments", TypedError.invalid(messages
                            .getMessage("errors.field.is.required", null, locale)));
                    successfulValidation = false;
                } else {
                    for (var entry : dto.getSegments().entrySet()) {
                        if (entry.getValue() == null) {
                            errors.addError(fieldKey + "." + i + ".segments." + entry.getKey(),
                                    TypedError.invalid(messages.getMessage("errors.field.is.required", null, locale)));
                            successfulValidation = false;
                        }
                    }
                }
            }

            if (successfulValidation) {
                Set<ExternalAccountUrlTemplate> validated = listTemplates.stream()
                        .map(dto -> new ExternalAccountUrlTemplate(
                                dto.getSegments(),
                                dto.getDefaultTemplate(),
                                dto.getUrlTemplates(),
                                dto.getUrlsForSegments()))
                        .collect(Collectors.toSet());
                setter.accept(validated);
            }
        }
    }

    private void validateVersion(Supplier<Optional<Long>> getter, Supplier<Long> existingGetter,
                                 Consumer<Long> setter, ErrorCollection.Builder errors, String fieldKey,
                                 Locale locale) {
        Optional<Long> value = getter.get();
        Long existingValue = existingGetter.get();
        if (value.isEmpty()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        } else if (!Objects.equals(value.get(), existingValue)) {
            errors.addError(fieldKey, TypedError.versionMismatch(messages
                    .getMessage("errors.version.mismatch", null, locale)));
        } else {
            setter.accept(existingValue + 1L);
        }
    }

    private Mono<Result<String>> validateKey(YdbTxSession session, Supplier<Optional<String>> getter,
                                             String fieldKey, Locale locale) {
        Optional<String> value = getter.get();
        if (value.isEmpty()) {
            ErrorCollection error = ErrorCollection.builder().addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)))
                    .build();
            return Mono.just(Result.failure(error));
        }
        if (value.get().isBlank()) {
            ErrorCollection error = ErrorCollection.builder().addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.non.blank.text.is.required", null, locale)))
                    .build();
            return Mono.just(Result.failure(error));
        }
        if (value.get().length() > MAX_KEY_LENGTH) {
            ErrorCollection error = ErrorCollection.builder().addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.text.is.too.long", null, locale)))
                    .build();
            return Mono.just(Result.failure(error));
        }
        return providersDao.existsByKey(session, Tenants.DEFAULT_TENANT_ID, value.get(), false).map(exists -> {
            if (exists) {
                ErrorCollection error = ErrorCollection.builder().addError(fieldKey,
                        TypedError.conflict(messages
                                .getMessage("errors.provider.key.already.exists", null, locale)))
                        .build();
                return Result.failure(error);
            }
            return Result.success(value.get());
        });
    }

    public Mono<Result<ProviderModel>> doSync(String id, YaUserDetails currentUser, Locale locale) {
        return validateId(id, locale).andThenMono(v -> securityManagerService
                .checkWritePermissionsForProvider(id, currentUser, locale))
                .flatMap(r -> r.andThenMono(v -> tableClient.usingSessionMonoRetryable(session -> providersDao
                        .getById(immediateTx(session), id, Tenants.DEFAULT_TENANT_ID)
                        .map(pO -> validateExists(pO.orElse(null), locale)
                                .andThen(p -> validateEligibleForSync(p, locale)))
                )))
                .flatMap(r -> r.andThenMono(provider ->
                        accountsSyncService.syncOneProvider(provider, locale, Clock.systemUTC())
                                .thenReturn(Result.success(provider))
                ));
    }

    private Result<ProviderModel> validateEligibleForSync(ProviderModel provider, Locale locale) {
        if ((provider.getRestApiUri().isPresent() || provider.getGrpcApiUri().isPresent())
                && provider.isSyncEnabled()) {
            return Result.success(provider);
        }

        return Result.failure(ErrorCollection.builder()
                .addError(TypedError.invalid(messages.getMessage("errors.provider.sync.unavailable", null, locale)))
                .build());
    }

    public Mono<Result<ProviderSyncStatusDto>> getSyncStatus(
            String providerId, YaUserDetails currentUser, Locale locale
    ) {
        TenantId tenantId = getTenantId(currentUser);
        return securityManagerService.checkReadPermissions(currentUser, locale).flatMap(res -> res.andThen(v ->
        validateId(providerId, locale)).andThenMono(u -> tableClient.usingSessionMonoRetryable(session ->
        providersDao.getById(immediateTx(session), providerId, tenantId).flatMap(providerOptional ->
        validateExists(providerOptional.orElse(null), locale).andThenMono(provider ->
        providersSyncStatusDao.getById(immediateTx(session), providerId, tenantId).flatMap(syncStatusOptional ->
            syncStatusOptional.map(Result::success).orElse(Result.failure(ErrorCollection.builder().addError(
                    TypedError.invalid(messages.getMessage("errors.provider.sync.unavailable", null, locale)))
                    .build())).applyMono(syncStatus ->
        providersSyncErrorsDao.getAllByProvider(immediateTx(session), tenantId, providerId)
            .map(WithTxId::get).map(syncErrors ->
        buildProviderSyncStatusDto(syncStatus, syncErrors)
        ))))))));
    }

    public Mono<Result<Void>> putProvisionsAsQuota(String providerId, YaUserDetails currentUser, Locale locale) {
        TenantId tenantId = getTenantId(currentUser);
        return securityManagerService.checkDictionaryWritePermissions(currentUser, locale).andThen(v ->
                validateProviderId(providerId, locale)).andThenMono(v ->
                providersLoader.getProviderByIdImmediate(providerId, tenantId)
                        .map(oProvider -> validateProviderForQuotaSet(oProvider, locale))).flatMap(result ->
                result.andThenMono(provider -> {
                    try {
                        putProvisionsAsQuotasExecutor.execute(() ->
                                putProvisionsAsQuotaBlocking(tenantId, providerId, currentUser));
                    } catch (RejectedExecutionException e) {
                        return Mono.just(Result.failure(ErrorCollection.builder().addError(TypedError.tooManyRequests(
                                messages.getMessage("errors.too.many.requests", null, locale))).build()));
                    }
                    return Mono.just(Result.success(null));
                }));
    }

    public Result<QuotaSetDto> putProvisionsAsQuotaBlocking(TenantId tenantId, String providerId,
                                                            YaUserDetails currentUser) {
        try {
            Result<QuotaSetDto> result = getFolderIdsFlux(tenantId).concatMap(folderIds ->
                    quotaSet(folderIds, tenantId, providerId, currentUser))
                    .collect(Collectors.summingLong(value -> value))
                    .map(sumLong -> Result.success(QuotaSetDto.builder()
                            .status(OK)
                            .quotaChangeCount(sumLong)
                            .build()))
                    .block();
            result.doOnSuccess(quotas -> LOG.info(quotas.toString()));
            result.doOnFailure(errors -> LOG.error(errors.toString()));
            return result;
        } catch (RuntimeException e) {
            LOG.error("Error in the background thread when executing putProvisionsAsQuota", e);
            throw e;
        }
    }

    public Mono<Result<ProviderUISettingsDto>> getDefaultUISettings(YaUserDetails currentUser, Locale locale) {
        return securityManagerService.checkReadPermissions(currentUser, locale).map(res -> res.apply(v ->
                new ProviderUISettingsDto(ProviderUISettings.defaultValue())));
    }

    private Flux<Long> quotaSet(List<String> rawFolderIds, TenantId tenantId, String providerId,
                                YaUserDetails currentUser) {
        if (rawFolderIds.isEmpty()) {
            return Flux.just(0L);
        }

        return tableClient.usingSessionMonoRetryable(session -> session.usingCompTxRetryable(
                ts -> folderDao.getByIdsStartTx(ts, rawFolderIds, tenantId),
                (ts, folders) -> {
                    Set<String> folderIdsSet = folders.stream()
                            .map(FolderModel::getId)
                            .collect(Collectors.toSet());
                    return accountsQuotasDao.getAllByFoldersAndProvider(ts, tenantId, folderIdsSet, providerId, false)
                            .flatMap(accountQuotasTx -> {
                                List<AccountsQuotasModel> accountQuotas = accountQuotasTx.get();
                                List<String> folderIds = accountQuotas.stream()
                                        .map(AccountsQuotasModel::getFolderId)
                                        .distinct()
                                        .collect(Collectors.toList());

                                if (folderIds.isEmpty()) {
                                    return Mono.just(0L);
                                }

                                return quotasDao.getByFoldersAndProvider(ts, folderIds, tenantId, providerId, false)
                                        .flatMap(quotas -> applyQuotaSet(ts, quotas, accountQuotas, folders, tenantId,
                                                currentUser));
                            });
                },
                (ts, l) -> ts.commitTransaction().thenReturn(l)))
                .flux();
    }

    private Mono<Long> applyQuotaSet(YdbTxSession ts, List<QuotaModel> quotas, List<AccountsQuotasModel> accountQuotas,
                                     List<FolderModel> folders, TenantId tenantId,
                                     YaUserDetails currentUser) {
        Instant now = Instant.now();

        Map<String, Map<String, Long>> oldQuotasByFolderId = new HashMap<>();
        Map<String, Map<String, Long>> oldBalanceByFolderId = new HashMap<>();
        List<QuotaModel> updatedQuotaModels = new ArrayList<>();

        Map<String, Map<String, Long>> sumByResourceIdByFolderId =  accountQuotas.stream()
                .collect(Collectors.groupingBy(AccountsQuotasModel::getFolderId,
                        Collectors.groupingBy(AccountsQuotasModel::getResourceId,
                                Collectors.summingLong(AccountsQuotasModel::getProvidedQuota))));

        quotas.stream()
                .filter(quotaModel -> sumByResourceIdByFolderId.get(quotaModel.getFolderId())
                        .containsKey(quotaModel.getResourceId()))
                .forEach(quotaModel -> {
                            String folderId = quotaModel.getFolderId();
                            oldQuotasByFolderId.computeIfAbsent(folderId,
                                    key -> new HashMap<>()).put(quotaModel.getResourceId(), quotaModel.getQuota());
                            oldBalanceByFolderId.computeIfAbsent(folderId,
                                    key -> new HashMap<>()).put(quotaModel.getResourceId(), quotaModel.getBalance());
                            updatedQuotaModels.add(QuotaModel.builder(quotaModel)
                                    .quota(sumByResourceIdByFolderId.get(folderId).get(quotaModel.getResourceId()))
                                    .balance(0L)
                                    .frozenQuota(0L)
                                    .build());
                        }
                );

        Map<String, Long> nextOpOrderByFolderId = folders.stream()
                .collect(Collectors.toMap(FolderModel::getId, FolderModel::getNextOpLogOrder));

        return quotasDao.upsertAllRetryable(ts, updatedQuotaModels)
                .then(Mono.defer(() -> {
                    var folderOperationLogModels = sumByResourceIdByFolderId.keySet().stream()
                            .map(folderId -> {
                                Map<String, Long> oldQuotas = oldQuotasByFolderId.get(folderId);
                                Map<String, Long> oldBalance = oldBalanceByFolderId.get(folderId);

                                Map<String, Long> newQuotas = sumByResourceIdByFolderId.get(folderId);
                                Map<String, Long> newBalance = sumByResourceIdByFolderId.get(folderId).entrySet()
                                        .stream()
                                        .collect(Collectors.toMap(Map.Entry::getKey, e -> 0L));


                                return FolderOperationLogModel.builder()
                                        .setTenantId(tenantId)
                                        .setFolderId(folderId)
                                        .setOperationDateTime(now)
                                        .setId(UUID.randomUUID().toString())
                                        .setProviderRequestId(null)
                                        .setOperationType(PROVISION_AS_QUOTA_BY_ADMIN)
                                        .setAuthorUserId(currentUser.getUser().orElseThrow().getId())
                                        .setAuthorUserUid(currentUser.getUser().orElseThrow().getPassportUid()
                                                .orElse(null))
                                        .setAuthorProviderId(null)
                                        .setSourceFolderOperationsLogId(null)
                                        .setDestinationFolderOperationsLogId(null)
                                        .setOldFolderFields(null)
                                        .setOldQuotas(new QuotasByResource(oldQuotas))
                                        .setOldBalance(new QuotasByResource(oldBalance))
                                        .setOldProvisions(new QuotasByAccount(Map.of()))
                                        .setOldAccounts(null)
                                        .setNewFolderFields(null)
                                        .setNewQuotas(new QuotasByResource(newQuotas))
                                        .setNewBalance(new QuotasByResource(newBalance))
                                        .setNewProvisions(new QuotasByAccount(Map.of()))
                                        .setActuallyAppliedProvisions(null)
                                        .setNewAccounts(null)
                                        .setAccountsQuotasOperationsId(null)
                                        .setQuotasDemandsId(null)
                                        .setOperationPhase(null)
                                        .setOrder(nextOpOrderByFolderId.get(folderId))
                                        .setCommentId(null)
                                        .setDeliveryMeta(null)
                                        .build();
                            })
                            .collect(Collectors.toList());

                    return folderOperationLogDao.upsertAllRetryable(ts, folderOperationLogModels)
                            .then(Mono.just((long) updatedQuotaModels.size()));
                }));
    }

    private Result<Void> validateProviderId(String providerId, Locale locale) {
        if (!Uuids.isValidUuid(providerId)) {
            ErrorCollection error = ErrorCollection.builder().addError(TypedError.notFound(messages
                    .getMessage("errors.provider.not.found", null, locale)))
                    .build();
            return Result.failure(error);
        }
        return Result.success(null);
    }

    private Result<ProviderModel> validateProviderForQuotaSet(Optional<ProviderModel> oProvider, Locale locale) {
        return validateExists(oProvider.orElse(null), locale).andThen(provider -> {
            if (provider.isManaged()) {
                ErrorCollection error = ErrorCollection.builder().addError(TypedError.notFound(messages
                        .getMessage("errors.provider.is.managed", null, locale)))
                        .build();
                return Result.failure(error);
            }
            return Result.success(provider);
        });
    }

    private Flux<List<String>> getFolderIdsFlux(TenantId tenantId) {
        return tableClient.usingSessionFluxRetryable(session ->
                folderDao.getAllFoldersIds(session, tenantId.getId()))
                .buffer(FOLDERS_IN_PREFETCH);
    }

    private ProviderSyncStatusDto buildProviderSyncStatusDto(
            ProvidersSyncStatusModel syncStatus, List<ProvidersSyncErrorsModel> syncErrors
    ) {
        return new ProviderSyncStatusDto(
                syncStatus.getLastSyncStart(),
                syncStatus.getLastSyncFinish(),
                ProviderSyncStatusDto.SyncStatuses.from(syncStatus.getLastSyncStatus()), // lastSyncStatus
                syncStatus.getLastSyncStats() != null ? syncStatus.getLastSyncStats().getAccountsCount() : null,
                syncStatus.getLastSyncStats() != null ? syncStatus.getLastSyncStats().getQuotasCount() : null,
                syncStatus.getLastSyncStats() != null ? syncStatus.getLastSyncStats().getSyncDuration() : null,
                syncErrors.stream().map(error -> new ProviderSyncErrorDto(
                        error.getRequestTimestamp(),
                        error.getErrors() != null ? error.getErrors().getErrorMessage() : null,
                        error.getErrors() != null ? error.getErrors().getDetails() : null
                )).collect(Collectors.toList())
        );
    }

}
